#include <stdio.h>
#include "xfibs.h"
#include "FibwP.h"
#include "graphics.h"
#include "mouse.h"
#include "popup.h"

/* Look in the header file for the prototypes - that's what it's for! */

static int bear_off(fw, board)
FibwWidget	fw;
int		board[];
{
    int		 i;
    FIBS_Rec	*pFIBS = &fw->fibw.fibs_rec;

    /* Run through all the points outside players home. Return 0 if
     * if there are pieces left 'outside'. Return 1 if you may bear off.
     */

    if (pFIBS->direction == 1) {
	for (i = 0; i < 19; i++)
	    if (pFIBS->color * board[i] > 0)
		return 0;
    }
    else {
	for (i = 7; i <= 25; i++)
	    if (pFIBS->color * board[i] > 0)
		return 0;
    }
    return 1;
}


static int find_roll(fw, move)
FibwWidget	fw;
int		move;
{
    int	i, j;

    if (!move)
	return 0; /* Piece dropped at same place as it was picked up */

    j = fw->fibw.moves_left;
    for (i = 0; i < j; i++) {
	if (fw->fibw.dice[i].used == 5)
	    j++;
	else if (!fw->fibw.dice[i].used && fw->fibw.dice[i].value == move)
            return i+1;
    }

    return 0;
}

void pipCount(fw)
FibwWidget	fw;
{
    int		 i, forward = 0, reverse = 0;
    int		 playerpip, opponentpip;
    int		*board = fw->fibw.fibs_rec.board;
    char	 string[STRING_LENGTH];

    for (i = 1; i <= 25; i++) {
	if (board[i])
	    if (board[i] > 0)
		forward += board[i] * i;
	    else reverse -= board[i] * (25 - i);
    }
    if (board[0])
	reverse -= board[0] * i;

    if (fw->fibw.fibs_rec.direction == 1) {
	playerpip = reverse;
	opponentpip = forward;
    }
    else {
	playerpip = forward;
	opponentpip = reverse;
    }
    sprintf(string, "%s: %d, %s: %d\n", fw->fibw.fibs_rec.playername,
	playerpip, fw->fibw.fibs_rec.opponentname, opponentpip);
    output_text(string);
}

static int findMove(fw, moveNum)
FibwWidget	fw;
{
    int	i, j;

    if (!fw->fibw.pieces_moved)
	return 0;

    j = fw->fibw.moves_left;
    for (i = 0; i < j; i++) {
	if (fw->fibw.dice[i].used == 5)
	    j++;
	else if (fw->fibw.dice[i].used == moveNum)
            return i+1;
    }

    return 0;
}


/* Returns the position number of the piece most points away from home */
static int highest(fw, dir, from_pos)
FibwWidget	fw;
int		dir;
int		from_pos;
{
    int		i;

    if (dir == 1) {
        for (i = 19; i <= 24; i++) {
	    if (fw->fibw.fibs_rec.board[i] * fw->fibw.fibs_rec.color > 0)
                return min(i, from_pos);
        }
    }
    else {
	for (i = 6; i >= 1; i--) {
	    if (fw->fibw.fibs_rec.board[i] * fw->fibw.fibs_rec.color > 0)
		return max(i, from_pos);
	}
    }
    /* There are two situations where we may drop down here:
     * 1) We're bearing off the last piece
     * 2) We don't have any pieces left in our home board, but do have one
     *    outside.
     * As we only use the result from this function when bearing off is legal,
     * we may return from_pos as the highest piece here...
     */
    return from_pos; 
}


/* Routine to find next available die */
static Dice * find_die(fw, die)
FibwWidget	 fw;
int		*die;
{
    int		i, j;
    Dice	*pDice;

    i = *die;		/* Start after index of last die */
    *die = 0;		/* Assume failure */

    pDice = &fw->fibw.dice[i];
    j = fw->fibw.moves_left;
    for (pDice = &fw->fibw.dice[i]; i < j; i++) {
	if (!pDice->used) {
	    *die = i + 1;
	    break;
	}
	else if (pDice->used == 5)
	    j++;
	pDice++;
    }
    return pDice;
}


/* This routine makes the move from pickUp to drop. Only adds piece to board
 * if it's not hidden (ie not an intermediate step).
 */
static void make_move(fw, board, move_info, pickUp, drop, hidden)
FibwWidget	 fw;
int		 board[];
Move		 move_info;
int		 pickUp, drop, hidden;
{
    char	 str[20];
    int		 dieNumber;
    Dice	*roll;
    FIBS_Rec	*pFIBS = &fw->fibw.fibs_rec;

    fw->fibw.pieces_moved++;
    dieNumber = find_roll(fw, move_info.dice_used);
    roll = &fw->fibw.dice[dieNumber - 1];
    roll->used = fw->fibw.pieces_moved;
    roll->from = pickUp;
    roll->to   = drop;
    roll->hit  = move_info.blot_hit;
    
    if (move_info.blot_hit) {
	/* If hidden just remove the old blot, otherwise put our blot there */
	board[drop] = (hidden) ? 0 : pFIBS->color;
	if (pFIBS->direction == 1)
	    board[25] -= pFIBS->color;
	else
	    board[0]  -= pFIBS->color;
    }
    else if (!hidden)
	board[drop] += pFIBS->color;

    /* Add the move to the command string */
    if ((pickUp == 0) || (pickUp == 25))
	sprintf(roll->move, "bar-");
    else
	sprintf(roll->move, "%i-", pickUp);
    if (drop >= 26)
	strcat(roll->move, "off ");
    else {
	sprintf(str, "%i ", drop);
	strcat(roll->move, str);
    }
}

static void undoMove(fw, dieNumber)
FibwWidget	fw;
int		dieNumber;
{
    int		 pickUp, drop;
    Dice	*roll;
    FIBS_Rec	*pFIBS = &fw->fibw.fibs_rec;

    roll = &fw->fibw.dice[dieNumber - 1];
    pickUp = roll->from;
    drop = roll->to;
    roll->used = 0;
    fw->fibw.pieces_moved--;

    if (roll->hit) {
	/* Just swap our piece with 1 off opponent's bar */
	pFIBS->board[drop] = -pFIBS->color;
	if (pFIBS->direction == 1)
	    pFIBS->board[25] += pFIBS->color;
	else
	    pFIBS->board[0]  += pFIBS->color;
    }
    else
	pFIBS->board[drop] -= pFIBS->color;

    /* Put the piece back where it came from */
    pFIBS->board[pickUp] += pFIBS->color;

    checkBearOff(fw);
    /* Change back to roll popup if no move to accept */
    if (!fw->fibw.pieces_moved)
	use_popup(fw, roll_popup_id);
}

int validUndo(fw, pickUp, drop)
FibwWidget	fw;
int		pickUp;
int		drop;
{
    Dice	*roll;
    int	i, j;

    if (!fw->fibw.pieces_moved)
	return 0;

    /* Look through the moves made to find a match pickUp->drop */
    if (fw->fibw.debug)
	fprintf(stderr, "undo: from %d->%d\n", pickUp, drop);
    j = fw->fibw.moves_left;
    for (i = 0; i < j; i++) {
	if (fw->fibw.dice[i].used == 5)
	    j++;
	else {
	    roll = &fw->fibw.dice[i];
	    if (drop == roll->from && pickUp == roll->to) {
		undoMove(fw, i + 1); 
		return 1;
	    }
	}
    }
    return 0;
}

void undoLastMove(fw)
FibwWidget	fw;
{
    int		 dieNumber;
    Dice	*roll;

    dieNumber = findMove(fw, fw->fibw.pieces_moved);
    if (!dieNumber)		/* Nothing to undo */
	return;

    undoMove(fw, dieNumber); 
}

void checkBearOff(fw)
FibwWidget	fw;
{
    fw->fibw.bear_off = bear_off(fw, fw->fibw.fibs_rec.board);
}


/* We now pass a pointer to the drop position value, this allows us to trim
 * it when bearing off. This way, we can make home appear as if it's next to
 * the home board. So multi-move testing is much (much) easier.
 *
 * Also 0 (bar) is a valid pickUp point, so return -1 for illegal moves.
 */
int legal_move(fw, pickUp, drop, move_info)
FibwWidget	 fw;
int		 pickUp;
int		*drop;
Move		*move_info;
{
    int		 i;
    int		 dropPos, top_piece;
    FIBS_Rec	*pFIBS = &fw->fibw.fibs_rec;

    dropPos = *drop;	/* For array accesses */
    if (dropPos >= 25) {
	dropPos = 27;
	*drop = 25;
    }
    else if (dropPos <= 0) {
	dropPos = 26;
	*drop = 0;
    }

    /* Blocked spot ? */
    if (pFIBS->board[dropPos] * pFIBS->color < -1) {
	move_info->reason = ms_blocked;
        return -1;
    }
    
    checkBearOff(fw);
    /* Not allowed to bear off ? */
    if ((*drop == 0 || *drop == 25) && (fw->fibw.bear_off != 1)) {
	move_info->reason = ms_cant_bear_off;
	return -1;
    }

    top_piece = highest(fw, pFIBS->direction, pickUp);
    if (fw->fibw.debug)
	fprintf(stderr, "legal: from %d->%d (%d)\n", pickUp, *drop, top_piece);
    /* Check if it's the highest possible numbered piece, and we're
     * bearing off... It's mildly confusing :)
     */
    if (((pFIBS->direction == -1) && (*drop == 0) &&
	    (!find_roll(fw, pickUp)) &&			/* not in dice_list  */
            (fw->fibw.bear_off) &&			/* legal to bear off */
            (pickUp == top_piece))) {			/* highest possible  */
        for (i = 6; i > pickUp; i--) {			/* any dices higher? */
            if (find_roll(fw, i)) {			/* Got one...        */
            /* It wasn't that bad... */
	        move_info->reason = ms_bear_off;
		move_info->dice_used = i;
		move_info->blot_hit = 0;
		*drop = 0;		/* Closer to home */
		if (fw->fibw.debug)
		    fprintf(stderr, "Higher, %d %d\n", pickUp, i);
		return pickUp;
            }
        }
	/* Couldn't find a dice that matched... */
	/* Could be a combi-move */
	move_info->reason = ms_no_such_dice;
	return -1;
    }

    if (((pFIBS->direction == 1) && (*drop == 25) && (fw->fibw.bear_off) &&
	    (!find_roll(fw, 25 - pickUp)) &&		/* not in dice_list  */
            (pickUp == top_piece))) {			/* highest possible  */
        for (i = 19; i < pickUp; i++) {
            if (find_roll(fw, 25-i)) {
	        move_info->reason = ms_bear_off;
		move_info->dice_used = 25 - i;
		move_info->blot_hit = 0;
		*drop = 25;		/* Closer to home */
		if (fw->fibw.debug)
		    fprintf(stderr, "Higher, %d %d\n", pickUp, i);
		return pickUp;
            }
        }
	/* No matching dice.... */
	/* Could be a combi-move */
	move_info->reason = ms_no_such_dice;
	return -1;
    } /* PHEW!! */

    /* Bearing off */
    if ((pFIBS->direction == -1) && (*drop == 0)) {
	if (find_roll(fw, pickUp)) {
	    move_info->reason = ms_bear_off;
	    move_info->dice_used = pickUp;
	    move_info->blot_hit = 0;
	    return pickUp;
	}
	else {
	/* Could be a combi-move */
	    move_info->reason = ms_no_such_dice;
	    move_info->dice_used = 0;
	    move_info->blot_hit = 0;
	    return -1;
	}
    }
    if ((pFIBS->direction == 1) && (*drop == 25)) { 
	if (find_roll(fw, 25-pickUp)) {
	    move_info->reason = ms_bear_off;
	    move_info->dice_used = 25-pickUp;
	    move_info->blot_hit = 0;
	    return pickUp;
	}
	else {
	/* Could be a combi-move */
	    move_info->reason = ms_no_such_dice;
	    move_info->dice_used = 0;
	    move_info->blot_hit = 0;
	    return -1;
	}
    }

    /* Not correct amount of points moved ? */
    if (!find_roll(fw, pFIBS->direction * (*drop - pickUp))) {
	/* Could be a combi-move */
	move_info->reason = ms_no_such_dice;
	move_info->dice_used = 0;
	move_info->blot_hit = 0;
	return -1;
    }
 
    /* 
     * As far as I can see, the move has to be legal, 
     * when we've come this far. All we got to do now, is to check if
     * we are hitting some blots
    */

    move_info->reason = ms_normal_move;
    move_info->dice_used = abs(pickUp - *drop);
    if (pFIBS->board[dropPos] * pFIBS->color == -1) 
	move_info->blot_hit = 1;
    else
	move_info->blot_hit = 0;

    if (fw->fibw.debug)
	fprintf(stderr, "Simple move %d->%d, %d %d\n", pickUp, *drop,
		move_info->dice_used, move_info->blot_hit);
    /* Make life simple for calling routine - home is next to the home board */
    if (*drop > 25)
	*drop = 25;
    else if (*drop < 0)
	*drop = 0;
    return pickUp;
}

/* This routine gets called to check all possible combinations of dice.
 * the rolled dice. We start by seeing if the first die gives a legal
 * move. If it does, then we try the second die and so on. If the first
 * die fails, we try the second, then the first. Any doubles will be
 * caught by the first case, so if that fails, it's illegal.
 *
 * When we do get a legal move, this routine makes all the moves except
 * the last, which is obviously a legal_move() and is handled as such
 * by the calling routine.
 */
int valid_move(fw, pickUp, drop, move_info)
FibwWidget	 fw;
int		 pickUp, drop;
Move		*move_info;
{
    int		 i, die;
    int		 step1, step2, step3, step4;
    Dice	*pDice;
    Move	 move1, move2, move3;
    FIBS_Rec	*pFIBS = &fw->fibw.fibs_rec;

    die = 0;
    pDice = find_die(fw, &die);		/* Try 1st die */

    /* To make life easier, let's make home right next to home board */
    if (drop == 27)
	drop = 25;
    else if (drop == 26)
	drop = 0;

    if (!die)		/* Can't move without dice, shouldn't happen! */
	return -1;	/* A very sick bird - totally ill eagle */

    step1 = pickUp + (pDice->value * pFIBS->direction);

    if (legal_move(fw, pickUp, &step1, move_info) != -1) {
	if (drop == step1)		/* Simple 1 die move */
	    return pickUp;

	pDice = find_die(fw, &die);		/* Try 2nd die */
	/* If no more dice, or a step too far - try second die */
	if (die && abs(step1 - pickUp) < abs(drop - pickUp)) {
	    move1 = *move_info;	/* Copy 1st move */

	    /* Try 2nd die by itself before all the clever tricks */
	    step2 = pickUp + (pDice->value * pFIBS->direction);
	    if (legal_move(fw, pickUp, &step2, move_info) != -1) {
		if (drop == step2)		/* Simple 1 die move */
		    return pickUp;
	    }

	    step2 = step1 + (pDice->value * pFIBS->direction);
	    /* Is it a legal move ? */
	    step1 = legal_move(fw, step1, &step2, move_info);

	    /* If not or too far, maybe we can do it the other way round */
	    if (step1 != -1 && abs(step2 - pickUp) <= abs(drop - pickUp)) {
		if (drop == step2) {
		    /* Jackpot! We made it to the drop point */
		    make_move(fw, pFIBS->board, move1, pickUp, step1, TRUE);
		    /* Tell calling routine new pickUp */
		    return step1;
		}
		pDice = find_die(fw, &die);		/* Try 3rd die */
		if (!die)		/* No more dice, can't reach drop */
		    return -1;

		move2 = *move_info;	/* Copy 2nd move */
		step3 = step2 + (pDice->value * pFIBS->direction);
		/* Is it a legal move ? */
		step2 = legal_move(fw, step2, &step3, move_info);
		if (step2 == -1)	/* Must be a double, can't reach drop */
		    return -1;

		if (drop == step3) {
		    /* Jackpot! We made it to the drop point */
		    make_move(fw, pFIBS->board, move1, pickUp, step1, TRUE);
		    make_move(fw, pFIBS->board, move2, step1, step2, TRUE);
		    /* Tell calling routine new pickUp */
		    return step2;
		}
		pDice = find_die(fw, &die);
		if (!die)		/* No more dice - can't make drop */
		    return -1;

		move3 = *move_info;	/* Copy 3rd move */
		step4 = step3 + (pDice->value * pFIBS->direction);
		step3 = legal_move(fw, step3, &step4, move_info);
		if (step3 == -1)	/* Can't land there */
		    return -1;

		if (drop == step4) {
		    /* We made it to the drop point */
		    make_move(fw, pFIBS->board, move1, pickUp, step1, TRUE);
		    make_move(fw, pFIBS->board, move2, step1, step2, TRUE);
		    make_move(fw, pFIBS->board, move3, step2, step3, TRUE);
		    return step3;	/* Tell calling routine new pickUp */
		}
		return -1;	/* Missed the drop with the last die */
	    }
	}
    }
    /* try second die first */

    die = 1;
    pDice = find_die(fw, &die);

    if (!die)		/* Only 1 die left - no combination then */
	return -1;

    step1 = pickUp + (pDice->value * pFIBS->direction);
    if (legal_move(fw, pickUp, &step1, move_info) != -1) {
	if (drop == step1)		/* Simple 1 die move */
	    return pickUp;

	die = 0;
	pDice = find_die(fw, &die);
	if (!die)		/* We've accessed it once, shouldn't happen */
	    return -1;

	move1 = *move_info;	/* Copy 1st move */
	step2 = step1 + (pDice->value * pFIBS->direction);
	/* Is it a legal move ? */
	step1 = legal_move(fw, step1, &step2, move_info);
	if (step1 == -1)
	    return -1;		/* Can't make it either way round */
	if (drop == step2) {
	    /* We made it to the drop point */
	    make_move(fw, pFIBS->board, move1, pickUp, step1, TRUE);
	    return step1;		/* Tell calling routine new pickUp */
	}
    }
    return -1;	/* Only 1 combination with reverse dice */
}


static int valid_pickup(w, event)
Widget		 w;
XButtonEvent	*event;
{
    FibwWidget	 fw = (FibwWidget) w;
    FIBS_Rec	*pFIBS = &fw->fibw.fibs_rec;
    XPoint	 point;
    int		 pos, i;

    /* Check if it's our turn. Don't want to move around otherwise do we? */
    if ((strcmp(pFIBS->playername, "You") != 0) ||
	(pFIBS->turn != pFIBS->color))
	return -1;

    /* Well, it's our turn, but do we have any more pieces left to move? */
    if (fw->fibw.pieces_moved == fw->fibw.moves_left)
	return -1;

    /* OK, it's our turn... Now save the original board setup, in case we
     * change our mind (hardly likely, but you never know... :))
     * Just do that for the first piece we move... (almost forgot that one)
    */
    if (!fw->fibw.pieces_moved)  /* We'll increase this number when we drop
                                    the piece */
        for (i = 0; i <= 27; i++)
            fw->fibw.oldboard[i] = pFIBS->board[i];

    /* Let's check if we may bear off... */
    checkBearOff(fw);
    point.x = event->x;
    point.y = event->y;

    /* get board position of the piece we're moving */
    pos = get_position_number(fw, &point);

    if (pFIBS->board[pos] * pFIBS->color <= 0)	/* It's not our piece */
        return -1;
    if (pFIBS->direction == 1) {
        if ((pFIBS->board[0] != 0) && (pos != 0))
            return -1;			/* Have to get off the bar first... */
    }
    else if ((pFIBS->board[25] != 0) && (pos != 25))
	return -1;

    return pos;
}


static int do_drop(fw, drop)
FibwWidget	fw;
int		drop;
{
    int	 pickUp = -1;
    Move move_info;

    /* Make sure we keep to piece inside our array bounds */
    if (drop >= 27)
	drop = 27;
    else if (drop <= 0)
	drop = 26;
    /* Check if move matches a rolled die */
    if (drop != fw->fibw.moved_from_pos)
	pickUp = valid_move(fw, fw->fibw.moved_from_pos, drop, &move_info);

    if (pickUp == -1) {
	/* not legal, put piece back */
	fw->fibw.fibs_rec.board[fw->fibw.moved_from_pos] +=
		fw->fibw.fibs_rec.color;
	/* Check if it's an undo */
	validUndo(fw, fw->fibw.moved_from_pos, drop);
	checkBearOff(fw);
    }
    else { /* legal move */
	/* Make the move (not hidden) */
	make_move(fw, fw->fibw.fibs_rec.board, move_info, pickUp, drop, FALSE);

	/* change to the "accept move, reject move etc." popup */
	if (fw->fibw.pieces_moved >= 1) 
            use_popup(fw, move_popup_id);
    }

    redrawBoard(fw);
    fw->fibw.moving = 0;
    /* Flag successful moves */
    return (pickUp != -1);
}

int pickupPiece(w, event)
Widget		 w;
XButtonEvent	*event;
{
    FibwWidget	 fw = (FibwWidget) w;
    int		 pos;
    XExposeEvent fake_event;
    Display	*dpy = XtDisplay(fw);
    Drawable	 drawable = RootWindow(dpy,DefaultScreen(dpy));
    unsigned int depth = DefaultDepthOfScreen(XtScreen(fw));
    FIBS_Rec	*pFIBS = &fw->fibw.fibs_rec;
    int		 i, pieceSize;

    pos = valid_pickup(w, event);

    if (pos != -1) {
	fw->fibw.moved_from_pos = pos;

	/* removes a piece from the board */
	pFIBS->board[pos] -= pFIBS->color;

	pieceSize = fw->fibw.piece_size;
	/* copy blank background onto board pixmap */
	XCopyArea(dpy, fw->fibw.back_pixmap,
		fw->fibw.board_pixmap, fw->fibw.draw_gc,
		fw->fibw.pos_rec[pos].x1,
		fw->fibw.pos_rec[pos].y1,
		fw->fibw.pos_rec[pos].x2 - fw->fibw.pos_rec[pos].x1,
		fw->fibw.pos_rec[pos].y2 - fw->fibw.pos_rec[pos].y1,
		fw->fibw.pos_rec[pos].x1,
		fw->fibw.pos_rec[pos].y1);
	/* and fill it with pieces */
	draw_pieces(fw,pos);
	/* Create small pixmap and fill it with current background... */
	fw->fibw.small_pixmap = XCreatePixmap(dpy, drawable,
		2 * pieceSize,
		2 * pieceSize,
		depth);
	XCopyArea(dpy, fw->fibw.board_pixmap,
		fw->fibw.small_pixmap, fw->fibw.draw_gc,
		event->x - pieceSize,
		event->y - pieceSize,
		2 * pieceSize,
		2 * pieceSize,
		0, 0);
	/* then show the picked up piece */
	draw_one_piece_at_loc(fw, event->x - pieceSize / 2,
		event->y - pieceSize / 2, pFIBS->color);
	/* Where is it now? */
	fw->fibw.current_piece_pos.x = event->x;
	fw->fibw.current_piece_pos.y = event->y;
	/* Now, let's put it up on screen */
	fake_event.x = min(fw->fibw.pos_rec[pos].x1, event->x - pieceSize);
	fake_event.y = min(fw->fibw.pos_rec[pos].y1, event->y - pieceSize);
	fake_event.width = max(fw->fibw.pos_rec[pos].x2, event->x + pieceSize)
			   - fake_event.x;
	fake_event.height = max(fw->fibw.pos_rec[pos].y2, event->y + pieceSize)
			    - fake_event.y;
	Redisplay((Widget)fw, &fake_event);
	fw->fibw.moving = 1; /* We're on our way... */
    }
    return pos;
}


void dropPiece(w, event)
Widget		 w;
XButtonEvent	*event;
{
    FibwWidget	 fw = (FibwWidget) w;
    Display	*dpy = XtDisplay(fw);
    XPoint	 point;
    int		 drop;

    if (!fw->fibw.moving) return;

    point.x=event->x;
    point.y=event->y;

    /* Undraw piece, in case it's off the board */
    undraw_current(dpy, fw, event);
    /* Destroy small pixmap */
    if (fw->fibw.small_pixmap) {
        XFreePixmap(dpy, fw->fibw.small_pixmap);
        fw->fibw.current_piece_pos.x = -1;
        fw->fibw.current_piece_pos.y = -1;
    }
    drop = get_position_number(fw, &point);

    do_drop(fw, drop);
}


void clickPiece(w, event)
Widget w;
XButtonEvent *event;
{
    FibwWidget	 fw = (FibwWidget) w;
    Dice	*pDice;
    Move	 move_info;
    int		 pickUp, drop;
    int		 temp, die = 0;

    pickUp = pickupPiece(w, event);
    if (pickUp == -1)
	return;

    do {
	/* Find first available die */
	pDice = find_die(fw, &die);

	drop = pickUp + (pDice->value * fw->fibw.fibs_rec.direction);
	if (drop >= 25)
	    drop = 27;
	else if (drop <= 0)
	    drop = 26;

	temp = valid_move(fw, pickUp, drop, &move_info);
    } while ((temp == -1) && die);

    /* If it can't drop it, it'll put it back */
    do_drop(fw, drop);
}


void movePiece(w, event)
Widget w;
XButtonEvent *event;
{
    FibwWidget fw = (FibwWidget) w;
    Display *dpy = XtDisplay(fw);
    XExposeEvent fake_event;


    /* Making sure we're actually moving a piece */
    if (!fw->fibw.moving) return;

    /* First, remove the image of the moving piece, then put it in again
       at the new location. Finally show the image on screen */
    remove_current(dpy, fw, event);

    /* Now, draw the moving piece... */
    draw_one_piece_at_loc(fw,event->x-fw->fibw.piece_size/2,
            event->y-fw->fibw.piece_size/2,fw->fibw.fibs_rec.color);

    fake_event.x=min(fw->fibw.current_piece_pos.x-fw->fibw.piece_size,
                     event->x-fw->fibw.piece_size);
    fake_event.y=min(fw->fibw.current_piece_pos.y-fw->fibw.piece_size,
                     event->y-fw->fibw.piece_size);
    fake_event.width=max(fake_event.x+2*fw->fibw.piece_size,
                         fw->fibw.current_piece_pos.x+fw->fibw.piece_size)
                    -fake_event.x;
    fake_event.height=max(fake_event.y+2*fw->fibw.piece_size,
                         fw->fibw.current_piece_pos.y+fw->fibw.piece_size)
                    -fake_event.y;
    Redisplay((Widget)fw,&fake_event);
    fw->fibw.current_piece_pos.x=event->x;
    fw->fibw.current_piece_pos.y=event->y;
    return;
}
