/*
  xbomber.c
  
  XBomber 0.2
  
  A 4-player networked "bombing" game (based loosely on "Super Bomberman")
  for X-Window.
  
  by Bill Kendrick
  January 28, 1998 - February 19, 1998
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <sys/types.h>
#include <sys/time.h>
#include <math.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "window.h"
#include "connect.h"
#include "hints.h"
#include "visual.h"
#include "gc.h"
#include "color.h"
#include "randnum.h"
#include "text.h"


#define MAX_PLAYERS 4
#define CHANCE_OF_UPGRADE 3
#define MAX_FIRE_SIZE 8
#define BOMB_COUNTDOWN 160

#define OBJ_DIRT 0
#define OBJ_MAN_DOWN 1
#define OBJ_MAN_RIGHT 2
#define OBJ_MAN_UP 3
#define OBJ_MAN_LEFT 4
#define OBJ_BLOCK 5
#define OBJ_BLOCK2 6
#define OBJ_BOX 7
#define OBJ_BOMB 8
#define OBJ_RADIO_BOMB 9
#define OBJ_EXPLOSION1 10
#define OBJ_EXPLOSION2 11
#define OBJ_MOREFIRE 12
#define OBJ_TOTALFIRE 13
#define OBJ_PUSHER 14
#define OBJ_KICKER 15
#define OBJ_MOREBOMB 16
#define OBJ_RADIOUPGRADE 17
#define OBJ_NUKE 18
#define OBJ_INVERT 19
#define OBJ_WARP 20
#define OBJ_TNT 21
#define OBJ_DEATH 22
#define OBJ_DEATH_EATING 23
#define NUM_OBJECTS 24

#define FIRST_UPGRADE OBJ_MOREFIRE
#define NUM_UPGRADES (OBJ_INVERT - FIRST_UPGRADE + 1)

#define YOU_COLOR "LightYellow"
#define THEM_COLOR "light goldenrod"
#define GOODIE_COLOR "green"

char * object_names[NUM_OBJECTS] = {
  "dirt", "man_down", "man_right", "man_up", "man_left", "block", "block2",
  "box", "bomb", "radio_bomb", "explosion1", "explosion2",
  "morefire", "totalfire", "pusher", "kicker", "morebomb",
  "radioupgrade", "nuke", "invert", "warp", "tnt", "death", "death_eating"};

char * object_colors[NUM_OBJECTS] = {
  "Black", THEM_COLOR, THEM_COLOR, THEM_COLOR, THEM_COLOR,
  "deep sky blue", "blue", "red", "grey", "grey", "yellow", "yellow",
  GOODIE_COLOR, GOODIE_COLOR, GOODIE_COLOR, GOODIE_COLOR, GOODIE_COLOR,
  GOODIE_COLOR, GOODIE_COLOR, GOODIE_COLOR, "white", "red", "yellow",
  "yellow"};


#define FLT_BALLOON 0
#define FLT_BOMB_BALLOON 1
#define FLT_CLOUD 2
#define NUM_FLOATERS 3

char * floater_names[NUM_FLOATERS] = {
  "balloon", "bomb-balloon", "cloud"};

char * floater_colors[NUM_FLOATERS] = {
  "blue", "red", "white"};


#define SND_EXPLODE 0
#define SND_GOT 1
#define SND_KICKER 2
#define SND_OUCH1 3
#define SND_OUCH2 4
#define SND_OUCH3 5
#define SND_PUSHER 6
#define SND_WARP 7
#define SND_LEVELEND1 8
#define SND_LEVELEND2 9
#define SND_LEVELEND3 10
#define SND_LEVELEND4 11
#define SND_LAUGH 12
#define SND_INVERT 13
#define NUM_SOUNDS 14

char * sound_names[NUM_SOUNDS] = {
  "explode", "got", "kicker", "ouch1", "ouch2", "ouch3", "pusher", "warp",
  "levelend1", "levelend2", "levelend3", "levelend4", "laugh", "invert"};

#define NSND_LEVEL 0
#define NSND_1 1
#define NSND_2 2
#define NSND_3 3
#define NSND_4 4
#define NSND_5 5
#define NSND_6 6
#define NSND_7 7
#define NSND_8 8
#define NSND_9 9
#define NSND_10 10
#define NSND_11 11
#define NSND_12 12
#define NSND_13 13
#define NSND_14 14
#define NSND_15 15
#define NSND_20 16
#define NSND_30 17
#define NSND_40 18
#define NSND_50 19
#define NSND_60 20
#define NSND_70 21
#define NSND_80 22
#define NSND_90 23
#define NSND_TEEN 24
#define NUM_NSOUNDS 25

char * nsound_names[NUM_NSOUNDS] = {
  "level", "1", "2", "3", "4", "5", "6", "7", "8", "9", "10", "11", "12",
  "13", "14", "15", "20", "30", "40", "50", "60", "70", "80", "90", "teen"};

unsigned char au_file_data[NUM_SOUNDS][25000],
  nau_file_data[NUM_NSOUNDS][6000];

int sound_len[NUM_SOUNDS], nsound_len[NUM_NSOUNDS];


/* Global X-Window Variables: */

char server[MAX_PLAYERS][512];
Cursor cursor[MAX_PLAYERS];
XColor blackxcolor[MAX_PLAYERS], whitexcolor[MAX_PLAYERS];
Display * display[MAX_PLAYERS];
Colormap colormap[MAX_PLAYERS];
Window window[MAX_PLAYERS], root[MAX_PLAYERS];
GC whitegc[MAX_PLAYERS], blackgc[MAX_PLAYERS];
GC colorgcs[MAX_PLAYERS][NUM_OBJECTS], yougc[MAX_PLAYERS];
GC floatergcs[MAX_PLAYERS][NUM_FLOATERS],
  floatermaskgcs[MAX_PLAYERS][NUM_FLOATERS];
XFontStruct * font[MAX_PLAYERS];
int fh[MAX_PLAYERS];
int screen[MAX_PLAYERS], black[MAX_PLAYERS], white[MAX_PLAYERS],
  has_color[MAX_PLAYERS];
Pixmap object_pixmaps[MAX_PLAYERS][NUM_OBJECTS];
Pixmap floater_pixmaps[MAX_PLAYERS][NUM_FLOATERS],
  floater_masks[MAX_PLAYERS][NUM_FLOATERS];
Pixmap cursor_pixmap[MAX_PLAYERS], cursor_mask[MAX_PLAYERS];


/* Level stats: */

int lev_usebox[10] =  {1, 1, 1, 1, 0, 1, 1, 0, 0, 0};
int lev_usetnt[10] =  {0, 0, 1, 1, 1, 0, 1, 1, 0, 0};
int lev_usewarp[10] = {0, 0, 0, 0, 0, 1, 1, 1, 1, 0};


/* Arena size: */

#define MAP_WIDTH 17
#define MAP_HEIGHT 17
#define WIDTH (MAP_WIDTH * 32)
#define HEIGHT ((MAP_HEIGHT + 1) * 32)


/* How many randomly placed boxes to place in a level: */

#define HOW_MANY_BOXES 100
#define HOW_MANY_TNTS 50
#define HOW_MANY_WARPS 25


/* How many bombs and explosions max on the screen? */

#define MAX_BOMBS 20
#define MAX_EXPLOSIONS (MAP_WIDTH * MAP_HEIGHT)


/* How many floaters max on the screen? */

#define MAX_FLOATERS 5


/* How long a player gets after all other players are dead: */

#define FREETIME 400


/* Speed controls: */

#define FRAMERATE 10000


/* Size of sound buffer queue: */

#define SOUND_BUF 40960


/* Typedefs: */

typedef struct player_type {
  int bombs_out, bombs_max, fire_size;
  int alive;
  int x, y;
  int score;
  int shape;
  int seenmap[MAP_HEIGHT][MAP_WIDTH];
  int radiobomb, pusher, kicker;
  int radio_detonate;
  int cmp_time, cmp_dir;
} player_type;

typedef struct bomb_type {
  int alive, time;
  int x, y;
  int shape, size;
  int owner;
  int moving, xm, ym;
} bomb_type;

typedef struct explosion_type {
  int alive, time;
  int x, y;
} explosion_type;

typedef struct death_type {
  int alive, time;
  int x, y;
  int seek;
  int shape;
} death_type;

typedef struct floater_type {
  int f_type;
  int alive, time;
  int x, y;
  int xm, ym;
} floater_type;


/* Global game Variables: */

int level, use_sound, num_servers, winner, leveltime, cmp_movetoggle, toggle,
  ttoggle;
int map[MAP_HEIGHT][MAP_WIDTH], seenmap[MAP_HEIGHT][MAP_WIDTH];
int xms[5], yms[5];
char message[128];
struct bomb_type bomb[MAX_BOMBS];
struct explosion_type explosion[MAX_EXPLOSIONS];
struct player_type player[MAX_PLAYERS];
struct death_type death;
struct floater_type floaters[MAX_FLOATERS];
int sound_position;
int pid;
int pipefds[2];
FILE * sound_fs;
int devaudio_fd;
unsigned char sound_data[SOUND_BUF];


/* Local function prototypes: */

void eventloop(void);
void setup(int argc, char * argv[]);
void Xsetup(int pln);
unsigned long MyAllocNamedColor(Display *display, Colormap colormap,
                                char* colorname, unsigned long default_color,
                                int has_color);
void playsound(int whichfile);
void playnsound(int whichfile);
void drawblock(int pln, int x, int y, int c);
void makeexplosion(int x, int y, int size);
int addexplosion(int x, int y);
int moveok(int pln, int xm, int ym);
void redrawstatus(int pln);
int dropbomb(int pln);
void initlevel(void);
void initplayergoodies(int i);
void addexplosion_atomic(int x, int y);
void control_computer(int pln);
void make_death_seek(void);
int empty(int x, int y);
void kill_bomb(int i);
int floater_touching(int i, int x, int y);


/* ---- MAIN FUNCTION ---- */

int main(int argc, char * argv[])
{
  int i;
  
  
  /* Program setup: */
  
  setup(argc, argv);
  randinit();
  
  
  /* Connect to the X Servers and draw the program's windows: */
  
  for (i = 0; i < num_servers; i++)
    Xsetup(i);
  
  
  /* Run the main loop: */
  
  eventloop();
}


/* --- MAIN EVENT LOOP --- */

void eventloop(void)
{
  int i, j, e_loop, pln, done, x, y, found, c, erase_it, killed_one,
    foundblocker, freetime, num_alive, humans_alive, z, best, score;
  int newblocks_x, newblocks_y, newblocks_dir;
  int newblocks_top, newblocks_bottom, newblocks_left, newblocks_right;
  char string[128];
  XEvent event;
  char key[1024];
  KeySym keysym;
  XComposeStatus composestatus;
  struct timeval now, then;
  
  
  /* Init player stuff: */
  
  for (i = 0; i < MAX_PLAYERS; i++)
    {
      player[i].score = 0;
      
      initplayergoodies(i);
    }
  
  
  initlevel();

  
  /* MAIN LOOP: */
  
  toggle = 0;
  ttoggle = 0;
  cmp_movetoggle = 0;
  done = False;
  
  do
    {
      /* Toggle our toggle switches: */
      
      toggle = 1 - toggle;
      
      if (toggle)
	ttoggle = 1 - ttoggle;
      
      cmp_movetoggle = (cmp_movetoggle + 1) % 5;
      
      
      gettimeofday(&then, NULL);
      
      
      /* Count how many HUMAN players are currently alive: */
      
      humans_alive = 0;
      
      for (pln = 0; pln < num_servers; pln++)
	if (player[pln].alive)
	  humans_alive++;
      
      
      /* Human controlled players: */
      
      for (pln = 0; pln < num_servers; pln++)
	{
	  /* Get and handle events: */
	  
	  strcpy(key, "");
	  
	  while (XPending(display[pln]))
	    {
	      XNextEvent(display[pln], &event);
	      
	      if (event.type == KeyPress)
		{
		  /* Get the key's name: */
		  
		  XLookupString(&event.xkey, key, 1, &keysym,
				&composestatus);
		  
		  if (XKeysymToString(keysym) != NULL)
		    strcpy(key, XKeysymToString(keysym));
		}
	      else if (event.type == Expose)
		{
		  for (y = 0; y < MAP_HEIGHT; y++)		  
		    for (x = 0; x < MAP_WIDTH; x++)
		      player[pln].seenmap[y][x] = -1;
		  
		  redrawstatus(pln);
		}
	    }
	  
	  
	  if (strcasecmp(key, "Q") == 0)
	    {
	      /* Q: Quit: */
	      
	      done = True;
	    }
	  
	  
	  /* These controls only work if the player is actually alive still! */
	  
	  if (player[pln].alive)
	    {
	      if (strcasecmp(key, "Up") == 0)
		{
		  /* Up: Move up */
		  
		  player[pln].shape = OBJ_MAN_UP;
		  
		  if (moveok(pln, 0, -1))
		    {
		      player[pln].y--;
		      
		      if (player[pln].y < 0)
			player[pln].y = MAP_HEIGHT - 1;
		    }
		}
	      else if (strcasecmp(key, "Down") == 0)
		{
		  /* Down: Move down */
		  
		  player[pln].shape = OBJ_MAN_DOWN;
		  
		  if (moveok(pln, 0, 1))
		    {
		      player[pln].y++;
		      
		      if (player[pln].y > MAP_HEIGHT - 1)
			player[pln].y = 0;
		    }
		}
	      else if (strcasecmp(key, "Left") == 0)
		{
		  /* Left: Move left */
		  
		  player[pln].shape = OBJ_MAN_LEFT;
		  
		  if (moveok(pln, -1, 0))
		    {
		      player[pln].x--;
		      
		      if (player[pln].x < 0)
			player[pln].x = MAP_WIDTH -1;
		    }
		}
	      else if (strcasecmp(key, "Right") == 0)
		{
		  /* Right: Move right */
		  
		  player[pln].shape = OBJ_MAN_RIGHT;
		  
		  if (moveok(pln, 1, 0))
		    {
		      player[pln].x++;
		      
		      if (player[pln].x > MAP_WIDTH - 1)
			player[pln].x = 0;
		    }
		}
	      else if ((strcasecmp(key, " ") == 0 ||
			strcasecmp(key, "Space") == 0) &&
		       winner == -1)
		{
		  /* Space: Drop bomb */
		  
		  dropbomb(pln);
		}
	      else if (strcasecmp(key, "Return") == 0 ||
		       strcasecmp(key, "Enter") == 0)
		{
		  /* Return: Explode radio bomb */
		  
		  player[pln].radio_detonate = 1;
		}
	    }
	}
      
      
      /* Copy main ("underneath") map into seen map: */
      
      for (y = 0; y < MAP_HEIGHT; y++)
	for (x = 0; x < MAP_WIDTH; x++)
	  seenmap[y][x] = map[y][x];
      
      
      /* Draw bombs onto main seen map: */
      
      for (i = 0; i < MAX_BOMBS; i++)
	if (bomb[i].alive == 1)
	  seenmap[bomb[i].y][bomb[i].x] = bomb[i].shape;
      
      
      /* Draw death onto main seen map: */
      
      if (death.alive)
	seenmap[death.y][death.x] = death.shape;
      
      
      /* Handle bombs: */
      
      for (i = 0; i < MAX_BOMBS; i++)
	{
	  if (bomb[i].alive == 1)
	    {
	      if (bomb[i].moving == 0)
		{
		  if (bomb[i].shape != OBJ_RADIO_BOMB)
		    bomb[i].time--;
		  else if (bomb[i].shape == OBJ_RADIO_BOMB)
		    {
		      if (player[bomb[i].owner].radio_detonate == 1)
			bomb[i].time = 0;
		    }
		}
	      else
		{
		  /* Move bomb until it finds land: */
		  
		  if (ttoggle)
		    {
		      bomb[i].x = bomb[i].x + bomb[i].xm;	
		      if (bomb[i].x < 0)
			bomb[i].x = MAP_WIDTH - 1;
		      if (bomb[i].x > MAP_WIDTH - 1)
			bomb[i].x = 0;
		      
		      bomb[i].y = bomb[i].y + bomb[i].ym;	
		      if (bomb[i].y < 0)
			bomb[i].y = MAP_HEIGHT - 1;
		      if (bomb[i].y > MAP_HEIGHT - 1)
			bomb[i].y = 0;
		    }
		  
		  if (seenmap[bomb[i].y][bomb[i].x] == OBJ_DIRT)
		    bomb[i].moving = 0;
		}
	      
	      if (bomb[i].time <= 0)
		{
		  /* Explode the bomb: */
		  
		  kill_bomb(i);
		  makeexplosion(bomb[i].x, bomb[i].y, bomb[i].size);
		}
	    }
	}
      
      
      /* Handle explosions: */
      
      for (i = 0; i < MAX_EXPLOSIONS; i++)
	{
	  if (explosion[i].alive == 1)
	    {
	      explosion[i].time--;
	      
	      if (explosion[i].time <= 0)
		explosion[i].alive = 0;
	    }
	}
      
      
      /* Handle floaters: */
      
      if (cmp_movetoggle == 0 && 0 == 1)
	{
	  for (i = 0; i < MAX_FLOATERS; i++)
	    {
	      if (floaters[i].alive == 1)
		{
		  /* Make their map so that it needs to be redrawn: */
		  
		  x = floaters[i].x / 32;
		  y = floaters[i].y / 32;
		  
		  /*
		  for (pln = 0; pln < num_servers; pln++)
		    {
		      player[pln].seenmap[y][x] = -1;
		      if (x < MAP_WIDTH)
			player[pln].seenmap[y][x + 1] = -1;
		      if (y < MAP_HEIGHT)
			{
			  player[pln].seenmap[y + 1][x] = -1;
			  if (x < MAP_WIDTH)
			    player[pln].seenmap[y + 1][x + 1] = -1;
			  
			  if (y < MAP_HEIGHT - 1)
			    {
			      player[pln].seenmap[y + 2][x] = -1;
			      if (x < MAP_WIDTH)
				player[pln].seenmap[y + 2][x + 1] = -1;
			    }
			}
		    }
		    */
		  
		    
		  /* Move floater: */
		  
		  floaters[i].x = floaters[i].x + floaters[i].xm;
		  
		  if (floaters[i].x < 0)
		    floaters[i].x = MAP_WIDTH * 32 - 32;
		  else if (floaters[i].x > MAP_WIDTH * 32 - 32)
		    floaters[i].x = 0;
		  
		  floaters[i].y = floaters[i].y + floaters[i].ym;
		  
		  if (floaters[i].y < 0)
		    floaters[i].y = MAP_HEIGHT * 32 - 48 - 32;
		  else if (floaters[i].y > MAP_HEIGHT * 32 - 48 - 32)
		    floaters[i].y = 0;
		  
		  
		  if (floaters[i].f_type == FLT_BALLOON)
		    {
		      for (j = 0; j < MAX_BOMBS; j++)
			{
			  if (floater_touching(i, bomb[j].x, bomb[j].y) &&
			      bomb[j].alive == 1)
			    {
			      kill_bomb(j);
			      
			      floaters[i].f_type = FLT_BOMB_BALLOON;
			      floaters[i].time = randnum(100) + 100;
			    }
			}
		    }
		  else if (floaters[i].f_type == FLT_BOMB_BALLOON)
		    {
		      floaters[i].time--;
		      
		      if (floaters[i].time <= 0)
			{
			  floaters[i].alive = 0;
			  makeexplosion(floaters[i].x / 32,
					floaters[i].y / 32, 3);
			}
		    }
		  else if (floaters[i].f_type == FLT_CLOUD)
		    {
		      /* Keep clouds vertically stable: */
		      
		      if (floaters[i].ym < 0)
			floaters[i].ym = floaters[i].ym + 1;
		      else
			floaters[i].ym = floaters[i].ym - 1;
		      
		      
		      /* Make it appear as if "new" clouds are appearing
			 on the left and right: */
		      
		      if (floaters[i].x == 0 ||
			  floaters[i].x == MAP_WIDTH * 32 - 32)
			{
			  floaters[i].y = randnum(MAP_HEIGHT * 32 - 32);
			}
		    }
		}
	      else
		{
		  if (randnum(100) < 50)
		    {
		      /* Turn on a floater: */
		      
		      floaters[i].alive = 1;
		      
		      
		      /* Pick a type: */
		      
		      do
			{
			  floaters[i].f_type = randnum(NUM_FLOATERS);
			}
		      while (floaters[i].f_type == FLT_BOMB_BALLOON);
		      
		      
		      /* Start it at the top, bottom, left or right
			 (no middle) */
		      
		      if (randnum(10) < 5)
			{
			  floaters[i].x = randnum(MAP_WIDTH * 32);
			  
			  if (randnum(10) < 5)
			    floaters[i].y = 0;
			  else
			    floaters[i].y = MAP_HEIGHT * 32;
			}
		      else
			{
			  floaters[i].y = randnum(MAP_HEIGHT * 32);
			  
			  if (randnum(10) < 5)
			    floaters[i].x = 0;
			  else
			    floaters[i].x = MAP_WIDTH * 32;
			}

		      
		      /* Pick a direction/speed: */
		      
		      do
			{
			  floaters[i].xm = randnum(10) - 5;
			}
		      while (floaters[i].xm == 0);
		      
		      do
			{
			  floaters[i].ym = randnum(10) - 5;
			}
		      while (floaters[i].ym == 0);
		    }
		}
	    }
	}
      
      
      /* Draw explosions onto main seen map: */
      
      killed_one = 0;
      
      for (i = 0; i < MAX_EXPLOSIONS; i++)
	{
	  if (explosion[i].alive == 1)
	    {
	      seenmap[explosion[i].y][explosion[i].x] = (OBJ_EXPLOSION2 -
							 (explosion[i].time /
							  8));
	      
	      for (j = 0; j < MAX_PLAYERS; j++)
		{
		  if (player[j].alive &&
		      player[j].x == explosion[i].x &&
		      player[j].y == explosion[i].y)
		    {
		      player[j].alive = 0;
		      initplayergoodies(j);
		      killed_one = 1;
		      
		      z = randnum(3);
		      
		      if (z == 0)
			playsound(SND_OUCH1);
		      else if (z == 1)
			playsound(SND_OUCH2);
		      else if (z == 3)
			playsound(SND_OUCH3);
		    }
		}
	    }
	}
      
      
      /* Make players pick up things: */
      
      for (pln = 0; pln < MAX_PLAYERS; pln++)
	{
	  if (player[pln].alive)
	    {
	      erase_it = 0;
	      c = map[player[pln].y][player[pln].x];
	      
	      if (c == OBJ_MOREFIRE)
		{
		  /* Now we have more fire: */
		  
		  player[pln].fire_size++;
		  if (player[pln].fire_size > MAX_FIRE_SIZE)
		    player[pln].fire_size = MAX_FIRE_SIZE;
		  
		  erase_it = 1;
		}
	      else if (c == OBJ_TOTALFIRE)
		{
		  /* Now we have the max. fire!: */
		  
		  player[pln].fire_size = MAX_FIRE_SIZE;
		  erase_it = 1;
		}
	      else if (c == OBJ_PUSHER)
		{
		  /* Now we can push blocks: */
		  
		  player[pln].pusher = 1;
		  erase_it = 1;
		}
	      else if (c == OBJ_KICKER)
		{
		  /* Now we can kick bombs:*/
		  
		  player[pln].kicker = 1;
		  erase_it = 1;
		}
	      else if (c == OBJ_MOREBOMB)
		{
		  /* Now we can drop more bombs: */
		     
		  player[pln].bombs_max++;
		  if (player[pln].bombs_max > MAX_BOMBS)
		    player[pln].bombs_max = MAX_BOMBS;
		  
		  player[pln].radiobomb = 0;
		  erase_it = 1;
		}
	      else if (c == OBJ_RADIOUPGRADE)
		{
		  /* Pick up a radio bomb (now we can only drop one bomb
		     at a time: */
		  
		  player[pln].radiobomb = 1;
		  player[pln].bombs_max = 1;
		  erase_it = 1;
		}
	      else if (c == OBJ_NUKE)
		{
		  /* Nuker causes all bombs to go off NOW, including
		     radiobombs! */
		  
		  for (i = 0; i < MAX_BOMBS; i++)
		    bomb[i].time = 1;
		  
		  for (i = 0; i < MAX_PLAYERS; i++)
		    player[i].radio_detonate = 1;
		  
		  erase_it = 1;
		}
	      else if (c == OBJ_INVERT)
		{
		  /* Invert causes blocks to turn to bricks and vice-versa */
		  
		  for (y = 0; y < MAP_HEIGHT; y++)
		    {
		      for (x = 0; x < MAP_WIDTH; x++)
			{
			  if (map[y][x] == OBJ_BOX)
			    map[y][x] = OBJ_BLOCK;
			  else if (map[y][x] == OBJ_BLOCK)
			    map[y][x] = OBJ_BOX;
			  else if (map[y][x] == OBJ_TNT)
			    map[y][x] = OBJ_BLOCK2;
			  else if (map[y][x] == OBJ_BLOCK2)
			    map[y][x] = OBJ_TNT;
			}
		    }
		  
		  playsound(SND_INVERT);
		  
		  erase_it = 1;
		}
	      else if (c == OBJ_WARP)
		{
		  /* Pick a random place to warp: */
		  
		  do
		    {
		      player[pln].x = randnum(MAP_WIDTH);
		      player[pln].y = randnum(MAP_HEIGHT);
		    }
		  while (map[player[pln].y][player[pln].x] != OBJ_DIRT);

		  playsound(SND_WARP);
		}
	      
	      
	      /* If it's a pick-up item, remove it from the map and
		 play the "picked-up-an-item" sound: */
	      
	      if (erase_it == 1)
		{
		  playsound(SND_GOT);
		  
		  map[player[pln].y][player[pln].x] = OBJ_DIRT;
		  
		  redrawstatus(pln);
		}
	    }
	}
      
      
      /* Draw players onto main seen map: */
      
      for (pln = 0; pln < MAX_PLAYERS; pln++)
	{
	  if (player[pln].alive)
	    seenmap[player[pln].y][player[pln].x] = player[pln].shape;
	}
      
      
      /* Computer controlled players: */
      
      if (ttoggle)
	{
	  for (pln = num_servers; pln < MAX_PLAYERS; pln++)
	    {
	      if (player[pln].alive)
		control_computer(pln);
	    }
	}
      
      
      /* Redraw the changed part of player's windows: */
      
      for (pln = 0; pln < num_servers; pln++)
	{
	  for (y = 0; y < MAP_HEIGHT; y++)
	    {
	      for (x = 0; x < MAP_WIDTH; x++)
		{
		  if (seenmap[y][x] != player[pln].seenmap[y][x])
		    {
		      drawblock(pln, x, y, seenmap[y][x]);
		      player[pln].seenmap[y][x] = seenmap[y][x];
		    }
		}
	    }
	}
      
      
      /* Draw floaters: */
      
      for (pln = 0; pln < num_servers; pln++)
	{
	  for (i = 0; i < MAX_FLOATERS; i++)
	    {
	      if (floaters[i].alive == 1)
		{
		  /* Draw the floater: */
		  
		  x = floaters[i].x;
		  y = floaters[i].y;
		  
		  XCopyPlane(display[pln],
			     floater_masks[pln][floaters[i].f_type],
			     window[pln],
			     floatermaskgcs[pln][floaters[i].f_type],
			     0, 0, 32, 48, x, y, 1);
		  
		  XCopyPlane(display[pln],
			     floater_pixmaps[pln][floaters[i].f_type],
			     window[pln], floatergcs[pln][floaters[i].f_type],
			     0, 0, 32, 48, x, y, 1);
		  
		  XFlush(display[pln]);
		}
	    }
	}
      
      
      /* Handle end-of-level stuff: */
      
      if (winner != -1)
	{
	  freetime--;
	  
	  if (freetime <= 0)
	    {
	      level = (level + 1) % 100;
	      
	      initlevel();
	      
	      for (i = 0; i < num_servers; i++)
		redrawstatus(i);
	    }
	}
      
      
      /* Keep track of level's time: */
      
      if (leveltime > 0)
	{
	  leveltime--;
	  
	  if (leveltime <= 0)
	    {
	      newblocks_x = 0;
	      newblocks_y = 0;
	      newblocks_dir = 1;
	      
	      newblocks_top = 0;
	      newblocks_bottom = MAP_HEIGHT - 1;
	      newblocks_left = 0;
	      newblocks_right = MAP_WIDTH - 1;
	      
	      strcpy(message, "WATCH OUT!");
	      
	      for (i = 0; i < num_servers; i++)
		redrawstatus(i);
	    }
	}
      
      
      /* Add new blocks if level is "over!" */
      
      if (leveltime <= 0 && ttoggle == 0 && winner == -1)
	{
	  addexplosion_atomic(newblocks_x, newblocks_y);
	  
	  map[newblocks_y][newblocks_x] = OBJ_BLOCK2;
	  
	  
	  if (newblocks_x + xms[newblocks_dir] >= newblocks_left &&
	      newblocks_x + xms[newblocks_dir] <= newblocks_right &&
	      newblocks_y + yms[newblocks_dir] >= newblocks_top &&
	      newblocks_y + yms[newblocks_dir] <= newblocks_bottom)
	    {
	      newblocks_x = newblocks_x + xms[newblocks_dir];
	      newblocks_y = newblocks_y + yms[newblocks_dir];
	    }
	  else
	    {
	      /* Cause inward motion: */
	      
	      if (newblocks_dir == 1)
		newblocks_top++;
	      else if (newblocks_dir == 2)
		newblocks_right--;
	      else if (newblocks_dir == 3)
		newblocks_bottom--;
	      else if (newblocks_dir == 0)
		newblocks_left++;
	      
	      
	      /* Cause spiral motion: */
	      
	      newblocks_dir = (newblocks_dir + 1) % 4;
	    }
	}
      
      
      /* Turn death on randomly: */
      
      if (((level > 10 && randnum(1000) < 1) || randnum(20000) < 1) &&
	  death.alive == 0 && winner == -1)
	{
	  playsound(SND_LAUGH);
	  
	  death.alive = 1;
	  death.x = randnum(MAP_WIDTH);
	  death.y = randnum(MAP_HEIGHT);
	  death.shape = OBJ_DEATH_EATING;
	  
	  /* Pick a sucker to chase (but not a dead one): */
	  
	  make_death_seek();

	  death.time = randnum(200) + 100;
	}
      
      
      /* Handle death: */
      
      if (death.alive && cmp_movetoggle == 0 && ttoggle == 0 && toggle == 0)
	{
	  /* No more, if someone won.. */
	  
	  if (winner != -1)
	    death.alive = 0;
	  
	  
	  /* Chase the person we're seeking (if we're not eating): */
	  
	  if (death.shape == OBJ_DEATH)
	    {
	      if (death.x < player[death.seek].x &&
		  empty(death.x + 1, death.y))
		death.x = death.x + 1;
	      else if (death.x > player[death.seek].x &&
		       empty(death.x - 1, death.y))
		death.x = death.x - 1;
	      else if (death.y < player[death.seek].y &&
		       empty(death.x, death.y + 1))
		death.y = death.y + 1;
	      else if (death.y > player[death.seek].y &&
		       empty(death.x, death.y - 1))
		death.y = death.y - 1;
	      else
		{
		  i = randnum(4);
		  
		  if (empty(death.x + xms[i], death.y + yms[i]))
		    {
		      death.x = death.x + xms[i];
		      death.y = death.y + yms[i];
		    }
		}
	      
	      
	      /* Eat bombs: */
	      
	      for (i = 0; i < MAX_BOMBS; i++)
		{
		  if (bomb[i].y == death.y && bomb[i].x == death.x &&
		      bomb[i].alive)
		    {
		      death.shape = OBJ_DEATH_EATING;
		      
		      kill_bomb(i);
		      
		      addexplosion_atomic(death.x, death.y);
		    }
		}


	      /* Eat players: */
	      
	      for (i = 0; i < MAX_PLAYERS; i++)
		{
		  if (player[i].y == death.y && player[i].x == death.x &&
		      player[i].alive)
		    {
		      death.shape = OBJ_DEATH_EATING;
		      
		      player[i].alive = 0;
		      initplayergoodies(i);

		      addexplosion_atomic(death.x, death.y);
		      
		      killed_one = 1;
		    }
		}
	      
	      
	      /* Eat upgrades: */
	      
	      if (map[death.y][death.x] >= FIRST_UPGRADE &&
		  map[death.y][death.x] < FIRST_UPGRADE + NUM_UPGRADES)
		{
		  death.shape = OBJ_DEATH_EATING;
		  
		  map[death.y][death.x] = OBJ_DIRT;
		}
	    }
	  else
	    death.shape = OBJ_DEATH;
	  
	  
	  /* If that guy died, choose another guy: */
	  
	  if (player[death.seek].alive == 0)
	    make_death_seek();
	  
	  
	  /* Count death down: */
	  
	  if (death.time <= 0)
	    death.alive = 0;
	}
      

      /* Do stuff is someone died: */
      
      if (killed_one && winner == -1)
	{
	  /* See how many people are alive: */
	  
	  num_alive = 0;
	  found = -1;
	  
	  for (i = 0; i < MAX_PLAYERS; i++)
	    {
	      if (player[i].alive)
		{
		  num_alive++;
		  found = i;
		}
	    }
	  
	  
	  /* Results of one person (or no people) alive: */
	  
	  if (num_alive == 0 || num_alive == 1)
	    {
	      if (num_alive == 1)
		{
		  winner = found;
		  player[found].score++;
		  sprintf(message, "Player %d won!", found + 1);
		}
	      else if (num_alive == 0)
		{
		  winner = -2;
		  strcpy(message, "A draw!");
		}
	      
	      
	      freetime = FREETIME;
	      
	      
	      /* Turn off all bombs: */
	      
	      for (i = 0; i < MAX_BOMBS; i++)
		bomb[i].alive = 0;
	      
	      
	      /* Explode all boxes and warps (may get some goodies!): */
	      
	      for (y = 0; y < MAP_HEIGHT; y++)
		{
		  for (x = 0; x < MAP_WIDTH; x++)
		    {
		      if (map[y][x] == OBJ_BOX || map[y][x] == OBJ_TNT ||
			  map[y][x] == OBJ_WARP)
			{
			  addexplosion_atomic(x, y);
			  
			  if (randnum(CHANCE_OF_UPGRADE) < 1)
			    map[y][x] = randnum(NUM_UPGRADES) + FIRST_UPGRADE;
			  else
			    map[y][x] = 0;
			}
		    }
		}
	      
	      z = randnum(6) + SND_LEVELEND1;
	      playsound(z);
	    }

	  
	  
	  /* Update everyone's status: */
	  
	  for (i = 0; i < num_servers; i++)
	    redrawstatus(i);
	}
      
      

      /* Keep framerate exact: */
      
      do
	{
	  gettimeofday(&now, NULL);
	}
      while (now.tv_sec == then.tv_sec &&
	     now.tv_usec < then.tv_usec + FRAMERATE);
    }
  while (done == False);
}


/* Program set-up (check usage, load data, etc.): */

void setup(int argc, char * argv[])
{
  FILE * fi;
  char temp[512], color[512], file[1024];
  int i, len, z, zc, cntl, want_length, sound_pos_tmp, which_sound;
  char c;
  struct timeval now, then;
  
  
  /* Get -sound switch: */
  
  use_sound = 0;
  
  if (argc >= 2)
    {
      if (strcmp(argv[argc - 1], "-sound") == 0 ||
	  strcmp(argv[argc - 1], "-s") == 0)
	{
	  use_sound = 1;
	  

	  /* Open audio device: */
	  
	  devaudio_fd = open("/dev/audio", O_RDWR);
	  
	  if (devaudio_fd == -1)
	    {
	      fprintf(stderr, "Can't open /dev/audio!");
	      exit(0);
	    }
	  
	  
	  /* Fork off into a sound manager process: */
	  
	  if (pipe(pipefds) < 0)
	    {
	      perror("pipe");
	      exit(1);
	    }
	  
	  pid = fork();
	  
	  
	  /* Sound manager routine: */
	  
	  if (pid == 0)
	    {
	      /* Figure out our file descriptors: */
	      
	      close(0);
	      dup(pipefds[0]);
	      close(pipefds[0]);
	      close(pipefds[1]);
	      
	      cntl = fcntl(0, F_GETFL, 0);
	      cntl = cntl | O_NONBLOCK;
	      fcntl(0, F_SETFL, cntl);
	      
	      len = 0;
	      
	      sound_position = 0;
	      
	      for (i = 0; i < SOUND_BUF; i++)
		sound_data[i] = 0;
	      
	      
	      /* Load game sounds: */
	      
	      for (i = 0; i < NUM_SOUNDS; i++)
		{
		  sprintf(file, "sounds/%s.au", sound_names[i]);
		  
		  fi = fopen(file, "r");
		  if (fi != NULL)
		    {
		      sound_pos_tmp = 0;
		      
		      do
			{
			  zc = fgetc(fi);
			  
			  if (zc != EOF)
			    {
			      au_file_data[i][sound_pos_tmp] =
				(unsigned char) zc;
			      
			      sound_pos_tmp++;
			    }
			}
		      while (zc != EOF);
		      
		      sound_len[i] = sound_pos_tmp;
		      
		      fclose(fi);
		    }
		  else
		    {
		      perror(file);
		      exit(0);
		    }
		}
	      

	      /* Load number sounds: */
	      
	      for (i = 0; i < NUM_NSOUNDS; i++)
		{
		  sprintf(file, "sounds/numbers/%s.au", nsound_names[i]);
		  
		  fi = fopen(file, "r");
		  if (fi != NULL)
		    {
		      sound_pos_tmp = 0;
		      
		      do
			{
			  zc = fgetc(fi);
			  
			  if (zc != EOF)
			    {
			      nau_file_data[i][sound_pos_tmp] =
				(unsigned char) zc;
			      
			      sound_pos_tmp++;
			    }
			}
		      while (zc != EOF);
		      
		      nsound_len[i] = sound_pos_tmp;
		      
		      fclose(fi);
		    }
		  else
		    {
		      perror(file);
		      exit(0);
		    }
		}
	      
	      printf("(Sound manager: running!)\n");
	      
	      
	      /* Loop: */
	      
	      do
		{
		  gettimeofday(&then, NULL);
		  
		  
		  /* Read from parent: */
		  
		  z = read(0, &c, 1);
		  
		  if (z > 0)
		    {
		      if (c != '\n')
			{
			  file[len] = c;
			  len++;
			}
		      else
			{
			  /* Copy sound into queue: */
			  
			  file[len] = '\0';
			  len = 0;
			  
			  if (file[0] != 'N')
			    {
			      which_sound = atoi(file);
			      
			      for (i = 0; i < sound_len[which_sound]; i++)
				{
				  sound_data[(i + sound_position) % SOUND_BUF]
				    = au_file_data[which_sound][i];
				}
			    }
			  else
			    {
			      which_sound = atoi(file + 1);
			      
			      for (i = 0; i < nsound_len[which_sound]; i++)
				{
				  sound_data[(i + sound_position) % SOUND_BUF]
				    = nau_file_data[which_sound][i];
				}
			    }
			}
		    }
		  
		  
		  /* Play sound: */
		  
		  want_length = 10;
		  
		  if (sound_data[sound_position] != 0)
		    {
		      write(devaudio_fd, &sound_data[sound_position],
			    want_length);
		      
		      for (sound_pos_tmp = 0; sound_pos_tmp < want_length;
			   sound_pos_tmp++)
			sound_data[sound_position + sound_pos_tmp] = 0;
		    }
		  
		  sound_position = (sound_position + want_length) %
		    SOUND_BUF;

		  do
		    {
		      gettimeofday(&now, NULL);
		    }
		  while (now.tv_sec == then.tv_sec &&
			 now.tv_usec < then.tv_usec + 500);
		}
	      while (z != 0);
	      
	      printf("(Sound manager: quitting!)\n");
	      
	      exit(0);
	    }
	  else if (pid == -1)
	    {
	      perror("fork");
	      exit(1);
	    }
	  else
	    {
	      close(pipefds[0]);
	      
	      sound_fs = fdopen(pipefds[1], "w");
	      
	      fprintf(stderr, "Connected to sound manager process\n");
	    }
	  
	  argc = argc - 1;
	}
    }
  
  
  /* Get "-L" level switch: */
  
  level = 1;
  
  if (argc >= 2)
    {
      if (strstr(argv[argc - 1], "-l") == argv[argc - 1] ||
	  strstr(argv[argc - 1], "-L") == argv[argc - 1])
	{
	  level = atoi(argv[argc - 1] + 2);
	  
	  if (level < 1 || level > 99)
	    level = 1;
	  
	  argc--;
	}
    } 
  
  
  if (argc == 2)
    {
      if (strcmp(argv[1], "-version") == 0 ||
	  strcmp(argv[1], "-v") == 0)
	{
	  /* Check for "-version": */
	  
	  printf("\nxbomber version 0.2\n\n");
	  
	  printf("by Bill Kendrick\n");
	  printf("kendrick@zippy.sonoma.edu\n");
	  printf("http://zippy.sonoma.edu/kendrick/nbs/unix/x/xbomber/\n\n");
	  exit(0);
	}
      else if (strcmp(argv[1], "-help") == 0 ||
	       strcmp(argv[1], "-h") == 0)
	{
	  /* Check for "-help": */
	  
	  fprintf(stderr, "\n");
	  fprintf(stderr, "Usage: %s ", argv[0]);
	  fprintf(stderr, "server1 [server2 [server3 [server4]]]\n");
	  fprintf(stderr, "       [-sound]\n");
	  fprintf(stderr, "Controls:\n");
	  fprintf(stderr, "  [Arrow] - Move\n");
	  fprintf(stderr, "  [Space] - Drop bomb\n");
	  fprintf(stderr, "  [Enter] - Activate radio bomb\n");
	  fprintf(stderr, "    [Q]   - Quit\n");
	  fprintf(stderr, "\n");
	  
	  exit(0);
	}
    }
  
  if (argc == 1)
    {
      if (getenv("DISPLAY") != NULL)
	{
	  strcpy(server[0], getenv("DISPLAY"));
	  num_servers = 1;
	}
      else
	{
	  fprintf(stderr, "Can't determine your display!\n");
	  fprintf(stderr, "Please specify it on the command line.\n");
	  exit(1);
	}
    }
  else if (argc <= MAX_PLAYERS + 1)
    {
      num_servers = argc - 1;
      
      for (i = 0; i < argc - 1; i++)
	strcpy(server[i], argv[i + 1]);
    }
  else
    {
      fprintf(stderr, "Too many servers specified.\n");
      fprintf(stderr, "Max. players: %d\n", MAX_PLAYERS);
      exit(1);
    }
  
  
  /* Init. dir -> xm/ym table: */
  
  xms[0] = 0;
  yms[0] = -1;

  xms[1] = 1;
  yms[1] = 0;

  xms[2] = 0;
  yms[2] = 1;

  xms[3] = -1;
  yms[3] = 0;
  
  xms[4] = 0;
  yms[4] = 0;
}


/* Setup the application: */

void Xsetup(int pln)
{
  int i, tempint, ret;
  char title[1024], file[128];
  int status, temp_depth;
  Visual * temp_visual = CopyFromParent;
  FILE * fi;
  XGCValues gcvalues;
  
  
  /* Connect to display: */
  
  fprintf(stderr, "Connecting to: %s\n", server[pln]);
  
  display[pln] = ConnectToServer(server[pln], &screen[pln], &root[pln]);
  if (display[pln] == NULL)
    {
      perror(server[pln]);
      exit(1);
    }
  
  
  /* Load font: */
  
  font[pln] = LoadFont(display[pln], "variable", "fixed");
  fh[pln] = FontHeight(font[pln]);
  
  
  /* Get our primitve colors: */
  
  black[pln] = BlackPixel(display[pln], screen[pln]);
  white[pln] = WhitePixel(display[pln], screen[pln]);
  
  
  /* Open window: */
  
  window[pln] = OpenWindow(display[pln], root[pln], 10, 10, WIDTH, HEIGHT,
			   CopyFromParent, black[pln],
			   (KeyPressMask | ExposureMask),
			   (Visual *)CopyFromParent);
  
  sprintf(title, "X-Bomber by Bill Kendrick");
  SetStandardHints(display[pln], window[pln], "XBomber", title, 10, 10,
		   WIDTH, HEIGHT);
  
  
  /* Set up visual: */
  
  has_color[pln] = 0;
  
  if (SetUpVisual(display[pln], screen[pln], &temp_visual, &temp_depth))
    {
      if (!SetUpColormap(display[pln], screen[pln], window[pln], temp_visual,
                         &colormap[pln]))
        {
          fprintf(stderr, "Could not create a colormap!\n");
        }
      else
        has_color[pln] = 1;
    }
  else
    {
      fprintf(stderr, "Could not find a PseudoColor visual!\n");
    }
  
  
  /* Make cursor: */
  
  ret = XReadBitmapFile(display[pln], window[pln], "bitmaps/cursor.xbm",
			&tempint, &tempint, &cursor_pixmap[pln],
			&tempint, &tempint);
  
  if (ret != BitmapSuccess)
    {
      perror("bitmaps/cursor.xbm");
      exit(1);
    }
  
  ret = XReadBitmapFile(display[pln], window[pln], "bitmaps/cursor-mask.xbm",
			&tempint, &tempint, &cursor_mask[pln],
			&tempint, &tempint);
  
  if (ret != BitmapSuccess)
    {
      perror("bitmaps/cursor-mask.xbm");
      exit(1);
    }
  
  XLookupColor(display[pln], colormap[pln], "black", &blackxcolor[pln],
	       &blackxcolor[pln]);
  XLookupColor(display[pln], colormap[pln], "white", &whitexcolor[pln],
	       &whitexcolor[pln]);
  
  cursor[pln] = XCreatePixmapCursor(display[pln], cursor_pixmap[pln],
				    cursor_mask[pln], 
				    &blackxcolor[pln], &whitexcolor[pln],
				    0, 0);
  
  XDefineCursor(display[pln], window[pln], cursor[pln]);
  
  
  /* Create Primitive GC's: */
  
  whitegc[pln] = CreateGC(display[pln], window[pln],
			  white[pln], black[pln]);

  blackgc[pln] = CreateGC(display[pln], window[pln],
			  black[pln], black[pln]);
  
  
  /* Create object GC's: */
  
  for (i = 0; i < NUM_OBJECTS; i++)
    colorgcs[pln][i] = CreateGC(display[pln], window[pln],
				MyAllocNamedColor(display[pln], colormap[pln],
						  object_colors[i], white[i],
						  has_color[pln]),
				MyAllocNamedColor(display[pln], colormap[pln],
						  object_colors[0], black[i],
						  has_color[pln]));
  
  for (i = 0; i < NUM_FLOATERS; i++)
    {
      floatergcs[pln][i] = CreateGC(display[pln], window[pln],
				    MyAllocNamedColor(display[pln],
						      colormap[pln],
						      floater_colors[i],
						      white[i],
						      has_color[pln]),
				    black[pln]);
      
      floatermaskgcs[pln][i] = CreateGC(display[pln], window[pln],
					MyAllocNamedColor(display[pln],
							  colormap[pln],
							  floater_colors[i],
							  white[i],
							  has_color[pln]),
					black[pln]);
      
      XSetFunction(display[pln], floatermaskgcs[pln][i],
		   GXorInverted);
      XSetFunction(display[pln], floatergcs[pln][i],
		   GXor);
    }
  
  yougc[pln] = CreateGC(display[pln], window[pln],
			MyAllocNamedColor(display[pln], colormap[pln],
					  YOU_COLOR, white[i],
					  has_color[pln]),
			MyAllocNamedColor(display[pln], colormap[pln],
					  object_colors[0], black[i],
					  has_color[pln]));
  
  
  /* Load object pixmaps: */
  
  for (i = 0; i < NUM_OBJECTS; i++)
    {
      sprintf(file, "bitmaps/%s.xbm", object_names[i]);
      
      ret = XReadBitmapFile(display[pln], window[pln], file, 
			    &tempint, &tempint, &object_pixmaps[pln][i],
			    &tempint, &tempint);
      
      if (ret != BitmapSuccess)
	{
	  perror(file);
	  exit(1);
	}
    }
  
  
  /* Load floater pixmaps and masks: */
  
  for (i = 0; i < NUM_FLOATERS; i++)
    { 
      sprintf(file, "bitmaps/%s.xbm", floater_names[i]);
      
      ret = XReadBitmapFile(display[pln], window[pln], file,
			    &tempint, &tempint, &floater_pixmaps[pln][i],
			    &tempint, &tempint);
      
      if (ret != BitmapSuccess)
	{
	  perror(file);
	  exit(1);
	}

      sprintf(file, "bitmaps/%s-mask.xbm", floater_names[i]);
      
      ret = XReadBitmapFile(display[pln], window[pln], file,
			    &tempint, &tempint, &floater_masks[pln][i],
			    &tempint, &tempint);
      
      if (ret != BitmapSuccess)
	{
	  perror(file);
	  exit(1);
	}
   }
  
  
  /* Put us up!: */
  
  XMapWindow(display[pln], window[pln]);
  XMapRaised(display[pln], window[pln]);
  XFlush(display[pln]);
}


/* Allocate a color (or white): */

unsigned long MyAllocNamedColor(Display *display, Colormap colormap,
                                char* colorname, unsigned long default_color,
                                int has_color)
{
  if (has_color == 0)
    return(default_color);
  else
    return(AllocNamedColor(display, colormap, colorname, default_color));
}


/* Play a sound (if we can): */

void playsound(int whichfile)
{
  if (use_sound == 1)
    {
      fprintf(sound_fs, "%d\n", whichfile);
      fflush(sound_fs);
    }
}


/* Play a sound (if we can): */

void playnsound(int whichfile)
{
  if (use_sound == 1)
    {
      fprintf(sound_fs, "N%d\n", whichfile);
      fflush(sound_fs);
    }
}


/* (Re)draw a tile at some point on the player's window: */

void drawblock(int pln, int x, int y, int c)
{
  if (player[pln].alive && player[pln].x == x && player[pln].y == y)
    XCopyPlane(display[pln], object_pixmaps[pln][c], window[pln],
	       yougc[pln], 0, 0, 32, 32, x * 32, y * 32, 1);
  else
    XCopyPlane(display[pln], object_pixmaps[pln][c], window[pln],
	       colorgcs[pln][c], 0, 0, 32, 32, x * 32, y * 32, 1);
  
  XFlush(display[pln]);
}


/* Make an explosion (as much of it as we can, depending on where walls
   are: */

void makeexplosion(int x, int y, int size)
{
  int done, i, ii;
  
  playsound(SND_EXPLODE);
  
  done = 0;
  for (i = x; i >= x - size + 1 && done == 0; i--)
    {
      ii = i;
      if (ii < 0)
	ii = ii + MAP_WIDTH;
      
      done = addexplosion(ii, y);
    }
  
  done = 0;
  for (i = x + 1; i < x + size && done == 0; i++)
    {
      ii = i;
      if (ii > MAP_WIDTH - 1)
	ii = ii - MAP_WIDTH;
      
      done = addexplosion(ii, y);
    }

  done = 0;
  for (i = y; i >= y - size + 1 && done == 0; i--)
    {
      ii = i;
      if (ii < 0)
	ii = ii + MAP_HEIGHT;
      
      done = addexplosion(x, ii);
    }
  
  done = 0;
  for (i = y + 1; i < y + size && done == 0; i++)
    {
      ii = i;
      if (ii > MAP_HEIGHT - 1)
	ii = ii - MAP_HEIGHT;
      
      done = addexplosion(x, ii);
    }
}


/* Add one explosion to the screen (if it can).  Returns 1 when an
   obstacle is found: */

int addexplosion(int x, int y)
{
  int ret, i;
  
  
  ret = 0;
  
  if (map[y][x] == OBJ_BLOCK || map[y][x] == OBJ_BLOCK2)
    {
      /* Stop if we hit a block (they're indestructable): */
      
      ret = 1;
    }
  else
    {
      /* Add an explosion at this spot: */
      
      addexplosion_atomic(x, y);
      
      if (map[y][x] == OBJ_BOX)
	{
	  /* If it's a box, stop, and turn the box into an upgrade (maybe): */
	  
	  ret = 1;
	  
	  if (randnum(CHANCE_OF_UPGRADE) < 1)
	    map[y][x] = randnum(NUM_UPGRADES) + FIRST_UPGRADE;
	  else
	    map[y][x] = OBJ_DIRT;
	}
      else if (map[y][x] == OBJ_TNT)
	{
	  /* If it's TNT, stop, turn it into an upgrade (maybe) and
	     explode some more!: */
	  
	  ret = 1;
	  
	  map[y][x] = OBJ_DIRT;
	  
	  makeexplosion(x, y, 2);
	  
	  if (randnum(CHANCE_OF_UPGRADE) < 1)
	    map[y][x] = randnum(NUM_UPGRADES) + FIRST_UPGRADE;
	  else
	    map[y][x] = OBJ_DIRT;
	}
      else
	{
	  /* Other things (ie, upgrades, warps, etc.) turn into dirt: */
	  
	  map[y][x] = OBJ_DIRT;
	}
    }
  
  
  /* Blow up any bombs we touched: */
  
  for (i = 0; i < MAX_BOMBS; i++)
    {
      if (bomb[i].alive)
	{
	  if (bomb[i].x == x && bomb[i].y == y)
	    {
	      kill_bomb(i);
	      makeexplosion(bomb[i].x, bomb[i].y, bomb[i].size);
	      ret = 1;
	    }
	}
    }
  
  return (ret);
}


/* Returns whether a player's move is valid (not off the screen, and
   not into some object) */

int moveok(int pln, int xm, int ym)
{
  int c, x, y, ok, i, found, xx, yy;
  
  
  ok = 1;
  
  x = player[pln].x + xm;
  y = player[pln].y + ym;
  
  if (x < 0)
    x = MAP_WIDTH -1;
  if (x > MAP_WIDTH - 1)
    x = 0;
  if (y < 0)
    y = MAP_HEIGHT - 1;
  if (y > MAP_HEIGHT - 1)
    y = 0;
  
  c = seenmap[y][x];
  
  if (c == OBJ_BLOCK || c == OBJ_BLOCK2 || c == OBJ_MAN_UP ||
      c == OBJ_MAN_DOWN || c == OBJ_MAN_LEFT || c == OBJ_MAN_RIGHT)
    {
      /* You can't walk through indestructable blocks or through
	 other players: */
      
      ok = 0;
    }
  else if (c == OBJ_BOX || c == OBJ_TNT)
    {
      /* You can't walk through boxes unless you have the pusher: */
      
      ok = 0;
      
      if (player[pln].pusher == 1)
	{
	  /* See if there's room to shove the block out of the way: */
	  
	  xx = x + xm;

	  if (xx < 0)
	    xx = xx + MAP_WIDTH;
	  if (xx > MAP_WIDTH - 1)
	    xx = xx - MAP_WIDTH;

	  yy = y + ym;
	  
	  if (yy < 0)
	    yy = yy + MAP_HEIGHT;
	  if (yy > MAP_HEIGHT - 1)
	    yy = yy - MAP_HEIGHT;


	  if (seenmap[yy][xx] == OBJ_DIRT)
	    {
	      /* Shove the block: */
	      
	      map[yy][xx] = c;
	      map[y][x] = OBJ_DIRT;
	      
	      ok = 1;
	      
	      playsound(SND_PUSHER);
	    }
	}
    }
  else if (c == OBJ_BOMB || c == OBJ_RADIO_BOMB)
    {
      /* You can't walk through bombs unless you have the kicker: */
      
      ok = 0;
      
      if (player[pln].kicker == 1)
	{
	  found = -1;
	  
	  for (i = 0; i < MAX_BOMBS; i++)
	    if (bomb[i].x == x && bomb[i].y == y)
	      found = i;
	  
	  if (found != -1)
	    {
	      bomb[found].moving = 1;
	      bomb[found].xm = xm;
	      bomb[found].ym = ym;
	      
	      ok = 1;
	      
	      playsound(SND_KICKER);
	    }
	}
    }
  
  return(ok);
}


/* Updates the status line at the bottom: */

void redrawstatus(int pln)
{
  int i, which;
  char text[128];
  
  
  if (pln < num_servers)
    {
      /* Show who's alive: */
      
      for (i = 0; i < MAX_PLAYERS; i++)
	{
	  /* Draw white guy for you, tan guy for other players, or explosion
	     for dead players: */
	  
	  if (player[i].alive)
	    {
	      if (i == pln)
		XCopyPlane(display[pln], object_pixmaps[pln][OBJ_MAN_DOWN],
			   window[pln], yougc[pln],
			   0, 0, 32, 32, i * 32, MAP_HEIGHT * 32, 1);
	      else
		XCopyPlane(display[pln], object_pixmaps[pln][OBJ_MAN_DOWN],
			   window[pln], colorgcs[pln][OBJ_MAN_DOWN],
			   0, 0, 32, 32, i * 32, MAP_HEIGHT * 32, 1);
	    }
	  else
	    XCopyPlane(display[pln], object_pixmaps[pln][OBJ_EXPLOSION1],
		       window[pln], colorgcs[pln][OBJ_EXPLOSION1],
		       0, 0, 32, 32, i * 32, MAP_HEIGHT * 32, 1);
	  
	  
	  /* Draw their score on their dude icon: */
	  
	  sprintf(text, "%d", player[i].score);
	  
	  drawtext(display[pln], window[pln], whitegc[pln], 
		   i * 32, (MAP_HEIGHT + 1) * 32, text);
	}
      
      
      /* Show what type of bomb(s) you have and how many: */
      
      if (player[pln].alive)
	{
	  if (player[pln].radiobomb == 0)
	    {
	      which = OBJ_BOMB;
	      sprintf(text, "x%d ", player[pln].bombs_max);
	    }
	  else if (player[pln].radiobomb == 1)
	    {
	      which = OBJ_RADIO_BOMB;
	      strcpy(text, "   ");
	    }
	}
      else
	{
	  which = OBJ_DIRT;
	  strcpy(text, "   ");
	}
      
      XCopyPlane(display[pln], object_pixmaps[pln][which],
		 window[pln], colorgcs[pln][which],
		 0, 0, 32, 32, (MAX_PLAYERS + 1) * 32, MAP_HEIGHT * 32, 1);
      
      drawtext(display[pln], window[pln], whitegc[pln], 
	       (MAX_PLAYERS + 2) * 32, (MAP_HEIGHT + 1) * 32, text);
      
      
      /* Show how big their bombs are: */
      
      if (player[pln].alive)
	{
	  if (player[pln].fire_size < MAX_FIRE_SIZE)
	    which = OBJ_MOREFIRE;
	  else
	    which = OBJ_TOTALFIRE;
	  
	  sprintf(text, "x%d ", player[pln].fire_size);
	}
      else
	{
	  which = OBJ_DIRT;
	  strcpy(text, "   ");
	}
      
      XCopyPlane(display[pln], object_pixmaps[pln][which],
		 window[pln], colorgcs[pln][which],
		 0, 0, 32, 32, (MAX_PLAYERS + 3) * 32, MAP_HEIGHT * 32, 1);
      
      drawtext(display[pln], window[pln], whitegc[pln], 
	       (MAX_PLAYERS + 4) * 32, (MAP_HEIGHT + 1) * 32, text);
      
      
      /* Show pusher, if you have it: */
      
      if (player[pln].pusher == 1 && player[pln].alive)
	which = OBJ_PUSHER;
      else
	which = OBJ_DIRT;
      
      XCopyPlane(display[pln], object_pixmaps[pln][which],
		 window[pln], colorgcs[pln][which],
		 0, 0, 32, 32, (MAX_PLAYERS + 5) * 32, MAP_HEIGHT * 32, 1);
      
      
      /* Show kicker, if you have it: */
      
      if (player[pln].kicker == 1 && player[pln].alive)
	which = OBJ_KICKER;
      else
	which = OBJ_DIRT;
      
      XCopyPlane(display[pln], object_pixmaps[pln][which],
		 window[pln], colorgcs[pln][which],
		 0, 0, 32, 32, (MAX_PLAYERS + 7) * 32, MAP_HEIGHT * 32, 1);
      
      
      /* Draw current status message: */
      
      drawtext(display[pln], window[pln], whitegc[pln],
	       (MAX_PLAYERS + 9) * 32, (MAP_HEIGHT + 1) * 32 - fh[pln] / 2, 
      ".................................");
      drawtext(display[pln], window[pln], whitegc[pln],
	       (MAX_PLAYERS + 9) * 32, (MAP_HEIGHT + 1) * 32 - fh[pln] / 2, 
      message);
    }
}


/* Drop a bomb if we can: */

int dropbomb(int pln)
{
  int i, found, ret;
  
  if (player[pln].bombs_out < player[pln].bombs_max)
    {
      player[pln].bombs_out++;
      
      found = -1;
      
      for (i = 0; i < MAX_BOMBS; i++)
	if (bomb[i].alive == 0)
	  found = i;
      
      if (found != -1)
	{
	  bomb[found].alive = 1;
	  bomb[found].x = player[pln].x;
	  bomb[found].y = player[pln].y;
	  if (player[pln].radiobomb == 0)
	    bomb[found].shape = OBJ_BOMB;
	  else
	    bomb[found].shape = OBJ_RADIO_BOMB;
	  bomb[found].owner = pln;
	  bomb[found].time = BOMB_COUNTDOWN - (level / 10) * 10;
	  bomb[found].size = player[pln].fire_size;
	  bomb[found].moving = 0;
	  
	  player[pln].radio_detonate = 0;
	}
      
      ret = 1;
    }
  else
    ret = 0;
  
  return(ret);
}


void initlevel(void)
{
  int i, x, y, pln, c;
  FILE * fi;
  char filename[128];
  
  
  /* Init bombs: */
  
  for (i = 0; i < MAX_BOMBS; i++)
    bomb[i].alive = 0;

  
  /* Init explosions: */
  
  for (i = 0; i < MAX_EXPLOSIONS; i++)
    explosion[i].alive = 0;
  
  
  /* Init map: */
  
  /* (Clear) */
  for (y = 0; y < MAP_HEIGHT; y++)
    for (x = 0; x < MAP_WIDTH; x++)
      map[y][x] = OBJ_DIRT;
  
  if (lev_usetnt[(level - 1) % 10])
    {
      /* (Random TNT) */
      for (i = 0; i < HOW_MANY_TNTS; i++)
	map[randnum(MAP_HEIGHT)][randnum(MAP_WIDTH)] = OBJ_TNT;
    }
  
  if (lev_usebox[(level - 1) % 10])
    {
      /* (Random boxes) */
      for (i = 0; i < HOW_MANY_BOXES; i++)
	map[randnum(MAP_HEIGHT)][randnum(MAP_WIDTH)] = OBJ_BOX;
    }
  
  if (lev_usewarp[(level - 1) % 10])
    {
      /* (Random warp spots) */
      for (i = 0; i < HOW_MANY_WARPS; i++)
	map[randnum(MAP_HEIGHT)][randnum(MAP_WIDTH)] = OBJ_WARP;
    }
  
  sprintf(filename, "levels/level%.2d.dat", level);
  
  fi = fopen(filename, "r");
  if (fi == NULL)
    {
      perror(filename);
      exit(1);
    }
  
  for (y = 0; y < MAP_HEIGHT; y++)
    {
      for (x = 0; x < MAP_WIDTH; x++)
	{
	  c = fgetc(fi);
	  
	  if (c == '#')
	    map[y][x] = OBJ_BLOCK;
	  else if (c == '@')
	    map[y][x] = OBJ_BLOCK2;
	}
      fgetc(fi);
    }
  
  fclose(fi);
  
  
  /* (Clear spaces next to players, to keep them from being trapped) */
  map[1][1] = OBJ_DIRT;
  map[2][1] = OBJ_DIRT;
  map[1][2] = OBJ_DIRT;
  
  map[MAP_WIDTH - 2][1] = OBJ_DIRT;
  map[MAP_WIDTH - 3][1] = OBJ_DIRT;
  map[MAP_WIDTH - 2][2] = OBJ_DIRT;
  
  map[MAP_WIDTH - 2][MAP_HEIGHT - 2] = OBJ_DIRT;
  map[MAP_WIDTH - 3][MAP_HEIGHT - 2] = OBJ_DIRT;
  map[MAP_WIDTH - 2][MAP_HEIGHT - 3] = OBJ_DIRT;
  
  map[1][MAP_HEIGHT - 2] = OBJ_DIRT;
  map[2][MAP_HEIGHT - 2] = OBJ_DIRT;
  map[1][MAP_HEIGHT - 3] = OBJ_DIRT;
  
  
  /* Copy entire map onto main seen map: */
  
  for (y = 0; y < MAP_HEIGHT; y++)
    for (x = 0; x < MAP_WIDTH; x++)
      seenmap[y][x] = map[y][x];
  
  
  /* Clear each player's seen map: */
  
  for (pln = 0; pln < num_servers; pln++)
    for (y = 0; y < MAP_HEIGHT; y++)
      for (x = 0; x < MAP_WIDTH; x++)
	player[pln].seenmap[y][x] = -1;
  
  
  /* Init player stuff: */
  
  for (i = 0; i < MAX_PLAYERS; i++)
    {
      if (i == 0)
	{
	  player[i].x = 1;
	  player[i].y = 1;
	}
      else if (i == 1)
	{
	  player[i].x = MAP_WIDTH - 2;
	  player[i].y = MAP_HEIGHT - 2;
	}
      else if (i == 2)
	{
	  player[i].x = MAP_WIDTH - 2;
	  player[i].y = 1;
	}
      else if (i == 3)
	{
	  player[i].x = 1;
	  player[i].y = MAP_HEIGHT - 2;
	}

      player[i].alive = 1;
      player[i].bombs_out = 0;
    }
  
  
  /* Init level text: */
  
  sprintf(message, "LEVEL: %d", level);
  winner = -1;
  
  
  /* Init level countdown time: */
  
  leveltime = 8000;
  
  
  /* Init death: */
  
  death.alive = 0;
  
  
  /* Init floaters: */
  
  for (i = 0; i < MAX_FLOATERS; i++)
    {
      floaters[i].alive = 0;
    }
  
  
  /* Play sound: */
  
  playnsound(NSND_LEVEL);
  if (use_sound)
    sleep(1);
  
  if (level < 16)
    playnsound(level);
  else
    {
      if (level >= 16 && level < 20)
	{
	  playnsound(level - 10);
	  if (use_sound)
	    sleep(1);
	  playnsound(NSND_TEEN);
	}
      else
	{
	  if (level >= 20 && level < 30)
	    playnsound(NSND_20);
	  else if (level >= 30 && level < 40)
	    playnsound(NSND_30);
	  else if (level >= 40 && level < 50)
	    playnsound(NSND_40);
	  else if (level >= 50 && level < 60)
	    playnsound(NSND_50);
	  else if (level >= 60 && level < 70)
	    playnsound(NSND_60);
	  else if (level >= 70 && level < 80)
	    playnsound(NSND_70);
	  else if (level >= 80 && level < 90)
	    playnsound(NSND_80);
	  else if (level >= 90)
	    playnsound(NSND_90);

	  if ((level % 10) != 0)
	    {
	      if (use_sound)
		sleep(1);
	      playnsound(level % 10);
	    }
	}
    }
}


/* Init. player goodies (beginning of game, and when you die): */

void initplayergoodies(int i)
{
  player[i].bombs_max = 1;
  player[i].fire_size = 2;
  player[i].radiobomb = 0;
  player[i].pusher = 0;
  player[i].kicker = 0;
  
  player[i].cmp_time = 0;
  player[i].cmp_dir = 5;
  player[i].shape = OBJ_MAN_DOWN;
}


/* Actually looks for an explosion slot and places an explosion at x,y: */

void addexplosion_atomic(int x, int y)
{
  int i, found;
  
  
  found = -1;
  
  for (i = 0; i < MAX_EXPLOSIONS; i++)
    {
      if (explosion[i].alive == 0)
	found = i;
    }
  
  if (found != -1)
    {
      explosion[found].alive = 1;
      explosion[found].time = 16;
      explosion[found].x = x;
      explosion[found].y = y;
    }
}


/* Control a computer player: */

void control_computer(int pln)
{
  int pick_new_dir, c, count, i, old_dir;
  
  
  pick_new_dir = 0;
  
  if (cmp_movetoggle == 0)
    {
      if (player[pln].cmp_time >= 0)
	{
	  /* Move (if we still can): */
	  
	  if (moveok(pln, xms[player[pln].cmp_dir], yms[player[pln].cmp_dir]))
	    {
	      player[pln].x = player[pln].x + xms[player[pln].cmp_dir];
	      
	      if (player[pln].x < 0)
		player[pln].x = MAP_WIDTH - 1;
	      if (player[pln].x > MAP_WIDTH - 1)
		player[pln].x = 0;	
	      
	      player[pln].y = player[pln].y + yms[player[pln].cmp_dir];

	      if (player[pln].y < 0)
		player[pln].y = MAP_HEIGHT - 1;
	      if (player[pln].y > MAP_HEIGHT - 1)
		player[pln].y = 0;
	    }
	  else
	    {
	      if (player[pln].x > 0 && player[pln].x < MAP_WIDTH - 1 &&
		  player[pln].y > 0 && player[pln].y < MAP_HEIGHT - 1)
		{
		  c = (map[player[pln].y + yms[player[pln].cmp_dir]]
		       [player[pln].x + xms[player[pln].cmp_dir]]);
		  
		  if (((c == OBJ_BOX || c == OBJ_TNT) && randnum(10) < 3) ||
		      randnum(50) < 1)
		    {
		      if (winner == -1)
			dropbomb(pln);
		    }
		}
	    }
	  
	  player[pln].cmp_time--;
	}
      else
	{
	  pick_new_dir = 1;
	}
      
      
      /* Check for bombs and explosions!: */
      
      for (i = 0; i < MAX_BOMBS; i++)
	{
	  if (bomb[i].alive && (bomb[i].x == player[pln].x ||
				bomb[i].y == player[pln].y))
	    pick_new_dir = 2;
	}
      
      for (i = 0; i < MAX_EXPLOSIONS; i++)
	{
	  if (explosion[i].alive && (explosion[i].x == player[pln].x ||
				     explosion[i].y == player[pln].y))
	    pick_new_dir = 2;
	}
      
      
      /* Pick a new direction(?): */
      
      if (pick_new_dir != 0)
	{
	  count = 0;
	  
	  if (pick_new_dir == 1)
	    old_dir = (player[pln].cmp_dir + 2) % 4;
	  else
	    old_dir = -1;
	  
	  
	  /* Get the hell away from bombs right next to you! */
	  
	  if (map[player[pln].y - 1][player[pln].x] == OBJ_BOMB ||
	      map[player[pln].y - 1][player[pln].x] == OBJ_RADIO_BOMB)
	    {
	      player[pln].cmp_dir = 2;
	    }
	  else if (map[player[pln].y + 1][player[pln].x] == OBJ_BOMB ||
	      map[player[pln].y + 1][player[pln].x] == OBJ_RADIO_BOMB)
	    {
	      player[pln].cmp_dir = 0;
	    }
	  else if (map[player[pln].y][player[pln].x - 1] == OBJ_BOMB ||
	      map[player[pln].y][player[pln].x - 1] == OBJ_RADIO_BOMB)
	    {
	      player[pln].cmp_dir = 1;
	    }
	  else if (map[player[pln].y][player[pln].x + 1] == OBJ_BOMB ||
	      map[player[pln].y][player[pln].x + 1] == OBJ_RADIO_BOMB)
	    {
	      player[pln].cmp_dir = 3;
	    }
	  else
	    {
	      /* Look for a random direction: */
	      
	      do
		{
		  player[pln].cmp_dir = randnum(6 - pick_new_dir);
		  count++;
		}
	      while (moveok(pln, xms[player[pln].cmp_dir],
			    yms[player[pln].cmp_dir]) == 0 && count < 20 &&
		     player[pln].cmp_dir == old_dir);
	    }
	  
	  
	  /* Point the dude in that direction: */
	  
	  if (player[pln].cmp_dir == 0)
	    player[pln].shape = OBJ_MAN_UP;
	  else if (player[pln].cmp_dir == 1)
	    player[pln].shape = OBJ_MAN_RIGHT;
	  else if (player[pln].cmp_dir == 2)
	    player[pln].shape = OBJ_MAN_DOWN;
	  else if (player[pln].cmp_dir == 3)
	    player[pln].shape = OBJ_MAN_LEFT;
	  
	  
	  /* Make them walk for a while: */
	  
	  player[pln].cmp_time = randnum(30) + 5;
	}
    }
}


/* Make death find another player to chase, or turn death off if everyone's
   dead: */

void make_death_seek(void)
{
  int count;
  
  
  death.seek = -1;
  count = 0;
  
  do
    {
      count = count + 1;
      
      death.seek = randnum(MAX_PLAYERS);
    }
  while (player[death.seek].alive == 0 && count < 10);
  
  
  if (death.seek == -1)
    death.alive = 0;
}


/* 1 if a spot is empty, 0 if not: */

int empty(int x, int y)
{
  if (y < 0 || y > MAP_HEIGHT - 1 || x < 0 || x > MAP_WIDTH - 1)
    return(0);
  
  if (map[y][x] == OBJ_BLOCK || map[y][x] == OBJ_BLOCK2 ||
      map[y][x] == OBJ_BOX || map[y][x] == OBJ_TNT)
    return(0);
  else
    return(1);
}


/* Kill a bomb and give it back to the player: */

void kill_bomb(int i)
{
  bomb[i].alive = 0;
  player[bomb[i].owner].bombs_out--;
}


/* Returns whether a floater is "touching" some spot: */

int floater_touching(int i, int x, int y)
{
  if (floaters[i].x / 32 >= x && floaters[i].x / 32 <= x + 1 &&
      floaters[i].y / 32 >= y && floaters[i].y / 32 <= y + 2)
    return 1;
  else
    return 0;
}
