/*
 *
 * $Id: lp_protocols.cxx,v 1.1 1995/04/07 14:32:14 beust Exp beust $
 */

#include <assert.h>
#include "lp_lib.hxx"

///////////////////////////////////////////////////////////////////////////

#define WORD64COUNT(_bytes) (((unsigned int) ((_bytes) + 7)) >> 3)

//
// Necessary to set the userData field in protocolSetupProc
//
//static CProtocolOut *LastProtocolOut;
static CProtocolIn *LastProtocolIn;

static void
_outProcessMessage(IceConn iceConn, IcePointer userData, int opCode,
			   unsigned long length, Bool swap,
			   IceReplyWaitInfo *replyWait, Bool *replyReadyRet)
{
   cout << "_outProcessMessage: processing" << endl;
}

static void
_outWatchProc(IceConn iceConn, IcePointer clientData,
	      Bool opening, IcePointer *watchData)
{
//   cout << "watch proc called with opening = " << opening << endl;
}

CProtocolOut::CProtocolOut(CDescProtocol &protDesc,
			   const String recipient,
			   const String pName,
			   const String pVendor,
			   const String pRelease
   ) : _protDesc(protDesc)
{
   IcePoVersionRec callbackList[] = {
      { 1, 0, _outProcessMessage}
   };
   char error[128];

   _recipient = recipient;

   //
   // Default coding method
   //
   _coding = new CCoding;

   _iceProtocol = IceRegisterForProtocolSetup(pName,
					   pVendor,
					   pRelease,
				      1, callbackList,
				      0,   // auth count
				      0,   // auth names
				      0,   //auth callbacks
				      0    // err procedure
          );
   _iceConn = IceOpenConnection((char *) recipient,
			       0,  // don't create a new connection
			       False,  // no authentication
			       -1,    // major opcode
			       128, error    // size of error buffer
             );

   IceAddConnectionWatch(_outWatchProc, this);

   if (0 == _iceConn) {
      cerr << "Error in IceOpenConnection() : " << error << endl;
   }
   else {
      int majVer, minVer, status;
      char *vendor, *release;
      status = IceProtocolSetup(_iceConn,
                                _iceProtocol,
                                this,       // client data
				False,    // no auth needed
                                & majVer,
                                & minVer,
                                & vendor,
                                & release,
                                128, error);

      if (IceProtocolSetupFailure == status) {
         cerr << "*** "
	      << __FILE__ << __LINE__
              << " protocol failure: "
	      << status << error
	      << endl;
      }
      else if (IceProtocolSetupIOError == status) {
         fprintf(stderr, "*** %s,%d: %setupIOError: %d '%s'\n",
		 __FILE__, __LINE__,
		 status, error);
      }
      else {
      }
      
   }
}

void
CProtocolOut::setCoding(CCoding *coding)
{
   _coding = coding;
}

void
CProtocolOut::send(CMessage *msg)
{
   size_t codeSize;
   iceMsg *iMsg;
   char *extra, *fillBuffer;

   IceGetHeaderExtra(_iceConn, _iceProtocol, msg -> getOpCode(),
		     sizeof(iceMsg), 0,
		     iceMsg, iMsg, extra);
   fillBuffer = extra;

   codeSize = _coding -> encode(msg, fillBuffer);

   iMsg -> length = WORD64COUNT(codeSize);
   IceWriteData(_iceConn, (iMsg -> length << 3), extra);

   IceFlush(_iceConn);
}

int
CProtocolOut::getFileNumber(void) const
{
   return 0;
}


IceConn
CProtocolOut::getIceConn(void) const
{
   return _iceConn;
}

int
CProtocolOut::getProtocol(void) const
{
   return _iceProtocol;
}


///////////////////////////////////////////////////////////////////////////

static void
_inProcessMessage(IceConn iceConn, void *clientData,
		   int opCode, unsigned long length, Bool swap)
{
   assert(clientData);
   CProtocolIn *cpi = (CProtocolIn *) clientData;
   CDescProtocol &protDesc = cpi -> getProtocolDescription();
   CDescPacket **messages = protDesc.getMessages();
   CDescPacket *protocolMessage = messages[opCode];

   iceMsg *header;
   char *data, *fillBuffer;

   assert(iceConn);
   IceReadCompleteMessage(iceConn, sizeof(iceMsg),
			  iceMsg, header, data);

   fillBuffer = data;

   CDescPacket *msg = cpi -> decode(protocolMessage, fillBuffer);

   msg -> handle();
}

static Bool
hostBasedAuth(char *host)
{
   return True;
}

/* callback of type IceIOErrorProc */
static void
errorHandler(IceConn conn)
{
//   cout << "errorHandler: disconnection detected" << endl;
}

/*
** This callback will be called before the ProtocolReply is sent.
** It is only used to set the userdata field that will be passed
** on to function _processMessage.
*/
static Status
inProtocolSetupProc(IceConn iceConn, int majVersion, int minVersion,
		  char *vendor, char *release, IcePointer *userData,
		  char **failureReason)
{

   *userData = (IcePointer) LastProtocolIn;
   return 1;
}

//
// Type XtInputCallbackProc
//
void
inFileDescriptorReady(XtPointer clientData, int *source, XtInputId *id)
{
   CProtocolIn *cpi = (CProtocolIn *) clientData;
   fd_set fds;
   FD_ZERO(& fds);
   FD_SET(*source, & fds);
   cpi -> dispatch(1, & fds); // 1 file descriptor ready
}

CProtocolIn::CProtocolIn(CDescProtocol &protDesc,
			 const String pName,
			 const String pVendor,
			 const String pRelease)
   : _protDesc(protDesc),
     _openConnectionsCount(0), _newConn(0),
     _appContext(0)
{
   Status status;
   IcePaVersionRec callbackList[] = {
      { 1, 0, (IcePaProcessMsgProc) _inProcessMessage }
   };
   char error[128];
   status = IceListenForConnections(& _objCount,
				    & _obj,
				    128,
				    error);
   assert(2 == _objCount);
   LastProtocolIn = this;

   _iceProtocol = IceRegisterForProtocolReply(pName, pVendor, pRelease,
			       1, callbackList,	   /* callbacks */
			       0,            /* # of auth methods supported */
			       NULL,  /* auth names */
			       NULL,		   /* auth callbacks */
			       hostBasedAuth,     /* host-based auth */
			       inProtocolSetupProc,   /* cb before Protocol Reply is sent */
			       NULL,   /* cb after Protocol Reply is sent */
			       NULL    /* IO error handler */
   );

   assert(_iceProtocol);

   //
   // Assign a host-based authentication on each listen object
   // and add each listen object for XtMainLoop if required
   //
   for (int i = 0; i < _objCount; i++) {
      IceSetHostBasedAuthProc(_obj[i], hostBasedAuth);
   }

   //
   // Default coding method
   //
   _coding = new CCoding;

   //
   // Intercept errors
   //
   IceSetIOErrorHandler(errorHandler);

}

void
CProtocolIn::wedgeIntoXtMainLoop(XtAppContext appContext)
{
   _appContext = appContext;

   for (int i = 0; i < _objCount; i++) {

      int fd = IceGetListenConnectionNumber(_obj[i]);

      _inputIds[fd] = XtAppAddInput(_appContext,
		    fd,
		    (XtPointer) XtInputReadMask,
		    inFileDescriptorReady,
		    this);
   }

}

String
CProtocolIn::getConnectionStrings(void) const
{
   String result;
   size_t resultSize = 0;

   for (int i = 0; i < _objCount; i++) {
      resultSize += strlen(IceGetListenConnectionString(_obj[i]));
   }

   result = new char [resultSize + 2];
   strcpy(result, "");
   for (i = 0; i < _objCount; i++) {
      if (i != 0) strcat(result, ",");
      strcat(result, IceGetListenConnectionString(_obj[i]));
   }

   return result;
   
}

void
CProtocolIn::setCoding(CCoding *coding)
{
   _coding = coding;
}

CDescProtocol &
CProtocolIn::getProtocolDescription(void) const
{
    return _protDesc;
}

void
CProtocolIn::addConnection(IceConn iceConn)
{
   assert(_openConnectionsCount + 1 < MAX_CONNECTIONS);

   _openConnections[_openConnectionsCount++] = iceConn;
   if (0 != _appContext)

      XtAppAddInput(_appContext,
		    IceConnectionNumber(iceConn),
		    (XtPointer) XtInputReadMask,
		    inFileDescriptorReady,
		    this);
}

void
CProtocolIn::removeConnection(IceConn iceConn)
{
   XtRemoveInput(_inputIds[IceConnectionNumber(iceConn)]);
}

CDescPacket *
CProtocolIn::decode(CDescPacket *msg, const char *inBuffer)
{
   return _coding -> decode(msg, inBuffer);
}

void
CProtocolIn::dispatch(int ready, fd_set *fds)
{
   Status status;

   //
   // Check all our listenObj
   //
   for (int i = 0; i < _objCount; i++) {
      int thisFd = IceGetListenConnectionNumber(_obj[i]);
      if (ready > 0 && FD_ISSET(thisFd, fds)) {
	 IceAcceptStatus status;
	 ready--;
	 _newConn = IceAcceptConnection(_obj[i], & status);
	 if (0 != _newConn) {
	    IceProcessMessages(_newConn, NULL, NULL);
	    if (0 != _newConn && 0 != _appContext) {
	       _newConnInputId = XtAppAddInput(_appContext,
			     IceConnectionNumber(_newConn),
			     (XtPointer) XtInputReadMask,
			     inFileDescriptorReady,
			     this);
	    }
	 }
      }
   }

   //
   // Check open connections
   //
   for (i = 0; i < _openConnectionsCount; i++) {
      int thisFd = IceConnectionNumber(_openConnections[i]);
      if (ready > 0 && FD_ISSET(thisFd, fds)) {
	 ready--;
	 IceProcessMessages(_openConnections[i], NULL, NULL);
      }
   }

   //
   // Check dangling connection
   //
   if (ready > 0 && 0 != _newConn ) {
      if (FD_ISSET(IceConnectionNumber(_newConn), fds)) {
	 ready--;
	 IceProcessMessages(_newConn, NULL, NULL);
	 status = IceConnectionStatus(_newConn);
	 if (IceConnectAccepted == status) {
	    addConnection(_newConn);
	    XtRemoveInput(_newConnInputId);
	    _newConn = NULL;
	    _newConnInputId = 0;
	 }
      }
   }
}

void
CProtocolIn::mainLoop(void)
{
   Status status;
   fd_set fds;
   int ready, prot;

   while (1) {
      FD_ZERO(& fds);
      status = IceConnectPending;

      if (0 != _newConn) {
	 FD_SET(IceConnectionNumber(_newConn), & fds);
      }

      for (int i = 0; i < _objCount; i++) {
	 FD_SET(IceGetListenConnectionNumber(_obj[i]), & fds);
      }

      for (i=0; i < _openConnectionsCount; i++) {
	 FD_SET(IceConnectionNumber(_openConnections[i]), & fds);
      }

      int ready = select(20, & fds, 0, 0, 0);

      dispatch(ready, & fds);

   }  // while 1
}

ostream &
operator<<(ostream &os, CProtocolIn &p)
{
   for (int i = 0; i < p._objCount; i++) {
      cout << IceGetListenConnectionString(p._obj[i]) << " ";
   }
   cout.flush();
   return os;
}

///////////////////////////////////////////////////////////////////////////

typedef enum { CODING_BYTES = 0, CODING_NUM = 1 } FieldType;

size_t
CCoding::encode(const CMessage *msg, char *fillBuffer)
{
   size_t resultSize = 0;
   char *headBuffer = fillBuffer;

   CDescField **cdf = msg -> getFields();
   Cardinal n = msg -> getFieldsCount();

   //
   // Encode number of fields
   //
   encodeCARD32(n, fillBuffer); fillBuffer += sizeof(CARD32);

   for (int i = 0; i < n; i++) {
      CFieldValue *cfv = cdf[i] -> _cfv;
      size_t size = cfv -> getSize();

      //
      // Encode type of this field
      //
      if (cfv -> isNum())
	 encodeCARD32(CODING_NUM, fillBuffer);
      else
	 encodeCARD32(CODING_BYTES, fillBuffer);
      fillBuffer += sizeof(CARD32);

      //
      // Encode size of this field
      //
      encodeCARD32(size, fillBuffer); fillBuffer += sizeof(CARD32);

      resultSize = sizeof(iceMsg);
      if (cfv -> isNum()) {
	 encodeCARD32(cfv -> numValue(), fillBuffer);
      }
      else {
	 memcpy(fillBuffer, cfv -> bytesValue(), size);
      }

      fillBuffer += size;
   }

   resultSize += fillBuffer - headBuffer;
   return resultSize;
}


CDescPacket *
CCoding::decode(CDescPacket *msg, const char *fillBuffer)
{
   CDescPacket *result = msg;
   CDescField **fields = msg -> getFields();
   const char *headBuffer = fillBuffer;

   CARD32 fieldsCount;
   fieldsCount = decodeCARD32(fillBuffer); fillBuffer += sizeof(CARD32);

   for (int i = 0; i < fieldsCount; i++) {
      CARD32 sizeOfField, typeOfField;
      
      typeOfField = decodeCARD32(fillBuffer); fillBuffer += sizeof(CARD32);
      sizeOfField = decodeCARD32(fillBuffer); fillBuffer += sizeof(CARD32);
      if (CODING_NUM == typeOfField) {
	 CARD32 value;
	 value = decodeCARD32(fillBuffer); fillBuffer += sizeof(CARD32);
	 result -> setField(fields[i] -> _fieldName, value);
      }
      else if (CODING_BYTES == typeOfField) {
	 result -> setField(fields[i] -> _fieldName, fillBuffer, sizeOfField);
	 fillBuffer += sizeOfField;
      }
      else
	 assert(0);
   }

   return result;
}

/*
** Encoding/Decoding long 
** @@ These two functions should be optimized (e.g. inlined) for
** better performances : they are used and abused everywhere 
*/
CARD32
CCoding::decodeCARD32(const char *pt)
{
  return (pt[0] * (1 << 24) & 0xff000000) +
         (pt[1] * (1 << 16) & 0xff0000) +
         (pt[2] * (1 << 8) & 0xff00) +
         (pt[3] & 0xff);
}

void
CCoding::encodeCARD32(CARD32 n, char *pt)
{
  pt[0] = (n >> 24) & 0xff;
  pt[1] = (n >> 16) & 0xff;
  pt[2] = (n >> 8) & 0xff;
  pt[3] = n & 0xff;
}
