/* commands.c for firepower

   Check for and process message based commands sent by the player
   */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "global.h"
#include "map.h"

#include "network.h"
#include "newplayer.h"

#define V_NEWMAP 1

static int votes[MAXPLAYERS];

static void cmnd_newmap();
static void cmnd_help();

/* The help messages sent when the player sends himself "HELP" */
struct command_help {
    char *command, *text;
    void (*function) ();
}

chelp[] = {
    {
        "NEWMAP", "Vote to reset the server and load a new map", cmnd_newmap
    }
    ,
    {
        "HELP", "Show this help", cmnd_help
    }
    ,
};

#define NUMCOMMANDS (sizeof(chelp) / sizeof(struct command_help))

/* check_newmap()

   Someone voted for a new map, see if the motion passed yet

   Return: number of players who still need to vote for a new map
           for the motion to pass
*/
static int check_newmap()
{
    int c, i, l;
    char buf[100];

    /* loop through every player */
    for (i = 0, c = 0, l = 0; i < MAXPLAYERS; i++) {
        if (GLO_players[i].p_status == PEMPTY) {
            /* If this player is an empty slot, clear his votes */
            votes[i] = 0;

        } else {
            /* one more player is alive */
            l++;
        }
        if (votes[i] & V_NEWMAP) {
            /* and he's voted for a new map */
            c++;
        }
    }

    /* got enough votes? */
    if (c > l / 2) {
        /* Yup! Let everyone know */
        sprintf(buf, "the vote NEWMAP has passed.");
        NET_send_message_to_all(buf, M_SRV);

        /* load a new map */
        map_pick_map(GLO_maplist);

        /* tell everyone the name */
        sprintf(buf, "Map name is %s", GLO_map_info.m_name);
        NET_send_message_to_all(buf, M_SRV);

        /* And send everyone back to the motd */
        for (i = 0; i < MAXPLAYERS; i++) {
            if (GLO_players[i].p_status != PEMPTY) {
                NET_send_map(i);
                newplayer_new_player(i);
            }
        }
        return 0;
    } else {
        /* return the number of players who still need to vote */
        return (l / 2 + 1) - c;
    }
}

/* cmnd_newmap(arg, pl)

   arg: unused

   This player voted for a new map

   Return: none
   */
static void cmnd_newmap(char *arg, int pl)
{
    int n;
    char buf[80];

    /* set his newmap flag */
    votes[pl] |= V_NEWMAP;

    /* check if it passed */
    if ((n = check_newmap()) > 0) {
        sprintf(buf, "Player %d votes NEWMAP, %d more votes are needed", pl, n);
        NET_send_message_to_all(buf, M_SRV);
    }
}

/* cmnd_help(arg, pl)

   arg: unused

   this player sent himself "HELP", send him the help text

   Return: none
   */
static void cmnd_help(char *arg, int pl)
{
    int i;
    char buf[80];

    /* loop through all commands, send the help text */
    for (i = 0; i < NUMCOMMANDS; i++) {
        sprintf(buf, "%6s - %s", chelp[i].command, chelp[i].text);
        NET_send_message_to_indiv(buf, M_SRV, pl);
    }
}

/* commands_do_command

   Called anytime a player sends himself a message, to see if it's a
   special command

   Return: none
   */
void commands_do_command(char *msg, int pl)
{
    int i;

    for (i = 0; i < NUMCOMMANDS; i++) {
        if (!strncasecmp(msg, chelp[i].command, strlen(chelp[i].command))) {
            /* if the text at the beginning of this message matches the
               name of the command we're checking, call the function
               associated with that command.  The rest of the message is
               passed as the first argument to the command. */
            (*(chelp[i].function)) (msg + strlen(chelp[i].command) + 1, pl);
            break;
        }
    }
}


/* commands_clear_votes(pl)

   when a player dies, clear all his votes

   Return: none
   */
void commands_clear_votes(int pl)
{
    votes[pl] = 0;
}

/* commands_init_commands

   When the server starts up, no one has voted for anything, this makes sure of
   that.

   Return: none
   */
void commands_init_commands()
{
    int i;

    for (i = 0; i < MAXPLAYERS; i++)
        votes[i] = 0;
}
