#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "data.h"

#define DEF_BOOL 1
#define DEF_STRING 2
#define DEF_INT 3

struct def {
    char *name;
    int type;
    void *var;
    char *help;
};

char keymap[256];

struct def defs[] = {
    {"namesOnTanks", DEF_BOOL, &namesOnTanks, "Whether to show player names or just numbers on tanks"},
    {"showBuildMarker", DEF_BOOL, &showBuildMarker, "Show where a road or wall will be built"},
    {"splitWindows", DEF_BOOL, &splitWindows, "Use three message windows instead of one"},
    {"server", DEF_STRING, &server, "Server to connect to (if not given on command line)"},
    {"port", DEF_INT, &port, "Port to connect to (if not given on command line)"},
    {"name", DEF_STRING, &name, "Character name to use (if not given on command line)"},
    {"keymap", DEF_STRING, keymap, "Remap keyboard.  Format : <new_char><old_char><new_char><old_char>..."},
};

#define NUMDEFS (sizeof(defs)/sizeof(struct def))

char *expand_filename(char *fn)
{
    char *home, *expfn;

    if(strncmp(fn, "~/", 2))
	return strdup(fn);

    if(!(home = getenv("HOME")))
	home = ".";
    
    expfn = malloc(strlen(home) + strlen(fn) + 2);
    
    sprintf(expfn, "%s/%s", home, &fn[2]);
    return expfn;
}
	
void init_keymap()
{
    int i;

    for(i=0; i<256; i++)
	keymap[i]=i;
}

void add_keymap(char *map)
{
    char *k;

    for(k = map; k < map+strlen(map); k+=2)
	if(*(k+1)) /* don't remap if at the end of the map string */
	    keymap[(int)*k] = *(k+1);
}

char do_keymap(char k)
{
    return keymap[(int)k];
}

void do_default(char *name, char *val)
{
    int i;
    
    if(!strcasecmp(name, "keymap")) {
	add_keymap(val);
	return;
    }

    for(i=0; i<NUMDEFS; i++) {
	if(!strcasecmp(name, defs[i].name)) {
	    switch(defs[i].type) {
	      case DEF_BOOL:
		if((strncasecmp(val, "on", 2) == 0) || 
		   (strncasecmp(val, "1", 1)==0) || 
		   (strncasecmp(val, "true", 4)==0))
		    *((int *)defs[i].var) = 1;
		else
		    *((int *)defs[i].var) = 0;
		break;
	      case DEF_INT:
		*((int *)defs[i].var) = atoi(val);
		break;
	      case DEF_STRING:
		*((char **)defs[i].var) = strdup(val);
		break;
	      default:
		printf("uhh, what?\n");
		break;
	    }
	    continue;
	}
    }
}

void write_defaults(char *fn)
{
    FILE *df;
    int i, k;

    if(!(df= fopen(fn, "w"))) {
	printf("Couldn't open \"%s\" for writing\n", fn);
	return;
    }

    for(i=0;i<NUMDEFS;i++) {
	fprintf(df, "# %s\n", defs[i].help);
	if(strcmp(defs[i].name, "keymap") == 0) {
	    fprintf(df, "keymap: %c%c", 1, 1); /* the two chars at the beginning are in case
						  the first entry happens to be remapping
						  space to something(very likely!) */
	    for(k=0;k<256;k++) {
		if(keymap[k] != (char)k)
		    fprintf(df, "%c%c", k, keymap[k]);
	    }
	    fprintf(df, "\n");
	} else {
	    fprintf(df, "%s: ", defs[i].name);
	    switch(defs[i].type) {
	      case DEF_BOOL:
		fprintf(df, "%s\n", (*((int *)defs[i].var) ? "on" : "off"));
		break;
	      case DEF_INT:
		fprintf(df, "%d\n", *((int *)defs[i].var));
		break;
	      case DEF_STRING:
		fprintf(df, "%s\n", *((char **)defs[i].var));
		break;
	      default:
		fprintf(df, "\n");
		break;
	    }
	}
	fprintf(df, "\n");
    }
    fclose(df);
}

void read_defaults(char *filename)
{
    FILE *df;
    char *expfilename, *junk;
    char buf[1024];
    char *s, *val;

    init_keymap();

    if(!filename)
	filename = "~/.firepowerrc";
    
    expfilename = expand_filename(filename);

    if(!(df=fopen(expfilename, "r"))) {
	if(strcmp(expfilename, (junk=expand_filename("~/.firepowerrc"))) == 0) {
	    free(junk);
	    printf("Creating defaults file \"%s\"\n", expfilename);
	    write_defaults(expfilename);
	    free(expfilename);
	    return;
	}
	printf("Couldn't open defaults file \"%s\"\n", expfilename);
	free(expfilename);
	return;
    }

    while(!feof(df)) {
	if(fgets(buf, 1023, df)) {
	    if(buf[0] == '#')
		continue;
	    val = 0;
	    for(s=buf;s<buf+strlen(buf);s++)
		if(*s == ':') {
		    *s = 0;
		    val = s+1;
		    break;
		}
	    if(val) {
		while((*val == ' ' || *val == '\t') && *val != 0)
		    val++;
		val[strlen(val) - 1] = 0; /* get rid of newline */
		if(val) {
		    do_default(buf, val);
		}
	    }
	}
    }
    fclose(df);
    free(expfilename);
}
