/*****************************************************************************
 *                                                                           *
 *  Copyright (c) 1993-1997 Elan Feingold (elan@jeeves.net)                  *
 *                                                                           *
 *     PERMISSION TO USE, COPY, MODIFY, AND TO DISTRIBUTE THIS SOFTWARE      *
 *     AND ITS DOCUMENTATION FOR ANY PURPOSE IS HEREBY GRANTED WITHOUT       *
 *     FEE, PROVIDED THAT THE ABOVE COPYRIGHT NOTICE APPEAR IN ALL           *
 *     COPIES AND MODIFIED COPIES AND THAT BOTH THAT COPYRIGHT NOTICE AND    *
 *     THIS PERMISSION NOTICE APPEAR IN SUPPORTING DOCUMENTATION.  THERE     *
 *     IS NO REPRESENTATIONS ABOUT THE SUITABILITY OF THIS SOFTWARE FOR      *
 *     ANY PURPOSE.  THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS       *
 *     OR IMPLIED WARRANTY.                                                  *
 *                                                                           *
 *****************************************************************************/

#include <X11/X.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/AsciiText.h>

#include "colorEdit.h"
#include "dice.h"
#include "callbacks.h"
#include "colormap.h"
#include "client.h"
#include "utils.h"
#include "cards.h"
#include "riskgame.h"
#include "gui-vars.h"
#include "debug.h"
#include "riskgame.h"

/* Action tables */
static XtActionsRec actionTable[] =
{
  { "updateColor",      (XtActionProc)COLEDIT_UpdateColorWithInput },
  { "colorEditCancel",  (XtActionProc)COLEDIT_Cancel },
  { NULL, NULL }
};

/* Widgets */
static Widget wColorEditShell;
static Widget wColorEditForm;
static Widget wColorEditLabel;
static Widget wRedScrollbar, wBlueScrollbar, wGreenScrollbar;
static Widget wSampleCountryForm;
static Widget wColorInputLabel, wColorInputText;
static Widget wColorOK, wColorCancel;

/* Color editing related things */
#define COLOR_EDIT_OK           1
#define COLOR_EDIT_CANCEL       0
#define COLOR_EDIT_INPROGRESS  -1
static Int32 iQueryResult = COLOR_EDIT_OK;

extern Char strScratch[256];


/************************************************************************ 
 *  FUNCTION: COLEDIT_MapShiftClick
 *  HISTORY: 
 *     01.22.95  ESF  Created.
 *     01.24.95  ESF  Changed to get coordinates manually.
 *     01.25.95  ESF  Added check that the country was not empty.
 *     12.07.95  ESF  Fixed a bug -- don't allow editing of ocean.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void COLEDIT_MapShiftClick(void)
{
  Int32   iColor, iCountry;
  Int32   x, y, xRoot, yRoot;
  UInt32  uiButtons;
  Window  winDummy1, winDummy2;

  if (!XQueryPointer(hDisplay, XtWindow(wMap), &winDummy1, &winDummy2, 
		    &xRoot, &yRoot, &x, &y, &uiButtons))
    {
      /* Pointer has somehow left window */
      return;
    }

  /* Get the coordinates of the mouse */
  iColor = XGetPixel(pMapImage, x, y);
  iCountry = COLOR_ColorToCountry(iColor);

  /* See if there is a player on the country */
  if (iCountry<0 || iCountry>=NUM_COUNTRIES ||
      RISK_GetOwnerOfCountry(iCountry) == -1 ||
      RISK_GetNumArmiesOfCountry(iCountry) == 0)
    return;

  /* Edit this color */
  COLEDIT_EditColor(iColor, TRUE);
}


/************************************************************************ 
 *  FUNCTION: COLEDIT_BuildDialog
 *  HISTORY: 
 *     01.21.95  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void COLEDIT_BuildDialog(void)
{
  Widget  wColorDummy;

  /* Add Actions */
  XtAppAddActions(appContext, actionTable, XtNumber(actionTable));

  wColorEditShell = XtCreatePopupShell("wColorEditShell", 
				       transientShellWidgetClass,
				       wToplevel, pVisualArgs, iVisualCount);
  wColorEditForm = XtCreateManagedWidget("wColorEditForm", formWidgetClass, 
					 wColorEditShell, NULL, 0);

  /* The label for the dialog */
  wColorEditLabel = XtCreateManagedWidget("wColorEditLabel", labelWidgetClass, 
					  wColorEditForm, NULL, 0);

  /* The three (RGB) scrollbars */
  wRedScrollbar = XtCreateManagedWidget("wRedScrollbar", 
					scrollbarWidgetClass,
					wColorEditForm, NULL, 0);
  XtAddCallback(wRedScrollbar, XtNjumpProc, 
		(XtCallbackProc)COLEDIT_ScrollbarMoved, NULL);
  XawScrollbarSetThumb(wRedScrollbar, 0.0, 0.05);
  wGreenScrollbar = XtCreateManagedWidget("wGreenScrollbar", 
					  scrollbarWidgetClass,
					  wColorEditForm, NULL, 0);
  XtAddCallback(wGreenScrollbar, XtNjumpProc, 
		(XtCallbackProc)COLEDIT_ScrollbarMoved, NULL);
  XawScrollbarSetThumb(wGreenScrollbar, 0.0, 0.05);
  wBlueScrollbar = XtCreateManagedWidget("wBlueScrollbar", 
					 scrollbarWidgetClass,
					 wColorEditForm, NULL, 0);
  XtAddCallback(wBlueScrollbar, XtNjumpProc, 
		(XtCallbackProc)COLEDIT_ScrollbarMoved, NULL);
  XawScrollbarSetThumb(wBlueScrollbar, 0.0, 0.05);

  /* A sample country in the chosen color */
  wSampleCountryForm = XtCreateManagedWidget("wSampleCountryForm", 
					     labelWidgetClass,
					     wColorEditForm, NULL, 0);

  /* Input text label */
  wColorInputLabel = XtCreateManagedWidget("wColorInputLabel",
					   labelWidgetClass,
					   wColorEditForm, NULL, 0);

  /* The actual widget for inputting text */
  wColorInputText = XtVaCreateManagedWidget("wColorInputText",
					    asciiTextWidgetClass,
					    wColorEditForm, 
					    XtNeditType, XawtextEdit, NULL);

  /* This is used to align the OK/Cancel buttons */
  wColorDummy = XtCreateManagedWidget("wColorDummy", formWidgetClass,
				      wColorEditForm, NULL, 0);

  /* The buttons */
  wColorOK = XtCreateManagedWidget("wColorOK",
				   commandWidgetClass,
				   wColorEditForm, NULL, 0);
  XtAddCallback(wColorOK, XtNcallback, (XtCallbackProc)COLEDIT_Ok, NULL);  
  wColorCancel = XtCreateManagedWidget("wColorCancel",
				       commandWidgetClass,
				       wColorEditForm, NULL, 0);
  XtAddCallback(wColorCancel, XtNcallback, (XtCallbackProc)COLEDIT_Cancel, 
		NULL);
}

/* Utility for the next function */
#define COLEDIT_CloseColorDialog() \
 UTIL_DisplayError(""); \
 XtSetKeyboardFocus(wToplevel, wToplevel); \
 XtRemoveGrab(wColorEditShell); \
 XtUnrealizeWidget(wColorEditShell);

/************************************************************************ 
 *  FUNCTION: COLEDIT_EditColor
 *  HISTORY: 
 *     01.22.95  ESF  Created.
 *     01.26.95  ESF  Fixed last few bugs.
 *     23.08.95  JC   Use of COLOR_Depth.
 *     12.07.95  ESF  Made the editable color be the dice color.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
String COLEDIT_EditColor(Int32 iColor, Flag fStoreColor)
{
  XEvent   xEvent;
  Int32    x, y, c, iCardBlack;
  Pixmap   pixSampleImage; 
  Int32    iCountry;
  XImage  *pImage;
  CString  strNewColor=NULL;

  D_Assert(iColor >= COLOR_CountryToColor(0) && iColor <= COLOR_DieToColor(3), 
	   "Asked to edit bogus color!");
    
  /* Center the new shell */
  UTIL_CenterShell(wColorEditShell, wToplevel, &x, &y);
  XtVaSetValues(wColorEditShell, 
		XtNallowShellResize, False,
		XtNx, x, 
		XtNy, y, 
		XtNborderWidth, 1,
#ifdef ENGLISH
		XtNtitle, "Edit Color",
#endif
#ifdef FRENCH
		XtNtitle, "Edition des couleurs",
#endif
		NULL);

  /* Initialize the scrollbars to represent the right color */
  COLEDIT_InitDialogWithColor(iColor);

  iCountry = COLOR_ColorToCountry(iColor);

  /* If the country requested was not a valid country, then 
   * pick a random country for the caller.
   */

  if (iCountry >= NUM_COUNTRIES || iCountry < 0)
    iCountry = rand() % NUM_COUNTRIES;

  /* Hide the dice */
  DICE_Hide();

  /* Put the country on the sample */
  pImage = CARD_GetCountryImage(iCountry, 
				COLOR_DieToColor(0),
				COLOR_CountryToColor(NUM_COUNTRIES));
  
  /* Scale it to the sample country form */
  pImage = CARD_ScaleImage(pImage, 100, 100);
  
  /* Set the initial color of the sample country */
  COLOR_CopyColor(iColor, COLOR_DieToColor(0));

  /* Dump it on the sample country location */
  pixSampleImage = XCreatePixmap(hDisplay, pixMapImage, 140, 140, COLOR_Depth);
  XSetForeground(hDisplay, hGC, COLOR_CountryToColor(NUM_COUNTRIES));
  XFillRectangle(hDisplay, pixSampleImage, hGC, 0, 0, 140, 140);
  if (COLOR_IsTrueColors())
    {
      iCardBlack = BlackPixel(hDisplay, 0);
      for (y = 0; y < pImage->height; y++)
          for (x = 0; x < pImage->width; x++)
            {
              c = XGetPixel(pImage, x, y);
              if (c != iCardBlack)
                  c = COLOR_QueryColor(c);
              XSetForeground(hDisplay, hGC, c);
              XDrawPoint(hDisplay, pixSampleImage, hGC,
                         (140-pImage->width)/2 + x,
                         (140-pImage->height)/2 + y);
            }
    }
  else
    {
      XPutImage(hDisplay, pixSampleImage, hGC, pImage, 0, 0, 
	        (140-pImage->width)/2, (140-pImage->height)/2, 140, 140);
    }

  XtVaSetValues(wSampleCountryForm, XtNbitmap, pixSampleImage, NULL); 
  XtVaSetValues(wSampleCountryForm, XtNbackground, 
		COLOR_QueryColor(NUM_COUNTRIES), NULL);

  /* Popup the color editing dialog */
  XtMapWidget(wColorEditShell);
  XtAddGrab(wColorEditShell, True, True);
  XtSetKeyboardFocus(wToplevel, wColorEditShell); 
  XtSetKeyboardFocus(wColorEditShell, wColorInputText);

  /* Look until user presses one of the buttons */
  iQueryResult = COLOR_EDIT_INPROGRESS;
  while (iQueryResult == COLOR_EDIT_INPROGRESS) 
    {
      /* pass events */
      XNextEvent(hDisplay, &xEvent);
      XtDispatchEvent(&xEvent);
    }

  if (iQueryResult == COLOR_EDIT_OK)
    {
      Int32   i;
      
      /* Just in case the player typed a color name in and then hit "Ok"
       * update the color in the country in the last second that the dialog
       * is up, so that when we copy the color out of there it will be the
       * color that the player typed in.
       */

      COLEDIT_UpdateColorWithInput();

      /* Get the color from the text widget.  If the player moved the
       * scrollbars, then the color will have been written in here.
       * If the player typed something in, it will be in here.
       */

      XtVaGetValues(wColorInputText, XtNstring, &strNewColor, NULL);

      /* Are we trying to store the color? */
      if (fStoreColor)
	{
	  /* Sanity check */
	  if (iCountry < 0 || iCountry >= NUM_COUNTRIES ||
	      RISK_GetOwnerOfCountry(iCountry) == -1 ||
	      RISK_GetNumArmiesOfCountry(iCountry) <= 0)
	    {
	      (void)UTIL_PopupDialog("Error", 
				     "Cannot set new value for color!", 
				     1, "Ok", NULL, NULL);
	    }
	  else
	    {
	      /* And also the actual color for the player */
	      COLOR_StoreNamedColor(strNewColor, 
				    RISK_GetOwnerOfCountry(iCountry));
	      
	      /* Set all the countries that the player owns to be new color */
	      for (i=0; i!=NUM_COUNTRIES; i++)
		if (RISK_GetOwnerOfCountry(i) == 
		    RISK_GetOwnerOfCountry(iCountry))
		  COLOR_CopyColor(COLOR_DieToColor(0), 
				  COLOR_CountryToColor(i));
	  
	      /* If the player whose color changed is the current
	       * player, then recolor the current player indicator.
	       * Also, the dice really need to change color too
	       * (thanks, Kirsten, for reminding me about this -- who
	       * says English majors can't program?)  For now punt on
	       * the dice.
	       */
	      
	      if (RISK_GetOwnerOfCountry(iCountry) == iCurrentPlayer)
		COLOR_CopyColor(COLOR_DieToColor(0), COLOR_DieToColor(2));
	    }
	}
    }
  
  XDestroyImage(pImage);
  
  COLEDIT_CloseColorDialog();
  return strNewColor;
}

void COLEDIT_Ok(void) { iQueryResult = COLOR_EDIT_OK; }
void COLEDIT_Cancel(void) { iQueryResult = COLOR_EDIT_CANCEL; }


/************************************************************************ 
 *  FUNCTION: COLEDIT_ScrollbarMoved
 *  HISTORY: 
 *     01.22.95  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void COLEDIT_ScrollbarMoved(void)
{
  float   r, g, b;
  XColor  xColor;

  XtVaGetValues(wRedScrollbar, XtNtopOfThumb, &r, NULL);
  XtVaGetValues(wGreenScrollbar, XtNtopOfThumb, &g, NULL);
  XtVaGetValues(wBlueScrollbar, XtNtopOfThumb, &b, NULL);

  /* Setup the new color */
  xColor.red   = (Int32)((1.0-r)*65535.0);
  xColor.green = (Int32)((1.0-g)*65535.0);
  xColor.blue  = (Int32)((1.0-b)*65535.0);

  /* Color the sample country with the new color */
  COLOR_StoreColor(COLOR_DieToColor(0), xColor.red, xColor.green, xColor.blue);

  /* Display the name of the color */
  sprintf(strScratch, "#%02x%02x%02x", 
	  xColor.red/256, xColor.green/256, xColor.blue/256);
  XtVaSetValues(wColorInputText, 
		XtNstring, strScratch, 
		XtNinsertPosition, strlen(strScratch),
		NULL);
}

/************************************************************************ 
 *  FUNCTION: COLEDIT_InitDialogWithColor
 *  HISTORY: 
 *     01.22.95  ESF  Created.
 *     18.08.95  JC   Modified to call COLOR_GetColor.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void COLEDIT_InitDialogWithColor(Int32 iColor)
{
  float  f;
  UInt16 r, g, b;

  COLOR_GetColor(iColor, &r, &g, &b);

  f = 1.0 - ((float)r)/65535.0;
  XawScrollbarSetThumb(wRedScrollbar, f, 0.05);

  f = 1.0 - ((float)g)/65535.0;
  XawScrollbarSetThumb(wGreenScrollbar, f, 0.05);

  f = 1.0 - ((float)b)/65535.0;
  XawScrollbarSetThumb(wBlueScrollbar, f, 0.05);

  /* Display the name of the color */
  sprintf(strScratch, "#%02x%02x%02x", 
	  r/256, g/256, b/256);
  XtVaSetValues(wColorInputText, 
		XtNstring, strScratch, 
		XtNinsertPosition, strlen(strScratch),
		NULL);
}


/************************************************************************ 
 *  FUNCTION: COLEDIT_UpdateColorWithInput
 *  HISTORY: 
 *     01.23.95  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void COLEDIT_UpdateColorWithInput(void)
{
  CString  strColor;
  float    r, g, b;
  XColor   xColor1, xColor2;

  /* Get the color from the text widget */
  XtVaGetValues(wColorInputText, XtNstring, &strColor, NULL);

  /* Also get the color from the scrollbars. */
  XtVaGetValues(wRedScrollbar, XtNtopOfThumb, &r, NULL);
  XtVaGetValues(wGreenScrollbar, XtNtopOfThumb, &g, NULL);
  XtVaGetValues(wBlueScrollbar, XtNtopOfThumb, &b, NULL);
  sprintf(strScratch, "#%02x%02x%02x", 
	  (Int32)((1.0-r)*65535.0)/256, 
	  (Int32)((1.0-g)*65535.0)/256, 
	  (Int32)((1.0-b)*65535.0)/256);

  /* Adjust for the screen */
  xColor1.flags = DoRed | DoGreen | DoBlue;
  (void)XParseColor(hDisplay, cmapColormap, strScratch, &xColor1);

  /* Is it a valid color? */
  if (XParseColor(hDisplay, cmapColormap, strColor, &xColor2))
    {
      /* If they match, then take this to mean the same as the "Ok" button. */
      if (xColor1.red   == xColor2.red &&
	  xColor1.green == xColor2.green &&
	  xColor1.blue  == xColor2.blue)
	{
	  COLEDIT_Ok();
	}
      else
	{
	  COLOR_StoreColor(COLOR_DieToColor(0), 
			   xColor2.red, xColor2.green, xColor2.blue);
	  COLEDIT_InitDialogWithColor(COLOR_DieToColor(0));
	}
    }
  else
#ifdef ENGLISH
    (void)UTIL_PopupDialog("Error", "That color does not exist!", 
			   1, "Ok", NULL, NULL);
#endif
#ifdef FRENCH
    (void)UTIL_PopupDialog("Erreur", "Cette couleur n'existe pas!", 
			   1, "Ok", NULL, NULL);
#endif
}
