/*****************************************************************************
 *                                                                           *
 *  Copyright (c) 1993-1997 Elan Feingold (elan@jeeves.net)                  *
 *                                                                           *
 *     PERMISSION TO USE, COPY, MODIFY, AND TO DISTRIBUTE THIS SOFTWARE      *
 *     AND ITS DOCUMENTATION FOR ANY PURPOSE IS HEREBY GRANTED WITHOUT       *
 *     FEE, PROVIDED THAT THE ABOVE COPYRIGHT NOTICE APPEAR IN ALL           *
 *     COPIES AND MODIFIED COPIES AND THAT BOTH THAT COPYRIGHT NOTICE AND    *
 *     THIS PERMISSION NOTICE APPEAR IN SUPPORTING DOCUMENTATION.  THERE     *
 *     IS NO REPRESENTATIONS ABOUT THE SUITABILITY OF THIS SOFTWARE FOR      *
 *     ANY PURPOSE.  THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS       *
 *     OR IMPLIED WARRANTY.                                                  *
 *                                                                           *
 *****************************************************************************/

#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <signal.h>

#include "types.h"
#include "riskgame.h"
#include "network.h"
#include "server.h"
#include "deck.h"
#include "debug.h"
#include "version.h"

/* Move this to the Imakefile!! */
#ifdef __hpux
#define FDSET int 
#else
#define FDSET fd_set
#endif

/* This may be a static value (i.e. OPEN_MAX), or a dynamic value,
 * as in OSF/1, accessed through sysconf().  I'm not sure that all 
 * systems have these, so we'll just guess.  I don't think this is
 * exceptionally evil, since if we run out of descriptors, the socket
 * or accept calls will fail.
 */

#define MAX_DESCRIPTORS 128

#ifdef ENGLISH
#define SERVERNAME "Server"
#endif
#ifdef FRENCH
#define SERVERNAME "Serveur"
#endif

Int32      iServerCommLink;
Int32      iState;
fd_set     fdSet, fdBackup;
Deck      *pPlayerDeck = NULL;
Deck      *pCardDeck;
Int32      iReply, iAllClients;
Int32      iMaxFileDescUsed = -1;
Char       strScratch[256];
Int32      iServerMode = SERVER_REGISTERING;
Flag       fGameReset = TRUE;
Flag       fRememberKilled = FALSE;
Int32      iTurn;

/* Private server data that doesn't get replicated. */
typedef struct _Client
{
  Int32     iCommLink;
  CString   strAddress;
  Int32     iState, iType, iSpecies;
  Flag      fStarted;
} Client;

/* Structure to track failures */
typedef struct _Failure
{
  CString   strReason;
  Int32     iCount;
} Failure;

Failure   pFailures[MAX_CLIENTS];
Client    pClients[MAX_CLIENTS];
Int32     iNumClients=0, iNumFailures=0;

/* Private functions */
void     SRV_ResetGame(void);
void     SRV_ReplicateRegistrationData(Int32 iCommLinkDest);
void     SRV_ReplicateAllData(Int32 iCommLinkDest);
void     SRV_AttemptNewGame(void);
void     SRV_DistributeCountries(void);
Flag     SRV_DistributeMissions(void);
void     SRV_SetInitialArmiesOfPlayers(void);
void     SRV_SetInitialMissionOfPlayers(void);
void     SRV_NotifyClientsOfTurn(Int32 iTurn);
Int32    SRV_CommLinkToClient(Int32 iCommLink);
void     SRV_HandleRegistration(Int32 iCommLink);
void     SRV_HandleSignals(Int32 iParam);
void     UTIL_ExitProgram(Int32 iExitValue);
Int32    SRV_IterateTurn(void);
Int32    SRV_GetNumClientsStarted(void);

/* Very private functions */
Client *_SRV_GetNthClient(Int32 iNumClient);
 
/* Server message handlers */
void SRV_HandleEXIT(Int32 iExitValue);
void SRV_HandleALLOCPLAYER(Int32 iClient);
void SRV_HandleFREEPLAYER(void *pvMessage);
void SRV_HandleREPLYPACKET(void *pvMessage);
void SRV_HandleMESSAGEPACKET(Int32 iClient, void *pvMessage);
void SRV_HandleENTERSTATE(void *pvMessage);
void SRV_HandleDEREGISTERCLIENT(Int32 iClient);


/************************************************************************ 
 *  FUNCTION: SRV_Init
 *  HISTORY: 
 *     01.23.94  ESF  Created.
 *     02.22.94  ESF  Cleaned up a bit, removing warnings.
 *     05.08.94  ESF  Fixed MSG_MESSAGEPACKET handling.
 *     05.10.94  ESF  Fixed, not registering needed callback with DistObj.
 *     05.12.94  ESF  Added fd usage map.
 *     05.19.94  ESF  Fixed bug, passing &pvMessage instead of pvMessage.
 *     08.16.94  ESF  Cleaned up.
 *     10.01.94  ESF  Added initialization of failures.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_Init(void)
{
  struct sockaddr_in   server;
  Int32                i;

  /* Print an informative message */
#ifdef ENGLISH
  printf("SERVER: Starting Frisk %s.\n", VERSION);
#endif
#ifdef FRENCH
  printf("SERVEUR: Dmarage de Frisk %s.\n", VERSION);
#endif

  /* Catch signals so that we can clean up upon termination */
  signal(SIGHUP,  SRV_HandleSignals);
  signal(SIGINT,  SRV_HandleSignals);
  signal(SIGQUIT, SRV_HandleSignals);
  signal(SIGILL,  SRV_HandleSignals);
  signal(SIGTRAP, SRV_HandleSignals);
  signal(SIGFPE,  SRV_HandleSignals);
  signal(SIGKILL, SRV_HandleSignals);
  signal(SIGBUS,  SRV_HandleSignals);
  signal(SIGTERM, SRV_HandleSignals);

  /* We want to ignore this signal, because we don't want a I/O
   * failure to terminate the program.
   */

  signal(SIGPIPE, SIG_IGN);

  /* Init. clients and players */
  SRV_SetNumClients(0);

  /* Here we use the internals, not good */
  for (i=0; i!=MAX_CLIENTS; i++)
    {
      SRV_SetAllocationStateOfClient(i, ALLOC_NONE);
      pFailures[i].strReason = NULL;
      pFailures[i].iCount = 0;
    }

  /* Initialize the pool of free players */
  pPlayerDeck = DECK_Create(MAX_PLAYERS);

  /* Create server socket */
  if ((iServerCommLink = socket(AF_INET, SOCK_STREAM, 0)) < 0)
    {
      perror("Creating CommLink");
      UTIL_ExitProgram(1);
    }
  
  /* Update the max */
  iMaxFileDescUsed = MAX(iMaxFileDescUsed, iServerCommLink);
  
  /* Name sockets using wildcards */
  server.sin_family = AF_INET;
  server.sin_addr.s_addr = INADDR_ANY;
  server.sin_port = htons(RISK_PORT);
  
  /* Bind the socket to the port */
  if (bind(iServerCommLink, (struct sockaddr *)&server, sizeof(server)))
    {
#ifdef ENGLISH
      printf("SERVER: The Frisk port (%d), is already in use.  Perhaps a\n"
	     "        server is already running, or else a server crashed\n"
	     "        badly, in which case you will probably have to wait\n"
	     "        a few minutes till the port clears.  I'm exiting.\n", 
#endif
#ifdef FRENCH
      printf("SERVEUR: Le port (%d) de Frisk est utilis. Le serveur est\n"
	     "         dj lanc, ou un serveur est salement plant,\n"
	     "         dans ce cas il faut attendre quelques minutes pour\n"
	     "         que le port soit  nouveau disponible. Je quitte.\n", 
#endif
	     RISK_PORT);
      UTIL_ExitProgram(1);
    }

  /* Add the socket options to the socket */
  NET_SetCommLinkOptions(iServerCommLink);

  SRV_PlayGame();
}


/************************************************************************ 
 *  FUNCTION: SRV_BroadcastTextMessage
 *  HISTORY: 
 *     01.26.94  ESF  Created.
 *     01.15.95  ESF  Initialized .iTo field to avoid uninitialized memory.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_BroadcastTextMessage(CString strMessage)
{
  MsgMessagePacket   msgMess;

  msgMess.strMessage  = strMessage;
  msgMess.iFrom       = FROM_SERVER;
  msgMess.iTo         = DST_ALLPLAYERS;

  /* Send the message out to all clients */
  SRV_BroadcastMessage(MSG_MESSAGEPACKET, &msgMess);
}


/************************************************************************ 
 *  FUNCTION: SRV_PlayGame
 *  HISTORY: 
 *     02.04.94  ESF  Created.
 *     02.05.94  ESF  Fixed broadcast loop bug. 
 *     02.05.94  ESF  Fixed message receive bug.
 *     03.03.94  ESF  Changed to send _UPDATE to all clients but sender.
 *     03.28.94  ESF  Added _DEADPLAYER & _ENDOFGAME.
 *     03.29.94  ESF  Added _REQUESTCARD.
 *     04.01.94  ESF  Fixed card exchange to work right with jokers.
 *     04.11.94  ESF  Fixed CARDPACKET to broadcast the card.
 *     05.05.94  ESF  Added MSG_OBJ* msgs.
 *     05.06.94  ESF  Factored out dealing cards code.
 *     05.15.94  ESF  Added MSG_[ALLOC|FREE]PLAYER.
 *     05.15.94  ESF  Added MSG_REPLYPACKET.
 *     05.17.94  ESF  Added MSG_NETMESSAGE.
 *     06.24.94  ESF  Fixed memory leak bug.
 *     07.27.94  ESF  Completely revamped, combined with CollectPlayers().
 *     09.31.94  ESF  Fixed so that a new deck is created upon reset.
 *     10.01.94  ESF  Fixed MSG_ENDOFGAME to pass message and not NULL.
 *     10.02.94  ESF  Fixed so in case of bogus message, client is killed.
 *     10.03.94  ESF  Fixed bug, excessive processing of MSG_DEREGISTERCLIENT.
 *     10.08.94  ESF  Added SERVER_FORTIFYING mode to fix a bug.
 *     10.29.94  ESF  Added handling for MSG_DICEROLL.
 *     10.30.94  ESF  Fixed serious bug: SERVER[_REGISTERING -> _FORTIFYING]. 
 *     25.08.95  JC   Don't call perror if errno is equal to 4.
 *     28.08.95  JC   Added handling for MSG_ENDOFMISSION and MSG_VICTORY.
 *     30.08.95  JC   Added handling for MSG_FORCEEXCHANGECARDS.
 *     30.08.95  JC   The first AI-Client can't start the game, but other
 *                    AI-Client can do this for computer's battles.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_PlayGame(void)
{
  Int32             i, n, iMessageType;
  void             *pvMessage;

  /* Create the card deck */
  pCardDeck = DECK_Create(NUM_COUNTRIES + 2);

  /* Add the initial fd to keep an eye on -- the connect socket */
  FD_ZERO(&fdBackup);
  FD_SET(iServerCommLink, &fdBackup);
  
  /* Start accepting connections */
  listen(iServerCommLink, 5);
  
  /* Loop for the entirety of the game */
  for(;;)
    {
      extern int errno;
      fdSet = fdBackup;

      /* If there have been any failures, then deal with them */
      SRV_RecoverFailures();

      /* Wait for a message to come in */
      if (select(iMaxFileDescUsed+1, (FDSET *)&fdSet, (FDSET *)0, (FDSET *)0,
          NULL) < 0)
           /* errno = 4 is caused by SRV_HandleSignals */
          if (errno != 4)
              perror("Select");

      /* Two things might have happened here.  Either an existing client
       * sent a message to the server, or a new client sent a message to
       * the connect port, trying to join the game.  If the former occurred
       * process it normally.  If the latter occurred, let the client
       * connect, and add its fd to the fd map.  If we are in the middle
       * of a game, send it a message saying this, and then perform a 
       * RISK_SelectiveReplicate() so as to get the new client in the
       * same state as the others.  Also send a message to the other
       * clients telling them what is happening.
       */
      
      if (FD_ISSET(iServerCommLink, &fdSet))
	{
	  Int32 iNewCommLink;

	  /* Try to accept the new connection */
	  if ((iNewCommLink = accept(iServerCommLink, 0, 0)) < 0)
	    {
	      /* Couldn't do it, go back to top */
#ifdef ENGLISH
	      printf("SERVER: Connect to client failed, its loss...\n");
#endif
#ifdef ENGLISH
	      printf("SERVEUR: Une connexion choue, le client est perdu...\n");
#endif
	      continue;
	    }
	  else /* connection went fine */
	    {
	      /* Does Frisk have enough resources to hold this client? */
	      if (SRV_GetNumClients() >= MAX_CLIENTS ||
		  iNewCommLink >= MAX_DESCRIPTORS)
		{
		  (void)RISK_SendMessage(iNewCommLink, MSG_EXIT, NULL);
		  close(iNewCommLink);
		  continue;
		}
	    }

	  /* Assert: At this point, connection is complete and we have
	   * enough resources to keep it around.  Begin connection protocol.
	   */

	  /* Set options on new CommLink */
	  NET_SetCommLinkOptions(iNewCommLink);
	  SRV_HandleRegistration(iNewCommLink);
	}
      else
	{
	  Flag fHandledMessage = FALSE;

	  /* There is a client trying to send a message */
	  for (i=0; i!=MAX_CLIENTS && !fHandledMessage; i++)
	    if (SRV_GetAllocationStateOfClient(i) == ALLOC_COMPLETE &&
		FD_ISSET(SRV_GetCommLinkOfClient(i), &fdSet))
	      {
		/* We got the message we were looking */
		fHandledMessage = TRUE;
		
		/* Actually pick it up, skip it if the was failure */
		if (!RISK_ReceiveMessage(SRV_GetCommLinkOfClient(i), 
					 &iMessageType, &pvMessage))
		  continue;

		/* Depending on the message type, dispatch it to one
		 * of the handlers.  They usually take just a message,
		 * but depending on what they do they could take the
		 * and/or the client index.
		 */

		switch (iMessageType)
		  {
		  case MSG_NETMESSAGE:
		  case MSG_DICEROLL:
		  case MSG_PLACENOTIFY:
		  case MSG_ATTACKNOTIFY:
		  case MSG_MOVENOTIFY:
		    SRV_CompleteBroadcast(i, iMessageType, pvMessage);
		    break;
		  case MSG_ENDOFMISSION:
		  case MSG_VICTORY:
		    SRV_BroadcastMessage(iMessageType, pvMessage);
		    break;
		  case MSG_MISSION:
		    if (SRV_DistributeMissions())
		        SRV_BroadcastMessage(iMessageType, pvMessage);
		    break;

		  case MSG_ENDOFGAME:
		    {
		      iServerMode = SERVER_REGISTERING;
		      /* Let everyone else know */
		      SRV_BroadcastMessage(MSG_ENDOFGAME, NULL);
		      SRV_ResetGame();
		    }
		    break;

		  case MSG_ALLOCPLAYER:
		    SRV_HandleALLOCPLAYER(i);
		    break;
		    
		  case MSG_FREEPLAYER:
		    SRV_HandleFREEPLAYER(pvMessage);
		    break;
		    
		  case MSG_REPLYPACKET:
		    SRV_HandleREPLYPACKET(pvMessage);
		    break;

		  case MSG_MESSAGEPACKET:
		    SRV_HandleMESSAGEPACKET(i, pvMessage);
		    break;

		  case MSG_STARTGAME:
		    {
		      /* Mark the client as started */
		      SRV_SetStartStateOfClient(i, TRUE);

		      /* Informative message */
		      sprintf(strScratch, 
#ifdef ENGLISH
			      "%s has finished registering players.", 
			      SRV_GetAddressOfClient(i));
		      printf("SERVER: %s\n", strScratch);
#endif
#ifdef FRENCH
			      "%s a fini d'enregistrer ses joueurs.", 
			      SRV_GetAddressOfClient(i));
		      printf("SERVEUR: %s\n", strScratch);
#endif
		      SRV_BroadcastTextMessage(strScratch); 

		      if (iServerMode == SERVER_REGISTERING)
			{
			  /* Reset the game, and possible start it */
			  SRV_ResetGame();
			  /* The first AI-Client can't start the game */
			  if (    (SRV_GetTypeOfClient(i) != CLIENT_AI)
			       || (SRV_GetNumClientsStarted() > 1))
			    SRV_AttemptNewGame();
			}
		      else
			{
			  /* Some random client tried to start in the middle 
			   * of a game, probably an overeager AI client.
			   */
			  ; 
			}
		    }
		    break;

		  case MSG_DEREGISTERCLIENT:
		    {
		      /* Do the actual deregistration */
		      SRV_HandleDEREGISTERCLIENT(i);
		      
		      /* Informative message */
#ifdef ENGLISH
		      sprintf(strScratch, "%s has deregistered.", 
			      SRV_GetAddressOfClient(i));
		      printf("SERVER: %s\n", strScratch);
#endif
#ifdef FRENCH
		      sprintf(strScratch, "%s a quitt.", 
			      SRV_GetAddressOfClient(i));
		      printf("SERVEUR: %s\n", strScratch);
#endif
		      SRV_BroadcastTextMessage(strScratch);
		    }
		    break;

		  case MSG_ENDTURN:
		    {
		      Int32 iPlayer;

		      /* Sanity check */
		      D_Assert(RISK_GetNumLivePlayers(), "Modulo by zero!");

  		      if (iServerMode == SERVER_FORTIFYING)
  			{
  			  Int32  i;
  			  Flag   fFoundPlayer;
  
  			  /* If noone has any more armies, then move to
  			   * SERVER_PLAYING mode.  Otherwise, go along 
  			   * the list of players, looking for a player who 
  			   * still has armies to place.
  			   */
  
  			  for (i=0, fFoundPlayer=FALSE; 
  			       i!=RISK_GetNumLivePlayers() && !fFoundPlayer; 
  			       i++)
  			    {
  			      /* The next player who's turn it is */ 
  			      iPlayer = SRV_IterateTurn();
  
  			      /* Is there a player with armies left? */
  			      if (RISK_GetNumArmiesOfPlayer(iPlayer) != 0)
  				{
  				  SRV_NotifyClientsOfTurn(iPlayer);
  				  fFoundPlayer = TRUE;
  				}
  			    }
  
  			  /* If we have not found a player with armies, 
  			   * then there are no longer any players with any 
  			   * armies to fortify with.  Let the game begin!!  
  			   * Theorem (proof left to reader):  When some 
  			   * players have more armies to to fortify with 
  			   * than others (because they got less countries), 
  			   * then the last player to fortify here will be 
  			   * the last player in relation to the first player 
  			   * who fortified.  So if we move to the next 
  			   * player, it will be the player who fortified 
  			   * first, which is what we want.
  			   */
  			  
  			  if (!fFoundPlayer)
  			    {
  			      iServerMode = SERVER_PLAYING;
  			      iPlayer = SRV_IterateTurn();
  			      SRV_NotifyClientsOfTurn(iPlayer);
  			    }
  			}
  		      else /* iServerMode == SERVER_PLAYING */
  			{
  			  /* Get the next player */
			  iPlayer = SRV_IterateTurn();
			  SRV_NotifyClientsOfTurn(iPlayer);
  			} 
		    }
		    break;
		    
		  case MSG_ENTERSTATE:
		    SRV_HandleENTERSTATE(pvMessage);
		    break;
		    
		  case MSG_EXCHANGECARDS:
		    {
		      MsgExchangeCards *pMess = (MsgExchangeCards *)pvMessage;
		      MsgReplyPacket    msgMess;
		      Int32             iNumJokers;
		      
		      /* Put cards back on the deck and change them to type */
		      for (n=iNumJokers=0; n!=3; n++)
			{
			  DECK_PutCard(pCardDeck, pMess->piCards[n]);
			  
			  if (pMess->piCards[n] < NUM_COUNTRIES)
			    pMess->piCards[n] %= 3;
			  else
			    pMess->piCards[n] = -1, iNumJokers++;
			}
		      
		      /* Find out how many armies the player gets in 
		       * exchange for the cards and send them to him or
		       * her, in an _UPDATEARMIES message.  Right now
		       * the only option is fixed return values for card
		       * exchanges.
		       */
		      
		      /* Do we have one of each (possibly with jokers)? */
		      if ((pMess->piCards[0] != pMess->piCards[1] &&
			   pMess->piCards[1] != pMess->piCards[2] &&
			   pMess->piCards[0] != pMess->piCards[2]) ||
			  iNumJokers >= 2)
			{
			  msgMess.iReply = 10;
			}
		      else if (pMess->piCards[0]==0 ||
			       pMess->piCards[1]==0 ||
			       pMess->piCards[2]==0)
			{
			  msgMess.iReply = 8;  
			}
		      else if (pMess->piCards[0]==1 ||
			       pMess->piCards[1]==1 ||
			       pMess->piCards[2]==1)
			{
			  msgMess.iReply = 6;  
			}
		      else 
			{
			  msgMess.iReply = 4;  
			}
		      
		      (void)RISK_SendMessage(SRV_GetCommLinkOfClient(i), 
					     MSG_REPLYPACKET, &msgMess);
		    }
		    break;
		    
		  case MSG_REQUESTCARD:
		    {
		      MsgRequestCard *pMess = (MsgRequestCard *)pvMessage;
		      
		      RISK_SetCardOfPlayer(pMess->iPlayer, 
					   RISK_GetNumCardsOfPlayer
					   (pMess->iPlayer),
					   DECK_GetCard(pCardDeck));
		      RISK_SetNumCardsOfPlayer(pMess->iPlayer,
					       RISK_GetNumCardsOfPlayer
					       (pMess->iPlayer)+1);
		    }
		    break;
		    
		  case MSG_FORCEEXCHANGECARDS:
		    {
		      Int32 iPlayer ;

		      iPlayer = ((MsgForceExchangeCards *)pvMessage)->iPlayer;
		      if (RISK_GetNumCardsOfPlayer(iPlayer) < 5)
		        ((MsgForceExchangeCards *)pvMessage)->iPlayer = -1;
		      (void)RISK_SendMessage(SRV_GetCommLinkOfClient(i), 
					     MSG_FORCEEXCHANGECARDS,
					     pvMessage);
		    }
		    break;

		  case MSG_EXIT:
		    SRV_HandleEXIT(0);
		    break;
		  
		  case MSG_NOMESSAGE:
		    break;

		  default:
		    {
		      MsgNetPopup msg;

		      /* Assume that client is messed up.  Consider it 
		       * a failure and kill the client.
		       */

#ifdef ENGLISH
		      printf("SERVER: Considering client %s dead (it send a"
			     " bogus message.)\n",
#endif
#ifdef FRENCH
		      printf("SERVEUR: Le client %s est considr comme mort"
		             " (il a mis un mauvais message.)\n",
#endif
		             SRV_GetAddressOfClient(i));

		      /* In case the client is alive and well, send it a 
		       * message to inform it why it is being axed.
		       */
		      
		      sprintf(strScratch,
#ifdef ENGLISH
			    "The server received an invalid message from me.");
#endif
#ifdef FRENCH
			    "Le serveur a reu un message invalide de ma part.");
#endif
		      msg.strMessage = strScratch;
		      (void)RISK_SendMessage(SRV_GetCommLinkOfClient(i), 
					     MSG_NETPOPUP, &msg);
		      
		      /* Log the failure */
		      SRV_LogFailure("Sent bogus message", 
				     SRV_GetCommLinkOfClient(i));
		    }
		  }
		
		/* Free up the memory the message was taking */
		NET_DeleteMessage(iMessageType, pvMessage);

		D_Assert(i != MAX_CLIENTS && fHandledMessage == TRUE, 
			 "Message received from unknown source!!");
	      }
	}
    }
}


/************************************************************************ 
 *  FUNCTION: SRV_NotifyClientsOfTurn
 *  HISTORY: 
 *     08.27.94  ESF  Created.
 *     30.08.95  JC   Send a message if the player have too many cards.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_NotifyClientsOfTurn(Int32 iPlayer)
{
  MsgTurnNotify msgTurnNotify;

  /* Let all clients know whos turn it is */
  msgTurnNotify.iPlayer = iPlayer;
  msgTurnNotify.iClient = RISK_GetClientOfPlayer(iPlayer);
  SRV_BroadcastMessage(MSG_TURNNOTIFY, &msgTurnNotify);
}
  

/************************************************************************ 
 *  FUNCTION: SRV_DistributeCountries
 *  HISTORY: 
 *     05.12.94  ESF  Created.
 *     07.25.94  ESF  Changed to support TEST_GAMEs.
 *     10.30.94  ESF  Changed to support TEST_GAMEs better.
 *     10.30.94  ESF  Changed to refer to LivePlayer instead of Player.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_DistributeCountries(void)
{
  Deck   *pCountryDeck = DECK_Create(NUM_COUNTRIES);
  Int32   iCountry, iPlayer, i;

  /* Dole them out starting with the first player */
  iTurn = RISK_GetNthLivePlayer(0);

  /* Dole out the countries */
  for(i=0; i!=NUM_COUNTRIES; i++)
    {
      /* Pick a country, any country */
      iCountry = DECK_GetCard(pCountryDeck);

#ifdef TEST_GAME
      /* Give countries to the first player, leave one for the rest */
      iPlayer = (i <= NUM_COUNTRIES-RISK_GetNumLivePlayers()) 
	  ? RISK_GetNthLivePlayer(0) 
	  : RISK_GetNthLivePlayer(i-(NUM_COUNTRIES-RISK_GetNumLivePlayers()));
#else      
      iPlayer = iTurn;
#endif

      /* Update the game object */
      RISK_SetOwnerOfCountry(iCountry, iPlayer);
      RISK_SetNumCountriesOfPlayer(iPlayer, 
				   RISK_GetNumCountriesOfPlayer(iPlayer)+1);
      RISK_SetNumArmiesOfCountry(iCountry, 1);
      RISK_SetNumArmiesOfPlayer(iPlayer, RISK_GetNumArmiesOfPlayer(iPlayer)-1);

      /* Iterate to next player */
      (void)SRV_IterateTurn();
    }

#ifdef TEST_GAME
  /* Set the number of armies to be small, to let the game start soon. */
  for (i=0; i!=RISK_GetNumLivePlayers(); i++)
    RISK_SetNumArmiesOfPlayer(RISK_GetNthLivePlayer(i), 2);
#endif
  
  DECK_Destroy(pCountryDeck);  
}


/************************************************************************ 
 *  FUNCTION: SRV_SetInitialArmiesOfPlayers
 *  HISTORY: 
 *     08.27.94  ESF  Created.
 *     12.07.95  ESF  Fixed initial number of armies to be by the rules.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_SetInitialArmiesOfPlayers(void)
{
  Int32 i, iPlayer, iNumArmies;
  const Int32 iNumPlayers = RISK_GetNumPlayers();

  D_Assert(iNumPlayers >= 2, "Not enough players!");

  /* Calculate the number of armies. */
  iNumArmies = MAX( 50 - iNumPlayers*5, /* According to the rules */
		    NUM_COUNTRIES/iNumPlayers+1 ); /* At least one per country */

  /* Make sure it's enough */
  D_Assert(iNumPlayers*iNumPlayers >= NUM_COUNTRIES, "Not enough armies!");

  /* Set the initial number of armies for all the players */
  for (i=0; i<RISK_GetNumPlayers(); i++)
    {
      iPlayer = RISK_GetNthPlayer(i);
      RISK_SetNumArmiesOfPlayer(iPlayer, iNumArmies);

      /* Sanity check */
      D_Assert(RISK_GetNumArmiesOfPlayer(iPlayer) > 0,
	       "Number of armies is negative!");
    }
}


/************************************************************************ 
 *  FUNCTION: SRV_SetInitialMissionOfPlayers
 *  HISTORY: 
 *     24.08.95  JC   Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_SetInitialMissionOfPlayers(void)
{
  Int32 i, iPlayer, nb;

  nb = RISK_GetNumPlayers();
  /* Set the initial mission's type for all the players */
  for (i=0; i<nb; i++)
    {
      iPlayer = RISK_GetNthPlayer(i);
      RISK_SetMissionTypeOfPlayer(iPlayer, NO_MISSION);
    }
}


/************************************************************************ 
 *  FUNCTION: SRV_DistributeMissions
 *  HISTORY: 
 *     24.08.94  JC   Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Flag SRV_DistributeMissions(void)
{
  Int32 iMission[MAX_PLAYERS];
  Int32 iPlayer, i, j, cont1, cont2, n, nb, m;

  nb = RISK_GetNumLivePlayers();
  for (n = 0; n<nb; n++)
    {
      iPlayer = RISK_GetNthPlayer(n);
      if (RISK_GetMissionTypeOfPlayer(iPlayer) != NO_MISSION)
          return FALSE;
    }
  for (n = 0; n<MAX_PLAYERS; n++)
      iMission[n] = -1;
  for (n = 0; n<nb; n++)
    {
      iPlayer = RISK_GetNthLivePlayer(n);
      i = rand() % (2 + (NUM_CONTINENTS * (NUM_CONTINENTS - 1))/2 + nb - 1);
      j = 0;
      while (j<MAX_PLAYERS)
        {
          if (i == iMission[j])
            {
              j = 0;
              i = (i + 1) %(2 + NUM_CONTINENTS * (NUM_CONTINENTS - 1)
                            + nb - 1);
            }
          else
              j++;
        }
      iMission[iPlayer]=i;
      if (i == 0)
          RISK_SetMissionTypeOfPlayer(iPlayer, CONQUIER_WORLD);
      else if (i == 1)
        {
          RISK_SetMissionTypeOfPlayer(iPlayer, CONQUIER_Nb_COUNTRY);
          RISK_SetMissionNumberOfPlayer(iPlayer, NUM_COUNTRIES / 2);
        }
      else if (i <= ((NUM_CONTINENTS * (NUM_CONTINENTS - 1))/2 + 1))
        {
          i = i - 2;
          cont1 = -1;
          m = 0;
          while (cont1 == -1)
            {
              cont2 = m + i + 1;
              if (cont2 < NUM_CONTINENTS)
                  cont1 = m;
              i = i - (NUM_CONTINENTS - m - 1);
              m++;
            }
          RISK_SetMissionTypeOfPlayer(iPlayer, CONQUIER_TWO_CONTINENTS);
          RISK_SetMissionContinent1OfPlayer(iPlayer, cont1);
          RISK_SetMissionContinent2OfPlayer(iPlayer, cont2);
        }
      else
        {
          i = i - (NUM_CONTINENTS * (NUM_CONTINENTS - 1))/2 - 2;
          j = RISK_GetNthLivePlayer(i);
          if (j == iPlayer)
            {
              i++;
              i = i % nb;
              j = RISK_GetNthLivePlayer(i);
            }
          RISK_SetMissionTypeOfPlayer(iPlayer, KILL_A_PLAYER);
          RISK_SetMissionMissionPlayerToKillOfPlayer(iPlayer, j);
          RISK_SetMissionPlayerIsKilledOfPlayer(iPlayer, FALSE);
        }
    }
  return TRUE;
}


/************************************************************************ 
 *  FUNCTION: SRV_BroadcastMessage
 *  HISTORY: 
 *     02.05.94  ESF  Created.
 *     08.16.94  ESF  Rewrote a bit.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_BroadcastMessage(Int32 iMessType, void *pvMessage)
{
  Int32 i;
  
  /* If there are no clients, then do nothing */
  if (SRV_GetNumClients() == 0)
    return;

  /* Loop through and send the message to each active client */
  for (i=0; i!=MAX_CLIENTS; i++)
    if (SRV_GetAllocationStateOfClient(i) == ALLOC_COMPLETE)
      (void)RISK_SendMessage(SRV_GetCommLinkOfClient(i), iMessType, pvMessage);
}


/************************************************************************ 
 *  FUNCTION: SRV_CompleteBroadcast
 *  HISTORY: 
 *     03.28.94  ESF  Created.
 *     08.16.94  ESF  Rewrote a bit.
 *     10.02.94  ESF  Fixed a heinous bug, was going 'till NumClients.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_CompleteBroadcast(Int32 iClientExclude, Int32 iMessageType, 
			   void *pvMess)
{
  Int32 i;

  for (i=0; i!=MAX_CLIENTS; i++)
    if (i!=iClientExclude && 
	SRV_GetAllocationStateOfClient(i) == ALLOC_COMPLETE)
      (void)RISK_SendMessage(SRV_GetCommLinkOfClient(i), iMessageType, pvMess);
}


/************************************************************************ 
 *  FUNCTION: SRV_Replicate
 *  HISTORY: 
 *     08.18.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_Replicate(Int32 iMessType, void *pvMess, Int32 iType, Int32 iSrc)
{
  if (iType == MESS_OUTGOING)
    SRV_BroadcastMessage(iMessType, pvMess);
  else /* (iType == MESS_INCOMING) */
    {
      /* Convert the CommLink to a client */
      SRV_CompleteBroadcast(SRV_CommLinkToClient(iSrc), iMessType, pvMess);
    }
}


/************************************************************************ 
 *  FUNCTION: SRV_HandleEXIT
 *  HISTORY: 
 *     06.10.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_HandleEXIT(Int32 iExitValue)
{
  Int32 n;

  /* close the listening socket */
  close(iServerCommLink);

  /* close all of the player sockets */
  for(n=0; n!=MAX_CLIENTS; n++)
    if (SRV_GetAllocationStateOfClient(n) == ALLOC_COMPLETE)
      {
        RISK_SendMessage(SRV_GetCommLinkOfClient(n), MSG_EXIT, NULL);
        close(SRV_GetCommLinkOfClient(n));
      }

  UTIL_ExitProgram(iExitValue);
}


/************************************************************************ 
 *  FUNCTION: SRV_HandleALLOCPLAYER
 *  HISTORY: 
 *     06.10.94  ESF  Created.
 *     10.04.94  ESF  Fixed a bug, creating player not a transaction.
 *     04.01.95  ESF  Changed to use a more robust method.  Simplified.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_HandleALLOCPLAYER(Int32 iClient)
{
  MsgReplyPacket  mess;
  Int32           iPlayer;
  
  /* Get a player and return it, or -1 if none available */
  iPlayer = mess.iReply = DECK_GetCard(pPlayerDeck);
  
  /* If there was a valid player, make a note */
  if (iPlayer != -1)
    {
      /* Reset all of the player fields */
      RISK_SetAttackModeOfPlayer(iPlayer, 1);
      RISK_SetStateOfPlayer(iPlayer, PLAYER_ALIVE);
      RISK_SetClientOfPlayer(iPlayer, -1);
      RISK_SetNumCountriesOfPlayer(iPlayer, 0);
      RISK_SetNumArmiesOfPlayer(iPlayer, 0);
      RISK_SetNumCardsOfPlayer(iPlayer, 0);
      RISK_SetMissionTypeOfPlayer(iPlayer, NO_MISSION);

      /* Note that the player is in the process of being allocated.
       * The client will complete the procedure.  The client that
       * sets the allocation state to ALLOC_COMPLETE is the one
       * responsible for increasing the number of players.
       */

      RISK_SetAllocationStateOfPlayer(iPlayer, ALLOC_INPROGRESS);
    }
  
  RISK_SendMessage(SRV_GetCommLinkOfClient(iClient), MSG_REPLYPACKET, &mess);
}


/************************************************************************ 
 *  FUNCTION: SRV_HandleREPLYPACKET
 *  HISTORY: 
 *     06.10.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_HandleREPLYPACKET(void *pvMessage)
{
  iReply = ((MsgReplyPacket *)pvMessage)->iReply; 
}


/************************************************************************ 
 *  FUNCTION: SRV_HandleFREEPLAYER
 *  HISTORY: 
 *     06.10.94  ESF  Created.
 *     10.15.94  ESF  Fixed a bug, only Set LivePlayers if player is alive.
 *     11.06.94  ESF  Fixed a bug, return player's cards to server.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_HandleFREEPLAYER(void *pvMessage)
{
  Int32 iPlayer, i;

  /* Put the player ID back onto pool of free players */
  iPlayer = ((MsgFreePlayer *)(pvMessage))->iPlayer;
  DECK_PutCard(pPlayerDeck, iPlayer);

  /* If the player has cards at this point, take them away and put
   * them back onto the card deck.
   */

  for (i=0; i!=RISK_GetNumCardsOfPlayer(iPlayer); i++)
    DECK_PutCard(pCardDeck, RISK_GetCardOfPlayer(iPlayer, i));
 
  /* Details, details... */
  RISK_SetAllocationStateOfPlayer(iPlayer, ALLOC_NONE);

  /* If there were no live players at the client then we can let it go
   * without ending the game.  Otherwise, end the game if we're in the
   * PLAY or FORTIFICATION stages of the game.  If the number of live
   * players went down to 0, then we must go back to registering mode.
   * Kind of moot, but safe... 

  /* Was the player alive? */
  if (RISK_GetStateOfPlayer(iPlayer) == PLAYER_ALIVE &&
      (iServerMode == SERVER_PLAYING ||
       iServerMode == SERVER_FORTIFYING))
    {
      MsgNetPopup msgNetPopup;
      
      /* Informative message */
      sprintf(strScratch, 
#ifdef ENGLISH
	      "The game is over: a live player (or an entire client) "
	      "left the game.");
#endif
#ifdef FRENCH
	      "Le jeu est termin: un joueur vivant (ou un client complet) "
	      "quitte le jeu.");
#endif
      msgNetPopup.strMessage = strScratch;
      SRV_BroadcastMessage(MSG_NETPOPUP, &msgNetPopup);

      /* There was no winner, reset the game */
      SRV_BroadcastMessage(MSG_ENDOFGAME, NULL);

      /* It was the end of the game, so reset things */
      SRV_ResetGame();
    }
  else if (RISK_GetNumLivePlayers() == 0)
    SRV_ResetGame();
}


/************************************************************************ 
 *  FUNCTION: SRV_HandleMESSAGEPACKET
 *  HISTORY: 
 *     06.10.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_HandleMESSAGEPACKET(Int32 iClient, void *pvMessage)
{
  MsgMessagePacket *pMess = (MsgMessagePacket *)pvMessage;
  
  if (pMess->iTo == DST_ALLPLAYERS)
    SRV_BroadcastMessage(MSG_MESSAGEPACKET, pvMessage);
  else if (pMess->iTo == DST_ALLBUTME)
    SRV_CompleteBroadcast(iClient, MSG_MESSAGEPACKET, pvMessage);
  else
    (void)RISK_SendMessage(SRV_GetCommLinkOfClient(
                               RISK_GetClientOfPlayer(pMess->iTo)), 
			   MSG_MESSAGEPACKET, pvMessage);
}


/************************************************************************ 
 *  FUNCTION: SRV_HandleENTERSTATE
 *  HISTORY: 
 *     06.10.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_HandleENTERSTATE(void *pvMessage)
{
  iState = ((MsgEnterState *)pvMessage)->iState; 
}


#define _SRV_CleanupRegistration() \
  /* NET_DeleteMessage(iMessType, pvMess); */ \
  FD_CLR(iCommLink, &fdBackup);\
  close(iCommLink);

/************************************************************************ 
 *  FUNCTION: SRV_HandleRegistration
 *  HISTORY: 
 *     06.10.94  ESF  Created.
 *     08.31.94  ESF  Changed to use only one socket.
 *     09.28.94  ESF  Fixed to handle errant clients correctly (free 'em).
 *     09.28.94  ESF  Fixed the process of filling in the new clients.
 *     09.31.94  ESF  Added notification of current turn for new clients.
 *     10.01.94  ESF  Fixed to check for failure of ReceiveMessage.
 *     10.15.94  ESF  Fixed, was freeing client that I wasn't allocating.
 *     10.30.94  ESF  Added quotes to the "A new client..." message.
 *     01.01.95  ESF  Fixed a bug in notifying clients of current player.
 *     01.15.95  ESF  Initilized iTo field to avoid uninitialized memory.
 *     01.15.95  ESF  Fixed memory leak.
 *     02.21.95  ESF  Add support for computer players.
 *     02.21.95  ESF  Cleaned up, fixed minor bugs.
 *     25.08.95  JC   Add || (RISK_GetNumSpecies() > 1) because the
 *                    presence of AI-Client witch have no player.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_HandleRegistration(Int32 iCommLink)
{
  Int32                 n, iMessType;
  MsgClientIdent        msgClientIdent;
  MsgVersion            msgVersion;
  MsgMessagePacket      msgMess;
  void                 *pvMess, *pvDeleteMe = NULL;
  Int32                 iNewClient, iClientType = CLIENT_NORMAL;
  Int32                 iClientSpecies = SPECIES_HUMAN;
  Flag                  fOldClient = FALSE;
  CString               strClientAddress;

  /* The Protocol:
   *
   *                      Receive MSG_HELLO           --> OK!
   *                              Other               --> Protocol bogosity.
   *  Send MSG_VERSION. 
   *                      Receive MSG_REGISTERCLIENT  --> OK!
   *                              MSG_VERSION         --> Version mismatch.
   *                              Other               --> Protocol bogosity.
   *  Send MSG_CLIENTIDENT if server is not full.
   *  Send MSG_CLIENTEXIT  if server is full.
   */

  /* It should be a MSG_HELLO! */
  if (!RISK_ReceiveMessage(iCommLink, &iMessType, &pvMess) ||
      (iMessType != MSG_HELLO && iMessType != MSG_OLDREGISTERCLIENT))
    {
#ifdef ENGLISH
      printf("SERVER: Error in protocol (%d).  Ignoring wanna-be client.\n", 
#endif
#ifdef FRENCH
      printf("SERVEUR: Erreur de protocol (%d).  Ignore ce client.\n", 
#endif
	     iMessType);
      _SRV_CleanupRegistration();
      return;
    }

  /* See if it's an old client */
  fOldClient = (iMessType == MSG_OLDREGISTERCLIENT);
  
  /* If it is an old client, then get its name */
  if (fOldClient)
    {
      strClientAddress = ((MsgOldRegisterClient *)pvMess)->strClientAddress;
      
      /* Mark this message for future deletion */
      pvDeleteMe = pvMess;
    }
  else
    NET_DeleteMessage(MSG_HELLO, pvMess);

  /* If we're talking to a new client, send the version information */
  if (!fOldClient)
    {
      /* Send the version */
      msgVersion.strVersion = VERSION;
      (void)RISK_SendMessage(iCommLink, MSG_VERSION, &msgVersion);
      
      /* It should be a MSG_REGISTERCLIENT or a MSG_VERSION! */
      if (!RISK_ReceiveMessage(iCommLink, &iMessType, &pvMess) ||
	  iMessType != MSG_REGISTERCLIENT)
	{
	  if (iMessType == MSG_VERSION)
#ifdef ENGLISH
	    printf("SERVER: Client version mismatch (%s).\n", 
#endif
#ifdef FRENCH
	    printf("SERVEUR: Mauvaise version de client (%s).\n", 
#endif
		   ((MsgVersion *)pvMess)->strVersion);
	  else
#ifdef ENGLISH
	    printf("SERVER: Error in connect protocol (%d).  "
		   "Ignoring client.\n", iMessType);
#endif
#ifdef FRENCH
	    printf("SERVEUR: Erreur dans le protocol de connexion(%d).  "
		   "Client ignor.\n", iMessType);
#endif
	  _SRV_CleanupRegistration();
	  return;
	}

      /* Now we know the name (and type) of the client */
      strClientAddress = ((MsgRegisterClient *)pvMess)->strClientAddress;
      iClientType      = ((MsgRegisterClient *)pvMess)->iClientType;
    }

  /* If the client is an "AI" client, then send it the 
   * dynamic ID of its species, which we allocate on-the-fly. 
   */

  if (iClientType == CLIENT_AI)
    {
      MsgSpeciesIdent msg;

      iClientSpecies = msg.iSpeciesID = SRV_AllocSpecies();
      (void)RISK_SendMessage(iCommLink, MSG_SPECIESIDENT, &msg);
    }
  
  /* Get the handle to a new client */
  iNewClient = SRV_AllocClient();
  
  /* Did the allocation fail? */
  if (iNewClient == -1)
    {
#ifdef ENGLISH
      printf("%s: Unable to allocate room for new client!",SERVERNAME);
#endif
#ifdef FRENCH
      printf("%s: Plus de place pour un nouveau client!",SERVERNAME);
#endif
      (void)RISK_SendMessage(iCommLink, MSG_EXIT, NULL);
      _SRV_CleanupRegistration();
      return;
    }

  /* Send its ID */
  msgClientIdent.iClientID = iNewClient;
  (void)RISK_SendMessage(iCommLink, MSG_CLIENTIDENT, &msgClientIdent);
#ifdef ENGLISH
  printf("SERVER: A new client, \"%s\", has registered.\n",
#endif
#ifdef FRENCH
  printf("SERVEUR: Un nouveau client, \"%s\", est enregistr.\n",
#endif
        strClientAddress);

  /* Set up its fields */
  SRV_SetCommLinkOfClient(iNewClient, iCommLink);
  SRV_SetAddressOfClient(iNewClient, strClientAddress);
  SRV_SetTypeOfClient(iNewClient, iClientType);
  SRV_SetSpeciesOfClient(iNewClient, iClientSpecies);
  
  /* Remember to look for messages from it with select(), 
   * and update the maximum file descriptor.
   */
  
  FD_SET(iCommLink, &fdBackup);
  iMaxFileDescUsed = MAX(iMaxFileDescUsed, iCommLink);
  
  /* Let the new client know about old clients, if needed */
  if (SRV_GetNumClients() > 1 && iClientType != CLIENT_AI)
    {
      /* Let the new client know about old clients */
      msgMess.strMessage = strScratch;
      msgMess.iFrom      = FROM_SERVER;
      msgMess.iTo        = DST_OTHER;
      
      /* Send info about everybody but the new client. */
      for (n=0; n!=MAX_CLIENTS; n++)
	if (SRV_GetAllocationStateOfClient(n) == ALLOC_COMPLETE &&
	    SRV_GetCommLinkOfClient(n) != iCommLink)
	  {
#ifdef ENGLISH
	    sprintf(strScratch, "The client \"%s\", is registered.",
#endif
#ifdef FRENCH
	    sprintf(strScratch, "Le client \"%s\", est enregistr.",
#endif
		    SRV_GetAddressOfClient(n));
	    
	    (void)RISK_SendMessage(iCommLink, MSG_MESSAGEPACKET, &msgMess);
	  }
    }
      
  /* Let all other clients know about it */
  if (iClientType == CLIENT_AI)
#ifdef ENGLISH
    sprintf(strScratch, "A new AI client, %s,  has registered.",
#endif
#ifdef FRENCH
    sprintf(strScratch, "Un nouveau client AI, %s, est enregistr.",
#endif
	    SRV_GetAddressOfClient(iNewClient));
  else
#ifdef ENGLISH
    sprintf(strScratch, "A new client, %s, has registered.",
#endif
#ifdef FRENCH
    sprintf(strScratch, "Un nouveau client, %s, est enregistr.",
#endif
	    SRV_GetAddressOfClient(iNewClient));

  msgMess.strMessage = strScratch;
  msgMess.iFrom      = FROM_SERVER;
  msgMess.iTo        = DST_OTHER;
  SRV_CompleteBroadcast(iNewClient, MSG_MESSAGEPACKET, &msgMess);

  /* And tell the client that just joined about itself */
#ifdef ENGLISH
  sprintf(strScratch, "Hello new client, you are `%s'.", 
#endif
#ifdef FRENCH
  sprintf(strScratch, "Bonjour nouveau client, vous tes `%s'.", 
#endif
	  SRV_GetAddressOfClient(iNewClient));
  RISK_SendMessage(SRV_GetCommLinkOfClient(iNewClient), 
		   MSG_MESSAGEPACKET, &msgMess);

  /* Tell the new client to pop up its registration box, if we
   * are not in game playing mode.
   */
  
  if (iServerMode == SERVER_REGISTERING && iClientType != CLIENT_AI)
    (void)RISK_SendMessage(iCommLink, MSG_POPUPREGISTERBOX, NULL);
  
  /* If there are any players, let the new client 
   * know about them, so that it can set up the 
   * colors and other things.
   */
  
  if (RISK_GetNumPlayers() || (RISK_GetNumSpecies() > 1))
    {
      /* Let the clients know the situation */
#ifdef ENGLISH
      sprintf(strScratch, "The client %s is being updated.",
#endif
#ifdef FRENCH
      sprintf(strScratch, "Le client %s va tre mis  jour.",
#endif
	      SRV_GetAddressOfClient(iNewClient));
      SRV_BroadcastTextMessage(strScratch);
      
      if (iServerMode == SERVER_REGISTERING)
	SRV_ReplicateRegistrationData(iCommLink);
      else if (iServerMode == SERVER_PLAYING ||
	       iServerMode == SERVER_FORTIFYING)
	{
	  MsgTurnNotify  msgTurnNotify;
	  
	  SRV_ReplicateAllData(iCommLink);
	  
	  /* Let the client know who's turn it is currently */
	  msgTurnNotify.iPlayer = iTurn;
	  msgTurnNotify.iClient = 
	    RISK_GetClientOfPlayer(msgTurnNotify.iPlayer);
	  (void)RISK_SendMessage(iCommLink, MSG_TURNNOTIFY, 
				 &msgTurnNotify);
	}
    }      
  
  /* Don't need these anymore */
  if (pvDeleteMe)
    NET_DeleteMessage(MSG_OLDREGISTERCLIENT, pvDeleteMe);
  else
    NET_DeleteMessage(MSG_REGISTERCLIENT, pvMess);
}


/************************************************************************ 
 *  FUNCTION: SRV_HandleDEREGISTERCLIENT
 *  HISTORY: 
 *     06.10.94  ESF  Created.
 *     08.31.94  ESF  Revamped, made more sophisticated.
 *     09.30.94  ESF  Fixed bug, freeing player before MSG_DELETEMSGDST.
 *     09.31.94  ESF  Fixed so that FreeClient is called last.
 *     10.01.94  ESF  Fixed so that it doesn't return too soon.
 *     01.15.95  ESF  Removed MSG_DELETEMSGDST stuff.
 *     01.24.95  ESF  Fixed bug, reset iServerMode if SRV_NumClients()==1.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_HandleDEREGISTERCLIENT(Int32 iClient)
{
  const Int32 iNumPlayersOfClient = RISK_GetNumLivePlayersOfClient(iClient);

  D_Assert(iClient>=0 && iClient<MAX_CLIENTS, "Bogus client!");
  /* We need to free the client */
  SRV_FreeClient(iClient);
}


/************************************************************************ 
 *  FUNCTION: UTIL_ExitProgram
 *  HISTORY: 
 *     06.16.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void UTIL_ExitProgram(Int32 iExitValue)
{
  MEM_TheEnd();

  /* Don't let any more data be received on this socket */
  shutdown(iServerCommLink, 0);

  exit(iExitValue);
}


/************************************************************************ 
 *  FUNCTION: SRV_HandleSignals
 *  HISTORY: 
 *     07.31.94  ESF  Created.
 *     24.08.95  JC   Don't quit if a game is started, but remember.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_HandleSignals(Int32 iParam)
{
  Int32          i;
  MsgNetPopup    msg;

#ifdef ENGLISH
  printf("Ouch, I've been killed...\n");
#endif
#ifdef FRENCH
  printf("Ouch, Je suis tu...\n");
#endif

  if (    (    (iServerMode != SERVER_FORTIFYING)
            && (iServerMode != SERVER_PLAYING   ))
       || (    (iParam != SIGHUP ) && (iParam != SIGINT)
            && (iParam != SIGQUIT) && (iParam != SIGKILL))
       || (SRV_GetNumClients() <= 0))
      SRV_HandleEXIT(-1);

  for (i=0; i!=MAX_CLIENTS; i++)
    if (    (SRV_GetAllocationStateOfClient(i) == ALLOC_COMPLETE)
         && (SRV_GetStartStateOfClient(i) == TRUE))
      {
        RISK_SendMessage(SRV_GetCommLinkOfClient(i),
                         MSG_DEREGISTERCLIENT, NULL);
        close(SRV_GetCommLinkOfClient(i));
      }
  fRememberKilled = TRUE;
  select(iMaxFileDescUsed+1, (FDSET *)&fdSet, (FDSET *)0, (FDSET *)0, NULL);
  signal(iParam, SRV_HandleSignals);
}


/************************************************************************ 
 *  FUNCTION: SRV_AllocClient
 *  HISTORY: 
 *     05.12.94  ESF  Created.
 *     08.08.94  ESF  Moved here from server.c
 *     08.21.94  ESF  Fixed to work.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int32 SRV_AllocClient(void)
{
  Int32 i;

  /* Since we wouldn't have accepted the connection if there weren't
   * a slot available, this call cannot fail.
   */

  D_Assert(SRV_GetNumClients() < MAX_CLIENTS, "Not good!");
  SRV_SetNumClients(SRV_GetNumClients()+1);
  
  /* Find an available client */
  for (i=0; i!=MAX_CLIENTS; i++)
    if (SRV_GetAllocationStateOfClient(i) == ALLOC_NONE)
      {
	SRV_SetAllocationStateOfClient(i, ALLOC_COMPLETE);
	SRV_SetCommLinkOfClient(i, -1);
	SRV_SetStartStateOfClient(i, FALSE);
	return (i);
      }

  D_Assert(FALSE, "Something wierd happened!");
  
  /* For the compiler */
  return (0);
}


/************************************************************************ 
 *  FUNCTION: SRV_FreeClient
 *  HISTORY: 
 *     05.12.94  ESF  Created.
 *     08.08.94  ESF  Moved here from server.c.
 *     08.08.94  ESF  Fixed to delete players at client.
 *     09.09.94  ESF  Fixed to take into consideration side effects.
 *     09.13.94  ESF  Fixed loop, was doing i<=0, changed to i>=0 :)
 *     09.14.94  ESF  Fixed to deal with iNumClientsStarted.
 *     03.25.95  ESF  Added species considerations.
 *     24.08.95  JC   Quit if no client and fRememberKilled == TRUE.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_FreeClient(Int32 iClient)
{
  Int32            i, iNumPlayers;
  MsgFreePlayer    msgFreePlayer;

  /* Get rid of the client.  We do this first so that broadcasts
   * won't include this client.  This would normally result in
   * a SIGPIPE, if the exiting client had already shut down its
   * sockets, but since we ignore SIGPIPE, it would result in
   * a failed broadcast, which is technically wrong.
   */

  SRV_SetAllocationStateOfClient(iClient, ALLOC_NONE);
  SRV_SetNumClients(SRV_GetNumClients()-1);

  /* Note that the client won't be participating in the game */
  SRV_SetStartStateOfClient(iClient, FALSE);

  /* Reset the fields, close the sockets */
  close(SRV_GetCommLinkOfClient(iClient));

  /* Remove the socket from the fd_set */
  FD_CLR(SRV_GetCommLinkOfClient(iClient), &fdBackup);

  /* Free all of the players at the client that is being nixed.
   * The client would call CLNT_FreePlayer to do this, which
   * is an RPC type call that sends MSG_FREEPLAYER to the
   * server (that's me!)  So what we do here is manufacture a
   * MSG_FREEPLAYER and call the handler for it.  A bit roundabout,
   * perhaps, but nevertheless a consistant way of dealing with
   * freeing players.  Could probably be improved.  Notice that
   * we get the Nth player, and _not_ the Nth live player.
   * Note also, that freeing a player has the side effect of
   * changing the value of RISK_GetNumPlayers().
   */

  iNumPlayers = RISK_GetNumPlayers();

  for (i=iNumPlayers-1; i>=0; i--)
    if (RISK_GetClientOfPlayer(RISK_GetNthPlayer(i)) == iClient)
      {
	msgFreePlayer.iPlayer = RISK_GetNthPlayer(i);
	SRV_HandleFREEPLAYER((void *)&msgFreePlayer);
      }

  /* Additionally, if the client is an AI client, then we have to
   * remove the species from the list of species.
   */

  if (SRV_GetTypeOfClient(iClient) == CLIENT_AI)
    SRV_FreeSpecies(SRV_GetSpeciesOfClient(iClient));

  if (fRememberKilled && (SRV_GetNumClientsStarted() <= 0))
      SRV_HandleEXIT(-1);

  if (iServerMode == SERVER_REGISTERING)
    {
      /* Reset the game, and possible start it */
      SRV_ResetGame();
      SRV_AttemptNewGame();
    }
}



/***************************************/
void SRV_SetAddressOfClient(Int32 iNumClient, CString strAddress)
{
  D_Assert(iNumClient>=0 && iNumClient<MAX_CLIENTS, 
	   "Client out of range!");

  if (pClients[iNumClient].strAddress != NULL)
    MEM_Free(pClients[iNumClient].strAddress);
  pClients[iNumClient].strAddress = 
    (CString)MEM_Alloc(strlen(strAddress)+1);
  strcpy(pClients[iNumClient].strAddress, 
	 strAddress);
}

/***************************************/
void SRV_SetAllocationStateOfClient(Int32 iNumClient, Int32 iState)
{
  D_Assert(iNumClient>=0 && iNumClient<MAX_CLIENTS, 
	   "Client out of range!");

  pClients[iNumClient].iState = iState;
}

/***************************************/
void SRV_SetCommLinkOfClient(Int32 iNumClient, Int32 iCommLink)
{
  D_Assert(iNumClient>=0 && iNumClient<MAX_CLIENTS, 
	   "Client out of range!");
  
  pClients[iNumClient].iCommLink = iCommLink;
}

/***************************************/
void SRV_SetTypeOfClient(Int32 iNumClient, Int32 iType)
{
  D_Assert(iNumClient>=0 && iNumClient<MAX_CLIENTS, 
	   "Client out of range!");
  
  pClients[iNumClient].iType = iType;
}

/***************************************/
void SRV_SetSpeciesOfClient(Int32 iNumClient, Int32 iSpecies)
{
  D_Assert(iNumClient>=0 && iNumClient<MAX_CLIENTS, 
	   "Client out of range!");
  
  pClients[iNumClient].iSpecies = iSpecies;
}

/***************************************/
void SRV_SetNumClients(Int32 iClients)
{
  iNumClients = iClients;
}

/***************************************/
CString SRV_GetAddressOfClient(Int32 iNumClient)
{
  D_Assert(iNumClient>=0 && iNumClient<MAX_CLIENTS, 
	   "Client out of range!");

  return pClients[iNumClient].strAddress;
}

/***************************************/
Int32 SRV_GetTypeOfClient(Int32 iNumClient)
{
  D_Assert(iNumClient>=0 && iNumClient<MAX_CLIENTS, 
	   "Client out of range!");

  return pClients[iNumClient].iType;
}

/***************************************/
Int32 SRV_GetCommLinkOfClient(Int32 iNumClient)
{
  D_Assert(iNumClient>=0 && iNumClient<MAX_CLIENTS, 
	   "Client out of range!");

  return pClients[iNumClient].iCommLink;
}


/***************************************/
Int32 SRV_GetAllocationStateOfClient(Int32 iNumClient)
{
  D_Assert(iNumClient>=0 && iNumClient<MAX_CLIENTS, 
	   "Client out of range!");

  return pClients[iNumClient].iState;
}

/***************************************/
Int32 SRV_GetSpeciesOfClient(Int32 iNumClient)
{
  D_Assert(iNumClient>=0 && iNumClient<MAX_CLIENTS, 
	   "Client out of range!");

  return pClients[iNumClient].iSpecies;
}

/***************************************/
Int32 SRV_GetNumClients(void)
{
  return iNumClients;
}


/************************************************************************ 
 *  FUNCTION: SRV_CommLinkToClient
 *  HISTORY: 
 *     09.01.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int32 SRV_CommLinkToClient(Int32 iCommLink)
{
  Int32 i;

  D_Assert(iCommLink>=0 && iCommLink<MAX_DESCRIPTORS, "Bogus CommLink!");

  for (i=0; i!=MAX_CLIENTS; i++)
    if ((SRV_GetAllocationStateOfClient(i) == ALLOC_COMPLETE) &&
	(SRV_GetCommLinkOfClient(i) == iCommLink))
      return (i);

  return (-1);
}


/************************************************************************ 
 *  FUNCTION: _SRV_GetNthClient
 *  HISTORY: 
 *     08.21.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Client *_SRV_GetNthClient(Int32 iIndex)
{
  Int32 i, iNumFound;

  /* Find the nth client and return it.  It is an error to call
   * this function with an index that is out of range.
   */

  D_Assert(iIndex>=0 && iIndex<MAX_CLIENTS,
	   "_SRV_GetNthClient called with bogus parameter!");

  for (i=0, iNumFound=-1; i!=MAX_CLIENTS && iNumFound!=iIndex ; i++)
    if (pClients[i].iState == TRUE)
      iNumFound++;
  
  if (iNumFound==iIndex)
    return &pClients[i-1];

  D_Assert(FALSE, "Bogus index!");

  /* For compiler */
  return NULL;
}


/************************************************************************ 
 *  FUNCTION: SRV_ReplicateRegistrationData
 *  HISTORY: 
 *     08.28.94  ESF  Created.
 *     02.27.95  ESF  Added Species.
 *     25.08.95  JC   Moved Species before.
 *     29.08.95  JC   Don't send always the first allocated species.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_ReplicateRegistrationData(Int32 iCommLinkDest)
{
  Int32 i, iPlayer, iSpecies;

  /* Ignore the "human" entry... */
  for (i=iSpecies=1; i<RISK_GetNumSpecies(); i++, iSpecies++)
    {
      while (RISK_GetAllocationStateOfSpecies(iSpecies) == ALLOC_NONE)
	iSpecies++;

      RISK_SelectiveReplicate(iCommLinkDest, SPE_NAME, iSpecies, 0);
      RISK_SelectiveReplicate(iCommLinkDest, SPE_VERSION, iSpecies, 0);
      RISK_SelectiveReplicate(iCommLinkDest, SPE_DESCRIPTION, iSpecies, 0);
      RISK_SelectiveReplicate(iCommLinkDest, SPE_AUTHOR, iSpecies, 0);
      RISK_SelectiveReplicate(iCommLinkDest, SPE_CLIENT, iSpecies, 0);
      RISK_SelectiveReplicate(iCommLinkDest, SPE_ALLOCATION, iSpecies, 0);
    }

  for (i=0; i<RISK_GetNumPlayers(); i++)
    {
      iPlayer = RISK_GetNthPlayer(i);
      
      RISK_SelectiveReplicate(iCommLinkDest, PLR_STATE, iPlayer, 0);
      RISK_SelectiveReplicate(iCommLinkDest, PLR_COLORSTRING, iPlayer, 0); 
      RISK_SelectiveReplicate(iCommLinkDest, PLR_NAME, iPlayer, 0);
      RISK_SelectiveReplicate(iCommLinkDest, PLR_SPECIES, iPlayer, 0);
      RISK_SelectiveReplicate(iCommLinkDest, PLR_CLIENT, iPlayer, 0);
      RISK_SelectiveReplicate(iCommLinkDest, PLR_ALLOCATION, iPlayer, 0);
    }
}


/************************************************************************ 
 *  FUNCTION: SRV_ReplicateAllData
 *  HISTORY: 
 *     08.28.94  ESF  Created.
 *     02.27.95  ESF  Added Species.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_ReplicateAllData(Int32 iCommLinkDest)
{
  Int32 i, j, iPlayer;

  /* First replicate the registration data */
  SRV_ReplicateRegistrationData(iCommLinkDest);

  /* Now everything else */
  for (i=0; i<RISK_GetNumPlayers(); i++)
    {
      iPlayer = RISK_GetNthPlayer(i);
      
      RISK_SelectiveReplicate(iCommLinkDest, PLR_NUMCOUNTRIES, iPlayer, 0);
      RISK_SelectiveReplicate(iCommLinkDest, PLR_NUMARMIES, iPlayer, 0);
      RISK_SelectiveReplicate(iCommLinkDest, PLR_NUMCARDS, iPlayer, 0);
      
      for (j=0; j!=RISK_GetNumCardsOfPlayer(iPlayer); j++)
	RISK_SelectiveReplicate(iCommLinkDest, PLR_CARD, iPlayer, j);
    }

  for (i=0; i!=NUM_COUNTRIES; i++)
    {
      RISK_SelectiveReplicate(iCommLinkDest, CNT_OWNER, i, 0);
      RISK_SelectiveReplicate(iCommLinkDest, CNT_NUMARMIES, i, 0);
    }
}


/************************************************************************ 
 *  FUNCTION: SRV_LogFailure
 *  HISTORY: 
 *     10.01.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_LogFailure(CString strReason, Int32 iCommLink)
{
  const Int32 iClient = SRV_CommLinkToClient(iCommLink);

  D_Assert(iClient >= -1 && iClient < MAX_CLIENTS, "Bogus client...");

  /* This stops any more bogus messages from coming in from the 
   * dead client, which probably has nothing interesting to say
   * anyway...
   */
  
  FD_CLR(iCommLink, &fdBackup);

  /* If this is -1, then it is not a known client */
  if (iClient == -1)
    {
      /* This shouldn't happen much */
#ifdef ENGLISH
      printf("SERVER: CommLink operation failed, recovering.\n");
#endif
#ifdef FRENCH
      printf("SERVEUR: CommLink operation choue, recupre.\n");
#endif
      close(iCommLink);
    }
  else
    {
      /* Log the failure, to be dealt with later. */
      pFailures[iClient].strReason = strReason;
      pFailures[iClient].iCount++; 
      iNumFailures++;
    }
}


/************************************************************************ 
 *  FUNCTION: SRV_RecoverFailures
 *  HISTORY: 
 *     08.28.94  ESF  Created.
 *     08.31.94  ESF  Revised to do something.
 *     09.29.94  ESF  Fixed to deregister client in case of failure.
 *     09.30.94  ESF  Fixed to call the right deregistration function.
 *     09.31.94  ESF  Added MSG_NETMESSAGE to be sent also.
 *     10.01.94  ESF  Changed to reflect new failure handling.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_RecoverFailures(void)
{
  Int32 iClient;

  /* An optimization */
  if (iNumFailures == 0)
    return;

  /* Go through all of the failures and handle them */
  for (iClient=0; iClient!=MAX_CLIENTS; iClient++)
    if (pFailures[iClient].iCount > 0)
      {
	MsgNetPopup       msgNetPopup;
	MsgMessagePacket  msgMessagePacket;
	
	/* It is in the list of clients, so we need to destroy it.
	 * Make as if we received a MSG_DEREGISTERCLIENT.
	 */
	
	SRV_HandleDEREGISTERCLIENT(iClient);
	
	/* Tell the clients what's happening (N.B. this broadcast call
	 * has to be made AFTER the FreeClient call, otherwise, the
	 * broadcast will try to send a message to the client that
	 * just failed, and we will enter a recursive situation.)
	 */

	/* Information */
#ifdef ENGLISH
	sprintf(strScratch, "%s has failed (%s) (%d failure%s)", 
#endif
#ifdef FRENCH
	sprintf(strScratch, "%s a chou (%s) (%d chec%s)", 
#endif
		SRV_GetAddressOfClient(iClient), 
		pFailures[iClient].strReason, 
		pFailures[iClient].iCount, pFailures[iClient].iCount>1 ? 
		"s" : "");
#ifdef ENGLISH
	printf("SERVER: %s\n", pFailures[iClient].strReason);
#endif
#ifdef FRENCH
	printf("SERVEUR: %s\n", pFailures[iClient].strReason);
#endif
	msgNetPopup.strMessage = strScratch;
	SRV_BroadcastMessage(MSG_NETPOPUP, &msgNetPopup);
	
	msgMessagePacket.strMessage = strScratch;
	msgMessagePacket.iFrom      = FROM_SERVER;
	msgMessagePacket.iTo        = DST_ALLPLAYERS;
	SRV_BroadcastMessage(MSG_MESSAGEPACKET, &msgMessagePacket);
	
	/* Pheww... Finished. */
#ifdef ENGLISH
	printf("SERVER: Successfully handled client failure.\n");
#endif
#ifdef FRENCH
	printf("SERVEUR: Rcupration correcte de la perte d'un client.\n");
#endif

	pFailures[iClient].iCount = 0;
	pFailures[iClient].strReason = 0;
      }

  /* Reset this */
  iNumFailures = 0;
}


/************************************************************************ 
 *  FUNCTION: SRV_IterateTurn
 *  HISTORY: 
 *     10.29.94  ESF  Created.
 *     11.27.94  ESF  Fixed a really stupid and serious bug.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int32 SRV_IterateTurn(void)
{
  /* It may be that the player who's turn it just was killed a few
   * players, and the number of live players has gone down.  The way
   * players are organized is as if they were stacked on one top of
   * the next, with the 0th live player at the bottom.  In other words,
   * if the 2nd live player gets killed, then the 3rd live player becomes
   * the 2nd live player, etc.  This is done so that one can loop from the
   * [0th -- NumLivePlayer()th] live player.  However, when calculating 
   * turns, this may screw us up.  If is the 2nd live player's turn, and
   * during this turn the 1st live player gets killed, then the next turn
   * belongs to the 2nd live player (who used to be the 3rd live player).
   * So to calculate the next turn, we iterate through the players, NOT
   * the live players.  We simply search for the next live player.  In
   * case this sounds obvious, it probably is, but the code use to read:
   * "iTurn = (iTurn+1) % RISK_GetNumLivePlayers()" which is obviously wrong.
   */

  do 
    {
      /* Go to the next player, wrap-around if we're at the end */
      iTurn = (iTurn+1) % MAX_PLAYERS;
    } 
  while (RISK_GetStateOfPlayer(iTurn) == FALSE ||
	 RISK_GetAllocationStateOfPlayer(iTurn) != ALLOC_COMPLETE);

  /* Return the player who's turn it is */
  return iTurn;
}


/************************************************************************ 
 *  FUNCTION: SRV_AllocSpecies
 *  HISTORY: 
 *     02.23.95  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int32 SRV_AllocSpecies(void)
{
  /* Simply loop through the species looking for an empty slot.
   * Since they are created on demand, if the list is full, when
   * we get to the end the Dist. Obj. will build us a new one.
   */

  Int32 i;

  for (i=0; 
       RISK_GetAllocationStateOfSpecies(i) != ALLOC_NONE;
       i++)
    ; /* TwiddleThumbs */

  /* Mark the species as allocated.  Otherwise, if the requesting ai Client
   * hasn't done anything with the new species, and another ai Client 
   * is asking for one, the first ai Client doesn't set the name to be 
   * something non-NULL, then the server will return the SAME species 
   * number -- race condition!
   */

  RISK_SetAllocationStateOfSpecies(i, ALLOC_INPROGRESS);

  return i;
}


/************************************************************************ 
 *  FUNCTION: SRV_FreeSpecies
 *  HISTORY: 
 *     02.32.95  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_FreeSpecies(Int32 i)
{
  /* Reset the allocation state of the species */
  RISK_SetAllocationStateOfSpecies(i, ALLOC_NONE);
}


/************************************************************************ 
 *  FUNCTION: SRV_ResetGame
 *  HISTORY: 
 *     04.04.95  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_ResetGame(void)
{
  /* Things we have to do when a game is restarted:
   *   1. Set the server state to be SERVER_REGISTERING
   *   2. Actually reset the dist. obj.
   *   3. Create a new deck of cards.
   */

  iServerMode = SERVER_REGISTERING;
  
  /* Reset the game if it has not been already reset */
  if (!fGameReset)
    {
      RISK_ResetGame();
      fGameReset = TRUE;
			  
      /* Get a new deck, destroy the old one. */
      DECK_Destroy(pCardDeck);
      pCardDeck = DECK_Create(NUM_COUNTRIES + 2);
    }
}


/************************************************************************ 
 *  FUNCTION: SRV_GetStartStateOfClient
 *  HISTORY: 
 *     04.06.95  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int32 SRV_GetStartStateOfClient(Int32 iNumClient)
{
  D_Assert(iNumClient>=0 && iNumClient<MAX_CLIENTS, 
	   "Client out of range!");
  
  return pClients[iNumClient].fStarted;
}


/************************************************************************ 
 *  FUNCTION: SRV_SetStartStateOfClient
 *  HISTORY: 
 *     04.06.95  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_SetStartStateOfClient(Int32 iNumClient, Int32 iState)
{
  D_Assert(iNumClient>=0 && iNumClient<MAX_CLIENTS, 
	   "Client out of range!");
  
  pClients[iNumClient].fStarted = iState;
}


/************************************************************************ 
 *  FUNCTION: SRV_GetNumClientsStarted
 *  HISTORY: 
 *     04.06.95  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int32 SRV_GetNumClientsStarted(void)
{
  Int32 i, iCount;

  for (i=iCount=0; i!=MAX_CLIENTS; i++)
    if (SRV_GetAllocationStateOfClient(i) == ALLOC_COMPLETE &&
	SRV_GetStartStateOfClient(i) == TRUE)
      iCount++;

  return iCount;
}


/************************************************************************ 
 *  FUNCTION: SRV_AttemptNewGame
 *  HISTORY: 
 *     04.11.95  ESF  Created.
 *     30.08.95  JC   if (iServerMode == SERVER_REGISTERING && ... ?
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_AttemptNewGame(void)
{
  Int32 n, iIndex;

  /* See if we can start a game */
  if (    (iServerMode != SERVER_REGISTERING)
       || (SRV_GetNumClientsStarted() != SRV_GetNumClients())
       || (RISK_GetNumPlayers() < 2))
    return;

  D_Assert(RISK_GetNumLivePlayers()>=2, "Bogus number of players!");

#ifdef ENGLISH
  printf("SERVER: Clients have finished registering, beginning game.\n");
#endif
#ifdef FRENCH
  printf("SERVEUR: Les clients ont fini de s'enregistrer, le jeu commence.\n");
#endif
  /* We're going to need to reset it before playing again, so mark this. */
  fGameReset = FALSE;

  iServerMode = SERVER_FORTIFYING;

  SRV_SetInitialArmiesOfPlayers();
  SRV_SetInitialMissionOfPlayers();

  /* Dole out the countries */
  SRV_DistributeCountries();

  /* Compute a random player and notify clients */
  iIndex = rand() % RISK_GetNumLivePlayers();
  iTurn = RISK_GetNthLivePlayer(iIndex);
  SRV_NotifyClientsOfTurn(iTurn);

  /* Reset this for the next time */
  for (n=0; n!=MAX_CLIENTS; n++)
    if (SRV_GetAllocationStateOfClient(n) == ALLOC_COMPLETE)
      SRV_SetStartStateOfClient(n, FALSE);
}
