/* Low level event handling
   Copyright (C) 1995 Viacom New Media.

	This file is part of e93.

	e93 is free software; you can redistribute it and/or modify
	it under the terms of the e93 LICENSE AGREEMENT.

	e93 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	e93 LICENSE AGREEMENT for more details.

	You should have received a copy of the e93 LICENSE AGREEMENT
	along with e93; see the file "LICENSE.TXT".
*/

#include "includes.h"

#define	BLINKTIMEOUTSTART		40			/* when restarting the timeout value, this is used */
#define	BLINKTIMEOUTCONTINUE	30			/* when continuing timeouts, this value is used */
#define	INTERVALTIMEOUT			60			/* approx 1 second between calls to the timer script */
#define	ABORTCHECKINTERVAL		30			/* check for abort every 1/2 second */

static volatile UINT32
	waitTimer;								/* used by the WaitTicks routine, this counts up at the alarm rate, but can be reset */
static volatile UINT32
	blinkTimeout;							/* when this gets to 0, it is time to flash the cursor */
static volatile UINT32
	intervalTimeout;						/* when this gets to 0, it is time to wake the shell */
static UINT32
	nextAbortCheckTime;						/* when timer is >= to this value, it is ok to check for an abort */

int XErrorEventHandler(Display *theDisplay,XErrorEvent *theEvent)
/* handle protocol errors (generated by strange X implementations, or window managers)
 */
{
/*
	char
		errorMessage[256];

	XGetErrorText(theDisplay,theEvent->error_code,errorMessage,256);
	fprintf(stderr,"X Error: %s\n",errorMessage);
*/
	xFail=TRUE;
	return(0);
}

static Bool AllButUpdateEventsPredicate(Display *theDisplay,XEvent *theEvent,char *arg)
/* match all events (used in XCheckIfEvent below)
 */
{
	if((theEvent->type!=Expose)&&(theEvent->type!=GraphicsExpose)&&(theEvent->type!=NoExpose)&&(theEvent->type!=ConfigureNotify))
		{
		return(True);
		}
	return(False);
}

static Bool UpdateEventsPredicate(Display *theDisplay,XEvent *theEvent,char *arg)
/* match all update events
 */
{
	if((theEvent->type==Expose)||(theEvent->type==GraphicsExpose)||(theEvent->type==NoExpose)||(theEvent->type==ConfigureNotify))
		{
		return(True);
		}
	return(False);
}

static Bool KeyEventsPredicate(Display *theDisplay,XEvent *theEvent,char *arg)
/* match all keyboard press events
 */
{
	if(theEvent->type==KeyPress)
		{
		return(True);
		}
	return(False);
}

/**** The following comment is brought to you by: emacs 19.22 ****
 * This holds the state XLookupString needs to implement dead keys
 * and other tricks known as "compose processing".  _X Window System_ 
 * says that a portable program can't use this, but Stephen Gildea assures
 * me (emacs person) that letting the compiler initialize it to zeros will
 * work okay.
 *
 * If it is good enough for emacs, it is good enough for e93!
 */

static XComposeStatus
	composeStatus;						/* keep compose status around, and pass it in to all calls to XLookupString */

int ComposeXLookupString(XKeyEvent *event_struct,char *buffer_return,int bytes_buffer,KeySym *keysym_return)
/* Call this instead of XLookupString, it does the same thing, but handles the state of
 * compose processing properly
 * This should allow people to enter umlaut characters more easily
 */
{
	return(XLookupString(event_struct,buffer_return,bytes_buffer,keysym_return,&composeStatus));
}

BOOLEAN KeyEventToEditorKey(XEvent *theEvent,EDITORKEY *theKeyData)
/* take a keyboard event, and fill in an editor key record
 * If theEvent does not yield a editor key, then
 * return FALSE
 */
{
	KeySym
		theKeySym;
	char
		theAscCode;
	BOOLEAN
		virtualKey;
	UINT32
		theKey;
	int
		numAsc;
	XKeyEvent
		ourEvent;

	ourEvent=*((XKeyEvent *)theEvent);		/* copy the event, so we can play with it a bit */
	ourEvent.state&=(~Mod1Mask);			/* remove this from consideration */
	numAsc=ComposeXLookupString(&ourEvent,&theAscCode,1,&theKeySym);
	if(!IsModifierKey(theKeySym))			/* make sure it is not a modifier key (these do not get passed back) */
		{
		virtualKey=TRUE;					/* assume virtual key */
		switch(theKeySym)
			{
			case XK_Left:
				theKey=VVK_LEFTARROW;
				break;
			case XK_Right:
				theKey=VVK_RIGHTARROW;
				break;
			case XK_Up:
				theKey=VVK_UPARROW;
				break;
			case XK_Down:
				theKey=VVK_DOWNARROW;
				break;
			case XK_Prior:
				theKey=VVK_PAGEUP;
				break;
			case XK_Next:
				theKey=VVK_PAGEDOWN;
				break;
			case XK_Home:
				theKey=VVK_HOME;
				break;
			case XK_End:
				theKey=VVK_END;
				break;
			case XK_Insert:
				theKey=VVK_INSERT;
				break;
			case XK_Delete:
				theKey=VVK_DELETE;
				break;
			case XK_BackSpace:
				theKey=VVK_BACKSPACE;
				break;
			case XK_Help:
				theKey=VVK_HELP;
				break;
			case XK_Undo:
			case XK_F14:
				theKey=VVK_UNDO;
				break;
			case XK_Redo:
			case XK_F12:
				theKey=VVK_REDO;
				break;
			case XK_Return:
				theKey=VVK_RETURN;
				break;
			case XK_KP_Enter:
				theKey=VVK_RETURN;
				break;
			case XK_Tab:
				theKey=VVK_TAB;
				break;
			case XK_F20:
				theKey=VVK_CUT;
				break;
			case XK_F16:
				theKey=VVK_COPY;
				break;
			case XK_F18:
				theKey=VVK_PASTE;
				break;
			case XK_F1:
				theKey=VVK_F1;
				break;
			case XK_F2:
				theKey=VVK_F2;
				break;
			case XK_F3:
				theKey=VVK_F3;
				break;
			case XK_F4:
				theKey=VVK_F4;
				break;
			case XK_F5:
				theKey=VVK_F5;
				break;
			case XK_F6:
				theKey=VVK_F6;
				break;
			case XK_F7:
				theKey=VVK_F7;
				break;
			case XK_F8:
				theKey=VVK_F8;
				break;
			case XK_F9:
				theKey=VVK_F9;
				break;
			case XK_F10:
				theKey=VVK_F10;
				break;
			default:
				virtualKey=FALSE;
				break;
			}
		if(theKeyData->isVirtual=virtualKey)
			{
			theKeyData->keyCode=theKey;
			XStateToEditorModifiers(theEvent->xkey.state,0,&theKeyData->modifiers);
			return(TRUE);
			}
		else
			{
			if(numAsc)
				{
				theKeyData->keyCode=theAscCode;
				XStateToEditorModifiers(theEvent->xkey.state,0,&theKeyData->modifiers);
				return(TRUE);
				}
			}
		}
	return(FALSE);
}

BOOLEAN EditorKeyNameToKeyCode(char *theName,UINT32 *theCode)
/* given a string which is the name of an unmodified key on the keyboard,
 * convert it into a keycode (in the case of X, we use KeySyms)
 * if the name is not the name of any key on our keyboard, then return FALSE
 */
{
	KeySym
		theKeySym;

	if((theKeySym=XStringToKeysym(theName))!=NoSymbol)
		{
		*theCode=(UINT32)theKeySym;
		return(TRUE);
		}
	return(FALSE);
}

char *EditorKeyCodeToKeyName(UINT32 theCode)
/* given a keycode, get its name
 * if the code is invalid, return NULL
 * NOTE: the returned string is static, and should not be modified or deleted
 */
{
	return(XKeysymToString((KeySym)theCode));
}

void XStateToEditorModifiers(unsigned int state,UINT32 numClicks,UINT32 *editorModifiers)
/* convert modifiers and numClicks into modifier bits for the editor
 */
{
	(*editorModifiers)=0;

	if(state&LockMask)
		{
		(*editorModifiers)|=EEM_CAPSLOCK;
		}
	if(state&ShiftMask)
		{
		(*editorModifiers)|=EEM_SHIFT;
		}
	if(state&ControlMask)
		{
		(*editorModifiers)|=EEM_CTL;
		}
	if(state&Mod1Mask)						/* Meta L key */
		{
		(*editorModifiers)|=EEM_MOD0;
		}
	if(state&Mod4Mask)						/* Alt key */
		{
		(*editorModifiers)|=EEM_MOD1;
		}

	numClicks<<=EES_STATE0;
	numClicks&=EEM_STATE0;
	(*editorModifiers)|=numClicks;
}

BOOLEAN EditorGetKeyPress(UINT32 *keyCode,UINT32 *editorModifiers,BOOLEAN wait,BOOLEAN clearBuffered)
/* used by editor when it wants to get a key press
 * if a code is found, it is returned in keyCode/editorModifiers
 * if none is ready, return FALSE
 * if wait is TRUE, wait here until one is ready (TRUE is always returned)
 * if clearBuffered is TRUE, then clear any keypresses that may have been in the
 * buffer before proceeding.
 */
{
	XEvent
		theEvent;
	BOOLEAN
		gotOne;
	KeySym
		theKeySym;

	gotOne=FALSE;
	XSync(xDisplay,False);										/* make sure the events we need are waiting */
	if(clearBuffered)
		{
		while(XCheckIfEvent(xDisplay,&theEvent,KeyEventsPredicate,NULL))		/* inhale all keypresses */
			;
		}
	if(wait)
		{
		do
			{
			XIfEvent(xDisplay,&theEvent,KeyEventsPredicate,NULL);
			ComposeXLookupString((XKeyEvent *)&theEvent,NULL,0,&theKeySym);		/* get KeySym to see if modifiers or real press */
			}
		while(IsModifierKey(theKeySym));
		gotOne=TRUE;
		}
	else
		{
		while(!gotOne&&XCheckIfEvent(xDisplay,&theEvent,KeyEventsPredicate,NULL))
			{
			ComposeXLookupString((XKeyEvent *)&theEvent,NULL,0,&theKeySym);		/* get KeySym to see if modifiers or real press */
			if(!IsModifierKey(theKeySym))
				{
				gotOne=TRUE;
				}
			}
		}
	if(gotOne)
		{
		XStateToEditorModifiers(theEvent.xkey.state,0,editorModifiers);		/* make modifiers */
		theEvent.xkey.state=0;												/* get rid of all modifiers, so we end up with un-modified keysym */
		ComposeXLookupString((XKeyEvent *)&theEvent,NULL,0,&theKeySym);		/* get back unmodified keysym */
		*keyCode=(UINT32)theKeySym;
		}
	return(gotOne);
}

/* pseudo globals needed to handle multiple clicks
 */

static UINT32
	lastClickTime;
static Window
	lastClickWindow;
static INT32
	lastClickX,
	lastClickY;
static UINT32
	numClicks;

#define	CLICKSLOP		5				/* number of pixels of slop in either direction */
#define	CLICKTIME		300				/* number of milliseconds between clicks considered multiple */

void ResetMultipleClicks()
/* clears the double click information, so that next click starts from scratch
 */
{
	lastClickTime=0;					/* initialize multiple click variables */
	lastClickWindow=(Window)0;
	lastClickX=lastClickY=0;
	numClicks=0;
}

UINT32 HandleMultipleClicks(XEvent *theEvent)
/* updates and returns numClicks which tells how many clicks were in roughly the same area
 * of the same window.
 * where each click followed the previous by no more than CLICKTIME milliseconds
 * This is how double, triple, etc clicks are handled
 * NOTE: due to wrap around in the time field, this routine could possibly
 * fail to work correctly once every 49.7 days
 */
{
	if((lastClickTime+CLICKTIME)>theEvent->xbutton.time&&
		(lastClickX>=(theEvent->xbutton.x_root-CLICKSLOP))&&(lastClickX<=(theEvent->xbutton.x_root+CLICKSLOP))&&
		(lastClickY>=(theEvent->xbutton.y_root-CLICKSLOP))&&(lastClickY<=(theEvent->xbutton.y_root+CLICKSLOP))&&
		theEvent->xbutton.window==lastClickWindow)
		{
		numClicks++;
		}
	else
		{
		numClicks=0;
		}
	lastClickTime=theEvent->xbutton.time;
	lastClickWindow=theEvent->xbutton.window;
	lastClickX=theEvent->xbutton.x_root;
	lastClickY=theEvent->xbutton.y_root;
	return(numClicks);
}

static Bool ButtonEventPredicate(Display *theDisplay,XEvent *theEvent,char *arg)
/* return True if the event is a button press, or a button release
 * all button events are looked at while in StillDown
 */
{
	if((theEvent->type==ButtonPress)||(theEvent->type==ButtonRelease))
		{
		return(True);
		}
	return(False);
}

BOOLEAN StillDown(unsigned int theButton,UINT32 numTicks)
/* call this after receiving a Button Pressed event for theButton, and
 * it will return TRUE if the button is still held down
 * numTicks can be used to give some time back to the system while
 * waiting, it can also be used to time scrolls and such so that they
 * do not run too fast on fast machines
 */
{
	XEvent
		theEvent;
	BOOLEAN
		locatedUp;

	locatedUp=FALSE;
	XSync(xDisplay,False);
	WaitTicks(numTicks);
	while(!locatedUp&&XCheckIfEvent(xDisplay,&theEvent,ButtonEventPredicate,NULL))
		{
		if(theEvent.type==ButtonRelease&&theEvent.xbutton.button==theButton)
			{
			locatedUp=TRUE;
			}
		}
	return(!locatedUp);
}

static void CollectInvalidations(XEvent *theEvent)
/* given an Expose, or GraphicsExpose event, find the window
 * it belongs to, and add the event's invalid area to the window's
 * invalid region
 */
{
	WINDOWLISTELEMENT
		*theWindowElement;
	XRectangle
		exposeRect;

	if(theWindowElement=LocateWindowElement(theEvent->xany.window))
		{
		if(theEvent->xexpose.width&&theEvent->xexpose.height)	/* NOTE: this gets around a bug in X11/MOTIF (does not properly handle adding 0 width or height to a region!) */
			{
			exposeRect.x=theEvent->xexpose.x;
			exposeRect.y=theEvent->xexpose.y;
			exposeRect.width=theEvent->xexpose.width;
			exposeRect.height=theEvent->xexpose.height;
			XUnionRectWithRegion(&exposeRect,theWindowElement->invalidRegion,theWindowElement->invalidRegion);	/* add to invalid region of the window */
			}
		}
}

static void ConfigureWindow(XEvent *theEvent)
/* given a ConfigureNotify event event, find the window
 * it belongs to, and tell it to adjust to its size
 */
{
	WINDOWLISTELEMENT
		*theWindowElement;

	if(theWindowElement=LocateWindowElement(theEvent->xany.window))
		{
		switch(theWindowElement->theEditorWindow->windowType)
			{
			case EWT_DOCUMENT:
				AdjustDocumentWindowForNewSize(theWindowElement->theEditorWindow);
				break;
			case EWT_DIALOG:
			case EWT_MENU:
				/* these do not configure at the moment, so ignore the events */
				break;
			default:
				break;
			}
		}
}

static BOOLEAN AttemptBoundKeyPress(XEvent *theEvent)
/* see if the keypress event can be handled by one of the bound keys
 * if so, handle it, and return TRUE
 * if the key is not bound, return FALSE
 */
{
	KeySym
		theKeySym;
	XKeyEvent
		ourEvent;
	UINT32
		editorModifiers;

	XStateToEditorModifiers(theEvent->xkey.state,0,&editorModifiers);	/* make modifiers */
	ourEvent=*((XKeyEvent *)theEvent);
	ourEvent.state=0;												/* get rid of all modifiers, so we end up with un-modified keysym */
	ComposeXLookupString(&ourEvent,NULL,0,&theKeySym);				/* get back unmodified keysym */
	if(!IsModifierKey(theKeySym))									/* do not pass lone modifiers */
		{
		return(HandleBoundKeyEvent((UINT32)theKeySym,editorModifiers));	/* attempt to handle it */
		}
	return(FALSE);
}

static void DoEvent(XEvent *theEvent)
/* Passed an event, do what is needed to take care of it
 */
{
	WINDOWLISTELEMENT
		*theWindowElement;
	Window
		xWindow;
	EDITORWINDOW
		*theWindow;
	int
		focusRevert;

	switch(theEvent->type)
		{
		case KeyPress:
			if(!AttemptMenuKeyPress(theEvent))								/* look for menu key first */
				{
				if(!AttemptBoundKeyPress(theEvent))							/* then try for bound key */
					{
					XGetInputFocus(xDisplay,&xWindow,&focusRevert);			/* find out who has the focus */
					if(theWindowElement=LocateWindowElement(xWindow))		/* see if it is one of our windows */
						{
						theEvent->xany.window=xWindow;						/* fudge top window into event */
						switch(theWindowElement->theEditorWindow->windowType)
							{
							case EWT_MENU:									/* keys in menu window which are not menu keys get passed to top document window */
								if(theWindow=GetTopEditorWindowType(EWT_DOCUMENT))
									{
									theEvent->xany.window=((WINDOWLISTELEMENT *)theWindow->userData)->xWindow;
									DocumentWindowEvent(theWindow,theEvent);
									}
								break;
							case EWT_DOCUMENT:
								DocumentWindowEvent(theWindowElement->theEditorWindow,theEvent);
								break;
							}
						}
					}
				}
			break;

		default:
			if(theWindowElement=LocateWindowElement(theEvent->xany.window))
				{
				switch(theWindowElement->theEditorWindow->windowType)
					{
					case EWT_DOCUMENT:
						DocumentWindowEvent(theWindowElement->theEditorWindow,theEvent);
						break;
					case EWT_DIALOG:
						DialogWindowEvent(theWindowElement->theEditorWindow,theEvent);
						break;
					case EWT_MENU:
						MenuWindowEvent(theWindowElement->theEditorWindow,theEvent);
						break;
					default:
						break;
					}
				}
			break;
		}
}

static void DoDialogEvent(EDITORWINDOW *theDialog,XEvent *theEvent)
/* Passed an event, do what is needed to take care of it
 * this is called only when a dialog is up
 */
{
	WINDOWLISTELEMENT
		*theWindowElement;
	Window
		xWindow;
	int
		focusRevert;

	switch(theEvent->type)
		{
		case KeyPress:
			XGetInputFocus(xDisplay,&xWindow,&focusRevert);				/* find out who has the focus */
			if((theWindowElement=LocateWindowElement(xWindow))&&(theWindowElement->theEditorWindow==theDialog))	/* see if it is the active dialog */
				{
				theEvent->xany.window=xWindow;							/* fudge top window into event */
				DialogWindowEvent(theWindowElement->theEditorWindow,theEvent);
				}
			else
				{
				EditorBeep();											/* ugly kludge caused by X not allowing me to force the dialog to the top of the application */
				SetTopEditorWindow(theDialog);
				}
			break;

		default:
			if(theWindowElement=LocateWindowElement(theEvent->xany.window))
				{
				if((theWindowElement->theEditorWindow==theDialog)||(theEvent->type!=ButtonPress&&theEvent->type!=ClientMessage))	/* dont let the user do things to windows other than the dialog we are handling */
					{
					switch(theWindowElement->theEditorWindow->windowType)
						{
						case EWT_DOCUMENT:
							DocumentWindowEvent(theWindowElement->theEditorWindow,theEvent);
							break;
						case EWT_DIALOG:
							DialogWindowEvent(theWindowElement->theEditorWindow,theEvent);
							break;
						case EWT_MENU:
							MenuWindowEvent(theWindowElement->theEditorWindow,theEvent);
							break;
						default:
							break;
						}
					}
				else
					{
					SetTopEditorWindow(theDialog);
					}
				}
			break;
		}
}

static void RedrawInvalidWindows()
/* run though the window list, and redraw anything that
 * has invalid area
 */
{
	WINDOWLISTELEMENT
		*theElement;

	theElement=windowListHead;									/* update all windows that need it */
	while(theElement)
		{
		switch(theElement->theEditorWindow->windowType)
			{
			case EWT_DOCUMENT:
				UpdateDocumentWindow(theElement->theEditorWindow);
				break;
			case EWT_DIALOG:
				UpdateDialogWindow(theElement->theEditorWindow);
				break;
			case EWT_MENU:
				UpdateMenuWindow(theElement->theEditorWindow);
				break;
			default:
				break;
			}
		theElement=theElement->nextElement;
		}
}

void UpdateEditorWindows()
/* attend to changes brought about in our windows
 * redraw all editor windows that have been invalidated by previous expose or
 * graphics expose events
 */
{
	BOOLEAN
		needToRedraw;
	XEvent
		theEvent;

	XSync(xDisplay,False);										/* make sure the events we need are waiting */
/* NOTE: this second XSync is here because some update problems were
 * seen when dismissing dialogs.
 * The dialog window would be deleted, then the unmap event would be waited for.
 * Finally, this would be called (which with only one XSync, would sometimes
 * fail to update the windows that were invalidated by the dialog, because the
 * Expose events somehow did not come in??? Grrrrrr.)
 * The second XSync seems to resolve the problem...
 */
	XSync(xDisplay,False);
	needToRedraw=FALSE;											/* do not draw unless we need to (load the system less) */
	while(XCheckIfEvent(xDisplay,&theEvent,UpdateEventsPredicate,NULL))		/* get pending update messages */
		{
		switch(theEvent.type)
			{
			case Expose:
			case GraphicsExpose:
				CollectInvalidations(&theEvent);				/* collect any rectangles from expose events */
				needToRedraw=TRUE;								/* got an event that will make us redraw */
				break;
			case ConfigureNotify:
				ConfigureWindow(&theEvent);						/* go configure the window */
				XSync(xDisplay,False);							/* (could make more update messages, so get them) */
				break;
			default:
				break;
			}
		}
	if(needToRedraw)
		{
		RedrawInvalidWindows();
		}
}

void ResetCursorBlinkTime()
/* reset the timeout until the next cursor blink arrives
 */
{
	blinkTimeout=BLINKTIMEOUTSTART;			/* number of ticks of system timer before flashing the cursor */
}

void ResetEditorIntervalTimer()
/* reset the timeout until the next editor time event should be sent
 * the editor requests time events approx once per second, but is
 * not picky about it, and can go indefinitely without them
 */
{
	intervalTimeout=INTERVALTIMEOUT;		/* number of ticks of system timer before sending editor time event */
}

void WaitTicks(UINT32 numTicks)
/* wait for numTicks of the system timer before continuing
 * NOTE: the delay is measured from the last time this
 * was called, so if it has not been called in a while, the
 * first call usually returns immediately, it is useful
 * for slowing down scrolls on fast machines
 */
{
	while(waitTimer<numTicks)
		{
		pause();						/* give some time back to the machine, while we wait */
		}
	waitTimer=0;						/* reset the wait timeout */
}

void AlarmIntervalProc()
/* when the alarm goes off, this executes (then main wakes up)
 */
{
	if(blinkTimeout)
		{
		blinkTimeout--;
		}
	if(intervalTimeout)
		{
		intervalTimeout--;
		}
	waitTimer++;
	timer++;
}

void StartAlarmTimer()
/* called at the start of execution, this sets up the time base
 * used during the rest of the program
 * it also installs the signal handler
 */
{
	struct sigaction
		*theVector;								/* used to set the alarm vector */
	struct itimerval
		timerStruct;

	theVector=calloc(sizeof(struct sigaction),1);	/* cheap way to clear most fields of sigaction that we do not care about (because fields are named differently on each machine) */
#ifdef SOLARIS
	theVector->_funcptr._handler=AlarmIntervalProc;		/* handle the signal with this routine */
#else
	theVector->sa_handler=AlarmIntervalProc;	/* handle the signal with this routine */
#endif
	sigaction(SIGALRM,theVector,NULL);			/* when the alarm signal goes off, call here */
	free(theVector);

	timerStruct.it_interval.tv_sec=0;			/* 1/60 of a second intervals (fast enough to avoid sluggish feeling, not so fast as to bog machine) */
	timerStruct.it_interval.tv_usec=16667;
	timerStruct.it_value.tv_sec=0;
	timerStruct.it_value.tv_usec=16667;
	setitimer(ITIMER_REAL,&timerStruct,NULL);
}

static void DoPeriodicProc()
/* when it is time to flash the cursor (or whatever), attempt to locate the window with input
 * focus, and call it to do its thing
 */
{
	Window
		xWindow;
	int
		focusRevert;
	WINDOWLISTELEMENT
		*theElement;

	XGetInputFocus(xDisplay,&xWindow,&focusRevert);					/* find out who has the focus */
	if(theElement=LocateWindowElement(xWindow))						/* see if it is one of our windows */
		{
		switch(theElement->theEditorWindow->windowType)
			{
			case EWT_DOCUMENT:
				DocumentPeriodicProc(theElement->theEditorWindow);
				break;
			case EWT_DIALOG:
				DialogPeriodicProc(theElement->theEditorWindow);
				break;
			}
		}
}

void EditorQuit()
/* this is called from the editor when it wants to quit
 * it needs to inform the event loop that it should stop looping
 */
{
	timeToQuit=TRUE;								/* set the quit request flag */
}

void EditorDoBackground()
/* do the background tasks of updating windows, tasks and timers
 */
{
	UpdateEditorWindows();							/* update anything that is invalid now */
	pause();										/* wait for alarm signal to wake us up again */
	UpdateBufferTasks();							/* see if any buffer have tasks that need to be serviced/updated */
	if(!blinkTimeout)								/* time to flash cursor */
		{
		blinkTimeout=BLINKTIMEOUTCONTINUE;			/* reset the timer */
		DoPeriodicProc();
		}
	if(!intervalTimeout)							/* time to handle a shell interval event */
		{
		ResetEditorIntervalTimer();					/* reset the timer */
		HandleShellCommand("timer",0,NULL);			/* give the shell some time */
		}
}

void EditorEventLoop(int argc,char *argv[])
/* get the next event that the editor needs to know about and return it
 * if other events occur, this routine should stay here handling them until
 * one specifically related to the editor arrives, it should then pass that
 * event up to the editor for processing
 * if no events arrive, it is the responsibility of this routine to
 * give time back to the OS if necessary.
 */
{
	XEvent
		theEvent;

	HandleShellCommand("initialargs",argc-1,&(argv[1]));	/* pass off initial arguments to shell for processing */

	while(!timeToQuit)
		{
		if(XCheckIfEvent(xDisplay,&theEvent,AllButUpdateEventsPredicate,NULL))
			{
			DoEvent(&theEvent);
			}
		else
			{
			EditorDoBackground();					/* handle background stuff */
			}
		}
}

void DialogEventLoop(EDITORWINDOW *theDialog,BOOLEAN *dialogComplete)
/* when a dialog is up, this handles getting events, it does not send any
 * events other than update to document windows, or the menus
 * ALL events are either discarded, or given to the current dialog window
 * dialogComplete must be cleared when calling this, and it
 * must be set for this to return
 * NOTE: it is possible for this loop to be called BEFORE EditorEventLoop
 * has ever been executed, since dialogs can be brought up during the
 * startup script (which executes before the main event loop is called for the
 * first time)
 */
{
	XEvent
		theEvent;

	while(!(*dialogComplete))
		{
		if(XCheckIfEvent(xDisplay,&theEvent,AllButUpdateEventsPredicate,NULL))
			{
			DoDialogEvent(theDialog,&theEvent);
			}
		else
			{
			EditorDoBackground();					/* handle background stuff */
			}
		}
}

static Bool AbortEventPredicate(Display *theDisplay,XEvent *theEvent,char *arg)
/* Check the Event to see if it is an abort event
 * if it is, return True
 */
{
	KeySym
		theKeySym;

	if((theEvent->type==KeyPress))
		{
		ComposeXLookupString((XKeyEvent *)theEvent,NULL,0,&theKeySym);
		if((theKeySym==XK_Break)||(theKeySym==XK_Cancel))		/* see if we got a break, or cancel key */
			{
			return(True);
			}
		}
	return(False);
}

void ClearAbort()
/* call this to clear any pending abort status
 */
{
	XEvent
		theEvent;

	while(XCheckIfEvent(xDisplay,&theEvent,AbortEventPredicate,NULL))	/* get all abort events out of the queue */
		;
	nextAbortCheckTime=timer+ABORTCHECKINTERVAL;				/* set next time it will be ok to check for abort */
}

BOOLEAN CheckAbort()
/* This will return TRUE if the user is requesting an abortion of the
 * current job
 * This should be made to execute as efficiently as possible, since it will
 * be called often during certain loops
 * When an abort is sent back, SetError should have been called
 * To make it more efficient, this version checks the time, and if enough time
 * has not passed, returns instantly
 */
{
	XEvent
		theEvent;

	if(timer>=nextAbortCheckTime)
		{
		nextAbortCheckTime=timer+ABORTCHECKINTERVAL;			/* reset the interval */
		XSync(xDisplay,False);									/* get the events into the queue */
		if(XCheckIfEvent(xDisplay,&theEvent,AbortEventPredicate,NULL))	/* see if there is an abort event floating around in the queue */
			{
			SetError("events","abort","User Aborted");
			return(TRUE);
			}
		}
	return(FALSE);
}

BOOLEAN InitEvents()
/* intialize event stuff
 * if there is a problem, SetError, return FALSE
 */
{
	waitTimer=0;									/* init the wait timer */
	ResetMultipleClicks();
	return(TRUE);
}

void UnInitEvents()
/* unintialize event stuff
 */
{
}
