/* Undo/redo functions
   Copyright (C) 1995 Viacom New Media.

	This file is part of e93.

	e93 is free software; you can redistribute it and/or modify
	it under the terms of the e93 LICENSE AGREEMENT.

	e93 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	e93 LICENSE AGREEMENT for more details.

	You should have received a copy of the e93 LICENSE AGREEMENT
	along with e93; see the file "LICENSE.TXT".
*/

#include "includes.h"

static void DoClearUndo(TEXTUNIVERSE *undoText,ARRAYUNIVERSE *undoArray)
/* clear the passed text, and undo arrays
 */
{
	ARRAYCHUNKHEADER
		*undoChunk;
	UINT32
		undoOffset;

	DeleteUniverseArray(undoArray,undoArray->firstChunkHeader,0,undoArray->totalElements,&undoChunk,&undoOffset);
	DeleteUniverseText(undoText,0,undoText->totalBytes);
}

void BeginUndoGroup(EDITORUNIVERSE *theUniverse)
/* all of the things that are done between now, and the matching EndUndoGroup
 * will be remembered as a group in the undo buffer
 */
{
	if(!theUniverse->undoUniverse->groupingDepth++)			/* increment the depth, if it was 0, then start a new group */
		{
		theUniverse->undoUniverse->currentFrame++;			/* bump the frame number */
		}
}

void EndUndoGroup(EDITORUNIVERSE *theUniverse)
/* call this to balance a call to BeginUndoGroup
 * it ends the undo group, but will allow subsequent undos to be joined
 */
{
	if(theUniverse->undoUniverse->groupingDepth)
		{
		--theUniverse->undoUniverse->groupingDepth;			/* decrement the group nesting count */
		}
}

void StrictEndUndoGroup(EDITORUNIVERSE *theUniverse)
/* call this to balance a call to BeginUndoGroup
 * it ends the undo group, and will NOT allow subsequent undos to be joined
 */
{
	if(theUniverse->undoUniverse->groupingDepth)
		{
		if(!(--theUniverse->undoUniverse->groupingDepth))	/* decrement the group nesting count, if it went to 0, force new group */
			{
			theUniverse->undoUniverse->currentFrame++;		/* bump the frame number */
			}
		}
}

static BOOLEAN CreateUndoDeleteElement(EDITORUNIVERSE *theUniverse,UINT32 startPosition,UINT32 numBytes,TEXTUNIVERSE *undoText,ARRAYUNIVERSE *undoArray)
/* create a new element at the end of the undo array
 * if there is a problem, set the error, and return FALSE
 */
{
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	ARRAYCHUNKHEADER
		*undoChunk;
	UINT32
		undoOffset;
	UNDOELEMENT
		*undoElement;

	if(InsertUniverseArray(undoArray,NULL,0,1,&undoChunk,&undoOffset))	/* create the element */
		{
		undoElement=&(((UNDOELEMENT *)undoChunk->data)[undoOffset]);
		undoElement->frameNumber=theUniverse->undoUniverse->currentFrame;	/* set the frame number to the current undo frame */
		undoElement->textStartPosition=startPosition;						/* when the text is blasted back into place, this is where it goes */
		undoElement->textLength=0;											/* number of bytes to overwrite (none, we are deleting) */
		undoElement->undoStartPosition=undoText->totalBytes;				/* start at what is now the current end of the undo text */
		undoElement->undoLength=numBytes;									/* number of bytes to put back */
		PositionToChunkPosition(theUniverse->textUniverse,startPosition,&theChunk,&theOffset);	/* point to bytes about to be deleted */
		if(InsertUniverseChunks(undoText,undoText->totalBytes,theChunk,theOffset,numBytes))	/* copy into undo holding area */
			{
			return(TRUE);													/* if we got them, then we are done */
			}
		DeleteUniverseArray(undoArray,undoChunk,undoOffset,1,&undoChunk,&undoOffset);	/* we failed, so get rid of the last element */
		}
	return(FALSE);
}

static BOOLEAN MergeUndoDeleteElement(EDITORUNIVERSE *theUniverse,UNDOELEMENT *lastElement,UINT32 startPosition,UINT32 numBytes,TEXTUNIVERSE *undoText,ARRAYUNIVERSE *undoArray)
/* a delete is happening, and the last element of the undo list is in the current
 * frame. See if the delete can be combined with the last element of the undo list,
 * or if another entirely new element needs to be created
 * if there is a problem, set the error, and return FALSE
 */
{
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	ARRAYCHUNKHEADER
		*undoChunk;
	UINT32
		undoOffset;
	UINT32
		bytesToEdge;

	if((lastElement->textStartPosition==startPosition)||(lastElement->textStartPosition+lastElement->textLength==startPosition+numBytes))	/* for a delete to be grouped, it must start exactly in line with last element, or end exactly in line with it */
		{
		if(startPosition>=lastElement->textStartPosition)
			{
			bytesToEdge=lastElement->textStartPosition+lastElement->textLength-startPosition;
			if(bytesToEdge>=numBytes)
				{
				lastElement->textLength-=numBytes;
				if(!lastElement->textLength&&!lastElement->undoLength)	/* if this element has become empty, then get rid of it */
					{
					DeleteUniverseArray(undoArray,undoArray->lastChunkHeader,undoArray->lastChunkHeader->totalElements-1,1,&undoChunk,&undoOffset);
					}
				return(TRUE);											/* all that was needed was to remove bytes */
				}
			else
				{
				startPosition+=bytesToEdge;
				numBytes-=bytesToEdge;
				PositionToChunkPosition(theUniverse->textUniverse,startPosition,&theChunk,&theOffset);	/* point to bytes about to be deleted */
				if(InsertUniverseChunks(undoText,undoText->totalBytes,theChunk,theOffset,numBytes))		/* copy into undo holding area */
					{
					lastElement->textLength-=bytesToEdge;
					lastElement->undoLength+=numBytes;
					return(TRUE);																		/* if we got them, then we are done */
					}
				else
					{
					return(FALSE);
					}
				}
			}
		else
			{
			if((startPosition+numBytes)<=(lastElement->textStartPosition+lastElement->textLength))		/* make sure it does not span entire element */
				{
				bytesToEdge=startPosition+numBytes-lastElement->textStartPosition;
				numBytes-=bytesToEdge;
				PositionToChunkPosition(theUniverse->textUniverse,startPosition,&theChunk,&theOffset);				/* point to bytes about to be deleted */
				if(InsertUniverseChunks(undoText,lastElement->undoStartPosition,theChunk,theOffset,numBytes))	/* copy into undo holding area */
					{
					lastElement->textStartPosition=startPosition;
					lastElement->textLength-=bytesToEdge;
					lastElement->undoLength+=numBytes;
					return(TRUE);																		/* if we got them, then we are done */
					}
				else
					{
					return(FALSE);
					}
				}
			else
				{
				return(CreateUndoDeleteElement(theUniverse,startPosition,numBytes,undoText,undoArray));	/* this delete spans entire previous element, so make new one */
				}
			}
		}
/* if we cannot merge, then create a new element */
	if(!theUniverse->undoUniverse->groupingDepth)						/* if not currently in a group, then make this a new frame */
		{
		theUniverse->undoUniverse->currentFrame++;						/* bump the frame number */
		}
	return(CreateUndoDeleteElement(theUniverse,startPosition,numBytes,undoText,undoArray));	/* delete in a different place */
}

static BOOLEAN RegisterDelete(EDITORUNIVERSE *theUniverse,UINT32 startPosition,UINT32 numBytes,TEXTUNIVERSE *undoText,ARRAYUNIVERSE *undoArray)
/* register the delete into the given text buffer, and chunk array
 */
{
	ARRAYCHUNKHEADER
		*lastUndoChunk;
	UNDOELEMENT
		*lastElement;

	if((lastUndoChunk=undoArray->lastChunkHeader))							/* is there a current item in the array? */
		{
		lastElement=&(((UNDOELEMENT *)lastUndoChunk->data)[lastUndoChunk->totalElements-1]);	/* point to last element of undo array */
		if(theUniverse->undoUniverse->currentFrame==lastElement->frameNumber)	/* is it in the current frame? */
			{
			return(MergeUndoDeleteElement(theUniverse,lastElement,startPosition,numBytes,undoText,undoArray));
			}
		else
			{
			return(CreateUndoDeleteElement(theUniverse,startPosition,numBytes,undoText,undoArray));
			}
		}
	else
		{
		return(CreateUndoDeleteElement(theUniverse,startPosition,numBytes,undoText,undoArray));
		}
}

BOOLEAN RegisterUndoDelete(EDITORUNIVERSE *theUniverse,UINT32 startPosition,UINT32 numBytes)
/* call this BEFORE deleting numBytes of text from startPosition of theUniverse
 * if there is a problem, return FALSE
 * NOTE: for speed, it is the caller's responsibility to determine theChunk and theOffset
 * which point to startPosition of theUniverse
 */
{
	UNDOUNIVERSE
		*theUndoUniverse;

	if(numBytes)
		{
		theUndoUniverse=theUniverse->undoUniverse;

		if(!theUndoUniverse->undoActive)
			{
			DoClearUndo(theUndoUniverse->redoText,&(theUndoUniverse->redoChunks));	/* get rid of redo information */
			}
		if(theUndoUniverse->undoing)					/* if undoing, then place undo info into redo buffers */
			{
			return(RegisterDelete(theUniverse,startPosition,numBytes,theUndoUniverse->redoText,&theUndoUniverse->redoChunks));
			}
		else
			{
			return(RegisterDelete(theUniverse,startPosition,numBytes,theUndoUniverse->undoText,&theUndoUniverse->undoChunks));
			}
		}
	else
		{
		return(TRUE);
		}
}

static BOOLEAN CreateUndoInsertElement(EDITORUNIVERSE *theUniverse,UINT32 startPosition,UINT32 numBytes,TEXTUNIVERSE *undoText,ARRAYUNIVERSE *undoArray)
/* create a new element at the end of the undo array
 * if there is a problem, set the error, and return FALSE
 */
{
	ARRAYCHUNKHEADER
		*undoChunk;
	UINT32
		undoOffset;
	UNDOELEMENT
		*undoElement;

	if(InsertUniverseArray(undoArray,NULL,0,1,&undoChunk,&undoOffset))	/* create the element */
		{
		undoElement=&(((UNDOELEMENT *)undoChunk->data)[undoOffset]);
		undoElement->frameNumber=theUniverse->undoUniverse->currentFrame;	/* set the frame number to the current undo frame */
		undoElement->textStartPosition=startPosition;						/* when the text is blasted back into place, this is where it goes */
		undoElement->textLength=numBytes;									/* number of bytes to overwrite (all that is being inserted) */
		undoElement->undoStartPosition=undoText->totalBytes;				/* start at what is now the current end of the undo text */
		undoElement->undoLength=0;											/* number of bytes to put back (none, we are inserting) */
		return(TRUE);
		}
	return(FALSE);
}

static BOOLEAN MergeUndoInsertElement(EDITORUNIVERSE *theUniverse,UNDOELEMENT *lastElement,UINT32 startPosition,UINT32 numBytes,TEXTUNIVERSE *undoText,ARRAYUNIVERSE *undoArray)
/* an insert is happening, and the last element of the undo list is in the current
 * frame. See if the insert can be combined with the last element of the undo list,
 * or if another entirely new element needs to be created
 * if there is a problem, set the error, and return FALSE
 */
{
	if(startPosition==lastElement->textStartPosition+lastElement->textLength)	/* see if insert just after end of current start */
		{
		lastElement->textLength+=numBytes;							/* add these bytes to what should be deleted during undo */
		return(TRUE);
		}
/* if we cannot merge, then create a new element */
	if(!theUniverse->undoUniverse->groupingDepth)						/* if not currently in a group, then make this a new frame */
		{
		theUniverse->undoUniverse->currentFrame++;						/* bump the frame number */
		}
	return(CreateUndoInsertElement(theUniverse,startPosition,numBytes,undoText,undoArray));	/* insert in a different place */
}

static BOOLEAN RegisterInsert(EDITORUNIVERSE *theUniverse,UINT32 startPosition,UINT32 numBytes,TEXTUNIVERSE *undoText,ARRAYUNIVERSE *undoArray)
/* register the insert into the given text buffer, and chunk array
 * if there is a problem, set the error, and return FALSE
 */
{
	ARRAYCHUNKHEADER
		*lastUndoChunk;
	UNDOELEMENT
		*lastElement;

	if((lastUndoChunk=undoArray->lastChunkHeader))							/* is there a current item in the array? */
		{
		lastElement=&(((UNDOELEMENT *)lastUndoChunk->data)[lastUndoChunk->totalElements-1]);	/* point to last element of undo array */
		if(theUniverse->undoUniverse->currentFrame==lastElement->frameNumber)	/* is it in the current frame? */
			{
			return(MergeUndoInsertElement(theUniverse,lastElement,startPosition,numBytes,undoText,undoArray));
			}
		else
			{
			return(CreateUndoInsertElement(theUniverse,startPosition,numBytes,undoText,undoArray));
			}
		}
	else
		{
		return(CreateUndoInsertElement(theUniverse,startPosition,numBytes,undoText,undoArray));
		}
}

BOOLEAN RegisterUndoInsert(EDITORUNIVERSE *theUniverse,UINT32 startPosition,UINT32 numBytes)
/* call this AFTER inserting numBytes of text into theUniverse at startPosition
 * if there is a problem, set the error, return FALSE
 */
{
	UNDOUNIVERSE
		*theUndoUniverse;

	if(numBytes)
		{
		theUndoUniverse=theUniverse->undoUniverse;

		if(!theUndoUniverse->undoActive)
			{
			DoClearUndo(theUndoUniverse->redoText,&(theUndoUniverse->redoChunks));	/* get rid of redo information */
			}
		if(theUndoUniverse->undoing)					/* if undoing, then place undo info into redo buffers */
			{
			return(RegisterInsert(theUniverse,startPosition,numBytes,theUndoUniverse->redoText,&theUndoUniverse->redoChunks));
			}
		else
			{
			return(RegisterInsert(theUniverse,startPosition,numBytes,theUndoUniverse->undoText,&theUndoUniverse->undoChunks));
			}
		}
	else
		{
		return(TRUE);
		}
}

static BOOLEAN FastUndoSelectionChange(SELECTIONUNIVERSE *theSelectionUniverse,UINT32 thePosition,UINT32 numBytesOut,UINT32 numBytesIn,UINT32 *minSelectionPosition,UINT32 *maxSelectionPosition)
/* While undoing, the text is changing size, selections are being created for each change in
 * the text, and the selection universe is being traversed to make sure it reflects the
 * changes to the text.
 * The editor provides two general purpose functions for adding to a selection universe,
 * and for updating one due to a change in the text (EditorAddSelection, and EditorFixSelections.)
 * Unfortunately, these functions can become quite slow when there are many selections in the list,
 * and changes are happening near the bottom. Since undo typically works in a constant direction,
 * and typically does not cause selections to overlap, it can be sped up by taking advantage of
 * these cases. This routine does that by taking responsibility for creating the selection
 * during undo, and exploiting the fast cases, calling the slow routines only when needed.
 * thePosition is the place being modified, numBytesOut tells how many bytes have been removed at
 * thePosition, and numBytesIn tells how many have been added (and should be selected.)
 * minSelectionPosition and maxSelectionPosition remember the start and end points of
 * all selections so far, so this can case on those selections being added outside the current
 * bounds.
 * If there is a problem, set the error, and return FALSE
 * NOTE: if something goes wrong, attempt to leave the selection universe in a sane state
 */
{
	ARRAYCHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	BOOLEAN
		fail;

	fail=FALSE;
	if(!theSelectionUniverse->selectionChunks.firstChunkHeader)			/* if nothing selected so far, then init min and max */
		{
		if(numBytesIn)													/* only create a selection if bytes are added */
			{
			if(InsertUniverseSelection(theSelectionUniverse,NULL,0,1,&theChunk,&theOffset))		/* make a new selection element */
				{
				((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset=thePosition;
				((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset=numBytesIn;
				*minSelectionPosition=thePosition;
				*maxSelectionPosition=thePosition+numBytesIn;
				}
			else
				{
				fail=TRUE;
				}
			}
		}
	else
		{
		if(thePosition+numBytesOut<=*minSelectionPosition)				/* can we add it at the start? */
			{
			if(InsertUniverseSelection(theSelectionUniverse,theSelectionUniverse->selectionChunks.firstChunkHeader,0,1,&theChunk,&theOffset))		/* make a new selection element at the start */
				{
				((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset=thePosition;
				((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset=numBytesIn;
				theOffset++;											/* move to next selection (so it can be adjusted to account for the change just made) */
				if(theOffset>=theChunk->totalElements)
					{
					theChunk=theChunk->nextHeader;
					theOffset=0;
					}
				((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset-=(thePosition+numBytesOut);	/* move to account for changes */
				*minSelectionPosition=thePosition;
				*maxSelectionPosition+=(numBytesIn-numBytesOut);
				}
			else
				{
				fail=TRUE;
				}
			}
		else
			{
			if(thePosition>=*maxSelectionPosition)					/* can we add it at the end? */
				{
				if(InsertUniverseSelection(theSelectionUniverse,NULL,0,1,&theChunk,&theOffset))		/* make a new selection element at the en */
					{
					((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset=thePosition-*maxSelectionPosition;
					((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset=numBytesIn;
					*maxSelectionPosition=thePosition+numBytesIn;
					}
				else
					{
					fail=TRUE;
					}
				}
			else			/* hmmm, added in the middle, so call generalized routines */
				{
				EditorFixSelections(theSelectionUniverse,thePosition,thePosition+numBytesOut,thePosition+numBytesIn);
				if(EditorAddSelection(theSelectionUniverse,thePosition,numBytesIn))
					{
					if(thePosition<*minSelectionPosition)
						{
						*minSelectionPosition=thePosition;
						}
					if(thePosition+numBytesIn-numBytesOut>*maxSelectionPosition)
						{
						*maxSelectionPosition=thePosition+numBytesIn;
						}
					else
						{
						*maxSelectionPosition+=(numBytesIn-numBytesOut);
						}
					}
				else
					{
					fail=TRUE;
					}
				}
			}
		}
	return(!fail);
}

static BOOLEAN DoUndo(EDITORUNIVERSE *theUniverse,TEXTUNIVERSE *undoText,ARRAYUNIVERSE *undoArray,BOOLEAN *didSomething)
/* undo from the given text and array into theUniverse
 * NOTE: if there is nothing to undo, return FALSE in *didSomething
 * else return TRUE.
 * If something bad happens while trying to undo, set the error, and return FALSE
 */
{
	SELECTIONUNIVERSE
		*theSelectionUniverse,
		*newSelectionUniverse;
	ARRAYCHUNKHEADER
		*lastUndoChunk;
	ARRAYCHUNKHEADER
		*selectionChunk;
	UINT32
		selectionOffset;
	UINT32
		tempOffset;
	UINT32
		minSelectionPosition,
		maxSelectionPosition;
	UNDOELEMENT
		*lastElement;
	UINT32
		frameToUse;
	BOOLEAN
		haveMinimumPosition;
	UINT32
		minimumPosition;													/* used to set the cursor position after the change */
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	BOOLEAN
		fail;

	fail=*didSomething=FALSE;
	haveMinimumPosition=FALSE;
	minimumPosition=0;
	if(lastUndoChunk=undoArray->lastChunkHeader)							/* is there a current item in the array? */
		{
		lastElement=&(((UNDOELEMENT *)lastUndoChunk->data)[lastUndoChunk->totalElements-1]);	/* point to last element of undo array */
		frameToUse=lastElement->frameNumber;								/* undo all elements of this frame */

		if(newSelectionUniverse=OpenSelectionUniverse())					/* create a new selection universe where the output selection will be created */
			{
			theSelectionUniverse=theUniverse->selectionUniverse;			/* point to current selection universe */
			EditorStartSelectionChange(theUniverse);
			DeleteUniverseSelection(theSelectionUniverse,theSelectionUniverse->selectionChunks.firstChunkHeader,0,theSelectionUniverse->selectionChunks.totalElements,&selectionChunk,&selectionOffset);	/* kill off all old selections */
			EditorEndSelectionChange(theUniverse);

			EditorStartReplace(theUniverse);
			BeginUndoGroup(theUniverse);										/* group the redo of this undo */
			while(!fail&&lastElement&&lastElement->frameNumber==frameToUse)
				{
				PositionToChunkPosition(undoText,lastElement->undoStartPosition,&theChunk,&theOffset);	/* point to bytes about to be placed into the text */
				if(ReplaceEditorChunks(theUniverse,lastElement->textStartPosition,lastElement->textStartPosition+lastElement->textLength,theChunk,theOffset,lastElement->undoLength))
					{
					if(FastUndoSelectionChange(newSelectionUniverse,lastElement->textStartPosition,lastElement->textLength,lastElement->undoLength,&minSelectionPosition,&maxSelectionPosition))
						{
						if(haveMinimumPosition)
							{
							if(lastElement->textStartPosition<minimumPosition)
								{
								minimumPosition=lastElement->textStartPosition;
								}
							}
						else
							{
							minimumPosition=lastElement->textStartPosition;
							haveMinimumPosition=TRUE;
							}

						DeleteUniverseText(undoText,lastElement->undoStartPosition,lastElement->undoLength);
						DeleteUniverseArray(undoArray,lastUndoChunk,lastUndoChunk->totalElements-1,1,&lastUndoChunk,&tempOffset);
						if((lastUndoChunk=undoArray->lastChunkHeader))					/* fetch the last element again */
							{
							lastElement=&(((UNDOELEMENT *)lastUndoChunk->data)[lastUndoChunk->totalElements-1]);	/* point to last element of undo array */
							}
						else
							{
							lastElement=NULL;
							}
						}
					else
						{
						fail=TRUE;
						}
					}
				else
					{
					fail=TRUE;
					}
				}

			EditorEndReplace(theUniverse);
			StrictEndUndoGroup(theUniverse);									/* end this group, do not let anything combine with it */

			/* place new selection into the universe */

			EditorStartSelectionChange(theUniverse);
			newSelectionUniverse->cursorPosition=minimumPosition;				/* place cursor at the start */
			theUniverse->selectionUniverse=newSelectionUniverse;
			CloseSelectionUniverse(theSelectionUniverse);						/* close old universe */
			EditorEndSelectionChange(theUniverse);

			*didSomething=TRUE;
			}
		else
			{
			fail=TRUE;
			}
		}
	return(!fail);
}

BOOLEAN EditorUndo(EDITORUNIVERSE *theUniverse)
/* undo whatever was done last (if there is nothing in the undo buffer
 * then do not do anything)
 * If there is no undo information, return FALSE
 * otherwise return TRUE.
 * NOTE: if there is a failure, report it here
 */
{
	BOOLEAN
		didSomething,
		result;

	didSomething=FALSE;
	ShowBusy();
	theUniverse->undoUniverse->undoActive=TRUE;							/* changes are coming from undo/redo, not user */
	theUniverse->undoUniverse->undoing=TRUE;							/* remember that we are undoing so if register calls come in, we can place them in redo buffer */
	result=DoUndo(theUniverse,theUniverse->undoUniverse->undoText,&(theUniverse->undoUniverse->undoChunks),&didSomething);
	theUniverse->undoUniverse->undoing=FALSE;
	theUniverse->undoUniverse->undoActive=FALSE;
	ShowNotBusy();
	if(!result)
		{
		GetError(&errorFamily,&errorFamilyMember,&errorDescription);
		ReportMessage("Failed to undo: %.256s\nNOTE: graceful recovery from this error is not likely.\nThe buffer contents may now be corrupt :-(\n",errorDescription);
		didSomething=TRUE;												/* tell the caller that something was done */
		}
	return(didSomething);
}

BOOLEAN EditorRedo(EDITORUNIVERSE *theUniverse)
/* redo whatever was undone last
 * If there is no undo information, return FALSE in *didSomething
 * otherwise return TRUE.
 * NOTE: if there is a failure, report it here
 */
{
	BOOLEAN
		didSomething,
		result;

	didSomething=FALSE;
	ShowBusy();
	theUniverse->undoUniverse->undoActive=TRUE;							/* changes are coming from undo/redo, not user */
	result=DoUndo(theUniverse,theUniverse->undoUniverse->redoText,&(theUniverse->undoUniverse->redoChunks),&didSomething);
	theUniverse->undoUniverse->undoActive=FALSE;
	ShowNotBusy();
	if(!result)
		{
		GetError(&errorFamily,&errorFamilyMember,&errorDescription);
		ReportMessage("Failed to redo: %.256s\nNOTE: graceful recovery from this error is not likely.\nThe buffer contents may now be corrupt :-(\n",errorDescription);
		didSomething=TRUE;												/* tell the caller that something was done */
		}
	return(didSomething);
}

BOOLEAN EditorToggleUndo(EDITORUNIVERSE *theUniverse)
/* if there is a redo that can be done, then do it, otherwise, do an undo
 * if neither can be done, return FALSE
 * NOTE: if there is a failure, report it here
 */
{
	if(!EditorRedo(theUniverse))
		{
		return(EditorUndo(theUniverse));
		}
	return(TRUE);
}

void EditorClearUndo(EDITORUNIVERSE *theUniverse)
/* when, undo/redo must be cleared, call this to do it
 */
{
	DoClearUndo(theUniverse->undoUniverse->undoText,&(theUniverse->undoUniverse->undoChunks));	/* get rid of undo information */
	DoClearUndo(theUniverse->undoUniverse->redoText,&(theUniverse->undoUniverse->redoChunks));	/* get rid of redo information */
	theUniverse->undoUniverse->currentFrame=0;							/* reset the frame number */
}

UNDOUNIVERSE *OpenUndoUniverse()
/* open an undo universe
 * if there is a problem, set the error, and return NULL
 */
{
	UNDOUNIVERSE
		*theUniverse;

	if(theUniverse=(UNDOUNIVERSE *)MNewPtr(sizeof(UNDOUNIVERSE)))
		{
		theUniverse->currentFrame=0;
		theUniverse->groupingDepth=0;
		theUniverse->undoActive=FALSE;
		theUniverse->undoing=FALSE;
		if(theUniverse->undoText=OpenTextUniverse())
			{
			if(theUniverse->redoText=OpenTextUniverse())
				{
				if(InitArrayUniverse(&(theUniverse->undoChunks),UNDOCHUNKELEMENTS,sizeof(UNDOELEMENT)))
					{
					if(InitArrayUniverse(&(theUniverse->redoChunks),UNDOCHUNKELEMENTS,sizeof(UNDOELEMENT)))
						{
						return(theUniverse);
						}
					UnInitArrayUniverse(&(theUniverse->undoChunks));
					}
				CloseTextUniverse(theUniverse->redoText);
				}
			CloseTextUniverse(theUniverse->undoText);
			}
		MDisposePtr(theUniverse);
		}
	return(NULL);
}

void CloseUndoUniverse(UNDOUNIVERSE *theUniverse)
/* dispose of an undo universe
 */
{
	UnInitArrayUniverse(&(theUniverse->redoChunks));
	UnInitArrayUniverse(&(theUniverse->undoChunks));
	CloseTextUniverse(theUniverse->redoText);
	CloseTextUniverse(theUniverse->undoText);
	MDisposePtr(theUniverse);
}
