/* TCL commands that e93 supports
   Copyright (C) 1995 Viacom New Media.

	This file is part of e93.

	e93 is free software; you can redistribute it and/or modify
	it under the terms of the e93 LICENSE AGREEMENT.

	e93 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	e93 LICENSE AGREEMENT for more details.

	You should have received a copy of the e93 LICENSE AGREEMENT
	along with e93; see the file "LICENSE.TXT".
*/

/* NOTES about TCL, and its subtle interactions with the editor
 *
 * First, Tcl should never be allowed to evaluate script while an editor
 * window,view,buffer,selectionlist,etc... is in a state of flux that would
 * cause it to draw improperly if the script caused the editor to update its
 * windows. This is because certain editor operations that scripts can call,
 * can force updates of all editor windows.
 *
 * Second, if a script is running, and decides to operate on a window that
 * is in some state of flux, (say in the middle of a replace-all), it must
 * be prohibited from doing so, and the operation must fail. NOTE, that by
 * rule one, if you are executing script in the middle of replace-all, that
 * all buffers must be in a state where they can be drawn.
 *
 * Finally, if a script is run while some code of the editor is relying on
 * a given window, or buffer to be around, the code MUST set the
 * buffer Busy, or check that the buffer still exists after
 * the script is run.
 * If a buffer is set Busy, no script command, or side-effect can allow the
 * buffer to be modified in ANY way, including text/selection changes, undo
 * changes, new views, deleting views, etc...
 */

#include "includes.h"

typedef struct tokenlist						/* token list definition */
	{
	char	*token;
	int		tokenNum;
	} TOKENLIST;

TOKENLIST menuRelationshipTokens[]=
	{
	{"NEXTSIBLING",MR_NEXTSIBLING},
	{"PREVIOUSSIBLING",MR_PREVIOUSSIBLING},
	{"FIRSTCHILD",MR_FIRSTCHILD},
	{"LASTCHILD",MR_LASTCHILD},
	{"",0}
	};

typedef enum
	{
	STRICT,
	SEMISTRICT,
	LENIENT
	} HOMEWAY;

TOKENLIST homeWayTokens[]=
	{
	{"STRICT",STRICT},
	{"SEMISTRICT",SEMISTRICT},
	{"LENIENT",LENIENT},
	{"",0}
	};

TOKENLIST cursorMovementTokens[]=
	{
	{"LEFTCHAR",RPM_mBACKWARD|RPM_CHAR},
	{"RIGHTCHAR",RPM_CHAR},
	{"LEFTWORD",RPM_mBACKWARD|RPM_WORD},
	{"RIGHTWORD",RPM_WORD},
	{"UPLINE",RPM_mBACKWARD|RPM_LINE},
	{"DOWNLINE",RPM_LINE},
	{"STARTLINE",RPM_mBACKWARD|RPM_LINEEDGE},
	{"ENDLINE",RPM_LINEEDGE},
	{"STARTPARAGRAPH",RPM_mBACKWARD|RPM_PARAGRAPHEDGE},
	{"ENDPARAGRAPH",RPM_PARAGRAPHEDGE},
	{"UPPAGE",RPM_mBACKWARD|RPM_PAGE},
	{"DOWNPAGE",RPM_PAGE},
	{"STARTDOC",RPM_mBACKWARD|RPM_DOCEDGE},
	{"ENDDOC",RPM_DOCEDGE},
	{"",0}
	};

#define	TEMPSTRINGBUFFERLENGTH	1024

static BOOLEAN MatchToken(char *theString,TOKENLIST *theList,int *theToken)
/* given a string, see if it matches any of the tokens in the token list, if so, return
 * its token number in theToken, otherwise, return FALSE
 * This should not be case sensitive.
 */
{
	int
		i;
	BOOLEAN
		found,reachedEnd;

	found=reachedEnd=FALSE;
	i=0;
	while(!found&&!reachedEnd)
		{
		if(theList[i].token[0]!='\0')
			{
			if(strcasecmp(theString,theList[i].token)==0)
				{
				found=TRUE;
				*theToken=theList[i].tokenNum;
				}
			i++;
			}
		else
			{
			reachedEnd=TRUE;
			}
		}
	return(found);
}

static UINT32 ForcePositionIntoRange(EDITORUNIVERSE *theUniverse,UINT32 thePosition)
/* take thePosition, and make sure it is in range for theUniverse
 */
{
	if(thePosition>theUniverse->textUniverse->totalBytes)
		{
		thePosition=theUniverse->textUniverse->totalBytes;
		}
	return(thePosition);
}

static EDITORWINDOW *GetEditorWindow(Tcl_Interp *localInterpreter,char *theString)
/* see if theString is a window, if so, return it
 * if not, return NULL, and set the Tcl result
 */
{
	EDITORWINDOW
		*theWindow;

	if(theWindow=LocateEditorDocumentWindow(theString))
		{
		return(theWindow);
		}
	Tcl_AppendResult(localInterpreter,"Failed to locate window '",theString,"'",NULL);
	return(NULL);
}

static EDITORBUFFER *GetEditorBuffer(Tcl_Interp *localInterpreter,char *theString)
/* see if theString is a buffer, if so, return it
 * if not, return NULL, and set the Tcl result
 */
{
	EDITORBUFFER
		*theBuffer;

	if(theBuffer=LocateEditorBuffer(theString))
		{
		return(theBuffer);
		}
	Tcl_AppendResult(localInterpreter,"Failed to locate buffer '",theString,"'",NULL);
	return(NULL);
}

static BOOLEAN GetINT32(Tcl_Interp *localInterpreter,char *theString,INT32 *theNumber)
/* convert theString to a INT32
 * return FALSE if the conversion failed, and fill the Tcl result
 */
{
	int
		theInt;

	if(Tcl_GetInt(localInterpreter,theString,&theInt)==TCL_OK)
		{
		*theNumber=(INT32)theInt;
		return(TRUE);
		}
	return(FALSE);
}

static BOOLEAN GetUINT32(Tcl_Interp *localInterpreter,char *theString,UINT32 *theNumber)
/* convert theString to a UINT32
 * return FALSE if the conversion failed, and fill the Tcl result
 */
{
	int
		theInt;

	if(Tcl_GetInt(localInterpreter,theString,&theInt)==TCL_OK)
		{
		*theNumber=(UINT32)theInt;					/* ### this should really be fixed to be a true unsigned */
		return(TRUE);
		}
	return(FALSE);
}

static BOOLEAN GetRectangle(Tcl_Interp *localInterpreter,char **theArgs,EDITORRECT *theRect)
/* convert theArgs (of which there must be at least 4) to an EDITORRECT
 * return FALSE if the conversion failed, and fill the Tcl result
 */
{
	if(GetINT32(localInterpreter,theArgs[0],&theRect->x))
		{
		if(GetINT32(localInterpreter,theArgs[1],&theRect->y))
			{
			if(GetUINT32(localInterpreter,theArgs[2],&theRect->w))
				{
				if(GetUINT32(localInterpreter,theArgs[3],&theRect->h))
					{
					return(TRUE);
					}
				}
			}
		}
	return(FALSE);
}

static BOOLEAN GetNibble(char theCharacter,UINT32 *theNibble)
/* interpret theCharacter as a hex nibble, and return it
 * if there is a problem return FALSE
 */
{
	if(theCharacter>='0'&&theCharacter<='9')
		{
		*theNibble=theCharacter-'0';
		return(TRUE);
		}
	else
		{
		if(theCharacter>='A'&&theCharacter<='F')
			{
			*theNibble=10+theCharacter-'A';
			return(TRUE);
			}
		else
			{
			if(theCharacter>='a'&&theCharacter<='f')
				{
				*theNibble=10+theCharacter-'a';
				return(TRUE);
				}
			}
		}
	return(FALSE);
}

static BOOLEAN GetColor(Tcl_Interp *localInterpreter,char *theString,EDITORCOLOR *theColor)
/* convert theString to a color value
 * return FALSE if the conversion failed, and fill the Tcl result
 */
{
	UINT32
		index,
		theNibble;
	char
		theCharacter;
	
	BOOLEAN
		fail;

	fail=FALSE;
	if(!EditorColorNameToColor(theString,theColor))
		{
		*theColor=0;
		index=0;
		while(!fail&&(theCharacter=theString[index]))
			{
			if(GetNibble(theCharacter,&theNibble))
				{
				(*theColor)<<=4;
				(*theColor)|=theNibble;
				index++;
				}
			else
				{
				fail=TRUE;
				}
			}
		(*theColor)&=0xFFFFFF;		/* if number was large, ignore preceding digits */
		if(!index)					/* no characters in the input string */
			{
			fail=TRUE;
			}
		}
	if(fail)
		{
		Tcl_AppendResult(localInterpreter,"Invalid color",NULL);
		}
	return(!fail);
}

static BOOLEAN GetBoolean(Tcl_Interp *localInterpreter,char *theString,BOOLEAN *theValue)
/* convert theString to a boolean value
 * return FALSE if the conversion failed, and fill the Tcl result
 */
{
	int
		intValue;

	if(Tcl_GetBoolean(localInterpreter,theString,&intValue)==TCL_OK)
		{
		*theValue=intValue?TRUE:FALSE;
		return(TRUE);
		}
	return(FALSE);
}

static UINT32
	modifierFlagTable[]=
		{
		EEM_CAPSLOCK,
		EEM_SHIFT,
		EEM_CTL,
		EEM_MOD0,
		EEM_MOD1,
		EEM_MOD2,
		EEM_MOD3,
		EEM_MOD4,
		EEM_MOD5,
		EEM_MOD6,
		EEM_MOD7
		};

static BOOLEAN GetModifierMaskAndValue(Tcl_Interp *localInterpreter,char *theString,UINT32 *modifierMask,UINT32 *modifierValue)
/* convert a string of modifiers into a mask, and value
 * if there is a problem, set the Tcl result, and return FALSE
 */
{
	int
		i,
		theLength;
	BOOLEAN
		fail;

	*modifierMask=0;
	*modifierValue=0;
	theLength=strlen(theString);
	fail=FALSE;
	if(theLength==11)				/* must be an entry for each flag */
		{
		for(i=0;(!fail)&&(i<theLength);i++)
			{
			switch(theString[i])
				{
				case 'x':
				case 'X':
					break;
				case '0':
					*modifierMask|=modifierFlagTable[i];
					break;
				case '1':
					*modifierMask|=modifierFlagTable[i];
					*modifierValue|=modifierFlagTable[i];
					break;
				default:
					Tcl_AppendResult(localInterpreter,"Invalid modifiers",NULL);
					fail=TRUE;
					break;
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"Invalid modifiers (incorrect length)",NULL);
		fail=TRUE;
		}
	return(!fail);
}

static void ModifierMaskAndValueToString(UINT32 modifierMask,UINT32 modifierValue,char *theString)
/* given a modifier mask, and value, make a modifier string
 * NOTE: theString must be long enough to hold the result (which is never longer than 12 characters
 * including the terminator)
 */
{
	int
		i;

	for(i=0;i<11;i++)
		{
		if(modifierMask&modifierFlagTable[i])
			{
			if(modifierValue&modifierFlagTable[i])
				{
				theString[i]='1';
				}
			else
				{
				theString[i]='0';
				}
			}
		else
			{
			theString[i]='X';
			}
		}
	theString[i]='\0';
}


static BOOLEAN GetFlagList(Tcl_Interp *localInterpreter,int argc,char *argv[],BOOLEAN *flagList)
/* write values from the given constant array to the given boolean array
 * if there is a problem, set the TCL error, and return FALSE
 */
{
	int
		currentCount;
	BOOLEAN
		fail;

	fail=FALSE;
	for(currentCount=0;currentCount<argc&&!fail;currentCount++)
		{
		if(GetBoolean(localInterpreter,argv[currentCount],&(flagList[currentCount])))
			{
			}
		else
			{
			fail=TRUE;
			}
		}
	return(!fail);
}

static BOOLEAN GetVarFlagList(Tcl_Interp *localInterpreter,int argc,char *argv[],BOOLEAN *flagList)
/* write values from the given variable array to the given boolean array
 * if there is a problem, set the TCL error, and return FALSE
 */
{
	char
		*varValue;
	int
		currentCount;
	BOOLEAN
		fail;

	fail=FALSE;
	for(currentCount=0;currentCount<argc&&!fail;currentCount++)
		{
		if(varValue=Tcl_GetVar(localInterpreter,argv[currentCount],TCL_LEAVE_ERR_MSG))
			{
			if(GetBoolean(localInterpreter,varValue,&(flagList[currentCount])))
				{
				}
			else
				{
				fail=TRUE;
				}
			}
		else
			{
			fail=TRUE;
			}
		}
	return(!fail);
}

static BOOLEAN PutVarFlagList(Tcl_Interp *localInterpreter,int argc,char *argv[],BOOLEAN *flagList)
/* write the boolean variables back into Tcl variables
 * if there is a problem, set the TCL error, and return FALSE
 */
{
	int
		currentCount;
	BOOLEAN
		fail;

	fail=FALSE;
	for(currentCount=0;currentCount<argc&&!fail;currentCount++)
		{
		if(Tcl_SetVar(localInterpreter,argv[currentCount],flagList[currentCount]?"1":"0",TCL_LEAVE_ERR_MSG))
			{
			}
		else
			{
			fail=TRUE;
			}
		}
	return(!fail);
}

static void SetBufferBusy(EDITORBUFFER *theBuffer)
/* set the buffer's busy state
 */
{
	theBuffer->shellBusy=TRUE;
}

static void ClearBufferBusy(EDITORBUFFER *theBuffer)
/* clear the buffer's busy state
 */
{
	theBuffer->shellBusy=FALSE;
}

static BOOLEAN BufferNotBusy(Tcl_Interp *localInterpreter,EDITORBUFFER *theBuffer)
/* check the given buffer's busy state
 * if it is TRUE, set the Tcl result, and return FALSE
 */
{
	if(!theBuffer->shellBusy)
		{
		return(TRUE);
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"Buffer '",theBuffer->contentName,"' is busy",NULL);
		}
	return(FALSE);
}

static BOOLEAN CurrentClipboardNotBusy(Tcl_Interp *localInterpreter)
/* make sure the current clipboard is not busy, if it is, set the Tcl result
 * and return FALSE
 */
{
	EDITORBUFFER
		*clipboardBuffer;

	if(clipboardBuffer=EditorGetCurrentClipboard())
		{
		return(BufferNotBusy(localInterpreter,clipboardBuffer));
		}
	return(TRUE);
}

/* ACTUAL TCL COMMAND IMPLEMENTATIONS */

static int Tcl_AddMenu(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* add a menu to the menu list
 * NOTE: if a menu already exists, this will delete it before attempting
 * a creation
 * NOTE ALSO: if this deletes an element, it will also delete the element's children
 */
{
	int
		theRelationship;
	BOOLEAN
		active;
	int
		pathArgc;
	char
		**pathArgv;
	BOOLEAN
		succeed;

	succeed=FALSE;
	if(argc==7)
		{
		if(MatchToken(argv[2],menuRelationshipTokens,&theRelationship))
			{
			if(GetBoolean(localInterpreter,argv[3],&active))
				{
				if(Tcl_SplitList(localInterpreter,argv[1],&pathArgc,&pathArgv)==TCL_OK)
					{
					if(CreateEditorMenu(pathArgc,pathArgv,theRelationship,argv[4],argv[5],argv[6],active))
						{
						succeed=TRUE;
						}
					else
						{
						GetError(&errorFamily,&errorFamilyMember,&errorDescription);
						Tcl_AppendResult(localInterpreter,errorDescription,NULL);
						}
					ckfree(pathArgv);
					}
				}
			}
		else
			{
			Tcl_AppendResult(localInterpreter,"Syntax error in menu relationship field",NULL);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," menuPathList relationship active menuName menuModifiers menuFunction\"",NULL);
		}
	return(succeed?TCL_OK:TCL_ERROR);
}

static int Tcl_DeleteMenus(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* delete a menu from the menu list
 * NOTE: if this deletes an element, it will also delete the element's children
 */
{
	EDITORMENU
		*theMenu;
	UINT32
		theIndex;
	int
		pathArgc;
	char
		**pathArgv;
	BOOLEAN
		fail;

	fail=FALSE;
	if(argc>1)
		{
		for(theIndex=1;!fail&&theIndex<argc;theIndex++)
			{
			if(Tcl_SplitList(localInterpreter,argv[theIndex],&pathArgc,&pathArgv)==TCL_OK)
				{
				theMenu=NULL;										/* start at the root */
				if(GetEditorMenu(pathArgc,pathArgv,&theMenu))
					{
					DisposeEditorMenu(theMenu);
					}
				else
					{
					Tcl_AppendResult(localInterpreter,"Invalid menu specification",NULL);
					fail=TRUE;
					}
				ckfree(pathArgv);
				}
			else
				{
				fail=TRUE;
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," menuPathList ?menuPathList ...?\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_AddKeyBinding(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* add a key binding to the bindings list
 * NOTE: if a binding already exists, this will delete it before attempting
 * a creation
 */
{
	UINT32
		keyCode,
		modifierMask,
		modifierValue;

	if(argc==4)
		{
		if(EditorKeyNameToKeyCode(argv[1],&keyCode))						/* find the code for this key */
			{
			if(GetModifierMaskAndValue(localInterpreter,argv[2],&modifierMask,&modifierValue))
				{
				if(CreateEditorKeyBinding(keyCode,modifierMask,modifierValue,argv[3]))
					{
					return(TCL_OK);
					}
				else
					{
					GetError(&errorFamily,&errorFamilyMember,&errorDescription);
					Tcl_AppendResult(localInterpreter,errorDescription,NULL);
					}
				}
			}
		else
			{
			Tcl_AppendResult(localInterpreter,"Invalid key code",NULL);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," keyName modifiers keyFunction\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_DeleteKeyBinding(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* delete a key binding from the bindings list
 */
{
	UINT32
		keyCode,
		modifierMask,
		modifierValue;

	if(argc==3)
		{
		if(EditorKeyNameToKeyCode(argv[1],&keyCode))						/* find the code for this key */
			{
			if(GetModifierMaskAndValue(localInterpreter,argv[2],&modifierMask,&modifierValue))
				{
				if(DeleteEditorKeyBinding(keyCode,modifierMask,modifierValue))
					{
					return(TCL_OK);
					}
				else
					{
					Tcl_AppendResult(localInterpreter,"Key binding does not exist",NULL);
					}
				}
			}
		else
			{
			Tcl_AppendResult(localInterpreter,"Invalid key code",NULL);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," keyName modifiers\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_ListKeyBindings(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* dump all key bindings
 */
{
	UINT32
		keyCode,
		modifierMask,
		modifierValue;
	EDITORKEYBINDING
		*currentBinding;
	char
		*keyName,
		modifierString[TEMPSTRINGBUFFERLENGTH];
	BOOLEAN
		haveName,
		haveModifiers;
	BOOLEAN
		fail;
	BOOLEAN
		didOne;

	if(argc>=1&&argc<=3)
		{
		fail=FALSE;
		haveName=haveModifiers=FALSE;
		if(argc>=2)
			{
			haveName=TRUE;
			if(!EditorKeyNameToKeyCode(argv[1],&keyCode))						/* find the code for this key */
				{
				Tcl_AppendResult(localInterpreter,"Invalid key code",NULL);
				fail=TRUE;
				}
			}
		if(!fail&&argc>=3)
			{
			haveModifiers=TRUE;
			fail=!GetModifierMaskAndValue(localInterpreter,argv[2],&modifierMask,&modifierValue);
			}
		if(!fail)
			{
			currentBinding=keyBindingListHead;
			didOne=FALSE;
			while(currentBinding)
				{
				if(!haveName||(currentBinding->keyCode==keyCode))
					{
					if(!haveModifiers||((currentBinding->modifierMask==modifierMask)&&(currentBinding->modifierValue==modifierValue)))
						{
						if(keyName=EditorKeyCodeToKeyName(currentBinding->keyCode))
							{
							ModifierMaskAndValueToString(currentBinding->modifierMask,currentBinding->modifierValue,&(modifierString[0]));
							if(didOne)				/* do not put return on last line */
								{
								Tcl_AppendResult(localInterpreter,"\n",NULL);
								}
							Tcl_AppendResult(localInterpreter,"bindkey",NULL);
							Tcl_AppendElement(localInterpreter,keyName);
							Tcl_AppendElement(localInterpreter,&(modifierString[0]));
							Tcl_AppendElement(localInterpreter,&(currentBinding->dataText[0]));
							didOne=TRUE;
							}
						}
					}
				currentBinding=currentBinding->nextBinding;
				}
			return(TCL_OK);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," ?keyName? ?modifiers?\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_WaitKey(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* wait for a keyboard press, return the string for the code, and modifiers
 */
{
	UINT32
		keyCode,
		modifierValue;
	char
		*keyName,
		modifierString[TEMPSTRINGBUFFERLENGTH];

	if(argc==1)
		{
		EditorGetKeyPress(&keyCode,&modifierValue,TRUE,FALSE);
		if(keyName=EditorKeyCodeToKeyName(keyCode))
			{
			ModifierMaskAndValueToString(0xFFFFFFFF,modifierValue,&(modifierString[0]));
			Tcl_AppendElement(localInterpreter,keyName);
			Tcl_AppendElement(localInterpreter,&(modifierString[0]));
			return(TCL_OK);
			}
		else
			{
			Tcl_AppendResult(localInterpreter,"Unnamed key",NULL);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0],"\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_GetKey(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* check to see if there is a keyboard event in the queue, if so, get it, if not, return an error
 */
{
	UINT32
		keyCode,
		modifierValue;
	char
		*keyName,
		modifierString[TEMPSTRINGBUFFERLENGTH];

	if(argc==1)
		{
		if(EditorGetKeyPress(&keyCode,&modifierValue,FALSE,FALSE))
			{
			if(keyName=EditorKeyCodeToKeyName(keyCode))
				{
				ModifierMaskAndValueToString(0xFFFFFFFF,modifierValue,&(modifierString[0]));
				Tcl_AppendElement(localInterpreter,keyName);
				Tcl_AppendElement(localInterpreter,&(modifierString[0]));
				return(TCL_OK);
				}
			else
				{
				Tcl_AppendResult(localInterpreter,"Unnamed key",NULL);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0],"\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_NewBuffer(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* create a new buffer with the passed name
 * if a buffer already exists with the passed name fail
 */
{
	EDITORBUFFER
		*theBuffer;

	if(argc==2)
		{
		if(theBuffer=EditorNewBuffer(argv[1]))					/* create a new buffer */
			{
			theBuffer->shellBusy=FALSE;							/* not being modified at the moment */
			Tcl_AppendResult(localInterpreter,theBuffer->contentName,NULL);
			return(TCL_OK);
			}
		else
			{
			GetError(&errorFamily,&errorFamilyMember,&errorDescription);
			Tcl_AppendResult(localInterpreter,errorDescription,NULL);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_OpenBuffer(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* open the passed file into a buffer, return the buffer name
 */
{
	EDITORBUFFER
		*theBuffer;

	if(argc==2)
		{
		if(theBuffer=EditorOpenBuffer(argv[1]))
			{
			theBuffer->shellBusy=FALSE;							/* not being modified at the moment */
			Tcl_AppendResult(localInterpreter,theBuffer->contentName,NULL);
			return(TCL_OK);
			}
		else
			{
			GetError(&errorFamily,&errorFamilyMember,&errorDescription);
			Tcl_AppendResult(localInterpreter,errorDescription,NULL);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," pathName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_CloseBuffer(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* close the passed buffer as long as it is not in use, or being modified
 */
{
	EDITORBUFFER
		*theBuffer;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				EditorCloseBuffer(theBuffer);
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_SaveBuffer(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* save the passed buffer
 */
{
	EDITORBUFFER
		*theBuffer;
	BOOLEAN
		result;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				result=EditorSaveBuffer(theBuffer);
				ClearBufferBusy(theBuffer);
				if(result)
					{
					return(TCL_OK);
					}
				else
					{
					GetError(&errorFamily,&errorFamilyMember,&errorDescription);
					Tcl_AppendResult(localInterpreter,errorDescription,NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_SaveBufferAs(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* save the passed buffer to newPath
 * rename the buffer to the new name, clear dirty
 * NOTE: this returns the new name of theBuffer after it is saved
 */
{
	EDITORBUFFER
		*theBuffer;
	BOOLEAN
		result;

	if(argc==3)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				result=EditorSaveBufferAs(theBuffer,argv[2]);
				ClearBufferBusy(theBuffer);
				if(result)
					{
					Tcl_AppendResult(localInterpreter,theBuffer->contentName,NULL);
					return(TCL_OK);
					}
				else
					{
					GetError(&errorFamily,&errorFamilyMember,&errorDescription);
					Tcl_AppendResult(localInterpreter,errorDescription,NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName newPath\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_SaveBufferTo(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* save the passed buffer to a file, leave the passed buffer dirty if it was before
 */
{
	EDITORBUFFER
		*theBuffer;
	BOOLEAN
		result;

	if(argc==3)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				result=EditorSaveBufferTo(theBuffer,argv[2]);
				ClearBufferBusy(theBuffer);
				if(result)
					{
					return(TCL_OK);
					}
				else
					{
					GetError(&errorFamily,&errorFamilyMember,&errorDescription);
					Tcl_AppendResult(localInterpreter,errorDescription,NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName pathName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_RevertBuffer(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* revert the passed buffer
 */
{
	EDITORBUFFER
		*theBuffer;
	BOOLEAN
		result;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				result=EditorRevertBuffer(theBuffer);
				ClearBufferBusy(theBuffer);
				if(result)
					{
					return(TCL_OK);
					}
				else
					{
					GetError(&errorFamily,&errorFamilyMember,&errorDescription);
					Tcl_AppendResult(localInterpreter,errorDescription,NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_IsBufferDirty(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* return 1 if the passed buffer is dirty, 0 if not
 * if the buffer cannot be located, fail
 */
{
	EDITORBUFFER
		*theBuffer;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(theBuffer->editorUniverse->dirty)
				{
				Tcl_AppendResult(localInterpreter,"1",NULL);
				}
			else
				{
				Tcl_AppendResult(localInterpreter,"0",NULL);
				}
			return(TCL_OK);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_IsBufferFromFile(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* return 1 if the passed buffer is linked to a file, 0 if not
 * if the buffer cannot be located, fail
 */
{
	EDITORBUFFER
		*theBuffer;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(theBuffer->fromFile)
				{
				Tcl_AppendResult(localInterpreter,"1",NULL);
				}
			else
				{
				Tcl_AppendResult(localInterpreter,"0",NULL);
				}
			return(TCL_OK);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_BufferHasWindow(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* return 1 if the passed buffer has a window, 0 if not
 * if the buffer cannot be located, fail
 */
{
	EDITORBUFFER
		*theBuffer;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(theBuffer->theWindow)
				{
				Tcl_AppendResult(localInterpreter,"1",NULL);
				}
			else
				{
				Tcl_AppendResult(localInterpreter,"0",NULL);
				}
			return(TCL_OK);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_ClearBufferDirty(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* clear the dirty flag for the passed buffer
 * if the buffer cannot be located, fail
 */
{
	EDITORBUFFER
		*theBuffer;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			SetBufferBusy(theBuffer);
			EditorStartTextChange(theBuffer->editorUniverse);			/* start text change, so that status bars can update if needed */
			theBuffer->editorUniverse->dirty=FALSE;
			EditorEndTextChange(theBuffer->editorUniverse);				/* change is complete */
			ClearBufferBusy(theBuffer);
			return(TCL_OK);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_BufferList(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* return the current list of document buffers
 */
{
	BOOLEAN
		fail;
	EDITORBUFFER
		*testBuffer;

	fail=FALSE;
	if(argc==1)
		{
		testBuffer=EditorGetFirstBuffer();
		while(testBuffer)
			{
			Tcl_AppendElement(localInterpreter,testBuffer->contentName);
			testBuffer=EditorGetNextBuffer(testBuffer);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0],"\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_WindowList(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* return the current list of document windows
 */
{
	BOOLEAN
		fail;
	UINT32
		windowIndex,
		numWindows;
	EDITORWINDOW
		**windowList;

	fail=FALSE;
	if(argc==1)
		{
		if(GetSortedDocumentWindowList(&numWindows,&windowList))
			{
			for(windowIndex=0;windowIndex<numWindows;windowIndex++)
				{
				Tcl_AppendElement(localInterpreter,windowList[windowIndex]->theBuffer->contentName);
				}
			MDisposePtr(windowList);
			}
		else
			{
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0],"\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_OpenWindow(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* open a window onto the given buffer
 * if there is already a window open on the buffer, just bring it to the top
 */
{
	EDITORBUFFER
		*theBuffer;
	EDITORWINDOW
		*theWindow;
	EDITORRECT
		theRect;
	UINT32
		tabSize;
	EDITORCOLOR
		foreground,
		background;

	if(argc==10)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(GetRectangle(localInterpreter,&(argv[2]),&theRect))
				{
				if(GetUINT32(localInterpreter,argv[7],&tabSize))
					{
					if(GetColor(localInterpreter,argv[8],&foreground))
						{
						if(GetColor(localInterpreter,argv[9],&background))
							{
							if(BufferNotBusy(localInterpreter,theBuffer))
								{
								SetBufferBusy(theBuffer);
								theWindow=EditorOpenDocumentWindow(theBuffer,&theRect,argv[6],tabSize,foreground,background);
								ClearBufferBusy(theBuffer);
								if(theWindow)
									{
									return(TCL_OK);
									}
								else
									{
									GetError(&errorFamily,&errorFamilyMember,&errorDescription);
									Tcl_AppendResult(localInterpreter,errorDescription,NULL);
									}
								}
							}
						}
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName x y w h fontName tabSize foregroundColor backgroundColor\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_CloseWindow(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* close a given window
 */
{
	EDITORWINDOW
		*theWindow;
	EDITORBUFFER
		*theBuffer;

	if(argc==2)
		{
		if(theWindow=GetEditorWindow(localInterpreter,argv[1]))
			{
			theBuffer=theWindow->theBuffer;								/* hang onto this, we will need it after the window closes */
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				EditorCloseDocumentWindow(theWindow);
				ClearBufferBusy(theBuffer);
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_SetRect(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* set the rectangle of a window
 */
{
	EDITORWINDOW
		*theWindow;
	EDITORRECT
		theRect;
	BOOLEAN
		result;

	if(argc==6)
		{
		if(theWindow=GetEditorWindow(localInterpreter,argv[1]))
			{
			if(GetRectangle(localInterpreter,&(argv[2]),&theRect))
				{
				if(BufferNotBusy(localInterpreter,theWindow->theBuffer))
					{
					SetBufferBusy(theWindow->theBuffer);
					result=SetEditorDocumentWindowRect(theWindow,&theRect);
					ClearBufferBusy(theWindow->theBuffer);
					if(result)
						{
						return(TCL_OK);
						}
					else
						{
						GetError(&errorFamily,&errorFamilyMember,&errorDescription);
						Tcl_AppendResult(localInterpreter,errorDescription,NULL);
						}
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName x y w h\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_GetRect(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* get the rectangle of a window
 */
{
	EDITORWINDOW
		*theWindow;
	EDITORRECT
		theRect;
	BOOLEAN
		result;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];

	if(argc==2)
		{
		if(theWindow=GetEditorWindow(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theWindow->theBuffer))
				{
				SetBufferBusy(theWindow->theBuffer);
				result=GetEditorDocumentWindowRect(theWindow,&theRect);
				ClearBufferBusy(theWindow->theBuffer);
				if(result)
					{
					sprintf(tempString,"%d %d %d %d",theRect.x,theRect.y,theRect.w,theRect.h);
					Tcl_AppendResult(localInterpreter,tempString,NULL);
					return(TCL_OK);
					}
				else
					{
					GetError(&errorFamily,&errorFamilyMember,&errorDescription);
					Tcl_AppendResult(localInterpreter,errorDescription,NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_SetFont(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* set the font of a window
 */
{
	EDITORWINDOW
		*theWindow;
	EDITORVIEW
		*theView;
	BOOLEAN
		result;

	if(argc==3)
		{
		if(theWindow=GetEditorWindow(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theWindow->theBuffer))
				{
				SetBufferBusy(theWindow->theBuffer);
				theView=GetDocumentWindowCurrentView(theWindow);
				result=SetEditorViewFont(theView,argv[2]);
				ClearBufferBusy(theWindow->theBuffer);
				if(result)
					{
					return(TCL_OK);
					}
				else
					{
					GetError(&errorFamily,&errorFamilyMember,&errorDescription);
					Tcl_AppendResult(localInterpreter,errorDescription,NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName fontName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_GetFont(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* get the font of a window
 */
{
	EDITORWINDOW
		*theWindow;
	EDITORVIEW
		*theView;
	BOOLEAN
		result;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];

	if(argc==2)
		{
		if(theWindow=GetEditorWindow(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theWindow->theBuffer))
				{
				SetBufferBusy(theWindow->theBuffer);
				theView=GetDocumentWindowCurrentView(theWindow);
				result=GetEditorViewFont(theView,&tempString[0],TEMPSTRINGBUFFERLENGTH);
				ClearBufferBusy(theWindow->theBuffer);
				if(result)
					{
					Tcl_AppendResult(localInterpreter,tempString,NULL);
					return(TCL_OK);
					}
				else
					{
					Tcl_AppendResult(localInterpreter,"Failed to get font",NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_SetTabSize(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* set the tab size of a window
 */
{
	EDITORWINDOW
		*theWindow;
	EDITORVIEW
		*theView;
	UINT32
		tabSize;
	BOOLEAN
		result;

	if(argc==3)
		{
		if(theWindow=GetEditorWindow(localInterpreter,argv[1]))
			{
			if(GetUINT32(localInterpreter,argv[2],&tabSize))
				{
				if(BufferNotBusy(localInterpreter,theWindow->theBuffer))
					{
					SetBufferBusy(theWindow->theBuffer);
					theView=GetDocumentWindowCurrentView(theWindow);
					result=SetEditorViewTabSize(theView,tabSize);
					ClearBufferBusy(theWindow->theBuffer);
					if(result)
						{
						return(TCL_OK);
						}
					else
						{
						GetError(&errorFamily,&errorFamilyMember,&errorDescription);
						Tcl_AppendResult(localInterpreter,errorDescription,NULL);
						}
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName tabSize\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_GetTabSize(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* get the tab size of a window
 */
{
	EDITORWINDOW
		*theWindow;
	EDITORVIEW
		*theView;
	UINT32
		tabSize;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];

	if(argc==2)
		{
		if(theWindow=GetEditorWindow(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theWindow->theBuffer))
				{
				SetBufferBusy(theWindow->theBuffer);
				theView=GetDocumentWindowCurrentView(theWindow);
				tabSize=GetEditorViewTabSize(theView);
				ClearBufferBusy(theWindow->theBuffer);
				sprintf(tempString,"%d",tabSize);
				Tcl_AppendResult(localInterpreter,tempString,NULL);
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_SetColors(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* set the colors of a window
 */
{
	EDITORWINDOW
		*theWindow;
	EDITORCOLOR
		foregroundColor,
		backgroundColor;
	BOOLEAN
		result;

	if(argc==4)
		{
		if(theWindow=GetEditorWindow(localInterpreter,argv[1]))
			{
			if(GetColor(localInterpreter,argv[2],&foregroundColor))
				{
				if(GetColor(localInterpreter,argv[3],&backgroundColor))
					{
					if(BufferNotBusy(localInterpreter,theWindow->theBuffer))
						{
						SetBufferBusy(theWindow->theBuffer);
						result=SetEditorDocumentWindowColors(theWindow,foregroundColor,backgroundColor);
						ClearBufferBusy(theWindow->theBuffer);
						if(result)
							{
							return(TCL_OK);
							}
						else
							{
							GetError(&errorFamily,&errorFamilyMember,&errorDescription);
							Tcl_AppendResult(localInterpreter,errorDescription,NULL);
							}
						}
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName foregroundColor backgroundColor\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_GetColors(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* get the colors of a window
 */
{
	EDITORWINDOW
		*theWindow;
	EDITORCOLOR
		foregroundColor,
		backgroundColor;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];
	BOOLEAN
		result;

	if(argc==2)
		{
		if(theWindow=GetEditorWindow(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theWindow->theBuffer))
				{
				SetBufferBusy(theWindow->theBuffer);
				result=GetEditorDocumentWindowColors(theWindow,&foregroundColor,&backgroundColor);
				ClearBufferBusy(theWindow->theBuffer);
				if(result)
					{
					sprintf(tempString,"%06X %06X",foregroundColor,backgroundColor);
					Tcl_AppendResult(localInterpreter,tempString,NULL);
					return(TCL_OK);
					}
				else
					{
					GetError(&errorFamily,&errorFamilyMember,&errorDescription);
					Tcl_AppendResult(localInterpreter,errorDescription,NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_SetTopWindow(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* make the passed window the top/active window
 */
{
	EDITORWINDOW
		*theWindow;

	if(argc==2)
		{
		if(theWindow=GetEditorWindow(localInterpreter,argv[1]))
			{
			SetTopDocumentWindow(theWindow);
			return(TCL_OK);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_GetActiveWindow(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* return the active window name, or error if none
 */
{
	EDITORWINDOW
		*theWindow;

	if(argc==1)
		{
		if(theWindow=GetActiveDocumentWindow())
			{
			Tcl_AppendResult(localInterpreter,theWindow->theBuffer->contentName,NULL);
			return(TCL_OK);
			}
		else
			{
			Tcl_AppendResult(localInterpreter,"No active window",NULL);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0],"\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_UpdateWindows(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* update any window that has invalid areas
 * on some implementations, this may do nothing
 */
{
	if(argc==1)
		{
		UpdateEditorWindows();
		return(TCL_OK);
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0],"\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_ScreenSize(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* return the dimensions of the screen that the editor is running on
 */
{
	UINT32
		theWidth,
		theHeight;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];

	if(argc==1)
		{
		GetEditorScreenDimensions(&theWidth,&theHeight);
		sprintf(tempString,"%lu %lu",theWidth,theHeight);
		Tcl_AppendResult(localInterpreter,tempString,NULL);
		return(TCL_OK);
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0],"\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_UndoRedoToggle(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* perform an undo toggle in the given buffer
 */
{
	EDITORBUFFER
		*theBuffer;
	BOOLEAN
		result;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				result=EditorToggleUndo(theBuffer->editorUniverse);
				Tcl_AppendResult(localInterpreter,result?"1":"0",NULL);
				ClearBufferBusy(theBuffer);
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_Undo(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* perform an undo in the given buffer
 */
{
	EDITORBUFFER
		*theBuffer;
	BOOLEAN
		result;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				result=EditorUndo(theBuffer->editorUniverse);
				Tcl_AppendResult(localInterpreter,result?"1":"0",NULL);
				ClearBufferBusy(theBuffer);
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_Redo(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* perform a redo in the given buffer
 */
{
	EDITORBUFFER
		*theBuffer;
	BOOLEAN
		result;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				result=EditorRedo(theBuffer->editorUniverse);
				Tcl_AppendResult(localInterpreter,result?"1":"0",NULL);
				ClearBufferBusy(theBuffer);
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_BreakUndo(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* cause a break in the undo flow for the passed buffer
 */
{
	EDITORBUFFER
		*theBuffer;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				BeginUndoGroup(theBuffer->editorUniverse);
				StrictEndUndoGroup(theBuffer->editorUniverse);
				ClearBufferBusy(theBuffer);
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_FlushUndos(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* perform a clear of the undo/redo buffers in the given buffer
 */
{
	EDITORBUFFER
		*theBuffer;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				EditorClearUndo(theBuffer->editorUniverse);
				ClearBufferBusy(theBuffer);
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_Cut(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* perform a cut in the given buffer
 */
{
	EDITORBUFFER
		*theBuffer;
	EDITORBUFFER
		*theClipboard;
	BOOLEAN
		fail;

	fail=FALSE;
	if(argc==2||argc==3)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				if(argc==3)
					{
					theClipboard=LocateEditorBuffer(argv[2]);
					}
				else
					{
					theClipboard=EditorGetCurrentClipboard();
					}
				if(theClipboard)
					{
					if(BufferNotBusy(localInterpreter,theClipboard))
						{
						SetBufferBusy(theClipboard);
						EditorCut(theBuffer->editorUniverse,theClipboard,FALSE);
						ClearBufferBusy(theClipboard);
						}
					else
						{
						fail=TRUE;
						}
					}
				else
					{
					Tcl_AppendResult(localInterpreter,"No clipboard",NULL);
					fail=TRUE;
					}
				ClearBufferBusy(theBuffer);
				}
			else
				{
				fail=TRUE;
				}
			}
		else
			{
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName ?clipboardBuffer?\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_Copy(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* perform a copy in the given buffer
 */
{
	EDITORBUFFER
		*theBuffer;
	EDITORBUFFER
		*theClipboard;
	BOOLEAN
		fail;

	fail=FALSE;
	if(argc==2||argc==3)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				if(argc==3)
					{
					theClipboard=LocateEditorBuffer(argv[2]);
					}
				else
					{
					theClipboard=EditorGetCurrentClipboard();
					}
				if(theClipboard)
					{
					if(BufferNotBusy(localInterpreter,theClipboard))
						{
						SetBufferBusy(theClipboard);
						EditorCopy(theBuffer->editorUniverse,theClipboard,FALSE);
						ClearBufferBusy(theClipboard);
						}
					else
						{
						fail=TRUE;
						}
					}
				else
					{
					Tcl_AppendResult(localInterpreter,"No clipboard",NULL);
					fail=TRUE;
					}
				ClearBufferBusy(theBuffer);
				}
			else
				{
				fail=TRUE;
				}
			}
		else
			{
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName ?clipboardBuffer?\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_Paste(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* perform a paste in the given buffer
 */
{
	EDITORBUFFER
		*theBuffer;
	EDITORBUFFER
		*theClipboard;
	BOOLEAN
		fail;

	fail=FALSE;
	if(argc==2||argc==3)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				if(argc==3)
					{
					theClipboard=LocateEditorBuffer(argv[2]);
					}
				else
					{
					theClipboard=EditorGetCurrentClipboard();
					}
				if(theClipboard)
					{
					if(BufferNotBusy(localInterpreter,theClipboard))
						{
						SetBufferBusy(theClipboard);
						EditorPaste(theBuffer->editorUniverse,theClipboard);
						ClearBufferBusy(theClipboard);
						}
					else
						{
						fail=TRUE;
						}
					}
				else
					{
					Tcl_AppendResult(localInterpreter,"No clipboard",NULL);
					fail=TRUE;
					}
				ClearBufferBusy(theBuffer);
				}
			else
				{
				fail=TRUE;
				}
			}
		else
			{
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName ?clipboardBuffer?\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_ColumnarPaste(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* perform a columnar paste in the given window
 */
{
	EDITORWINDOW
		*theWindow;
	EDITORBUFFER
		*theClipboard;
	EDITORVIEW
		*theView;
	BOOLEAN
		fail;

	fail=FALSE;
	if(argc==2||argc==3)
		{
		if(theWindow=GetEditorWindow(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theWindow->theBuffer))
				{
				theView=GetDocumentWindowCurrentView(theWindow);
				SetBufferBusy(theWindow->theBuffer);
				if(argc==3)
					{
					theClipboard=LocateEditorBuffer(argv[2]);
					}
				else
					{
					theClipboard=EditorGetCurrentClipboard();
					}
				if(theClipboard)
					{
					if(BufferNotBusy(localInterpreter,theClipboard))
						{
						SetBufferBusy(theClipboard);
						EditorColumnarPaste(theView,theClipboard);
						ClearBufferBusy(theClipboard);
						}
					else
						{
						fail=TRUE;
						}
					}
				else
					{
					Tcl_AppendResult(localInterpreter,"No clipboard",NULL);
					fail=TRUE;
					}
				ClearBufferBusy(theWindow->theBuffer);
				}
			else
				{
				fail=TRUE;
				}
			}
		else
			{
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName ?clipboardBuffer?\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_GetClipboard(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* return the current clipboard, fail if there is none
 */
{
	EDITORBUFFER
		*theClipboard;

	if(argc==1)
		{
		if(theClipboard=EditorGetCurrentClipboard())
			{
			Tcl_AppendResult(localInterpreter,theClipboard->contentName,NULL);
			return(TCL_OK);
			}
		else
			{
			Tcl_AppendResult(localInterpreter,"No clipboard",NULL);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0],"\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_SetClipboard(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* set the current clipboard to the given buffer
 */
{
	EDITORBUFFER
		*theClipboard;

	if(argc==2)
		{
		if(theClipboard=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theClipboard))
				{
				EditorSetCurrentClipboard(theClipboard);
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," clipboardBuffer\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_UnSetClipboard(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* clear the current clipboard
 */
{
	if(argc==1)
		{
		EditorSetCurrentClipboard(NULL);
		return(TCL_OK);
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0],"\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_OkDialog(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* create a dialog with some text, and an OK button
 */
{
	BOOLEAN
		fail;

	fail=FALSE;
	if(argc==2)
		{
		if(OkDialog(argv[1]))
			{
			ClearAbort();				/* if user managed to send break sequence during the dialog, we would rather not abort now! */
			}
		else
			{
			GetError(&errorFamily,&errorFamilyMember,&errorDescription);
			Tcl_AppendResult(localInterpreter,errorDescription,NULL);
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," dialogText\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_OkCancelDialog(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* create a dialog with some text, and an OK button, and a cancel button
 */
{
	BOOLEAN
		cancel,
		fail;

	fail=FALSE;
	if(argc==2)
		{
		if(OkCancelDialog(argv[1],&cancel))
			{
			if(cancel)
				{
				fail=TRUE;
				}
			ClearAbort();				/* if user managed to send break sequence during the dialog, we would rather not abort now! */
			}
		else
			{
			GetError(&errorFamily,&errorFamilyMember,&errorDescription);
			Tcl_AppendResult(localInterpreter,errorDescription,NULL);
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," dialogText\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_YesNoCancelDialog(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* create a dialog with some text, and yes, no, and cancel buttons
 */
{
	BOOLEAN
		yes,
		cancel,
		fail;

	fail=FALSE;
	if(argc==2)
		{
		if(YesNoCancelDialog(argv[1],&yes,&cancel))
			{
			if(cancel)
				{
				fail=TRUE;
				}
			else
				{
				Tcl_AppendResult(localInterpreter,yes?"1":"0",NULL);
				}
			ClearAbort();				/* if user managed to send break sequence during the dialog, we would rather not abort now! */
			}
		else
			{
			GetError(&errorFamily,&errorFamilyMember,&errorDescription);
			Tcl_AppendResult(localInterpreter,errorDescription,NULL);
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," dialogText\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_TextDialog(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* create a get text dialog
 * if InitialText is not specified, then no initial text will be present in the dialog
 * NOTE: since this can indirectly modify the current clipboard, the clipboard must not be busy
 */
{
	BOOLEAN
		cancel,
		fail;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];

	fail=FALSE;
	if(argc==2||argc==3)
		{
		if(argc==3)
			{
			strncpy(tempString,argv[2],TEMPSTRINGBUFFERLENGTH-1);
			tempString[TEMPSTRINGBUFFERLENGTH-1]='\0';
			}
		else
			{
			tempString[0]='\0';
			}
		if(CurrentClipboardNotBusy(localInterpreter))
			{
			if(GetSimpleTextDialog(argv[1],&(tempString[0]),TEMPSTRINGBUFFERLENGTH,&cancel))
				{
				if(cancel)
					{
					fail=TRUE;
					}
				else
					{
					Tcl_AppendResult(localInterpreter,&(tempString[0]),NULL);
					}
				ClearAbort();				/* if user managed to send break sequence during the dialog, we would rather not abort now! */
				}
			else
				{
				GetError(&errorFamily,&errorFamilyMember,&errorDescription);
				Tcl_AppendResult(localInterpreter,errorDescription,NULL);
				fail=TRUE;
				}
			}
		else
			{
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," dialogTitle ?initialText?\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_SearchDialog(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* create a search dialog
 * findBuffer, and replaceBuffer must exist, or an error will be returned
 * NOTE: currently DialogTitle is ignored, later it should be implemented
 * NOTE: since this can indirectly modify the current clipboard, the clipboard must not be busy
 */
{
	BOOLEAN
		flagList[6];
	UINT16
		searchMode;
	EDITORBUFFER
		*findBuffer,
		*replaceBuffer;
	BOOLEAN
		cancel,
		result;

	result=FALSE;
	if(argc==10)
		{
		if(findBuffer=GetEditorBuffer(localInterpreter,argv[2]))
			{
			if(BufferNotBusy(localInterpreter,findBuffer))
				{
				if(replaceBuffer=GetEditorBuffer(localInterpreter,argv[3]))
					{
					if(BufferNotBusy(localInterpreter,replaceBuffer))
						{
						if(GetVarFlagList(localInterpreter,6,&(argv[4]),&(flagList[0])))
							{
							SetBufferBusy(findBuffer);
							SetBufferBusy(replaceBuffer);
							if(CurrentClipboardNotBusy(localInterpreter))
								{
								if(SearchReplaceDialog(findBuffer->editorUniverse,replaceBuffer->editorUniverse,&(flagList[0]),&(flagList[1]),&(flagList[2]),&(flagList[3]),&(flagList[4]),&(flagList[5]),&searchMode,&cancel))
									{
									if(!cancel)
										{
										result=TRUE;
										}
									ClearAbort();				/* if user managed to send break sequence during the dialog, we would rather not abort now! */
									PutVarFlagList(localInterpreter,6,&(argv[4]),&(flagList[0]));	/* put these back even if user cancelled */
									}
								else
									{
									GetError(&errorFamily,&errorFamilyMember,&errorDescription);
									Tcl_AppendResult(localInterpreter,errorDescription,NULL);
									}
								}
							ClearBufferBusy(replaceBuffer);
							ClearBufferBusy(findBuffer);
							if(result)
								{
								switch(searchMode)
									{
									case ST_FIND:
										Tcl_AppendResult(localInterpreter,"find",NULL);
										break;
									case ST_FINDALL:
										Tcl_AppendResult(localInterpreter,"findall",NULL);
										break;
									case ST_REPLACE:
										Tcl_AppendResult(localInterpreter,"replace",NULL);
										break;
									case ST_REPLACEALL:
										Tcl_AppendResult(localInterpreter,"replaceall",NULL);
										break;
									}
								return(TCL_OK);
								}
							}
						}
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," dialogTitle findBuffer replaceBuffer backwardsVar wrapAroundVar selectionExprVar ignoreCaseVar limitScopeVar replaceProcVar\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_ListDialog(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* create a list dialog, handle it, and if all is well, return a new list
 * that contains the chosen elements of the old list
 */
{
	UINT32
		theIndex;
	int
		listArgc;
	char
		**listArgv;
	BOOLEAN
		*selectedElements;
	BOOLEAN
		cancel,
		fail;

	fail=FALSE;
	if(argc==3)
		{
		if(Tcl_SplitList(localInterpreter,argv[2],&listArgc,&listArgv)==TCL_OK)
			{
			if(selectedElements=(BOOLEAN *)MNewPtrClr(sizeof(BOOLEAN)*listArgc))	/* get array of "FALSEs" */
				{
				if(listArgc)
					{
					selectedElements[0]=TRUE;	/* select the first element if there is one */
					}
				if(SimpleListBoxDialog(argv[1],listArgc,listArgv,selectedElements,&cancel))	/* let user choose elements */
					{
					if(!cancel)
						{
						for(theIndex=0;theIndex<listArgc;theIndex++)
							{
							if(selectedElements[theIndex])
								{
								Tcl_AppendElement(localInterpreter,listArgv[theIndex]);	/* make a result that contains the selected elements */
								}
							}
						}
					else
						{
						fail=TRUE;
						}
					ClearAbort();				/* if user managed to send break sequence during the dialog, we would rather not abort now! */
					}
				else
					{
					GetError(&errorFamily,&errorFamilyMember,&errorDescription);
					Tcl_AppendResult(localInterpreter,errorDescription,NULL);
					fail=TRUE;
					}
				MDisposePtr(selectedElements);
				}
			else
				{
				Tcl_AppendResult(localInterpreter,"Failed to allocate memory for list result",NULL);
				fail=TRUE;
				}
			ckfree(listArgv);
			}
		else
			{
			/* SplitList will leave a result if it fails */
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," dialogTitle inputList\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_OpenDialog(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* create a get file(s) dialog
 * if InitialText is not specified, then no initial text will be present in the dialog
 * NOTE: since this can indirectly modify the current clipboard, the clipboard must not be busy
 */
{
	BOOLEAN
		cancel,
		fail;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];
	UINT32
		theIndex;
	char
		**thePaths;

	fail=FALSE;
	if(argc==2||argc==3)
		{
		if(argc==3)
			{
			strncpy(tempString,argv[2],TEMPSTRINGBUFFERLENGTH-1);
			tempString[TEMPSTRINGBUFFERLENGTH-1]='\0';
			}
		else
			{
			tempString[0]='\0';
			}
		if(CurrentClipboardNotBusy(localInterpreter))
			{
			if(OpenFileDialog(argv[1],&(tempString[0]),TEMPSTRINGBUFFERLENGTH,&thePaths,&cancel))
				{
				if(!cancel)
					{
					theIndex=0;
					while(thePaths[theIndex])
						{
						Tcl_AppendElement(localInterpreter,thePaths[theIndex++]);
						}
					FreeOpenFileDialogPaths(thePaths);
					}
				else
					{
					fail=TRUE;
					}
				ClearAbort();				/* if user managed to send break sequence during the dialog, we would rather not abort now! */
				}
			else
				{
				GetError(&errorFamily,&errorFamilyMember,&errorDescription);
				Tcl_AppendResult(localInterpreter,errorDescription,NULL);
				fail=TRUE;
				}
			}
		else
			{
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," dialogTitle ?initialText?\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_SaveDialog(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* create a save file dialog
 * if InitialText is not specified, then no initial text will be present in the dialog
 * NOTE: since this can indirectly modify the current clipboard, the clipboard must not be busy
 */
{
	BOOLEAN
		cancel,
		fail;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];

	fail=FALSE;
	if(argc==2||argc==3)
		{
		if(argc==3)
			{
			strncpy(tempString,argv[2],TEMPSTRINGBUFFERLENGTH-1);
			tempString[TEMPSTRINGBUFFERLENGTH-1]='\0';
			}
		else
			{
			tempString[0]='\0';
			}
		if(CurrentClipboardNotBusy(localInterpreter))
			{
			if(SaveFileDialog(argv[1],&(tempString[0]),TEMPSTRINGBUFFERLENGTH,&cancel))
				{
				if(!cancel)
					{
					Tcl_AppendResult(localInterpreter,&(tempString[0]),NULL);
					}
				else
					{
					fail=TRUE;
					}
				ClearAbort();				/* if user managed to send break sequence during the dialog, we would rather not abort now! */
				}
			else
				{
				GetError(&errorFamily,&errorFamilyMember,&errorDescription);
				Tcl_AppendResult(localInterpreter,errorDescription,NULL);
				fail=TRUE;
				}
			}
		else
			{
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," dialogTitle ?initialText?\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_PathDialog(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* create a path choose dialog
 * if InitialText is not specified, then no initial text will be present in the dialog
 * NOTE: since this can indirectly modify the current clipboard, the clipboard must not be busy
 */
{
	BOOLEAN
		cancel,
		fail;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];

	fail=FALSE;
	if(argc==2||argc==3)
		{
		if(argc==3)
			{
			strncpy(tempString,argv[2],TEMPSTRINGBUFFERLENGTH-1);
			tempString[TEMPSTRINGBUFFERLENGTH-1]='\0';
			}
		else
			{
			tempString[0]='\0';
			}
		if(CurrentClipboardNotBusy(localInterpreter))
			{
			if(ChoosePathDialog(argv[1],&(tempString[0]),TEMPSTRINGBUFFERLENGTH,&cancel))
				{
				if(!cancel)
					{
					Tcl_AppendResult(localInterpreter,&(tempString[0]),NULL);
					}
				else
					{
					fail=TRUE;
					}
				ClearAbort();				/* if user managed to send break sequence during the dialog, we would rather not abort now! */
				}
			else
				{
				GetError(&errorFamily,&errorFamilyMember,&errorDescription);
				Tcl_AppendResult(localInterpreter,errorDescription,NULL);
				fail=TRUE;
				}
			}
		else
			{
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," dialogTitle ?initialText?\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_FontDialog(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* create a font selection dialog
 * if InitialFont is not specified, then no initial text will be present in the dialog
 * NOTE: since this can indirectly modify the current clipboard, the clipboard must not be busy
 */
{
	BOOLEAN
		cancel,
		fail;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];

	fail=FALSE;
	if(argc==2||argc==3)
		{
		if(argc==3)
			{
			strncpy(tempString,argv[2],TEMPSTRINGBUFFERLENGTH-1);
			tempString[TEMPSTRINGBUFFERLENGTH-1]='\0';
			}
		else
			{
			tempString[0]='\0';
			}
		if(CurrentClipboardNotBusy(localInterpreter))
			{
			if(ChooseFontDialog(argv[1],&(tempString[0]),TEMPSTRINGBUFFERLENGTH,&cancel))
				{
				if(!cancel)
					{
					Tcl_AppendResult(localInterpreter,&(tempString[0]),NULL);
					}
				else
					{
					fail=TRUE;
					}
				ClearAbort();				/* if user managed to send break sequence during the dialog, we would rather not abort now! */
				}
			else
				{
				GetError(&errorFamily,&errorFamilyMember,&errorDescription);
				Tcl_AppendResult(localInterpreter,errorDescription,NULL);
				fail=TRUE;
				}
			}
		else
			{
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," dialogTitle ?initialFont?\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_Find(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* find within the given buffer
 * findBuffer must exist, or an error will be returned
 * if there is no error, then if there as a match, the first match position is returned, if no
 * matches, -1 is returned
 */
{
	BOOLEAN
		flagList[4];
	EDITORBUFFER
		*theBuffer;
	EDITORBUFFER
		*findBuffer;
	BOOLEAN
		foundMatch;
	UINT32
		matchPosition;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];
	BOOLEAN
		fail;

	fail=FALSE;
	if(argc==7)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				if(findBuffer=GetEditorBuffer(localInterpreter,argv[2]))
					{
					if(BufferNotBusy(localInterpreter,findBuffer))
						{
						SetBufferBusy(findBuffer);
						if(GetFlagList(localInterpreter,4,&(argv[3]),&(flagList[0])))
							{
							if(EditorFind(theBuffer->editorUniverse,findBuffer->editorUniverse->textUniverse,flagList[0],flagList[1],flagList[2],flagList[3],&foundMatch,&matchPosition))
								{
								sprintf(tempString,"%ld",foundMatch?matchPosition:-1);
								Tcl_AppendResult(localInterpreter,tempString,NULL);
								}
							else
								{
								GetError(&errorFamily,&errorFamilyMember,&errorDescription);
								Tcl_AppendResult(localInterpreter,errorDescription,NULL);
								fail=TRUE;
								}
							}
						else
							{
							fail=TRUE;
							}
						ClearBufferBusy(findBuffer);
						}
					else
						{
						fail=TRUE;
						}
					}
				else
					{
					fail=TRUE;
					}
				ClearBufferBusy(theBuffer);
				}
			else
				{
				fail=TRUE;
				}
			}
		else
			{
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," searchInBuffer findBuffer backwards wrapAround selectionExpr ignoreCase\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_FindAll(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* find all within the given buffer
 * findBuffer must exist, or an error will be returned
 * if there is no error, then if there as a match, the first match position is returned, if no
 * matches, -1 is returned
 */
{
	BOOLEAN
		flagList[5];
	EDITORBUFFER
		*theBuffer;
	EDITORBUFFER
		*findBuffer;
	BOOLEAN
		foundMatch;
	UINT32
		firstMatchPosition;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];
	BOOLEAN
		fail;

	fail=FALSE;
	if(argc==8)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				if(findBuffer=GetEditorBuffer(localInterpreter,argv[2]))
					{
					if(BufferNotBusy(localInterpreter,findBuffer))
						{
						SetBufferBusy(findBuffer);
						if(GetFlagList(localInterpreter,5,&(argv[3]),&(flagList[0])))
							{
							if(EditorFindAll(theBuffer->editorUniverse,findBuffer->editorUniverse->textUniverse,flagList[0],flagList[1],flagList[2],flagList[3],flagList[4],&foundMatch,&firstMatchPosition))
								{
								sprintf(tempString,"%ld",foundMatch?firstMatchPosition:-1);
								Tcl_AppendResult(localInterpreter,tempString,NULL);
								}
							else
								{
								GetError(&errorFamily,&errorFamilyMember,&errorDescription);
								Tcl_AppendResult(localInterpreter,errorDescription,NULL);
								fail=TRUE;
								}
							}
						else
							{
							fail=TRUE;
							}
						ClearBufferBusy(findBuffer);
						}
					else
						{
						fail=TRUE;
						}
					}
				else
					{
					fail=TRUE;
					}
				ClearBufferBusy(theBuffer);
				}
			else
				{
				fail=TRUE;
				}
			}
		else
			{
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," searchInBuffer findBuffer backwards wrapAround selectionExpr ignoreCase limitScope\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_Replace(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* replace within the given buffer
 * findBuffer and replaceBuffer must exist, or an error will be returned
 * if there is no error, then if there as a match, the first match position is returned, if no
 * matches, -1 is returned
 */
{
	BOOLEAN
		flagList[5];
	EDITORBUFFER
		*theBuffer;
	EDITORBUFFER
		*findBuffer,
		*replaceBuffer;
	BOOLEAN
		foundMatch;
	UINT32
		matchPosition;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];
	BOOLEAN
		fail;

	fail=FALSE;
	if(argc==9)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				if(findBuffer=GetEditorBuffer(localInterpreter,argv[2]))
					{
					if(BufferNotBusy(localInterpreter,findBuffer))
						{
						SetBufferBusy(findBuffer);
						if(replaceBuffer=GetEditorBuffer(localInterpreter,argv[3]))
							{
							if(BufferNotBusy(localInterpreter,replaceBuffer))
								{
								SetBufferBusy(replaceBuffer);
								if(GetFlagList(localInterpreter,5,&(argv[4]),&(flagList[0])))
									{
									if(EditorReplace(theBuffer->editorUniverse,findBuffer->editorUniverse->textUniverse,replaceBuffer->editorUniverse->textUniverse,flagList[0],flagList[1],flagList[2],flagList[3],flagList[4],&foundMatch,&matchPosition))
										{
										sprintf(tempString,"%ld",foundMatch?matchPosition:-1);
										Tcl_AppendResult(localInterpreter,tempString,NULL);
										}
									else
										{
										GetError(&errorFamily,&errorFamilyMember,&errorDescription);
										Tcl_AppendResult(localInterpreter,errorDescription,NULL);
										fail=TRUE;
										}
									}
								else
									{
									fail=TRUE;
									}
								ClearBufferBusy(replaceBuffer);
								}
							else
								{
								fail=TRUE;
								}
							}
						else
							{
							fail=TRUE;
							}
						ClearBufferBusy(findBuffer);
						}
					else
						{
						fail=TRUE;
						}
					}
				else
					{
					fail=TRUE;
					}
				ClearBufferBusy(theBuffer);
				}
			else
				{
				fail=TRUE;
				}
			}
		else
			{
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," searchInBuffer findBuffer replaceBuffer backwards wrapAround selectionExpr ignoreCase replaceProc\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_ReplaceAll(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* replace all within the given buffer
 * findBuffer and replaceBuffer must exist, or an error will be returned
 * if there is no error, then if there as a match, the first match position is returned, if no
 * matches, -1 is returned
 */
{
	BOOLEAN
		flagList[6];
	EDITORBUFFER
		*theBuffer;
	EDITORBUFFER
		*findBuffer,
		*replaceBuffer;
	BOOLEAN
		foundMatch;
	UINT32
		firstMatchPosition;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];
	BOOLEAN
		fail;

	fail=FALSE;
	if(argc==10)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				if(findBuffer=GetEditorBuffer(localInterpreter,argv[2]))
					{
					if(BufferNotBusy(localInterpreter,findBuffer))
						{
						SetBufferBusy(findBuffer);
						if(replaceBuffer=GetEditorBuffer(localInterpreter,argv[3]))
							{
							if(BufferNotBusy(localInterpreter,replaceBuffer))
								{
								SetBufferBusy(replaceBuffer);
								if(GetFlagList(localInterpreter,6,&(argv[4]),&(flagList[0])))
									{
									if(EditorReplaceAll(theBuffer->editorUniverse,findBuffer->editorUniverse->textUniverse,replaceBuffer->editorUniverse->textUniverse,flagList[0],flagList[1],flagList[2],flagList[3],flagList[4],flagList[5],&foundMatch,&firstMatchPosition))
										{
										sprintf(tempString,"%ld",foundMatch?firstMatchPosition:-1);
										Tcl_AppendResult(localInterpreter,tempString,NULL);
										}
									else
										{
										GetError(&errorFamily,&errorFamilyMember,&errorDescription);
										Tcl_AppendResult(localInterpreter,errorDescription,NULL);
										fail=TRUE;
										}
									}
								else
									{
									fail=TRUE;
									}
								ClearBufferBusy(replaceBuffer);
								}
							else
								{
								fail=TRUE;
								}
							}
						else
							{
							fail=TRUE;
							}
						ClearBufferBusy(findBuffer);
						}
					else
						{
						fail=TRUE;
						}
					}
				else
					{
					fail=TRUE;
					}
				ClearBufferBusy(theBuffer);
				}
			else
				{
				fail=TRUE;
				}
			}
		else
			{
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," searchInBuffer findBuffer replaceBuffer backwards wrapAround selectionExpr ignoreCase limitScope replaceProc\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_SelectAll(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* perform a select all in the given buffer
 */
{
	EDITORBUFFER
		*theBuffer;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				EditorSelectAll(theBuffer->editorUniverse);
				ClearBufferBusy(theBuffer);
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_GetSelectedText(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* return the currently selected text
 */
{
	EDITORBUFFER
		*theBuffer;
	ARRAYCHUNKHEADER
		*currentChunk;
	UINT32
		currentPosition,
		currentOffset;
	UINT32
		actualLength;
	UINT8
		*selectionBuffer;
	BOOLEAN
		done;
	BOOLEAN
		fail;

	fail=FALSE;
	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				currentChunk=NULL;
				currentPosition=currentOffset=0;
				done=FALSE;
				while(!done&&!fail)
					{
					if(selectionBuffer=EditorNextSelectionToBuffer(theBuffer->editorUniverse,&currentPosition,&currentChunk,&currentOffset,1,&actualLength,&done))
						{
						selectionBuffer[actualLength-1]='\0';					/* terminate the string */
						Tcl_AppendElement(localInterpreter,(char *)selectionBuffer);
						MDisposePtr(selectionBuffer);
						}
					else
						{
						if(!done)
							{
							GetError(&errorFamily,&errorFamilyMember,&errorDescription);
							Tcl_ResetResult(localInterpreter);
							Tcl_AppendResult(localInterpreter,errorDescription,NULL);
							fail=TRUE;
							}
						}
					}
				}
			else
				{
				fail=TRUE;
				}
			}
		else
			{
			fail=TRUE;
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_SelectLine(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* locate the given line, and select it
 * if lineNumber is out of range, go to the nearest one
 */
{
	EDITORBUFFER
		*theBuffer;
	UINT32
		theLine;

	if(argc==3)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				if(GetUINT32(localInterpreter,argv[2],&theLine))
					{
					if(theLine==0)		/* if user enters 0, just take it as 1 */
						{
						theLine=1;
						}
					SetBufferBusy(theBuffer);
					EditorLocateLine(theBuffer->editorUniverse,theLine-1);
					ClearBufferBusy(theBuffer);
					return(TCL_OK);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName lineNumber\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_GetSelectionEnds(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* return the start and end of the current selection
 * of the passed buffer
 * if there is no selection, the cursor position is returned in both values
 */
{
	EDITORBUFFER
		*theBuffer;
	UINT32
		startPosition,
		endPosition;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				GetSelectionEndPositions(theBuffer->editorUniverse->selectionUniverse,&startPosition,&endPosition);
				sprintf(tempString,"%lu %lu",startPosition,endPosition);
				Tcl_AppendResult(localInterpreter,tempString,NULL);
				ClearBufferBusy(theBuffer);
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_SetSelectionEnds(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* set the selection of the passed buffer
 * if the given end position is < the start position,
 * the positions will be reversed
 * if any position is out of range, it will be forced in range
 */
{
	EDITORBUFFER
		*theBuffer;
	UINT32
		startPosition,
		endPosition;

	if(argc==4)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				if(GetUINT32(localInterpreter,argv[2],&startPosition))
					{
					if(GetUINT32(localInterpreter,argv[3],&endPosition))
						{
						SetBufferBusy(theBuffer);
						startPosition=ForcePositionIntoRange(theBuffer->editorUniverse,startPosition);
						endPosition=ForcePositionIntoRange(theBuffer->editorUniverse,endPosition);
						if(startPosition<=endPosition)
							{
							EditorSetNormalSelection(theBuffer->editorUniverse,startPosition,endPosition);
							}
						else
							{
							EditorSetNormalSelection(theBuffer->editorUniverse,endPosition,startPosition);	/* do it upside-down if asked */
							}
						ClearBufferBusy(theBuffer);
						return(TCL_OK);
						}
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName startPosition endPosition\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_SetMark(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* set a mark in the given buffer
 */
{
	EDITORBUFFER
		*theBuffer;
	MARKLIST
		*theMark;

	if(argc==3)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				theMark=SetEditorMark(theBuffer->editorUniverse,argv[2]);
				ClearBufferBusy(theBuffer);
				if(theMark)
					{
					return(TCL_OK);
					}
				else
					{
					GetError(&errorFamily,&errorFamilyMember,&errorDescription);
					Tcl_AppendResult(localInterpreter,errorDescription,NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName markName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_ClearMark(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* clear a mark in the given buffer
 */
{
	EDITORBUFFER
		*theBuffer;
	MARKLIST
		*theMark;

	if(argc==3)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				if(theMark=LocateEditorMark(theBuffer->editorUniverse,argv[2]))
					{
					ClearEditorMark(theBuffer->editorUniverse,theMark);
					}
				ClearBufferBusy(theBuffer);
				if(theMark)
					{
					return(TCL_OK);
					}
				else
					{
					Tcl_AppendResult(localInterpreter,"Failed to locate mark '",argv[2],"'",NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName markName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_GotoMark(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* goto a mark in the given buffer
 */
{
	EDITORBUFFER
		*theBuffer;
	MARKLIST
		*theMark;

	if(argc==3)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				if(theMark=LocateEditorMark(theBuffer->editorUniverse,argv[2]))
					{
					GotoEditorMark(theBuffer->editorUniverse,theMark);
					}
				ClearBufferBusy(theBuffer);
				if(theMark)
					{
					return(TCL_OK);
					}
				else
					{
					Tcl_AppendResult(localInterpreter,"Failed to locate mark '",argv[2],"'",NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName markName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_MarkList(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* return a list of marks for the given buffer
 */
{
	EDITORBUFFER
		*theBuffer;
	MARKLIST
		*theMark;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			theMark=theBuffer->editorUniverse->theMarks;
			while(theMark)
				{
				if(theMark->markName)
					{
					Tcl_AppendElement(localInterpreter,theMark->markName);
					}
				theMark=theMark->nextMark;
				}
			return(TCL_OK);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_HomeWindow(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* home a document view to a given position, in the given way
 * if no position is specified, it is assumed to mean the current cursor position
 * if no way is specified, it is assumed to mean SEMISTRICT
 */
{
	EDITORWINDOW
		*theWindow;
	EDITORVIEW
		*theView;
	BOOLEAN
		havePosition,
		haveWay;
	UINT32
		thePosition,
		endPosition;
	int
		theWay;
	int
		argIndex;
	BOOLEAN
		fail,
		parseFail;

	havePosition=haveWay=FALSE;
	fail=parseFail=FALSE;
	if(argc>=2)
		{
		for(argIndex=1;argIndex<argc&&!parseFail&&!fail;argIndex++)
			{
			if(argIndex==1)					/* first argument is always the window name */
				{
				if(theWindow=GetEditorWindow(localInterpreter,argv[argIndex]))
					{
					if(BufferNotBusy(localInterpreter,theWindow->theBuffer))
						{
						}
					else
						{
						fail=TRUE;
						}
					}
				else
					{
					fail=TRUE;
					}
				}
			else
				{
				if(!haveWay&&MatchToken(argv[argIndex],homeWayTokens,&theWay))
					{
					haveWay=TRUE;
					}
				else
					{
					if(!havePosition)
						{
						if(GetUINT32(localInterpreter,argv[argIndex],&thePosition))
							{
							thePosition=ForcePositionIntoRange(theWindow->theBuffer->editorUniverse,thePosition);
							havePosition=TRUE;
							}
						else
							{
							fail=TRUE;
							}
						}
					else
						{
						parseFail=TRUE;
						}
					}
				}
			}
		if(!parseFail&&!fail)
			{
			if(!haveWay)
				{
				theWay=SEMISTRICT;
				}
			if(!havePosition)
				{
				GetSelectionEndPositions(theWindow->theBuffer->editorUniverse->selectionUniverse,&thePosition,&endPosition);
				}
			theView=GetDocumentWindowCurrentView(theWindow);
			SetBufferBusy(theWindow->theBuffer);
			switch(theWay)
				{
				case STRICT:
					EditorHomeViewToPositionStrict(theView,thePosition);
					break;
				case SEMISTRICT:
					EditorHomeViewToPositionSemiStrict(theView,thePosition);
					break;
				case LENIENT:
					EditorHomeViewToPositionLenient(theView,thePosition);
					break;
				default:
					break;
				}
			ResetEditorViewCursorBlink(theView);				/* reset cursor blinking when homing view */
			ClearBufferBusy(theWindow->theBuffer);
			return(TCL_OK);
			}
		}
	else
		{
		parseFail=TRUE;
		}
	if(parseFail)
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName ?position? ?STRICT|SEMISTRICT|LENIENT?\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_GetTopLeft(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* get the top line, and lefthand coordinate of the active view in the given window
 * NOTE: to be consistent with line numbering starting at 1, we fudge the number in here
 */
{
	EDITORWINDOW
		*theWindow;
	EDITORVIEW
		*theView;
	UINT32
		topLine,
		numLines,
		numPixels;
	INT32
		leftPixel;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];

	if(argc==2)
		{
		if(theWindow=GetEditorWindow(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theWindow->theBuffer))
				{
				SetBufferBusy(theWindow->theBuffer);
				theView=GetDocumentWindowCurrentView(theWindow);
				GetEditorViewTextInfo(theView,&topLine,&numLines,&leftPixel,&numPixels);
				sprintf(tempString,"%lu %ld",topLine+1,leftPixel);
				Tcl_AppendResult(localInterpreter,tempString,NULL);
				ClearBufferBusy(theWindow->theBuffer);
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_SetTopLeft(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* get the top line, and lefthand coordinate of the active view in the given window
 * NOTE: to be consistent with line numbering starting at 1, we fudge the number in here
 */
{
	EDITORWINDOW
		*theWindow;
	EDITORVIEW
		*theView;
	UINT32
		topLine;
	INT32
		leftPixel;

	if(argc==4)
		{
		if(theWindow=GetEditorWindow(localInterpreter,argv[1]))
			{
			if(GetUINT32(localInterpreter,argv[2],&topLine))
				{
				if(GetINT32(localInterpreter,argv[3],&leftPixel))
					{
					if(BufferNotBusy(localInterpreter,theWindow->theBuffer))
						{
						SetBufferBusy(theWindow->theBuffer);
						theView=GetDocumentWindowCurrentView(theWindow);
						if(topLine==0)				/* if line 0 was given, just make it line 1 */
							{
							topLine=1;
							}
						SetViewTopLeft(theView,topLine-1,leftPixel);
						ClearBufferBusy(theWindow->theBuffer);
						return(TCL_OK);
						}
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName topLine leftPixel\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_TextInfo(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* return the number of lines, and number of characters of text in the given buffer
 */
{
	EDITORBUFFER
		*theBuffer;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				sprintf(tempString,"%lu %lu",theBuffer->editorUniverse->textUniverse->totalLines,theBuffer->editorUniverse->textUniverse->totalBytes);
				Tcl_AppendResult(localInterpreter,tempString,NULL);
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_SelectionInfo(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* return information about the selection:
 * startPosition,endPosition,startLine,endLine,startLinePosition,endLinePosition,totalSegments,totalSpan
 * NOTE: to be consistent with line numbering starting at 1, we fudge the numbers in here
 */
{
	UINT32
		startPosition,						/* selection information */
		endPosition,
		startLine,
		endLine,
		startLinePosition,
		endLinePosition,
		totalSegments,
		totalSpan;
	EDITORBUFFER
		*theBuffer;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				EditorGetSelectionInfo(theBuffer->editorUniverse,&startPosition,&endPosition,&startLine,&endLine,&startLinePosition,&endLinePosition,&totalSegments,&totalSpan);
				sprintf(tempString,"%lu %lu %lu %lu %lu %lu %lu %lu",startPosition,endPosition,startLine+1,endLine+1,startLinePosition,endLinePosition,totalSegments,totalSpan);
				Tcl_AppendResult(localInterpreter,tempString,NULL);
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_PositionToLineAndOffset(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* given a position in a buffer, return a line number, and offset in characters into the line
 * that matches the position
 * NOTE: to be consistent with line numbering starting at 1, we fudge the number in here
 */
{
	EDITORBUFFER
		*theBuffer;
	UINT32
		thePosition,
		theLine,
		theLineOffset;
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];

	if(argc==3)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(GetUINT32(localInterpreter,argv[2],&thePosition))
				{
				if(BufferNotBusy(localInterpreter,theBuffer))
					{
					SetBufferBusy(theBuffer);
					PositionToLinePosition(theBuffer->editorUniverse->textUniverse,thePosition,&theLine,&theLineOffset,&theChunk,&theOffset);
					sprintf(tempString,"%lu %lu",theLine+1,theLineOffset);
					Tcl_AppendResult(localInterpreter,tempString,NULL);
					ClearBufferBusy(theBuffer);
					return(TCL_OK);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName position\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_LineAndOffsetToPosition(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* given a line, and offset in a buffer, return a position in the buffer that matches
 * NOTE: to be consistent with line numbering starting at 1, we fudge the number in here
 */
{
	EDITORBUFFER
		*theBuffer;
	UINT32
		thePosition,
		theLine,
		theLineOffset,
		distanceMoved;
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];

	if(argc==4)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(GetUINT32(localInterpreter,argv[2],&theLine))
				{
				if(GetUINT32(localInterpreter,argv[3],&theLineOffset))
					{
					if(BufferNotBusy(localInterpreter,theBuffer))
						{
						SetBufferBusy(theBuffer);
						if(theLine==0)
							{
							theLine=1;
							}
						LineToChunkPosition(theBuffer->editorUniverse->textUniverse,theLine-1,&theChunk,&theOffset,&thePosition);	/* get position that is the start of the given line */
						ChunkPositionToNextLine(theBuffer->editorUniverse->textUniverse,theChunk,theOffset,&theChunk,&theOffset,&distanceMoved);
						if(theLineOffset>distanceMoved)		/* see if offset is past end of line, if so, pin it */
							{
							theLineOffset=distanceMoved;
							}
						sprintf(tempString,"%lu",thePosition+theLineOffset);
						Tcl_AppendResult(localInterpreter,tempString,NULL);
						ClearBufferBusy(theBuffer);
						return(TCL_OK);
						}
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName lineNumber offset\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_Clear(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* perform a selection clear in the given buffer
 */
{
	EDITORBUFFER
		*theBuffer;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				EditorDeleteSelection(theBuffer->editorUniverse);
				ClearBufferBusy(theBuffer);
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_Insert(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* insert the given text into the current selection
 * of the passed buffer
 */
{
	EDITORBUFFER
		*theBuffer;

	if(argc==3)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				EditorInsert(theBuffer->editorUniverse,(UINT8 *)argv[2],strlen(argv[2]));
				ClearBufferBusy(theBuffer);
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName textToInsert\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_InsertFile(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* insert the given file into the current selection
 * of the passed buffer
 */
{
	EDITORBUFFER
		*theBuffer;
	BOOLEAN
		result;

	if(argc==3)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				result=EditorInsertFile(theBuffer->editorUniverse,argv[2]);
				ClearBufferBusy(theBuffer);
				if(result)
					{
					return(TCL_OK);
					}
				else
					{
					GetError(&errorFamily,&errorFamilyMember,&errorDescription);
					Tcl_AppendResult(localInterpreter,errorDescription,NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName pathName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_AutoIndent(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* perform an autoindent on the passed buffer
 */
{
	EDITORBUFFER
		*theBuffer;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				EditorAutoIndent(theBuffer->editorUniverse);
				ClearBufferBusy(theBuffer);
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_MoveCursor(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* move the cursor in the given window by the relative mode
 */
{
	EDITORWINDOW
		*theWindow;
	EDITORVIEW
		*theView;
	int
		relativeMode;

	if(argc==3)
		{
		if(theWindow=GetEditorWindow(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theWindow->theBuffer))
				{
				if(MatchToken(argv[2],cursorMovementTokens,&relativeMode))
					{
					SetBufferBusy(theWindow->theBuffer);
					theView=GetDocumentWindowCurrentView(theWindow);
					EditorMoveCursor(theView,(UINT16)relativeMode);
					ResetEditorViewCursorBlink(theView);				/* reset cursor blinking when done moving */
					ClearBufferBusy(theWindow->theBuffer);
					return(TCL_OK);
					}
				else
					{
					Tcl_AppendResult(localInterpreter,"Invalid relative mode",NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName relativeMode\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_Delete(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* delete the characters in the window between the current cursor position, and
 * the given relative mode away, if there is a selection, it will be deleted instead
 */
{
	EDITORWINDOW
		*theWindow;
	EDITORVIEW
		*theView;
	int
		relativeMode;

	if(argc==3)
		{
		if(theWindow=GetEditorWindow(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theWindow->theBuffer))
				{
				if(MatchToken(argv[2],cursorMovementTokens,&relativeMode))
					{
					SetBufferBusy(theWindow->theBuffer);
					theView=GetDocumentWindowCurrentView(theWindow);
					EditorDelete(theView,(UINT16)relativeMode);
					ClearBufferBusy(theWindow->theBuffer);
					return(TCL_OK);
					}
				else
					{
					Tcl_AppendResult(localInterpreter,"Invalid relative mode",NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName relativeMode\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_ExpandSelection(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* expand the selection in the given window in the given relative direction
 */
{
	EDITORWINDOW
		*theWindow;
	EDITORVIEW
		*theView;
	int
		relativeMode;

	if(argc==3)
		{
		if(theWindow=GetEditorWindow(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theWindow->theBuffer))
				{
				if(MatchToken(argv[2],cursorMovementTokens,&relativeMode))
					{
					SetBufferBusy(theWindow->theBuffer);
					theView=GetDocumentWindowCurrentView(theWindow);
					EditorExpandNormalSelection(theView,(UINT16)relativeMode);
					ClearBufferBusy(theWindow->theBuffer);
					return(TCL_OK);
					}
				else
					{
					Tcl_AppendResult(localInterpreter,"Invalid relative mode",NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName relativeMode\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_ReduceSelection(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* reduce the selection in the given window in the given relative direction
 */
{
	EDITORWINDOW
		*theWindow;
	EDITORVIEW
		*theView;
	int
		relativeMode;

	if(argc==3)
		{
		if(theWindow=GetEditorWindow(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theWindow->theBuffer))
				{
				if(MatchToken(argv[2],cursorMovementTokens,&relativeMode))
					{
					SetBufferBusy(theWindow->theBuffer);
					theView=GetDocumentWindowCurrentView(theWindow);
					EditorReduceNormalSelection(theView,(UINT16)relativeMode);
					ClearBufferBusy(theWindow->theBuffer);
					return(TCL_OK);
					}
				else
					{
					Tcl_AppendResult(localInterpreter,"Invalid relative mode",NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," windowName relativeMode\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_SetWordChars(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* set the wordSpaceTable so that all of the characters in the passed argument
 * are not considered space
 */
{
	int
		i;

	if(argc==2)
		{
		for(i=0;i<256;i++)
			{
			wordSpaceTable[i]=TRUE;
			}
		for(i=0;argv[1][i];i++)
			{
			wordSpaceTable[argv[1][i]]=FALSE;
			}
		return(TCL_OK);
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," wordCharacters\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_GetWordChars(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* return the list of characters that are considered parts of words
 */
{
	char
		tempString[256];				/* hold the list of characters that are part of a word */
	int
		i,j;

	if(argc==1)
		{
		j=0;
		for(i=1;i<256;i++)				/* never look at character 0, because it would terminate the string, and confuse things */
			{
			if(!wordSpaceTable[i])
				{
				tempString[j++]=(char)i;	/* set this one into the string */
				}
			}
		tempString[j++]='\0';				/* terminate the string */
		Tcl_AppendElement(localInterpreter,&(tempString[0]));
		return(TCL_OK);
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0],"\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_Execute(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* execute the given tcl script, place the results in the given buffer
 * if it still exists after the script is finished, otherwise throw them away.
 */
{
	EDITORBUFFER
		*theBuffer;
	UINT32
		startPosition,
		endPosition;
	char
		tempString[TEMPSTRINGBUFFERLENGTH];
	int
		tclResult;

	if(argc==3)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				/* NOTE: do not set busy here, we will allow executed script to mess with the window, but we must be careful:
				 * If the buffer goes away as the result of executing the script, we have to make sure it is not referenced again
				 */
				EditorHoldBuffer(theBuffer);									/* dont let this get deleted */
				/* do not call ClearAbort here, it will have already been called when the script that got us here was started */
				tclResult=Tcl_Eval(localInterpreter,argv[2]);
				if(*(localInterpreter->result))									/* if no result, do not try to insert into buffer */
					{
					if(theBuffer->contentName)									/* see if this buffer is alive */
						{
						GetSelectionEndPositions(theBuffer->editorUniverse->selectionUniverse,&startPosition,&endPosition);
						theBuffer->editorUniverse->auxSelectionUniverse->cursorPosition=endPosition;	/* move the aux cursor to the end of any selection/cursor, and start output there */
						if(tclResult!=TCL_OK)
							{
							sprintf(tempString,"Error in line %d: ",localInterpreter->errorLine);
							EditorAuxInsert(theBuffer->editorUniverse,(UINT8 *)tempString,(UINT32)strlen(tempString));
							}
						Tcl_AppendResult(localInterpreter,"\n",NULL);
						EditorAuxInsert(theBuffer->editorUniverse,(UINT8 *)localInterpreter->result,(UINT32)strlen(localInterpreter->result));
						}
					}
				EditorReleaseBuffer(theBuffer);									/* do not need this any more */
				return(TCL_OK);
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName script\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_BufferTaskData(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* start a task running in the given buffer, or send data to a task in a buffer
 */
{
	EDITORBUFFER
		*theBuffer;
	UINT32
		startPosition,
		endPosition;

	if(argc==3)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				GetSelectionEndPositions(theBuffer->editorUniverse->selectionUniverse,&startPosition,&endPosition);
				theBuffer->editorUniverse->auxSelectionUniverse->cursorPosition=endPosition;	/* move the aux cursor to the end of any selection/cursor, and start output there */
				if(WriteBufferTaskData(theBuffer,(UINT8 *)argv[2],(UINT32)strlen(argv[2])))
					{
					return(TCL_OK);
					}
				else
					{
					GetError(&errorFamily,&errorFamilyMember,&errorDescription);
					Tcl_AppendResult(localInterpreter,errorDescription,NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName taskData\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_BufferHasTask(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* see if a task is running in the given buffer, if so, return 1, else 0
 */
{
	EDITORBUFFER
		*theBuffer;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			Tcl_AppendResult(localInterpreter,theBuffer->theTask?"1":"0",NULL);
			return(TCL_OK);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_UpdateBufferTask(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* if a task is running in the given buffer, this can be called to
 * update the status of the task
 * If the buffer is busy, this will fail
 * NOTE: while scripts are not running, the tasks on all buffers are
 * continously updated, but while scripts are running, this must
 * be called explicitly to get a buffer to update
 */
{
	EDITORBUFFER
		*theBuffer;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				if(UpdateBufferTask(theBuffer))
					{
					return(TCL_OK);
					}
				else
					{
					GetError(&errorFamily,&errorFamilyMember,&errorDescription);
					Tcl_AppendResult(localInterpreter,errorDescription,NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_EOFBufferTask(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* send an EOF to the task on the passed buffer, if there is a problem, complain
 */
{
	EDITORBUFFER
		*theBuffer;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				if(SendEOFToBufferTask(theBuffer))
					{
					return(TCL_OK);
					}
				else
					{
					GetError(&errorFamily,&errorFamilyMember,&errorDescription);
					Tcl_AppendResult(localInterpreter,errorDescription,NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_KillBufferTask(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* send a signal to the task on the passed buffer, if there is a problem, complain
 */
{
	EDITORBUFFER
		*theBuffer;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				if(KillBufferTask(theBuffer))
					{
					return(TCL_OK);
					}
				else
					{
					GetError(&errorFamily,&errorFamilyMember,&errorDescription);
					Tcl_AppendResult(localInterpreter,errorDescription,NULL);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_Beep(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* make a noise
 */
{
	if(argc==1)
		{
		EditorBeep();
		return(TCL_OK);
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0],"\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_CheckBuffer(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* check the text universe on the given buffer, report the results
 * NOTE: this is mainly for debugging, and draws its own dialog boxes
 */
{
	EDITORBUFFER
		*theBuffer;
	BOOLEAN
		result;

	if(argc==2)
		{
		if(theBuffer=GetEditorBuffer(localInterpreter,argv[1]))
			{
			if(BufferNotBusy(localInterpreter,theBuffer))
				{
				SetBufferBusy(theBuffer);
				result=UniverseSanityCheck(theBuffer->editorUniverse->textUniverse);
				ClearBufferBusy(theBuffer);
				if(result)
					{
					return(TCL_OK);
					}
				}
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0]," bufferName\"",NULL);
		}
	return(TCL_ERROR);
}

static int Tcl_ForceQuitFunction(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* execute editor quit function (the editor WILL quit, no questions asked if you call this)
 */
{
	BOOLEAN
		fail;

	fail=FALSE;
	if(argc==1)
		{
		EditorQuit();
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0],"\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

static int Tcl_VersionFunction(ClientData clientData,Tcl_Interp *localInterpreter,int argc,char *argv[])
/* get version numbers
 */
{
	char
		*localVersion;
	BOOLEAN
		fail;

	fail=FALSE;
	if(argc==1)
		{
		Tcl_AppendResult(localInterpreter,programName," ",VERSION,EDITION,NULL);
		if(localVersion=GetEditorLocalVersion())
			{
			Tcl_AppendResult(localInterpreter," (",localVersion,")",NULL);
			}
		}
	else
		{
		Tcl_AppendResult(localInterpreter,"wrong # args: should be \"",argv[0],"\"",NULL);
		fail=TRUE;
		}
	return(fail?TCL_ERROR:TCL_OK);
}

/* Create commands so TCL can find them */

BOOLEAN CreateEditorShellCommands(Tcl_Interp *theInterpreter)
/* create the shell commands that are implemented by the editor
 * if there is a problem, SetError, return FALSE
 */
{
/* menus and key bindings */
	Tcl_CreateCommand(theInterpreter,"addmenu",Tcl_AddMenu,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"deletemenu",Tcl_DeleteMenus,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"bindkey",Tcl_AddKeyBinding,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"unbindkey",Tcl_DeleteKeyBinding,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"keybindings",Tcl_ListKeyBindings,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"waitkey",Tcl_WaitKey,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"getkey",Tcl_GetKey,NULL,NULL);

/* buffers and windows */
	Tcl_CreateCommand(theInterpreter,"newbuffer",Tcl_NewBuffer,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"openbuffer",Tcl_OpenBuffer,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"closebuffer",Tcl_CloseBuffer,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"savebuffer",Tcl_SaveBuffer,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"savebufferas",Tcl_SaveBufferAs,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"savebufferto",Tcl_SaveBufferTo,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"revertbuffer",Tcl_RevertBuffer,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"isdirty",Tcl_IsBufferDirty,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"fromfile",Tcl_IsBufferFromFile,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"haswindow",Tcl_BufferHasWindow,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"cleardirty",Tcl_ClearBufferDirty,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"bufferlist",Tcl_BufferList,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"windowlist",Tcl_WindowList,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"openwindow",Tcl_OpenWindow,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"closewindow",Tcl_CloseWindow,NULL,NULL);

	Tcl_CreateCommand(theInterpreter,"setrect",Tcl_SetRect,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"getrect",Tcl_GetRect,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"setfont",Tcl_SetFont,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"getfont",Tcl_GetFont,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"settabsize",Tcl_SetTabSize,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"gettabsize",Tcl_GetTabSize,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"setcolors",Tcl_SetColors,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"getcolors",Tcl_GetColors,NULL,NULL);

	Tcl_CreateCommand(theInterpreter,"settopwindow",Tcl_SetTopWindow,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"activewindow",Tcl_GetActiveWindow,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"updatewindows",Tcl_UpdateWindows,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"screensize",Tcl_ScreenSize,NULL,NULL);

/* undo/clipboard */
	Tcl_CreateCommand(theInterpreter,"undotoggle",Tcl_UndoRedoToggle,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"undo",Tcl_Undo,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"redo",Tcl_Redo,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"breakundo",Tcl_BreakUndo,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"flushundos",Tcl_FlushUndos,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"cut",Tcl_Cut,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"copy",Tcl_Copy,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"paste",Tcl_Paste,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"columnarpaste",Tcl_ColumnarPaste,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"getclipboard",Tcl_GetClipboard,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"setclipboard",Tcl_SetClipboard,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"unsetclipboard",Tcl_UnSetClipboard,NULL,NULL);

/* dialogs */
	Tcl_CreateCommand(theInterpreter,"okdialog",Tcl_OkDialog,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"okcanceldialog",Tcl_OkCancelDialog,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"yesnodialog",Tcl_YesNoCancelDialog,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"textdialog",Tcl_TextDialog,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"searchdialog",Tcl_SearchDialog,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"listdialog",Tcl_ListDialog,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"opendialog",Tcl_OpenDialog,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"savedialog",Tcl_SaveDialog,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"pathdialog",Tcl_PathDialog,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"fontdialog",Tcl_FontDialog,NULL,NULL);

/* search/replace */
	Tcl_CreateCommand(theInterpreter,"find",Tcl_Find,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"findall",Tcl_FindAll,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"replace",Tcl_Replace,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"replaceall",Tcl_ReplaceAll,NULL,NULL);

/* selection */
	Tcl_CreateCommand(theInterpreter,"selectall",Tcl_SelectAll,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"selectedtextlist",Tcl_GetSelectedText,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"selectline",Tcl_SelectLine,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"getselectionends",Tcl_GetSelectionEnds,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"setselectionends",Tcl_SetSelectionEnds,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"setmark",Tcl_SetMark,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"clearmark",Tcl_ClearMark,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"gotomark",Tcl_GotoMark,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"marklist",Tcl_MarkList,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"homewindow",Tcl_HomeWindow,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"gettopleft",Tcl_GetTopLeft,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"settopleft",Tcl_SetTopLeft,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"textinfo",Tcl_TextInfo,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"selectioninfo",Tcl_SelectionInfo,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"positiontolineoffset",Tcl_PositionToLineAndOffset,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"lineoffsettoposition",Tcl_LineAndOffsetToPosition,NULL,NULL);

/* editing functions */
	Tcl_CreateCommand(theInterpreter,"clear",Tcl_Clear,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"insert",Tcl_Insert,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"insertfile",Tcl_InsertFile,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"autoindent",Tcl_AutoIndent,NULL,NULL);

	Tcl_CreateCommand(theInterpreter,"movecursor",Tcl_MoveCursor,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"delete",Tcl_Delete,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"expandselection",Tcl_ExpandSelection,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"reduceselection",Tcl_ReduceSelection,NULL,NULL);

/* misc functions */
	Tcl_CreateCommand(theInterpreter,"setwordchars",Tcl_SetWordChars,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"getwordchars",Tcl_GetWordChars,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"execute",Tcl_Execute,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"task",Tcl_BufferTaskData,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"updatetask",Tcl_UpdateBufferTask,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"eoftask",Tcl_EOFBufferTask,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"killtask",Tcl_KillBufferTask,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"hastask",Tcl_BufferHasTask,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"beep",Tcl_Beep,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"checkbuffer",Tcl_CheckBuffer,NULL,NULL);
	Tcl_CreateCommand(theInterpreter,"forceQUIT",Tcl_ForceQuitFunction,NULL,NULL);

	Tcl_CreateCommand(theInterpreter,"version",Tcl_VersionFunction,NULL,NULL);

/* Built-in Tcl commands we DONT want */
	Tcl_DeleteCommand(theInterpreter,"exit");

	return(TRUE);
}
