/* Key binding interface
   Copyright (C) 1995 Viacom New Media.

	This file is part of e93.

	e93 is free software; you can redistribute it and/or modify
	it under the terms of the e93 LICENSE AGREEMENT.

	e93 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	e93 LICENSE AGREEMENT for more details.

	You should have received a copy of the e93 LICENSE AGREEMENT
	along with e93; see the file "LICENSE.TXT".
*/

#include "includes.h"

static EDITORKEYBINDING **GetBindingHashListHead(UINT32 keyCode)
/* perform hashing function on keyCode, return pointer to list head
 * for bindings
 */
{
	return(&(keyBindingTable[keyCode&0xFF]));
}

static BOOLEAN LocateKeyBinding(UINT32 keyCode,UINT32 modifierMask,UINT32 modifierValue,EDITORKEYBINDING ***hashListHead,EDITORKEYBINDING **previousHashEntry,EDITORKEYBINDING **theEntry)
/* locate a key binding in the binding table, return pointers to important entries
 */
{
	BOOLEAN
		found;
	EDITORKEYBINDING
		*currentEntry;

	currentEntry=*(*hashListHead=GetBindingHashListHead(keyCode));	/* point to the head of the linked list at this hash entry */
	*previousHashEntry=NULL;
	found=FALSE;
	while(currentEntry&&!found)
		{
		if((currentEntry->keyCode==keyCode)&&(currentEntry->modifierMask==modifierMask)&&(currentEntry->modifierValue==modifierValue))
			{
			found=TRUE;
			*theEntry=currentEntry;
			}
		else
			{
			*previousHashEntry=currentEntry;
			currentEntry=currentEntry->nextHashBinding;
			}
		}
	return(found);
}

EDITORKEYBINDING *LocateKeyBindingMatch(UINT32 keyCode,UINT32 modifierValue)
/* find a key binding that matches the keyCode, and modifierValue, and return it
 * if there is none, return NULL
 */
{
	BOOLEAN
		found;
	EDITORKEYBINDING
		*currentEntry;

	currentEntry=*GetBindingHashListHead(keyCode);	/* point to the head of the linked list at this hash entry */
	found=FALSE;
	while(currentEntry&&!found)
		{
		if((currentEntry->keyCode==keyCode)&&((currentEntry->modifierMask&modifierValue)==currentEntry->modifierValue))
			{
			found=TRUE;
			}
		else
			{
			currentEntry=currentEntry->nextHashBinding;
			}
		}
	return(currentEntry);
}

BOOLEAN DeleteEditorKeyBinding(UINT32 keyCode,UINT32 modifierMask,UINT32 modifierValue)
/* attempt to locate the given binding in the key bindings table, and
 * remove it if it is found
 * if it is not located, return FALSE
 */
{
	EDITORKEYBINDING
		**hashListHead,
		*previousHashEntry,
		*theEntry;

	if(LocateKeyBinding(keyCode,modifierMask,modifierValue,&hashListHead,&previousHashEntry,&theEntry))
		{
		if(previousHashEntry)
			{
			previousHashEntry->nextHashBinding=theEntry->nextHashBinding;		/* unlink the located entry */
			}
		else
			{
			*hashListHead=theEntry->nextHashBinding;
			}
		if(theEntry->previousBinding)
			{
			if(theEntry->previousBinding->nextBinding=theEntry->nextBinding)
				{
				theEntry->nextBinding->previousBinding=theEntry->previousBinding;
				}
			}
		else
			{
			if(keyBindingListHead=theEntry->nextBinding)
				{
				keyBindingListHead->previousBinding=NULL;
				}
			}
		MDisposePtr(theEntry);
		return(TRUE);
		}
	return(FALSE);
}

BOOLEAN CreateEditorKeyBinding(UINT32 keyCode,UINT32 modifierMask,UINT32 modifierValue,char *dataText)
/* search the key binding table for an entry that matches the one given, and delete it if
 * it exists. Then add this entry to the table
 * if there is a problem, SetError, and return FALSE
 */
{
	EDITORKEYBINDING
		**hashListHead,
		*newEntry;

	DeleteEditorKeyBinding(keyCode,modifierMask,modifierValue);	/* try to delete one if it exists */
	hashListHead=GetBindingHashListHead(keyCode);						/* point to head of list in hash table where this binding will go */
	if(newEntry=(EDITORKEYBINDING *)MNewPtr(sizeof(EDITORKEYBINDING)+strlen(dataText)+1))
		{
		newEntry->keyCode=keyCode;
		newEntry->modifierMask=modifierMask;
		newEntry->modifierValue=modifierValue;
		newEntry->nextHashBinding=*hashListHead;
		strcpy(&(newEntry->dataText[0]),dataText);				/* copy in the data */
		*hashListHead=newEntry;									/* link to head of hash list */
		newEntry->previousBinding=NULL;
		if(newEntry->nextBinding=keyBindingListHead)			/* point to the next entry */
			{
			keyBindingListHead->previousBinding=newEntry;
			}
		keyBindingListHead=newEntry;
		return(TRUE);
		}
	return(FALSE);
}

BOOLEAN InitKeyBindingTable()
/* clear the key binding table so that there are no bindings present
 * if there is a problem, SetError, and return FALSE
 */
{
	int
		i;

	for(i=0;i<256;i++)
		{
		keyBindingTable[i]=NULL;
		}
	keyBindingListHead=NULL;
	return(TRUE);
}

void UnInitKeyBindingTable()
/* remove all key bindings from the bindings table
 */
{
	int
		i;
	EDITORKEYBINDING
		*currentEntry,
		*nextEntry;

	for(i=0;i<256;i++)
		{
		currentEntry=keyBindingTable[i];
		while(currentEntry)
			{
			nextEntry=currentEntry->nextHashBinding;
			MDisposePtr(currentEntry);
			currentEntry=nextEntry;
			}
		keyBindingTable[i]=NULL;
		}
	keyBindingListHead=NULL;
}
