/*	This file is part of e93.

	e93 is free software; you can redistribute it and/or modify
	it under the terms of the e93 LICENSE AGREEMENT.

	e93 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	e93 LICENSE AGREEMENT for more details.

	You should have received a copy of the e93 LICENSE AGREEMENT
	along with e93; see the file "LICENSE.TXT".
*/

/* This include file is used to communicate with the GUI
 * it contains all prototypes for editor functions that the GUI is
 * allowed to call, and all GUI functions that the editor is allowed
 * to call
 * It also defines all data structures and types that they have in common
 */

/* when the editor needs to read and write files, it communicates with the gui through this */
struct
	editorFile;					/* declare it to put it into scope */

#define	EDITORFILE	struct editorFile	/* this type is hidden from the editor (the editor only needs pointers to it) */

struct
	editorMenu;

#define	EDITORMENU	struct editorMenu	/* menu definition (actual structure hidden from the editor) */

struct
	editorTask;

#define	EDITORTASK	struct editorTask	/* task definition (actual structure hidden) */


/* modifier bits (can come from keyboard, or elsewhere (like mouse)) */

/* masks */
enum
	{
	EEM_CAPSLOCK=0x00000001,
	EEM_SHIFT=0x00000002,
	EEM_CTL=0x00000004,
	EEM_MOD0=0x00000008,
	EEM_MOD1=0x00000010,
	EEM_MOD2=0x00000020,
	EEM_MOD3=0x00000040,
	EEM_MOD4=0x00000080,
	EEM_MOD5=0x00000100,
	EEM_MOD6=0x00000200,
	EEM_MOD7=0x00000400,
	EEM_STATE0=0x00F0000,		/* traditionally used to keep track of double, triple, quadruple, etc... clicks of mouse */
	EEM_STATE1=0x0F00000,
	EEM_STATE2=0xF000000
	};

/* shifts */
enum
	{
	EES_CAPSLOCK=0,
	EES_SHIFT=1,
	EES_CTL=2,
	EES_MOD0=3,
	EES_MOD1=4,
	EES_MOD2=5,
	EES_MOD3=6,
	EES_MOD4=7,
	EES_MOD5=8,
	EES_MOD6=9,
	EES_MOD7=10,
	EES_STATE0=16,
	EES_STATE1=20,
	EES_STATE2=24
	};

typedef struct keyBinding
	{
	UINT32
		keyCode,
		modifierMask,
		modifierValue;
	struct keyBinding
		*previousBinding,					/* pointer to previous binding in the global binding list */
		*nextBinding,						/* pointer to next binding in the global binding list (in user defined order) */
		*nextHashBinding;					/* pointer to next binding in local hash list */
	char
		dataText[1];						/* variable length array of characters of data (would like to declare as dataText[0] but some compilers complain) */
	} EDITORKEYBINDING;

/* view event types */

enum
	{
	VET_KEYDOWN,			/* saw key/virtual key press within view */
	VET_KEYREPEAT,			/* saw repeat within view */
	VET_POSITIONVERTICAL,	/* attempt to position the view to given absolute line */
	VET_POSITIONHORIZONTAL,	/* attempt to position the view to given absolute pixel */
	VET_CLICK,				/* initial click in view */
	VET_CLICKHOLD			/* send while click is being held in view */
	};

/* virtual view keys */

enum
	{
	VVK_SCROLLUP,
	VVK_SCROLLDOWN,
	VVK_DOCUMENTPAGEUP,
	VVK_DOCUMENTPAGEDOWN,
	VVK_SCROLLLEFT,
	VVK_SCROLLRIGHT,
	VVK_DOCUMENTPAGELEFT,
	VVK_DOCUMENTPAGERIGHT,
/* keys that are normally found on a keyboard, but not clearly associated with an ASCII code */
/* these are allowed to have modifier data sent with them */
	VVK_LEFTARROW,
	VVK_RIGHTARROW,
	VVK_UPARROW,
	VVK_DOWNARROW,
	VVK_HOME,
	VVK_END,
	VVK_PAGEUP,
	VVK_PAGEDOWN,
	VVK_INSERT,
	VVK_BACKSPACE,
	VVK_DELETE,
	VVK_HELP,
	VVK_UNDO,
	VVK_REDO,
	VVK_UNDOTOGGLE,
	VVK_CUT,
	VVK_COPY,
	VVK_PASTE,
	VVK_FIND,
	VVK_RETURN,
	VVK_TAB,
	VVK_ESC,
	VVK_F1,
	VVK_F2,
	VVK_F3,
	VVK_F4,
	VVK_F5,
	VVK_F6,
	VVK_F7,
	VVK_F8,
	VVK_F9,
	VVK_F10,
	VVK_F11,
	VVK_F12,
	VVK_F13,
	VVK_F14,
	VVK_F15,
	VVK_F16,
	VVK_F17,
	VVK_F18,
	VVK_F19,
	VVK_F20
	};

/* menu relationships */
enum
	{
	MR_NEXTSIBLING,			/* next sibling */
	MR_PREVIOUSSIBLING,		/* previous sibling */
	MR_FIRSTCHILD,			/* first child */
	MR_LASTCHILD			/* last child */
	};

/* search types */
enum
	{
	ST_FIND,
	ST_FINDALL,
	ST_REPLACE,
	ST_REPLACEALL
	};

typedef struct
	{
	INT32
		x,y;
	UINT32
		w,h;
	} EDITORRECT;

typedef UINT32	EDITORCOLOR;		/* intensity values for each of the colors in RGB space (3 bytes R,G,B) */

typedef struct chunkHeader
	{
	struct chunkHeader
		*nextHeader,
		*previousHeader;
	UINT8
		*data;						/* pointer to the data buffer for this chunk */
	UINT32
		totalBytes;					/* number of bytes of this chunk that are in use */
	UINT32
		totalLines;					/* number of new line characters in this chunk */
	} CHUNKHEADER;

typedef struct
	{
	CHUNKHEADER
		*firstChunkHeader,			/* pointer to the first chunk (NULL if there is none) */
		*lastChunkHeader,			/* pointer to the last chunk (NULL if there is none) */
		*cacheChunkHeader;			/* pointer to the cached chunk (NULL if there is none) */
	UINT32
		totalBytes;					/* total number of bytes of text */
	UINT32
		totalLines;					/* total number of new-line characters in text */
	UINT32
		cacheBytes;					/* total number of bytes up until the cacheChunkHeader */
	UINT32
		cacheLines;					/* total number of lines up until the cacheChunkHeader */
	} TEXTUNIVERSE;

typedef struct arrayChunkHeader
	{
	struct arrayChunkHeader
		*nextHeader,
		*previousHeader;
	UINT8
		*data;						/* pointer to the element data for this chunk */
	UINT32
		totalElements;				/* number of elements of this chunk that are in use */
	} ARRAYCHUNKHEADER;

typedef struct
	{
	ARRAYCHUNKHEADER
		*firstChunkHeader,			/* points to the first header of the chunk array (or NULL if there are no chunks) */
		*lastChunkHeader;
	UINT32
		totalElements;				/* total number of elements in this universe */
	UINT32
		maxElements,				/* maximum number of elements in a chunk of this universe */
		elementSize;				/* size of an element of a chunk in this universe */
	} ARRAYUNIVERSE;

typedef struct undoElement
	{
	UINT32
		frameNumber,				/* tells which frame of undo this belongs to */
		textStartPosition,			/* position in text where replacement begins */
		textLength,					/* number of bytes to replace */
		undoStartPosition,			/* position in undo buffer of replacement text */
		undoLength;					/* length of replacement text */
	} UNDOELEMENT;

typedef struct
	{
	UINT32
		currentFrame;				/* keeps track of the current undo frame index */
	UINT32
		groupingDepth;				/* knows the depth of begin/end undo group calls */
	BOOLEAN
		undoActive;					/* set true while undoing/redoing (tells not to delete redo information) */
	BOOLEAN
		undoing;					/* set true while undoing (tells to place undo info into redo buffer) */
	ARRAYUNIVERSE
		undoChunks;
	ARRAYUNIVERSE
		redoChunks;
	TEXTUNIVERSE
		*undoText;
	TEXTUNIVERSE
		*redoText;
	} UNDOUNIVERSE;

typedef struct selectionElement
	{
	UINT32
		startOffset,			/* offset from previous selection's end to start of this selection */
		endOffset;				/* offset from the start of this selection to the end */
	} SELECTIONELEMENT;

typedef struct
	{
	UINT32
		cursorPosition;			/* if there is no selection, this is the cursor's position */
	ARRAYUNIVERSE
		selectionChunks;
	} SELECTIONUNIVERSE;

typedef struct markList			/* list of marks hanging off of editor universe */
	{
	char
		*markName;				/* if NULL, then this mark is considered temporary */
	SELECTIONUNIVERSE
		*selectionUniverse;		/* the selection universe that defines this mark */
	struct markList
		*nextMark;				/* next mark in the list, NULL if no more */
	} MARKLIST;

typedef struct
	{
	TEXTUNIVERSE
		*textUniverse;			/* all the text */
	SELECTIONUNIVERSE
		*selectionUniverse;		/* selection and cursor information */
	SELECTIONUNIVERSE
		*auxSelectionUniverse;	/* aux selection and cursor information (allows tasks to write to universe at place other than cursor) */
	MARKLIST
		*theMarks;				/* pointer to head of list of marks for this universe, NULL if none */
	UNDOUNIVERSE
		*undoUniverse;			/* undo/redo information */
	BOOLEAN
		dirty;					/* tells if the universe has been edited */
	UINT32
		anchorStartPosition,	/* hold the ends of the selection while tracking */
		anchorEndPosition;
	UINT32
		anchorLine;				/* holds one corner of columnar selection while tracking */
	INT32
		anchorX;
	UINT32
		columnarTopLine;		/* when building a columnar selection, these remember the "rect" that bounds the selection */
	INT32
		columnarLeftX;			/* they are used when adding to the selection */
	UINT32
		columnarBottomLine;
	INT32
		columnarRightX;
	BOOLEAN
		haveStartX,				/* tell if desiredStartX/desiredEndX are valid */
		haveEndX;
	INT32
		desiredStartX,			/* desired X position of the start of the selection when expanding/reducing, or of the cursor when no selection (used when moving the cursor up or down) */
		desiredEndX;
	struct editorView
		*firstView;				/* head of linked list of views onto this universe */
	} EDITORUNIVERSE;

typedef struct editorBuffer
	{
	UINT32
		bufferHoldCount;		/* this is used to make sure that buffers which are deleted do not just become invalid (sometimes things are relying on them) */
	BOOLEAN
		shellBusy;				/* used by the shell to make sure it cannot modify a buffer while it is in use by some previous shell command */
	BOOLEAN
		fromFile;				/* tells if contents of this buffer represent an actual file */
	char
		*contentName;			/* name that describes the contents of the buffer (if fromFile=TRUE, then this is full absolute pathname of file) */
	EDITORUNIVERSE
		*editorUniverse;		/* pointer to the editor universe that contains the data in this buffer */
	EDITORTASK
		*theTask;				/* pointer to the task record for this buffer, or NULL if no task linked to this buffer */
	struct editorWindow
		*theWindow;				/* pointer to document window for this buffer (if there is one, NULL if none) */
	struct editorBuffer
		*nextBuffer;			/* next buffer in linked list of buffers */
	} EDITORBUFFER;

typedef struct editorWindow
	{
	UINT16
		windowType;				/* which type of editor window this is */
	EDITORBUFFER
		*theBuffer;				/* if this is a document window, then this points to the buffer, otherwise it is NULL */
	void
		*windowInfo;			/* pointer to gui specific structure that keeps all info needed for this window type */
	void
		*userData;				/* pointer to any additional gui specific info for this window (like the gui's actual window structure) */
	} EDITORWINDOW;

/* NOTE: it is possible to have multiple views of the same editor universe */
typedef struct editorView
	{
	EDITORWINDOW
		*parentWindow;			/* the window that contains this view */
	EDITORUNIVERSE
		*editorUniverse;		/* the editor universe on which this view is placed */
	void
		*viewInfo;				/* pointer to gui specific info about this view (gui needs to remember font, size, bounds, top line, etc) */
	struct editorView
		*nextUniverseView;		/* next view of the editor universe that this view is on */
	void
		(*viewTextChangedVector)(struct editorView *theView);	/* this vector is called by the view when it has finished a text change */
	void
		(*viewSelectionChangedVector)(struct editorView *theView);	/* this vector is called by the view when it has finished a selection change */
	void
		(*viewPositionChangedVector)(struct editorView *theView);	/* this vector is called by the view when it has finished a position change */
	/* all of these variables are used to facilitate updates of views */
	BOOLEAN
		wantHome;				/* used during buffer update, tells if given view should be homed */
	INT32
		updateStartX;			/* used during update, keeps the X position into the line being updated (if it is on the view) */
	UINT32
		startPosition,			/* used during update, keep track of the character positions that the view spans */
		endPosition;
	} EDITORVIEW;


typedef struct
	{
	BOOLEAN
		isVirtual;				/* tells if the keyCode is a virtual one or not */
	UINT32
		keyCode;				/* gives keyboard position of non-modifier key that was hit, or the virtual key number if it was a virtual key */
	UINT32
		modifiers;				/* modifier bits (enumerated above) */
	} EDITORKEY;

typedef struct
	{
	UINT32
		modifiers;				/* keyboard style modifier bits (enumerated above) */
	UINT32
		position;				/* absolute position (in pixels, or lines) */
	} VIEWPOSEVENTDATA;

typedef struct
	{
	UINT32
		keyCode;				/* mouse button (0-n) */
	UINT32
		modifiers;				/* keyboard style modifier bits (enumerated above) */
	INT32
		xClick,
		yClick;					/* view relative click point */
	} VIEWCLICKEVENTDATA;

typedef struct					/* high level view events (sent to the shell from the view) */
	{
	UINT16
		eventType;
	EDITORVIEW
		*theView;				/* the view that this event belongs to */
	void
		*eventData;				/* pointer to event specific data for event */
	} VIEWEVENT;

#define	HORIZONTALSCROLLTHRESHOLD	32				/* number of "pixels" to scroll horizontally whenever we do so */
#define	HORIZONTALMAXPIXELS			32767			/* maximum number of "pixels" that we can display horizontally */

/* ALL the routines that the editor requires ------------------------------------------------------------------------------------------ */

void EditorBeep();
void ReportMessage(char *format,...);
void GetEditorScreenDimensions(UINT32 *theWidth,UINT32 *theHeight);

void ResetEditorViewCursorBlink(EDITORVIEW *theView);
void InvalidateViewPortion(EDITORVIEW *theView,UINT32 startLine,UINT32 endLine,UINT32 startPixel,UINT32 endPixel);
void ScrollViewPortion(EDITORVIEW *theView,UINT32 startLine,UINT32 endLine,INT32 numLines,UINT32 startPixel,UINT32 endPixel,INT32 numPixels);
void SetViewTopLeft(EDITORVIEW *theView,UINT32 newTopLine,INT32 newLeftPixel);
void ViewsStartTextChange(EDITORUNIVERSE *theUniverse);
void ViewsEndTextChange(EDITORUNIVERSE *theUniverse);
void ViewsStartSelectionChange(EDITORUNIVERSE *theUniverse);
void ViewsEndSelectionChange(EDITORUNIVERSE *theUniverse);
void GetEditorViewGraphicToTextPosition(EDITORVIEW *theView,CHUNKHEADER *theChunk,UINT32 theOffset,INT32 xPosition,UINT32 *betweenOffset,UINT32 *charOffset);
void GetEditorViewTextToGraphicPosition(EDITORVIEW *theView,CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 linePosition,INT32 *xPosition);
void GetEditorViewTextToLimitedGraphicPosition(EDITORVIEW *theView,CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 linePosition,INT32 maxXPosition,INT32 *xPosition);
void GetEditorViewTextInfo(EDITORVIEW *theView,UINT32 *topLine,UINT32 *numLines,INT32 *leftPixel,UINT32 *numPixels);
void SetEditorViewTopLine(EDITORVIEW *theView,UINT32 lineNumber);
UINT32 GetEditorViewTabSize(EDITORVIEW *theView);
BOOLEAN SetEditorViewTabSize(EDITORVIEW *theView,UINT32 theSize);
BOOLEAN GetEditorViewFont(EDITORVIEW *theView,char *theFont,UINT32 stringBytes);
BOOLEAN SetEditorViewFont(EDITORVIEW *theView,char *theFont);
BOOLEAN GetEditorViewColors(EDITORVIEW *theView,EDITORCOLOR *foregroundColor,EDITORCOLOR *backgroundColor);
BOOLEAN SetEditorViewColors(EDITORVIEW *theView,EDITORCOLOR foregroundColor,EDITORCOLOR backgroundColor);

BOOLEAN EditorColorNameToColor(char *colorName,EDITORCOLOR *theColor);

void UpdateEditorWindows();

void SetTopDocumentWindow(EDITORWINDOW *theWindow);
EDITORWINDOW *GetActiveDocumentWindow();
EDITORWINDOW *GetTopDocumentWindow();
BOOLEAN GetSortedDocumentWindowList(UINT32 *numElements,EDITORWINDOW ***theList);
EDITORVIEW *GetDocumentWindowCurrentView(EDITORWINDOW *theWindow);

BOOLEAN GetDocumentWindowTitle(EDITORWINDOW *theWindow,char *theTitle,UINT32 stringBytes);
BOOLEAN SetDocumentWindowTitle(EDITORWINDOW *theWindow,char *theTitle);
BOOLEAN GetEditorDocumentWindowColors(EDITORWINDOW *theWindow,EDITORCOLOR *foregroundColor,EDITORCOLOR *backgroundColor);
BOOLEAN SetEditorDocumentWindowColors(EDITORWINDOW *theWindow,EDITORCOLOR foregroundColor,EDITORCOLOR backgroundColor);
BOOLEAN GetEditorDocumentWindowRect(EDITORWINDOW *theWindow,EDITORRECT *theRect);
BOOLEAN SetEditorDocumentWindowRect(EDITORWINDOW *theWindow,EDITORRECT *theRect);
EDITORWINDOW *OpenDocumentWindow(EDITORUNIVERSE *theUniverse,EDITORRECT *theRect,char *theTitle,char *fontName,UINT32 tabSize,EDITORCOLOR foreground,EDITORCOLOR background);
void CloseDocumentWindow(EDITORWINDOW *theWindow);

char *CreateWindowTitleFromPath(char *absolutePath);
char *CreateAbsolutePath(char *relativePath);
void FreeStartupScriptLocationList(char **theList);
char **GenerateStartupScriptLocationList();

void DeactivateEditorMenu(EDITORMENU *theMenu);
void ActivateEditorMenu(EDITORMENU *theMenu);
char *GetEditorMenuName(EDITORMENU *theMenu);
BOOLEAN SetEditorMenuName(EDITORMENU *theMenu,char *theName);
char *GetEditorMenuAttributes(EDITORMENU *theMenu);
BOOLEAN SetEditorMenuAttributes(EDITORMENU *theMenu,char *theAttributes);
char *GetEditorMenuDataText(EDITORMENU *theMenu);
BOOLEAN SetEditorMenuDataText(EDITORMENU *theMenu,char *theDataText);
EDITORMENU *GetRelatedEditorMenu(EDITORMENU *theMenu,int menuRelationship);
BOOLEAN GetEditorMenu(int argc,char *argv[],EDITORMENU **theMenu);
EDITORMENU *CreateEditorMenu(int argc,char *argv[],int menuRelationship,char *theName,char *theAttributes,char *dataText,BOOLEAN active);
void DisposeEditorMenu(EDITORMENU *theMenu);

BOOLEAN EditorKeyNameToKeyCode(char *theName,UINT32 *theCode);
char *EditorKeyCodeToKeyName(UINT32 theCode);
BOOLEAN EditorGetKeyPress(UINT32 *keyCode,UINT32 *editorModifiers,BOOLEAN wait,BOOLEAN clearBuffered);

void EditorQuit();
void EditorEventLoop(int argc,char *argv[]);
void ClearAbort();
BOOLEAN CheckAbort();

void ShowBusy();
void ShowNotBusy();

BOOLEAN ImportClipboard();
BOOLEAN ExportClipboard();

EDITORFILE *OpenEditorReadFile(char *thePath);
EDITORFILE *OpenEditorWriteFile(char *thePath);
BOOLEAN ReadEditorFile(EDITORFILE *theFile,UINT8 *theBuffer,UINT32 bytesToRead,UINT32 *bytesRead);
BOOLEAN WriteEditorFile(EDITORFILE *theFile,UINT8 *theBuffer,UINT32 bytesToWrite,UINT32 *bytesWritten);
void CloseEditorFile(EDITORFILE *theFile);

BOOLEAN OkDialog(char *theText);
BOOLEAN OkCancelDialog(char *theText,BOOLEAN *cancel);
BOOLEAN YesNoCancelDialog(char *theText,BOOLEAN *yes,BOOLEAN *cancel);
BOOLEAN GetSimpleTextDialog(char *theTitle,char *enteredText,UINT32 stringBytes,BOOLEAN *cancel);
BOOLEAN SearchReplaceDialog(EDITORUNIVERSE *searchUniverse,EDITORUNIVERSE *replaceUniverse,BOOLEAN *backwards,BOOLEAN *wrapAround,BOOLEAN *selectionExpr,BOOLEAN *ignoreCase,BOOLEAN *limitScope,BOOLEAN *replaceProc,UINT16 *searchType,BOOLEAN *cancel);
BOOLEAN SimpleListBoxDialog(char *theTitle,UINT32 numElements,char **listElements,BOOLEAN *selectedElements,BOOLEAN *cancel);
BOOLEAN OpenFileDialog(char *theTitle,char *fullPath,UINT32 stringBytes,char ***listElements,BOOLEAN *cancel);
void FreeOpenFileDialogPaths(char **thePaths);
BOOLEAN SaveFileDialog(char *theTitle,char *fullPath,UINT32 stringBytes,BOOLEAN *cancel);
BOOLEAN ChoosePathDialog(char *theTitle,char *fullPath,UINT32 stringBytes,BOOLEAN *cancel);
BOOLEAN ChooseFontDialog(char *theTitle,char *theFont,UINT32 stringBytes,BOOLEAN *cancel);

BOOLEAN WriteBufferTaskData(EDITORBUFFER *theBuffer,UINT8 *theData,UINT32 numBytes);
BOOLEAN SendEOFToBufferTask(EDITORBUFFER *theBuffer);
void DisconnectBufferTask(EDITORBUFFER *theBuffer);
BOOLEAN UpdateBufferTask(EDITORBUFFER *theBuffer);
BOOLEAN KillBufferTask(EDITORBUFFER *theBuffer);

void *MNewPtr(UINT32 theSize);
void *MNewPtrClr(UINT32 theSize);
void MDisposePtr(void *thePointer);
void MMoveMem(void *source,void *dest,UINT32 numBytes);
UINT32 MGetNumAllocatedPointers();

char *GetEditorLocalVersion();
BOOLEAN AddSupplementalShellCommands(Tcl_Interp *theInterpreter);
BOOLEAN InitEnvironment();
void UnInitEnvironment();
BOOLEAN EarlyInit();
void EarlyUnInit();

/* Editor routines and globals visible to the GUI ------------------------------------------------------------------------------------------ */

extern char
	*programName;

void PositionToLinePosition(TEXTUNIVERSE *theUniverse,UINT32 thePosition,UINT32 *theLine,UINT32 *theLineOffset,CHUNKHEADER **theChunk,UINT32 *theChunkOffset);
void PositionToChunkPositionPastEnd(TEXTUNIVERSE *theUniverse,UINT32 thePosition,CHUNKHEADER **theChunk,UINT32 *theOffset);
void PositionToChunkPosition(TEXTUNIVERSE *theUniverse,UINT32 thePosition,CHUNKHEADER **theChunk,UINT32 *theOffset);
void LineToChunkPosition(TEXTUNIVERSE *theUniverse,UINT32 theLine,CHUNKHEADER **theChunk,UINT32 *theOffset,UINT32 *thePosition);
void ChunkPositionToNextLine(TEXTUNIVERSE *theUniverse,CHUNKHEADER *theChunk,UINT32 theOffset,CHUNKHEADER **newChunk,UINT32 *newOffset,UINT32 *distanceMoved);
BOOLEAN InsertUniverseChunks(TEXTUNIVERSE *theUniverse,UINT32 thePosition,CHUNKHEADER *textChunk,UINT32 textOffset,UINT32 numBytes);
BOOLEAN InsertUniverseText(TEXTUNIVERSE *theUniverse,UINT32 thePosition,UINT8 *theText,UINT32 numBytes);
BOOLEAN ExtractUniverseText(TEXTUNIVERSE *theUniverse,CHUNKHEADER *theChunk,UINT32 theOffset,UINT8 *theText,UINT32 numBytes,CHUNKHEADER **nextChunk,UINT32 *nextOffset);
BOOLEAN DeleteUniverseText(TEXTUNIVERSE *theUniverse,UINT32 thePosition,UINT32 numBytes);
TEXTUNIVERSE *OpenTextUniverse();
void CloseTextUniverse(TEXTUNIVERSE *theUniverse);

EDITORBUFFER *EditorGetCurrentClipboard();
void EditorSetCurrentClipboard(EDITORBUFFER *theClipboard);
void EditorCopy(EDITORUNIVERSE *theEditorUniverse,EDITORBUFFER *theClipboard,BOOLEAN append);
void EditorCut(EDITORUNIVERSE *theEditorUniverse,EDITORBUFFER *theClipboard,BOOLEAN append);
void EditorPaste(EDITORUNIVERSE *theEditorUniverse,EDITORBUFFER *theClipboard);
void EditorColumnarPaste(EDITORVIEW *theView,EDITORBUFFER *theClipboard);
EDITORUNIVERSE *EditorStartImportClipboard();
void EditorEndImportClipboard();
EDITORUNIVERSE *EditorStartExportClipboard();
void EditorEndExportClipboard();

void DeleteUniverseSelection(SELECTIONUNIVERSE *theUniverse,ARRAYCHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numSelections,ARRAYCHUNKHEADER **nextChunk,UINT32 *nextOffset);
BOOLEAN InsertUniverseSelection(SELECTIONUNIVERSE *theUniverse,ARRAYCHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numSelections,ARRAYCHUNKHEADER **nextChunk,UINT32 *nextOffset);
SELECTIONUNIVERSE *OpenSelectionUniverse();
void CloseSelectionUniverse(SELECTIONUNIVERSE *theUniverse);
SELECTIONUNIVERSE *CopySelectionUniverse(SELECTIONUNIVERSE *theOldUniverse);

void EditorGetSelectionInfo(EDITORUNIVERSE *theEditorUniverse,UINT32 *startPosition,UINT32 *endPosition,UINT32 *startLine,UINT32 *endLine,UINT32 *startLinePosition,UINT32 *endLinePosition,UINT32 *totalSegments,UINT32 *totalSpan);
void EditorHomeViewToPositionLenient(EDITORVIEW *theView,UINT32 thePosition);
void EditorHomeViewToPositionStrict(EDITORVIEW *theView,UINT32 thePosition);
void EditorHomeViewToPositionSemiStrict(EDITORVIEW *theView,UINT32 thePosition);
void EditorHomeViewToCursorLenient(EDITORVIEW *theView,BOOLEAN useEnd);
void EditorHomeViewToCursorStrict(EDITORVIEW *theView,BOOLEAN useEnd);
void EditorHomeViewToCursorSemiStrict(EDITORVIEW *theView,BOOLEAN useEnd);
void EditorVerticalScroll(EDITORVIEW *theView,INT32 amountToScroll);
void EditorVerticalScrollByPages(EDITORVIEW *theView,INT32 pagesToScroll);
void EditorHorizontalScroll(EDITORVIEW *theView,INT32 amountToScroll);
void EditorHorizontalScrollByPages(EDITORVIEW *theView,INT32 pagesToScroll);
void EditorStartReplace(EDITORUNIVERSE *theUniverse);
BOOLEAN ReplaceEditorChunks(EDITORUNIVERSE *theUniverse,UINT32 startOffset,UINT32 endOffset,CHUNKHEADER *textChunk,UINT32 textOffset,UINT32 numBytes);
BOOLEAN ReplaceEditorText(EDITORUNIVERSE *theUniverse,UINT32 startOffset,UINT32 endOffset,UINT8 *theText,UINT32 numBytes);
void EditorEndReplace(EDITORUNIVERSE *theUniverse);
void EditorDeleteSelection(EDITORUNIVERSE *theEditorUniverse);
void EditorTrackViewPointer(EDITORVIEW *theView,INT32 viewLine,INT32 viewXPosition,UINT16 trackMode);
void EditorMoveCursor(EDITORVIEW *theView,UINT16 relativeMode);
void EditorExpandNormalSelection(EDITORVIEW *theView,UINT16 relativeMode);
void EditorReduceNormalSelection(EDITORVIEW *theView,UINT16 relativeMode);
void EditorLocateLine(EDITORUNIVERSE *theEditorUniverse,UINT32 theLine);
void EditorSelectAll(EDITORUNIVERSE *theUniverse);
void EditorDelete(EDITORVIEW *theView,UINT16 relativeMode);
void EditorInsert(EDITORUNIVERSE *theEditorUniverse,UINT8 *theText,UINT32 textLength);
void EditorAuxInsert(EDITORUNIVERSE *theEditorUniverse,UINT8 *theText,UINT32 textLength);
BOOLEAN EditorInsertFile(EDITORUNIVERSE *theEditorUniverse,char *thePath);
void EditorAutoIndent(EDITORUNIVERSE *theEditorUniverse);
BOOLEAN ClearEditorUniverse(EDITORUNIVERSE *theUniverse);
BOOLEAN LoadEditorUniverse(EDITORUNIVERSE *theUniverse,char *thePath);
BOOLEAN SaveEditorUniverse(EDITORUNIVERSE *theUniverse,char *thePath,BOOLEAN clearDirty);
void LinkViewToEditorUniverse(EDITORVIEW *theView,EDITORUNIVERSE *theUniverse);
void UnlinkViewFromEditorUniverse(EDITORVIEW *theView);

BOOLEAN GotoEditorMark(EDITORUNIVERSE *theUniverse,MARKLIST *theMark);
MARKLIST *LocateEditorMark(EDITORUNIVERSE *theUniverse,char *markName);
MARKLIST *SetEditorMark(EDITORUNIVERSE *theUniverse,char *markName);
void ClearEditorMark(EDITORUNIVERSE *theUniverse,MARKLIST *theMark);

EDITORUNIVERSE *OpenEditorUniverse();
void CloseEditorUniverse(EDITORUNIVERSE *theEditorUniverse);

void SetError(char *errorFamily,char *familyMember,char *memberDescription);
void SetStdCLibError();
void GetError(char **errorFamily,char **familyMember,char **memberDescription);

EDITORBUFFER *EditorGetFirstBuffer();
EDITORBUFFER *EditorGetNextBuffer(EDITORBUFFER *fromBuffer);

void InsertBufferTaskData(EDITORBUFFER *theBuffer,UINT8 *theData,UINT32 numBytes);
BOOLEAN HandleBoundKeyEvent(UINT32 keyCode,UINT32 modifierValue);
void HandleMenuEvent(EDITORMENU *theMenu);
void HandleViewEvent(VIEWEVENT *theEvent);
BOOLEAN HandleShellCommand(char *theCommand,int argc,char *argv[]);
