/* Handle high level buffer stuff
   Copyright (C) 1995 Viacom New Media.

	This file is part of e93.

	e93 is free software; you can redistribute it and/or modify
	it under the terms of the e93 LICENSE AGREEMENT.

	e93 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	e93 LICENSE AGREEMENT for more details.

	You should have received a copy of the e93 LICENSE AGREEMENT
	along with e93; see the file "LICENSE.TXT".
*/

/*
 * Some handy notes about buffers to help understand what some of the
 * flags mean:
 * bufferHoldCount
 *   This exists because shell scripts can delete buffers that the editor has
 *   pointers to, and the editor needs some method of determining that this
 *   happened. When the editor calls Tcl while it has pointers to buffers that
 *   it may wish to use again, it calls EditorHoldBuffer on the pointers.
 *   This will increment the hold count. When a buffer is deleted, the hold
 *   count is checked, and if it is non-zero, the buffer record in the
 *   linked list of buffers is not actually deleted. The editor can then
 *   check the content name field of the buffer to see if it is still active
 *   NULL=no. Once a buffer has been completely released, it's record is deleted
 * shellBusy
 *   This is used exclusively by the the shell. It indicates to the
 *   shell that it is already doing something with the buffer, and that it is
 *   not allowed to affect it in any way (useful when doing search and replace,
 *   where replace is a shell script... keeps script from being able to modify
 *   the buffer that is in the middle of being modified by replace command)
 */

#include "includes.h"

static EDITORBUFFER
	*startBuffer;								/* pointer to start of editor buffer list */

static char localErrorFamily[]="buffer";

enum
	{
	NOTFROMFILE,
	DUPLICATENAME
	};

static char *errorMembers[]=
	{
	"NotFromFile",
	"DuplicateName"
	};

static char *errorDescriptions[]=
	{
	"Buffer is not linked to a file",
	"Duplicate buffer names are not allowed"
	};

EDITORBUFFER *EditorGetFirstBuffer()
/* return the first active buffer, or NULL if none
 */
{
	EDITORBUFFER
		*theBuffer;

	theBuffer=startBuffer;
	while(theBuffer&&!(theBuffer->contentName))
		{
		theBuffer=theBuffer->nextBuffer;
		}
	return(theBuffer);
}

EDITORBUFFER *EditorGetNextBuffer(EDITORBUFFER *fromBuffer)
/* return the next active buffer, or NULL if none
 */
{
	EDITORBUFFER
		*theBuffer;

	theBuffer=fromBuffer->nextBuffer;
	while(theBuffer&&!(theBuffer->contentName))
		{
		theBuffer=theBuffer->nextBuffer;
		}
	return(theBuffer);
}

EDITORBUFFER *LocateEditorBuffer(char *bufferName)
/* search through the list of buffers, and attempt to locate
 * one with the given name
 * if one is found, return it, else return NULL
 */
{
	BOOLEAN
		found;
	EDITORBUFFER
		*testBuffer;

	testBuffer=startBuffer;
	found=FALSE;
	while(testBuffer&&!found)
		{
		if((testBuffer->contentName)&&(strcmp(testBuffer->contentName,bufferName)==0))
			{
			found=TRUE;
			}
		else
			{
			testBuffer=testBuffer->nextBuffer;
			}
		}
	return(testBuffer);
}

static void UnlinkAndRemoveBuffer(EDITORBUFFER *theBuffer)
/* remove a buffer from the list, and free it from memory
 */
{
	EDITORBUFFER
		*previousBuffer,
		*testBuffer;

	previousBuffer=NULL;
	testBuffer=startBuffer;
	while(testBuffer&&testBuffer!=theBuffer)
		{
		previousBuffer=testBuffer;
		testBuffer=testBuffer->nextBuffer;
		}
	if(testBuffer==theBuffer)									/* make sure we found it before trying to unlink */
		{
		if(previousBuffer)
			{
			previousBuffer->nextBuffer=theBuffer->nextBuffer;
			}
		else
			{
			startBuffer=theBuffer->nextBuffer;
			}
		}
	MDisposePtr(theBuffer);
}

void EditorCloseBuffer(EDITORBUFFER *theBuffer)
/* close the given edit buffer
 * if the buffer has a task, it will be disconnected
 * if the buffer has a window, it will be closed too
 * this will close the window without saving, even if the text has been modified
 */
{
	if(theBuffer->theTask)
		{
		DisconnectBufferTask(theBuffer);						/* close down the connection to any task that may be attached to this buffer */
		}
	if(theBuffer->theWindow)
		{
		EditorCloseDocumentWindow(theBuffer->theWindow);		/* close down the window if one exists */
		}
	CloseEditorUniverse(theBuffer->editorUniverse);
	MDisposePtr(theBuffer->contentName);
	theBuffer->contentName=NULL;								/* mark entry as deceased */
	if(!theBuffer->bufferHoldCount)								/* if it is not being held, get rid of it */
		{
		UnlinkAndRemoveBuffer(theBuffer);
		}
}

void EditorReleaseBuffer(EDITORBUFFER *theBuffer)
/* decrement the hold count on a buffer, and if the buffer is dead (as can be
 * gleaned by the fact that the contentName field is NULL), then
 * unlink, and remove it
 */
{
	if(!(--theBuffer->bufferHoldCount))							/* decrement, see if 0 */
		{
		if(!theBuffer->contentName)								/* see if it should be removed */
			{
			UnlinkAndRemoveBuffer(theBuffer);					/* remove it */
			}
		}
}

void EditorHoldBuffer(EDITORBUFFER *theBuffer)
/* increment the hold count on a buffer
 * if a buffer is closed that has a hold count other than 0,
 * it will not be unlinked, so that routines that are referencing
 * it will not be left without a valid pointer
 */
{
	theBuffer->bufferHoldCount++;
}

BOOLEAN EditorRevertBuffer(EDITORBUFFER *theBuffer)
/* revert theBuffer, no questions asked
 * if there is a problem, SetError, and return FALSE
 * NOTE: the window could be left with only part of the reverted text
 * if there was a problem
 */
{
	if(theBuffer->fromFile)
		{
		if(LoadEditorUniverse(theBuffer->editorUniverse,theBuffer->contentName))
			{
			return(TRUE);
			}
		}
	else
		{
		if(ClearEditorUniverse(theBuffer->editorUniverse))
			{
			return(TRUE);
			}
		}
	return(FALSE);
}

BOOLEAN EditorSaveBufferTo(EDITORBUFFER *theBuffer,char *thePath)
/* save the text of theBuffer into thePath
 * NOTE: the text will be saved even if it is not dirty
 * if there is a problem, SetError, and return FALSE
 * NOTE: the dirty flag is not cleared by this call
 * NOTE: certain problems may cause the text to be only partially saved
 * thus corrupting the file
 */
{
	return(SaveEditorUniverse(theBuffer->editorUniverse,thePath,FALSE));
}

BOOLEAN EditorSaveBufferAs(EDITORBUFFER *theBuffer,char *thePath)
/* save the text of theBuffer into thePath, change theBuffer's name
 * to the absolute version of thePath, and update its window if it has one
 * NOTE: the text will be saved even if it is not dirty
 * if there is a problem, SetError, and return FALSE
 * NOTE: certain problems may cause the text to be only partially saved
 * thus corrupting the file
 * NOTE: if after a save, the buffer's name becomes the same as that
 * of another already existing buffer, the name change will fail, even though
 * the contents of the buffer have been written into the file
 * this is a little ugly, but all editors have to cope with this problem somehow.
 */
{
	EDITORBUFFER
		*otherBuffer;
	char
		*absolutePath;

	if(SaveEditorUniverse(theBuffer->editorUniverse,thePath,TRUE))	/* save the file using the regular path */
		{
		if(absolutePath=CreateAbsolutePath(thePath))				/* attempt to make an absolute path out of the one given (this could fail if someone deleted the file we just saved before we get here!) */
			{
			if(!(otherBuffer=LocateEditorBuffer(absolutePath))||(otherBuffer==theBuffer))	/* see if there is another buffer down this path */
				{
				MDisposePtr(theBuffer->contentName);					/* wipe out the old content name */
				theBuffer->contentName=absolutePath;					/* point it at the new path */
				theBuffer->fromFile=TRUE;								/* linked to file now */
				EditorReassignDocumentWindowTitle(theBuffer);			/* make new title to reflect new name */
				return(TRUE);
				}
			else
				{
				SetError(localErrorFamily,errorMembers[DUPLICATENAME],errorDescriptions[DUPLICATENAME]);
				}
			MDisposePtr(absolutePath);
			}
		}
	return(FALSE);
}

BOOLEAN EditorSaveBuffer(EDITORBUFFER *theBuffer)
/* save the text of theBuffer into its file if it has one
 * if it does not have a file, this will fail
 */
{
	if(theBuffer->fromFile)
		{
		return(SaveEditorUniverse(theBuffer->editorUniverse,theBuffer->contentName,TRUE));
		}
	SetError(localErrorFamily,errorMembers[NOTFROMFILE],errorDescriptions[NOTFROMFILE]);
	return(FALSE);
}

EDITORBUFFER *EditorNewBuffer(char *bufferName)
/* create a new buffer with the given name
 * NOTE: if there is already a buffer with the given name, SetError
 * if there is a problem, SetError, return NULL
 */
{
	EDITORBUFFER
		*theBuffer;

	if(!LocateEditorBuffer(bufferName))
		{
		if(theBuffer=(EDITORBUFFER *)MNewPtr(sizeof(EDITORBUFFER)))
			{
			if(theBuffer->contentName=(char *)MNewPtr(strlen(bufferName)+1))
				{
				strcpy(theBuffer->contentName,bufferName);
				theBuffer->bufferHoldCount=0;					/* not being held */
				theBuffer->shellBusy=FALSE;						/* set this to a known value */
				theBuffer->fromFile=FALSE;
				theBuffer->theTask=NULL;
				theBuffer->theWindow=NULL;
				if(theBuffer->editorUniverse=OpenEditorUniverse())
					{
					theBuffer->nextBuffer=startBuffer;
					startBuffer=theBuffer;
					return(theBuffer);
					}
				MDisposePtr(theBuffer->contentName);
				}
			MDisposePtr(theBuffer);
			}
		}
	else
		{
		SetError(localErrorFamily,errorMembers[DUPLICATENAME],errorDescriptions[DUPLICATENAME]);
		}
	return(NULL);
}

EDITORBUFFER *EditorOpenBuffer(char *thePath)
/* open an editor buffer to the file at thePath
 * if possible, open it, and return the buffer
 * if there is a problem, set the error, and return NULL
 * NOTE: if thePath points to a buffer that is already open,
 * then that buffer is simply returned
 */
{
	EDITORBUFFER
		*theBuffer;
	char
		*absolutePath;

	if(absolutePath=CreateAbsolutePath(thePath))				/* attempt to make an absolute path out of the one given */
		{
		if(theBuffer=LocateEditorBuffer(absolutePath))
			{
			MDisposePtr(absolutePath);
			return(theBuffer);
			}
		else
			{
			if(theBuffer=(EDITORBUFFER *)MNewPtr(sizeof(EDITORBUFFER)))
				{
				theBuffer->contentName=absolutePath;
				theBuffer->bufferHoldCount=0;					/* not being held */
				theBuffer->shellBusy=FALSE;						/* set this to a known value */
				theBuffer->fromFile=TRUE;
				theBuffer->theTask=NULL;
				theBuffer->theWindow=NULL;
				if(theBuffer->editorUniverse=OpenEditorUniverse())
					{
					if(LoadEditorUniverse(theBuffer->editorUniverse,absolutePath))
						{
						theBuffer->nextBuffer=startBuffer;
						startBuffer=theBuffer;
						return(theBuffer);
						}
					CloseEditorUniverse(theBuffer->editorUniverse);
					}
				MDisposePtr(theBuffer);
				}
			}
		MDisposePtr(absolutePath);
		}
	return(NULL);
}

BOOLEAN InitBuffers()
/* initialize buffer handling
 * if there is a problem, SetError, and return FALSE
 */
{
	startBuffer=NULL;										/* no start buffer as of yet */
	return(TRUE);
}

void UnInitBuffers()
/* undo whatever InitBuffers did
 */
{
	while(startBuffer)
		{
		if(startBuffer->bufferHoldCount)
			{
			fprintf(stderr,"Buffer dying with hold count (%d)\n",startBuffer->bufferHoldCount);
			startBuffer->bufferHoldCount=0;					/* clear the count so the buffer will DIE */
			}
		EditorCloseBuffer(startBuffer);						/* get rid of any buffers left lying about */
		}
}
