/****************************************************************************
*  Copyright (C) 1996-98 by Leo Khramov
*  email:     leo@unix1.jinr.dubna.su
*  
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
 ****************************************************************************/

//This file contains routines for FTP filesystem (FTPFS).
//There are two parts: Child and parent.
//Parent sends commands to child, and child connects and talking to actual ftp.

#include "xh.h"
#include "panel.h"
#include "ftplib/ftp_communication.h"
#include "bookmark.h"
#include "query_windows.h"
#include "infowin.h"
#include "ftpfs.h"
#include "ftpvisuals.h"


#define         DEBDELAY        20        //DEBUG delay - FTP operation delay ->sleep(DEBDELAY)

static     char ftpsyspath[PATH_MAX];

int ftp_cache_disabled=0;                //If set to 1 then we skip cache routines.

extern BookMark *bmark;
extern "C" void process_x_event(XEvent *);
extern void simple_flist_add(FList * base, FList * it);
extern void simple_mes(char *head, char *mes);
extern char *syspath;
extern int    xnc_ex;        //Placed in main.cxx: if xnc_ex==0 then no exit clean function for XNC
extern FtpVisual *fvis;        //Placed in main.cxx
extern Switcher *fsw1;        //Placed in main.cxx

static char ftptmpstr[2*PATH_MAX];
static char ftptmpstr2[2*PATH_MAX];

FTP *ftparr[MaxFtps];        //Array of active FTPs

void  (*xnc_look_at_pipes)(fd_set*)=ftp_backgrounds;        //Function for update FTP state in bg.
int    (*xnc_set_pipes)(fd_set*, int)=ftp_backgrounds_fdset;        //Function for select FTP bg pipes.


//Looking through ftparr and see 'is there empty place?'
int is_ftp_place()
{
 int i;
 for(i=0;i<MaxFtps;i++)
   if(ftparr[i]==NULL)
      return 1;
 return 0;
}


int where_is_ftp_place(FTP *o)
{
 int i;
 for(i=0;i<MaxFtps;i++)
   if(ftparr[i]==0)
      return i;
 return -1;
}


//Set background operation code to FTP and to its dup in ftparr.
void ftp_set_bgcode(FTP *ftp,int opcode)
{
  ftp->bgbit=opcode;
  if(ftparr[ftp->dup_entry])
    ftparr[ftp->dup_entry]->bgbit=opcode;
}

//Register FTP in ftparr.
void register_ftp(FTP* ftp)
{ 
  int i;
  fsw1->switch_to(1);        //Switch MenuBar to FtpVisual.
  for(i=0;i<MaxFtps;i++)
    if(ftparr[i]==NULL)
    {
       ftparr[i]=ftp;
       fvis->rescan();
       return;
    }
}

//Find 'prev' in ftparr and change it to 'ftp'
int reregister_ftp(FTP* prev, FTP* ftp)
{ 
  int i;
  for(i=0;i<MaxFtps;i++)
    if(ftparr[i]==prev)
    {
       ftparr[i]=ftp;
       return i;
    }
  for(i=0;i<MaxFtps;i++)
    if(ftparr[i]==NULL)
    {
       ftparr[i]=ftp;
       return i;
    }
 return -1;
}

//Delete 'ftp' from ftparr
void unregister_ftp(FTP* ftp)
{ 
  int i;
  for(i=0;i<MaxFtps;i++)
    if(ftparr[i]==ftp)
    {
       ftparr[i]=NULL;
       fvis->rescan();
       break;
    }
  for(i=0;i<MaxFtps;i++)
    if(ftparr[i])
            return;
//If we are here then no active FTP found -> switch MenuBar to MainMenu.
  fsw1->switch_to(0);
}

//Looking through all background FTP
//and receive events from its.
void ftp_backgrounds(fd_set *fds)
{
 int i;
 for(i=0;i<MaxFtps;i++)
   if(ftparr[i] && ftparr[i]->bgbit)
           ftparr[i]->bg_select_if_set(fds);
}

int ftp_backgrounds_fdset(fd_set *fds, int fd)
{
 int i;
 for(i=0;i<MaxFtps;i++)
   if(ftparr[i] && ftparr[i]->bgbit)
          fd = ftparr[i]->bg_fd_set(fd, fds);
 return fd;
}

void init_ftp_globals()
{
  int i;
  for(i=0;i<MaxFtps;i++)
    ftparr[i]=NULL;
}

//This will be run on exit. (Clean up ftp).
void deinit_all_ftp()
{
  int i;
  for(i=0;i<MaxFtps;i++)
    if(ftparr[i])
       ftparr[i]->close_fs();
}

int write_select(int fp)
{
  fd_set fds;
  struct timeval tv;
  tv.tv_sec=tv.tv_usec=0;
  FD_ZERO(&fds);
  FD_SET(fp,&fds);
  if(select(fp+1,NULL,&fds,NULL,&tv))
          return FD_ISSET(fp,&fds);
  return 0;
}

int read_select(int fp)
{
  fd_set fds;
  struct timeval tv;
  tv.tv_sec=tv.tv_usec=0;
  FD_ZERO(&fds);
  FD_SET(fp,&fds);
  if(select(fp+1,&fds,NULL,NULL,&tv))
          return FD_ISSET(fp,&fds);
  return 0;
}

int write_select_wait(int fp)
{
  fd_set fds;
  FD_ZERO(&fds);
  FD_SET(fp,&fds);
  if(select(fp+1,NULL,&fds,NULL,NULL))
          return FD_ISSET(fp,&fds);
  return 0;
}

int read_select_wait(int fp)
{
  fd_set fds;
  FD_ZERO(&fds);
  FD_SET(fp,&fds);
  if(select(fp+1,&fds,NULL,NULL,NULL))
          return FD_ISSET(fp,&fds);
  return 0;
}

void dummy_func() {};

//Open pipes and fork ftp child.
int FTP::ftp_fork()
{
  if(pipe(wp)==-1)
  {
          errno2mes();
          return -1;
  }
  if(pipe(rp)==-1)
  {
          errno2mes();
          return -1;
  }
  if((ftpid=fork())==-1)
  {
          errno2mes();
          return -1;
  }
  if(ftpid)        //We are parent process.
    return 1;

//Child part of FTP begins here
//  close(1);
//  close(2);
  close(XConnectionNumber(disp));
  xnc_ex=0;                //Disable XFree.. functions on exit.
  ftplib_debug=0;      //Ftp debugging ON -> stderr
  signal (SIGHUP, SIG_DFL);
  signal (SIGINT, SIG_DFL);
  signal (SIGQUIT, SIG_DFL);
  signal (SIGTERM, SIG_DFL);
  signal (SIGCHLD, SIG_DFL);
  ftpid=getpid();
  while(1)
  {
    read_select_wait(wp[0]);        //Wait for event from parent
    read(wp[0],&com,S_Ftp_Com);
    child_ftp_command();        //Do the event!
  }
  return 0;
}

int FTP::child_get_single_file(int mode, char *fname)
{
  char ftptmpstr3[PATH_MAX];
  int ret;
  add_path_content(com.path, fname);
  add_path_content(com.remotepath, fname);

  if(mode & S_IFDIR)
        {
                    if(FtpGetDir(netb, com.remotepath, com.path, ftptmpstr, ftptmpstr2, ftptmpstr3)!=0)
                    {
                      com.response=-1;
                      upper_path(com.remotepath);
                      upper_path(com.path);
                      strcpy(com.reason,ftp_reason);
                      return 1;
                    } else
                           strcpy(com.reason, netb->response);
                   upper_path(com.remotepath);
                   upper_path(com.path);
                   return 0;
        }
        
         ret=FtpBinGet(com.path,com.remotepath,mode,netb);

         upper_path(com.remotepath);
         upper_path(com.path);
         return ret ? 0 : 1;
}  


void FTP::child_ftp_command()
{
 char tstr[40];
 com.response=1;
 int len,mode,l;
 char localtmp1[PATH_MAX];
 char localtmp2[PATH_MAX];
 char localtmp3[PATH_MAX];
 char err_str[256];
 FILE *fp,*ofp;
 switch(com.command)
 {
         case FTP_CONNECT:
                 if(FtpConnect(com.hostname, &netb)==0)
                 {
                         com.response=-1;
                         if(netb)
                         {
                                 strcpy(com.reason, netb->response);
                                 free(netb);
                         } else
                                 strcpy(com.reason, "Memory allocation error!!");
                         com.command=FTP_BYE;
                         break;
                 }
                 if(FtpLogin(com.user,com.passwd,netb)==0)
                 {
                         com.response=-1;
                         strcpy(com.reason,netb->response);
                         FtpQuit(netb);
                         com.command=FTP_BYE;
                         break;
                 }
       create_ftp_cachedir(curdir);
      if(ftp_cache_disabled)
              sprintf(com.tmppath,"%s/FTPFS/.Xnc.Ftp%d.ls",syspath,ftpid);
      else
              sprintf(com.tmppath,"%s/FTPFS/%s/%s/.Xnc.Ftp.ls",syspath,com.hostname,com.remotepath);

        case FTP_LS:
#if FDEB
  fprintf(stderr,"FTP_LS: '%s'\n",com.remotepath);
#endif
//                 sprintf(tstr,"Ftp.%d.ls",getpid());
//                 add_path_content(com.tmppath,tstr);
        sleep(1);
                 if(FtpChdir(com.remotepath,netb)==0)
                 {
                         com.response=-1;
                         upper_path(com.tmppath);
                         strcpy(com.reason,netb->response);
                         FtpPwd(com.remotepath, netb);
                         break;
                 }
                 if(FtpLS(com.tmppath,NULL,netb)==0)
                 {
                         com.response=-1;
                 }
//                 upper_path(com.tmppath);
                 strcpy(com.reason, netb->response);
                 break;
                 
         case FTP_LCD:
                 break;
                 
         case FTP_CD:
#if FDEB
  fprintf(stderr,"FTP_CD: '%s'\n",com.remotepath);
#endif
                 if(FtpChdir(com.remotepath,netb)==0)
                 {
                         com.response=-1;
                 }
                 strcpy(com.reason, netb->response);
                 FtpPwd(com.remotepath, netb);
//              sleep(DEBDELAY);
                 break;
         case FTP_MGET:
                 fp=fopen(com.tmppath,"r");
                 if(fp==NULL)
                 {
                   strcpy(com.reason,"Can't read list file!");
                   com.response=-1;
                   break;
                 }
                 strcpy(localtmp1,com.tmppath);
                 strcat(com.tmppath,".ret");
                 ofp=fopen(com.tmppath,"w");
                 if(ofp==NULL)
                 {
                   fclose(fp);
                   rename(localtmp1,com.tmppath);
                   strcpy(com.response, "Can't create return list!");
                   com.response=-1;
                   break;
                 }
                 
                 sprintf(ftptmpstr,"%s/%s.%d.ls", syspath, "FTPMGET",this);
                 
                 while(fgets(localtmp2,PATH_MAX,fp)!=NULL)
                 {
                   sscanf(localtmp2,"%o %[^\n]\n", &mode, localtmp3);
                   l=strlen(localtmp3);
                   if(localtmp3[l-1]=='\r')
                     localtmp3[l-1]=0;
                   if(child_get_single_file(mode, localtmp3)!=0)
                   {
                      strcpy(err_str,netb->response);
                      fprintf(ofp,"%s\n",localtmp3);
                      com.response=-1;
                   }
                 }
                 fclose(fp);
                 fclose(ofp);
                 unlink(localtmp1);
                 unlink(ftptmpstr);
                 break;
                 
         case FTP_GET:
//              sleep(DEBDELAY);
                 if(com.file_attr & S_IFDIR)
                 {
                    sprintf(tstr,"FtpTMP.%d.ls",getpid());
                    add_path_content(com.tmppath,tstr);
                    strcpy(localtmp1, com.remotepath);
                    strcpy(localtmp2, com.path);
                    if(FtpGetDir(netb, localtmp1, localtmp2, com.tmppath, ftptmpstr,ftptmpstr2)!=0)
                    {
                      com.response=-1;
                      strcpy(com.reason,ftp_reason);
                    } else
                           strcpy(com.reason, netb->response);
                   upper_path(com.remotepath);
                   upper_path(com.path);
                   upper_path(com.tmppath);
                   break;
                 }
                 com.response=FtpBinGet(com.path,com.remotepath,com.file_attr,netb) ? 0 : -1;
                 strcpy(com.reason, netb->response);
                 upper_path(com.remotepath);
                 upper_path(com.path);
                 break;
                 
         case FTP_BYE:
                 FtpQuit(netb);
                 strcpy(com.reason, netb->response);
                 break;
 };                

//Send reply to parent.
 len=strlen(com.reason);
 if(com.reason[len-2]=='\r')
   com.reason[len-2]='\0';
 if(com.reason[len-1]=='\n')
   com.reason[len-1]='\0';
 write_select_wait(rp[1]);
 write(rp[1],&com, S_Ftp_Com);
 if(com.command==FTP_BYE)
 {
         close(wp[0]);
         close(wp[1]);
         close(rp[0]);
         close(rp[1]);
         exit(0);
 }
}

//Send data to ftp child.
int FTP::send_to_ftpchild(Ftp_Com *com)
{
  while(!write_select(wp[1]));
  if(write(wp[1],com,S_Ftp_Com)==-1)
  {
    errno2mes();
    return 0;
  }
  return 1;
}

//If reply from child is ready, then process it!
void FTP::bg_select()
{
  int i;
  if(read_select(rp[0]))
  {
          read(rp[0],&com,S_Ftp_Com);
          work=0;
          fvis->refresh(this);
          if(autoraise==0)
                  return;
          i=where_is_ftp_place(this);
          if(i!=-1)
             fvis->animate_moving(i);
          bg_commands();
  }
}                           

void FTP::bg_select_if_set(fd_set *fds)
{
  int i;
  if(FD_ISSET(rp[0], fds))
  {
          read(rp[0],&com,S_Ftp_Com);
          work=0;
          fvis->refresh(this);
          if(autoraise==0)
                  return;
          i=where_is_ftp_place(this);
          if(i!=-1)
             fvis->animate_moving(i);
          bg_commands();
  }
}                           

int FTP::bg_fd_set(int fd, fd_set *fds)
{
  FD_SET(rp[0], fds);
  return rp[0]>fd ? rp[0] : fd;
}


void FTP::bg_switch()
{
  if(work)
  {
    simple_mes("FTP","Operation not complete!");
    return;
  }
  bg_commands();
}

void FTP::bg_commands()
{
          switch(bgbit)
          {
                  case FF_CON:
                          if(com.response==-1)
                          {
                                  vfs_error("FTP",com.reason);
                                  show_vfs_error();
                          }
                          else
                          {
                            need_reread=0;
                            bgbit=0;
                            panel->push_n_pop(this);
                            reregister_ftp(this,&panel->ftp);
                            create_ftp_cachedir(curdir);
                            delete this;
                          }
                          break;
                  case FF_CLO:
                          close(wp[0]);
                          close(wp[1]);
                          close(rp[0]);
                          close(rp[1]);
                          bgbit=0;
                          unregister_ftp(this);
                          wait(NULL);
                          delete this;
                          break;
                  case FF_CD:
                          need_reread = 1;
                          if(com.response==-1)
                          {
                               vfs_error("FTP",com.reason);
                               show_vfs_error();
                          } else
                          {
                                  bgbit=0;
                                  panel->push_n_pop(this);
                                  reregister_ftp(this,&panel->ftp);
                                  create_ftp_cachedir(curdir);
                                  delete this;
                          }
                          break;
                  case FF_DIR:
                          need_reread=0;
                          bgbit=0;
                          panel->push_n_pop(this);
                          reregister_ftp(this,&panel->ftp);
                          delete this;
                          break;
                  case FF_VIE:
                  case FF_SVI:
                          need_reread=0;
                          if(com.response!=-1)
                                  panel->bg_view(tmps);
                          else
                            {
                                 vfs_error("FTP",com.reason);
                                 show_vfs_error();
                            }
                          bgbit=0;
                          panel->push_n_pop(this);
                          reregister_ftp(this,&panel->ftp);
                          delete this;
                          break;
                  case FF_EDI:
                          need_reread=0;
                          if(com.response!=-1)
                                  panel->bg_edit(tmps);
                          else
                            {
                                 vfs_error("FTP",com.reason);
                                 show_vfs_error();
                            }
                          bgbit=0;
                          panel->push_n_pop(this);
                          reregister_ftp(this,&panel->ftp);
                          delete this;
                          break;
                  case FF_GET:
                          need_reread=0;
                          if(com.response==-1)
                          {
                              vfs_error("FTP",com.reason);
                              show_vfs_error();
                          }
                          bgbit=0;
                          panel->push_n_pop(this);
                          reregister_ftp(this,&panel->ftp);
                          delete this;
                          break;
          }
   bgbit=0;
}

//Wait while child complete operation
//This routine can change FTP status to BG.
int FTP::ftp_wait(char *mes,int opcode)
{
  time_t tt,tt2;
  disable_reread=1;
  work=1;
  fvis->refresh(this);
  if(mes)
  {
          create_ftp_infowin("FTP");
          to_infowin(mes);
  }
  time(&tt);
  while(!read_select(rp[0]))
  {
      if (XPending(disp))
        {
          XNextEvent(disp, &ev);
          process_x_event(&ev);
        }
      time(&tt2);
      if(bgbit==0 && infowin->bgbit && xnc_ex)
      {
              bgbit=opcode;
              delete_vfs_list();
              FTP *ftp=new FTP;
              *ftp=*this;
              dup_entry=reregister_ftp(this,ftp);
              if(mes)
                    del_infowin();
              return BGM;        //We are in background mode! (operation not completed yet)
      }
  }
  disable_reread=0;
  if(mes)
    del_infowin();
  work=0;
  fvis->refresh(this);
  return FGM;
}

void FTP::create_ftp_cachedir(char *dir)
{
   char str[PATH_MAX];
   char str2[FLIST_NAME];
         if(ftp_cache_disabled)
                   return;
         sprintf(str,"%s/FTPFS/%s",syspath,host);
         if(::mkdir(str,0755)==-1 && errno!=EEXIST)
                 return;
         if(strcmp(dir,"/"))
         {
                 char *s=dir;
                 if(*s=='/')         //Skip leading '/'
                         s++;
                 do
                 {
                         s=get_first_content(s,str2);
                         add_path_content(str,str2);
                         if(::mkdir(str,0755)==-1 && errno!=EEXIST)
                                 break;
                 } while(s);
         }
}

//Initialize FTPlink and run ftp.xnc if need.
//Return 1 if OK and 0 if failed.    
int    FTP::init_support(char *rhost)
{
  char  *b;

  if(is_ftp_place()==0)
  {
          vfs_error("FTP","Too many active connections");
          return 0;
  }
  com.command = FTP_CONNECT;
  com.xncpid = getpid();
  b = strchr(rhost, '@');
  if (b)                        //We found '@' then before this sign we have username
    {
      *b++ = 0;
      strcpy(host, b);
      b = strchr(rhost, ':');
      if (b)                        //We found ':' - before username before it and password after
        {
          *b++ = 0;
          strcpy(com.passwd, b);
        }
      else
        strcpy(com.passwd, "unknown@ftp.none.net");
      strcpy(com.user, rhost);
    }
  else
    {
      strcpy(com.user, "anonymous");
      strcpy(com.passwd, "unknown@ftp.none.net");
      strcpy(host, rhost);
    }
  strcpy(com.remotepath, "/");
  strcpy(com.tmppath, syspath);
// *INDENT-OFF*        
  ::getcwd(com.path, 1024);
// *INDENT-ON*        

  if(ftp_fork()==-1)
  {
        vfs_error("FTP","Forking Child");
        return 0;
  }
  if (send_to_ftpchild(&com))
    {
          make_empty_list();
          if(ftp_wait("Connecting", FF_CON)==BGM)
                  return 1;
          read(rp[0],&com,S_Ftp_Com);
          if(com.response==-1)
          {
                  vfs_error("FTP",com.reason);
                  return 0;
          }
          need_reread=0;
          register_ftp(this);
          create_ftp_cachedir(curdir);
          return 1;
    }
  vfs_error("FTP","Child process failed");
  return 0;
}

extern "C" void set_cmdchild_signal();
//Close ftp and wait for child DIE!
void   FTP::close_fs()
{
  signal(SIGCHLD, SIG_DFL);
  com.xncpid = getpid();
  com.command = FTP_BYE;
  send_to_ftpchild(&com);
  if(ftp_wait("Closing connection", FF_CLO)==BGM)
          return;
  read(rp[0],&com,S_Ftp_Com);
  close(wp[0]);
  close(wp[1]);
  close(rp[0]);
  close(rp[1]);
  unregister_ftp(this);
  wait(NULL);
  set_cmdchild_signal();
}

char*   FTP::get_file_for_execute(char *fname)
{
  com.xncpid = getpid();
  com.command = FTP_GET;
  com.file_attr=0755;
  strcpy(com.path,syspath);
  add_path_content(com.path,fname);
  strcpy(tmps,com.path);
  add_path_content(com.remotepath,fname);
  send_to_ftpchild(&com);
  upper_path(com.remotepath);
  upper_path(com.path);
  if(ftp_wait("Getting file", FF_GET)==BGM)
  {
          panel->switch_to_prev_vfs();
          return NULL;
  }
  read(rp[0],&com,S_Ftp_Com);
  if(com.response==-1)
     return NULL;
  return fname;
}

int   FTP::copy(FList *o, VFS* vfs)
{
  if(vfs->fstype!=DFS_TYPE)        //Currently we can copy only to DFS
          return 255;
  del_infowin();
  
  struct stat dstat;
  char tmps[PATH_MAX];
  FList *ko;
  FILE *fp;
  if(opt==0) //If no optimization then we copy only one file from FTP
  {  
          int checkres=vfs->is_file_exist(vfs->curdir, o->name);
//Check file existance.
              if (ow_all == 0 && dont_ask == 0)
                if (checkres)
                  {
                    init_overwrite_query("Copy file", o->name);
                    wait_for_query_done();
                    if (ow_all == 0 && ow_file == 0)
                      return ow_cancel;        //if cancel we return as error!!!
        
                  }
              ow_file = 0;
        
          com.xncpid = getpid();
          com.command = FTP_GET;
          com.file_attr=o->mode;
          strcpy(com.tmppath, syspath);
          strcpy(com.path,vfs->curdir);
          if (!S_ISDIR(o->mode) || (::stat(vfs->curdir, &dstat) != -1 && S_ISDIR(dstat.st_mode)))
                add_path_content(com.path,o->name);
          add_path_content(com.remotepath,o->name);
          send_to_ftpchild(&com);
          upper_path(com.remotepath);
          upper_path(com.path);
          if(ftp_wait("Getting file", FF_GET)==BGM)
          {
                  panel->switch_to_prev_vfs();
                  return 0;
          }
          read(rp[0],&com,S_Ftp_Com);
          if(com.response==-1)
          {
             vfs_error("FTP",com.reason);
             return 255;
          }
  } else         //If we are here then doing list file copy
  {
          ko = dl.next;
          strcpy(tmps,syspath);
          sprintf(com.tmppath,"TMPLIST.%d",this);
          add_path_content(tmps,com.tmppath);
          fp=fopen(tmps,"w");
          if(fp==NULL)
          {
             vfs_error("FTP","Can't create temporary list file!");
             return 255;
          }
          while (ko != NULL)
            {
              if (ko->mode & S_SELECT)
                {
                      //Check file existance.
                      if (ow_all == 0 && dont_ask == 0)
                        if (vfs->is_file_exist(vfs->curdir,ko->name))
                          {
                            init_overwrite_query("Copy file", ko->name);
                            wait_for_query_done();
                            if (ow_cancel)
                            {
                              fclose(fp);
                              unlink(tmps);
                              return ow_cancel;        //if cancel we return as error!!!
                            }
                          }
                       ow_file = 0;
                       if(ow_no)
                       {
                         ko->mode^=S_SELECT;
                         ow_no=0;
                       } else
                          fprintf(fp,"%o %s\n",ko->mode, ko->name);
                }
              ko = ko->next;
            }
      fclose(fp);

          com.xncpid = getpid();
          com.command = FTP_MGET;
          strcpy(com.tmppath, tmps);
          strcpy(com.path,vfs->curdir);

          send_to_ftpchild(&com);

          if(ftp_wait("MGetting files", FF_MGET)==BGM)
          {
                  panel->switch_to_prev_vfs();
                  return 0;
          }
          read(rp[0],&com,S_Ftp_Com);
          if(com.response==-1)
          {
             vfs_error("FTP",com.reason);
             return 255;
          }
      
      opt=0;
  }
     
  return 0;
}

int    FTP::chdir(char *d)
{
  if(strcmp(d,"..")==0 && strcmp(curdir,"/")==0)
  {
    need_change_vfs=1;
    return 0;
  }
  need_change_vfs=0;
  if(strcmp(com.remotepath,d)==0)
          return 0;
  com.command = FTP_CD;
  strcpy(com.remotepath, d);
  if (send_to_ftpchild(&com))
    {
      if(ftp_wait("Changing directory",FF_CD)==BGM)
      {
              panel->switch_to_prev_vfs();
              return 0;
      }
      read(rp[0],&com,S_Ftp_Com);
      need_reread = 1;
      if(com.response==-1)
         return -1;
      create_ftp_cachedir(curdir);
      return 0;
    }
  return -1;
}

char  *FTP::get_dir_header()
{
  sprintf(ftptmpstr, "ftp://%s%s", host, curdir);
  return ftptmpstr;
}

//Convert file info from list file to stat struct.
char  *convert_ftpstring(char *s, struct stat *st)
{
  static char strbuf[PATH_MAX];
  int l;
/*
  char  *r = ++s;
  while (*s != '\'')
    s++;
  *s++ = 0;
  st->st_mode = 0;
  if (strcmp(r, ".") == 0)
    return NULL;
*/
  sscanf(s, "%d %o %X %[^\n]\n", &st->st_size, &st->st_mode, &st->st_mtime,strbuf);
  l=strlen(strbuf);
  if(strbuf[l-1]=='\r')
          strbuf[l-1]=0;
  st->st_uid = st->st_gid = 0;
  if (strcmp(strbuf, "..") == 0 || strcmp(strbuf,".")==0)
    return NULL;
  return strbuf;
}

int    FTP::direc(char *dir_ch)
{
  FList *ol, *cur, *hd;
  Namer *namer=NULL,*n;
  char  *d_name;
  FILE  *d;
  int    ct = 0;
  struct stat st;
  char   ftpname[1024];
  char   lbuf1[FLIST_NAME];
  char   lbuf2[FLIST_NAME];
  int linkcoun=0;

  if (need_reread)
    {
      com.command = FTP_LS;
      if(ftp_cache_disabled)
              sprintf(com.tmppath,"%s/FTPFS/.Xnc.Ftp%d.ls",syspath,ftpid);
      else
              sprintf(com.tmppath,"%s/FTPFS/%s/%s/.Xnc.Ftp.ls",syspath,host,curdir);
      if(ftp_cache_disabled || stat(com.tmppath,&st)==-1)
      {
       create_ftp_cachedir(curdir);
       if (send_to_ftpchild(&com))
       {
        make_empty_list();
        if(ftp_wait("Directory listing",FF_DIR)==BGM)
                return 1;
       }
       else
         return 0;
       read(rp[0],&com,S_Ftp_Com);
       if(com.response==-1)
               return 0;
      }
      need_reread = 0;
    }
//  sprintf(ftpname, "%s/Ftp.%d.ls", syspath, ftpid);
  d = fopen(com.tmppath, "r");
  if (d == NULL)
    return 0;

  delete_vfs_list();
  while (fgets(ftpname, 1024, d) != NULL)
    {
      if ((d_name = convert_ftpstring(ftpname, &st)) == NULL)
        {
          if (st.st_mode)
            {
              hd = new_alloc(S_FList);
              hd->init("..", st.st_mode, st.st_size, st.st_uid, st.st_gid, st.st_mtime);
            }
          continue;
        }
      if (hide_hidden && d_name[0] == '.')
        continue;
      bmark->animate();
      if(st.st_mode & S_IFLNK)        //Symbolic link
      {
              linkcoun++;
              breakdown_link(d_name,lbuf1,lbuf2);        //breakdown  lbuf1 -> lbuf2
              ol=new_alloc(S_FList);
              ol->init(lbuf1, st.st_mode, st.st_size, st.st_uid, st.st_gid, st.st_mtime);        //init link
              n=new Namer(lbuf2,ol);        //and remember pointer for post detection.
              n->next=namer;
              namer=n;
      } else
      {
              ol=new_alloc(S_FList);
              ol->init(d_name, st.st_mode, st.st_size, st.st_uid, st.st_gid, st.st_mtime);
      }
      if (st.st_mode & S_IFDIR)
        dl.add(ol);
      else
        {
          fl.add(ol);
          if (ext_find(ol->name) != -1)
            ol->mode |= S_EXTEND;
          else if (is_it_afs_file(ol->name))
            ol->mode |= S_EXTEND;
        }
    }
  fclose(d);


  if(linkcoun)
  {                        //Doing post action linking
        n=namer;
        while(n)
        {
              ol=new_alloc(S_FList);
              if(flist_find_by_name(dl.next,n->name,&st) || flist_find_by_name(fl.next,n->name,&st))
                      ol->init(n->name, st.st_mode, st.st_size, st.st_uid, st.st_gid, st.st_mtime);
              else
                      ol->init(n->name,0755,0,0,0,st.st_mtime);
              n->o->link=ol;
              ol=n->o;
              if((ol->mode & S_IFDIR)!=S_IFDIR && (st.st_mode & S_IFDIR)==S_IFDIR)
              {        //We are in wrong list. We must be inside dl but now we in fl.
                      ol->mode|=S_IFDIR;
                      if(ol->prev)
                              ol->prev->next=ol->next;
                      if(ol->next)
                              ol->next->prev=ol->prev;
                      ol->next=ol->prev=NULL;
                      dl.add(ol);
              } else
                      if((ol->mode & S_IFDIR)==S_IFDIR && (st.st_mode & S_IFDIR)!=S_IFDIR)
                      {        //We are in wrong list. We must be inside fl but now we in dl.
                              ol->mode&=~S_IFDIR;
                              if(ol->prev)
                                      ol->prev->next=ol->next;
                              if(ol->next)
                                      ol->next->prev=ol->prev;
                              ol->next=ol->prev=NULL;
                              fl.add(ol);
                      }              
              ol->mode=(ol->mode & ~0777) | (st.st_mode & 0777);
              ol->size=st.st_size;
              n=n->next;
              delete namer;
              namer=n;
        }
  }
  
  bmark->animate();
  if(hd==NULL)
  {
              hd = new_alloc(S_FList);
              hd->init("..", S_IFDIR | 0755, 0, st.st_uid, st.st_gid, st.st_mtime);
  }
  simple_flist_add(&dl, hd);
  ol = new_alloc(S_FList);
  ol->init(".", S_IFDIR | 0755, 0, st.st_uid, st.st_gid, st.st_mtime);
  simple_flist_add(&dl, ol);

/*
   ol=dl.next;
   while(ol)
   {
   fprintf(stderr,"Dir: %s\n",ol->name);
   ol=ol->next;
   }
   fprintf(stderr,"-----------------------------------\n");
 */

  return 1;
}

void FTP::make_empty_list()
{
  FList *ol;
  delete_vfs_list();
  ol=new_alloc(S_FList);
  dl.next = fl.next = NULL;
  ol->init("..", S_IFDIR, 0, 0, 0, 5000);
  ol->next=NULL;
  simple_flist_add(&dl, ol);
  panel->dl.next=panel->base=panel->cur=dl.next;
  panel->curn=0;
}
