/* File: parser.c
 *
 *	Parse the toolbar resource file, producing the Motif
 *	toolbar.
 */

#include <stdio.h>
#include <stdlib.h>

#include <Xm/Xm.h>
#include <Xm/RowColumn.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/ArrowB.h>
#include <Xm/Separator.h>
#include <Xm/MwmUtil.h>
#include <xpm.h>

#include "parser.h"

#include "construction.xpm"
#include "resize.xpm"


Toolbar * CreateToolbar(Widget parent, char *file_name, unsigned char orientation)
{
	Toolbar		*user_tools;
	FILE		*fp;

	/*** open the resource file ***/
	fp = fopen(file_name, "r");
	if (!fp) {
		fprintf(stderr, "ERROR opening file \"%s\".\n", file_name);
		return NULL;
	}

	/*** allocate new structure ***/
	user_tools = malloc(sizeof(Toolbar));
	if (!user_tools) {
		fprintf(stderr, "ERROR allocating memory.\n");
		return NULL;
	}
	user_tools->CommandCount = 0;
	user_tools->Pixmaps = NULL;
	user_tools->file_name = malloc(strlen(file_name) + 1);
	strcpy(user_tools->file_name, file_name);
	user_tools->parent = parent;

	BuildToolbar(user_tools, fp, orientation);

	/*** close the resource file ***/
	fclose(fp);

	return user_tools;
}

char BuildToolbar(Toolbar *user_tools, FILE *fp, unsigned char orientation)
{
	Widget		separator;
	struct PixList	*pix;

	/*** create shell and manager ***/
	user_tools->topShell = XtVaCreatePopupShell("topShell",
						    topLevelShellWidgetClass,
						    user_tools->parent,
						    XmNoverrideRedirect, True,
						    /*XmNmwmFunctions, MWM_FUNC_MOVE,
						    XmNmwmDecorations, MWM_DECOR_BORDER,*/
						    XmNx, 0,
						    XmNy, 0,
						    XmNallowShellResize, True,
						    NULL);

	user_tools->manager = XtVaCreateManagedWidget("manager",
						      xmRowColumnWidgetClass,
						      user_tools->topShell,
						      XmNorientation, orientation,
						      XmNentryAlignment, XmALIGNMENT_CENTER,
						      XmNadjustMargin, False,
						      XmNspacing, 0,
						      XmNpacking, XmPACK_TIGHT,
						      NULL);

	/*** create default pixmap ***/
	user_tools->Pixmaps = malloc(sizeof(struct PixList));
	pix = user_tools->Pixmaps;
	pix->Next = NULL;
	pix->Attr.valuemask = XpmReturnPixels;
	if (XpmCreateImageFromData(XtDisplay(user_tools->topShell),
				   construction_xpm, &(pix->Image), NULL,
				   &(pix->Attr)) != XpmSuccess) {
		fprintf(stderr, "ERROR creating default image.\n");
		return False;
	}
	XmInstallImage(pix->Image, "default_xpm");
	pix->Pix = XmGetPixmap(XtScreen(user_tools->topShell), "default_xpm", 0, 1);

	/*** create resize button pixmap ***/
	pix->Next = malloc(sizeof(struct PixList));
	pix = pix->Next;
	pix->Next = NULL;
	pix->Attr.valuemask = XpmReturnPixels;
	if (XpmCreateImageFromData(XtDisplay(user_tools->topShell),
				   resize_xpm, &(pix->Image), NULL,
				   &(pix->Attr)) != XpmSuccess) {
		fprintf(stderr, "ERROR creating close button image.\n");
		return False;
	}
	XmInstallImage(pix->Image, "resize_xpm");
	pix->Pix = XmGetPixmap(XtScreen(user_tools->topShell), "resize_xpm", 0, 1);

	/*** create user buttons from resource file ***/
	ParseFile(user_tools, fp, orientation);

	/*** create restart & quit buttons with separator ***/
	separator = XtVaCreateManagedWidget("bottomSeparator",
					    xmSeparatorWidgetClass,
					    user_tools->manager,
					    XmNorientation, (orientation == XmVERTICAL ? XmHORIZONTAL : XmVERTICAL),
					    NULL);

	user_tools->restartButton = XtVaCreateManagedWidget("restartButton",
							    xmPushButtonWidgetClass,
							    user_tools->manager,
							    XmNlabelString, XmStringCreateSimple("Restart"),
							    NULL);
	XtAddCallback(user_tools->restartButton, XmNactivateCallback,
		      (XtCallbackProc)Toolbar_restartButton, (XtPointer)user_tools);
	user_tools->quitButton = XtVaCreateManagedWidget("quitButton",
							 xmPushButtonWidgetClass,
							 user_tools->manager,
							 XmNlabelString, XmStringCreateSimple("Quit"),
							 NULL);
	XtAddCallback(user_tools->quitButton, XmNactivateCallback,
		      (XtCallbackProc)Toolbar_quitButton, (XtPointer)user_tools);

	/*** display the toolbar ***/
	XtPopup(user_tools->topShell, XtGrabNone);

	return True;
}

void DeleteToolbar(Toolbar *tools)
{
	int		i;
	struct PixList	*pix;

	/*** free the list of commands ***/
	for (i = 0; i < tools->CommandCount; i++)
		free(tools->Command[i]);
	tools->CommandCount = 0;

	/*** free the pixmaps and pixmap data ***/
	while (tools->Pixmaps) {
		pix = tools->Pixmaps;
		tools->Pixmaps = tools->Pixmaps->Next;
		XFreeColors(XtDisplay(tools->topShell),
			    DefaultColormapOfScreen(XtScreen(tools->topShell)),
			    pix->Attr.pixels, pix->Attr.npixels, 0);
		if (pix->Pix) XmDestroyPixmap(XtScreen(tools->topShell), pix->Pix);
		if (pix->Image) XmUninstallImage(pix->Image);
		free(pix);
	}

	/*** free the widgets ***/
	XtDestroyWidget(tools->topShell);
}

char ParseFile(Toolbar *tools, FILE *fp, unsigned char orientation)
{
	TokenType	token;
	char		title[256], button[256], command[1024];
	char		return_val = True;
	Widget		userButton, userTitle, separator, parent, popup, manager, form, iconify, resize;
	int		buttonNumber = 0, sepNumber = 0, popupNumber = 0;
	char		skipTitle = False;
	char		image_path[256], image_file[512];
	Pixmap		pix;
	unsigned char	flipped = (orientation == XmVERTICAL ? XmHORIZONTAL : XmVERTICAL);
	unsigned char	arrow_dir = (orientation == XmVERTICAL ? XmARROW_LEFT : XmARROW_UP);

	/*** initial image path, in case it is not specified ***/
	strcpy(image_path, "./");

	/*** initial parent, changed by popup windows ***/
	parent = tools->manager;

	do {
		token = GetToken(fp);

		switch(token) {
		case tkTitle:
			/*** get title of toolbar - must be first token in file ***/
			if (skipTitle) break;
			fgetstring(fp, title);
			form = XtVaCreateManagedWidget("titleForm",
						       xmFormWidgetClass,
						       tools->manager,
						       NULL);
			userTitle = XtVaCreateManagedWidget("userTitle",
							    xmLabelWidgetClass,
							    form,
							    XmNlabelString, XmStringCreateSimple(title),
							    XmNtopAttachment, XmATTACH_FORM,
							    XmNbottomAttachment, XmATTACH_FORM,
							    XmNleftAttachment, XmATTACH_FORM,
							    NULL);
			resize = XtVaCreateManagedWidget("resizeButton",
							 xmPushButtonWidgetClass,
							 form,
							 XmNlabelType, XmPIXMAP,
							 XmNlabelPixmap, tools->Pixmaps->Next->Pix,
							 XmNbottomAttachment, XmATTACH_FORM,
							 XmNrightAttachment, XmATTACH_FORM,
							 XmNleftAttachment, XmATTACH_WIDGET,
							 XmNleftWidget, userTitle,
							 NULL);
			XtAddCallback(resize, XmNactivateCallback,
				      (XtCallbackProc)Toolbar_resizeButton, (XtPointer)tools);
			iconify = XtVaCreateManagedWidget("iconifyButton",
							  xmArrowButtonWidgetClass,
							  form,
							  XmNtopAttachment, XmATTACH_FORM,
							  XmNbottomAttachment, XmATTACH_WIDGET,
							  XmNbottomWidget, resize,
							  XmNleftAttachment, XmATTACH_WIDGET,
							  XmNleftWidget, userTitle,
							  XmNrightAttachment, XmATTACH_FORM,
							  NULL);
			XtAddCallback(iconify, XmNactivateCallback,
				      (XtCallbackProc)Toolbar_iconifyButton, (XtPointer)tools);
			separator = XtVaCreateManagedWidget("titleSeparator",
							    xmSeparatorWidgetClass,
							    tools->manager,
							    XmNorientation, flipped,
							    NULL);
			/*** create icon window ***/
			tools->iconShell = XtVaCreatePopupShell("iconShell",
								topLevelShellWidgetClass,
								XtParent(tools->topShell),
								NULL);
			iconify = XtVaCreateManagedWidget("deiconifyButton",
							  xmPushButtonWidgetClass,
							  tools->iconShell,
							  XmNlabelString, XmStringCreateSimple(title),
							  NULL);
			XtAddCallback(iconify, XmNactivateCallback,
				      (XtCallbackProc)Toolbar_deiconifyButton, (XtPointer)tools);
			break;
		case tkButton:
			/*** create a standard text button ***/
			fgetstring(fp, button);
			fgetstring(fp, command);
			/*** add the command to the permanent list ***/
			tools->Command[tools->CommandCount] = malloc(strlen(command) + 1);
			strcpy(tools->Command[tools->CommandCount], command);
			/*** unique widget name ***/
			sprintf(title, "userButton%02d", buttonNumber++);
			/*** create the button ***/
			userButton = XtVaCreateManagedWidget(title,
							     xmPushButtonWidgetClass,
							     parent,
							     XmNlabelString, XmStringCreateSimple(button),
							     XmNuserData, tools->CommandCount,
							     NULL);
			XtAddCallback(userButton, XmNactivateCallback,
				      (XtCallbackProc)Toolbar_userButton, (XtPointer)tools);
			++(tools->CommandCount);
			break;
		case tkSeparator:
			/*** create a separator line ***/
			sprintf(title, "userSeparator%02d", sepNumber++);
			separator = XtVaCreateManagedWidget(title,
							    xmSeparatorWidgetClass,
							    parent,
							    XmNorientation, flipped,
							    NULL);
			break;
		case tkIPath:
			/*** change the path used to find images ***/
			fgetstring(fp, image_path);
			strcat(image_path, "/");
			break;
		case tkImage:
			/*** create a button containing an image ***/
			fgetstring(fp, button);
			fgetstring(fp, command);
			strcpy(image_file, image_path);
			strcat(image_file, button);
			/*** load XPM file to Pixmap ***/
			pix = LoadImage(tools, image_file, button);
			/*** add the command to the permanent list ***/
			tools->Command[tools->CommandCount] = malloc(strlen(command) + 1);
			strcpy(tools->Command[tools->CommandCount], command);
			/*** unique widget name ***/
			sprintf(title, "userButton%02d", buttonNumber++);
			/*** create the button ***/
			userButton = XtVaCreateManagedWidget(title,
							     xmPushButtonWidgetClass,
							     parent,
							     XmNuserData, tools->CommandCount,
							     XmNlabelType, XmPIXMAP,
							     NULL);
			if (pix) XtVaSetValues(userButton, XmNlabelPixmap, pix, NULL);
			XtAddCallback(userButton, XmNactivateCallback,
				      (XtCallbackProc)Toolbar_userButton, (XtPointer)tools);
			++(tools->CommandCount);
			break;
		case tkPopup:
		case tkPopupImage:
			if (parent != tools->manager) {
				fprintf(stderr, "ERROR!  Popup within a popup is not allowed.\n");
				token = tkError;
				break;
			}
			/*** create the popup window ***/
			sprintf(title, "popup%02d", popupNumber);
			popup = XtVaCreatePopupShell(title,
						     topLevelShellWidgetClass,
						     parent,
						     XmNallowShellResize, True,
						     /*XmNmwmFunctions, MWM_FUNC_MOVE,
						     XmNmwmDecorations, 0,*/
						     XmNoverrideRedirect, True,
						     NULL);
			sprintf(title, "popupManager%02d", popupNumber);
			manager = XtVaCreateManagedWidget(title,
							  xmRowColumnWidgetClass,
							  popup,
							  XmNorientation, flipped,
							  XmNentryAlignment, XmALIGNMENT_CENTER,
							  NULL);
			/*** create a button to popup a sub-window ***/
			fgetstring(fp, button);
			sprintf(title, "popupButton%02d", popupNumber);
			if (token == tkPopup) {
				userButton = XtVaCreateManagedWidget(title,
								     xmPushButtonWidgetClass,
								     parent,
								     XmNlabelString, XmStringCreateSimple(button),
								     XmNuserData, popup,
								     XmNshadowThickness, 0,
								     NULL);
			}
			else {
				strcpy(image_file, image_path);
				strcat(image_file, button);
				/*** load XPM file to Pixmap ***/
				pix = LoadImage(tools, image_file, button);
				/*** create button ***/
				userButton = XtVaCreateManagedWidget(title,
								     xmPushButtonWidgetClass,
								     parent,
								     XmNlabelType, XmPIXMAP,
								     XmNuserData, popup,
								     XmNshadowThickness, 0,
								     NULL);
				if (pix) XtVaSetValues(userButton, XmNlabelPixmap, pix, NULL);
			}
			XtAddCallback(userButton, XmNactivateCallback,
				      (XtCallbackProc)Toolbar_userPopup, (XtPointer)tools);
			parent = manager;
			/*** add close button ***/
			sprintf(title, "closeButton%02d", popupNumber++);
			userButton = XtVaCreateManagedWidget(title,
							     xmArrowButtonWidgetClass,
							     parent,
							     XmNarrowDirection, arrow_dir,
							     NULL);
			XtAddCallback(userButton, XmNactivateCallback,
				      (XtCallbackProc)Toolbar_closePopup, (XtPointer)popup);
			break;
		case tkEndPopup:
			parent = tools->manager;
			break;
		case tkError:
			fprintf(stderr, "ERROR parsing \".toolbarrc\".\n");
			return_val = False;
			break;
		}
		/*** title must be first in file, else it is skipped ***/
		skipTitle = True;
	} while ((token != tkEOF) && (token != tkError));

	return return_val;
}

TokenType GetToken(FILE *fp)
{
	char	string[32], c;
	int	len;

	/*** next word in file should be a recognizable token ***/
	/*** or a comment beginning with '#' ***/
	len = fscanf(fp, "%s", string);

	if (len == EOF)
		return tkEOF;
	else if (!strcmp(string, "TITLE"))
		return tkTitle;
	else if (!strcmp(string, "BUTTON"))
		return tkButton;
	else if (!strcmp(string, "SEPARATOR"))
		return tkSeparator;
	else if (!strcmp(string, "IPATH"))
		return tkIPath;
	else if (!strcmp(string, "IMAGE"))
		return tkImage;
	else if (!strcmp(string, "POPUP"))
		return tkPopup;
	else if (!strcmp(string, "POPUP_IMAGE"))
		return tkPopupImage;
	else if (!strcmp(string, "END_POPUP"))
		return tkEndPopup;
	else if (string[0] == '#') {
		/*** comment goes to end of line ***/
		do {
			c = fgetc(fp);
		} while (c != EOF && c != '\n');
		if (c == EOF)
			return tkEOF;
		else
			return GetToken(fp);
	}
	else {
		printf("ERROR!  Unknown token \"%s\".\n", string);
		return tkError;
	}
}

char * fgetstring(FILE *fp, char *s)
{
	char	c, *return_val, escape;

	return_val = s;

	/*** skip blank space ***/
	do {
		c = fgetc(fp);
		if (c == EOF) return NULL;
	} while (c == ' ' || c == '\n' || c== '\t');

	/*** copy quoted string ***/
	escape = False;
	if (c == '\"') {
		c = fgetc(fp);
		while ((c != '\"' || escape) && c != EOF) {
			if (c == '\\' && !escape)
				escape = True;
			else {
				*s++ = c;
				escape = False;
			}
			c = fgetc(fp);
		}
	}
	/*** copy until blank space ***/
	else {
		do {
			*s++ = c;
			c = fgetc(fp);
		} while (c != ' ' && c != '\n' && c != '\t' && c != EOF);
	}
	if (c == EOF) return NULL;

	*s = '\0';
	return return_val;
}

void Toolbar_restartButton(Widget widget, Toolbar *tools)
{
	int	pid;

	/*** start new toolbar as separate process ***/
	pid = fork();
	if (pid == 0) {
		DeleteToolbar(tools);
		execlp("toolbar", "toolbar", NULL);
		perror("ERROR restarting \"toolbar\"");
	}

	/*** end this process ***/
	DeleteToolbar(tools);

	exit(0);
}

void Toolbar_quitButton(Widget widget, Toolbar *tools)
{
	/*** free memory, colors, etc. ***/
	DeleteToolbar(tools);

	free(tools->file_name);
	free(tools);

	exit(0);
}

void Toolbar_userButton(Widget widget, Toolbar *tools)
{
	int		command, i = 0, pid;
	char		string[1024], *s, in_quotes = False;
	static char	*args[256];

	XtVaGetValues(widget, XmNuserData, &command, NULL);
	strncpy(string, tools->Command[command], 1024);

	/*** build argument list ***/
	s = strtok(string, " ");
	args[i++] = s;
	while (s) {
		/*** check for beginning of quoted string ***/
		if (s[0] == '\"') {
			in_quotes = True;
			++(args[i-1]);	/* skip leading quotes */
		}
		/*** check for next word in quoted string, or end of string ***/
		if (in_quotes) {
			if (s[strlen(s) - 1] != '\"')
				s[strlen(s)] = ' ';
			else {
				in_quotes = False;
				s[strlen(s) - 1] = '\0';	/* skip trailing quotes */
			}
		}
		/*** get next word ***/
		s = strtok(NULL, " ");
		/*** add word to argument list ***/
		if (!in_quotes) args[i++] = s;
	}

	/*** start new process and execute command ***/
	pid = fork();
	if (pid == 0) {
		DeleteToolbar(tools);
		execvp(args[0], args);
		perror("ERROR executing command!");
	}
}

void Toolbar_userPopup(Widget widget, Toolbar *tools)
{
	Dimension	x, y;
	Widget		popup;
	unsigned char	orientation;

	XtVaGetValues(tools->manager, XmNorientation, &orientation, NULL);

	if (orientation == XmVERTICAL) {
		XtVaGetValues(widget, XmNy, &y, XmNuserData, &popup, NULL);
		XtVaGetValues(tools->topShell, XmNwidth, &x, NULL);
	}
	else {
		XtVaGetValues(widget, XmNx, &x, XmNuserData, &popup, NULL);
		XtVaGetValues(tools->topShell, XmNheight, &y, NULL);
	}

	XtVaSetValues(popup, XmNx, x, XmNy, y, NULL);
	XtPopup(popup, XtGrabNone);
}

void Toolbar_closePopup(Widget widget, Widget popup)
{
	XtPopdown(popup);
}

void Toolbar_iconifyButton(Widget widget, Toolbar *tools)
{
	XtPopdown(tools->topShell);
	XtPopup(tools->iconShell, XtGrabNone);
}

void Toolbar_deiconifyButton(Widget widget, Toolbar *tools)
{
	XtPopdown(tools->iconShell);
	XtPopup(tools->topShell, XtGrabNone);
}


Pixmap LoadImage(Toolbar *tools, char *file_name, char *image_name)
{
	struct PixList	*pix;

	/*** create new struct for image data ***/
	pix = tools->Pixmaps;
	while (pix->Next) pix = pix->Next;
	pix->Next = malloc(sizeof(struct PixList));
	pix = pix->Next;
	pix->Next = NULL;

	/*** create pixmap from XPM file ***/
	pix->Attr.valuemask = XpmReturnPixels;
	if (XpmReadFileToImage(XtDisplay(tools->topShell),
			       file_name, &(pix->Image), NULL,
			       &(pix->Attr)) != XpmSuccess) {
		fprintf(stderr, "ERROR reading image from \"%s\".\n", file_name);
		pix->Pix = XmGetPixmap(XtScreen(tools->topShell), "default_xpm", 0, 1);
	}
	else {
		XmInstallImage(pix->Image, image_name);
		pix->Pix = XmGetPixmap(XtScreen(tools->topShell),
				       image_name, 0, 1);
	}

	return pix->Pix;
}

void Toolbar_resizeButton(Widget widget, Toolbar *tools)
{
	unsigned char	orientation;
	FILE		*fp;

	/*** open the resource file ***/
	fp = fopen(tools->file_name, "r");
	if (!fp) {
		fprintf(stderr, "ERROR opening file \"%s\".\n", tools->file_name);
		return;
	}

	XtVaGetValues(tools->manager, XmNorientation, &orientation, NULL);
	DeleteToolbar(tools);
	BuildToolbar(tools, fp, (orientation == XmVERTICAL ? XmHORIZONTAL : XmVERTICAL));

	/*** close the resource file ***/
	fclose(fp);
}
