/*
 * Freedom Desktop
 * Copyright 1994 by Freedom Software
 *
 * Freedom Software retains all rights to Freedom Desktop (hereafter Software)
 * in binary and in source code form.
 *
 * The commercial use of this Software shall be governed by a separate License
 * agreement. Any individual or institution wishing to make commercial use of
 * the Software must sign a license agreement with Freedom Software. In such
 * cases, the Licensee agrees to abide by the terms contained in the License
 * Agreement and not those contained in this document. Examples of commercial
 * use include (without limitation): (i) integration of the Software (source
 * code form), in whole or in part, into a commercial product sold by or on
 * on behalf of the Licensee; (ii) distribution of the Software (binary form or
 * source code form) in combination with a commercial product sold by or on
 * behalf of the Licensee.
 *
 * Freedom Software (Licensor) grants you (Licensee) a license: (i) to use,
 * copy and make changes and improvements to this Software for licensee's
 * internal business purposes; (ii) to use, copy, and distribute this Software
 * or the derivative works provided that the copyright notice and this
 * permission notice appear on all copies and that NO CHARGE is associated
 * with such copies. However, if Licensee distributes any derivative work
 * based on the Software, then Licensee shall (i) notify Licensor in writing
 * (ii) clearly state that such derivative work is a modified and not the
 * original Freedom Desktop distributed by Freedom Software (iii) publish
 * the corresponding machine-readable source code or information as to
 * where it may be obtained. Each time Licensee redistribute the Software
 * or any derivative work, the recipient automatically agrees to abide
 * by the same terms as the Licensee. Licensee may not impose terms
 * more restrictive than the terms granted herein.
 *
 * By using, copying, modifying or distributing this Software (or any
 * derivative work based on this Software) Licensee indicates acceptance
 * of the terms and conditions set forth in this License.
 *
 * Licensor reserves the right to terminate this License immediately on written
 * notice, for material breach by the Licensee.
 *
 * FREEDOM SOFTWARE DISCLAIMS ALL WARRANTIES EXPRESS OR IMPLIED WITH REGARD
 * TO THIS SOFTWARE INCLUDING BUT NOT LIMITED TO ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND  FITNESS,  IN  NO  EVENT  SHALL LICENSOR BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE
 */

#ifndef List_h
#define List_h

#include <stdio.h>
#include <stdlib.h>


/*
 * Simple List Element
 */

typedef struct _elem {
   struct _elem *elm_nxt, *elm_prv; /* Next and previous Elements */
   int elm_stat;		    /* Element Status		  */
} _ELEM;

/*
 * Simple List
 */

typedef struct SimpleList {
   _ELEM *lst_first;    /* First Element        */
   _ELEM *lst_curr;     /* Current Element      */
   _ELEM *lst_last;     /* Last Element         */
   size_t lst_cnt;      /* Number of slots occupied for the list */
                        /* of elements                           */
   int  lst_stat;	/* Status		*/
} _SimpleList;



/*
 * Simple List definitions
 */

#define  _ELM_DELETED		1
#define _SLSTSIZE        	(sizeof(_SimpleList))
#define _ELMSIZE        	(sizeof(_ELEM))
#define _LST_BEGINNING		1	/* Look from the beginning */
#define _LST_CURRENT_POS	2	/* Look from the current position */

#define _InvalidArg(s)  ((fprintf(stderr, s)?NULL:NULL)) /* This returns
 NULL */

extern size_t hdrsize;

/*
 * Next and previous and current elements of the list
 */

#define _lst_next(lst) ((lst)?((lst)->lst_curr = ((lst)->lst_curr?((lst)->lst_curr->elm_nxt):NULL)):(_ELEM *)_InvalidArg("_lst_next:NULL argument\n"))

#define _lst_prev(lst) ((lst)?((lst)->lst_curr = ((lst)->lst_curr?((lst)->lst_curr->elm_prv):NULL)):(_ELEM *)_InvalidArg("_lst_prev:NULL argument\n"))

#define _lst_curr(lst) ((lst)?(lst)->lst_curr:(_ELEM *)_InvalidArg("_lst_prev:NULL argument\n"))


/*
 * First and last elements of the list
 */

#define _lst_first(lst) ((lst)?((lst)->lst_curr=(lst)->lst_first):(_ELEM *)_InvalidArg("_lst_first:NULL argument\n"))

#define _lst_last(lst) ((lst)?((lst)->lst_curr=(lst)->lst_last):(_ELEM *)_InvalidArg("_lst_last:NULL argument\n"))

/*
 * Other list operations
 */

#define _lst_length(lst)        ((lst)?(lst)->lst_cnt:!fprintf(stderr, "_lst_length: Invalid argument\n"))
#define _lst_empty(lst)        ((lst)?((lst)->lst_cnt==0):fprintf(stderr, "_lst_empty: Invalid argument\n"))

/*
 * Set Current position
 */

#define _lst_set_curr(lst,elm) ((lst)?(lst)->lst_curr=(elm):(_ELEM *)_InvalidArg("_lst_first:NULL argument\n"))
#define _lst_get_curr(lst) ((lst)?(lst)->lst_curr:(_ELEM *)_InvalidArg("_lst_first:NULL argument\n"))


/*
 * _lst_reset: Reset list pointers
 * 
 * Does not free list elements
 */

#define _lst_reset(lst) { \
	if (lst) {\
        lst->lst_curr = lst->lst_last = lst->lst_first = NULL; \
        lst->lst_stat = 0; \
        lst->lst_cnt = 0; \
	} else _InvalidArg("_lst_reset: Invalid argument\n");\
}


/*
 * Append to the end of the list
 */

#define _lst_append(lst,elm) { \
   if ((lst) && (elm)) { \
   	(elm)->elm_stat = 0; \
   	if ((lst)->lst_last == NULL) {\
           (lst)->lst_curr = (lst)->lst_first = (lst)->lst_last = (elm); \
           (elm)->elm_prv = (elm)->elm_nxt = NULL; \
   	} else { \
           (lst)->lst_curr = (lst)->lst_last->elm_nxt = (elm); \
           (elm)->elm_nxt = NULL; \
           (elm)->elm_prv = (lst)->lst_last; \
           (lst)->lst_last = (elm); \
   	} \
   	lst->lst_cnt++; \
   } else _InvalidArg("_lst_append: Invalid arguments\n"); \
}

/*
 * Unlink element and mark it as deleted (_ELM_DELETED)
 * The next element of the list (or NULL) becomes the current 
 * 
 */

#define _lst_delete(lst,elm) { \
   if ((lst) && (elm)) { \
   	if ((lst)->lst_first == (lst)->lst_last) \
           (lst)->lst_first = (lst)->lst_curr = (lst)->lst_last = NULL; \
   	else \
           if ((elm) == (lst)->lst_first) { \
           	(lst)->lst_curr = (lst)->lst_first = (elm)->elm_nxt; \
           	(lst)->lst_first->elm_prv = NULL; \
           } else \
           	if ((elm) == (lst)->lst_last) { \
                   (lst)->lst_last = (elm)->elm_prv; \
                   (lst)->lst_curr = (lst)->lst_last->elm_nxt = NULL; \
           	} else { \
                   (lst)->lst_curr = (elm)->elm_prv->elm_nxt = (elm)->elm_nxt; \
                   (elm)->elm_nxt->elm_prv = (elm)->elm_prv; \
           	} \
   	lst->lst_cnt--; \
        (elm)->elm_prv = (elm)->elm_nxt = NULL; /* just in case */ \
	free(elm); \
	(elm) = NULL;  /* just in case */ \
   }  else _InvalidArg("_lst_delete: Invalid argument\n"); \
}

/*
 * Disconnect element from the list
 * The next element of the list (or NULL) becomes the current 
 */

#define _lst_unlink(lst,elm) { \
   if ((lst) && (elm)) { \
      (elm)->elm_stat = _ELM_DELETED; \
      if ((lst)->lst_first == (lst)->lst_last) \
        (lst)->lst_first = (lst)->lst_curr = (lst)->lst_last = NULL; \
      else \
      	if ((elm) == (lst)->lst_first) { \
           (lst)->lst_curr = (lst)->lst_first = (elm)->elm_nxt; \
           (lst)->lst_first->elm_prv = NULL; \
        } else \
           if ((elm) == (lst)->lst_last) { \
                (lst)->lst_last = (elm)->elm_prv; \
                (lst)->lst_curr = (lst)->lst_last->elm_nxt = NULL; \
           } else { \
                (lst)->lst_curr = (elm)->elm_prv->elm_nxt = (elm)->elm_nxt; \
                (elm)->elm_nxt->elm_prv = (elm)->elm_prv; \
           } \
      (elm)->elm_prv = (elm)->elm_nxt = NULL; /* just in case */ \
      lst->lst_cnt--; \
  } else _InvalidArg("_lst_unlink: Invalid argument\n"); \
}

/*
 * Insert element 
 * Bug - It did not consider deleted elements (Fixed 1/17/93) ??
 * Bug - _ELMSIZE should not be used here
 */

#define _lst_insert_o(lst,elm,cmp) { \
 if ((lst) && (elm) && (cmp)) { \
   _ELEM *elm1; \
   (elm)->elm_stat = 0; \
   if ((lst)->lst_last == NULL) {\
        (lst)->lst_curr = (lst)->lst_first = (lst)->lst_last = (elm); \
        (elm)->elm_nxt = (elm)->elm_prv = NULL; \
   } else { \
	elm1 = _lst_first(lst); \
	while (elm1 && cmp ((char *) (elm), (char *) (elm1)) >= 0) \
		elm1 = _lst_next(lst); \
	if (!elm1) { \
        	(lst)->lst_curr = (lst)->lst_last->elm_nxt = (elm); \
                (elm)->elm_nxt = NULL; \
        	(elm)->elm_prv = (lst)->lst_last; \
        	(lst)->lst_last = (elm); \
	} else { \
	   if (elm1 == (lst)->lst_first) { \
		(lst)->lst_first = (elm); \
		(elm)->elm_prv = NULL; \
	   } else { \
		elm1->elm_prv->elm_nxt = elm; \
		(elm)->elm_prv = elm1->elm_prv; \
	   } \
		(elm)->elm_nxt = elm1; \
		(lst)->lst_curr = elm1->elm_prv = (elm); \
	} \
   } \
   lst->lst_cnt++; \
 } else _InvalidArg("_lst_insert_o: Invalid argument\n"); \
}


/*
 * Simple List macros (user interface to the simple list functions)
 */

#define _SListAppend(lst,elm) _lst_append(lst, (_ELEM *) elm)
#define _SListInsertO(lst,elm,cmp) _lst_insert_o(lst, (_ELEM *) elm,cmp)
#define _SListSetCurr(lst,elm) _lst_set_curr(lst, (_ELEM *) elm)
#define _SListGetCurr(lst) _lst_get_curr(lst)
#define _SListFirst(lst) _lst_first(lst)
#define _SListLast(lst) _lst_last(lst)
#define _SListNext(lst) _lst_next(lst)
#define _SListPrev(lst) _lst_prev(lst)
#define _SListDelete(lst,elm) _lst_delete(lst,(_ELEM *) elm)
#define _SListUnlink(lst,elm) _lst_unlink(lst,(_ELEM *) elm)
#define _SListLength(lst) _lst_length(lst)
#define _SListEmpty(lst) _lst_empty(lst)
#define _SListReset(lst) _lst_reset(lst)


/*
 * Simple List Operations (user interface)
 */

extern _SimpleList *_SListCreate ();
extern int _SListDestroy ();
extern int _SListTraverse ();
extern char *_SListFind ();
extern char *_SListSeek ();
extern int _SListInvert ();
extern void _SListClear ();


/*
 * List structure (use Simple List)
 */


typedef struct _lst {
   _ELEM *lst_first;	/* First Element	*/
   _ELEM *lst_curr;	/* Current Element	*/
   _ELEM *lst_last;	/* Last Element		*/
   size_t lst_cnt;    	/* Number of slots occupied for the list */
			/* of elements				 */

   char *lst_arr;	/* Array of free slots	*/
   int  lst_stat;	/* Status		*/

   size_t lst_n_org;

   size_t lst_free;	/* Index of the next free slot */
   size_t lst_n_ovf;	/* Number of free slots - overflow block */
   size_t lst_n_elm;	/* Total # of elements - array of free slots */
   size_t lst_elm_siz;  /* Slot size in bytes */
} _LST;


/*
 * List Operations
 */

extern char 	*_ListAppend ();
extern _LST 	*_ListCreate ();
extern char  	*_ListDelete ();
extern char 	*_ListFind ();
extern unsigned int _ListGetPost ();
extern char 	*_ListInsertO ();
extern char  	*_ListReinsert ();
extern char 	*_ListSeek ();
extern int  	_ListSort ();
extern int  	_ListTraverse ();
extern int  	_ListTrvSeq ();	/* Fast Traverse - use the array directly */
extern char 	*_ListUnlink ();



/*
 * List definitions
 */

#define _LSTSIZE        (sizeof(_LST))

/*
 * List macros
 */

/*
 * Next and previous
 */

#define _ListNext(lst) ((lst)?((lst)->lst_curr?(_lst_next(lst)?(char*)(lst)->lst_curr+hdrsize:NULL):NULL):(char *) _InvalidArg("_ListNext:NULL argument\n"))

#define _ListPrev(lst) ((lst)?((lst)->lst_curr?(_lst_prev(lst)?(char*)(lst)->lst_curr+hdrsize:NULL):NULL):(char *) _InvalidArg("_ListPrev:NULL argument\n"))

/*
 * First and last
 */

#define _ListFirst(lst) ((lst)?(_lst_first(lst)?(char*)(lst)->lst_curr+hdrsize:NULL):(char *) _InvalidArg("_ListFirst:NULL argument\n"))
#define _ListLast(lst) ((lst)?(_lst_last(lst)?(char*)(lst)->lst_curr+hdrsize:NULL):(char *) _InvalidArg("_ListLast:NULL argument\n"))

/*
 * Other list operations
 */

#define _ListLength(lst)        _SListLength(lst)
#define _ListEmpty(lst)        	_SListEmpty(lst)
#define _ListInvert(lst)        _SListInvert(lst)

#define _ListArrayOffset(lst,item) ((char *)(item)-(lst)->lst_arr)

/*
#define _ListReinsertO(lst,elm,cmp) \
{ \
   char *elm1; \
   elm1 = (char *) (elm) - hdrsize; \
   _lst_unlink((lst),(_ELEM *)elm1); \
   _lst_insert_o ((lst),(_ELEM *) elm1, (cmp)); \
}
*/

/* Rearrange pointers */
#define _PointerOffset(p,old,new) \
((p)?((_ELEM *) ((new) + ((char *) (p) - (old)))):NULL)

#endif
