
/*
 * Freedom Desktop
 * Copyright 1994 by Freedom Software
 *
 * Freedom Software retains all rights to Freedom Desktop (hereafter Software)
 * in binary and in source code form.
 *
 * The commercial use of this Software shall be governed by a separate License
 * agreement. Any individual or institution wishing to make commercial use of
 * the Software must sign a license agreement with Freedom Software. In such
 * cases, the Licensee agrees to abide by the terms contained in the License
 * Agreement and not those contained in this document. Examples of commercial
 * use include (without limitation): (i) integration of the Software (source
 * code form), in whole or in part, into a commercial product sold by or on
 * on behalf of the Licensee; (ii) distribution of the Software (binary form or
 * source code form) in combination with a commercial product sold by or on
 * behalf of the Licensee.
 *
 * Freedom Software (Licensor) grants you (Licensee) a license: (i) to use,
 * copy and make changes and improvements to this Software for licensee's
 * internal business purposes; (ii) to use, copy, and distribute this Software
 * or the derivative works provided that the copyright notice and this
 * permission notice appear on all copies and that NO CHARGE is associated
 * with such copies. However, if Licensee distributes any derivative work
 * based on the Software, then Licensee shall (i) notify Licensor in writing
 * (ii) clearly state that such derivative work is a modified and not the
 * original Freedom Desktop distributed by Freedom Software (iii) publish
 * the corresponding machine-readable source code or information as to
 * where it may be obtained. Each time Licensee redistribute the Software
 * or any derivative work, the recipient automatically agrees to abide
 * by the same terms as the Licensee. Licensee may not impose terms
 * more restrictive than the terms granted herein.
 *
 * By using, copying, modifying or distributing this Software (or any
 * derivative work based on this Software) Licensee indicates acceptance
 * of the terms and conditions set forth in this License.
 *
 * Licensor reserves the right to terminate this License immediately on written
 * notice, for material breach by the Licensee.
 *
 * FREEDOM SOFTWARE DISCLAIMS ALL WARRANTIES EXPRESS OR IMPLIED WITH REGARD
 * TO THIS SOFTWARE INCLUDING BUT NOT LIMITED TO ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND  FITNESS,  IN  NO  EVENT  SHALL LICENSOR BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "lst.h"

/*
 * List manipulation functions
 */

size_t hdrsize = _ELMSIZE;
static int (*usrcmp) ();

/*
 * _cmp: Compare two elements in the list 
 */

static int _cmp (elm, elm1)
register char *elm, *elm1;
{
#ifdef DEBUG
   fprintf (stderr, "_cmp: Starting ...\n");
#endif
   if (!elm || !elm1) {
	_InvalidArg ("_cmp: Null arguments\n");
	return (elm == elm1);
   }

   return (usrcmp (elm + hdrsize, elm1 + hdrsize));
}

/*
 * _ecmp: Compare two elements in the list (used for ListSort only)
 *        Consider deleted elements and compact then toward the end
 */

static int _ecmp (e1, e2)
register _ELEM *e1, *e2;
{
   
   if (!e1 || !e2) {
	_InvalidArg ("_ecmp: Null arguments\n");
	return (e1 == e2);
   }

#ifdef DEBUG
   if ((e1->elm_stat & _ELM_DELETED) || (e2->elm_stat & _ELM_DELETED))
        fprintf (stderr, "-------------> Bug in _ecmp <-------------\n");
#endif

   if ((e1->elm_stat & _ELM_DELETED) && (e2->elm_stat & _ELM_DELETED))
        return (0);

   if (e1->elm_stat & _ELM_DELETED)
        return (1);

   if (e2->elm_stat & _ELM_DELETED)
        return (-1);
   
   return (usrcmp ((char *) e1 + hdrsize, (char *) e2 + hdrsize));
}

/*
 * _SListCreate: create a simple list
 */

_SimpleList *_SListCreate () {
   register _SimpleList *lst;
   static size_t lstsize = _SLSTSIZE;

	if ((lst = (_SimpleList *) malloc (lstsize)) == NULL) {
		fprintf (stderr, "_SListCreate: no enough memory\n");
		exit (1);
	}
	lst->lst_curr = lst->lst_last = lst->lst_first = NULL;
	lst->lst_stat = 0;
	lst->lst_cnt = 0;
	return (lst);
}


/*
 * _SListDestroy: destroy a simple list
 */

_SListDestroy (lst) 
register _SimpleList *lst;
{ 
   register _ELEM *elm, *aux;
	if (!lst) {
		_InvalidArg ("_ListDestroy: Null arguments\n");
		return; 
	}

        elm = _lst_first (lst);
        while (elm) {
                aux = elm;
                elm = _lst_next(lst);
		free ((void *) aux);
        }
	free ((void *) lst);
}

/*
 * _SListClear: Delete all the elements without deleting the header of
 *              the list
 */

void _SListClear (lst) 
register _SimpleList *lst;
{
   register _ELEM *elm, *aux;
        if (!lst) {
                _InvalidArg ("_ListClear: Null arguments\n");
                return;
        }

        elm = _lst_first (lst);
        while (elm) {
                aux = elm;
                elm = _lst_next(lst);
		_lst_delete(lst,aux);
        }
}


/*
 * _SListTraverse: traverse a simple list
 */

_SListTraverse (lst, func)
register _SimpleList *lst;
int (*func) ();
{
   register _ELEM *elm, *elm1;
   register _ELEM *tmp;

	if (!lst || !func) {
		_InvalidArg ("_SListTraverse: Null arguments\n");
		return; 
	}

	tmp = _lst_get_curr (lst);

        elm = _lst_first (lst);
        while (elm) {
                (*func) (elm);
                elm = _lst_next(lst);
        }
	_lst_set_curr (lst, tmp);
}

/*
 * _SListCheck: check a simple list
 */

_SListCheck (lst, func)
register _SimpleList *lst;
int (*func) ();
{
   register _ELEM *elm;
   register _ELEM *tmp;

        if (!lst || !func) {
		_InvalidArg ("_ListAppend: Null arguments\n");
		return; 
	}

	tmp = _lst_get_curr (lst);

        elm = _lst_first (lst);
	if (_lst_prev (lst) != NULL)
		fprintf (stderr, "Error: first element\n");
        while (elm) {
                (*func) (elm);
                elm = _lst_next(lst);
        }

        _lst_set_curr (lst, tmp);
}


/*
 * _SListFind: find an element the list
 */

char *_SListFind(lst,elm,cmp,flag)
_SimpleList *lst;
char *elm; 
int (*cmp) ();
int flag;
{ 
   _ELEM *elm1, *tmp; 
   short found;
	if (!lst || !elm || !cmp) {
	   _InvalidArg ("_SListFind: Null arguments\n");
	   return (NULL);
	}

	tmp = _lst_get_curr (lst);

        if (flag & _LST_BEGINNING) 
                elm1 = _lst_first(lst);
        else 
                elm1 = _lst_get_curr (lst);

        while (elm1) {
 		if (found = !cmp(elm, (char *) elm1)) {
        		_lst_set_curr (lst, tmp);
			return ((char *) elm1);
		}

		if (found > 0) {
        		_lst_set_curr (lst, tmp);
			return (NULL);
		}

                elm1 = _lst_next(lst);
	}

        _lst_set_curr (lst, tmp);
	return (NULL);
}

/*
 *  _SListSeek: set the list pointer
 */

char *_SListSeek (lst,pos) 
_SimpleList *lst;
size_t pos; 
{

   if (!lst) {
	_InvalidArg ("_ListSeek: Null arguments\n");
	return (NULL);
   }
   _lst_first(lst);
	
   while (pos--)
	_lst_next(lst); 

   return ((char *) lst->lst_curr);
}

/*
 * _SListInvert: invert a simple list
 */

_SListInvert (lst)
_SimpleList *lst;
{
   _ELEM *elm, *elm1;

   if (!lst) {
	_InvalidArg ("_SListInvert: Invalid argument\n");
	return;
   }

   if (_lst_length (lst) <= 1)
	return;


   /* At least two elements */

   elm = _lst_last (lst);    
   elm = _lst_prev (lst);

   do {
   	elm1 = _lst_prev (lst);
	_lst_unlink (lst, elm); /* This alters current */
	_lst_append (lst, elm); /* This alters current */
	_lst_set_curr (lst, elm1);
	elm = elm1;
   } while (elm != NULL);

}



/*
 * _ListCreate: Create a List
 */

_LST *_ListCreate (n_elm, elm_siz, n_ovf)
   size_t n_elm;
   size_t elm_siz;
   size_t n_ovf;
{
static size_t lstsize = _LSTSIZE;
register _LST *lst;

   if (n_elm <= 0 || elm_siz <= 0 || n_ovf <= 0) {
	_InvalidArg ("_ListCreate: Invalid arguments\n");
	return (NULL);
   }

   elm_siz += hdrsize;

   if ((lst = (_LST *) malloc (lstsize)) == NULL) {
	fprintf (stderr, "ListCreate: no enough memory\n");
        return (NULL);
   }

   if ((lst->lst_arr = (char *) malloc (elm_siz*n_elm)) == NULL) {
        fprintf (stderr, "create_lst: no enough memory\n");
        return (NULL);
   }

   lst->lst_free = 0;
   lst->lst_n_ovf = n_ovf;
   lst->lst_n_org = lst->lst_n_elm = n_elm;
   lst->lst_cnt = 0;
   lst->lst_elm_siz = elm_siz;
   lst->lst_curr = NULL;
   lst->lst_last = NULL;
   lst->lst_first = NULL;
   lst->lst_stat = 0;
   return (lst);
}

/*
 * _ListAppend: Append an element to the list
 */

char *_ListAppend (lst,elm)
register _LST  *lst;
register char *elm;
{
register char *pelm;
char *elm_lst;
size_t new_n;
size_t elm_siz;
_ELEM *elp;

   if (!lst || !elm) {
	_InvalidArg ("_ListAppend: Null arguments\n");
	return (NULL); 
   }
   elm_siz = lst->lst_elm_siz;

   if (lst->lst_free == lst->lst_n_elm)  {
	new_n = lst->lst_n_elm + lst->lst_n_ovf;
        if ((elm_lst = (char *) realloc ((void *) lst->lst_arr,
                        new_n * lst->lst_elm_siz)) == NULL) {
#ifdef DEBUG
			fprintf (stderr, "ListAppend: Not enough memory\n");
#endif
			return (NULL); 
#ifdef OLD
                        exit (1); /* too drastic */
#endif
	}

	/* Rearrange pointers */


	lst->lst_first =_PointerOffset (lst->lst_first,lst->lst_arr,elm_lst);
	lst->lst_last =_PointerOffset (lst->lst_last,lst->lst_arr,elm_lst);

	elp = _lst_first (lst);
	while (elp) { 
	   elp->elm_nxt=_PointerOffset (elp->elm_nxt,lst->lst_arr,elm_lst);
	   elp->elm_prv=_PointerOffset (elp->elm_prv,lst->lst_arr,elm_lst);
	   elp = _lst_next (lst);
	}

	/* Restore the lst_curr */
 	lst->lst_curr =_PointerOffset (lst->lst_curr,lst->lst_arr,elm_lst);

        lst->lst_arr = elm_lst;
        lst->lst_n_elm = new_n;
   }


   pelm = lst->lst_arr + lst->lst_free * elm_siz;


   memcpy (((char *) pelm + hdrsize), (char *) elm, elm_siz - hdrsize);

   _lst_append(lst,(_ELEM *) pelm);
   lst->lst_free++;
   return ((char *) pelm + hdrsize);

}


/*
 * _ListTraverse: Traverse a list
 */

_ListTraverse (lst,func)
register _LST *lst;
int (*func) ();
{
   register char *elm;
   _ELEM *tmp; 


        if (!lst || !func) {
		_InvalidArg ("_ListTraverse: Null arguments\n");
		return;
	}

	tmp = _lst_get_curr (lst);

        elm = (char *) _lst_first (lst);
        while (elm) {
                (*func) (elm + hdrsize);
                elm = (char *) _lst_next(lst);
        }

	_lst_set_curr (lst, tmp);
}



/*
 * _ListSort: Sort a list
 */

_ListSort (lst, cmp)
register _LST *lst;
int (*cmp) ();
{
   register int cnt;
   register _ELEM *pelm;

   if (!lst || !cmp) {
	_InvalidArg ("ListSort: Null arguments\n");
	return;
   }

   if (lst->lst_free <= 0)
	return;

   usrcmp = cmp;
#ifdef DEBUG
   _ListTrvSeq (lst, eprint);
#endif
   qsort((char *) lst->lst_arr, lst->lst_free, lst->lst_elm_siz, _ecmp);
   cnt = lst->lst_cnt;
#ifdef DEBUG
   fprintf (stderr, "cnt = %d, lst_free = %d\n", cnt, lst->lst_free);
#endif

   /* rearrange  pointers */

   
   lst->lst_free = 0;
   lst->lst_cnt = 0;

   lst->lst_curr = NULL;   /* be careful */
   lst->lst_last = NULL;
   lst->lst_first = NULL;
   lst->lst_stat = 0;

   pelm = (_ELEM *) lst->lst_arr;
   while (cnt--) {
#ifdef DEBUG
        eprint (((char *) pelm) + hdrsize);
#endif
	_lst_append(lst, pelm);
	pelm = (_ELEM *) ((char *) pelm + lst->lst_elm_siz);
   }

   /* compact memory */
   lst->lst_free = lst->lst_cnt;

   
}

/*
 * _ListTrvSeq: traverse the list using its array 
 */

_ListTrvSeq (lst, f)
register _LST *lst;
void (*f) ();
{
   register _ELEM *pelm;
   register size_t cnt;

   if (!lst || !f) {
	_InvalidArg ("_ListTrvSeq: Null arguments\n");
	return;
   }

   cnt = lst->lst_free;
   pelm = (_ELEM *) lst->lst_arr;
   while (cnt--) {
        (*f) (((char *) pelm) + hdrsize);
	pelm = (_ELEM *) ((char *) pelm + lst->lst_elm_siz);
   }
}

/*
 * _ListDestroy: destroy a list
 */

_ListDestroy (lst)
register _LST *lst;
{
   if (!lst) {
	_InvalidArg ("_ListDestroy: Null argument\n");
	return;
   }
	
   free ((void *) lst->lst_arr);
   lst->lst_arr = NULL;
   free ((void *) lst);
}


/*
 * _ListFind: look for an element in the list
 */

char *_ListFind(lst,elm,cmp,flag)
_LST *lst;
char *elm;
int (*cmp) ();
int flag;
{
   _ELEM *elm1;
   short found;

   _ELEM *tmp; 


	if (!lst || !elm || !cmp) {
		_InvalidArg ("_ListFind: Null arguments\n");
		return (NULL);
	}

	tmp = _lst_get_curr (lst);

        if (flag & _LST_BEGINNING)
                elm1 = _lst_first(lst);
        else
                elm1 = _lst_get_curr (lst);

	elm -=  hdrsize; /* careful */
        while (elm1) {
		usrcmp = cmp;
 		if (found = !_cmp(elm, (char *) elm1)) {
			_lst_set_curr (lst, tmp);
			elm1 = (_ELEM *) ((char *) elm1 + hdrsize);
			return ((char *) elm1);
		}

		if (found > 0) {
			_lst_set_curr (lst, tmp);
			return (NULL);
		}

                elm1 = _lst_next(lst);
	}

	_lst_set_curr (lst, tmp);
	return (NULL);
}

/*
 * _ListSeek: set the current element
 */

char *_ListSeek(lst,pos)
_LST *lst;
size_t pos;
{

	if (!lst || pos < 0) {
		_InvalidArg ("_ListSeek: Invalid arguments\n");
		return (NULL);
   	}

        /* Check pos */
	_lst_first(lst);

	while (pos--)
		_lst_next(lst);

   	return ((char *) lst->lst_curr + hdrsize);
}



/*
 * _ListReinsertO: Reinsert an element - Returns a pointer to the 
 * element after its insertion
 */

char *_ListReinsertO (lst, elm, cmp)
_LST *lst;
char *elm;
int (*cmp) ();
{

   if (!lst || !elm || !cmp) {
	_InvalidArg ("_ListReinsertO: Null arguments\n");
	return (NULL);
   }

   elm -= hdrsize; 
   _lst_unlink(lst,(_ELEM *)elm); 
   usrcmp = cmp;
   _lst_insert_o (lst,(_ELEM *) elm, _cmp);
   return elm;
}

/*
 * _ListInsertO: insert an element
 */

char *
_ListInsertO (lst,elm,cmp)
_LST *lst;
char *elm;
int (*cmp) ();
{
   char *elm_lst;
   size_t new_n;
   _ELEM *elp;
   char *pelm;
   size_t elm_siz;

   if (!lst || !elm || !cmp) {
	_InvalidArg ("_ListInsertO: Null arguments\n");
	return (NULL);
   }

   elm_siz = lst->lst_elm_siz;

   if (lst->lst_free == lst->lst_n_elm)  {
	new_n = lst->lst_n_elm + lst->lst_n_ovf;
        if ((elm_lst = (char *) realloc ((void *) lst->lst_arr,
                        new_n * elm_siz)) == NULL) {
			fprintf (stderr, "ListInsertO: Not enough memory\n");
                        return (NULL);
	}

	/* Rearrange pointers */


	lst->lst_first =_PointerOffset (lst->lst_first,lst->lst_arr,elm_lst);
	lst->lst_last =_PointerOffset (lst->lst_last,lst->lst_arr,elm_lst);

	elp = _lst_first (lst); /* This set lst_curr */
	while (elp) { 
	   elp->elm_nxt=_PointerOffset (elp->elm_nxt,lst->lst_arr,elm_lst);
	   elp->elm_prv=_PointerOffset (elp->elm_prv,lst->lst_arr,elm_lst);
	   elp = _lst_next (lst);
	}

	/* Restore lst_curr */

	lst->lst_curr =_PointerOffset (lst->lst_curr,lst->lst_arr,elm_lst);

        lst->lst_arr = elm_lst;
        lst->lst_n_elm = new_n;
   }


   pelm = lst->lst_arr + lst->lst_free * elm_siz;

   memcpy ((char *) pelm + hdrsize, (char *) elm, elm_siz - hdrsize);
   usrcmp = cmp;
   _lst_insert_o (lst,(_ELEM *) pelm, _cmp); 
   lst->lst_free++;
   return ((char *) pelm + hdrsize);
}

/*
 * _ListDelete: delete an element
 */

char *_ListDelete (lst,elm)
_LST *lst;
char *elm;
{
   if (!lst || !elm) {
	_InvalidArg ("_ListDelete: Null arguments\n");
	return (NULL);
   }

   elm -= hdrsize;
   _SListUnlink (lst, elm); /* I Must not free the memory */
   return ((char *) _SListGetCurr (lst));
}

/*
 * _ListUnlink: unlink an element
 */

char *_ListUnlink (lst,elm)
_LST *lst;
char *elm;
{
   if (!lst || !elm) {
	_InvalidArg ("_ListUnlink: Null arguments\n");
	return (NULL);
   }
   elm -= hdrsize;
   _SListUnlink (lst, elm);
   return ((char *) _SListGetCurr (lst));
}

/*
 * _ListGetPos: get the position of an element
 */

size_t _ListGetPos (lst, elm)
_LST *lst;
char *elm;
{
   if (!lst || !elm || lst->lst_elm_siz <= 0) {
	_InvalidArg ("_ListGetPos: Invalid arguments\n");
	return (0); 
   }

   elm -=hdrsize;
   return ((elm - lst->lst_arr) / lst->lst_elm_siz);
}
