/*
 * Freedom Desktop
 * Copyright 1994 by Freedom Software
 *
 * Freedom Software retains all rights to Freedom Desktop (hereafter Software)
 * in binary and in source code form.
 *
 * The commercial use of this Software shall be governed by a separate License
 * agreement. Any individual or institution wishing to make commercial use of
 * the Software must sign a license agreement with Freedom Software. In such
 * cases, the Licensee agrees to abide by the terms contained in the License
 * Agreement and not those contained in this document. Examples of commercial
 * use include (without limitation): (i) integration of the Software (source
 * code form), in whole or in part, into a commercial product sold by or on
 * on behalf of the Licensee; (ii) distribution of the Software (binary form or
 * source code form) in combination with a commercial product sold by or on
 * behalf of the Licensee.
 *
 * Freedom Software (Licensor) grants you (Licensee) a license: (i) to use,
 * copy and make changes and improvements to this Software for licensee's
 * internal business purposes; (ii) to use, copy, and distribute this Software
 * or the derivative works provided that the copyright notice and this
 * permission notice appear on all copies and that NO CHARGE is associated
 * with such copies. However, if Licensee distributes any derivative work
 * based on the Software, then Licensee shall (i) notify Licensor in writing
 * (ii) clearly state that such derivative work is a modified and not the
 * original Freedom Desktop distributed by Freedom Software (iii) publish
 * the corresponding machine-readable source code or information as to
 * where it may be obtained. Each time Licensee redistribute the Software
 * or any derivative work, the recipient automatically agrees to abide
 * by the same terms as the Licensee. Licensee may not impose terms
 * more restrictive than the terms granted herein.
 *
 * By using, copying, modifying or distributing this Software (or any
 * derivative work based on this Software) Licensee indicates acceptance
 * of the terms and conditions set forth in this License.
 *
 * Licensor reserves the right to terminate this License immediately on written
 * notice, for material breach by the Licensee.
 *
 * FREEDOM SOFTWARE DISCLAIMS ALL WARRANTIES EXPRESS OR IMPLIED WITH REGARD
 * TO THIS SOFTWARE INCLUDING BUT NOT LIMITED TO ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND  FITNESS,  IN  NO  EVENT  SHALL LICENSOR BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE
 */

#include <stdio.h>
#include "etc.h"
#include <sys/types.h>
#include <sys/stat.h>
#include "Openscript.h"

/*
 * _OpenScriptDescWrite: Write out the description field
 */

static _OpenScriptDescWrite (fp, desc)
FILE *fp;
char *desc;
{
   register char *cp = desc;

	if (!*cp)
	   return;

	fprintf (fp, "#(OS)");
	while (*cp) {
	   if (*cp == '\n')
		fprintf (fp, "\n#(OS)");
	   else
		fprintf (fp, "%c", *cp);
	   cp++;
	}
	fprintf (fp, "\n");
}


/*
 * _OpenScriptWrite: write out OpenScript
 */

int _OpenScriptWrite (path, data)
char *path;
char **data;
{
FILE *fp;
int flags = 0;

   fp = fopen (path, "w");
   
   if (!fp) {
#ifdef DEBUG
	perror (path);
#endif
	return (1);
   }

   /* use Borne shell */
   fprintf (fp, "#! /bin/sh\n#OpenScript\n");

   /* write description field */
   if (strcmp ("", data[6]))
	_OpenScriptDescWrite (fp, data[6]);

   if (!strcmp ("ON", data[3])) {
	flags+= _XTERM;
	if (!strcmp ("ON", data[4])) {
	   flags += _PAUSE;
	}
	if (!strcmp ("ON", data[5])) {
	   flags += _PAGE;
	}
   }
   /* write out execution flags */
   fprintf (fp, "OPENFLAGS=%d\n", flags);

   /* write out icon path */
   if (strcmp ("", data[2]))
	fprintf (fp, "OPENICON=\"%s\"\n", data[2]);

   
   /* current directory */
   if (strcmp ("", data[1])) {
	fprintf (fp, *(data[0])?"cd %s;":"cd %s\n", data[1]);
   }

   /* command */
   if (strcmp ("", data[0])) {
        fprintf (fp, "OPENCOMMAND=\"%s\"; eval $OPENCOMMAND\n", data[0]);
   }
   fprintf (fp, "#(OS)Do not change anything above this line\n");

out:
   fclose (fp);
   /* set execution mode */
   if (chmod (path, S_IRWXU | S_IXGRP | S_IXOTH) < 0) {
	perror (path);
        return (2);
   }
   return (0);
}

/*
 * lprocess: parse one line of the OpenScript
 */

int lprocess (line, data)
register char *line;
register char **data;
{
   char *token;
   char *aux; 
   int flags;

   if (*line == '#')
	if (!strncmp (line, "#(OS)Do not change", 18))
	   return (1);
	else if (!strncmp (line, "#(OS)", 5)) {
	   if (!data[_DESC]) {
		data[_DESC] = strdup (line+5);
		if (!data[_DESC])
		   return (-1);
		return (0);
	   } else {
		aux = data[_DESC];
		data[_DESC] = _strconcat (data[_DESC], line+5);
		free (aux);
		if (!data[_DESC])
		   return (-1);
		return (0);
	   }

	}

   token = _sgettoken (line);
   if (!token)
	return (0);
   do {
      
      if (!strcmp (token, ";")) {
	free (token);
	continue;
      }
      /* icon */
      if (!strcmp (token, "OPENICON")) {
	free (token);
	if (!(token = _sgettoken (NULL)))
	   return (-1);
	if (strcmp (token, "=")) {
	   free (token);
	   return (-1);
	}
	free (token);
        if (!(token = _sgettoken (NULL)))
	   return (-1);
        data[_ICON] = token;
      /* execution flags */
      } else if (!strcmp (token, "OPENFLAGS")) {
	free (token);
	if (!(token = _sgettoken (NULL)))
	   return (-1);
	if (strcmp (token, "=")) {
	   free (token);
	   return (-1);
	}
	free (token);
        if (!(token = _sgettoken (NULL)))
	   return (-1);

	if (!strcmp (token, "0")) {
	   data[_XTERM_FLAG] = strdup ("OFF");
	   data[_PAUSE_FLAG] = strdup ("OFF");
	   data[_PAGE_FLAG] = strdup ("OFF");
	   if (!data[_XTERM_FLAG] || !data[_PAUSE_FLAG] || !data[_PAGE_FLAG]) {
		free (token);
		return (-1);
	   }
        }
	flags = atoi (token);
	free (token);
	if (flags & _XTERM)
	   data[_XTERM_FLAG] = strdup ("ON");
	else
	   data[_XTERM_FLAG] = strdup ("OFF");
        if (flags & _PAUSE)
           data[_PAUSE_FLAG] = strdup ("ON");
        else
           data[_PAUSE_FLAG] = strdup ("OFF");
        if (flags & _PAGE)
           data[_PAGE_FLAG] = strdup ("ON");
        else
           data[_PAGE_FLAG] = strdup ("OFF");
	if (!data[_XTERM_FLAG] || !data[_PAUSE_FLAG] || !data[_PAGE_FLAG])
		return (-1);
      /* command */
      } else if (!strcmp (token, "OPENCOMMAND")) {
	free (token);
	if (!(token = _sgettoken (NULL)))
	   return (-1);
	if (strcmp (token, "=")) {
	   free (token);
	   return (-1);
	}
	free (token);
        if (!(token = _sgettoken (NULL)))
           return (-1);
        data[_CMD] = token;
      /* current directory */
      } else if (!strcmp (token, "cd")) {
	free (token);
        if (!(token = _sgettoken (NULL)))
           return (-1);
        data[_WD] = token;
      }

   } while (token = _sgettoken (NULL));
   return (0);
}

#define MAXLINE1 1024

/*
 * _OpenScriptRead: read OpenScript from file
 */

char **_OpenScriptRead (path)
char *path;
{
FILE *fp;
char line[MAXLINE1]; 
char **data;
register int i = 0;
int status;


   fp = fopen (path, "r");
  
   if (!fp) {
        perror (path);
        return (NULL); 
   }

   data =  (char **) malloc (sizeof (char *) * (_MAXDATA + 1));

   if (!data) {
	fprintf (stderr, "_OpenScriptRead: not enough memory\n");
	return (NULL);
   }


   while (i < _MAXDATA)
	data[i++] = NULL;

   data[0] = strdup (_basename(path));

   if (!data[0])
	goto out;

   if (!fgets (line, MAXLINE1, fp))
        goto out;

   if (strncmp (line, "#! /bin/sh", 10))
        goto out;

   if (!fgets (line, MAXLINE1, fp))
        goto out;

   if (strncmp (line, "#OpenScript", 11))
        goto out;

   while (fgets (line, MAXLINE1, fp)) 
	if ((status = lprocess (line, data)) < 0)
	   goto out;
	else if (status > 0)
		break;

   i = 0;
   while (i < _MAXDATA) {
     if (data[i] == NULL) {
	data[i] = strdup ("");
	if (!data[i])
	   goto out;
     }
     i++;
   }

   _stripnl(data[_DESC]);
   data[_MAXDATA] = '\0';
   fclose (fp);
   return (data);

out:
   _freeargv (data); 
   fclose (fp);
   return (NULL);
}


/*
 * OpenScriptGetFlags: get execution flags
 */

int OpenScriptGetFlags (path)
char *path;
{
   char **data;
   int flags = 0;
   
        /* read OpenScript into data */
	data = _OpenScriptRead(path);
	if (!data)
	   return (-1);

	/* check execution flags */
        if (!strcmp (data[_XTERM_FLAG], "ON"))
	   flags |= _XTERM;
        if (!strcmp (data[_PAUSE_FLAG], "ON"))
	   flags |= _PAUSE;
        if (!strcmp (data[_PAGE_FLAG], "ON"))
	   flags |= _PAGE;


	_freeargv (data); 
        return (flags);

}


/*
 * _OpenScriptUpdate: Update OpenScript
 */

int _OpenScriptUpdate (path, data)
char *path;
char **data;
{
char *tmpname;
char *basename;
char *tmppath;
char *dirname;
FILE *ifp, *ofp;
char line[MAXLINE1]; 

   if (!path || !data)
	return (-1);

   basename = _basename (path);
   
   /* use temporal file */
   tmpname = _mktmp (basename);
   if (!tmpname)
	  return (-1);

   dirname = _dirname (path);

   if (dirname)
	tmppath = _dircat(dirname, tmpname);
   free (dirname);

   if (!tmppath)
	return (-1);

   if (_exists (tmppath))
	return (-1);

   if (_cp (path, tmppath) < 0)
	return (-1);
   
   /* open tmp file */
   ifp = fopen (tmppath, "r");

   if (!ifp) {
	unlink (tmppath);
	return (-1);
   }

   /* skip header */
   if (_OpenScriptHeaderRead (ifp) < 0) {
	unlink (tmppath);
	fclose (ifp);
	return (-1);
   }

   ofp = fopen (path, "w");

   if (!ofp) {
	unlink (tmppath);
	fclose (ifp);
	return (-1);
   }

   /* Update header information */
   _OpenScriptHeaderWrite (ofp, data);


   /* transfer existing information*/
   while (fgets (line, MAXLINE1, ifp))
	fputs (line, ofp);

   fclose (ofp);
   fclose (ifp);
   unlink (tmppath);	/* remove temporal file */
   return (0);
}

/*
 * _OpenScriptHeaderWrite: write out header
 */

int _OpenScriptHeaderWrite (fp, data)
FILE *fp;
char **data;
{
   int flags = 0;

   fprintf (fp, "#! /bin/sh\n#OpenScript\n");

   /* write out description field */
   if (strcmp ("", data[6]))
        _OpenScriptDescWrite (fp, data[6]);

   /* write out execution flags */
   if (!strcmp ("ON", data[3])) {
        flags+= _XTERM;
        if (!strcmp ("ON", data[4])) {
           flags += _PAUSE;
        }
        if (!strcmp ("ON", data[5])) {
           flags += _PAGE;
        }
   }
   fprintf (fp, "OPENFLAGS=%d\n", flags);

   /* icon */
   if (strcmp ("", data[2]))
        fprintf (fp, "OPENICON=\"%s\"\n", data[2]);

  
   /* current directory */
   if (strcmp ("", data[1])) {
        fprintf (fp, *(data[0])?"cd %s;":"cd %s\n", data[1]);
   }

   /* command */
   if (strcmp ("", data[0])) {
        fprintf (fp, "OPENCOMMAND=\"%s\"; eval $OPENCOMMAND\n", data[0]);
   }
   fprintf (fp, "#(OS)Do not change anything above this line\n");
   return (1); 
}


/*
 * _OpenScriptHeaderRead: read header information
 */

int _OpenScriptHeaderRead (fp)
register FILE *fp;
{
char line[MAXLINE1]; 
int status;

   if (!fgets (line, MAXLINE1, fp))
        return (-1);

   if (strncmp (line, "#! /bin/sh", 10))
        return (-1);

   if (!fgets (line, MAXLINE1, fp))
        return (-1);

   if (strncmp (line, "#OpenScript", 11))
        return (-1);

   while (fgets (line, MAXLINE1, fp))
      if (*line == '#')
        if (!strncmp (line, "#(OS)Do not change", 18))
                return (1);

   return (-1);
}



