/*
 * Freedom Desktop
 * Copyright 1994 by Freedom Software
 *
 * Freedom Software retains all rights to Freedom Desktop (hereafter Software)
 * in binary and in source code form.
 *
 * The commercial use of this Software shall be governed by a separate License
 * agreement. Any individual or institution wishing to make commercial use of
 * the Software must sign a license agreement with Freedom Software. In such
 * cases, the Licensee agrees to abide by the terms contained in the License
 * Agreement and not those contained in this document. Examples of commercial
 * use include (without limitation): (i) integration of the Software (source
 * code form), in whole or in part, into a commercial product sold by or on
 * on behalf of the Licensee; (ii) distribution of the Software (binary form or
 * source code form) in combination with a commercial product sold by or on
 * behalf of the Licensee.
 *
 * Freedom Software (Licensor) grants you (Licensee) a license: (i) to use,
 * copy and make changes and improvements to this Software for licensee's
 * internal business purposes; (ii) to use, copy, and distribute this Software
 * or the derivative works provided that the copyright notice and this
 * permission notice appear on all copies and that NO CHARGE is associated
 * with such copies. However, if Licensee distributes any derivative work
 * based on the Software, then Licensee shall (i) notify Licensor in writing
 * (ii) clearly state that such derivative work is a modified and not the
 * original Freedom Desktop distributed by Freedom Software (iii) publish
 * the corresponding machine-readable source code or information as to
 * where it may be obtained. Each time Licensee redistribute the Software
 * or any derivative work, the recipient automatically agrees to abide
 * by the same terms as the Licensee. Licensee may not impose terms
 * more restrictive than the terms granted herein.
 *
 * By using, copying, modifying or distributing this Software (or any
 * derivative work based on this Software) Licensee indicates acceptance
 * of the terms and conditions set forth in this License.
 *
 * Licensor reserves the right to terminate this License immediately on written
 * notice, for material breach by the Licensee.
 *
 * FREEDOM SOFTWARE DISCLAIMS ALL WARRANTIES EXPRESS OR IMPLIED WITH REGARD
 * TO THIS SOFTWARE INCLUDING BUT NOT LIMITED TO ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND  FITNESS,  IN  NO  EVENT  SHALL LICENSOR BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE
 */


/*
 * Miscellaneous functions
 */


#define VALIDATE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>
#include <varargs.h>
#include <errno.h>
#include <grp.h>
#include "etc.h"
#include <sys/wait.h>
#include <signal.h>



#include <ctype.h>
#include <dirent.h>
#ifdef SUNOS
#include <strings.h>
#endif



#define CHUNKSIZE 10


/*
 * _pathcmp (s1, s2) - Compare two directory paths
 *
 * s1, s2 - NULL-terminated directory paths
 *
 * Determine if s1 and s2 point to the same directory
 * if so, 0 is returned. Otherwise 1 is returned. _pathcmp
 * is similar to strcmp but take into account paths ended
 * with '/' and NULL pointers.
 */

int _pathcmp(s1,s2)
register char *s1, *s2;
{

   if (s1 == NULL && s2 == NULL)
	return (0);
   if (s1 == NULL && s2 != NULL)
	return (-1);
   if (s1 != NULL && s2 == NULL)
	return (1);


   while (*s1 == *s2) {

        if (*s1 == '\0' && *s2 == '\0')
                return (0);
        s1++;
        s2++;
   }

   /* If I get to this point *s1 isn't equal to *s2 */

   if (*s1 != '/' && *s2 != '/')
        return (*s1 - *s2);

   if (*s1 != '\0' && *s2 != '\0')
        return (*s1 - *s2);


   /* ((*s1 == NULL && *s2 == '/') || (*s1 == '/' && *s2 == NULL)) */

   if (*s1 == '\0')
   /* check if there is something after the '/' in s2 */
        return (*(s2+1) != '\0');

   /* check if there is something after the '/' in s1 */
   return (*(s1+1) != '\0');

} 


/*
 * _pathcat: Append to path
 */

char *_pathcat (s, s1)
char *s;
char *s1;
{
	if (!strcmp (s, "/"))
		return (strcat (s, s1));

	if (s[strlen(s) - 1] == '/')
		return (strcat (s, s1));

	strcat (s, "/");
	return (strcat (s, s1));
}

/*
 * _strclone: Create a new instance of a string
 */

char *_strclone (s)
char *s;
{
   char *s1;

	if ((s1 = (char *) malloc (strlen(s) + 1 )) == NULL) {
		fprintf (stderr, "strclone: no enough memory\n");
		exit (1);
	}
	return (strcpy (s1, s));
}

/*
 * _direntcmp: Compare directory entries
 */

int _direntcmp (s1, s2)
register char *s1;
register char *s2;
{
   int c1, c2; 

   while (1) {
	if (*s1 != *s2) {
   	   c1 = *s1;
   	   c2 = *s2;
   	   if (isupper (c1))
        	c1 = tolower (c1);
   	   if (isupper (c2))
        	c2 = tolower (c2);
	   if (c1 != c2)
	   	return (c1 - c2);
	   s1++;
	   s2++;
	   continue;
	}

	if (*s1 == '\0')
	  return (0);
	s1++;
	s2++;
   }
}

/*
 * _panic: print message and abort process
 */

int _panic (s)
char *s;
{
	fprintf (stderr, "%s", s);
	abort ();

}

/*
 * _strconcat:	Concatenate two strings
 * 		s1, s2: strings
 * 
 * Bug: Does not handle NULL pointers properly
 */

char *_strconcat (s1, s2)
register char *s1, *s2;
{
   register char *news;
   if (!s1 || !s2) {
	_InvalidArg("_strconcat: invalid argument\n");
	return (NULL);
   }

   if ((news = (char *) malloc (strlen (s1) + strlen (s2) + 1)) == NULL) {
	return (NULL);
   } else 
	return ((strcat (strcpy (news, s1), s2)));
}

/*
 * vstrconcat: Convert a variable argument list into a string
 */

char *_vstrconcat (va_alist)
va_dcl
{
   va_list ap;
   char *arg;
   register size_t cmd_siz = 0;
   char *cmd;
   register char *cp;

   va_start (ap);

   while (arg = va_arg (ap, char *))
        cmd_siz += _strlen (arg);
   va_end (ap);

   if (!cmd_siz)
        return (NULL);

   cmd_siz++;

   if (!(cmd = (char *) malloc (cmd_siz)))
        return (NULL);

   cp = cmd;
   *cmd = '\0';

   va_start (ap);

   while (arg = va_arg (ap, char *)) {
        strcat (cp, arg);
        cp += _strlen (arg);
        *cp = '\0';
   }
   va_end (ap);

   return (cmd);

}

/*
 * _stralloc: Allocate memory for a string
 */

char *_stralloc (size)
size_t size;
{
   char *s;
	if ((s = (char *) malloc (size + 1)) == NULL) /* Reserve space for \0 */
		_panic ("_stralloc: not enough memory\n");
	else
		return (s);
}
	
/*
 * _dircat: Append a filename to a directory path
 *
 *  Bugs:  Does not consider Null pointers
 *         Return NULL in more than one case
 */

char *_dircat (path, subpath)
register char *path, *subpath;
{

   size_t length;
   char *newpath;

   if (!path) {
	_InvalidArg("_dircat: Invalid arguments\n");
	return (NULL);
   }
#ifdef VALIDATE
   if (!subpath)
	fprintf (stderr, "_dircat: subpath is NULL\n");
#endif

   length = _strlen (path) + _strlen (subpath) + 2;/* One byte might be wasted */


   if ((newpath = malloc (length)) == NULL) {
        fprintf (stderr, "dircat: not enough memory\n");
        return (NULL);
   }

   newpath[0] = '\0';
   if (path[strlen(path) - 1] == '/')
   	strcpy (newpath, path);
   else
        sprintf (newpath, "%s/", path);

   _strcat (newpath, subpath);


   return (newpath);
}

/*
 * _isfilename: Determine if s has the syntax of a filename
 */

int _isfilename (s)
register char *s;
{
   while (*s && *s != '/')
	s++;
   return (*s != '/');
}


#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>

#ifndef TRUE_SYSV
#include <sys/file.h>
#endif

#include <pwd.h>
#include <grp.h>


/*
 * long_listing_entry: Build a long listing entry
 */

char *long_listing_entry (path, filename, lmode)
register char *path, *filename;
register int lmode;
{

  int length;
  char * output_line;
  struct stat st;
  struct stat *filestatus = &st;
  char *fullname;

  char buf[LINK_MAX];
  int cnt;


  if (lmode & LONG) {
    fullname=(char *) XtMalloc(strlen(filename)+strlen(path)+4);


    strcpy(fullname, path);
    strcat(fullname, "/");
    strcat(fullname, filename);
#ifdef TRUE_SYSV
    if (stat(fullname, filestatus) == -1) {    /* jcc, no symbolic links */
#else
    if (lstat(fullname, filestatus) == -1) {
#endif
      /* maybe a link to a nonexistent file? */
      XtFree((char *)fullname);
      return(NULL);
    } else {
      {
	/* Long Listing, construct the line*/

	length = 0;
	output_line = NULL;

	length = (lmode & PERMS?1:0) * 12 + (lmode & NLINKS?1:0) * 7 +
		(lmode & OWNER?1:0) * 12 + (lmode & GROUP?1:0) * 12 + 
		(lmode & SIZE?1:0) * 13 + (lmode & MODTM?1:0) * 39 +
		(lmode & ACCTM?1:0) * 39 + strlen(filename) + 4;


	if (S_ISLNK(filestatus->st_mode)) {
   	   if ((cnt =  readlink(fullname, buf, LINK_MAX-1)) < 0) {
            perror (path);
      	    XtFree((char *)fullname);
            return (NULL); 
           }
           buf[cnt] = '\0';
	   length += (cnt+4); /* reserve slot for ' -> ' */

#ifdef DEBUG
           fprintf (stderr, "%s\n", buf);
#endif
	}


#ifdef DEBUG
	fprintf (stderr, "%d\n", length);
#endif

	output_line = (char *) XtMalloc (sizeof(char) * length);

	if (lmode & PERMS) {
	  char type;
	  char xusr, xgrp, xoth;
	  unsigned short mode = (filestatus->st_mode);

	  /* I could use the S_IS* macros here but I prefer to see what's
	   * going on.
	   */
	  if ((mode & S_IFMT) == S_IFDIR) type = 'd';      /* Directory */
	  else if ((mode & S_IFMT) == S_IFBLK) type = 'b'; /* Block device */
	  else if ((mode & S_IFMT) == S_IFCHR) type = 'c'; /* Character device */
	  else if ((mode & S_IFMT) == S_IFREG) type = ' '; /* Plain file */
	  else if ((mode & S_IFMT) == S_IFIFO) type = 'f'; /* Fifo */
#ifndef TRUE_SYSV					   /* jcc */
	  else if ((mode & S_IFMT) == S_IFLNK) type = 'l'; /* Symbolic link */ 
	  else if ((mode & S_IFMT) == S_IFSOCK) type = 's'; /* Socket */
#endif
	  else type = '?';
	  
	  if (mode & S_ISUID) 	/* Set User Id */
	    xusr = 's';
	  else if (mode & S_IXUSR)
	    xusr = 'x';
	  else 
	    xusr = '-';

	  if (mode & S_ISGID)	/* Set Group Id */
	    xgrp = 's';
	  else if (mode & S_IXGRP)
	    xgrp = 'x';
	  else
	    xgrp = '-';

	  if (mode & S_ISVTX)	/* Save Text */
	    xoth = 't';
	  else if (mode & S_IXOTH)
	    xoth = 'x';
	  else 
	    xoth = '-';

	  sprintf(output_line, 
		  "%c%c%c%c%c%c%c%c%c%c ",
		  type,
		  (mode & S_IRUSR) ? 'r' : '-',
		  (mode & S_IWUSR) ? 'w' : '-',
		  xusr,
		  (mode & S_IRGRP) ? 'r' : '-',
		  (mode & S_IWGRP) ? 'w' : '-',
		  xgrp,
		  (mode & S_IROTH) ? 'r' : '-',
		  (mode & S_IWOTH) ? 'w' : '-',
		  xoth);
	} 

	if (lmode & NLINKS) {
	  
	  sprintf(output_line, "%s%3d ", output_line, 
		  filestatus->st_nlink);
	}
	if (lmode & OWNER) {
	  struct passwd *pw;

	  if ((pw = getpwuid(filestatus->st_uid)) == NULL)
	    sprintf(output_line, "%s%-8d ", output_line, filestatus->st_uid);
	  else
	    sprintf(output_line, "%s%-8s ", output_line, pw->pw_name);

	}
	if (lmode & GROUP) {
	  struct group *gr;
	  
	  if ((gr = getgrgid(filestatus->st_gid)) == NULL)
	    sprintf(output_line, "%s%-8d ", output_line, filestatus->st_gid);
	  else
	    sprintf(output_line, "%s%-8s ", output_line, gr->gr_name);

	}
	if (lmode & SIZE)  {
	  unsigned short mode = (filestatus->st_mode);

	  if ((mode & S_IFMT) == S_IFBLK || (mode & S_IFMT) == S_IFCHR)
	    sprintf(output_line, "%s%5d,%3d ", output_line, 
		    (filestatus->st_rdev >> 8) & 0377,
		    (filestatus->st_rdev) & 0377);
	  else
	    sprintf(output_line, "%s%9d ", output_line, filestatus->st_size);
	}
	if (lmode & MODTM) {
	  char * time;

	  time = (char *)ctime(&(filestatus->st_mtime));
	  *(time + (strlen(time) - 1)) = '\0';
	  sprintf(output_line, "%s%s ", output_line, time);
	}
	if (lmode & ACCTM) {
	  char * time;

	  time = (char *)ctime(&(filestatus->st_atime));
	  *(time + (strlen(time) - 1)) = '\0';
	  sprintf(output_line, "%s%s ", output_line, time);
	}
        if (S_ISLNK(filestatus->st_mode))
		sprintf(output_line, "%s%s -> %s", output_line, filename, buf);
	else
		sprintf(output_line, "%s%s", output_line, filename);
#ifdef DEBUG
        fprintf (stderr, "%s", output_line);
#endif
      }
    }
    XtFree((char *)fullname);
  }


  return (output_line);
}

/*
 * _isldir: Determines if fname is a directory
 *	    Returns false (0) is fname is a symbolic link
 */

int _isldir (fname)
char *fname;
{
struct stat fstat;

   if(lstat(fname, &fstat) < 0) {
        return (0);
   } else
        return (S_ISDIR(fstat.st_mode));
}

/*
 * _isldir: Determines if fname is a directory or if fname points
 *          to a directory (symbolic links)
 */

int _isdir (fname)
char *fname;
{
struct stat fstat;

   if(stat(fname, &fstat) < 0) { 
	return (0);
   } else
	return (S_ISDIR(fstat.st_mode));
}

/*
 * _exists: check if fname or if the file pointed by fname (symbolic 
 *          links) exists
 */

int _exists (fname)
char *fname;
{
struct stat fstat;

   if(stat(fname, &fstat) < 0) { 
        return (0);
   } else
        return (1);
}

/*
 * _lexists: check if fname exists 
 */

int _lexists (fname)
char *fname;
{
struct stat fstat;

   if(lstat(fname, &fstat) < 0) { 
        return (0);
   } else
        return (1);
}


/*
 * _cp: Copy files
 */

int _cp (source, target)
char *source;
char *target;
{
   char *argv[5];
   int status;

   if (!source || !target)
	return (-1);

   argv[0] = "/bin/cp";
   argv[1] = "cp";
   argv[2] = source;
   argv[3] = target;
   argv[4] = '\0';

   if ((status = executev (argv, NULL, 1)) < 0)
	return (-1);

   if (WIFEXITED (status)) {
        if (WEXITSTATUS(status))
	   return (-1);
   } else
	   return (-1);
   return (1);
}


/*
 * cmcat: Convert a variable argument list of commands into a string.
 *        Appends a space after each command
 */

char *cmcat (va_alist)
va_dcl
{
   va_list ap;
   char *arg;
   register size_t cmd_siz = 0;
   char *cmd;
   register char *cp;

   va_start (ap);

   while (arg = va_arg (ap, char *))
	cmd_siz += _strlen (arg) + 1;
   va_end (ap);

   if (!cmd_siz)
	return (NULL);

   cmd_siz++;

   if (!(cmd = (char *) malloc (cmd_siz)))
   	return (NULL);
   
   cp = cmd;
   *cmd = '\0';

   va_start (ap);

   while (arg = va_arg (ap, char *)) {
	strcat (cp, arg);
	cp += _strlen (arg);
	*cp++ = ' ';
	*cp = '\0';
   }
   *(cp - 1) = '\0';
   va_end (ap);

   return (cmd);

}

/*
 * cnvStrToArgv: Convert a string into a string vector. The string
 *		 consists of a sequence of zero  or  more text tokens 
 *		 separated by spaces.
 */

char **cnvStrToArgv (cmdlin)
char *cmdlin;
{
   char **argv;
   register int i = 0;
   register char *fname, *cp;
   char **cmd;

   if (!cmdlin)
        return (NULL);
    
   i++;
   cp = cmdlin;
   while (cp = index(cp, ' ')) {
	cp++;
	i++;
   }

   if (!(argv = (char **) malloc (sizeof (char *) * (i + 1)))) {
	fprintf (stderr, "cnvStrToArgv: not enough memory\n");
	return (NULL);
   }

   cmd = argv;
   fname = strtok (cmdlin, " ");

   while (fname) {
        *cmd++ = (fname);
        fname = strtok (NULL, " ");
   }
   *cmd = '\0';
   return (argv);
}


/*
 * _Cargvtocmd: Convert a string vector (char **) into a single string.
 *		Insert one space after each vector string.
 */


char *_Cargvtocmd (argv, quotes)
char **argv;
int quotes;
{
register char **vp = argv;
size_t cmd_siz = 0;
register char *cmd, *cp;

   if (!argv)
	return (NULL);

   while (*vp)
        cmd_siz += strlen (*vp++) + 1;

   if (!cmd_siz)
        return (NULL);

   if (quotes)
	cmd_siz+=3;
   else
   	cmd_siz++;

   if (!(cmd = (char *) malloc (cmd_siz)))
        return (NULL);

   cp = cmd;
   if (quotes) {
	*cmd = '"';
	*(cmd+1) = '\0';
        cp++;
   } else
   	*cmd = '\0';

   vp = argv;

   while (*vp) {
        strcat (cp, *vp);
        cp += strlen (*vp++);
        *cp++ = ' ';
        *cp = '\0';
   }
   if (quotes) {
   	*(cp - 1) = '"';
   } else
	*(cp - 1) = '\0';

   return (cmd);
}



/* protections, links, owner, group, size, modification date */

static char permissions[11];
static char nlinks[4];
static char owner[25]; 
static char group[25]; 
static char devid[10];
static char size[10];
static char acctm[27];
static char modtm[27];
static char sctm[27];
static char filename[31]; 
static char linked[LINK_MAX+1]; 

char *entries[] = {permissions, nlinks, owner, group, devid, size,
		acctm, modtm, sctm, filename, linked};
		

/*
 * long_list_entry: Build a long listing entry
 */

char **long_list_entry (path, filename, lmode)
register char *path, *filename;
register int lmode;
{
  int length;
  struct stat st;
  struct stat *filestatus = &st;
  char *fullname;

  char buf[LINK_MAX+1];
  int cnt;


  if (lmode & LONG) { 
    fullname=(char *) XtMalloc(strlen(filename)+strlen(path)+4);


    strcpy(fullname, path);
    strcat(fullname, "/");
    strcat(fullname, filename);
#ifdef TRUE_SYSV
#else
    if (lstat(fullname, filestatus) == -1) {
#endif
      /* maybe a link to a nonexistent file? */
      XtFree((char *)fullname);
      return(NULL);
    } else {
	/* Long Listing, construct the line */

	length = 0;

	if (S_ISLNK(filestatus->st_mode)) {
   	   if ((cnt =  readlink(fullname, buf, LINK_MAX)) < 0) {
            perror (path);
      	    XtFree((char *)fullname);
            return (NULL); 
           }
           buf[cnt] = '\0';

#ifdef DEBUG
           fprintf (stderr, "%s\n", buf);
#endif
	}


#ifdef DEBUG
	fprintf (stderr, "%d\n", length);
#endif

	if (lmode & PERMS) {
	  char type;
	  char xusr, xgrp, xoth;
	  unsigned short mode = (filestatus->st_mode);

	  /* I could use the S_IS* macros here but I prefer to see what's
	   * going on.
	   */
	  if ((mode & S_IFMT) == S_IFDIR) type = 'd';      /* Directory */
	  else if ((mode & S_IFMT) == S_IFBLK) type = 'b'; /* Block device */
	  else if ((mode & S_IFMT) == S_IFCHR) type = 'c'; /* Character device */
	  else if ((mode & S_IFMT) == S_IFREG) type = ' '; /* Plain file */
	  else if ((mode & S_IFMT) == S_IFIFO) type = 'f'; /* Fifo */
#ifndef TRUE_SYSV					   /* jcc */
	  else if ((mode & S_IFMT) == S_IFLNK) type = 'l'; /* Symbolic link */ 
	  else if ((mode & S_IFMT) == S_IFSOCK) type = 's'; /* Socket */
#endif
	  else type = '?';
	  
	  if (mode & S_ISUID) 	/* Set User Id */
	    xusr = 's';
	  else if (mode & S_IXUSR)
	    xusr = 'x';
	  else 
	    xusr = '-';

	  if (mode & S_ISGID)	/* Set Group Id */
	    xgrp = 's';
	  else if (mode & S_IXGRP)
	    xgrp = 'x';
	  else
	    xgrp = '-';

	  if (mode & S_ISVTX)	/* Save Text */
	    xoth = 't';
	  else if (mode & S_IXOTH)
	    xoth = 'x';
	  else 
	    xoth = '-';

	  sprintf(entries[_PERMS], 
		  "%c%c%c%c%c%c%c%c%c%c",
		  type,
		  (mode & S_IRUSR) ? 'r' : '-',
		  (mode & S_IWUSR) ? 'w' : '-',
		  xusr,
		  (mode & S_IRGRP) ? 'r' : '-',
		  (mode & S_IWGRP) ? 'w' : '-',
		  xgrp,
		  (mode & S_IROTH) ? 'r' : '-',
		  (mode & S_IWOTH) ? 'w' : '-',
		  xoth);
	} 

	if (lmode & NLINKS) {
	  sprintf(entries[_NLINKS], "%-3d", filestatus->st_nlink);
	}
	if (lmode & OWNER) {
	  struct passwd *pw;

	  if ((pw = getpwuid(filestatus->st_uid)) == NULL)
	    sprintf(entries[_OWNER], "%d", filestatus->st_uid);
	  else
	    sprintf(entries[_OWNER], "%.24s", pw->pw_name);

	}
	if (lmode & GROUP) {
	  struct group *gr;
	  
	  if ((gr = getgrgid(filestatus->st_gid)) == NULL)
	    sprintf(entries[_GROUP], "%d", filestatus->st_gid);
	  else
	    sprintf(entries[_GROUP], "%.24s", gr->gr_name);
	}
	if (lmode & SIZE)  {
	  unsigned short mode = (filestatus->st_mode);

          if ((mode & S_IFMT) == S_IFBLK || (mode & S_IFMT) == S_IFCHR)
            sprintf(entries[_DEVID], "%5d,%3d",
                    (filestatus->st_rdev >> 8) & 0377,
                    (filestatus->st_rdev) & 0377);
          else
            sprintf(entries[_SIZE], "%-9d", filestatus->st_size);
	}
	if (lmode & MODTM) {
	  char * time;

	  time = (char *)ctime(&(filestatus->st_mtime));
	  *(time + (strlen(time) - 1)) = '\0';
	  sprintf(entries[_MODTM], "%s", time);
	}
	if (lmode & ACCTM) {
	  char * time;

	  time = (char *)ctime(&(filestatus->st_atime));
	  *(time + (strlen(time) - 1)) = '\0'; 
	  sprintf(entries[_ACCTM], "%s", time);
	}

        if (lmode & SCTM) {
          char * time;

          time = (char *)ctime(&(filestatus->st_ctime));
          *(time + (strlen(time) - 1)) = '\0';
          sprintf(entries[_SCTM], "%s", time);
        }

        if (S_ISLNK(filestatus->st_mode)) {
		sprintf(entries[_FILENAME], "%s", filename); 
		sprintf(entries[_LINKED], "%s", buf);
	} else {
		sprintf(entries[_FILENAME], "%s", filename);
		*(entries[_LINKED]) = '\0';
	}
#ifdef DEBUG
        fprintf (stderr, "%s", output_line);
#endif
    }
    XtFree((char *)fullname);
  }


   return (entries);
}

/*
 * permsToData: Convert permissions to a data array to be used with the
 *              entry form library
 */

permsToData (perms, data)
char *perms;
char **data;
{
   static char ON[] = "ON";
   static char OFF[] = "OFF";

     perms++;	/* skip type */

     *data++ = (*perms++ == 'r') ? ON:OFF;
     *data++ = (*perms++ == 'w') ? ON:OFF;
     *data++ = (*perms == '-') ? OFF:ON;
     *data++ = (*perms++ == 's') ? ON:OFF;
	   
     *data++ = (*perms++ == 'r') ? ON:OFF;
     *data++ = (*perms++ == 'w') ? ON:OFF;
     *data++ = (*perms == '-') ? OFF:ON;
     *data++ = (*perms++ == 's') ? ON:OFF;

     *data++ = (*perms++ == 'r') ? ON:OFF;
     *data++ = (*perms++ == 'w') ? ON:OFF;
     *data++ = (*perms == '-') ? OFF:ON;
     *data++ = (*perms++ == 't') ? ON:OFF;
}


/*
 * dataToMode: Convert the data coming from the entry form library
 *             into the corresponding octal permissions mode.
 */

int dataToMode (data)
char **data;
{
  int mode = 0;
    if (!strcmp (*data++, "ON"))
	mode = mode | 0400;
    if (!strcmp (*data++, "ON"))
	mode = mode | 0200;
    if (!strcmp (*data++, "ON"))
	mode = mode | 0100;
    if (!strcmp (*data++, "ON"))
	mode = mode | 04000;

    if (!strcmp (*data++, "ON"))
	mode = mode | 040;
    if (!strcmp (*data++, "ON"))
	mode = mode | 020;
    if (!strcmp (*data++, "ON"))
	mode = mode | 010;
    if (!strcmp (*data++, "ON"))
	mode = mode | 02000;

    if (!strcmp (*data++, "ON"))
	mode = mode | 04;
    if (!strcmp (*data++, "ON"))
	mode = mode | 02;
    if (!strcmp (*data++, "ON"))
	mode = mode | 01;
    if (!strcmp (*data++, "ON"))
	mode = mode | 01000;

    return (mode);

}

/*
 * permsToMode: Convert a string into the corresponding octal permissions mode.
 */

int permsToMode (perms)
char *perms;
{
  int mode = 0;

     perms++;

     if (*perms++ == 'r')
	mode |= 0400;

     if (*perms++ == 'w')
	mode |= 0200;

     if (*perms == 'x')
	mode |= 0100;

     if (*perms++ == 's')
	mode |= 04000;


     if (*perms++ == 'r')
	mode |= 040;

     if (*perms++ == 'w')
	mode |= 020;

     if (*perms == 'x')
	mode |= 010;

     if (*perms++ == 's')
	mode |= 02000;


     if (*perms++ == 'r')
	mode |= 04;

     if (*perms++ == 'w')
	mode |= 02;

     if (*perms == 'x')
	mode |= 01;

     if (*perms++ == 's')
	mode |= 01000;

     return (mode);
}

/*
 * _append: Append a variable argument list to a string vector
 */

char **_append (va_alist) va_dcl
{
   va_list ap;
   char **argv, **arg1, **argv1;
   char *str;
   register int argc = 0, nargc = 0;


   va_start (ap);

   argv = va_arg (ap, char **);
   if (!argv)
	return (NULL);
   while (*argv++)
	argc++;

   while (str = va_arg (ap, char *))
        nargc++;

   va_end (ap);

   va_start (ap);
   argv1 =  va_arg (ap, char **);

   argv = (char **) realloc (argv1, (sizeof (char *)) * (argc + nargc + 1));

   if (!argv) {
        fprintf (stderr, "_append: not enough memory\n");
        return (NULL);
   }

   arg1 = argv + argc;
   while (str = va_arg (ap, char *))
        *arg1++ = str;

   *arg1 = '\0';

   va_end (ap);
   return (argv);
}


/*
 * _preappend: Preappend a variable argument list to a string vector
 *
 * bug - Does not handle NULL pointers in a general way
 */

char **_preappend (va_alist) va_dcl
{
   va_list ap;
   register char **argv, **argv1, **arg1, **arg2;
   register int argc = 0, nargc = 0;
   char *str;

   va_start (ap);

   argv =  va_arg (ap, char **);
   if (!argv)
        return (NULL);
   while (*argv++)
        argc++;

   while (str = va_arg (ap, char *))
        nargc++;

   va_end (ap);

   va_start (ap);
   argv1 =  va_arg (ap, char **);

   argv = (char **) realloc (argv1, (sizeof (char *)) * (argc + nargc + 1));

   if (!argv) {
        fprintf (stderr, "_preappend: not enough memory\n");
        return (NULL);
   }

   arg1 = argv + argc;
   arg2 = argv + argc + nargc;
   while (arg1 >= argv)
        *arg2-- = *arg1--;

   arg2 = argv;
   while (str = va_arg (ap, char *))
        *arg2++ = str; 

   va_end (ap);
   return (argv);

}


/*
 * _ustrerror: return the system error message associated to errno
 */

char *_ustrerror (errno)
int errno;
{
static char unknown[] = "unknown system error";
extern int sys_nerr; 
extern char *sys_errlist[];

   if (errno < sys_nerr)
        return (sys_errlist[errno]);
   else
        return (unknown);

}

char *_syserror ()
{
static char unknown[] = "unknown system error";
extern int sys_nerr; 	
extern char *sys_errlist[];

   if (errno < sys_nerr)
        return (sys_errlist[errno]);
   else
	return (unknown);

}

/*
 * _talkto - Open a bidirectional channel to a process
 */

int _talkto(argv)
   char   **argv;
{
  int   to_child[2], /* pipe descriptors from parent->child */
        to_parent[2];/* pipe descriptors from child->parent */
  int   pid;
  if (pipe(to_child) < 0) {
	perror ("_talkto");
	return;
  }
  if (pipe(to_parent) < 0) {
	perror ("_talkto");
	return;
  }
  if (pid = vfork(), pid == 0){    /* in the child   */
     close(0);                    /* redirect stdin */
     if (dup(to_child[0]) < 0)
	perror ("_talkto");
     close(1);                    /* redirect stdout*/
     if (dup(to_parent[1]) < 0)
	perror ("_talkto");
     close(to_child[0]);          /* close pipes    */
     close(to_child[1]);
     close(to_parent[0]);
     close(to_parent[1]);
     execv(argv[0], &argv[1]);
     perror ("_talkto");
     exit (1);			  
   }
   else if (pid > 0){             /* in the parent  */
#ifdef DEBUG
      fprintf (stderr, "pid = %d\n", pid);
#endif
      close(0);                   /* redirect stdin */
      if (dup(to_parent[0]) < 0)
	perror ("_talkto");
      close(to_child[0]);         /* close pipes */
      close(to_child[1]);
      close(to_parent[0]);
      close(to_parent[1]);
      return (pid);
    }
    else {                        /* error!       */
      fprintf(stderr,"Couldn't fork process\n");
      return(-1);
    }
}

#define BSIZE 256

/*
 * _ugetcwd - get pathname of the current working directory
 */

char *_ugetcwd ()
{
int size = BSIZE;
char *ptr;
   while (1) {
	ptr = (char *) malloc (size);
	
	if (!ptr) {
	   fprintf (stderr, "_ugetcwd: not enough memory\n");
	   return (NULL);
	}

	if (getcwd (ptr, size-1) != NULL)
	   return (ptr);

	if (errno != ERANGE) {
	   perror ("_getcwd");
	   return (NULL);
	}

	free (ptr);
	size += BSIZE;
   }
}

/*
 * executev - execute command (argv)
 */

int executev (argv, wd, waitv)
char **argv;
char *wd;
int waitv;
{
   int status, pid, w;
   register void (*istat) (), (*qstat) ();
   register void (*cstat) ();

   _usignal(SIGTTOU, SIG_IGN); 
   _usignal(SIGTTIN, SIG_IGN); 
   cstat = _usignal(SIGCHLD, SIG_DFL);
   if ((pid = vfork()) == 0) { 
        _usignal(SIGINT, SIG_DFL);
        _usignal(SIGQUIT, SIG_DFL);
        _usignal(SIGHUP, SIG_DFL);
	if (wd) {
	   char *pwd_var;
       	   chdir (wd); 
	   pwd_var = _strconcat ("PWD=", wd);
	   if (pwd_var) {
	      putenv (pwd_var);
	      free (pwd_var);
	   }
	}
        execvp(argv[0], &argv[1]);
        _exit(127); 
   }
   if (pid < 0) {
        _usignal(SIGCHLD, cstat);
        return (-1);
   }
   if (waitv) {
        istat = _usignal(SIGINT, SIG_IGN);
        qstat = _usignal(SIGQUIT, SIG_IGN);
        while ((w = wait (&status)) != pid && w != -1)
        ;
        _usignal(SIGCHLD, cstat);
        _usignal(SIGINT, istat);
        _usignal(SIGQUIT, qstat);
        return(status); 
   }
   _usignal(SIGCHLD, cstat);
   return (0);
}

/*
 * _isnum: determines if c is a numeric string
 */

_isnum (s)
register char *s;
{
   if (!s || !*s)
	return (0);
   while (*s) {
	if (!isdigit(*s++))
	   return (0);
   }
   return (1);
}

/*
 * _argvconcat: concatenate two string vectors
 */

char **_argvconcat (argv1, argv2)
char **argv1, **argv2;
{
   register char **ap1, **ap2, **ap;
   register int cnt1 = 0, cnt2 = 0;
   register char **argv;

   if (!argv1 || !argv2) /* for now */
	return (NULL);

   ap1 = argv1;
   ap2 = argv2;

   while (*ap1++)
	cnt1++;

   while (*ap2++)
	cnt2++;

   cnt1 += (cnt2 + 1);

   argv = (char **) malloc (sizeof (char *) * cnt1);

   if (!argv) {
	fprintf (stderr, "_argvconcat: not enough memory\n"); 
	return (NULL);
   }

   ap1 = argv1;
   ap2 = argv2;
   ap = argv;

   while (*ap1)
	*ap++ = *ap1++;
   while (*ap2)
	*ap++ = *ap2++;
	
   *ap = '\0';
   return (argv);
}

/*
 * _isblankline: determines if s is a blank line
 */

int _isblankline (s)
register char *s;
{
   if (!s)
	return (0);
   while (*s) {
	if (*s != ' ' || *s != '\t' || *s != '\n')
	   return (0);
	s++;
   }
   return (1);

}

/*
 * _fexecutable: determines is a path references a executable file
 */
 
int _fexecutable (path)
char *path;
{
struct stat fstat;
uid_t euid, egid;

   if(stat(path, &fstat) < 0) { 
        return (-1);
   }
   
   euid = getuid();

   egid = geteuid();
   
   if (!isreadable (&fstat, euid, egid ) || !isexecutable (&fstat, euid, egid))
   	return (0);
   	
   return (1);

}
 

/*
 * isreadable: determines if a file or directory is readable
 *             by a user which effective user id and group id 
 *	       are euid and egid
 */

int isreadable (st, euid, egid)
struct stat *st;
uid_t euid;
uid_t egid;
{
   mode_t    mode = st->st_mode;

   if (!euid)
	return (1);

   if (euid == st->st_uid)
	return (mode & S_IRUSR);
#ifdef OLD
   if (egid == st->st_gid)
	return (mode & S_IRGRP);
#endif
   if (user_is_member( st->st_gid ))
   	return (mode & S_IRGRP);
   return (mode & S_IROTH);
}

/*
 * _fwritable: determines is a path references a writable file
 */
 
int _fwrite_execute (path)
char *path;
{
struct stat fstat;
uid_t euid, egid;

   if(stat(path, &fstat) < 0) { 
        return (-1);
   }
   
   euid = getuid();

   egid = geteuid();
   
   if (!iswritable (&fstat, euid, egid ) || 
   	!isexecutable (&fstat, euid, egid))
   	return (0);
   	
   return (1);

}

/*
 * isexecutable: check the search (if a directory) or 
 * 		 execute (otherwise) permission.
 */

int isexecutable (st, euid, egid)
struct stat *st;
uid_t euid;
uid_t egid;
{
   mode_t    mode = st->st_mode;

   if (!euid)
	return (1);

   if (euid == st->st_uid)
        return (mode & S_IXUSR);

#ifdef OLD
   if (egid == st->st_gid)
        return (mode & S_IXGRP);
#endif
   if (user_is_member( st->st_gid ))
   	return (mode & S_IXGRP);

   return (mode & S_IXOTH);
}

/*
 * _sskipspaces: Skip trailing spaces
 */

char *_sskipspaces (s)
register char *s;
{
   if (!s)
	return (NULL);

   while (*s && isspace (*s))
	s++;
   return (s);
}

/*
 * fskipBlanks: skip blanks in file
 */

char fskipBlanks (fp)
FILE *fp;
{
   int c;

   while ((c = getc (fp)) == ' ' || c == '\t' || c == '\n'); 

   ungetc (c, fp);

   return (c);
}

/*
 * _sgetdword: Reads a word delimited by two sc's  (from s)
 */

char *_sgetdword (s, sc)
char *s;
int sc; /* separator character */
{
   register char *cp = s;
   int cnt;
   char *token;

   if (*cp != sc)
	return (NULL);

   cp++;

   if (!(cp = rindex (cp, sc)))
	return (NULL); /* careful -  what if there are more that two   */
		       /* variables in the same line ?                 */

   cp++;
   cnt = cp - s;
   token = malloc (sizeof (char) * (cnt + 1));

   if (!token)
	return (NULL);
	
   strncpy (token, s, cnt);
   token[cnt] = '\0';
   return (token);
}


/*
 * _fgetdword: Reads a word delimited by two sc's  (from fp)
 */

char *fgetdword (fp, sc) 
FILE *fp;
int sc; /* separator character */
{
  static char token[MAXLINE];
  char *cp = token;
  int n = MAXLINE;
  int c;
  int cnt = 0;


   while (--n > 0 && (c = getc (fp)) != EOF) {
	if (c == '\n')
	   return (NULL);
        if ((c == sc) && cnt) {
	   cnt++;
           *cp++ = c;
	   break;
	}
	if (c == sc)
	   cnt++;
        *cp++ = c;
   }

   if (cnt != 2)
	return (NULL);
   *cp = '\0';

   return (token);
}


/*
 * fgetword: reads an alphanumeric string from a file
 */

char *fgetword (fp)
FILE *fp;
{
  int c;
  static char token[MAXLINE];
  char *cp = token;

   while ((c = getc (fp)) != EOF)
	if (isalpha (c) || isdigit (c))
	  *cp++ = c;
	else
	  break;
   
   *cp = '\0';
   ungetc (c, fp);
   return (token);
}

/*
 * _sgetnum: reads a numeric string from a file
 */

char *_sgetnum (s)
char *s;
{

    register char *cp = s;
    char *token;
    int token_length;

    if (!*cp || !isdigit (*cp))
        return (NULL);

    while (*cp && isdigit (*cp))
        cp++;
    
    if (*cp && !isspace (*cp) && *cp != ';')
	return (NULL);

    token_length = cp - s;
    token = (char *) malloc (sizeof (char) * (token_length + 1));

    if (token) {
        strncpy (token, s, token_length);
        token[token_length] = '\0';
    }

    return (token);
}

/*
 * _sgetany: reads a token from s
 */

char *_sgetany (s)
char *s;
{

    register char *cp = s;
    char *token;
    int token_length;

    if (!s)
        return (NULL);

    while (*cp && !isspace (*cp) && (*cp != ';'))
        cp++;

    token_length = cp - s;
    if (!token_length)
	return (NULL);

    token = (char *) malloc (sizeof (char) * (token_length + 1));

    if (token) {
        strncpy (token, s, token_length);
        token[token_length] = '\0';
    }

    return (token);
}

/*
 * _sgetword: reads a word from s
 */

char *_sgetword (s)
char *s;
{

    register char *cp = s;
    char *token;
    int token_length;

    if (!cp || !isalpha (*cp))
	return (NULL);

    while (*cp && (isalpha (*cp) || isdigit (*cp)))
	cp++;

    if (*cp && !isspace (*cp) && (*cp != '=') && (*cp != ';')) 
        return (NULL);

    token_length = cp - s;
    token = (char *) malloc (sizeof (char) * (token_length + 1));

    if (token) {
	strncpy (token, s, token_length);
	token[token_length] = '\0';
    }

    return (token);
}

/*
 * fgetop: reads an operator from a file
 */

char *fgetop (fp)
FILE *fp;
{
  int c;
  static char token[2];
  if ((c = getc (fp)) == EOF)
	return (NULL);
  token[0] = c;
  token[1] = '\0';
  return (token);
}

/*
 * fgettoken: reads a token from a file
 */

char *fgettoken (fp)
FILE *fp;
{
  int c;
  static char token[MAXLINE];

  if ((c = fskipBlanks (fp)) == EOF)
	return (NULL);
  if (isalpha(c))
	return (fgetword (fp));
  if (c == '"')
	return (fgetdword (fp, '"'));
  if (c == '=') {
	return (fgetop (fp));
  }
  return (fgetop (fp));  /* for now */
	
}

/*
 * fgettoken: reads a token from a string
 */

char *_sgettoken (s)
char *s;
{
  int c;
  static char *sp;
  char *token;

  if (s)
	sp = s;

  sp = _sskipspaces (sp);

  if (!sp)
	return (NULL);
	
	
  if ((c = *sp) == '\0')
	return (NULL);

  if (isdigit (c)) {
	token = _sgetnum (sp);
	sp += _strlen (token);
	return (token);
  }

  if (isalpha(c)) {
	token = _sgetword (sp);
	sp += _strlen (token);
        return (token);
  }

  if (c == '"') {
	token = _sgetdword (sp, '"');
	sp += _strlen (token);
        return (_stripquotes(token));
  }
  if (c == '=' || c == ';') {
	token = strdup (" ");
	if (!token)
	  return (NULL);
	*token = c;
	sp++;
        return (token);
  }
  token = _sgetany (sp);
  sp += _strlen (token);
  return (token);
}

/*
 * fskipline: skip a line
 */

int fskipline (fp)
FILE *fp;
{
  char line[MAXLINE];

   fgets (line, MAXLINE, fp);
}

/*
 * _stripquotes: Remove double quotes from token
 */

char *_stripquotes (token)
char *token;
{
 register char *cp, *cp1;

   cp = cp1 = token;

   if (!token)
	return (NULL);

   if (*cp != '"')
        return (NULL);

   cp1++;

   while (*cp++ = *cp1++);

   if (!(cp = rindex (token, '"')))
	return (NULL);

   *cp = '\0';

   return (token);
}

/*
 * _freeargv: free a string vector
 */

_freeargv (argv)
register char **argv;
{
   register char **ap = argv;
   while (*ap) {
	free (*ap);
	*ap++ = NULL;	/* just in case */
   }
   free (argv);
}

/*
 * _dirtoargv: stuff directory entries into a string vector (char **)
 */

char **_dirtoargv (path, flag)
char *path;
int flag;
{
DIR             *dirp;
register struct dirent   *dp;
char **argv, **aux, **ap;
int  tblsiz = CHUNKSIZE;
register int cnt = 0;
register char *tmp;

  argv = (char **) malloc (sizeof (char *) * CHUNKSIZE);
  *argv = '\0';
  ap = argv;

  if (!argv) {
#ifdef DEBUG
        fprintf (stderr, "_dirtoargv: not enough memory\n");
#endif
        return (NULL);
  }

  if ((dirp = opendir(path)) == NULL) {
        perror (path);
        return (NULL);
  }

  while ((dp = readdir(dirp)) != NULL) {
        if (!_strcmp (dp->d_name, "."))
           continue;
        if (!_strcmp (dp->d_name, ".."))
           continue;
        tmp = _dircat (path, dp->d_name);
        if (!tmp) {
#ifdef DEBUG
           fprintf (stderr, "dirtoargv: _dircat failed\n");
#endif
           return (0);
        }
        /* look for directories */
        if (flag && !_isdir (tmp)) {
           free (tmp);
           continue;
        }
        free (tmp);

        if (cnt >= tblsiz-1) { /* reserve space for NULL */
           tblsiz += CHUNKSIZE;
           aux = (char **) realloc (argv, tblsiz * sizeof (char *));
           if (!aux) {
#ifdef DEBUG
                fprintf (stderr, "buildGroupsMenu: not enough memory");
#endif
                _freeargv (argv);
                return (NULL);
           }
           argv = aux;
           ap = &aux[cnt];
        }
	*ap = strdup (dp->d_name);

	if (!*ap) {
           _freeargv (argv);
           return (NULL);
	}
	ap++;
        cnt++;
  }
  *ap = '\0';
  closedir (dirp);
  return (argv);
}

/*
 * _isempty: determine if the directory pointed by path is empty
 */

int _isempty (path)
char *path;
{
DIR             *dirp;
register struct dirent   *dp;
register int cnt = 0;

  if ((dirp = opendir(path)) == NULL) {
        perror (path);
        return (-1);
  }

  while ((dp = readdir(dirp)) != NULL) {
        if (!_strcmp (dp->d_name, "."))
           continue;
        if (!_strcmp (dp->d_name, ".."))
           continue;
	cnt++;
  }
  closedir(dirp);
  return (cnt?0:1);
}

/*
 * _basename: returns the last component of the path
 */

char *_basename(name)
char *name;
{
        register char *cp;

	if (!name)
	  return (NULL);

        cp = rindex(name, '/');
        return(cp ? cp + 1 : name);
}


/* 
 * _dirname: return a path with its last component removed 
 */

char *_dirname(name)
char *name;
{
   register char *cp;
   register char *dname;
   int len;

   if (!name)
	return (NULL);

   cp = rindex (name, '/');

   if (!cp)
	return (strdup("."));

   len = cp - name;

   if (!len)
	return (strdup ("/"));

   dname = (char *) malloc (sizeof (char) * (len + 1));
   if (!dname)
	return (NULL);
   strncpy (dname, name, len);
   dname[len] = '\0';
   return (dname);
}

/*
 * _stripnl: remove the ending newline from s
 */

char *_stripnl (s)
register char *s;
{
   int l;

   if (!s || !*s)
	return (s);

   l = strlen (s);

   if (s[l - 1] == '\n')
	s[l - 1] = '\0';

   return (s);
	
}

/*
 * _mktmp: return a temporal filename
 */

char *_mktmp (filename)
char *filename;
{
   static char tmp[256];
   int len;


   if (!filename)
	return (NULL);

   len = strlen (filename);

   if (len >= 255)
	return (NULL);

   strcpy (tmp, filename);
   tmp[len] = '~';
   tmp[len+1] = '\0';
   return (tmp);

}

/*
 * _emptydir: remove all the files and directories under path
 */

char * _emptydir (path)
char *path;
{
DIR             *dirp;
struct dirent   *dp;
char            *fullname;
char            *retval;


   if (!path)
	return (NULL);
   if (!_isldir (path))
        return (NULL);

retry:

   if ((dirp = opendir(path)) == NULL)
        return (strdup (path));

   while ((dp = readdir(dirp)) != NULL) {
        if (!_strcmp (dp->d_name, "."))
           continue;
        if (!_strcmp (dp->d_name, ".."))
           continue;
        fullname = _dircat (path, dp->d_name);
        if (!fullname) {
           closedir (dirp);
           return (NULL); /* bug - NULL means ok  */
        }
        if (_isldir (fullname)) {
           closedir (dirp);	/* We do not want too many file descriptors 
open at the same time */
           retval = _emptydir (fullname);
           if (retval) {
		free (fullname);
                return (retval);
           }
           if (rmdir (fullname) < 0) {
                return (fullname);
           }
	   goto retry;
        }
        if (unlink (fullname) < 0) {
           closedir (dirp);
           return (fullname);
        }
	free (fullname);
          
   }

   closedir (dirp);
   return (NULL);

}

/*
 * user_is_member: user is member of the group gid
 */
 
int user_is_member( gid )
gid_t gid ;
{
  struct group *gp;
  register int ngroups;
  gid_t *glist, *aux;


  /* determine the size of the grouplist array to allocate */
  ngroups = getgroups (0, NULL); /* number of supplementary group IDs */
  
  if (ngroups < 0) { 
#ifdef DEBUG
  	perror ("_belong_to_group");
#endif
  	return (0);
  }
  
  /* allocate grouplist array */
  aux = glist = (gid_t *) malloc (ngroups * sizeof (gid_t)); 
  
  if (!glist) {
#ifdef DEBUG
	fprintf (stderr, "_belong_to_group: not enough memory\n");
#endif
        return (0);
  }      	
  
  /* get supplementary group IDs */
  
  if ((ngroups = getgroups (ngroups, glist)) < 0) { 
#ifdef DEBUG
  	perror ("_belong_to_group");
#endif
  	return (0);
  }
  
  while (ngroups--)
    if (*glist++ == gid) {/* gid in the group list */
    	free (aux);
    	return (1);
    }
  
  free (aux);
  return (0); /* gid not found in the user's group list */  	  

}

/*
 * iswritable: determines if a file or directory is writable
 *             by a user which effective user id and group id 
 *	       are euid and egid
 */

int iswritable (st, euid, egid)
struct stat *st;
uid_t euid;
uid_t egid;
{
   mode_t    mode = st->st_mode;

   if (!euid)
	return (1);

   if (euid == st->st_uid)
	return (mode & S_IWUSR);
   if (user_is_member( st->st_gid ))
   	return (mode & S_IWGRP);
   return (mode & S_IWOTH);
}

/*
 * _in_grouplist: Return 1 if 'group' belongs to the list of 
 *                groups of the calling process; 0 otherwise
 */
 
int _in_groups (group)
char *group;
{
  struct group *gp;
  register int ngroups;
  gid_t *glist;

#ifdef OLD  
  if (!group)
    return (0);
#endif
    
 /* get the group structure associated to the group name */
      
  gp = getgrnam (group); 
  
  if (!gp) { /* error or EOF  */
#ifdef DEBUG
     fprintf (stderr, "_belong_to_group: getgrnam failed\n");
#endif 
     return (0);
  }
       
  /* determine the size of the grouplist array to allocate */
  ngroups = getgroups (0, NULL); /* number of supplementary group IDs */
  
  if (ngroups < 0) { 
#ifdef DEBUG
  	perror ("_belong_to_group");
#endif
  	return (0);
  }
  
  /* allocate grouplist array */
  glist = (gid_t *) malloc (ngroups * sizeof (gid_t)); 
  
  if (!glist) {
#ifdef DEBUG
	fprintf (stderr, "_belong_to_group: not enough memory\n");
#endif
        return (0);
  }      	
  
  /* get supplementary group IDs */
  
  if ((ngroups = getgroups (ngroups, glist)) < 0) { 
#ifdef DEBUG
  	perror ("_belong_to_group");
#endif
  	return (0);
  }
  
  while (ngroups--)
    if (*glist++ == gp->gr_gid) /* group matches one of the list */
    	return (1);
  
  return (0); /* group not found in the grouplist */  	  
  
}


_traversedir (path, func, client)
char *path;
int (*func) ();
char *client;
{
DIR		*dirp;
struct dirent	*dp;


  if ((dirp = opendir(path)) == NULL) {
#ifdef DEBUG
	perror (newpath);
#endif	
	return (NULL);
  }

  while ((dp = readdir(dirp)) != NULL) {
	 (*func) (dp->d_name, client);
  }
  
  closedir(dirp);

}

#ifdef SOLARIS
char *index (s, c)
char *s;
char c;
{
   return (strchr (s, c));
}

char *rindex (s, c)
char *s;
char c;
{
   return (strrchr (s, c));
}
void bzero (sp, len)
register char *sp;
int len;
{
  if (!sp || len <= 0)
	return;
  while (len--)
	*sp++=0;
}

void bcopy (s1, s2, len)
register char *s1, *s2;
int len;
{
  if (!s1 || !s2 || len <= 0)
	return;
  while (len--)
	*s2++=*s1++;
}
#endif

