
/*
 * Freedom Desktop
 * Copyright 1994 by Freedom Software
 *
 * Freedom Software retains all rights to Freedom Desktop (hereafter Software)
 * in binary and in source code form.
 *
 * The commercial use of this Software shall be governed by a separate License
 * agreement. Any individual or institution wishing to make commercial use of
 * the Software must sign a license agreement with Freedom Software. In such
 * cases, the Licensee agrees to abide by the terms contained in the License
 * Agreement and not those contained in this document. Examples of commercial
 * use include (without limitation): (i) integration of the Software (source
 * code form), in whole or in part, into a commercial product sold by or on
 * on behalf of the Licensee; (ii) distribution of the Software (binary form or
 * source code form) in combination with a commercial product sold by or on
 * behalf of the Licensee.
 *
 * Freedom Software (Licensor) grants you (Licensee) a license: (i) to use,
 * copy and make changes and improvements to this Software for licensee's
 * internal business purposes; (ii) to use, copy, and distribute this Software
 * or the derivative works provided that the copyright notice and this
 * permission notice appear on all copies and that NO CHARGE is associated
 * with such copies. However, if Licensee distributes any derivative work
 * based on the Software, then Licensee shall (i) notify Licensor in writing
 * (ii) clearly state that such derivative work is a modified and not the
 * original Freedom Desktop distributed by Freedom Software (iii) publish
 * the corresponding machine-readable source code or information as to
 * where it may be obtained. Each time Licensee redistribute the Software
 * or any derivative work, the recipient automatically agrees to abide
 * by the same terms as the Licensee. Licensee may not impose terms
 * more restrictive than the terms granted herein.
 *
 * By using, copying, modifying or distributing this Software (or any
 * derivative work based on this Software) Licensee indicates acceptance
 * of the terms and conditions set forth in this License.
 *
 * Licensor reserves the right to terminate this License immediately on written
 * notice, for material breach by the Licensee.
 *
 * FREEDOM SOFTWARE DISCLAIMS ALL WARRANTIES EXPRESS OR IMPLIED WITH REGARD
 * TO THIS SOFTWARE INCLUDING BUT NOT LIMITED TO ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND  FITNESS,  IN  NO  EVENT  SHALL LICENSOR BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE
 */

#define _VALIDATE
#include <sys/param.h>


#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/CoreP.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "ool.h"
#include "pixmaps.h"
#include "XedwListP.h"
#include "XedwList.h"
#include "build_option.h"
#include "Dir.h"
#include "DirP.h"
#include "file.h"
#include "etc.h"
#include "rdd.h"
#include <string.h>
#ifdef SUNOS
#include <strings.h>
#endif
#include <stdlib.h>
#include <signal.h>
#include <sys/wait.h>
#include "X.h"

#include <dirent.h>


static uid_t euid, egid;
static char *cmdtbl [] = {
	"mkdir",
	"rm",
	"rmdir",
	"mv",
	"cp",
	"ln"
};

/*
 * Internal functions
 */

static void XwDirInitialize ();
static void XwDirDoubleClick ();
static void XwDirDrop ();
static void XwDirConvertProc ();
static void XwDirActivate ();
static void XwDirCleanup ();
static void XwDirInvokeAction ();
void XwDirNewInvokeAction ();
static void XwDirEmptyTrash ();
static void XwDirCD ();
void XwDirStart ();

static void UpdateDirWidget ();
static int _DReqTerminate ();
static char *BuildDataBlk ();
static char **buildCmdFromData ();


int _opcnt = 0;
static long _maxcycles = 0L;
static long _cycles = 0L;

#define DEMO_PERIOD 20		/* 20 minutes */


/*
 * Default Action table.
 */

static XtActionsRec actions[] = {
  {"Activate",          XwDirActivate},
  {"Cleanup",           XwDirCleanup},
  {"InvokeAction",      XwDirInvokeAction},
  {"ChangeDirectory",   XwDirCD},
  {"EmptyTrash",	XwDirEmptyTrash},
  {"NewInvokeAction",   XwDirNewInvokeAction},
  {"Start",		XwDirStart}
};


static _LST 	*XwDirCreateList ();
static Boolean	DirSetValues ();
extern char 	*_editor;
extern int	_demo_mode;
extern Action action_tbl[];
static int ftypecmp ();

static int (*fcmp) () = ftypecmp;

/*
 * Resources
 */

static XtResource resources[] = {
{XtNdirectoryPath, XtCDirectoryPath, XtRString, sizeof (char *), 
XtOffset (XwDirWidget, dir.directory_path), XtRString, NULL},
{XtNlongDirectory, XtCPty, XtRInt, sizeof(int), 
XtOffset (XwDirWidget, dir.long_directory), XtRImmediate, (caddr_t)-1},
{XtNupdatePeriod, XtCPty, XtRInt, sizeof(int), 
XtOffset (XwDirWidget, dir.update_period), XtRImmediate, (caddr_t)1},
{XtNpty, XtCPty, XtRInt, sizeof(int), 
XtOffset (XwDirWidget, dir.pty), XtRImmediate, (caddr_t)-1},
{XtNbaseDirectory, XtCBaseDirectory, XtRString, sizeof (char *),
XtOffset (XwDirWidget, dir.base_directory), XtRString, 
"/usr/local/guiWidgets/Directory"},
{XtNshowParent, XtCShowParent, XtRBoolean, sizeof (Boolean),
XtOffset (XwDirWidget, dir.showparent), XtRImmediate, (XtPointer) TRUE},
{XtNshowHidden, XtCShowHidden, XtRBoolean, sizeof (Boolean),
XtOffset (XwDirWidget, dir.showhidden), XtRImmediate, (XtPointer) FALSE},
{XtNtermMode, XtCTermMode, XtRBoolean, sizeof (Boolean),
XtOffset (XwDirWidget, dir.termmode), XtRImmediate, (XtPointer) FALSE},
{XtNkeepSynchronized, XtCKeepSynchronized, XtRBoolean, sizeof (Boolean),
XtOffset (XwDirWidget, dir.keepsynchronized), XtRImmediate, (XtPointer) True},
{XtNactivateProc, XtCActivateProc, XtRFunction, sizeof (void (*) ()),
XtOffset (XwDirWidget, dir.activateproc), XtRImmediate, (XtPointer) XwDirDoubleClick},
{XtNforceMove, XtCForceMove, XtRBoolean, sizeof (Boolean),
XtOffset (XwDirWidget, dir.force_move), XtRImmediate, (XtPointer) FALSE},
};

XwDirClassRec xwDirClassRec = {
  {
/* core_class fields */
#define superclass                  (&xedwListClassRec)
    /* superclass               */  (WidgetClass) superclass,
    /* class_name               */  "Dir",
    /* widget_size              */  sizeof(XwDirRec),
    /* class_initialize         */  NULL,
    /* class_part_initialize    */  NULL,
    /* class_inited             */  FALSE,
    /* initialize               */  (XtWidgetProc) XwDirInitialize,
    /* initialize_hook          */  NULL,
    /* realize                  */  XtInheritRealize,
    /* actions                  */  actions,
    /* num_actions              */  XtNumber(actions),
    /* resources                */  resources,
    /* num_resources            */  XtNumber(resources),
    /* xrm_class                */  NULLQUARK,
    /* compress_motion          */  TRUE,
    /* compress_exposure        */  FALSE,
    /* compress_enterleave      */  TRUE,
    /* visible_interest         */  FALSE,
    /* destroy                  */  NULL,
    /* resize                   */  XtInheritResize,
    /* expose                   */  XtInheritExpose,
    /* set_values               */  DirSetValues,
    /* set_values_hook          */  NULL,
    /* set_values_almost        */  XtInheritSetValuesAlmost,
    /* get_values_hook          */  NULL,
    /* accept_focus             */  XtInheritAcceptFocus,
    /* version                  */  XtVersion,
    /* callback_private         */  NULL,
    /* tm_table                 */  XtInheritTranslations,
    /* query_geometry           */  NULL,
				NULL,
				NULL,
  },
/* Dir class fields initialization */
  {
	0,
  },
  {
	0,
  }
};

WidgetClass xwDirWidgetClass = (WidgetClass)&xwDirClassRec;

/* _CreateHTbl: Create a hash table using the directory items */
/* 		Depends on our specific list implementation */

static Hash *_CreateHTbl (lst, n_buckets)
_LST *lst;
int n_buckets;
{
   register _ELEM *pelm;
   register size_t cnt;
   register size_t i = 0;
   DirItemObject item;
   Hash *htbl;

   
   if (!n_buckets)
	n_buckets = 10;

   htbl = hash__Create (n_buckets);

   if (!lst)
	return (htbl);

   cnt = lst->lst_free;
   pelm = (_ELEM *) lst->lst_arr;

   while (i++ < cnt) {
	item = (DirItemObject) (((char *) pelm) + hdrsize);
	hash__Store(htbl,item->item_name,
	(char *) _ListArrayOffset(lst, item));
        pelm = (_ELEM *) ((char *) pelm + lst->lst_elm_siz);
   }
#ifdef DEBUG
   for (i=0; i < cnt; i++)
   	_SListTraverse(htbl->buckets[i],_PrintHashEntry);
#endif
   return (htbl);
}

/*
 * _UpdateWidgetHTbl: 	Update the hash table associated to the widget.
 * 			This function must be called after sorting the 
 *			directory entries
 */

static void _UpdateWidgetHTbl (dw)
XwDirWidget dw;
{
   DirItemObject item;
   item = (DirItemObject) _ListFirst (dw->xedwList.xedw_item_lst);

   if (!dw->dir.htbl)
	return;

   while (item) {
	hash__Replace (dw->dir.htbl, item->item_name, 
		(char *) _ListArrayOffset(dw->xedwList.xedw_item_lst, item));
	item = (DirItemObject) _ListNext (dw->xedwList.xedw_item_lst);
   }
}

/*
 * _UpdateWidgetReqL: Update widget request list
 */

_UpdateWidgetReqL(dw) 
XwDirWidget dw;
{ 
   DirReqObject dr; 
   DirItemObject item; 

   dr = (DirReqObject) _ListFirst (dw->dir.req_lst);

   while (dr) { 
        if (dr->req.type == DELETE_ENTRY) {
           item = (DirItemObject) hash__Lookup(dw->dir.htbl,
                                                (char *) (dr->req.entry));
#ifdef _VALIDATE
	   if (!item)
	   	fprintf (stderr, 
			"_UpdateWidgetReqL: hash__Lookup returned NULL\n");
#endif
           dr->req.entry = item;
	}
        dr = (DirReqObject) _ListNext (dw->dir.req_lst); 
   } 
}

/*
 * _ReplaceOffsetsByNames: This function must be called before sorting the 
 *			   directory entries so that the request list can 
 *			   be updated later
 */

static void _ReplaceOffsetsByNames (dw)
XwDirWidget dw;
{
   DirReqObject dr; 
   DirItemObject item;

   dr = (DirReqObject) _ListFirst (dw->dir.req_lst); 

   while (dr) { 
        item = (DirItemObject) ((char *) dw->xedwList.xedw_item_lst->lst_arr + 
			(size_t) dr->req.entry);
        if (dr->req.type == DELETE_ENTRY) 
	   dr->req.entry = (DirItemObject) item->item_name;
	dr = (DirReqObject) _ListNext (dw->dir.req_lst);
   }
}

/*
 * fdatecmp: compare items based on their modification dates
 */

static int fdatecmp (f1, f2)
DirItemObject f1, f2;
{
   if (!strcmp (f1->item_name, ".."))
      if (!strcmp (f2->item_name, ".."))
        return (0);
      else
        return (-1);

   if (!strcmp (f2->item_name, ".."))
        return (1);

   if (f1->fileInfo.fstat.st_mtime > f2->fileInfo.fstat.st_mtime)
	return (-1);
   if (f1->fileInfo.fstat.st_mtime < f2->fileInfo.fstat.st_mtime)
	return (1);
   if (f1->fileInfo.fstat.st_mtime == f2->fileInfo.fstat.st_mtime)
        return (_direntcmp (f1->item_name, f2->item_name));
   return (1);
}

/*
 * fsizecmp: compare items based on their size
 */

static int fsizecmp (f1, f2)
DirItemObject f1, f2;
{
   if (!strcmp (f1->item_name, ".."))
      if (!strcmp (f2->item_name, ".."))
        return (0);
      else
        return (-1);

   if (!strcmp (f2->item_name, ".."))
        return (1);

   if (f1->fileInfo.fstat.st_size > f2->fileInfo.fstat.st_size)
	return (1);
   if (f1->fileInfo.fstat.st_size < f2->fileInfo.fstat.st_size)
        return (-1);

   if (f1->fileInfo.fstat.st_size == f2->fileInfo.fstat.st_size)
        return (_direntcmp (f1->item_name, f2->item_name));
   return (-1);
}

/*
 * ftypecmp: compare files based on their file types
 */

static int ftypecmp (f1, f2)
DirItemObject f1, f2;
{
   if (!strcmp (f1->item_name, ".."))
      if (!strcmp (f2->item_name, ".."))
        return (0);
      else
        return (-1);

   if (!strcmp (f2->item_name, ".."))
        return (1);

   if (f1->fileInfo.ftype != f2->fileInfo.ftype)
	return (f1->fileInfo.ftype - f2->fileInfo.ftype);
   return (_direntcmp (f1->item_name, f2->item_name));
}

/*
 * _namecmp: compare files based on their names
 */

static int _namecmp (f1, f2)
DirItemObject f1, f2;
{
   return (_direntcmp (f1->item_name, f2->item_name));
}

/*
 * XwDirSortByDate: sort directory by date
 */

void XwDirSortByDate(w)
XwDirWidget w;
{
#ifdef DEMO
   	_check_opcnt (w);
   	_opcnt++;
#endif
	DisplayWaitCursor(w);
        _ReplaceOffsetsByNames(w);
	fcmp = fdatecmp;
        XwCollSort(w, fcmp);
        _UpdateWidgetHTbl(w);
        _UpdateWidgetReqL(w);
	DisplayNormalCursor(w)
}

/*
 * XwDirSortBySize: sort directories by size
 */

void XwDirSortBySize(w)
XwDirWidget w;
{
#ifdef DEMO
        _check_opcnt (w);
        _opcnt++;
#endif
        DisplayWaitCursor(w);
        _ReplaceOffsetsByNames(w);
	fcmp = fsizecmp;
        XwCollSort(w, fcmp);
        _UpdateWidgetHTbl(w);
        _UpdateWidgetReqL(w);
        DisplayNormalCursor(w);
}


/*
 * XwDirSortByName: sort directory by filename
 */

void XwDirSortByName(w)
XwDirWidget w;
{
#ifdef DEMO
        _check_opcnt (w);
        _opcnt++;
#endif

        DisplayWaitCursor(w);
	_ReplaceOffsetsByNames(w);
        fcmp = _namecmp;
        XwCollSort(w, fcmp);
        _UpdateWidgetHTbl(w);
        _UpdateWidgetReqL(w);
        DisplayNormalCursor(w);
}

/*
 * XwDirSortByType: sort directory by type
 */

void XwDirSortByType(w)
XwDirWidget w;
{
#ifdef DEMO
        _check_opcnt (w);
        _opcnt++;
#endif

        DisplayWaitCursor(w);
        _ReplaceOffsetsByNames(w);
	fcmp = ftypecmp;
        XwCollSort(w, ftypecmp);
        _UpdateWidgetHTbl(w);
        _UpdateWidgetReqL(w);
        DisplayNormalCursor(w);
}


/*
 * XwUpdateItem: update item
 */

void XwUpdateItem (dw, name)
XwDirWidget dw;
char *name;
{

   char *fullname;
   _LST *lst = dw->xedwList.xedw_item_lst;
   DirItemObject item;
   struct stat st;


   if (!dw->dir.directory_path)
	return;

   fullname =  _dircat(dw->dir.directory_path, name);

   if (!fullname) {
#ifdef DEBUG
        fprintf (stderr, "loadProperties: _dircat failed\n");
#endif
	free (fullname);
        return;
   }

   item = (DirItemObject) _ListFirst (lst);
   while (item) {
      if (!strcmp (item->item_name, name))
	break;
      item = (DirItemObject) _ListNext (lst);
   }

   if (!item) {
	free (fullname);
	return;
   }

   /* get status */
   if (lstat (fullname, &st) < 0) { 
#ifdef DEBUG
        perror (fullname); /* no hope */
#endif
	free (fullname);
        return;
   }
   free (fullname);

   if (!isreadable (&st, euid, egid) || (S_ISDIR(st.st_mode)
	&& !isexecutable (&st, euid, egid)))
	item->item_stat |= _ItemGreyedOut;
   else
        item->item_stat &= ~_ItemGreyedOut;

   PaintItemName(dw, item);

}


/*
 * XwDirCreateList - Create a list of directory entries
 * XwDirWidget dw		- Directory widget
 * char *newpath 		- Directory path
 * _LST *XwDirCreateList	- Entry List
 */


static _LST *XwDirCreateList (dw, newpath)
XwDirWidget dw;
char *newpath;
{

DIR		*dirp;
struct dirent	*dp;
_LST		*lst ;
_DirItemRec	item;
 DirItemObject ip = &item;     /* Item pointer */
static int 	statsize = sizeof (struct stat);
struct stat	st;
char 		*path;
PicInfo		*Pic;
register int	nfiles = 0; /* number of files */


  if (!dw) {
	_InvalidArg ("XwDirCreateList: Invalid arg\n");
	return (NULL);
  }

  dw->dir.islong_dir = False;
    
  if ((dirp = opendir(newpath)) == NULL) {
	perror (newpath);
	return (NULL);
  }


   /*
    * Create list of directory entries
    */

    if ((lst = _ListCreate (100,sizeof(_DirItemRec),100)) == NULL) { 
	fprintf (stderr, "XwDirCreateList: Not enough memory\n");
	exit (1);
    }


    while ((dp = readdir(dirp)) != NULL) {
	   if (!_strcmp (dp->d_name, "."))
		continue;
	   if (!_strcmp (dp->d_name, "..")) {
		if (!dw->dir.showparent)
		  continue;
	   } else if (!dw->dir.showhidden && (*dp->d_name == '.'))
		continue; 
	   _DirItemClear (ip);
           ip->item_name = XtNewString(dp->d_name); 
           ip->item_label = ip->item_name; 
           if (dw->dir.long_listing_flags & LONG) {
              if (ip->fileInfo.long_label =  long_listing_entry
                (dw->dir.directory_path, ip->item_name,
                 dw->dir.long_listing_flags))
                ip->item_label = ip->fileInfo.long_label;
           }

           path = _dircat (newpath, ip->item_name);

	   /* Set File Information */
           if (lstat (path, &st) < 0) {
                perror (path); 
		ip->fileInfo.ftype = T_INVALID;
		continue;
	   } 
	   
	   /* if long_directory < 0 the detection of long directories */
	   /* is disabled                                             */ 
	   if (!dw->dir.islong_dir && dw->dir.long_directory > 0)  {
	   	if (++nfiles >= dw->dir.long_directory)
  	           dw->dir.islong_dir = True;
  	   }
	   
	   if (dw->dir.islong_dir) {
	      /* Don't do expensive calculations to determine the */
	      /* file type (include magic numbers)                */
	      _DirItemFastInit(dw,path,ip,&st);
	   } else
	      _DirItemInit(dw,path,ip,&st);
	      
	   if (!isreadable (&st, euid, egid) || (S_ISDIR(st.st_mode)
		&& !isexecutable (&st, euid, egid))) {
		/* gray out item */
		ip->item_stat |= _ItemGreyedOut;
	   } else {
		ip->item_stat &= ~_ItemGreyedOut;
	   }

	   /* Append to the directory list */
           _ListAppend(lst, (char *) &item);

	   /* free path */
           free (path);
    }

    if (_ListLength (lst) <= 0)
		goto out;

    _ListSort (lst, fcmp);	/* Sort entries	*/

out:
    closedir(dirp);
    return (lst);
}

/* 
 * DirNewPath: Update the internal state of the widget using a new path
 */

static _LST *DirNewPath (dw, newpath)
XwDirWidget dw;
register char *newpath;
{
register _LST *newlst;      /* List of directory entries */
struct stat newst;
static int statsize = sizeof (struct stat);


   /* get status */
   if (stat (newpath, &newst) < 0) {
        perror (newpath); /* no hope */
        return (NULL);
   }
   
   

   /* Create directory list */
   if (!(newlst  = XwDirCreateList (dw, newpath)))
	return (NULL);

   if (!isexecutable (&newst, euid, egid)) {
   	_ListDestroy (newlst);
   	return (NULL);
   }

   /* Reset request list */
   if (!dw->dir.req_lst)
	dw->dir.req_lst =  _SListCreate ();
   else {
	_SListTraverse (dw->dir.req_lst, _DReqTerminate);
	_SListClear (dw->dir.req_lst); 
   }

   if (!dw->dir.req_lst) {
	_ListDestroy (newlst);
        return (NULL);
   }

   /* Destroy previous Hash Table */
   if (dw->dir.htbl) {
        hash__Destroy (dw->dir.htbl);
	dw->dir.htbl = NULL;
   }

   memcpy (&(dw->dir.dir_stat), &newst, statsize);


   return (newlst);
}

static void XwDirInitialize (request, new, args, numArgs)
XwDirWidget request, new;
ArgList args;
Cardinal *numArgs;
{
char curr_path[MAXPATHLEN];  
char *path;
_LST *newflst;
static int statsize = sizeof (struct stat);
XtTranslations new_trans;



   /* Initial Values */

   memset ((void *) &(new->dir.dir_stat), '\0', statsize);
   new->dir.htbl = NULL;
   new->dir.req_lst = NULL;
   new->dir.delay = 0L;
   new->dir.long_listing_flags = 0;
   new->dir.widgets = NULL;
   euid = getuid();

   egid = geteuid();

   


   if (new->dir.directory_path == NULL)  {

        /* Use current directory */
#ifdef SOLARIS
        if (!getcwd (curr_path, MAXPATHLEN)) {
#else
        if (!getwd (curr_path)) {
#endif
                fprintf (stderr, "getwd failed\n");
		return;
	}
	path = curr_path;
	
   } else
	path = new->dir.directory_path;
   
  if (_isdir (new->dir.base_directory)) { 
  	new->dir.base_directory = XtNewString (new->dir.base_directory);
  } else {
	new->dir.base_directory = XtNewString (DEFAULT_BASE_DIRECTORY);
	fprintf (stderr, " XwDirInitialize: %s is not a directory\n",
		new->dir.base_directory);
  }

  new->dir.directory_path = path;  /* for now */
  /* Validate path and update widget internal state */
  if (!(newflst = DirNewPath (new, path)))
	return; /* Things went wrong */

  new->xedwList.xedw_item_lst = newflst;

  /* Create a copy of any resource passed by reference */
  new->dir.directory_path = XtNewString (path); 

  (*superclass->core_class.initialize) 
		((XedwListWidget) request, (XedwListWidget) new,
			args, numArgs);

   if (new->dir.activateproc)
      XtAddCallback(new, XtNdoubleClick,
                new->dir.activateproc, NULL);

   XtAddCallback(new, XtNdrop,
                XwDirDrop, NULL);

   XtAddCallback(new, XtNconvertProc,
                XwDirConvertProc, NULL);

   /* Install timer */

   new->dir.delay = _DwTimerDelay;
   new->dir.timer = XtAppAddTimeOut (XtWidgetToApplicationContext (new),
                                  new->dir.delay, UpdateDirWidget, new);
   /* maximun age of any request */
   new->dir.req_max_age = _DwReqMaxAge;

   if (_demo_mode) {
      if (_maxcycles == 0)
	_maxcycles = DEMO_PERIOD * (60000L/_DwTimerDelay);
   
   }
}

/*
 * Free volatile icons 
 */

free_vicons (dw)
XwDirWidget dw;
{
_LST *lst = dw->xedwList.xedw_item_lst;
DirItemObject item;


   if (!lst)
	return;

   item = (DirItemObject) _ListFirst (lst);
   while (item) {
      if (item->fileInfo.pic->nvolatile == 1) {
        _IconFree (dw, item->fileInfo.pic);
	free (item->fileInfo.pic);
	item->fileInfo.pic = NULL;
      }
      item = (DirItemObject) _ListNext (lst);
   }
}


/*
 * DirSetValues: set resource values
 */

static Boolean DirSetValues (current, request, new,  args, numArgs)
XwDirWidget current, request, new;
ArgList args;
Cardinal *numArgs;
{
_LST *newflst;		/* List of directory entries */
DirItemObject item;



   if (new->dir.activateproc != current->dir.activateproc) {
      XtRemoveCallback(new, XtNdoubleClick,  
                current->dir.activateproc, NULL);
      XtAddCallback(new, XtNdoubleClick,
                new->dir.activateproc, NULL);
   }

#ifdef DEBUG
   if (new->dir.directory_path != NULL) {
   	fprintf (stderr, "DirSetValues: new(%s)", new->dir.directory_path);
   }

   if (current->dir.directory_path != NULL) {
   	fprintf (stderr, "DirSetValues: current(%s)\n",
		 current->dir.directory_path);
   }
#endif
   new->dir.base_directory = current->dir.base_directory; /* changes allowed */
							  /* during creation */
							  /* only	     */

   if (new->dir.directory_path == NULL) { /* Don't remove this */
	XtWarning 
	("SetValues: Directory path is NULL, restoring previous value ...");
	/* Restore previous value */
	new->dir.directory_path = current->dir.directory_path;

	return (FALSE); 
   } 


   /* call _pathcmp which is similar to strcmp but deals, among other things, */
   /* with NULL pointers 						    */
   if ((new->dir.showhidden == current->dir.showhidden) && 
	!_pathcmp (new->dir.directory_path, current->dir.directory_path)) {

	/* Avoid allocating memory */
	new->dir.directory_path = current->dir.directory_path;
	return (FALSE);

   } 

   /* directory_path has changed */

#ifdef DEBUG
   fprintf (stderr, "DirSetValues: In\n");
#endif
   /* Validate new path and update widget internal state */
   if (!(newflst = DirNewPath (new, new->dir.directory_path))) {
        /* restore previous values */
#ifdef DEBUG
	XtWarning ("SetValues Failed: restoring previous path ...");
#endif
        new->dir.directory_path = current->dir.directory_path;
	return (FALSE); /* something went wrong */
   }

   new->xedwList.xedw_item_lst = newflst;

   free_vicons (current);

   /* Create a copy of any resource passed by reference */
   new->dir.directory_path = XtNewString (new->dir.directory_path);

   /* Free previously copied directory path */
   if (current->dir.directory_path != NULL) {
	XtFree (current->dir.directory_path);
	current->dir.directory_path =  NULL;
   }
	

   /* Layout things again */
   return (*superclass->core_class.set_values) ((Widget) current, 
		(Widget) request, (Widget) new, args, numArgs);
}

/*
 * XwDirCleanup: rearrange widget
 */

static void XwDirCleanup (widget, event, params, num_params)
XwDirWidget widget;
XEvent *event;
String *params;
Cardinal num_params;
{
 _LST *flst;

#ifdef DEMO
        _check_opcnt (widget);
        _opcnt++;
#endif


	if (!widget) {
		_InvalidArg ("XwDirCleanup: Invalid argument\n");
		return;
	}

        DisplayWaitCursor(widget);
   
	if (!(flst = DirNewPath (widget, widget->dir.directory_path))) {
        	DisplayNormalCursor(widget);
		return;
	}

        free_vicons (widget);
		
	/* Current hash table is no longer valid */
        /* Destroy it */
	/* We could  make changes to rearrange the hash table so we can */
	/* keep it. This is OK for now					*/
	if (widget->dir.htbl) {
        	hash__Destroy (widget->dir.htbl);
        	widget->dir.htbl = NULL;
   	}

	XtVaSetValues (widget, XtNxedwList, flst, NULL);
        DisplayNormalCursor(widget);
}


/*
 * Activate Action
 */

static void XwDirActivate (widget, event, params, num_params)
Widget widget;
XEvent *event;
String *params;
Cardinal num_params;
{

   XwDirWidget w = (XwDirWidget) widget;
   XedwCallbackStruct cbs;

#ifdef DEMO
        _check_opcnt (w);
        _opcnt++;
#endif

   /*
    * Set callback data
    */

   cbs.reason = XedwCR_DOUBLE_CLICK;
   cbs.event = NULL;
   cbs.xedw_h_item = w->xedwList.xedw_h_item;
   cbs.xedw_h_lst = NULL;
   cbs.xedw_receptor = NULL;

   XtCallCallbacks (w, XtNdoubleClick, &cbs);
   
}


/*
 * XwDirEntriesRemove - Remove list of highlighted entries 
 */

XwDirEntriesRemove (w)
XwDirWidget w;
{
  DirItemObject item;
  _SimpleList *hl;
  char *cmd;
  register _LST *ilst;
  int i = 0;
  char **entries;
  char **argv1, **argv2;
  

#ifdef DEMO
        _check_opcnt (w);
        _opcnt++;
#endif

     if (!w) {
	_InvalidArg ("XwDirEntriesRemove\n");
	return;
     }


     ilst = w->xedwList.xedw_item_lst;

     hl = w->xedwList.xedw_h_lst;

     /* Remove highlighted entries                  */

     if (!(item = (DirItemObject) _SListFirst (hl)))
	return;

     if (!w->dir.htbl)
        w->dir.htbl = _CreateHTbl (ilst, _ListLength (ilst));

     if (!w->dir.htbl)
        fprintf (stderr,
              "XwDirCreate: not enough memory for hash table\n");
     /* check this, I do not have a return */

     entries = XwCollVItemsSelected (w);
     if (!entries) {
	fprintf (stderr, "XwDirEntriesRemove: not enough memory\n"); 
	return;
     }

     XwDirTrashFiles (w, entries);
     return;

}

/*
 * XwDirCreate: create a new directory
 */

XwDirCreate (w, name)
XwDirWidget w;
char *name;
{
  char *mkdir = cmdtbl[MKDIR];
  int nlen = strlen (name);
  DirItemObject item;
  _LST *ilst;
  static size_t ditemsize = DITEMSIZE;
  char **argv;
  
  
        if (!w || !name) {
		_InvalidArg ("XwDirCreate: Invalid argument\n");
		return;
	}

  	ilst = w->xedwList.xedw_item_lst;

	if (*name == '\0')
		return;
	
	/* Free this memory later (UpdateDirectory) */
	item = (DirItemObject) XtMalloc (ditemsize);
	_DirItemClear (item);
        item->item_name = XtNewString (name);
	item->item_label = item->item_name;

	argv = (char **) malloc (sizeof (char *) * 4);

	if (!argv) {
	   fprintf (stderr, "XwDirCreate: not enough memory\n");
	   return;
	}

	argv[0] = "/bin/mkdir";
	argv[1] = "mkdir";
	argv[2] = name;
	argv[3] = '\0';

        DisplayWaitCursor(w);
        XwDirExecute1 (w, argv, 0);
        DisplayNormalCursor(w);

	free (argv);


	/* reserve space for '/' */

   	if (!w->dir.htbl)
        	w->dir.htbl = _CreateHTbl (ilst, _ListLength (ilst));

   	if (!w->dir.htbl)
        	fprintf (stderr, 
			"XwDirCreate: not enough memory for hash table\n");

	/* Check if the filename is valid and does not already exist  */
	/* If both conditions are met, send a request to add it       */
	/* to the item list					      */
	if (_isfilename(name) && !hash__Lookup(w->dir.htbl,name))
		_DReqEnQueue (w, w->dir.req_lst, ADD_ENTRY, item);


#ifdef DEMO
        _check_opcnt (w);
        _opcnt++;
#endif

}


/*
 * _freeArgv - free data 
 */

#define _freeArgv(argv) { \
   char **tmp = argv; \
	while (*tmp) \
	   free (*tmp++); \
	free (argv); \
}



/*
 * XwDirDrop: drop callback
 */

static void XwDirDrop (dw, client_data, call_data)
  XwDirWidget     dw;
  caddr_t    client_data;
  XedwCallbackStruct *call_data;
{
   _SimpleList *hl;
   DirItemObject item, rec;
   char **argv, **cmd;
   int argc;
   static char_p_size = sizeof (char *);
   int flags = 0;
   int ftype;
   struct stat linkstat;
   char *path;
   int lnflag = 0;

#ifdef DEMO
        _check_opcnt (dw);
        _opcnt++;
#endif

     rec = (DirItemObject) call_data->xedw_receptor;


     if (call_data->reason == XedwCR_EXTERNAL_MOVE) { 
#ifdef DEBUG
	fprintf (stderr, "%s", call_data->data);
#endif
	argv = buildCmdFromData (dw, dw->dir.directory_path, call_data->data, rec,
		&flags);
	if (!argv) {
	   fprintf (stderr, "XwDirDrop: not enough memory\n");
	   return;
	}
	if (!*argv) {
 	   _XmCErrorDialogPost (dw, 
		"No drop action defined for this kind of file", "Error");
	   _freeArgv(argv);
	   return;
	}
	XwDirExecute (dw, argv, dw->dir.directory_path, flags);

	XtFree (call_data->data);
	return;
     }

     hl = call_data->xedw_h_lst;

     /* create command */
     /* check the type of the receptor */
     /* .- directory		       */
     /* .- executable                  */
     /* .- link to		       */
     /* .- tar			       */
     /* .- ar			       */
     /* .- etc			       */

     if (!_SListLength (hl) || !rec)
	return;

     if (!rec->item_name) /* just in case */
	return;

     argc = 3 + _SListLength (hl); /* Command [flag] arg1 arg2 .. */

     if (!(argv = (char **) malloc ((argc + 1) * char_p_size))) { /* Null counts */
	fprintf (stderr, "XwDirDrop: not enough memory\n");
	return;
     }

     cmd = argv;

     ftype = rec->fileInfo.ftype;
     path = _dircat (dw->dir.directory_path, rec->item_name);

     if (!path) {
        fprintf (stderr, "XwDirDrop: _dircat failed\n");
        return;
     }

     if (ftype == T_IFLNK) {
        if (stat (path, &linkstat) == 0) {
           ftype = _fasttype (path, &linkstat); 
        } else {
           perror (path);
        }
     }

     switch (ftype) {
	case T_IFDIR:
           if (call_data->reason == XedwCR_INTERNAL_MOVE) 
	   	*cmd++ = cmdtbl[MV]; 
           else if (call_data->reason == XedwCR_INTERNAL_COPY) {
	   	*cmd++ = cmdtbl[CP];
                *cmd = strdup ("-r");
                if (!*cmd) {
                   free (argv);
                   return;
		}
		cmd++;
	   } else {
	   	*cmd++ = cmdtbl[LN];
                *cmd = strdup ("-s");
                if (!*cmd) {
                   free (argv);
                   return;
                }
                cmd++;
		lnflag++;
	   }
	   break;
	case T_IFEXC:
	case T_IFSCRPT:
	case T_IFOSCRPT:
	   argc--;
	   *cmd = _strconcat ("./", rec->item_name);

	   if (!*cmd) { 
     	        free (argv);
		return;
	   }

	   cmd++;
	   break;
	default:
     	   free (argv);
	   return;
     }

     item = (DirItemObject) _SListFirst (hl);
     while (item) {
	if (lnflag) {
	   *cmd = _dircat (dw->dir.directory_path, item->item_name); 
           if (!*cmd) {
                free (argv);
                return;
           }
           cmd++;
	} else
	  *cmd++ = item->item_name; /* careful */
        item = (DirItemObject) _SListNext (hl);
     }

     switch (ftype) {
        case T_IFDIR:
     	   *cmd++ = rec->item_name; 
     	   flags |= _SHELL;
           break;
        case T_IFEXC:
	case T_IFSCRPT:
     	   flags |= _XTERM;
     	   flags |= _PAUSE;
	   break;
	case T_IFOSCRPT:
           flags = OpenScriptGetFlags (path);
           if (flags < 0) {
                flags |= _XTERM;
                flags |= _PAUSE;
           }
	   if (flags == 0) {
           	flags |= _SHELL;
           	flags |= _NOWAIT;
	   }

           break;
        default:
           free (argv);	/* it should not get here */
           return;
     }

     free (path); /* this is not enough */
     *cmd ='\0';
     flags |= _SHELL;
     XwDirExecute (dw, argv, dw->dir.directory_path, flags);

}


/*
 * XwDirDoubleClick: double click callback
 */

static void XwDirDoubleClick (w, client_data, call_data)
  XwDirWidget     w;
  caddr_t    client_data;
  XedwCallbackStruct *call_data;
{
  char *path;
  int ftype;
  DirItemObject ditem = (DirItemObject) call_data->xedw_h_item;
  struct stat linkstat;
  String params[2];


#ifdef DEMO
        _check_opcnt(w);
        _opcnt++;
#endif

#ifdef DEBUG
   fprintf (stderr, "XwDirDoubleClick\n");
#endif
   params[0] = "0";
   params[1] = NULL;

   if (call_data->xedw_h_item == NULL) {
        XtCallActionProc (w, "InvokeAction", NULL, params, 1);
	return;
   }

   path = _dircat (w->dir.directory_path, ditem->item_name); 

   if (!path) {
	fprintf (stderr, "XwDirDoubleClick: _dircat failed\n");
	return;
   }
 
#ifdef DEBUG
   fprintf (stderr, "XwDirDoubleClick(%s)\n", path);
#endif
   ftype = _fasttype (path,  
	(struct stat *) &(ditem->fileInfo.fstat));

   if (ftype == T_IFLNK) {
	if (stat (path, &linkstat) == 0) {
	   ftype = _fasttype (path, &linkstat); 
	} else {
           perror (path);
	}
   }

   if (ftype == T_IFDIR) {
           XwDirChange (w, ditem->item_name);
	   free (path);
           return;
   }

   if (ftype < 0) {
	free (path);
	return;
   }

   free (path);
   XtCallActionProc (w, "InvokeAction", NULL, params, 1);

}

/*
 * chop: chop path
 */

char *chop (path)
char *path;
{
char *dir;

   if (!strcmp(path, "/"))
	return (path);

   path[strlen(path)-1] = '\0'; /* path must end with '/' */
   

   if ((dir = rindex(path, '/')) == NULL) { /* find previous '/' */
	fprintf (stderr, "chop: Unexpected bad path (%s)\n", path);
	return (path); 
   }
   *(++dir) = '\0';
   
   return (path);
}

/*
 * _parse: parse directory
 */

char *_parse(dir)
register char *dir;
{
char *buf = dir;
register char *path = dir;
int fs = 0; /* found first slash */

  if (dir == NULL)
	return (NULL);

  while (*dir) {

	switch (*dir) {
	   case ('/'):
	   	if (fs) 
		   dir++;
	   	else {
	   	   *path++ = *dir++;
		   fs++;
		}
		break;
	   case ('.'):

		if (!fs) {
		   *path++ = *dir++;                /* '.' inside filename */
		   break;
		}
		
		if (*(dir+1) == '.') {
		   
		   if (*(dir+2) != '/' && *(dir+2)){/* '/' or NULL must come */
						    /* right after ".."      */
			*path++ = *dir++;           /* '..' inside filename   */
			*path++ = *dir++;           /* '..' inside filename   */
		   	break;
		   }

		   dir += 2;			    /* skip ".." */
		   *path ='\0';
		   /* Chop path */
		   chop (buf);
		} else {
		   if (*(dir+1) != '/' && *(dir+1)){/* '/' or NULL must come */
						    /* right after '.'       */
		   	*path++ = *dir++; 	    /* '.' inside filename   */
			break;
		   }
		   /* ignore '.' */
		   dir++; 			    /* skip '.'	             */
		   *path ='\0';
		}
		break;
	   default:
		*path++ = *dir++;
		/* bug - It does not check length */
		fs = 0;
	}
  }
  *path = '\0';
  return (buf);

}


/*
 * XwDirChange: change directory
 */

XwDirChange (w, dir)
XwDirWidget w;
char *dir;
{
char path[MAXPATHLEN];
char cmd[520];


#ifdef DEMO
        _check_opcnt (w);
        _opcnt++;
#endif

   DisplayWaitCursor(w);
   /* home directory     */
   if (dir == NULL) {
	if ((dir = getenv("HOME")) == NULL) { 
      	   fprintf (stderr, "XwDirChange: Couldn't match HOME \n");
   	   DisplayNormalCursor(w);
	   return; 
    	} else {
    	   XtVaSetValues (w, XtNdirectoryPath, dir, NULL);
	   goto out;
	   
	}
   }

   /* absolute directory */

   if (*dir == '/') {
   	sprintf (path, "%s", dir);
   	_parse(path); 
    	XtVaSetValues (w, XtNdirectoryPath, path, NULL);
	goto out;
   }

   /* relative		*/
   /* bug ~		*/

   if (strcmp (w->dir.directory_path, "/")) 
   	sprintf (path, "%s/%s", w->dir.directory_path, dir);
   else
   	sprintf (path, "/%s", dir); 

   _parse(path);

   XtVaSetValues (w, XtNdirectoryPath, path, NULL);
out:
   if (w->dir.keepsynchronized)
	XwDirSynchronize (w);
   DisplayNormalCursor(w);

}


/*
 * XwDirSynchronize: synchronize widget directory and terminal widget directory
 */

XwDirSynchronize (w)
XwDirWidget w;
{
char *argv[4];
char *path;

   XtVaGetValues (w, XtNdirectoryPath, &path, NULL);

   argv[0] = "cd"; 
   argv[1] = path;
   argv[2] = '\0';

   sendtotty (w, argv, 0);
}

/*
 * XwDirFileSelected: return file selected
 */

char *XwDirFileSelected (w)
XwDirWidget w;
{
  XedwListWidget ilw = ( XedwListWidget ) w;
  _SimpleList *hl = ilw->xedwList.xedw_h_lst;
  DirItemObject item;

	item = (DirItemObject) _SListFirst (hl);

	if (item)
	   return (item->item_name);
	else
	   return (NULL);

}

/* XwDirSelectedItemCount - return the number of items selected */

int XwDirSelectedItemCount (w)
XwDirWidget w;
{
  XedwListWidget ilw = ( XedwListWidget ) w;
  _SimpleList *hl = ilw->xedwList.xedw_h_lst;
  return (_SListLength (hl));
}

#define requested(dr) ((dr->req.type == ADD_ENTRY)?dr->req.entry->item_name \
:((DirItemObject) ((char *) ilst->lst_arr + (size_t) dr->req.entry))->item_name)

/*
 * pending: check if the requested file has a previous request pending
 */

static int pending(w, dr) 
XwDirWidget w;
DirReqObject dr;
{
DirReqObject rp;
_LST *ilst;

   ilst = w->xedwList.xedw_item_lst;

   rp = (DirReqObject) _SListFirst (w->dir.req_lst);

   while (rp != dr) {
	if (!strcmp (requested(dr), requested (rp))) {
#ifdef DEBUG
		fprintf (stderr, "pending: requested before %s\n",
		requested(dr));
#endif
		return (1);
	}
	rp = (DirReqObject) _SListNext (w->dir.req_lst);
   }
   return (0);

}


#define _check_expiration(dw)  if (_cycles >= _maxcycles) { \
                           _XmCErrorDialogPost (dw, \
                           "Evaluation period has expired, please exit", "Error"); \
			   _opcnt = MAXOPCNT; \
                           return; \
                        }

#define UpdatePeriod	1000L	/* call opendir */

/*
 * UpdateDirWidget: Serve directory requests 
 */

static void UpdateDirWidget (w, id)
XwDirWidget w;
XtIntervalId *id;
{
 struct stat 	newst;
 struct stat 	*oldstp = &(w->dir.dir_stat), st;
 char 		*path = w->dir.directory_path;
 static int	statsize = sizeof(struct stat);
 DIR 		*dirp;
 struct dirent   *dp;
 DirItemObject item, newitem;
 DirReqObject dr;
 char *tmp;
 size_t 	itemoffset;
 static size_t ditemsize = DITEMSIZE;
 _LST *ilst;
 char 	* label;
 static unsigned long age;

 


   if (_demo_mode) {
#ifdef OLD
      if (_maxcycles == 0)
	_maxcycles = DEMO_PERIOD * (60000L/_DwTimerDelay);
#endif
      _check_expiration(w);
      _cycles++;
   }
   ilst = w->xedwList.xedw_item_lst;

   age += w->dir.delay;
   if (_SListEmpty (w->dir.req_lst)) {
	goto checkstat; 
   }

 
   dr = (DirReqObject) _SListFirst (w->dir.req_lst);

   for (;dr != NULL; dr = (DirReqObject) _SListNext(w->dir.req_lst)) {
   /* check if the requested file has a previous request pending. If so */
   /* skip this request							*/

   if (pending(w, dr)) {
#ifdef DEBUG
	fprintf (stderr, "Pending\n");
#endif
	continue;
   }
   
   /* Check if the directory path has changed or the request is too old */
   if (strcmp(w->dir.directory_path, dr->req.path)
	|| (dr->req.age > w->dir.req_max_age)) { /* If so ignore request */
	if (dr->req.type == ADD_ENTRY)
#ifdef DEBUG
		fprintf (stderr, "Couldn't serve request on %s\n",
			 dr->req.entry->item_name);
#endif
	;
	else {
           	item = (DirItemObject) ((char *) ilst->lst_arr +
                        (size_t) dr->req.entry);
#ifdef DEBUG
	        fprintf (stderr, "Couldn't serve delete request on\n");
#endif
	}

	/* Delete request from the list */
	_DReqDelete (w->dir.req_lst, dr);
	continue;
   }

   /* Update age */
   dr->req.age += w->dir.delay;
   /* Serve request */
   switch (dr->req.type) {
	case DELETE_ENTRY:
	   /* Check that this entry was removed successfully */
           item = (DirItemObject) ((char *) ilst->lst_arr + 
			(size_t) dr->req.entry);

	   tmp = _dircat (dr->req.path, item->item_name);
	   if (!tmp)
		continue;

	   if (lstat (tmp, &st) == 0) {
#ifdef DEBUG
		fprintf (stderr, "Couldn't delete %s\n", item->item_name);
#endif
		free (tmp);
		continue;
	   }
	   free (tmp);
	   
	   hash__Delete(w->dir.htbl,item->item_name);

           if (item->fileInfo.pic->nvolatile == 1) {
        	_IconFree (w, item->fileInfo.pic);
      	   }

           if (item->fileInfo.long_label) {
              free (item->fileInfo.long_label);
              item->fileInfo.long_label = NULL;
	      item->item_label = item->item_name; /* just in case */
           }

	   /* Delete item from the directory widget */
	   XwCollItemDelete ((Widget) w, (ObjectT) item);
	   
	   /* Delete request from the list         */
#ifdef DEBUG
           fprintf (stderr, "%s removed\n", item->item_name);
#endif
	   _DReqDelete (w->dir.req_lst, dr); 
	   break;
	case ADD_ENTRY:

           tmp = _dircat (dr->req.path, dr->req.entry->item_name);
           if (!tmp)
                continue;

	   /* Check that this entry was created successfully */
           if (lstat (tmp, &st) < 0) {
#ifdef DEBUG
                fprintf (stderr, "Couldn't create %s\n", dr->req.entry->item_name);
#endif
                free (tmp);
		continue;
           }
#ifdef DEBUG
           fprintf (stderr, "%s created\n", dr->req.entry->item_name);
#endif
	   item = dr->req.entry;
	   _DirItemInit (w, tmp, item, &st);
	   free (tmp);
           if (!isreadable (&st, euid, egid) || (S_ISDIR(st.st_mode)
                && !isexecutable (&st, euid, egid))) {
                /* gray out item */
                item->item_stat |= _ItemGreyedOut;
           } else {
                item->item_stat &= ~_ItemGreyedOut;
           }

           if (w->dir.long_listing_flags & LONG) {
              if (item->fileInfo.long_label =  long_listing_entry
                (w->dir.directory_path, item->item_name,
                 w->dir.long_listing_flags)) 
           	item->item_label = item->fileInfo.long_label;
           }
	   newitem = (DirItemObject) XwCollItemUInsertO ((Widget) w, 
		(ObjectT) item, fcmp); 

	   /* Calculate the position of the item in the array */
	   /* Does not consider deleted items                 */
	   /* This is implemetation dependent                 */
	   /* Insert item in the hash table */
           hash__Store(w->dir.htbl,newitem->item_name
	   ,(char *) _ListArrayOffset(w->xedwList.xedw_item_lst, newitem));

	   _DReqDelete (w->dir.req_lst, dr); 
	   XtFree (item);
   }

/*
 * All of the requests must be processed before doing the exhaustive
 * checking to avoid multiple requests being generated for the same file
 */

   }

    XtVaSetValues (w, NULL);
   goto out;
checkstat:

   if (_demo_mode)
      _check_expiration(w);
      
   if (!w->dir.update_period 
   	|| (age < w->dir.update_period * 1000L))
        goto out;

   if (lstat (path, &newst) < 0) {
	perror (path);
	goto out;  
   }

   /* If it is a long directory, update only when there has */
   /* been a change for sure				     */
   
   if (w->dir.islong_dir) {    
     /* Check the modification date of the directory */
     if (newst.st_mtime <= oldstp->st_mtime)
	  goto out;
   }
#ifdef DEBUG
   fprintf (stderr, "(b)new = %ld old = %ld\n", newst.st_mtime, oldstp->st_mtime);
#endif
   age = 0L;
   if (!w->dir.htbl)
        w->dir.htbl = _CreateHTbl (ilst, _ListLength (ilst));

   if (!w->dir.htbl) {
        fprintf (stderr, "UpdateDirWidget failed; not enough memory for hash table\n");
	goto out; /* bug - I should do something to avoid a long sequence of */
		  /* messages						     */
   }

   /* Open directory stream */
   if ((dirp = opendir(path)) == NULL) {
	perror (path);
	goto out; 
   }
   /* Traverse the directory looking for changes */
#ifdef DEBUG
   fprintf (stderr, "enter\n");
#endif

   /* mark all of the items */

   _MarkItems (ilst);

   while ((dp = readdir(dirp)) != NULL) {

	/* Skip . and .. */
	if (!strcmp (dp->d_name, ".") || !strcmp (dp->d_name, ".."))
	   continue;

	if (!w->dir.showhidden && (*dp->d_name == '.'))
                continue;

	if (!(itemoffset = (size_t) hash__Lookup(w->dir.htbl,dp->d_name))) {
           item = (DirItemObject) XtMalloc (ditemsize); 
	   _DirItemClear (item);
	   item->item_name =  XtNewString (dp->d_name); /* free this */
	   item->item_label = item->item_name;
           if (w->dir.long_listing_flags & LONG) {
              if (item->fileInfo.long_label =  long_listing_entry
                (w->dir.directory_path, item->item_name,
                 w->dir.long_listing_flags))
                item->item_label = item->fileInfo.long_label;
           }
	   /* Consider Mark */
           _DReqEnQueue (w, w->dir.req_lst, ADD_ENTRY, item);
	} else {
	   /* Consider deleted items */
	   /* Remove mark            */
	   item = (DirItemObject) ((char *) ilst->lst_arr + itemoffset);
	   if (item->item_stat & _ItemDeleted)
		fprintf (stderr, "UpdateDirWidget: %s was deleted previously\n",
			item->item_name);
	   item->item_stat &= ~_ItemMarked;
#ifdef DEBUG
	   fprintf (stderr, "%s is inside the hash table\n", item->item_name);
#endif
	}
    }
    /* Delete marked items */
    /* This depends on our list implementation */
    {
     register int i = ilst->lst_free;
     item = (DirItemObject) ((char *) ilst->lst_arr + hdrsize);
	while (i--) {
	   /* consider deleted items */
	   if (item->item_stat & _ItemDeleted) {
	   	item = (DirItemObject) ((char *) item + ilst->lst_elm_siz);
#ifdef DEBUG
	        fprintf (stderr, "%s has already been marked as deleted\n", 
			item->item_name);
#endif
		continue;
	   }
	   if (item->item_stat & _ItemMarked)  {
	   	item->item_stat &= ~_ItemMarked;
#ifdef DEBUG
		fprintf (stderr, "please delete %s\n", item->item_name);
#endif
    	   	_DReqEnQueue (w, w->dir.req_lst, DELETE_ENTRY, (DirItemObject)
			_ListArrayOffset(w->xedwList.xedw_item_lst, item));
	   }
	   item = (DirItemObject) ((char *) item + ilst->lst_elm_siz);
	}
    }

#ifdef DEBUG
    fprintf (stderr, "new = %ld old = %ld\n", newst.st_mtime, oldstp->st_mtime);
#endif
    memcpy (oldstp, &newst, statsize);
#ifdef DEBUG
    fprintf (stderr, "exit\n");
#endif

    /* Close directory stream*/
    closedir(dirp);
   
out:
   if (_demo_mode)
      _check_expiration(w);

   w->dir.timer = XtAppAddTimeOut (XtWidgetToApplicationContext (w),
				  w->dir.delay, UpdateDirWidget, w);
}
	

/*
 * _MarkItems: mark all the items in the list
 */

_MarkItems(lst) 
_LST *lst;
{                                       
  register int i = lst->lst_free;                            
  register DirItemObject item;

     item = (DirItemObject) ((char *) lst->lst_arr + hdrsize);  
        while (i--) {                                           
	   /* skip ".",  ".." and deleted items */
           if ((item->item_stat & _ItemDeleted) ||
		!strcmp (item->item_name, ".") ||               
                !strcmp (item->item_name, "..")) {
                item = (DirItemObject) ((char *) item + lst->lst_elm_siz);   
                continue;                                       
	   }
           item->item_stat |= _ItemMarked;                      
           item = (DirItemObject) ((char *) item + lst->lst_elm_siz);   
        }                                                       
}

/* _DReqDelete: Delete a request from the list  */
/* rl: Request list				*/
/* dr: Request					*/

static int _DReqTerminate(dr) 
DirReqObject dr;
{
   free (dr->req.path); 
   dr->req.path = NULL;
}


/*
 * XwDirLongListing: use the long listing format
 */

void XwDirLongListing (dw)
XwDirWidget dw;
{
   
   register DirItemObject item;
   register _LST *lst = dw->xedwList.xedw_item_lst;

#ifdef DEMO
        _check_opcnt (dw);
        _opcnt++;
#endif

   if (dw->dir.long_listing_flags & LONG)
	return;

   DisplayWaitCursor(dw);
   XtVaSetValues (dw, XtNlayoutPolicy, XedwSTATIC, NULL);

   item = (DirItemObject) _ListFirst (lst);
   dw->dir.long_listing_flags = 
	LONG | PERMS | NLINKS | OWNER | SIZE | GROUP | MODTM; 

   while (item) {
	if (!item->fileInfo.long_label) /* destroy this later */
	   item->fileInfo.long_label =  long_listing_entry
		(dw->dir.directory_path, item->item_name, 
		 dw->dir.long_listing_flags); 

	if (item->fileInfo.long_label)
	   item->item_label = item->fileInfo.long_label;
        item = (DirItemObject) _ListNext (lst);
   }
   XtVaSetValues(dw, XtNshowIcons, False, XtNforceColumns, True,
 		XtNdefaultColumns, 1,  NULL);
   DisplayNormalCursor(dw);
}


/*
 * XwDirSmallIcons: display directory using small icons
 */

void XwDirSmallIcons (dw)
XwDirWidget dw;
{
   register DirItemObject item;
   register _LST *lst = dw->xedwList.xedw_item_lst;

#ifdef DEMO
        _check_opcnt (dw);
        _opcnt++;
#endif

   DisplayWaitCursor(dw);
   if (dw->dir.long_listing_flags & LONG) {
	item = (DirItemObject) _ListFirst (lst);
	dw->dir.long_listing_flags = 0;

	while (item) {

           if (item->fileInfo.long_label) {
              free (item->fileInfo.long_label);
	      item->fileInfo.long_label = NULL;
	   }

           item->item_label = item->item_name;
           item = (DirItemObject) _ListNext (lst);
	}
   }

   XtVaSetValues(dw, XtNshowIcons, True, XtNforceColumns, False,
 		XtNdefaultColumns, 4, XtNlayoutPolicy, XedwDYNAMIC,  
		NULL); 
   DisplayNormalCursor(dw);
}

/*
 * XwDirShortListing: display directory using small icons
 */

void XwDirShortListing (dw)
XwDirWidget dw;
{
   register DirItemObject item;
   register _LST *lst = dw->xedwList.xedw_item_lst;

#ifdef DEMO
        _check_opcnt (dw);
        _opcnt++;
#endif

   DisplayWaitCursor(dw);
   if (dw->dir.long_listing_flags & LONG) {
        item = (DirItemObject) _ListFirst (lst);
        dw->dir.long_listing_flags = 0;

        while (item) {

           if (item->fileInfo.long_label) {
              free (item->fileInfo.long_label);
              item->fileInfo.long_label = NULL;
           }

           item->item_label = item->item_name;
           item = (DirItemObject) _ListNext (lst);
        }
   }

   XtVaSetValues(dw, XtNshowIcons, False, XtNforceColumns, False,
 		XtNdefaultColumns, 4,  XtNlayoutPolicy, XedwSTATIC, 
		NULL); 
   DisplayNormalCursor(dw);
}

#define checkstatus(status) { \
   if (WIFEXITED (status)) { \
	if (WEXITSTATUS(status)) { \
	   _XmCErrorDialogPost (dw, "Command terminated abnormally", "Error"); \
        } \
   } else { \
	   _XmCErrorDialogPost (dw, "Command terminated abnormally", "Error"); \
   } \
}


/*
 * XwDirExecute1: execute command
 */

int XwDirExecute1 (dw, argv, flags, cdpath)
XwDirWidget dw;
char **argv;
int flags;
char *cdpath;
{
char **argv1;
char *cmd;
int status;
extern char *_freedompath;
char *pausemepath = NULL;
char *tmp;
char *cwd;

    if (!dw)
       return;  /* just in case */
    
    /* current directory to be used at execution */   
    if (flags & _SETCWD)
    	cwd = cdpath;			/* use parameter 	  */
    else
        cwd = dw->dir.directory_path;	/* use widget's directory */
    
#ifdef DEMO
   _check_opcnt (dw);
   _opcnt++;
#endif

   if (_demo_mode)
      _check_expiration(dw);

   if (flags & _CHECKEDITOR)
        if (!strcmp (_basename(_editor), "vi"))
                flags |= _XTERM;
        else {
                flags |= _NOWAIT;
                argv1 = _preappend (argv, _editor, NULL);
                if (!argv1)
                   return; 
                argv = argv1; /* careful*/
        }

#ifdef DEMO
   _check_opcnt (dw);
#endif

   if (dw->dir.termmode) {
	if (flags & _XTERM || flags & _SHELL) {
		sendtotty (dw, argv, 1);
	} else {
		if (flags & _USEPATH) { 
		  tmp = argv[0];
		  argv[0] = argv[1];
		  argv[1] = tmp;
		}
		
		sendtotty (dw, &argv[1], 1);
	}
	return;
   }

#ifdef DEMO
   _check_opcnt (dw);
#endif

   if (flags & _XTERM) {
	if ((flags & _PAUSE) && (flags & _PAGE)) {
	   pausemepath = _dircat (_freedompath, "bin/pauseme1"); 
	   if (!pausemepath) {
	      fprintf (stderr, "XwDirExecute: _dircat failed\n");
	      return;
	   }
	   argv1 = _preappend (argv, "xterm", "xterm", "-e", pausemepath ,NULL);
        } else if (flags & _PAUSE) {
	   pausemepath = _dircat (_freedompath, "bin/pauseme");
	   if (!pausemepath) {
	      fprintf (stderr, "XwDirExecute: _dircat failed\n");
	      return;
	   }
	   argv1 = _preappend (argv, "xterm", "xterm", "-e", pausemepath ,NULL);
	} else if (flags & _PAGE) {
           pausemepath = _dircat (_freedompath, "bin/pageme");
           if (!pausemepath) {
              fprintf (stderr, "XwDirExecute: _dircat failed\n");
              return;
           }
           argv1 = _preappend (argv, "xterm", "xterm", "-e", pausemepath ,NULL);
	} else
	   argv1 = _preappend (argv, "xterm", "xterm", "-e", NULL);

	if (!argv1) {
	   return; 
	}
#ifdef DEMO
   	_check_opcnt (dw);
#endif
	status = executev (argv1, cwd, 0);
	if (status < 0)
	   _XmCErrorDialogPost (dw, "Command terminated abnormally", "Error");
	return;
   }

   if (flags & _SHELL) {
     char *argv2[6];
  
	cmd = _Cargvtocmd (argv, 0);
#ifdef DEBUG
	fprintf (stderr, "XwDirExecute1: %s\n", cmd);
#endif
	if (!cmd)
	   return;

	argv2[0] = "/bin/sh";
	argv2[1] = "sh";
	argv2[2] = "-c";
	argv2[3] = cmd;
	argv2[4] = '\0';

	if (!argv2) {
	   return; 
	}
#ifdef DEMO
   	_check_opcnt (dw);
#endif
	if (flags & _NOWAIT) {
	   status = executev (argv2, cwd, 0 );
	} else
	   status = executev (argv2, cwd, 1 );

#ifdef DEBUG
	fprintf (stderr, "executev: %d\n", status);
#endif
   	checkstatus (status);
	return;
   }
#ifdef DEMO
   _check_opcnt (dw);
#endif
   if (flags & _NOWAIT) {
	if ((status = executev (argv, cwd, 0)) < 0)
           _XmCErrorDialogPost (dw, _syserror (), "Error");
        return;
   } else {
   	if ((status = executev (argv, cwd, 1)) < 0)
           _XmCErrorDialogPost (dw, _syserror (), "Error");
   	checkstatus (status);
   }
}

/*
 * sendtotty: send command the terminal emulation window
 */

sendtotty (dw, argv, sync)
XwDirWidget dw;
char **argv;
int sync;
{
   int cnt = 0;

   if (sync) {
      unparsefputs ("cd ", dw->dir.pty);
      unparsefputs (dw->dir.directory_path, dw->dir.pty);
      unparsefputs (";", dw->dir.pty); /* for now */
   }
   while (*argv) {
	if (cnt)
	   if (cnt > 80)
		unparsefputs ("\\\n", dw->dir.pty); 
	   else
	   	unparsefputs (" ", dw->dir.pty); 
		
	cnt += strlen (*argv);
	unparsefputs (*argv++, dw->dir.pty); 
   }
   unparsefputs ("\n", dw->dir.pty);
}



#ifdef SYSV
#ifndef hpux
#define vfork() fork()
#endif
#endif

execute (wd, s)
char *wd;
char *s;
{
   int status, pid, w;
   register void (*istat) (), (*qstat) ();
   
   if ((pid = vfork()) == 0) {
	signal(SIGINT, SIG_DFL);
	signal(SIGQUIT, SIG_DFL);
	signal(SIGHUP, SIG_DFL);
	chdir (wd);
	execl("/bin/sh","sh","-c",s,0);
	_exit(127);
   }
   istat = signal(SIGINT, SIG_IGN);
   qstat = signal(SIGQUIT, SIG_IGN);
   while ((w = wait (&status)) != pid && w != -1)
	;
   signal(SIGINT, istat);
   signal(SIGQUIT, qstat);
   return(status);
}

static void XwDirConvertProc (dw, client_data, call_data)
  XwDirWidget     dw;
  caddr_t    client_data;
  XtPointer *call_data;
{
   char *data = BuildDataBlk (dw, (char *) call_data);

     if (!data)
	return; 
     rddAddSelection (data, strlen (data));
     free (data);
}

#define CHARSIZE (sizeof(char))

static char *BuildDataBlk (dw, opcode)
  XwDirWidget     dw;
  char *opcode;
{
   static char *machine = "dummy";
   size_t charsize = CHARSIZE;
   char *data;
   register char *dp;
   register char *dirp = dw->dir.directory_path;
   size_t dirl = strlen (dirp);
   size_t nbsize;

   nbsize = 1 + strlen (machine) + 1 + dirl + 1 + 1;

   if (!(data = malloc (nbsize * charsize))) {
	return (NULL);
   }

   dp = data;
   *dp++ = *opcode;
   sprintf (dp, "%s\n", machine);
   dp += (strlen (machine)+1);
   sprintf (dp, "%s\n", dirp);
   dp += (dirl + 1);
   return (data);
}

#define CHARPSIZ (sizeof(char *))
#define MAXARG	256

#define _checkNullPointer(argv,ptr) { \
	if (!(ptr)) { \
		_freeArgv (argv); \
		return (NULL); \
	} \
}

static int data_length (data)
char *data;
{
    char *dp = data;
    char *aux;
    register int cnt = 0;

	dp++;			 /* skip operation code */
	dp = index (data, '\n'); /* machine name        */

	if (!dp)
	  return (-1);
	
        dp++;
	dp = index (dp, '\n');	/* source directory     */

	if (!dp)
	  return (-1);

	dp++;
	if (!*dp)
	  return (0);

	do {
	   dp = index (dp, ' ');

	   if (dp) {
		cnt++;
		dp++;
	   }

	} while (dp);

	cnt++;	/* last one */
	return (cnt);

}

/*
 * buildCmdFromData: build command from data
 */

static char **buildCmdFromData (dw, path, data, rec, flags)
XwDirWidget dw;
char *path;
char *data;
DirItemObject rec;
int *flags;
{
register char **cmd, **argv; 
register int i = 0;
static size_t charpsize = CHARPSIZ;
char *target = NULL;
char *srcDir;
char *fname;
char *cmdname;
int n_slots;
int ftype = 0;



     if (!data)
	return (NULL);

     if ((n_slots = data_length (data)) < 0)
	return (NULL);

     n_slots += 2;	/* slots for the command name (i.e. ln -s) */
     n_slots++;		/* target				   */
     n_slots++;		/* NULL					   */

#ifdef DEBUG
     fprintf (stderr, "n_slots = %d\n", n_slots);
#endif

     if (!(argv = (char **) malloc (charpsize * n_slots)))
	return (NULL);

     cmd = argv;
     *cmd = '\0';
     if (!dw->dir.force_move)
     	switch (*data++) {
	case XedwCR_EXTERNAL_COPY:
	  cmdname = cmdtbl[CP];
	  break;
	case XedwCR_EXTERNAL_MOVE:
	  cmdname = cmdtbl[MV];
	  break;
	case XedwCR_EXTERNAL_LINK:
	  cmdname = cmdtbl[LN]; 
	  break;
	default:
	  goto out;
     	}
    else 
	  cmdname = cmdtbl[MV];

     if (rec) {
        ftype = rec->fileInfo.ftype;
	target = _dircat (path, rec->item_name);

	if (!target) {
	   goto out;
     	}

	if (ftype == T_IFLNK) {
	   ftype = _virtualtype (target);
	}

        if (ftype == -1)
           goto out;

        switch (ftype) {
        case T_IFDIR:
           *flags |= _SHELL;
	   
           *cmd = strdup (cmdname); 
	   if (!*cmd)
		goto out;
	   cmd++;
	   if (!strcmp (cmdname, cmdtbl[LN])) {
		*cmd = strdup ("-s");
	        if (!*cmd)
		   goto out;
	   	cmd++;
	   }
           if (!strcmp (cmdname, cmdtbl[CP])) {
                *cmd = strdup ("-r");
                if (!*cmd)
                   goto out;
                cmd++;
           }
	   if (!(target = strdup (rec->item_name))) {
		*cmd = '\0';
		goto out;
	   }
           break;
        case T_IFEXC:
        case T_IFSCRPT:
           *flags |= _XTERM;
           *flags |= _PAUSE;
           *cmd = _strconcat ("./", rec->item_name);
	   if (!*cmd)
		goto out;
	   cmd++;
           break;
        case T_IFOSCRPT:
           *cmd = _strconcat ("./", rec->item_name);
	   if (!*cmd)
		goto out;

	   *flags = OpenScriptGetFlags (target);
	   if (*flags < 0) {
		*flags |= _XTERM;
		*flags |= _PAUSE;
	   }
           if (*flags == 0) {
                *flags |= _SHELL;
                *flags |= _NOWAIT;
           }

	   cmd++;
           break;
        default:
	   *cmd = '\0';
           return (argv);
     }
     } else {
        *flags |= _SHELL;
	*cmd = strdup (cmdname);
	if (!*cmd)
	   goto out;
	cmd++;
        if (!strcmp (cmdname, cmdtbl[LN])) {
	   *cmd = strdup ("-s");
	   if (!*cmd)
	      goto out;
	   cmd++;
	}
        if (!strcmp (cmdname, cmdtbl[CP])) {
           *cmd = strdup ("-r");
           if (!*cmd)
              goto out;
           cmd++;
        }
	if (!(target = strdup ("."))) {
                *cmd = '\0';
		goto out;
        }
     }


     fname = strtok (data, "\n"); /* skip the machine name */
     srcDir = strtok (NULL, "\n");
     fname = strtok (NULL, " ");

     while (fname) {
	*cmd = _dircat (srcDir, fname); 
	if (!*cmd)
	   goto out;
	cmd++;
	fname = strtok (NULL, " ");
     }

     if (target && ftype != T_IFEXC && ftype != T_IFSCRPT && ftype != T_IFOSCRPT)
     	*cmd++ = target;
     *cmd = '\0';
     return (argv);
out:
     _freeArgv(argv);
     return (NULL);

}

/*
 * _parseargv: parse command
 */

char **_parseargv (argv, files)
char **argv;
char **files;
{
   char **v = argv;

	while (*v) {
	  if (**v == '$' || **v == '.') {
		if (!strcmp (*v, "$EDITOR"))
		   *v = _editor;
		else if (!strncmp (*v, ".\/$", 3)) {
		   *v = _strconcat ("./", files[0]);
		   if (!*v)
			return (NULL);
		} else
		   *v = files[0]; /* for now */

	  }
	  v++;
		
	}
   return (argv);
}

/*
 * XwDirCD: change directory
 */

static void XwDirCD (dw, event, params, num_params)
XwDirWidget dw;
XEvent *event;
String *params;
Cardinal *num_params;
{
  if (*num_params != 1) /* for now */
        return;

   DisplayWaitCursor(dw);

   XtVaSetValues (dw, XtNdirectoryPath, params[0], NULL); /* for now */

   if (dw->dir.keepsynchronized) 
        XwDirSynchronize (dw);

   DisplayNormalCursor(dw);
}

/*
 * XwDirInvokeAction: invoke a file action
 */

static void XwDirInvokeAction (dw, event, params, num_params)
XwDirWidget dw;
XEvent *event;
String *params;
Cardinal *num_params;
{
  char *files[2];
  DirItemObject item;
  char *cmd;
  int ftype = 0;
  int option;
  Action *action;
  char **argv;
  int flags;
  char *path;

  if (*num_params != 1)
	return;

  /*
   * Get option selected 
   */

  option = atoi (params[0]);
   
#ifdef OLD
   item = (DirItemObject) dw->xedwList.xedw_h_item; /* for now */
#endif
#ifdef OLD
   if (XwDirSelectedItemCount (dw) != 1)
   	return;
   /* 1/7/95 */
#endif 
   	
   item = (DirItemObject) _SListFirst (dw->xedwList.xedw_h_lst);

   if (!item) {
      files[0] = '\0';
      ftype = 0;
   } else {
      files[0] = item->item_name;
      files[1] = '\0';
      ftype = item->fileInfo.ftype;

#ifdef OLD
      if (ftype == T_IFLNK) {
        path = _dircat (dw->dir.directory_path, item->item_name);
        if (!path) {
           fprintf (stderr, "XwDirInvokeAction: _dircat failed\n");
           return;
        }
	ftype = _virtualtype (path);
	if (ftype == -1)
	   return;
      }
#endif
      path = _dircat (dw->dir.directory_path, item->item_name);
      if (!path) {
	 return;
      }
      ftype = _virtualtype (path);
      free (path);
      if (ftype == -1) /* invalid type or unable to get the file type */ 
        return;
   }


   /* check that ftype is not outside the array limits */

   action = &action_tbl[ftype];
   if (!action)
	return;

   /* check that option is not outside the array limits */

   if (!action->cmd_tbl)
	return;

   cmd = ((action->cmd_tbl)[option]).cmd;
   flags = ((action->cmd_tbl)[option]).flags;

   if (option == 0 && ftype ==  T_IFOSCRPT) {
        path = _dircat (dw->dir.directory_path, item->item_name);
        if (!path) {
           fprintf (stderr, "XwDirInvokeAction: _dircat failed\n");
           return;
        }

	flags = OpenScriptGetFlags (path);
	free (path);
	if (flags <= 0)
	      flags = ((action->cmd_tbl)[option]).flags;
   }

   if (!cmd)
	return;

   cmd = strdup (cmd);

   if (!cmd) {
	fprintf (stderr, "XwInvokeAction: strdup failed\n");
	return;
   }
   /* Convert string to argv */

   argv = cnvStrToArgv (cmd);

   if (!argv) {
	fprintf (stderr, "XwInvokeAction: cnvStrToArgv failed\n");
	return;
   }

   argv = _parseargv (argv, files); /* carefull argv memory has not been freed */

   if (!argv) {
	fprintf (stderr, "XwInvokeAction:_parseargv failed\n");
	return;
   }

   XwDirExecute (dw, argv, dw->dir.directory_path, flags);
#ifdef OLD
   XwDirActionMenuDelItems (dw);
#endif
}

int XwDirIfDefActions (dw)
XwDirWidget dw;
{
  DirItemObject item;
  Action *action;
  int ftype;
  register int n = 0;
  register MenuItem *mp; 
  char *path;

   n = XwDirSelectedItemCount (dw);
   
   if (n != 1)
   	return (0);

   item = (DirItemObject) _SListFirst (dw->xedwList.xedw_h_lst);
   if (!item)
        return (0);
   	
   path = _dircat (dw->dir.directory_path, item->item_name);
   if (!path) {
#ifdef DEBUG
      fprintf (stderr, "XwDirActionMenuAddItems: _dircat failed\n");
#endif
      return (0);
             
   }
   ftype = _virtualtype (path);
   free (path);
   
   if (ftype == -1)
      return (0);


   /* check that ftype is not outside the array limits */

   action = &action_tbl[ftype];
   if (!action)
        return (0);

   mp = action->menu_item;

   if (!mp)
        return (0);
   else
   	return (1);
}

/*
 * XwDirActionMenuAddItems: add items to the actions menu
 */

XwDirActionMenuAddItems (dw, cascade, flags)
XwDirWidget dw;
Widget cascade;
int flags;
{
  DirItemObject item;
  Action *action;
  int ftype;
  register MenuItem *mp; 
  register int n = 0;
  char *path;

#ifdef OLD
   item = (DirItemObject) dw->xedwList.xedw_h_item; /* for now */
   if (!item)
        return;
#endif

   n = XwDirSelectedItemCount (dw);
   
   if (n != 1)
   	return;

   item = (DirItemObject) _SListFirst (dw->xedwList.xedw_h_lst);
   if (!item)
        return;

#ifdef OLD   	
   ftype = item->fileInfo.ftype;
   if (ftype == T_IFLNK) {
        path = _dircat (dw->dir.directory_path, item->item_name);
        if (!path) {
           fprintf (stderr, "XwDirActionMenuAddItems: _dircat failed\n");
           return;
        }
        ftype = _virtualtype (path);
	free (path);
        if (ftype == -1)
           return;
   }
#endif
   path = _dircat (dw->dir.directory_path, item->item_name);
   if (!path) {
#ifdef DEBUG
	fprintf (stderr, "XwDirActionMenuAddItems: _dircat failed\n");
#endif
	return;
   }
   ftype = _virtualtype (path);
   free (path);
   if (ftype == -1) /* invalid type or unable to get the file type */ 
        return;
   
   /* check that ftype is not outside the array limits */

   action = &action_tbl[ftype];
   if (!action)
        return;


   mp = action->menu_item;

   if (!mp)
        return;

   mp = action->menu_item;
   BuildMenu(cascade, XmMENU_PULLDOWN,"", '\0', mp, flags);

}

#ifdef OLD
/*
 * XwDirActionMenuDelItems: delete items from the actions menu
 */

XwDirActionMenuDelItems (dw)
XwDirWidget dw;
{
   register Widget *widgets = dw->dir.widgets;

   if (!widgets)
	return;

   while (*widgets)
        XtDestroyWidget (*widgets++);

   free (dw->dir.widgets);
   dw->dir.widgets = NULL;

}
#endif
/*
 * confirmcmd: post confirmation dialog
 */

confirmcmd (dw, client, flags)
XwDirWidget dw;
caddr_t client;
int flags;
{
  static XwDirWidget widget;
  static int eflags;
  static char **argv;
  static char msg[] = "This action might overwrite existent files. Do you want to continue?";

   if (client) {
	widget = dw;
	argv = (char **) client;
	eflags = flags;
	_XmCConfirmationDialogPost (dw,
                msg, "Confirmation", confirmcmd, NULL,  NULL, NULL);
	return (0);
   }
   DisplayWaitCursor(widget);
   XwDirExecute1 (widget, argv, eflags);
   DisplayNormalCursor(widget);
   return (0);
}

/*
 * XwDirExecute: execute command (ask for confirmation when needed)
 */

int XwDirExecute (dw, argv, cwd, flags)
XwDirWidget dw;
char **argv;
char *cwd;
int flags;
{
   if (!dw || !argv || !cwd)
	return (-1);

   if (!strcmp(*argv, cmdtbl[MV]) ||
          !strcmp(*argv, cmdtbl[CP]))
	return (verifymv (dw, argv, cwd, flags));

   if (flags & _CONFIRM)
	return (confirmcmd (dw, argv, flags));

   DisplayWaitCursor(dw);
   XwDirExecute1 (dw, argv, flags);
   DisplayNormalCursor(dw);
   return (0);
}

/*
 * verifymv: ask for confirmation before clobbering files
 */

int verifymv (dw, client, cwd, flags)
XwDirWidget dw;
caddr_t client;
char *cwd;
int flags;
{
   static char **argv = NULL, **ap = NULL;
   static char *wd = NULL;
   static int  argc = 0;
   static char *target = NULL;
   static char confirm[] = ":File already exists. Do you want to overwrite it?";
   static XwDirWidget widget = NULL;
   char *msg; 
   char *source;
   static int  eflags;	/* execution flags */
   int cpflag = 0;
   

   if (client) {
	if (!cwd)
	   return (-1);
	ap = argv = (char **) client;
	wd = cwd;
	argc = 0;
	widget = dw;
	eflags = flags;
	while (*ap++)
	   argc++; 
	ap = argv;
	if (!strcmp(*argv, cmdtbl[CP]))
	  cpflag++;
	if ((cpflag && argc < 4) || (!cpflag && argc < 3))
	   return (-1);
	if (!strcmp (argv[argc-1], "."))
	   target = cwd;
	else if (!strcmp (argv[argc-1], "..")) {
	   target = _dircat (cwd, "..");
	   if (!target)
		return (-1);
	} else if (*(argv[argc-1]) == '/') {
	   target = argv[argc-1];
	   if (!target)
	   	return (-1);
	} else {
	   target = _dircat (cwd, argv[argc-1]);
	   if (!target)
	   	return (-1);
	}
	if (!_isdir (target)) {
	   if (cpflag && (argc > 4))
		return (1); /* too many parameters */
	   if (!cpflag && (argc > 3))
		return (1); /* too many parameters */
	   if (_exists (target)) {
		argc = 0;
		msg = _strconcat (target, confirm);
		if (!msg)
		   return (-1);
           	_XmCConfirmationDialogPost (dw,
           	msg, "Confirmation", verifymv, NULL,  NULL, NULL);
		free (msg);
		return (0);
		
	   }
		
	}
	ap++;
	argc-=2;
        if (cpflag) {
	   ap++;
	   argc--;
	}
   }

   while (argc--) {
	source = _basename (*ap);
	ap++;
	if (!source)
	   return (-1);
	source = _dircat (target, source);
	if (!source)
	   return (-1);
	if (_exists (source)) {
                msg = _strconcat (source, confirm);
		if (!msg)
		   return (-1);
                _XmCConfirmationDialogPost (widget,
                msg, "Confirmation", verifymv, NULL,  NULL, NULL);
                free (msg);
		return (0);
	}
   }

   /* execute */
   DisplayWaitCursor(widget);
   XwDirExecute1 (widget, argv, eflags);
   DisplayNormalCursor(widget);
   return (0); /* for now */

}


static struct ddstr {
XwDirWidget dw;
char *dir;
} st;

/*
 * deldir: delete directory
 */

static deldir (dw, client)
XwDirWidget dw;
caddr_t client;
{
   char *msg;
   struct ddstr *stp = (struct ddstr *) client;

   if (!stp->dir)
	return;

   if (rmdir (stp->dir) < 0)
      _XmCPerror (stp->dw, stp->dir, "Warning");
   free (stp->dir);
   stp->dir = NULL;
}


/*
 * XwDirTrashFile: trash file
 */

XwDirTrashFile (dw, path)
XwDirWidget dw;
char *path;
{
   char **argv;
   char **ap;
   int status;

	argv = (char **) malloc (6 * sizeof (char *));

	if (!argv)
	   return (-1);

	argv[0] = strdup ("/bin/mv");
	argv[1] = strdup ("mv");
	argv[2] = strdup ("-f");
	argv[3] = strdup (path);
	argv[4] = strdup (_trashpath);
	argv[5] = '\0';

	ap = argv;
	while (*ap)
	  ap++;

	if ((ap - argv) < 5)
	   return (-1);		/* does not free anything */

        status = executev (argv, NULL, 1);
	_freeargv (argv);

	if (status < 0)
	   return (-1);

   	if (WIFEXITED (status)) {
           if (WEXITSTATUS(status)) {
		return (-1);
           } 
   	} else { 
		return (-1);
   	} 
	return (1);

}


/*
 * XwDirTrashFiles: trash files
 */

int XwDirTrashFiles (w, client)
XwDirWidget w;
caddr_t client;
{
    static char **argv, **ap;
    static char *target;
    char *msg;
    char *msg1;
    char *source;
    static XwDirWidget dw;

        if (client) {
	   ap = argv = (char **) client;
	   dw = w;
	}
		
	while (*ap) {
	   target = _dircat (_trashpath, *ap);

	   if (_exists(target)) {
		msg = _strconcat (*ap, ": there is a file with this name in the trash.\nPlease empty the trash and try again."); 
		if (!msg) {
		   free (argv);
		   return (-1);
		}
		if (!*(ap+1)) {
		   _XmCErrorDialogPost (dw, msg, "Warning");
		   free (argv);
		   return (0);
		}
		msg1 = _strconcat (msg, 
		"\nDo you want to continue with the rest of the files ?");
		free (msg);
		if (!msg1) {
		   free (argv);
		   return (-1);
		}
		ap++;
                _XmCConfirmationDialogPost (dw,
                msg1, "Confirmation", XwDirTrashFiles, NULL,  NULL, NULL);
		free (target);
		free (msg1);
		return (0);
	   }

	   source = _dircat (dw->dir.directory_path, *ap);

	   if (rename (source, target) == 0) {
		ap++;
		free (source);
		free (target);
		continue;
	   }
	   free (target);
	   if (errno != EXDEV) {
		free (source);
		msg = _vstrconcat (*ap, ": ",  _ustrerror (errno), ".", NULL);
		if (!msg) {
		   free (argv);
		   return (-1);
		}

                if (!*(ap+1)) {
		   free (argv);
                   _XmCErrorDialogPost (dw, msg, "Warning");
                   return (0);
                }

		ap++;
		msg1 = _strconcat (msg, 
		"\nDo you want to continue with the rest of the files ?");
		free (msg);
		if (!msg1) {
		   free (argv);
		   return (-1);
		}
                _XmCConfirmationDialogPost (dw, 
                msg1, "Warning", XwDirTrashFiles, NULL,  NULL, NULL);
		free(msg1);
		return (0);
	   }

	   if (!_isldir (source)) {
		if (XwDirTrashFile (dw, source) > 0) {
		   free (source);
		   ap++;
		   continue;
		}

		msg = _strconcat (*ap, ": Could not be moved to the trash. Check the permissions of the parent directory");
		if (!*(ap+1)) {
                      _XmCErrorDialogPost (dw, msg, "Warning");
		      free (argv);
		      return (0);
		}

	        ap++;
		msg1 = _strconcat (msg, 
		"\nDo you want to continue with the rest of the files ?");
		free (msg);
		if (!msg1) {
		   free (argv);
		   return (-1);
		}
                _XmCConfirmationDialogPost (dw, 
                msg1, "Warning", XwDirTrashFiles, NULL,  NULL, NULL);
		free(msg1);
		free (source);
		return (0);
	   }

	
	   if (_isempty (source) > 0) {
		
	       /* If it is an empty directory, advise the user */
	       /* Directory and Trash can are on different filesystems */
	       /* therefore this directory can't be moved to the trash */
	       /* This is an empty directory. Do you want me to delete it */
	       msg = _strconcat (*ap, ": This empty directory cannot be moved to the trash directory\nbecause it is on a different file system.\nDo you want to delete it instead ?");
	       st.dir = source;
	       st.dw = dw;
	       _XmCConfirmationDialogPost (dw,
               msg, "Warning", deldir, &st,  XwDirTrashFiles, NULL);
	       free (msg);
	       ap++;
	       return (0);
	   }

	   /* It it is a non-empty directory advise the user */
	   /* Directory and Trash can are on different filesystems */
	   /* therefore this directory can't be moved to the trash */
	   /* Delete all the files inside this directory and try again */
	   msg = _strconcat (*ap, ": This directory cannot be moved to the trash directory\nbecause it is on a different file system and it is not empty.\nRemove the content of it ant try again.");

	   if (!*(ap+1)) {
           	_XmCErrorDialogPost (dw, msg, "Warning");
		free (argv);
	   	free (msg);
	   	return (0);
	   }

           msg1 = _strconcat (msg,
                "\nDo you want to continue with the rest of the files ?");
           free (msg);
           if (!msg1) {
		free (argv);
		return (-1);
	   }
	   ap++;
	   _XmCConfirmationDialogPost (dw,
                msg1, "Warning", XwDirTrashFiles, NULL,  NULL, NULL);
	   free(msg1);
	   free (source);
	   return (0);

	}

}

/*
 * XwDirEmptyTrash: empty trash
 */

static void XwDirEmptyTrash (dw, event, params, num_params)
XwDirWidget dw;
XEvent *event;
String *params;
Cardinal *num_params;
{
   char *path;
   extern char *_emptydir ();
   char *message;

   if (strcmp ("trash", _basename (_trashpath))) {
	fprintf (stderr, "XwDirEmptyTrash: The trash directory must be named 'trash'\n");
	return;
   }

   DisplayWaitCursor(dw);
   if (path = _emptydir (_trashpath)) {
   	DisplayNormalCursor(dw);
	message = _vstrconcat (path, ": ", _ustrerror (errno), ".\n",
"You might have to drag this item out of the trash\nin order to solve the problem. Check the protections.", NULL);
	if (!message) {
	  free (path);
	  return;
	}
   	_XmCErrorDialogPost (dw, message, "Warning");
   	free (message);
	free (path);
   }
   DisplayNormalCursor(dw);
}
