/*
 * Freedom Desktop
 * Copyright 1994 by Freedom Software
 *
 * Freedom Software retains all rights to Freedom Desktop (hereafter Software)
 * in binary and in source code form.
 *
 * The commercial use of this Software shall be governed by a separate License
 * agreement. Any individual or institution wishing to make commercial use of
 * the Software must sign a license agreement with Freedom Software. In such
 * cases, the Licensee agrees to abide by the terms contained in the License
 * Agreement and not those contained in this document. Examples of commercial
 * use include (without limitation): (i) integration of the Software (source
 * code form), in whole or in part, into a commercial product sold by or on
 * on behalf of the Licensee; (ii) distribution of the Software (binary form or
 * source code form) in combination with a commercial product sold by or on
 * behalf of the Licensee.
 *
 * Freedom Software (Licensor) grants you (Licensee) a license: (i) to use,
 * copy and make changes and improvements to this Software for licensee's
 * internal business purposes; (ii) to use, copy, and distribute this Software
 * or the derivative works provided that the copyright notice and this
 * permission notice appear on all copies and that NO CHARGE is associated
 * with such copies. However, if Licensee distributes any derivative work
 * based on the Software, then Licensee shall (i) notify Licensor in writing
 * (ii) clearly state that such derivative work is a modified and not the
 * original Freedom Desktop distributed by Freedom Software (iii) publish
 * the corresponding machine-readable source code or information as to
 * where it may be obtained. Each time Licensee redistribute the Software
 * or any derivative work, the recipient automatically agrees to abide
 * by the same terms as the Licensee. Licensee may not impose terms
 * more restrictive than the terms granted herein.
 *
 * By using, copying, modifying or distributing this Software (or any
 * derivative work based on this Software) Licensee indicates acceptance
 * of the terms and conditions set forth in this License.
 *
 * Licensor reserves the right to terminate this License immediately on written
 * notice, for material breach by the Licensee.
 *
 * FREEDOM SOFTWARE DISCLAIMS ALL WARRANTIES EXPRESS OR IMPLIED WITH REGARD
 * TO THIS SOFTWARE INCLUDING BUT NOT LIMITED TO ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND  FITNESS,  IN  NO  EVENT  SHALL LICENSOR BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE
 */

/*
 * Library to handle entry forms 
 */

#include <stdio.h>
#include <string.h>
#include <Xm/Label.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>
#include <Xm/RowColumn.h>
#include "X.h"
#include "xentry_form.h"

#define CHECKPAR

#include <stdio.h>
#include <varargs.h>
#include <malloc.h>
#include <errno.h>


/*
 * _syserror: return system error
 */

static char *_syserror ()
{
static char unknown[] = "unknown system error";
extern int sys_nerr;
extern char *sys_errlist[];

   if (errno < sys_nerr)
        return (sys_errlist[errno]);
   else
        return (unknown);

}

/*
 * _vstrconcat: concatenate the components of a variable argument list
 */

static char *_vstrconcat (va_alist) va_dcl
{
   va_list ap;
   register char *str, *str1, *result;
   register int length = 0;
   static int charsize = sizeof (char);

   va_start (ap);

   while (str = va_arg (ap, char *))
	length += strlen (str);
	
   va_end (ap);

   if (!length) {
	fprintf (stderr, "_strconcat: invalid parameters\n");
	return (NULL);
   }

   if (!(result = (char *) malloc ((length+1) * charsize))) {
	fprintf (stderr, "_strconcat: not enough memory\n");
	return (NULL);
   }

   *result = '\0';
   str1 = result;

   va_start (ap);

   while (str = va_arg (ap, char *)) {
	strcpy (str1, str);
        str1 += strlen (str);
   }

   va_end (ap);
   return (result);
}


/*
 * _strconcat: concatenate two strings
 */

static char *_strconcat (s1, s2)
char *s1, *s2;
{
   register char *result, *sp;
   register int length = 0;
   static int charsize = sizeof (char);

   if (s1)
	length += strlen (s1);

   if (s2)
	length += strlen (s2);

   if (!length) {
        fprintf (stderr, "_strconcat: invalid parameters\n");
        return (NULL);
   }

   if (!(result = (char *) malloc ((length+1) * charsize))) {
        fprintf (stderr, "_strconcat: not enough memory\n");
        return (NULL);
   }

   *result = '\0';
   sp = result;

   strcpy (sp, s1);
   if (s1)
	sp += strlen (s1);

   strcpy (sp, s2);
   return (result);
}


/*
 * _EntryFormCreate: create an entry form
 */

EntryFormObject _EntryFormCreate () 
{
EntryFormObject ef;
static size_t dfsize = _DFSIZE; 

   if (!(ef = (EntryFormObject) malloc (dfsize))) {
        fprintf (stderr, "_EntryFormCreate: Not enough memory\n");
        return (NULL);
    }
    memset ((char *) ef, '\0', dfsize);

   /* Start the form with enough room for 'NENTRIES' data fields */
   /* If needed increase using chunks of 'NENTRIES'              */
   ef->field_list = (_LST *) _ListCreate (NENTRIES, dfsize, NENTRIES);

   /*
    * Create the Field list
    */

   if (!ef->field_list) {
	free (ef);
	fprintf (stderr, "_EntryFormCreate: Not enough memory\n");
	return (NULL);
   }

   ef->entry_form.dfcnt = 0;	/* Number of data fields */

   return (ef);
} /* _EntryFormCreate */

/*
 * _EntryFormAddField: add field to an entry form
 */

_EntryFormAddField (ef, df)
EntryFormObject ef;
DataFieldObject df;
{

   if (!ef || !df) {
	fprintf (stderr, "_EntryFormAddField: Invalid parameters\n");
	return;
   }

   _ListAppend (ef->field_list, (char *) df); 

}

/*
 * cmp: compare data fields based on 'pos'
 */

static int cmp (e1, e2)
char *e1, *e2;
{
 DataFieldObject df1 = (DataFieldObject) e1;
 DataFieldObject df2 = (DataFieldObject) e2;

   return (df1->data_field.pos - df2->data_field.pos);
}

/*
 * _EntryFormFieldSetAppendMode: set append mode - (i.e. new values
 *				 should be appended to the content
 *				 of the field)
 */

_EntryFormFieldSetAppendMode(df, state)
DataFieldObject df;
int state;
{
   if (!df) {
	fprintf (stderr, "_EntryFormFieldSetAppendMode: Invalid parameters\n");
	return;
   }

   if (!state) {
	df->data_field.status &= ~_APPEND;
	return;
   }

   if ((df->data_field.widget_type == _XIsLABEL ||
	df->data_field.widget_type == _XIsTEXT)) {
        df->data_field.status |= _APPEND;
	if (!df->data_field.def_value)
		df->data_field.def_value = _EntryFormSGetField (df);
   } else
	fprintf (stderr, "_EntryFormFieldSetAppendMode: Invalid widget type\n");

}

/*
 * _EntryFormSInsertField: Insert a field
 */

_EntryFormSInsertField (ef, widget, pos)
EntryFormObject ef;
Widget widget;
size_t pos;
{

   static DataFieldRec dr;

   if (!ef || !widget || pos < 0) {
        fprintf (stderr, "_EntryFormSInsertField: Invalid parameters\n");
        return;
   }
   dr.data_field.widget = widget;
   dr.data_field.pos = pos;
   dr.data_field.widget_type = widgetType (widget);

   if (dr.data_field.widget_type == _XIsOPTION_MENU)
   	dr.data_field.def_value = _EntryFormSGetField (&dr); 
   else
	dr.data_field.def_value = NULL;

   if (dr.data_field.widget_type == _XIsUNKNOWN)
   	dr.data_field.status = _SENSITIVE|_MANAGED; 
   else
   	dr.data_field.status = _IN_OUT|_SENSITIVE|_MANAGED; 

   if (dr.data_field.widget_type == _XIsUNKNOWN)
       dr.data_field.value = NULL; 
   else {
       dr.data_field.value = _EntryFormSGetField (&dr); /* check for NULL */

       if (!dr.data_field.value) {
        fprintf (stderr, "_EntryFormSInsertField: _EntryFormSGetField failed\n") ;
       }
   }

   ef->entry_form.dfcnt++;
   _ListInsertO (ef->field_list, (char *) &dr, cmp);

   if (!ef->entry_form.shell)
      ef->entry_form.shell = locateShell (widget);

}

/*
 * widgetType: get the widget type
 */

int widgetType (widget) 
Widget widget;
{
   unsigned char rcType;

   if (XmIsToggleButton (widget))
	return (_XIsTOGGLE);

   if (XmIsLabel (widget))
	return (_XIsLABEL);

   if (XmIsText (widget))
	return (_XIsTEXT);

   if (XmIsRowColumn (widget)) {
	XtVaGetValues (widget, XmNrowColumnType, &rcType, NULL);
	if (rcType == XmMENU_OPTION)
	   return (_XIsOPTION_MENU);
	else
   	   return (_XIsUNKNOWN);
   }

   if (XmIsList (widget))
	return (_XIsLIST);

   return (_XIsUNKNOWN);
}


/*
 * _EntryFormPosToWidget: get widget ID
 */

Widget _EntryFormPosToWidget (ef, n)
EntryFormObject ef;
int n;          /* Field position - starting with 0 - all of the fields count */
{
register DataFieldObject df;
register int cnt = 0;

   if (!ef) {
        fprintf (stderr, "EntryFormPosToWidget: Invalid parameters\n");
        return (NULL);
   }

   df = (DataFieldObject) _ListFirst(ef->field_list);
   /* move to the right field in the field list */
   while (df) {
        if (df->data_field.pos == n) {
                return (df->data_field.widget);
        }
        df = (DataFieldObject) _ListNext(ef->field_list);
   }


   fprintf (stderr, "EntryFormPosToWidget: Could not get widget ID\n");
   return (NULL);
}

/*
 * _EntryFormPosToField: given the field position get the corresponding 
 *			 data field
 */

DataFieldObject _EntryFormPosToField (ef, n)
EntryFormObject ef;
int n;          /* Field position */
{
DataFieldObject df;

   if (!ef) {
        fprintf (stderr, "EntryFormPosToField: Invalid parameters\n");
        return (NULL);
   }

   df = (DataFieldObject) _ListFirst(ef->field_list);
   /* move to the field in the field list */
   while (df) {
        if (df->data_field.pos == n) { 
                return (df);
        }
        df = (DataFieldObject) _ListNext(ef->field_list);
   }


   fprintf (stderr, "EntryFormPosToField: Could not get field handle\n");
   return (NULL);
}


/*
 * _EntryFormOMenuDelItems: destroy menu items
 * bug: does not handle submenus properly
 */

_EntryFormOMenuDelItems (df)
DataFieldObject df;
{
   Widget *widgets, nullWidget = NULL;

   if (!df) {
        fprintf (stderr, "_EntryFormOMenuDelItems: Invalid parameters\n");
        return;
   }

   widgets = df->data_field.widgets;

   if (!widgets)
	return;

   while (*widgets)
        XtDestroyWidget (*widgets++);

   free (df->data_field.widgets);
   df->data_field.widgets = NULL;

   /* This does not seem to have any effect */
   XtVaSetValues (df->data_field.widget, XmNmenuHistory, nullWidget, NULL);
}

/*
 * _EntryFormOMenuAddItems: Add items to a menu
 */

_EntryFormOMenuAddItems (df, menuItems)
DataFieldObject df;
MenuItem *menuItems;
{
   MenuItem *mp = menuItems;
   int n = 0;
   Widget *widgets;

   if (!df || !menuItems) {
        fprintf (stderr, "_EntryFormOMenuAddItems: Invalid parameters\n");
        return;
   }
	
   while (mp++->label)
	n++;

#ifdef DEBUG
   fprintf (stderr, "_EntryFormOMenuAddItems: %d\n",  n);
#endif

   if (n <= 0) {
        fprintf (stderr, "_EntryFormOMenuAddItems: no menu items found\n");
	return;
   }

   widgets = (Widget *) malloc ((n + 1) * sizeof (Widget));

   if (!widgets) {
        fprintf (stderr, "_EntryFormOMenuAddItems: not enough memory\n");
	return;
   }
   memset ((char *) widgets, '\0', (n + 1) * sizeof (Widget));


   BuildMenu(df->data_field.widget, XmMENU_OPTION,"", '\0', menuItems, widgets);
   df->data_field.widgets = widgets;
   df->data_field.n_widgets = n;
   XtVaSetValues (df->data_field.widget, XmNmenuHistory, *widgets, NULL); 
   XtManageChild (df->data_field.widget);
}

/*
 * _EntryFormSChangeField: change a field value
 */

_EntryFormSChangeField (ef, newvalue, n)
EntryFormObject ef;
char *newvalue;
int n;		/* Field position - starting with 0 - all of the fields count */
{
DataFieldObject df;
int cnt = 0;

   if (!ef || !newvalue) {
        fprintf (stderr, "EntryFormChangeFieldValue: Invalid parameters\n");
        return;
   }

   df = (DataFieldObject) _ListFirst(ef->field_list);


   /* move to the right field in the field list */
   while (df) {
        if (df->data_field.pos == n) {
		_EntryFormSLoadField (df, newvalue);
		return;
        }
        df = (DataFieldObject) _ListNext(ef->field_list);
   }


   fprintf (stderr, "EntryFormChangeFieldValue: Could not change field value\n");
}


/*
 * _ListFirstDataField: get first data field
 */

DataFieldObject _ListFirstDataField (ef)
EntryFormObject ef;
{
DataFieldObject df;
int n;

   if (!ef)
        return (NULL);

   n = _ListLength (ef->field_list);

   if (!n)
        return (NULL);

   df = (DataFieldObject) _ListFirst(ef->field_list);

   while (n--) {
        if (df->data_field.type == _DATA) 
                break;

        df = (DataFieldObject) _ListNext(ef->field_list);
        if (df == NULL)
           df = (DataFieldObject) _ListFirst(ef->field_list);

   }

   if (df->data_field.type == _DATA)
        return df;
   else
        return (NULL);
}


/*
 * _ListNextDataField: get next data field
 */

DataFieldObject _ListNextDataField (ef)
EntryFormObject ef;
{
DataFieldObject df;

   if (!ef)
        return (NULL);

   df = (DataFieldObject) _ListNext(ef->field_list);

   while (df) {
        if (df->data_field.type == _DATA) 
                break;

        df = (DataFieldObject) _ListNext(ef->field_list);

   }
   return df;
}

   
/*
 * _EntryFormSPDumpPos: return value of field n
 */

char * _EntryFormSPDumpPos (ef, n)
EntryFormObject ef;
int n;
{
DataFieldObject df;

   df = _EntryFormPosToField (ef, n);
   if (!df) {
        fprintf (stderr, "_EntryFormPDumpPos: _EntryFormPosToField failed\n");
        return (NULL);
   }
   return (_EntryFormSGetField (df));
}


/*
 * _EntryFormSDumpData: extract data out of the form
 */

char **_EntryFormSDumpData (ef)
EntryFormObject ef;
{
DataFieldObject df;
register int i;
char **data_array, **tmp;

   if (!ef)
        return (NULL);

   
   /* no data fields  - nothing to do*/
   if (!(i = ef->entry_form.dfcnt)) 
	return (NULL);

   i++; /* reserve space for the NULL */

   /* create array of pointers to be returned */
   if (!(data_array = (char **) malloc (sizeof (char *) * i))) {
	fprintf (stderr, "_EntryFormDumpData: Not enough memory\n");
	return (NULL);
   }
   
   /* traverse the field list */
   df = (DataFieldObject) _ListFirst(ef->field_list);
   tmp = data_array;

   while (df) {
	if (df->data_field.status & _IN_OUT)  /* consider _IN_OUT fields only */
	   if (!(*tmp++ = _EntryFormSGetField (df))) {
                fprintf (stderr, "_EntryFormSDumpData: not enough memory\n");
                free (data_array); 
                return (NULL);
	   }
        df = (DataFieldObject) _ListNext(ef->field_list);

   }
   *tmp = NULL;
   return data_array; 
}

/*
 * _EntryFormSSave: Save content of the form
 */

int _EntryFormSSave (ef)
EntryFormObject ef;
{
DataFieldObject df;

   if (!ef)
        return (0);

   /* no data fields  - nothing to do*/
   if (!ef->entry_form.dfcnt)
        return (0);

   /* traverse the field list */
   df = (DataFieldObject) _ListFirst(ef->field_list);
   while (df) {
        if (df->data_field.status & _IN_OUT)  {/* consider _IN_OUT fields only */

	   if (df->data_field.value) {
		free (df->data_field.value);
		df->data_field.value = NULL;
	   }
           if (!(df->data_field.value = _EntryFormSGetField (df))) {
                fprintf (stderr, "_EntryFormSSave: _EntryFormSGetField failed\n");
                return (0);
           }
        }
        df = (DataFieldObject) _ListNext(ef->field_list);

   }
   return (1);

}

/*
 * _EntryFormSRestore: restore entry form values
 */

int _EntryFormSRestore (ef)
EntryFormObject ef;
{
DataFieldObject df;

   if (!ef)
        return (0);


   /* no data fields  - nothing to do*/
   if (!ef->entry_form.dfcnt)
        return (0);

   /* traverse the field list */
   df = (DataFieldObject) _ListFirst(ef->field_list);
   while (df) {
        if (df->data_field.status & _IN_OUT)  /* consider _IN_OUT fields only */
           _EntryFormSLoadField (df, df->data_field.value);
        df = (DataFieldObject) _ListNext(ef->field_list);

   }
   return (1);
}


/*
 * _EntryFormOptionNameToWidget: given a widget name, return a widget id
 */

Widget _EntryFormOptionNameToWidget (df, widget_name)
DataFieldObject df; /* option menu */
char *widget_name;
{
  Widget widget = df->data_field.widget;
  Widget child, submenu;
  size_t len;
  char *tmp;

   if  (!df || !widget_name) {
	fprintf (stderr, "_EntryFormOptionNameToWidget: invalid parameters\n");
	return (NULL);
   }

   len = strlen (widget_name);

   if (!(tmp = malloc (len+2 * sizeof (char)))) { /* NULL and "*" */
	fprintf (stderr, "_EntryFormOptionNameToWidget: not enough memory\n");
	return (NULL);
   }

   sprintf (tmp, "*%s", widget_name);

   XtVaGetValues (widget, XmNsubMenuId, &submenu, NULL);
   if (!submenu) {
	fprintf (stderr, 
		"_EntryFormOptionNameToWidget: couldn't get submenu widget\n");
	free (tmp);
	return (NULL);
   }

   child = XtNameToWidget (submenu, tmp);
   if (!child) {
	fprintf (stderr, "_EntryFormOptionNameToWidget: XtNameToWidget failed\n");
	free (tmp);
	return (NULL);
   }

   free (tmp);
   return (child);
}

/*
 * _EntryFormSLoadField: load a data field
 */

_EntryFormSLoadField (df, value)
DataFieldObject df;
char *value;
{
  Widget child, submenu;
  Widget widget = df->data_field.widget;

	
   switch (df->data_field.widget_type) {
	case _XIsLABEL:
	   if (df->data_field.status & _FIXED)
		return; 
		
	   if (!(df->data_field.status | _APPEND) || !df->data_field.def_value) {
           	XLabelSetString(widget, value);
		return;
	   }

	   value = _strconcat (df->data_field.def_value, value);
	   if (!value)
		return;

	   XLabelSetString(widget, value);
	   free (value);
	   
	   break;
	case _XIsTEXT:
	   if (df->data_field.status & _FIXED)
		return;
	   if (!(df->data_field.status | _APPEND) || !df->data_field.def_value) {
		XmTextSetString (widget, value);
		return;
	   }
	   value = _strconcat (df->data_field.def_value, value);
	   if (!value)
		return;

	   XLabelSetString(widget, value);
	   free (value);
	   break;
	case _XIsTOGGLE:
           if (strcmp (value, "ON"))
                XmToggleButtonSetState (widget, False, True);
           else
                XmToggleButtonSetState (widget, True, True);
           break;
	case _XIsLIST: {
	   XmString cs;
	   int pos;

	   if (!strcmp (value, _DEFAULT_VALUE))
		break;

	   cs = XmStringCreateSimple (value);
	   if (!(pos = XmListItemPos(widget, cs))) {
		if (!(df->data_field.status | _APPEND)) { 
		   fprintf (stderr, "_EntryFormSLoadField: item not found in list\n");
	   	   XmStringFree (cs);
		   return;
		}
		XmListAddItem (widget, cs, 0);
	        XmListSelectPos (widget, 0, False);
	        XmListSetPos (widget, 0);
		return;
	   }
	   XmStringFree (cs);
	   XmListSelectPos (widget, pos, False);
	   XmListSetPos (widget, pos);
	   }
	   break;
	case _XIsOPTION_MENU:
	   if (!strcmp (value, _DEFAULT_VALUE)) {
		if (df->data_field.widgets)
		   child = *(df->data_field.widgets);
		else
		   child = NULL;
		goto out;
	   }

           XtVaGetValues (widget, XmNsubMenuId, &submenu, NULL);
           if (!submenu) {
                fprintf (stderr, "_EntryFormSLoadField: couldn't get submenu widget\n");
                return;
           }

           child = XtNameToWidget (submenu, value);
           if (!child) {
                fprintf (stderr, "_EntryFormSLoadField: XtNameToWidget failed\n");
                return;
           }

out:
           XtVaSetValues (widget, XmNmenuHistory, child, NULL);
	   break;
	   
   }

}

#define _strdup(s)	(s?strdup(s):NULL)

/*
 * _EntryFormSGetField: get the value of a data field
 */

char *_EntryFormSGetField (df)
DataFieldObject df;
{
  Widget child;
  Widget widget = df->data_field.widget;
  char *optionName;

   switch (df->data_field.widget_type) {
        case _XIsLABEL:
           return (XLabelGetString(widget));
        case _XIsTEXT:
           return (XmTextGetString(widget));
        case _XIsTOGGLE:
	   return (XmToggleButtonGetState (widget)?strdup("ON"):strdup("OFF"));
        case _XIsOPTION_MENU:
           XtVaGetValues (widget, XmNmenuHistory, &child, NULL);
	   if (!child) {
		return (strdup(""));
	   }
	   if (!(optionName = XtName (child))) {
		fprintf (stderr, "_EntryFormSGetField: XtName failed\n"); 
		return (strdup(""));
	   } else
	        return (_strdup (optionName));
	case _XIsLIST: {
	   XmString *selItems;
	   int nsel;
	   char *text;

	   XtVaGetValues (widget, XmNselectedItems, &selItems, 
			XmNselectedItemCount, &nsel, NULL);

	   if (nsel == 0)
		return (strdup(""));
	   if (!_XmCStringToStr (*selItems, &text))
		return (strdup(""));
	   	return (text);
	   }
	   break;
	default:
	   return (NULL);
   }

}


/*
 * _EntryFormLoadWidget: load a value into a widget
 */

_EntryFormLoadWidget (widget, value)
Widget widget;
char *value;
{
  Widget child, submenu;

        if (XmIsToggleButton (widget)) {
           if (value == NULL)
                XmToggleButtonSetState (widget, False, True);
           else
                XmToggleButtonSetState (widget, True, True);

        }

        if (XmIsLabel (widget)) {
           XLabelSetString(widget, value);
           return;
        }

        if (XmIsText (widget)) {
           XmTextSetString (widget, value);
           return;
        }

	if (XmIsRowColumn (widget)) {
	   XtVaGetValues (widget, XmNsubMenuId, &submenu);
	   if (!submenu) {
		fprintf (stderr, "_EntryFormLoadWidget: couldn't get submenu widget\n"
);
                return;
	   }

	   child = XtNameToWidget (submenu, value);
	   if (!child) {
		fprintf (stderr, "_EntryFormLoadWidget: XtNameToWidget failed\n");
		return;
	   }
		
	   XtVaSetValues (widget, XmNmenuHistory, child, NULL);
	   return;
	}

}


/*
 * _EntryFormSLoadData: load data into the form
 */

EntryFormObject _EntryFormSLoadData (ef, data)
EntryFormObject ef;
char **data; /* null terminated array of pointers to string */
{
DataFieldObject df;
char **tmp;

   if (!ef)
        return (NULL);

   
   if (!ef->entry_form.dfcnt)
	return (NULL);

   /* traverse the field list */
   df = (DataFieldObject) _ListFirst(ef->field_list);
   tmp = data;

   while (df) {
	   
	if (!*tmp) { /* array has less entries than the form */
		     /* this might be a problem	        */
	   return (ef);
	}
        if (df->data_field.status & _IN_OUT)  /* consider _IN_OUT fields only */
	   _EntryFormSLoadField (df, *tmp++);

        df = (DataFieldObject) _ListNext(ef->field_list);

   }
   return ef;
}

/*
 * Clear entry form - does not affect the screen
 */

void _EntryFormSClear (ef)
EntryFormObject ef;
{
DataFieldObject df;

   if (!ef)
        return;

   
   if (!ef->entry_form.dfcnt)
	return;

   df = (DataFieldObject) _ListFirst(ef->field_list);

   while (df) {
        if (df->data_field.status & _IN_OUT) {
	   _EntryFormSClearField (ef, df);
	}

        df = (DataFieldObject) _ListNext(ef->field_list);

   }
}

/*
 * _EntryFormPopdown: popdown entry form
 */

_EntryFormPopdown (ef)
EntryFormObject ef;
{
#ifdef CHECKPAR
   if (!ef)
        fprintf (stderr, "_EntryFormPopdown: invalid parameters\n");
#endif

   if (!ef->entry_form.shell) {
        fprintf (stderr, "_EntryFormPopdown: couldn't get shell handle\n");
	return;
   }

   XtPopdown (ef->entry_form.shell);

}

/*
 * _EntryFormPopup: popup entry form
 */

_EntryFormPopup (ef)
EntryFormObject ef;
{
#ifdef CHECKPAR
   if (!ef)
        fprintf (stderr, "_EntryFormPopdown: invalid parameters\n");
#endif

   if (!ef->entry_form.shell) {
        fprintf (stderr, "_EntryFormPopdown: couldn't get shell handle\n");
	return;
   }

   XtPopup (ef->entry_form.shell, XtGrabNone);

}


/*
 * clearUnmanagedField: clear unmanaged fields
 */

int clearUnmanagedField (df)
DataFieldObject df;
{

   if (df->data_field.status & _IN_OUT && 
		!(df->data_field.status & _FIXED)) { /* check this */
	if (!XtIsManaged (df->data_field.widget) || !XtIsManaged (XtParent(df->data_field.widget)))
	   _EntryFormSClearField (ef, df);
   }
}

/*
 * _EntryFormTraverse: traverse an entry form
 */

void _EntryFormTraverse (ef, operation)
EntryFormObject ef;
int (*operation) ();
{
DataFieldObject df;

   if (!ef)
        return;


   if (!ef->entry_form.dfcnt)
        return;

   df = (DataFieldObject) _ListFirst(ef->field_list);

   while (df) {
	operation (df);

        df = (DataFieldObject) _ListNext(ef->field_list);
   }
}


/*
 * _EntryFormInvisibleField: make a field invisible
 */

void _EntryFormInvisibleField (df)
DataFieldObject df;
{
   if (!(df->data_field.saved_foreground))
	XtVaGetValues (df->data_field.widget, XmNforeground, 
	&(df->data_field.saved_foreground), NULL);
   if (!(df->data_field.saved_background))
	XtVaGetValues (df->data_field.widget, XmNbackground, 
	&(df->data_field.saved_background), NULL);
   XtVaSetValues (df->data_field.widget, XmNforeground, 
	df->data_field.saved_background, NULL);
}

/*
 * _EntryFormCDestroyData: destroy the entry form data vector
 */

_EntryFormCDestroyData (data)
char **data;
{
   char **tmp = data;

   if (!tmp)
	return;

   while (*tmp) {
	free (*tmp);
	*tmp++ = NULL; /* just in case */
   }
   free (data);
}

/*
 * _EntryFormPerror: popup a system error message
 */

_EntryFormPerror (ef,msg)
EntryFormObject ef;
char *msg;
{
  char *message;

   message = _vstrconcat (msg, ":", _syserror (), NULL);
   if (!ef->entry_form.shell || !message) 
     perror (msg);

   _XmCErrorDialogPost (ef->entry_form.shell, message, "Warning");
   free (message);
}

/*
 * _EntryFormWarning: popup a system warning
 */

_EntryFormWarning (ef,msg)
EntryFormObject ef;
char *msg;
{
   if (ef->entry_form.shell)
     _XmCErrorDialogPost (ef->entry_form.shell, msg, "Warning");
   else
     fprintf (stderr, "%s\n", msg); 

}



