/*
 * Freedom Desktop
 * Copyright 1994 by Freedom Software
 *
 * Freedom Software retains all rights to Freedom Desktop (hereafter Software)
 * in binary and in source code form.
 *
 * The commercial use of this Software shall be governed by a separate License
 * agreement. Any individual or institution wishing to make commercial use of
 * the Software must sign a license agreement with Freedom Software. In such
 * cases, the Licensee agrees to abide by the terms contained in the License
 * Agreement and not those contained in this document. Examples of commercial
 * use include (without limitation): (i) integration of the Software (source
 * code form), in whole or in part, into a commercial product sold by or on
 * on behalf of the Licensee; (ii) distribution of the Software (binary form or
 * source code form) in combination with a commercial product sold by or on
 * behalf of the Licensee.
 *
 * Freedom Software (Licensor) grants you (Licensee) a license: (i) to use,
 * copy and make changes and improvements to this Software for licensee's
 * internal business purposes; (ii) to use, copy, and distribute this Software
 * or the derivative works provided that the copyright notice and this
 * permission notice appear on all copies and that NO CHARGE is associated
 * with such copies. However, if Licensee distributes any derivative work
 * based on the Software, then Licensee shall (i) notify Licensor in writing
 * (ii) clearly state that such derivative work is a modified and not the
 * original Freedom Desktop distributed by Freedom Software (iii) publish
 * the corresponding machine-readable source code or information as to
 * where it may be obtained. Each time Licensee redistribute the Software
 * or any derivative work, the recipient automatically agrees to abide
 * by the same terms as the Licensee. Licensee may not impose terms
 * more restrictive than the terms granted herein.
 *
 * By using, copying, modifying or distributing this Software (or any
 * derivative work based on this Software) Licensee indicates acceptance
 * of the terms and conditions set forth in this License.
 *
 * Licensor reserves the right to terminate this License immediately on written
 * notice, for material breach by the Licensee.
 *
 * FREEDOM SOFTWARE DISCLAIMS ALL WARRANTIES EXPRESS OR IMPLIED WITH REGARD
 * TO THIS SOFTWARE INCLUDING BUT NOT LIMITED TO ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND  FITNESS,  IN  NO  EVENT  SHALL LICENSOR BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE
 */

/*
 * List object. 
 */

#include "List.h"
 
void LSTInitialize ();
void LSTDestroy ();
void LSTSetValues ();
void LSTProcessMessage ();
static void list_add ();
static Obj *allocate_slot ();
static Obj *list_find ();
static Obj *list_first ();
static Obj *list_next ();
static Obj *list_free_slot ();
static void list_delete ();
static int compare_resource ();
static Obj *list_ofind ();
#ifdef DEBUG
static void list_print ();
#endif
static void list_clear ();
static void broadcast_msg ();
static void destroy_objects ();

static RtResource resources[] = {
   {RtNlistCurrent, NULL, RtOBJ_TYPE, sizeof(Obj), 
   RtOffset(ListObj,lst.current), 0, NULL, 0, NULL }, 
   {RtNlistOrdered , NULL, RtBIT_TYPE, 1, 
   RtOffset(ListObj,lst.flags), 0, NULL, 0, 
   RtLIST_ORDERED},   
   {RtNlistKey, NULL, RtSTRING_TYPE, sizeof(char *), 
   RtOffset(ListObj,lst.key), 0, NULL, 0, NULL }, 
   {RtNlistCurrentPos, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(ListObj,lst.current_pos), 0, NULL, 0, NULL }, 
   {RtNlistDestroyObjects , NULL, RtBIT_TYPE, 1, 
   RtOffset(ListObj,lst.flags), 0, NULL, 0, 
   RtLIST_DESTROY_OBJECTS},   
};
  
ListClassRec listClassRec = {
  {
    &objClassRec,		/* superclass		*/
    "List",			/* class name		*/
    NULL,	
    0,
    resources,			/* resource table	*/
    RtNumber (resources),	/* number of resources  */
    /* object size              */  sizeof(ListRec),
    LSTInitialize,
    NULL,
    NULL,
    LSTSetValues,
    NULL,
    LSTDestroy,
    LSTProcessMessage,
    "list",
    "Rt1.00",
    NULL
  },
  {
  0,
  }
  
};

ObjClass listObjClass = (ObjClass) &listClassRec; 

/*
 * LSTSetValues: set object resources
 */
 
void LSTSetValues (current, new)
ListObj current, new;
{
   /* Check - This should be done by the superclass */
   
   if (current->core.object_name != new->core.object_name)
        RtUpdateString (current->core.object_name, 
        	new->core.object_name);
   RtUpdateString (current->lst.key, new->lst.key);
}


/*
 * LSTInitialize: initialize object
 */

void LSTInitialize (obj)
ListObj obj;
{
#ifdef DEBUG
   obj->core.object_name = strdup ("List Object");
#endif
}


/*
 * LSTDestroy: destroy object
 */
 
void LSTDestroy (obj)
ListObj obj;
{
   RtDestroyString (obj->lst.key);
   
   if (obj->lst.obj_list) {
   	free (obj->lst.obj_list);
   	obj->lst.obj_list = NULL;
   }
}

/*
 * LSTProcessMessage: process object messages
 */
 
void LSTProcessMessage (obj, data, client)
ListObj obj;
void *data;
char *client;
{
MessageObj msg;
char *content = NULL;
int flags;
int message_id;
Obj *tmp;
char *msg_client;

   msg = (MessageObj) data;
   RtGetValue (msg, RtNmsgId, &message_id);   
   RtGetValue (msg, RtNmsgContent, &content);   
   RtGetValue (msg, RtNmsgFlags, &flags);   
   
   /* ignore the function paramenter  		*/
   /* get the client info from the message 	*/
   RtGetValue (msg, RtNmsgClient, &msg_client); 

   switch (message_id) {
      case RtLIST_ADD:
      	if (!content)
      	  return;
      	list_add (obj, content);
#ifdef DEBUG
      	list_print (obj);
#endif      	
      	break;
      case RtLIST_FIRST:
        tmp = list_first (obj);
        obj->lst.curr_addr = tmp;
        obj->lst.current = tmp ? *tmp: NULL;
        break;      	 
      case RtLIST_NEXT:
        tmp = list_next (obj);
        obj->lst.curr_addr = tmp;
        obj->lst.current = tmp ? *tmp: NULL;
        break;      	 
      case RtLIST_FIND:
      	if (!msg_client && !content)
      	   return;
      	   
      	if (obj->lst.flags & RtLIST_ORDERED)
           tmp = list_ofind (obj, content, msg_client);
        else
           tmp = list_find (obj, content, msg_client);
        obj->lst.curr_addr = tmp;
        obj->lst.current = tmp ? *tmp: NULL;
        obj->lst.current_pos = tmp ? (tmp - obj->lst.obj_list): -1;
        break;
      case RtLIST_DELETE:
      	if (!obj->lst.current)
      	   return;
        list_delete (obj);
#ifdef DEBUG
      	list_print (obj);
#endif      	
        break;
      case RtLIST_CLEAR:
        list_clear (obj); 
        break;     
        break;
      case RtBROADCAST_MESSAGE:
        broadcast_msg (obj, content);
        break;      
      default:
#ifdef DEBUG
   	fprintf (stderr, 
   	"LSTProcessMessage: I don't know how to process this message\n");
#endif
	/* Invoke the superclass to process the message */
	RtSuperclassProcessMessage (listObjClass,
		obj, data, client); 
        break;
   }
}


/*
 * compare_resource: resource comparison
 */
 
static int compare_resource (current, new_obj, key)
Obj current;
Obj new_obj;
char *key;
{

char *value = NULL, *value1 = NULL;

   RtGetValue (current, key, &value);
   RtGetValue (new_obj, key, &value1);
   
   if (!value) {
   	value = "";
   }

   if (!value1) {
   	value1 = "";
   }
   
   /*
    * limitation: this works for strings only
    */
    
   return (strcmp (value1, value));
   
}



/*
 * find_insertion_point: find the slot where the new element should
 *			be inserted
 */
 
static Obj *find_insertion_point (obj, new_obj)
ListObj obj;
Obj new_obj;
{
 int mid, lo, hi, cnt;
 int aux;
 Obj *obj_list;
 Obj current;
    
   if (!new_obj)
   	return (NULL);

   cnt = obj->lst.cnt;	
   obj_list = obj->lst.obj_list;
   
   if (!obj_list || !obj->lst.key)
	return (NULL);
   
   if (cnt == 0)
   	return (obj_list);
   	
   lo = 0;
   hi = cnt - 1;

   while (lo <= hi) {
   	mid = (lo+hi)/2;
   	
   	current = obj_list[mid];

	aux = compare_resource (current, new_obj, 
		obj->lst.key);
   
	if (!aux)
   	   return (&(obj_list[mid]));
   	
   	if (aux > 0) {
	   lo = mid+1; /* upper portion ? */	   
   	} else
   	   hi = mid-1;

   }
   return (&(obj_list[lo]));

}

/*
 * list_right_shift: shift all the elements of the list to the left
 */
 
static void list_left_shift (obj, slot)
ListObj obj;
Obj *slot;
{
   register int j = obj->lst.cnt;
   register int i;
   
   if (!slot || !obj->lst.obj_list)
   	return;
   	
   i = slot - obj->lst.obj_list;  
   j--;
   
   while (i < j) {
   	(obj->lst.obj_list)[i] =  (obj->lst.obj_list)[i+1]; 
   	i++;
   } 
   (obj->lst.obj_list)[i] = NULL; 
}


/*
 * list_right_shift: shift all the elements of the list
 */
 
static void list_right_shift (obj, slot)
ListObj obj;
Obj *slot;
{
   register int j = obj->lst.cnt;
   int i;
   
   if (!slot || !obj->lst.obj_list)
   	return;
   	
   i = slot - obj->lst.obj_list;  
   
   while (i < j) {
   	(obj->lst.obj_list)[j] =  (obj->lst.obj_list)[j-1]; 
   	j--;
   }  
}

/*
 * list_oinsert: insert an element in the list. 
 */
 
static void list_oinsert (obj, new_obj)
ListObj obj;
Obj new_obj;
{
   Obj *tmp;
   
   tmp = find_insertion_point (obj, new_obj);
   
   if (!tmp)
   	return;
   
   /*
    * Make room for the new element
    */
    
   list_right_shift (obj, tmp);
   *tmp = new_obj;
   obj->lst.cnt++;
   
}


/*
 * list_add: add an object to the list
 */
 
static void list_add (obj, new_obj)
ListObj obj;
Obj new_obj;
{
Obj *tmp;

   /*
    * Allocate memory if needed
    */
    
   tmp = allocate_slot (obj);
    
   if (!tmp)
	return;

#ifdef OLD
   /* This does not look OK */
   if (!obj->lst.key) {
   	*tmp = new_obj;
   	obj->lst.cnt++;
   	return;
   }
#endif

   list_oinsert (obj, new_obj);
   
}

/*
 * allocate_slot: allocate a chunk of memory to store the new object
 */

static Obj *allocate_slot (obj)
ListObj obj;
{
  Obj *tmp, *mp;
  static unsigned int obj_size = sizeof (Obj); 
    	
    if (!obj->lst.obj_list) {
        
           
	obj->lst.obj_list = (Obj *) malloc 
			(obj_size * RtLST_CHUNK_SIZE);

        if (!obj->lst.obj_list)       
           return;

   	memset (obj->lst.obj_list, '\0', 
   	   	obj_size * RtLST_CHUNK_SIZE);
           	    
        obj->lst.max_cnt = RtLST_CHUNK_SIZE;
    } else if (obj->lst.cnt >= obj->lst.max_cnt) {
           mp = (Obj *) realloc ((char *) obj->lst.obj_list, 
           		obj_size 
                        * (obj->lst.max_cnt + RtLST_CHUNK_SIZE));
           if (mp) {
   	   	memset (((char *) mp) + (obj_size * obj->lst.max_cnt), 
   	   		'\0', 
   	   		obj_size * RtLST_CHUNK_SIZE);
                obj->lst.obj_list = mp;
                obj->lst.max_cnt += RtLST_CHUNK_SIZE;                    
           }
                           
    }
        
    if (obj->lst.cnt >= obj->lst.max_cnt)
           return (NULL);

    if (obj->lst.key)
    	tmp = obj->lst.obj_list+obj->lst.cnt;
    else
    	tmp = list_free_slot (obj);
    	
    return (tmp);
}

/*
 * list_free_slot: traverse the object list and return the next 
 * free slot
 */
 
static Obj *list_free_slot (obj)
ListObj obj;
{
  register Obj *op;
  register int i = 0;

   if (!obj->lst.obj_list)
   	return (NULL);
   
   op = obj->lst.obj_list;	
   
   while (i++ < obj->lst.max_cnt) {
	if (!*op)
	   return (op);
	op++;	   
   }
   return (NULL);
}

/*
 * list_find: find an object in the list
 */
 
static Obj *list_find (obj, res, value)
ListObj obj;
char *res;
char *value;
{
  register int i = 0;
  char *value1;
  register Obj *op;

   if (!res || !obj->lst.obj_list)
   	return (NULL);
   
   if (!value)
   	return (NULL);
   	
   op = obj->lst.obj_list;	
   
   while (i++ < obj->lst.max_cnt) {
   	if (*op == NULL) {
   	   op++;
   	   continue;
   	}
   	value1 = NULL;
	RtGetValue (*op, res, &value1);
	if (value1 && !strcmp (value, value1)) {	   
	   return (op);
	}
	op++;	   
   }
   return (NULL);
   
}

/*
 * list_ofind: find an object in a ordered list
 */
 
static Obj *list_ofind (obj, res, value)
ListObj obj;
char *res;
char *value;
{
  int lo, hi, cnt, mid, aux;
  char *value1;
  register Obj *obj_list;
  Obj current;

   if (!res || !obj->lst.obj_list)
   	return (NULL);
   
   if (!value)
   	return (NULL);
   	
   
   cnt = obj->lst.cnt;	

   obj_list = obj->lst.obj_list;
   
   lo = 0;
   hi = cnt - 1;

   while (lo <= hi) {
   	mid = (lo+hi)/2;
   	
   	current = obj_list[mid];
   	
   	if (!current) {
   	   obj->core.status = RtINTERNAL_ERROR;
   	   break;
   	}

   	value1 = NULL;
	RtGetValue (current, res, &value1);
	if (!value1)
	   value1= "";
	   
	aux = strcmp (value, value1);
	
	if (aux > 0)
	   aux = RtGREATER_THAN;
	else if (aux < 0)
	   aux = RtLESS_THAN;
	 
#ifdef BETTER
        /* This slows things down - Wait until computers get faster */
	RtSetValue (stdmsg, RtNmsgId, RtMATCH);
	RtSetValue (stdmsg, RtNmsgContent, res);
	RtSetValue (stdmsg, RtNmsgClient, value);
	
	RtSendMessage (current, stdmsg, NULL);
	RtGetValue (current, RtNobjStatus, &aux);
	if (aux < 0)
	   return (NULL); /* an error has been detected */  
#endif

	if (!aux)
   	   return (&(obj_list[mid]));
   	
   	if (aux == RtGREATER_THAN) {
	   lo = mid+1; /* upper portion ? */	   
   	} else if (aux == RtLESS_THAN)
   	   hi = mid-1;
	else
	   return (NULL);
   }
   return (NULL);

}

/*
 * list_first: get the first object in the list
 */
 
static Obj *list_first (obj)
ListObj obj;
{
  register int i = 0;
  register Obj *op;

   if (!obj->lst.obj_list)
   	return (NULL);
   
   op = obj->lst.obj_list;	
   
   while (i++ < obj->lst.max_cnt) {
   	if (*op != NULL) {
   	   return (op);
   	}
	op++;	   
   }
   return (NULL);
   
}

/*
 * list_next: get the next object in the list
 */
 
static Obj *list_next (obj)
ListObj obj;
{
  register int i = 0;
  register Obj *op;
  

   if (!obj->lst.obj_list)
   	return (NULL);
   
   if (!obj->lst.curr_addr)
   	return (list_first (obj));
   	
   i = obj->lst.curr_addr - obj->lst.obj_list;
   op = obj->lst.curr_addr;

   while (++i < obj->lst.max_cnt) {
   	if (*++op != NULL) {
   	   return (op);
   	}
   }
   return (NULL);   
}

/*
 * list_delete: delete current
 */
 
static void list_delete (obj)
ListObj obj;
{
   if (!obj->lst.current || !obj->lst.curr_addr)
	return;

   *(obj->lst.curr_addr) = NULL;

   if (obj->lst.key)
	list_left_shift (obj, obj->lst.curr_addr); 
   obj->lst.curr_addr = obj->lst.current = NULL;
   obj->lst.cnt--;
}

#ifdef DEBUG
static void list_print (obj)
ListObj obj;
{
  register Obj *op;
  register int i = 0;

   if (!obj->lst.obj_list)
   	return;
   
   op = obj->lst.obj_list;	
   
   while (i++ < obj->lst.max_cnt) {
	fprintf (stderr, "pos = %d, obj = %d\n", i-1, *op);
	if (*op) {
	   RtSetValue (stdmsg, RtNmsgId, RtPRINT_OBJECT);
	   RtSendMessage (*op, stdmsg, NULL);
	}
	op++;	   
   }
}
#endif

/*
 * list_clear: clear (reset) a list
 */
 
static void list_clear (obj)
ListObj obj;
{
   if (obj->lst.flags & RtLIST_DESTROY_OBJECTS)
   	destroy_objects (obj);

   if (obj->lst.obj_list) {
   	free (obj->lst.obj_list);
   	obj->lst.obj_list = NULL;
   }
   obj->lst.cnt = 0;
   obj->lst.max_cnt = 0;
   obj->lst.index = 0;
   obj->lst.current = NULL;
   obj->lst.curr_addr = NULL;
   obj->lst.current_pos = 0;

}

/*
 * broadcast_msg: broadcast a message to all the elements of the
 * list
 * check - changes the current
 */
 
static void broadcast_msg (obj, msg)
ListObj obj;
MessageObj msg;
{
Obj op;
MessageObj aux;
   
   aux = RtCreateMessage ();
   
   if (!aux)
   	return;
   
   RtSetValue (aux, RtNmsgId, RtLIST_FIRST); 	
   RtSendMessage (obj, aux, NULL);
   
   while (obj->lst.current) {
	RtSendMessage (obj->lst.current, msg, NULL);
   	RtSetValue (aux, RtNmsgId, RtLIST_NEXT); 	
   	RtSendMessage (obj, aux, NULL);
   }  
   
   RtDestroyObject (aux);
}

/*
 * destroy_objects: destroy the objects in the list
 * check - changes the current
 */
 
static void destroy_objects (obj)
ListObj obj;
{
Obj op;
MessageObj aux;
   
   aux = RtCreateMessage ();
   
   if (!aux)
   	return;
   
   RtSetValue (aux, RtNmsgId, RtLIST_FIRST); 	
   RtSendMessage (obj, aux, NULL);
   
   op = obj->lst.current;
   while (op) {
   	RtSetValue (aux, RtNmsgId, RtLIST_NEXT); 	
   	RtSendMessage (obj, aux, NULL);
   	RtDestroyObject (op);
   	op = obj->lst.current;
   }  
   
   RtDestroyObject (aux);
}
