/*
 * Freedom Desktop
 * Copyright 1994 by Freedom Software
 *
 * Freedom Software retains all rights to Freedom Desktop (hereafter Software)
 * in binary and in source code form.
 *
 * The commercial use of this Software shall be governed by a separate License
 * agreement. Any individual or institution wishing to make commercial use of
 * the Software must sign a license agreement with Freedom Software. In such
 * cases, the Licensee agrees to abide by the terms contained in the License
 * Agreement and not those contained in this document. Examples of commercial
 * use include (without limitation): (i) integration of the Software (source
 * code form), in whole or in part, into a commercial product sold by or on
 * on behalf of the Licensee; (ii) distribution of the Software (binary form or
 * source code form) in combination with a commercial product sold by or on
 * behalf of the Licensee.
 *
 * Freedom Software (Licensor) grants you (Licensee) a license: (i) to use,
 * copy and make changes and improvements to this Software for licensee's
 * internal business purposes; (ii) to use, copy, and distribute this Software
 * or the derivative works provided that the copyright notice and this
 * permission notice appear on all copies and that NO CHARGE is associated
 * with such copies. However, if Licensee distributes any derivative work
 * based on the Software, then Licensee shall (i) notify Licensor in writing
 * (ii) clearly state that such derivative work is a modified and not the
 * original Freedom Desktop distributed by Freedom Software (iii) publish
 * the corresponding machine-readable source code or information as to
 * where it may be obtained. Each time Licensee redistribute the Software
 * or any derivative work, the recipient automatically agrees to abide
 * by the same terms as the Licensee. Licensee may not impose terms
 * more restrictive than the terms granted herein.
 *
 * By using, copying, modifying or distributing this Software (or any
 * derivative work based on this Software) Licensee indicates acceptance
 * of the terms and conditions set forth in this License.
 *
 * Licensor reserves the right to terminate this License immediately on written
 * notice, for material breach by the Licensee.
 *
 * FREEDOM SOFTWARE DISCLAIMS ALL WARRANTIES EXPRESS OR IMPLIED WITH REGARD
 * TO THIS SOFTWARE INCLUDING BUT NOT LIMITED TO ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND  FITNESS,  IN  NO  EVENT  SHALL LICENSOR BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE
 */

/*
 * Command: object used to execute Unix commands. This object
 *          handles signals and error messages. Based in code
 *	    taken from Nedit. 
 */
 
#include "Command.h"

/*
 * Methods
 */
 
void CMDInitialize ();
void CMDActivate ();
int CMDRealize ();
void CMDDestroy ();
void CMDSetValues ();
void CMDProcessMessage ();
char *_vstrconcat ();
char static *get_target_directory ();

static int process_xevents ();
static int empty_trash ();
static int trash_file ();
static int move_files ();
static int move_file ();
static int copy_files ();
static int copy_file ();
static int validate_cpfile ();

void execute_cmd ();
extern char *_emptydir ();
extern char *_basename ();
extern char *_dircat ();
extern char *_get_cwd ();
extern char *_dirname ();

static RtResource resources[] = {
   {RtNcmdWaitPid, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(CommandObj,cmd.wait_pid), 0, NULL, 0, NULL },   
   {RtNcmdString, NULL, RtSTRING_TYPE, sizeof(char *), 
   RtOffset(CommandObj,cmd.cmd_string), 0, NULL, 0, NULL }, 
   {RtNcmdWidget, NULL, RtPOINTER_TYPE, sizeof(Widget), 
   RtOffset(CommandObj,cmd.widget), 0, NULL, 0, NULL }, 
   {RtNcmdMessageDialog, NULL, RtOBJ_TYPE, sizeof(Obj), 
   RtOffset(CommandObj,cmd.message_dialog), 0, NULL, 0, NULL }, 
   {RtNcmdTrashPath, NULL, RtSTRING_TYPE, sizeof(char *), 
   RtOffset(CommandObj,cmd.trash_path), 0, NULL, 0, NULL }, 
   {RtNcmdSrcPath, NULL, RtSTRING_TYPE, sizeof(char *), 
   RtOffset(CommandObj,cmd.src_path), 0, NULL, 0, NULL }, 
   {RtNcmdDestPath, NULL, RtSTRING_TYPE, sizeof(char *), 
   RtOffset(CommandObj,cmd.dest_path), 0, NULL, 0, NULL }, 
   {RtNcmdArgv, NULL, RtSTRING_TYPE, sizeof(char *), 
   RtOffset(CommandObj,cmd.argv), 0, NULL, 0, NULL }, 
   {RtNcmdCwd, NULL, RtSTRING_TYPE, sizeof(char *), 
   RtOffset(CommandObj,cmd.cwd), 0, NULL, 0, NULL }, 
   {RtNcmdFlags, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(CommandObj,cmd.flags), 0, NULL, 0, NULL }, 
   {RtNcmdDefaultButton, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(CommandObj,cmd.default_button), 0, NULL, 0, NULL },        
   {RtNshellUnsensitive, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(CommandObj,cmd.shell_unsensitive), 0, NULL, 0, NULL },        
   {RtNcmdStderrIsStdout, NULL, RtBIT_TYPE, 1, 
   RtOffset(CommandObj,cmd.flags), 0, NULL, 0, 
   (char *) RtCMD_STDERR_IS_STDOUT},        
   {RtNcmdBroadcastStdout, NULL, RtBIT_TYPE, 1, 
   RtOffset(CommandObj,cmd.flags), 0, NULL, 0, 
   (char *) RtCMD_BROADCAST_STDOUT},        
   {RtNcmdStdoutUnbuffered, NULL, RtBIT_TYPE, 1, 
   RtOffset(CommandObj,cmd.flags), 0, NULL, 0, 
   (char *) RtCMD_STDOUT_UNBUFFERED},        
};

CommandClassRec commandClassRec = {
  {
    (ObjClass) &objClassRec,	/* superclass		*/
    "Command",			/* class name		*/
    NULL,	
    0,
    resources,			/* resource table	*/
    RtNumber(resources),	/* number of resources  */
    /* object size              */  sizeof(CommandRec),
    CMDInitialize,
    CMDRealize,
    NULL,
    CMDSetValues,
    CMDActivate,
    CMDDestroy,
    CMDProcessMessage,
    "cmd",
    "Rt1.00",
    NULL
  },
  {
  0,
  }
  
};

ObjClass commandObjClass = (ObjClass) &commandClassRec; 

void CMDSetValues (current, new)
CommandObj current, new;
{
   if (current->cmd.cmd_string != new->cmd.cmd_string)
   	RtUpdateString (current->cmd.cmd_string, new->cmd.cmd_string);
   if (current->cmd.trash_path != new->cmd.trash_path)
   	RtUpdateString (current->cmd.trash_path, new->cmd.trash_path);
   if (current->cmd.src_path != new->cmd.src_path)
   	RtUpdateString (current->cmd.src_path, new->cmd.src_path);
   if (current->cmd.dest_path != new->cmd.dest_path)
   	RtUpdateString (current->cmd.dest_path, new->cmd.dest_path);
   if (current->cmd.cwd != new->cmd.cwd)
   	RtUpdateString (current->cmd.cwd, new->cmd.cwd);
      	
}


void CMDInitialize (obj)
CommandObj obj;
{
   obj->cmd.process_xevents  = 1; 
   obj->cmd.default_button  = RtCANCEL_BUTTON; 
}

void CMDActivate (obj)
CommandObj obj;
{
}


int CMDRealize (obj)
CommandObj obj;
{

  /* If a signal handler hasn't been defined, do so now */
   
  if (!obj->cmd.sig_handler) {
     obj->cmd.sig_handler = 
  	RtCreateObject (NULL, signalHandlerObjClass);
     RtRealizeObject (obj->cmd.sig_handler);
  }
  /* clock cursor */
  
  if (!obj->cmd.cursor) {
     obj->cmd.cursor =  
  	RtCreateObject (NULL, cursorObjClass);
     if (!obj->cmd.widget)
        return;
     RtSetValue (obj->cmd.cursor, RtNwidget, 
     		 obj->cmd.widget);
     RtRealizeObject (obj->cmd.cursor);
  
  }
  /* If a message object  hasn't been defined, do so now */
   
  if (!obj->cmd.message_dialog) {
     obj->cmd.message_dialog = (MessageDialogObj)
  	RtCreateObject (NULL, messageDialogObjClass);
     RtSetValue (obj->cmd.message_dialog, RtNwidgetTitle, "Warning");
     RtSetValue (obj->cmd.message_dialog, RtNwidgetParent, 
     	obj->cmd.widget);
     RtRealizeObject (obj->cmd.message_dialog);
  }
}


void CMDDestroy (obj)
CommandObj obj;
{
   if (obj->cmd.cmd_string)
      free (obj->cmd.cmd_string);
      
   if (obj->cmd.trash_path)
      free (obj->cmd.trash_path);
      
   if (obj->cmd.src_path)
      free (obj->cmd.src_path);
   
   if (obj->cmd.dest_path)
      free (obj->cmd.dest_path);
      
   if (obj->cmd.cwd)
      free (obj->cmd.cwd);
      
   if (obj->cmd.cursor)
      RtDestroyObject (obj->cmd.cursor);
   if (obj->cmd.message_dialog)
      RtDestroyObject (obj->cmd.message_dialog);
   if (obj->cmd.sig_handler)
      RtDestroyObject (obj->cmd.sig_handler);
      
   free (obj);
}

int spawn_command (obj)
CommandObj obj;
{

int fd[2];
int dup_fd;
int status = 0;

    obj->cmd.restore_signals = 0; 	
    if (!obj->cmd.cmd_string && !obj->cmd.argv)
        return (1);
           
    /* Create pipes to communicate with the sub process.  One end of each is
       returned to the caller, the other half is spliced to stdin, stdout
       and stderr in the child process */
    if (pipe(fd) != 0) {
    	perror("execute_command: Internal error (opening stdout pipe)");
        return (-1);
    }

    obj->cmd.stdout_fd = fd[0];
    obj->cmd.child_stdout_fd = fd[1];

    if (pipe(fd) != 0) {
    	perror("execute_command: Internal error (opening stdin pipe)");
        return (-1);
    }

    obj->cmd.stdin_fd = fd[0];
    obj->cmd.child_stdin_fd = fd[1];

    if (pipe(fd) != 0) {
    	perror("execute_command: Internal error (opening stdin pipe)");
        return (-1);
    }

    obj->cmd.stderr_fd = fd[0];
    obj->cmd.child_stderr_fd = fd[1];

    /*
     * Ask the signal handler to ignore
     * SIGINT & SIGQUIT and block SIGCHLD
     */
     
    RtSetValue (obj->cmd.sig_handler, 
    	RtNmsgId, RtSIG_IGNORE);
    RtSetValue (obj->cmd.sig_handler, 
    	RtNmsgContent, SIGINT);
    RtSendMessage (obj->cmd.sig_handler, 
    	obj->cmd.sig_handler);

    RtSetValue (obj->cmd.sig_handler, 
    	RtNmsgContent, SIGQUIT);
    RtSendMessage (obj->cmd.sig_handler, 
    	obj->cmd.sig_handler);

    RtSetValue (obj->cmd.sig_handler, 
    	RtNmsgId, RtSIG_BLOCK);
    RtSetValue (obj->cmd.sig_handler, 
    	RtNmsgContent, SIGCHLD);
    RtSendMessage (obj->cmd.sig_handler, 
    	obj->cmd.sig_handler);
    obj->cmd.restore_signals = 1; /* restore signals at a later time */	

    obj->cmd.child_pid = fork ();
    
    if (obj->cmd.child_pid == 0) {

    /*
     * Ask the signal handler to restore
     * the status of SIGINT, SIGQUIT and
     * the signal mask 
     */
     
    	RtSetValue (obj->cmd.sig_handler, 
    	   RtNmsgId, RtSIG_RESTORE);
    	RtSetValue (obj->cmd.sig_handler, 
    	   RtNmsgContent, SIGINT);
    	RtSendMessage (obj->cmd.sig_handler, 
    	   obj->cmd.sig_handler);

    	RtSetValue (obj->cmd.sig_handler, 
    	   RtNmsgContent, SIGQUIT);
    	RtSendMessage (obj->cmd.sig_handler, 
    	   obj->cmd.sig_handler);

    	RtSetValue (obj->cmd.sig_handler, 
    	   RtNmsgContent, RtSIG_MASK_RESTORE);
    	RtSendMessage (obj->cmd.sig_handler, 
    	   obj->cmd.sig_handler);
   	
     	/* close the parent end of the pipes in the child process */
	close (obj->cmd.stdin_fd);
	close (obj->cmd.stdout_fd);    
	if (obj->cmd.stderr_fd != NULL)
	    close(obj->cmd.stderr_fd);

	/* close current stdin, stdout, and stderr file descriptors before
	   substituting pipes */
	close(fileno(stdin));
	close(fileno(stdout));
	close(fileno(stderr));

	/* duplicate the child ends of the pipes to have the same numbers
	   as stdout & stderr, so it can substitute for stdout & stderr */
 	dup_fd = dup2(obj->cmd.child_stdin_fd, fileno(stdin));
	if (dup_fd == -1)
	    perror("dup of stdin failed");
 	dup_fd = dup2(obj->cmd.child_stdout_fd, fileno(stdout));
	if (dup_fd == -1)
	    perror("dup of stdout failed");
 	dup_fd = dup2(obj->cmd.child_stderr_fd, fileno(stderr));
	if (dup_fd == -1)
	    perror("dup of stderr failed");
	
#ifdef OLD	
	/* make this process the leader of a new process group, so the sub
	   processes can be killed, if necessary, with a killpg call */
	setsid();
#endif

	/* execute the command using sh */
	if (obj->cmd.flags & RtCMD_SHELL)
	   execl ("/bin/sh", "sh", "-c", obj->cmd.cmd_string, 
		(char *) 0);
	else
	   execvp (obj->cmd.argv[0], &obj->cmd.argv[1]);
	
  	/* if we reach here, execl failed */
	fprintf(stderr, "Error starting shell\n");
	_exit(-127); /* check */
	
  }
    
    /* Parent process context, check if fork succeeded */
    if (obj->cmd.child_pid == -1) {
    	perror ("fork");
    	status = -1;
    }

    /* close the child ends of the pipes */
    close(obj->cmd.child_stdin_fd);
    close(obj->cmd.child_stdout_fd);
    close(obj->cmd.child_stderr_fd);
    return (status);
}

void CMDProcessMessage (obj, data, client)
CommandObj obj;
void *data;
void *client;
{
int tmp, tmp1;
MessageObj msg;
char *content;
int message_id;

   msg = (MessageObj) data;
   RtGetValue (msg, RtNmsgId, &message_id);   

   if (obj->core.object_flags & RtOBJ_BUSY) {
#ifdef DEBUG
	fprintf (stderr, 
	"CMDProcessMessage: busy object: unable to process the message at this time");
#endif
	if (message_id == RtSTOP && 
		(obj->cmd.child_pid > 1))
    	   	kill (obj->cmd.child_pid, 9);

	return;   
   }
   
   /* save resource values. These values might be changed here */
   tmp = obj->cmd.wait_pid;   
   tmp1 = obj->cmd.confirm_remove;
   
   obj->core.status = 0;
   if (obj->cmd.cwd && message_id != RtCMD_EMPTY_TRASH) {
      if (!save_cwd (obj))
      	 return;

      obj->core.status = chdir (obj->cmd.cwd);
      if (obj->core.status < 0)
          return;

   }

   obj->core.object_flags |= RtOBJ_BUSY;
   if (obj->cmd.cursor)   
	RtSetValue (obj->cmd.cursor, RtNcursorEnabled, 1);
      
   switch (message_id) {
      case RtCMD_TRASH_FILES:
         trash_files (obj);
         break;
      case RtCMD_REMOVE_FILE:
         RtGetValue (msg, RtNmsgContent, &content);   
         if (content)
            remove_file (obj, content);
         break;
      case RtCMD_MOVE_FILES:
         move_files (obj);
         break;
      case RtCMD_COPY_FILES:
         copy_files (obj);
         break;
      case RtCMD_EMPTY_TRASH:
         empty_trash (obj);
         break;
      case RtCMD_EXECUTE:
         execute_cmd (obj);
         break;
      default:
   	 fprintf (stderr, 
   	 "CMDProcessMessage: I don't know how to process this message\n");
   }

   if (obj->cmd.cwd && message_id != RtCMD_EMPTY_TRASH) 
      restore_cwd (obj);

   /* Restore resource values */
   
   obj->cmd.wait_pid = tmp;   
   obj->cmd.confirm_remove = tmp1;
   obj->core.object_flags &= ~RtOBJ_BUSY;  
   if (obj->cmd.cursor)   
   	RtSetValue (obj->cmd.cursor, RtNcursorEnabled, 0);
}

/*
 * Broadcast the output of the command
 */
 
static void broadcast_stdout (obj)
CommandObj obj;
{
   char buf[EBUFSIZ];	/* Error buffer */
   int size, cnt;
   MessageObj msg;
   
   msg = RtCreateMessage ();
   if (!msg)
    	return;

   size = EBUFSIZ - 1;

   /* Check - this does not guarantee reaching EOF */
   for (;;) {
   	cnt = read (obj->cmd.stdout_fd, buf, size);

   
	/* check - flavor dependent */
   	if (cnt <= 0)
    	   cnt = 0;
   
   	RtSetValue (msg, RtNmsgId, RtCMD_OUTPUT);
   	RtSetValue (msg, RtNmsgContent, buf);
   	RtSetValue (msg, RtNmsgClient, cnt);
   	RtSendMessage (obj->core.obj_stdout, msg, NULL);

	/* check - flavor dependent */
   	if (cnt <= 0)
    	   break;
    	if (obj->cmd.process_xevents)
	   process_xevents (obj);
   }
   RtDestroyObject (msg);
}


/*
 * execute_cmd: execute command
 */
 
void execute_cmd (obj)
CommandObj obj;
{
   int status, w;
   int cnt;
   char buf[EBUFSIZ];	/* Error buffer */
   char discard[EBUFSIZ];
   char *aux;
   int size, index = 0;
   Widget shell;
   Widget locateShell ();
   int stderr_fd;

    obj->core.status = 0;
    
    	
    if ((obj->core.status = spawn_command (obj)) != 0) {
       if (obj->cmd.restore_signals)
          restore_signals (obj);
       return;
    }

    if (obj->cmd.flags & RtCMD_STDERR_IS_STDOUT)
    	stderr_fd = obj->cmd.stdout_fd;
    else
    	stderr_fd = obj->cmd.stderr_fd;

    if (obj->cmd.shell_unsensitive) {
        shell = locateShell(obj->cmd.widget);   
    	XtSetSensitive (shell, False);
    }
    
    /* set the pipes connected to the process for non-blocking i/o */
    if (fcntl(obj->cmd.stdin_fd, F_SETFL, O_NONBLOCK) < 0)
    	perror("execute_cmd: Internal error (fcntl)");
    if (fcntl(obj->cmd.stdout_fd, F_SETFL, O_NONBLOCK) < 0)
    	perror("execute_cmd: Internal error (fcntl1)");
    if (fcntl(obj->cmd.stderr_fd, F_SETFL, O_NONBLOCK) < 0)
    	    perror("execute_cmd: Internal error (fcntl2)");

    size = EBUFSIZ - 1;
    if (obj->cmd.wait_pid) {
        while ((w = waitpid (obj->cmd.child_pid, &status, WNOHANG))
        	 != obj->cmd.child_pid &&
        	 w != -1) {

    	   /* Read stderr */
    	   if (size) {
    	   	cnt = read (stderr_fd, buf+index, 
    	   		size);

    	   	if (cnt > 0) {
    	      	   index += cnt;
    	      	   size -= cnt;
    	   	}
    	   } else
    	   	read (stderr_fd, discard, 
    	   		EBUFSIZ - 1);
    	   		
    	   if (obj->cmd.flags &  RtCMD_BROADCAST_STDOUT)
    	   	broadcast_stdout (obj);
    	   	 
           if (obj->cmd.process_xevents)
           	process_xevents (obj);
        }
        
        if (obj->cmd.shell_unsensitive && shell)
           XtSetSensitive (shell, True);

	if (w == -1) {
	   if (errno != EINTR)
	        perror ("wait"); 
           if (obj->cmd.restore_signals)
          	restore_signals (obj);
	   return;
	}
    } else {
        if (obj->cmd.restore_signals)
          	restore_signals (obj);
        if (obj->cmd.shell_unsensitive && shell)  	
        	XtSetSensitive (shell, True);
    	return;
    }
    
    if (obj->cmd.restore_signals)
        restore_signals (obj);
    obj->core.status = status;
    
    if (obj->cmd.flags &  RtCMD_BROADCAST_STDOUT)
    	broadcast_stdout (obj);

    if (!abnormal(status)) {
       /* Close file descriptors */
       close (obj->cmd.stdin_fd);    
       close (obj->cmd.stdout_fd);    
       close (obj->cmd.stderr_fd);    
       return; 	/* normal termination */
    }
          
#ifdef OLD
    /* Read error message */
    cnt = read (obj->cmd.stderr_fd, buf, EBUFSIZ - 1);
#endif

    	
    if (size) {
	cnt = read (stderr_fd, buf+index, 
    	   		size);

    	if (cnt > 0) {
    	     index += cnt;
    	     size -= cnt;
    	}
    }
    	     
    
    if (!index) {
      /* Close file descriptors */
      close (obj->cmd.stdin_fd);    
      close (obj->cmd.stdout_fd);    
      close (obj->cmd.stderr_fd);    
      return;
    }
    
    buf[index] = '\0';
    
    /* Look for the last new line */
    if (aux = strrchr (buf, '\n'))
       *aux = '\0';
       
    if (obj->cmd.message_dialog) {
       _MDsyncErrorDialog(obj->cmd.message_dialog, buf);
    }
#ifdef OLD
    fprintf (stderr, "%s\n", buf);
#endif
    
    /* Close file descriptors */
    close (obj->cmd.stdin_fd);    
    close (obj->cmd.stdout_fd);    
    close (obj->cmd.stderr_fd);    
}

static int process_xevents (obj)
CommandObj obj;
{
   XtAppContext context;
   
   if (!obj->cmd.widget)
      return;
      
   context = XtWidgetToApplicationContext (obj->cmd.widget);
   while (XtAppPending(context)) {
#ifdef OLD   
   	fprintf (stderr, "Events\n");
#endif   	
	XtAppProcessEvent(context, XtIMAll);
   }
}

static int trash_file (obj)
CommandObj obj;
{
   int pressed_button = obj->cmd.default_button;
   char *msg;
   int saved_status;
   
   obj->core.status = 0;
   
   if (!obj->cmd.trash_path)
      return;
   
   /* Check if the source file exists */
   if (!_lexists (obj->cmd.single_path)) {
      obj->core.status = 2;
      msg = _vstrconcat (obj->cmd.single_path, 
      	": ", "file not found", NULL);
           
      if (!msg)
          return;
          
      _MDsyncErrorDialog(obj->cmd.message_dialog, msg);
      free (msg);
      return;
   }
      
retry:
       
   if (!_isdir (obj->cmd.trash_path)) {
#ifdef DEBUG
   	fprintf (stderr, 
   		"trash_file: trash path should be a directory\n");
#endif
        obj->core.status = 3;
   	return;
   }
   	  
   /* make sure this is the trash directory */
   if (!strstr (obj->cmd.trash_path, "trash")) {
#ifdef DEBUG
	fprintf (stderr, 
	"trash should be the last component of the trash directory path\n");
#endif
        obj->core.status = 3;
	return;	
   }
   
   obj->cmd.target_path = _dircat (obj->cmd.trash_path,
   				_basename(obj->cmd.single_path));
   				
   if (!obj->cmd.target_path) {
      obj->core.status = 1;
      return;
   }   
   
   /* If there is another file with the same name, check for confirmation */

   if (_lexists (obj->cmd.target_path)) {
        obj->core.status = 2;
        free (obj->cmd.target_path);
        msg = _vstrconcat (_basename(obj->cmd.single_path), ": ",
        "unable to continue, there is a file/directory in the trash with the same name",
        NULL);
           
        if (!msg)
          	return;
          
	_MDsyncErrorDialog(obj->cmd.message_dialog, msg);
#ifdef OLD
	_MDsyncConfirmationDialog(obj->cmd.message_dialog, msg);
#endif
	free (msg);
	return;
#ifdef OLD
	/* Check the pressed button */
	RtGetValue (obj->cmd.message_dialog, RtNmdPressedButton, 
		&pressed_button);
	if (pressed_button != RtOK_BUTTON) {
	   free (obj->cmd.target_path);
	   return;
	}
	   
	obj->cmd.confirm_remove = 0;
	remove_file (obj, obj->cmd.target_path);
		
	if (obj->core.status) {
	   free (obj->cmd.target_path);
	   return;
	}
#endif
   }

#ifdef OLD      
   if (obj->core.errno == ENOSPC) {
	errno = ENOSPC; /* for quality assurance purposes only */
	obj->core.errno = 0;
	saved_status = errno;
   } else 
#endif
   
   if ((obj->core.status = rename (obj->cmd.single_path, 
   		obj->cmd.target_path)) 
         	== 0) {
      free (obj->cmd.target_path);
      return;

   }	
   saved_status = errno;
   if (errno != ENOSPC && errno != EXDEV) {        
        msg = _vstrconcat ("mv : ",
        _basename(obj->cmd.single_path), " to trash",
        NULL);
        
        if (!msg)
           return;
        
       _MDsyncSystemErrorDialog(obj->cmd.message_dialog, 
	   msg);
	free (msg);
	try_to_remove (obj); /* Try to remove the file instead */	   
        free (obj->cmd.target_path);
        obj->core.status = saved_status;
	return;
   }

   if (errno == ENOSPC) {
	_MDsyncSystemErrorDialog(obj->cmd.message_dialog, 
	   "Trash Can")
   	
	if (_isempty (obj->cmd.trash_path)) {
           free (obj->cmd.target_path);
           obj->core.status = saved_status;
	   return;
	}
	
	/* Popup confirmation dialog */

	_MDsyncConfirmationDialog(obj->cmd.message_dialog, 
	   "Do you want to empty the trash directory ?")
	
	/* Check the pressed button */
	RtGetValue (obj->cmd.message_dialog, RtNmdPressedButton, 
		&pressed_button);
	if (pressed_button != RtOK_BUTTON) {
           free (obj->cmd.target_path);
	   obj->core.status = RtCMD_CANCELLED;
	   return;
	}
	   
        empty_trash (obj);
	goto retry;	/* Retry again - avoid infinite loop */ 
   } 

   /*
    * Check if single_path is a directory
    */
    
   if (!_isldir(obj->cmd.single_path)) {
	mv_file (obj); /* issue the Unix command */
   	free (obj->cmd.target_path);
	return;   
   }
   
   /* If it is on a diferent device, try to make a copy */
   
   different_device (obj);
   free (obj->cmd.target_path);
}

static int empty_trash (obj)
CommandObj obj;
{
   char *path;
   
   obj->core.status = 0;
   
   if (!obj->cmd.trash_path) {
       obj->core.status = 2;
       return;
   }
   
   /* make sure this is the trash directory */
   if (!strstr (obj->cmd.trash_path, "trash")) {
	fprintf (stderr, 
	"trash should be the last component of the trash directory path\n");
	return;	
   }
   /* check this - last call to closedir */
   path = _emptydir (obj->cmd.trash_path);
   
   if (path) {
        obj->core.status = 2;
	_MDsyncSystemErrorDialog(obj->cmd.message_dialog, 
	   path);
	free (path); /* check */
   }
}


static int copy_to_trash (obj)
CommandObj obj;
{
char *argv[10];
int flags;

   argv[0] = "/bin/cp";
   argv[1] = "cp";
   argv[2] = "-r";
   argv[3] = obj->cmd.single_path;
   argv[4] = obj->cmd.trash_path;
   argv[5] = '\0';
   RtSetValue (obj, RtNcmdArgv, argv); /* Check */
	
   RtSetValue (obj, RtNcmdWaitPid, 1);
   flags = obj->cmd.flags;
   obj->cmd.flags &= ~RtCMD_SHELL; /* Turn Off the shell option */
   execute_cmd (obj);
   obj->cmd.flags = flags; 	   /* Restore the flags */
}

static int different_device (obj) 
CommandObj obj;
{

   obj->core.status = 0;
   
   /* Attempt to make a copy */
   copy_to_trash (obj);
   
   if (!obj->core.status) {
	obj->cmd.confirm_remove = 0;
	remove_file (obj, obj->cmd.single_path);
	return;
   } 
   
   
   /* Remove copied file */
   if (_lexists (obj->cmd.target_path)) {
	obj->cmd.confirm_remove = 0;
	remove_file (obj, obj->cmd.target_path);
   }	   	
   /* Copy failed. Try to remove the file instead */
   try_to_remove (obj);
}

static int remove_file (obj, target)
CommandObj obj;
char *target;
{
  char *msg;
  int pressed_button = obj->cmd.default_button;
  
  
  /* Ask for Confirmation */
  if (obj->cmd.confirm_remove) {
  
      			    
      msg = _vstrconcat (_basename(obj->cmd.single_path), ": ",
   	"Do you really want to remove this file ?", NULL);
           
      if (!msg)
        return;
          
      _MDsyncConfirmationDialog(obj->cmd.message_dialog, msg);
      
      free (msg);

      /* Check the pressed button */
      RtGetValue (obj->cmd.message_dialog, RtNmdPressedButton, 
		&pressed_button);
   
      if (pressed_button != RtOK_BUTTON)
	   return;

   }
   	   
   /* determine if the file is a directory */
   if (!_isldir ((char *) target)) {
         obj->core.status = unlink ((char *) target);
         if (obj->core.status) 
	   _MDsyncSystemErrorDialog(obj->cmd.message_dialog, 
	   target);
         return;
   }
   
   if (!_emptydir (target)) { /* check */
         obj->core.status = rmdir ((char *) target);
         if (obj->core.status) {
	   _MDsyncSystemErrorDialog(obj->cmd.message_dialog, 
	   target);
	   return;
	 }
	 return;
   } 
   if (errno) {       
      _MDsyncSystemErrorDialog(obj->cmd.message_dialog, 
	   target);
      obj->core.status = errno;
   }
}


try_to_remove (obj)
CommandObj obj;
{
  char *msg;
  int pressed_button = obj->cmd.default_button;
        
   msg = _vstrconcat (_basename(obj->cmd.single_path), ": ",
   "unable to copy this file to the trash can ? \nDo you want to remove it instead ?",
   NULL);
           
   if (!msg)
      return;
          
   _MDsyncConfirmationDialog(obj->cmd.message_dialog, msg);
   
   free (msg);

   /* Check the pressed button */
   RtGetValue (obj->cmd.message_dialog, RtNmdPressedButton, 
		&pressed_button);
   
   if (pressed_button != RtOK_BUTTON)
	   return;
	   
   obj->cmd.confirm_remove = 0;
   remove_file (obj, obj->cmd.single_path);
}

restore_signals (obj)
CommandObj obj;
{

   RtSetValue (obj->cmd.sig_handler, 
    	   RtNmsgId, RtSIG_RESTORE);
   RtSetValue (obj->cmd.sig_handler, 
    	   RtNmsgContent, SIGINT);
   RtSendMessage (obj->cmd.sig_handler, 
    	   obj->cmd.sig_handler);

   RtSetValue (obj->cmd.sig_handler, 
    	   RtNmsgContent, SIGQUIT);
   RtSendMessage (obj->cmd.sig_handler, 
    	   obj->cmd.sig_handler);

   RtSetValue (obj->cmd.sig_handler, 
    	   RtNmsgContent, RtSIG_MASK_RESTORE);
   RtSendMessage (obj->cmd.sig_handler, 
    	   obj->cmd.sig_handler);

#ifdef OLD
    RtSendMessage (obj->cmd.sig_handler, 
    	   RtSIG_RESTORE, SIGINT);
    RtSendMessage (obj->cmd.sig_handler, 
    	   RtSIG_RESTORE, SIGQUIT);
    RtSendMessage (obj->cmd.sig_handler, 
    	   RtSIG_MASK_RESTORE, NULL);
#endif   	
}    

trash_files (obj)
CommandObj obj;
{
  char *tmp;
  
   if (!obj->cmd.src_path)
      return;
   
   tmp = malloc (strlen (obj->cmd.src_path) + 1);
   
   if (!tmp)
      return;
      
   strcpy (tmp, obj->cmd.src_path);
    
   obj->cmd.single_path = strtok (tmp, " ");

   while (obj->cmd.single_path) { 
      trash_file (obj);

      if (obj->core.status) {
         break;
      }
      obj->cmd.single_path = strtok (NULL, " ");
   }
   
   if (obj->core.status == RtCMD_CANCELLED)
      obj->core.status = 0;

   free (tmp);
   obj->cmd.single_path = NULL;   
}

/*
 * Save the current working directory
 */
 
int save_cwd (obj)
CommandObj obj;
{
#ifdef SUNOS
   if (!getwd (obj->cmd.saved_cwd)) {
      *(obj->cmd.saved_cwd) = '\0';
      return (0);
   }
#else 
   if (!getcwd (obj->cmd.saved_cwd, MAXPATHLEN)) {
      *(obj->cmd.saved_cwd) = '\0';
      return (0);	   
   }
#endif
   return (1);   
}

/*
 * Restore the current working directory.
 * This changes the status of the object.
 * Therefore, it can create confusion. 
 */

restore_cwd (obj)
CommandObj obj;
{
int status;

   if (*(obj->cmd.saved_cwd)) {
      status = chdir (obj->cmd.saved_cwd);
   }
   if (!obj->core.status)
     obj->core.status = status; /* It can be improved               */
     				/* The CWD is being changed behind  */
     				/* the scenes. We hope the previous */
     				/* call will never fail		    */
     				
}

/*
 * file_cnt: number of files to be copied or moved
 */
 
static int file_cnt (str)
char *str;
{
   char *file;
   char *files;
   register int cnt = 0;
   
   files = strdup (str);
   
   if (!files)
     return (0);
     
   file = strtok (files, " ");
   while (file) {
      cnt++;
      file = strtok (NULL, " ");
   }
   free (files);
   return (cnt);
}


/*
 * copy_files: copy files
 */
 
static int copy_files (obj)
CommandObj obj;
{
  char *tmp, *msg;
  char **argv, **vp;
  int cnt = 0;
  int flags;
  int nfiles;
  
   obj->core.status = 0;
   if (!obj->cmd.src_path || !obj->cmd.dest_path)
      return;
   
   /* determine the number of files to be copied */
   cnt = file_cnt (obj->cmd.src_path);
   
   if (!cnt) {
      obj->core.status = RtCMD_INVALID_ARGUMENTS;
      return;   
   }

   /* if two or more files are being copied, dest_path should 	*/
   /* be a directory						*/
   
   if (cnt >= 2) {
      if (!_isdir (obj->cmd.dest_path)) {
      	obj->core.status = RtCMD_NOT_SUCH_DIRECTORY;
      	msg = _vstrconcat (obj->cmd.dest_path, ": ",
        	"not such directory exists",
        	NULL);
        if (!msg)
      		return;
      	
       _MDsyncErrorDialog(obj->cmd.message_dialog, msg);
       free (msg);
       return;
      }   
   }
      
   tmp = malloc ((strlen (obj->cmd.src_path) + 1) * sizeof (char));
   
   if (!tmp) {
      obj->core.status = 1;
      return;
   }
      
   strcpy (tmp, obj->cmd.src_path);
          
   obj->cmd.single_path = strtok (tmp, " ");

   while (obj->cmd.single_path) {
      /* validate each file */
      validate_cpfile (obj);
      if (obj->core.status) {
   	 free (tmp);
         /* Did the user cancelled the operation  ?*/
         if (obj->core.status == RtCMD_COPY_CANCELLED)
         	obj->core.status = 0;
         return;
      }
      
      obj->cmd.single_path = strtok (NULL, " ");
   }
   
   
   cnt += 4;	/* Room for additional arguments */

   argv = (char **) malloc ((cnt + 1) * sizeof(char *));
   vp = argv;
   
   if (!argv) {
     free (tmp);   
     obj->core.status = 1;
   }
   
   /* Stuff the arguments for the command */
   
   argv[0] = "/bin/cp";
   argv[1] = "cp";
   argv[2] = "-r";
   argv[cnt] = '\0';

   vp += 3;
   
   strcpy (tmp, obj->cmd.src_path);
    
   *vp = strtok (tmp, " ");

   while (*vp) {
      *++vp = strtok (NULL, " ");
   }

   *vp++ = obj->cmd.dest_path;
   *vp = '\0';				/* just in case */
   
   RtSetValue (obj, RtNcmdArgv, argv); /* Check */
	

   /* Save the status of the Object */
	
   obj->cmd.wait_pid = 1;	   /* Careful	*/
   flags = obj->cmd.flags;
   obj->cmd.flags &= ~RtCMD_SHELL; /* Turn Off the shell option */
   execute_cmd (obj);
   obj->cmd.flags = flags; 	   /* Restore the flags */

   free (argv);
   free (tmp);
}

/*
 * validate_cpfile: validate the copy of a file
 */
 
static validate_cpfile (obj)
CommandObj obj;
{

   int pressed_button = obj->cmd.default_button;
   char *msg;
   char *dir = NULL;
   int current_dir = 0; /* target path relative to the current directory */

   
   obj->core.status = 0;
   if (!obj->cmd.dest_path || !obj->cmd.single_path) {
      obj->core.status = 2;
      return;
   }
   
   /* Check if the source file exists */
   if (!_lexists (obj->cmd.single_path)) {
      obj->core.status = RtCMD_FILE_DOES_NOT_EXISTS;
      msg = _vstrconcat (_basename(obj->cmd.single_path), ": ",
        "File does not exist",
        NULL);

      if (!msg)
      	return;
      	
      _MDsyncErrorDialog(obj->cmd.message_dialog, msg);
      free (msg);
      return;
   }

   /* get target directory */
      
   dir = get_target_directory (obj->cmd.dest_path);
   
   if (!dir) {
     obj->core.status = 2;
     return;   
   }
   
   if (!strcmp (dir, "."))
   	current_dir = 1;
   
   if (!current_dir && !_exists (dir)) {
      	obj->core.status = 2;
      	msg = _vstrconcat (obj->cmd.dest_path, ": ",
        	"directory does not exist",
        	NULL);
        	
        free (dir);
        if (!msg)
           return;
      	_MDsyncErrorDialog(obj->cmd.message_dialog, msg);

      	free (msg);
      	
      	return;
   }   
   
   if (!current_dir && !_isdir (dir)) {
      	msg = _vstrconcat (dir, ": ",
        	"not such directory. This is the name of a file",
        	NULL);
      	obj->core.status = 2;
	free (dir);
	
	if (!msg)
	   return;

      	_MDsyncErrorDialog(obj->cmd.message_dialog, msg);

      	free (msg);
      	
      	return;   
   }
   
   /* Assemble destination path */
   
   if (!strcmp (dir, obj->cmd.dest_path))
      obj->cmd.target_path = _dircat (obj->cmd.dest_path,
   				_basename(obj->cmd.single_path));
   else
      obj->cmd.target_path = strdup (obj->cmd.dest_path);
      
   free (dir);
     				
   if (!obj->cmd.target_path)
      return;
      
   
   /* If there is another file with the same name, check for confirmation */

   if (_lexists (obj->cmd.target_path)) {

        if (_isdir (obj->cmd.target_path)) {
           msg = _vstrconcat (obj->cmd.target_path,
           ": ",
           "unable to continue, there is already a directory with the same name",
           NULL);
     	   obj->core.status = 2;
	   free (obj->cmd.target_path);
	   
	   if (!msg)
	      return;
	      
	   _MDsyncErrorDialog(obj->cmd.message_dialog, msg);
	   free (msg);
           return;
        }

        msg = _vstrconcat (_basename(obj->cmd.target_path), ": ",
        "there is a file in the target directory with this same name.\nDo you want to overwrite it ?",
        NULL);
           
        if (!msg)
          	return;
          
	_MDsyncConfirmationDialog(obj->cmd.message_dialog, msg);

	free (msg);
	/* Check the pressed button */
	RtGetValue (obj->cmd.message_dialog, RtNmdPressedButton, 
		&pressed_button);
	if (pressed_button != RtOK_BUTTON) {
	   free (obj->cmd.target_path);
	   obj->core.status = RtCMD_COPY_CANCELLED;
	   return;
	}

        /* If it is a directory, don't do anything   */
        /* This is done here for a second time (just */
        /* in case)				     */
        
        if (_isdir (obj->cmd.target_path)) {
           free (obj->cmd.target_path);
           return;
        }

   }
   free (obj->cmd.target_path);
}
      
static int move_files (obj)
CommandObj obj;
{
  char *tmp, *msg;
  int cnt;
  
   obj->core.status = 0;
   if (!obj->cmd.src_path || !obj->cmd.dest_path)
      return;
   
   /* determine the number of files to be copied */
   cnt = file_cnt (obj->cmd.src_path);
   
   if (!cnt) {
      obj->core.status = 1;
      return;   
   }
   
   /* if two or more files are being copied, dest_path should 	*/
   /* be a directory						*/
   
   if (cnt >= 2) {
      if (!_isdir (obj->cmd.dest_path)) {
      	obj->core.status = RtCMD_NOT_SUCH_DIRECTORY;
      	msg = _vstrconcat (obj->cmd.dest_path, ": ",
        	"not such directory exists",
        	NULL);
        if (!msg)
      		return;
      	
       _MDsyncErrorDialog(obj->cmd.message_dialog, msg);
       free (msg);
       return;
      }   
   }
      
   tmp = malloc (strlen (obj->cmd.src_path) + 1);
   
   if (!tmp)
      return;
      
   strcpy (tmp, obj->cmd.src_path);
    
   obj->cmd.single_path = strtok (tmp, " ");

   while (obj->cmd.single_path) { 
      move_file (obj);
      if (obj->core.status) {
         break;
      }
      obj->cmd.single_path = strtok (NULL, " ");
   }
   if (obj->core.status == RtCMD_MOVE_CANCELLED)
      obj->core.status = 0;
    
   free (tmp);
}

/*
 * move_file: move files
 */

static int move_file (obj)
CommandObj obj;
{
   int pressed_button = obj->cmd.default_button;
   char *msg;
   char *dir = NULL; /* free this at a later time */
   int current_dir = 0; /* target path relative to the current directory */
   char cwd[MAXPATHLEN];
   
   obj->core.status = 0;
   if (!obj->cmd.dest_path || !obj->cmd.single_path) {
      obj->core.status = 2;
      return;
   }
   
   /* Check if the source file exists */
   if (!_lexists (obj->cmd.single_path)) {
      obj->core.status = 2;
      msg = _vstrconcat (_basename(obj->cmd.single_path), ": ",
        "File does not exist",
        NULL);

      if (!msg)
      	return;
      	
      _MDsyncErrorDialog(obj->cmd.message_dialog, msg);
      free (msg);
      return;
   }

   /* get target directory */
      
   dir = get_target_directory (obj->cmd.dest_path);
   
   if (!dir) {
     obj->core.status = 2;
     return;   
   }
   
   if (!strcmp (dir, "."))
   	current_dir = 1;
   
   if (!current_dir && !_exists (dir)) {
      	obj->core.status = 2;
      	msg = _vstrconcat (obj->cmd.dest_path, ": ",
        	"directory does not exist",
        	NULL);
        	
        free (dir);
        if (!msg)
           return;
      	_MDsyncErrorDialog(obj->cmd.message_dialog, msg);

      	free (msg);
      	
      	return;
   }   
   
   if (!current_dir && !_isdir (dir)) {
      	msg = _vstrconcat (dir, ": ",
        	"not such directory. This is the name of a file",
        	NULL);
      	obj->core.status = 2;
	free (dir);
	
	if (!msg)
	   return;

      	_MDsyncErrorDialog(obj->cmd.message_dialog, msg);

      	free (msg);
      	
      	return;   
   }
   
   /* Assemble destination path */
   
   if (!strcmp (dir, obj->cmd.dest_path))
      obj->cmd.target_path = _dircat (obj->cmd.dest_path,
   				_basename(obj->cmd.single_path));
#ifdef OLD
   } else if (!strcmp (obj->cmd.dest_path, "."))
      obj->cmd.target_path = _get_cwd (cwd);
#endif
   else
      obj->cmd.target_path = strdup (obj->cmd.dest_path);
      
#ifdef OLD
   free (dir);
#endif
     				
   if (!obj->cmd.target_path)
      return;
      
   
   /* If there is another file with the same name, check for confirmation */

   if (_lexists (obj->cmd.target_path)) {

	/* Conflict with the icons. I would need to remove */ 
	/* the file first				   */
        /* If it is a directory, don't do anything */
        if (_isdir (obj->cmd.target_path)) {
           msg = _vstrconcat (obj->cmd.target_path,
           ": ",
           "unable to continue, there is already a directory with the same name",
           NULL);
     	   obj->core.status = 2;
	   free (obj->cmd.target_path);
	   
	   if (!msg)
	      return;
	      
	   _MDsyncErrorDialog(obj->cmd.message_dialog, msg);
	   free (msg);
           return;
        }

        msg = _vstrconcat (_basename(obj->cmd.target_path), ": ",
        "there is a file in the target directory with this same name.\nDo you want to overwrite it ?",
        NULL);
           
        if (!msg)
          	return;
          
	_MDsyncConfirmationDialog(obj->cmd.message_dialog, msg);

	free (msg);
	/* Check the pressed button */
	RtGetValue (obj->cmd.message_dialog, RtNmdPressedButton, 
		&pressed_button);
	if (pressed_button != RtOK_BUTTON) {
	   obj->core.status = RtCMD_MOVE_CANCELLED;
	   return;
	}
	
        /* If it is a directory, don't do anything   */
        /* This is done here for a second time (just */
        /* in case)				     */
        
        if (_isdir (obj->cmd.target_path)) {
           free (obj->cmd.target_path);
           return;
        }
   }
      
   if ((obj->core.status = rename (obj->cmd.single_path, 
   		obj->cmd.target_path)) 
         	== 0) {
      free (obj->cmd.target_path);
      return;
   }

   if (errno != EXDEV) {        
        msg = _vstrconcat ("mv : ",
        _basename(obj->cmd.single_path),
        NULL);
        free (obj->cmd.target_path);
        
        if (!msg)
           return;
        
       _MDsyncSystemErrorDialog(obj->cmd.message_dialog, 
	   msg);

	free (msg);
	return;
   }

   /*
    * Check if single_path is a directory
    */
    
   if (!_isldir(obj->cmd.single_path)) {
	mv_file (obj); /* issue the Unix command */
   	free (obj->cmd.target_path);
	return;   
   }
   
   /* Make sure that the source directory is writable */
   
   dir = _dirname (obj->cmd.single_path);
   obj->core.status = _fwrite_execute (dir);
   
   if (obj->core.status == 0)
   	obj->core.status = 1;
   else
   	if (obj->core.status == 1)
   		obj->core.status = 0;
   			
   if (obj->core.status < 0) {
        
        _MDsyncSystemErrorDialog(obj->cmd.message_dialog, 
	   dir);
        free (dir);
	return;
   } else if (obj->core.status > 0) {
      	msg = _vstrconcat (dir, ": ",
        	"Permission denied",
        	NULL);
      	obj->core.status = 2;
	free (dir);
	
	if (!msg)
	   return;

      	_MDsyncErrorDialog(obj->cmd.message_dialog, msg);

      	free (msg);
      	
      	return;      
   }
   free (dir);
   mv_between_devices (obj);
   free (obj->cmd.target_path);
}

static int mv_between_devices (obj) 
CommandObj obj;
{

  int pressed_button = obj->cmd.default_button;
  char *msg;
  int saved_status;
  
   obj->core.status = 0;
   
   /* Attempt to make a copy */
   copy_file (obj);
   
   if (!obj->core.status) {
	obj->cmd.confirm_remove = 0;
	remove_file (obj, obj->cmd.single_path);
	if (obj->core.status) {
           msg = _vstrconcat ("Attempt to remove the original file (",
           _basename(obj->cmd.single_path),
           ") failed",
           NULL);

	   if (!msg)
	      return;
	   
	   /* This might not be specific enough */
       	   _MDsyncErrorDialog(obj->cmd.message_dialog, msg);

	   free (msg);
	}
	return;
   } 
   
   /* Copy failed. Remove copied portion of the file */
   /* check - We are not sure that this file was not there before */   
   if (_lexists (obj->cmd.target_path)) {
   	if (!_isldir (obj->cmd.target_path)) {

	   obj->cmd.confirm_remove = 0;
	   saved_status = obj->core.status;
	   remove_file (obj, obj->cmd.target_path);
	   obj->core.status = saved_status;
	   return;
   	}
        msg = _vstrconcat (_basename(obj->cmd.single_path), ": ",
        "Attempt to copy this file to the target directory failed.\nDo you want to remove the portion that has been copied so far ?",
        NULL);

	if (!msg)
	   return;
	   
       	_MDsyncConfirmationDialog(obj->cmd.message_dialog, msg);

	free (msg);
	/* Check the pressed button */
	RtGetValue (obj->cmd.message_dialog, RtNmdPressedButton, 
		&pressed_button);
       	
       	if (pressed_button != RtOK_BUTTON)
	   return;

	obj->cmd.confirm_remove = 0;
	saved_status = obj->core.status;
	remove_file (obj, obj->cmd.target_path);
	obj->core.status = saved_status;
   }	   	
}

static int mv_file (obj)
CommandObj obj;
{
char *argv[10];
int flags;

   argv[0] = "/bin/mv";
   argv[1] = "mv";
   argv[2] = obj->cmd.single_path;
   argv[3] = obj->cmd.target_path;
   argv[4] = '\0';
   RtSetValue (obj, RtNcmdArgv, argv); /* Check */
	
   /* Save the status of the Object */
	
   obj->cmd.wait_pid = 1;
   flags = obj->cmd.flags;
   obj->cmd.flags &= ~RtCMD_SHELL; /* Turn Off the shell option */
   execute_cmd (obj);
   obj->cmd.flags = flags; 	   /* Restore the flags */
}

static int copy_file (obj)
CommandObj obj;
{
char *argv[10];
int flags;

   argv[0] = "/bin/cp";
   argv[1] = "cp";
   argv[2] = "-r";
   argv[3] = obj->cmd.single_path;
   argv[4] = obj->cmd.target_path;
   argv[5] = '\0';
   RtSetValue (obj, RtNcmdArgv, argv); /* Check */
	
   /* Save the status of the Object */
	
   obj->cmd.wait_pid = 1;
   flags = obj->cmd.flags;
   obj->cmd.flags &= ~RtCMD_SHELL; /* Turn Off the shell option */
   execute_cmd (obj);
   obj->cmd.flags = flags; 	   /* Restore the flags */
}

static char *get_target_directory (path)
char *path;
{
   if (!path) {
	return (NULL);   
   }   

   if (_exists (path) && _isdir (path))
   	return (strdup (path));
   else
	return (_dirname (path));	   
}
