#include "state.h"
#include "tcp.h"
#include "io.h"
#include "command.h"

#include <errno.h>
#include <signal.h>

static void command_connect_machine(int fd);
static void command_state_machine(int fd);
static void data_connect_machine(int fd);
static void data_state_machine(int fd);
static void wakeup();


/********************************************************************
 * Start the connection building state machine.
 * We will try to connect to a RPTP server.  If we are playing sounds we also need
 * to build a second connection.
 */
void start_connect_machine()
{
	if (!state.command)
	{
		state.command = tcp_new();
		state.command->port = state.rptp_port;
		strcpy(state.command->hostname, state.rptp_host);
	}

	if (state.command_state != Connected)
	{
		display_message("Building connection with %s", state.rptp_host);

		if (tcp_initiate_connect(state.command) < 0)
		{
			/*
			 * The connection to the server failed somehow.
			 * This is a major problem!
			 */
			printf("%s: unable to build command connection\n", state.program_name);
			perror(state.program_name);
			exit(1);
		}

		/*
		 * Depending on the state of the command connection we need to either
		 * try to connect again later, wait for the connection to be established,
		 * or assume we are connected.
		 */
		switch (state.command->state)
		{
			case C_Not_connected:
				if (state.command->fd >= 0)
					register_read(state.command->fd, NULL);
				alarm(DELAY_BETWEEN_CONNECT_ATTEMPTS);
				signal(SIGALRM, wakeup);
				state.command_state = Waiting_between_connect_attempts;
				tcp_close(state.command);
				display_message("Connection refused.  Will try again in a while");
				break;
			case C_Building_connection:
				register_read(state.command->fd, command_connect_machine);
				state.command_state = Not_connected;
				break;
			case C_Connected:
				register_read(state.command->fd, command_state_machine);
				state.command_state = Connected;
				break;
		}
	}

	if (!state.playing_sounds || all_play())
		return;

	if (!state.data && state.playing_sounds)
	{
		state.data = tcp_new();
		state.data->port = state.rptp_port;
		strcpy(state.data->hostname, state.rptp_host);
	}

	if (state.data_state == Not_connected && state.playing_sounds)
	{
		if (tcp_initiate_connect(state.data) < 0)
		{
			/*
			 * The connection to the server failed somehow.
			 * This is a major problem!
			 */
			printf("%s: unable to build data connection\n", state.program_name);
			perror(state.program_name);
			exit(1);
		}

		/*
		 * Depending on the state of the command connection we need to either
		 * try to connect again later, wait for the connection to be established,
		 * or assume we are connected.
		 */
		switch (state.data->state)
		{
			case C_Not_connected:
				if (state.data->fd >= 0)
					register_read(state.data->fd, NULL);
				alarm(DELAY_BETWEEN_CONNECT_ATTEMPTS);
				signal(SIGALRM, wakeup);
				state.data_state = Waiting_between_connect_attempts;
				tcp_close(state.data);
				break;
			case C_Building_connection:
				register_read(state.data->fd, data_connect_machine);
				state.data_state = Not_connected;
				break;
			case C_Connected:
				register_read(state.data->fd, data_state_machine);
				state.data_state = Connected;
				break;
		}
	}
}


/********************************************************************
 * Wakeup call!  We need to attempt to connect again.
 */
static void wakeup()
{
	signal(SIGALRM, wakeup);
	if (state.debug)
		printf("%s: wakeup.  Attempting to reconnect\n", state.program_name);

	start_connect_machine();
}


/********************************************************************
 */
static void command_connect_machine(int fd)
{
	if (state.debug)
		printf("%s: Waiting for command connect\n", state.program_name);

	if (tcp_connect(state.command) < 0)
	{
		if (errno == EISCONN)
		{
			state.command_state = Connected;
			register_read(fd, command_state_machine);
		}
		else
		{
			/*
			 * The connection failed.  Try again later.
			 */
			register_read(fd, NULL);
			alarm(DELAY_BETWEEN_CONNECT_ATTEMPTS);
			signal(SIGALRM, wakeup);
			state.command_state = Waiting_between_connect_attempts;
			tcp_close(state.command);
			display_message("Connection refused.  Will try again in a while");
		}
	}
}


/********************************************************************
 * We are now ready to start communicating with rplayd.  We will send a string of
 * commands which will hopefully produce output that we can use.
 */
static void command_state_machine(int fd)
{
	char	command[1000];

	if (state.debug)
		printf("%s: command connected\n", state.program_name);

	io_write_line(fd, "set application=\"%s\"", state.version);
	io_write_line(fd, "set notify=continue,done,pause,play,skip,state,stop,volume,level,position");
	io_write_line(fd, "set position-notify-rate=0.5");

	register_read(fd, read_proc);
	register_process(fd, command_line);
}


/********************************************************************
 */
static void data_connect_machine(int fd)
{
	if (state.debug)
		printf("%s: Waiting for data connect\n", state.program_name);

	if (tcp_connect(state.data) < 0)
	{
		if (errno == EISCONN)
		{
			state.data_state = Connected;
			register_read(fd, data_state_machine);
		}
		else
		{
			/*
			 * The connection failed.  Try again later.
			 */
			register_read(fd, NULL);
			alarm(DELAY_BETWEEN_CONNECT_ATTEMPTS);
			signal(SIGALRM, wakeup);
			state.data_state = Waiting_between_connect_attempts;
			tcp_close(state.data);
		}
	}
}


/********************************************************************
 * We are now ready to start communicating with rplayd.
 */
static void data_state_machine(int fd)
{
	char	command[1000];

	if (state.debug)
		printf("%s: data connected\n", state.program_name);

	io_write_line(fd, "set application=\"%s (data only)\"", state.version);

	register_read(fd, read_proc);
	register_process(fd, data_line);

	data_connected();
}



/*
 * Local variables:
 * tab-width: 4
 * End:
 */
