/* 
 * Copyright (C) 1995 Andrew Scherpbier <Andrew@sdsu.edu>
 *
 * This file is part of rplay.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "state.h"
#include "command.h"
#include "forms.h"
#include "xrplay.h"
#include "gui.h"
#include "sound_list.h"

#include <sys/file.h>
#include <sys/fcntl.h>
#include <rplay.h>


extern FD_xrplay	*fd_xrplay;

static int			flow_fd = -1;
static int			chunk_size = 0;
static char			flow_buffer[8000];

static void command_event(char *line);
static void command_normal(char *line);
static void command_error(char *line);
static void data_event(char *line);
static void data_normal(char *line);
static void data_error(char *line);
void set_object_state(FL_OBJECT *obj, int state);
void continue_flow(int fd);


/********************************************************************
 * Process a line we got from the command rptp connection.
 */
void command_line(int fd, char *line)
{
	rptp_parse(line, NULL);

	switch (*line)
	{
		case '@':
			command_event(line);
			break;
		case '+':
			command_normal(line);
			break;
		case '-':
			command_error(line);
			break;
	}
	fl_check_forms();
}


/********************************************************************
 * Process a line we got from the data rptp connection.
 */
void data_line(int fd, char *line)
{
	rptp_parse(line, NULL);

	if (state.debug)
	{
		printf("%s: data: got '%s'\n", state.program_name, line);
	}
	switch (*line)
	{
		case '@':
			data_event(line);
			break;
		case '+':
			data_normal(line);
			break;
		case '-':
			data_error(line);
			break;
	}
	fl_check_forms();
}


/********************************************************************
 * The data connection was established.  If this was the first time,
 * we will try to play the sounds specified in the sound list
 */
void data_connected()
{
}


/********************************************************************
 * Process an event.
 */
static void command_event(char *line)
{
	char	string[RPTP_MAX_LINE];
	int		volume;
	char	*event = rptp_parse(NULL, "event");
	int		print = 1;

	if (strcmp(event, "volume") == 0)
	{
		if (!state.currently_playing)
		{
			volume = atoi(rptp_parse(NULL, "volume"));
			fl_set_slider_value(fd_xrplay->slider, (double) volume);
		}
	}
	else if (strcmp(event, "state") == 0)
	{
		int	playing = atoi(rptp_parse(NULL, "play"));
		int	paused = atoi(rptp_parse(NULL, "pause"));
		char *audio_port = rptp_parse(NULL, "audio-port");
		
		if (!state.playing_sounds)
			set_gray_state(playing, paused);
		else
		{
			if (state.current_id[0])
				set_gray_state(playing, paused);
			else
				set_gray_state(0, 0);
		}

		if (playing == 0 && paused == 0)
		{
			fl_set_slider_value(fd_xrplay->pos, 0.0);
		}

		if (audio_port && strcmp (audio_port, state.audio_port) != 0)
		{
			strcpy (state.audio_port, audio_port);

			fl_set_button (fd_xrplay->headphone,
						   (strstr (state.audio_port, "headphone")) ? 1 : 0);
			
			fl_set_button (fd_xrplay->speaker,
						   (strstr (state.audio_port, "speaker")) ? 1 : 0);
			
			fl_set_button (fd_xrplay->lineout,
						   (strstr (state.audio_port, "lineout")) ? 1 : 0);
		}
	}
	else if (strcmp(event, "level") == 0)
	{
		int	left = atoi(rptp_parse(NULL, "left"));
		int	right = atoi(rptp_parse(NULL, "right"));
		set_vu(left, right);
		print = 0;
	}
	else if (strcmp(event, "position") == 0)
	{
		double position = (double)atof(rptp_parse(NULL, "position"));
		double seconds = (double)atof(rptp_parse(NULL, "seconds"));
		fl_set_slider_bounds(fd_xrplay->pos, 0.0, seconds);
		fl_set_slider_value(fd_xrplay->pos, position);
		fl_redraw_object(fd_xrplay->message);
	}
	else if (strcmp(event, "done") == 0)
	{
		if (strcmp(rptp_parse(NULL, "id"), state.current_id) == 0)
		{
			/*
			 * Done playing the current sound.  Play the next one
			 */
			play_next_sound();
		}
	}
	if (print && state.debug)
	{
		printf("%s: got '%s'\n", state.program_name, line);
	}
}


/********************************************************************
 * Process a normal message
 */
static void command_normal(char *line)
{
	char	string[RPTP_MAX_LINE];
	char	*command = rptp_parse(NULL, "command");

	if (state.debug)
	{
		printf("%s: got '%s'\n", state.program_name, line);
	}

	if (rptp_parse(NULL, "host"))
	{
		/*
		 * This is the welcome message.  Extract the host and version from it
		 * so we can display it in the window.
		 */
		display_message("%s rplayd %s connected", rptp_parse(NULL, "host"), rptp_parse(NULL, "version"));
	}
	else if (strncmp(line, "+volume=", 8) == 0)
	{
		int		volume = atoi(rptp_parse(NULL, "volume"));
		fl_set_slider_value(fd_xrplay->slider, (double) volume);
	}
	else if (strncmp(line, "+application", 12) == 0)
	{
		/*
		 * Special case for when we got the reply from the set application command.
		 * We use this opportunity to start playing sounds...
		 */
		if (state.playing_sounds && !state.started_playing)
		{
			start_playing_sounds();
			state.started_playing = 1;
		}
	}
	else if (command && strcmp(command, "play") == 0)
	{
		/*
		 * Got a reply from a play command.  This means that we can now grab the current sound
		 * id.
		 */
		strcpy(state.current_id, rptp_parse(NULL, "id"));

		/*
		 * If we are playing a flow, we need to start sending data...
		 */
		if (current_sound && current_sound->sound_flow == Use_flow && state.data_state == Connected)
		{
			flow_fd = open(current_sound->sound_name, O_RDONLY);
			chunk_size = read(flow_fd, flow_buffer, sizeof(flow_buffer));
			if (chunk_size <= 0)
			{
				/*
				 * Weird...  The file exists but we were unable to read anything from it.
				 */
				if (state.debug)
				{
					printf("%s: flow file read error: file = %s, chunk_size = %d\n",
						state.program_name,
						current_sound->sound_name,
						chunk_size);
				}
			}
			else
			{
				/*
				 * We send the initial put which should either return a + or a -.  Only after
				 * we actually receive a + will we send the data we have in the flow_buffer.
				 */
				io_write_line(state.data->fd, "put id=%s size=%d", state.current_id, chunk_size);
			}
		}
	}
}


/********************************************************************
 * A the io system claims to have written a chunk of data.  Write the next one.
 */
void continue_flow(int fd)
{
	chunk_size = read(flow_fd, flow_buffer, sizeof(flow_buffer));
	if (chunk_size <= 0)
	{
		/*
		 * Done playing.  let rplayd know about it...
		 */
		io_write_line(state.command->fd, "done id=%s", state.current_id);
		close(flow_fd);
		flow_fd = -1;
		return;
	}

	if (state.debug)
		printf("%s: flow chunk %d, flow_fd = %d, data fd = %d, id = %s\n",
			state.program_name, chunk_size, flow_fd, fd, state.current_id);
	io_write_line(fd, "put id=%s size=%d", state.current_id, chunk_size);
}


/********************************************************************
 * Start playing sounds in the sound_list
 */
void start_playing_sounds()
{
	/*
	 * Make sure we start at the first sound in the list
	 */
	current_sound = NULL;

	play_button_state(0);
	play_next_sound();
}


/********************************************************************
 * Start playing the next sound in the sound_list
 */
void play_next_sound()
{
	Sound	*sound;

	sound = next_sound();
	if (!sound)
	{
		if (state.quit)
		{
			really_quit_cb();
		}
		
		/*
		 * No more sounds to be played.  Oh well....
		 */
		play_button_state(1);
		display_message("Done playing sounds");
		io_write_line(state.command->fd, "set volume");
		return;
	}
	state.currently_playing = 1;

	if (sound->sound_flow == Use_flow)
	{
		io_write_line(state.command->fd, "play input=flow volume=%d sound=\"%s\"", state.volume, sound->sound_name);
		state.current_id[0] = '\0';
	}
	else
	{
		io_write_line(state.command->fd, "play search=false volume=%d sound=\"%s\"", state.volume, sound->sound_name);
	}
	fl_set_slider_value(fd_xrplay->slider, (double) state.volume);
	display_message("Playing %s", sound->sound_name);
	state.started_playing = 1;
}


/********************************************************************
 * Process an error.
 */
static void command_error(char *line)
{
	if (state.debug)
	{
		printf("%s: got '%s'\n", state.program_name, line);
	}

}


/********************************************************************
 * Process an event.
 */
static void data_event(char *line)
{
	char	string[RPTP_MAX_LINE];
	int		volume;
	char	*event = rptp_parse(NULL, "event");

	if (strcmp(event, "volume") == 0)
	{
		volume = atoi(rptp_parse(NULL, "volume"));
		fl_set_slider_value(fd_xrplay->slider, (double) volume);
	}
	else if (strcmp(event, "state") == 0)
	{
		int	playing = atoi(rptp_parse(NULL, "play"));
		int	paused = atoi(rptp_parse(NULL, "pause"));

		set_gray_state(playing, paused);
	}
	else if (strcmp(event, "level") == 0)
	{
		int	left = atoi(rptp_parse(NULL, "left"));
		int	right = atoi(rptp_parse(NULL, "right"));
		set_vu(left, right);
	}
}


/********************************************************************
 * Process a normal message
 */
static void data_normal(char *line)
{
	if (strncmp(line, "+application=", 13) == 0)
	{
		/*
		 * If we are playing a flow, we need to start sending data...
		 */
		if (state.current_id[0] &&
			current_sound &&
			current_sound->sound_flow == Use_flow &&
			state.data_state == Connected)
		{
			io_write_line(state.data->fd, "put id=%s size=%d", state.current_id, current_sound->sound_size);
			flow_fd = open(current_sound->sound_name, O_RDONLY);
			continue_flow(state.data->fd);
		}
	}
	else if (rptp_parse(NULL, "command") && strcmp(rptp_parse(NULL, "command"), "put") == 0)
	{
		/*
		 * We got the ok to send the contents of the flow_buffer
		 */
		io_write(state.data->fd, flow_buffer, chunk_size, continue_flow);
		if (state.debug)
			printf("%s: got '+', writing %d bytes for put on %s\n", state.program_name, chunk_size, state.current_id);
	}
}


/********************************************************************
 * Process an error.
 */
static void data_error(char *line)
{
	/*
	 * We can only get here if the flow we are playing has been stopped for some reason.
	 */
	if (flow_fd >= 0)
		close(flow_fd);
	flow_fd = -1;
}


/********************************************************************
 * Attempt to stop the currently playing sound.
 */
void stop_sound(int all)
{
	io_write_line(state.command->fd, "stop id=%s", state.current_id);
	state.current_id[0] = '\0';
	state.currently_playing = 0;
	if (current_sound && current_sound->sound_flow == Use_flow)
	{
		if (flow_fd >= 0)
			close(flow_fd);
		flow_fd = -1;
	}
	if (all)
	{
		current_sound = NULL;
		play_button_state(1);
		display_message("Done playing sounds");
		io_write_line(state.command->fd, "set volume");
	}
}



/*
 * Local variables:
 * tab-width: 4
 * End:
 */
