/* 
 * Copyright (C) 1995 Andrew Scherpbier <Andrew@sdsu.edu>
 *
 * This file is part of rplay.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "forms.h"
#include "xrplay.h"
#include "state.h"
#include "io.h"
#include "command.h"
#include "sound_list.h"

extern FD_xrplay	*fd_xrplay;

/********************************************************************
 * Callback which really quits xrplay after stopping playing sounds.
 */
void really_quit_cb(void)
{
	fl_finish();
	exit(0);
}


/********************************************************************
 * Callback for the volume percentage buttons.  The data will contain
 * the volume as a percentage (0 - 100)
 */
void set_volume_cb(FL_OBJECT *ob, long data)
{
	if (state.command_state == Connected)
	{
		int		volume = data;
		if (state.currently_playing)
		{
			io_write_line(state.command->fd, "modify id=%s volume=%d", state.current_id, volume);
			state.volume = volume;
			fl_set_slider_value(fd_xrplay->slider, (double) volume);
		}
		else
		{
			io_write_line(state.command->fd, "set volume=%d", volume);
		}
	}
}


/********************************************************************
 * Callback routine for the File menu.
 */
void slider_cb(FL_OBJECT *ob, long data)
{
	if (state.command_state == Connected)
	{
		int		volume = (int) fl_get_slider_value(ob);
		if (state.currently_playing)
		{
			io_write_line(state.command->fd, "modify id=%s volume=%d", state.current_id, volume);
			state.volume = volume;
			fl_set_slider_value(fd_xrplay->slider, (double) volume);
		}
		else
		{
			io_write_line(state.command->fd, "set volume=%d", volume);
		}
	}
}


/********************************************************************
 * Callback routine for the File menu.
 */
void file_menu_cb(FL_OBJECT *ob, long data)
{
	char	*command = (char *) fl_get_menu_text(ob);
	if (strcmp(command, "Quit") == 0)
	{
		if (state.current_id[0])
		{
			char line[256];
			sprintf(line, "stop id=%s\r\n", state.current_id);
			io_write(state.command->fd, line, strlen(line), really_quit_cb);
		}
		else
		{
			really_quit_cb();
		}
	}
#if ALLOW_RESIZE
	else if (strcmp(command, "Smaller") == 0)
	{
		fl_scale_form(fd_xrplay->xrplay, (double) 0.92, (double) 0.92);
	}
	else if (strcmp(command, "Bigger") == 0)
	{
		fl_scale_form(fd_xrplay->xrplay, (double) 1.08, (double) 1.08);
	}
#endif
}


/********************************************************************
 * Callback routine for the Options menu.
 */
void options_menu_cb(FL_OBJECT *ob, long data)
{
  printf("options menu text: %s\n", fl_get_menu_text(ob));
}


/********************************************************************
 * Callback routine for the the skip back button (go to previous sound).
 */
void action_skip_back_cb(FL_OBJECT *ob, long data)
{
	if (state.command_state == Connected)
	{
		if (state.playing_sounds)
		{
			stop_sound(0);
			current_sound = previous_sound();
			play_next_sound();
		}
		else
			io_write_line(state.command->fd, "skip id=#0 count=-1");
	}
}


/********************************************************************
 * Callback routine for the pause button
 */
void action_pause_play_cb(FL_OBJECT *ob, long data)
{
	int		set = fl_get_button(ob);

	if (state.command_state == Connected)
	{
		if (state.playing_sounds)
		{
			if (set)
				io_write_line(state.command->fd, "pause id=%s", state.current_id);
			else
				io_write_line(state.command->fd, "continue id=%s", state.current_id);
		}
		else
		{
			if (set)
				io_write_line(state.command->fd, "pause id=#0");
			else
				io_write_line(state.command->fd, "continue id=#0");
		}
	}
}


/********************************************************************
 * Callback routine for the stop button.
 */
void action_stop_cb(FL_OBJECT *ob, long data)
{
	if (state.command_state == Connected)
	{
		if (state.playing_sounds)
			stop_sound(1);
		else
			io_write_line(state.command->fd, "stop id=#0");
	}
}


/********************************************************************
 * Callback routine for the skip forward (go to next sound).
 */
void action_skip_forward_cb(FL_OBJECT *ob, long data)
{
	if (state.command_state == Connected)
	{
		if (state.playing_sounds)
		{
			stop_sound(0);
			play_next_sound();
		}
		else
			io_write_line(state.command->fd, "skip id=#0 count=1");
	}
}


/********************************************************************
 * Callback routine for the start button
 */
void action_start_cb(FL_OBJECT *ob, long data)
{
	start_playing_sounds();
}


/********************************************************************
 * Callback routine for the "q" key to quit
 */
void quit_cb(FL_OBJECT *ob, long data)
{
	if (state.current_id)
	{
		char line[256];
		sprintf(line, "stop id=%s\r\n", state.current_id);
		io_write(state.command->fd, line, strlen(line), really_quit_cb);
	}
	else
	{
		really_quit_cb();
	}
}


/********************************************************************
 * Callback routine for keys which will increase the volume a bit
 */
void increase_volume_cb(FL_OBJECT *ob, long data)
{
	if (state.command_state == Connected)
	{
		int		volume = fl_get_slider_value(fd_xrplay->slider);
		volume += 8;
		if (volume > 255)
			volume = 255;
		if (state.currently_playing)
		{
			io_write_line(state.command->fd, "modify id=%s volume=%d", state.current_id, volume);
			state.volume = volume;
			fl_set_slider_value(fd_xrplay->slider, (double) volume);
		}
		else
		{
			io_write_line(state.command->fd, "set volume=%d", volume);
		}
	}
}


/********************************************************************
 * Callback routine for keys which will decrease the volume a bit
 */
void decrease_volume_cb(FL_OBJECT *ob, long data)
{
	if (state.command_state == Connected)
	{
		int		volume = fl_get_slider_value(fd_xrplay->slider);
		volume -= 8;
		if (volume < 0)
			volume = 0;
		if (state.currently_playing)
		{
			io_write_line(state.command->fd, "modify id=%s volume=%d", state.current_id, volume);
			state.volume = volume;
			fl_set_slider_value(fd_xrplay->slider, (double) volume);
		}
		else
		{
			io_write_line(state.command->fd, "set volume=%d", volume);
		}
	}
}


void
update_audio_port (FL_OBJECT *obj, long data)
{
	if (state.command_state == Connected)
	{
		char string[128];

		strcpy (string, "none,");
		if (fl_get_button (fd_xrplay->headphone))
		{
			strcat (string, "headphone,");
		}
		if (fl_get_button (fd_xrplay->speaker))
		{
			strcat (string, "speaker,");
		}
		if (fl_get_button (fd_xrplay->lineout))
		{
			strcat (string, "lineout,");
		}
		string[strlen(string)-1] = '\0';

		io_write_line (state.command->fd, "set audio-port=%s", string);
	}
}


/*
 * Local variables:
 * tab-width: 4
 * End:
 */
