/* $XConsortium: sundevice.c /main/5 1996/12/30 16:35:18 swick $ */

/*
Copyright (c) 1996  X Consortium

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from the X Consortium.
*/

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>  
#include <math.h>  
#include <sys/types.h>  
#include <sys/audioio.h>
#include "errors.h"
#include "au.h"
#include <Xa/atomdefs.h>
#include "cobject.h"
#include "cdevice.h"

/* Enumerate known device types */
enum SunDeviceType {
        SunDeviceUnknown,             /* unknown device type */
        SunDeviceAMD,                 /* AM79C30 */
        SunDeviceDBRI,                /* DBRI & MMCodec (SpeakerBox) */
        SunDeviceSBPRO,               /* SoundBlaster Pro */
        SunDeviceSB16,                /* SoundBlaster 16 */
        SunDeviceSPECTRUM,            /* MediaVision Audio Spectrum 16 */
        SunDeviceMULTISOUND,          /* Turtle Beach MultiSound */
        SunDeviceCODEC                /* MMCodec with no DBRI */
};

static int   device_fd = -1;
static int   dev_mode = -1;
static enum SunDeviceType  dev_type = SunDeviceUnknown;
static char  *audio_device = "/dev/audio";
static audio_info_t  curr_state;

/* 
 * Sun device module
 *   Open, Close, Read, Write, Reset, Drain
 */
int 
SunOpenFunc(void *object, int mode) {

  audio_device_t adev;
  int status;

/* Set up.
 * Then do the open.
 * return read(devAudio, buffer, size);
 */
    if(( device_fd = open( audio_device, mode)) == -1) {
      fprintf (stderr, " **Could not open dev audio file **\n");
      return -1;
    }
    dev_mode = mode;

/*
 * Save the current state.
 */
    if( ioctl(device_fd, AUDIO_GETINFO, &curr_state) == -1 ) {
        perror("ioctl");
	return -1;
    }
/*
 * Get the type of device.
 */
    status = ioctl(device_fd, AUDIO_GETDEV, &adev);

    if (strcmp(adev.name, "SUNW,dbri") == 0)
	dev_type = SunDeviceDBRI;
    else if (strcmp(adev.name, "SUNW,CS4231") == 0)
	dev_type = SunDeviceCODEC;
    else if (strcmp(adev.name, "SUNW,am79c30") == 0)
	dev_type = SunDeviceAMD;
    else if (strcmp(adev.name, "SUNW,sbpro") == 0)
	dev_type = SunDeviceSBPRO;
    else if (strcmp(adev.name, "SUNW,spectrum") == 0)
	dev_type = SunDeviceSPECTRUM;
    else if (strcmp(adev.name, "SUNW,multisound") == 0)
	dev_type = SunDeviceMULTISOUND;
    else if (strcmp(adev.name, "SUNW,sb16") == 0)
	dev_type = SunDeviceSB16;
    else
	dev_type = SunDeviceUnknown;

    return 0;
}

int 
SunCloseFunc(void *object) {

    /* Set up.
     * Then do the open.
     *return read(devAudio, buffer, size);
     */
    if (device_fd != -1)
      close (device_fd);

    return 0;
}




int 
SunReadFunc (void *object, void *connectId,
	     XaTime atTime, XaAtom refTime, 
	     char **buf, CARD32 bitsToRead,
	     CARD8 *leftPad, CARD32 *bitsRead,
	     XaTime *deviceTime)
{
    /* XXX mw: For now just assume we're to read byte-aligned audio data. */
    
    /* Set up.
     * Then do the read.
     */
    XaErrorCode err = XaESuccess;
    int bytesToRead = bitsToRead / 8;
    
    if (dev_mode != O_WRONLY)
	*bitsRead = read(device_fd, *buf, bytesToRead) * 8;
    else
	err = XaEFailure;

    return err;
}
	
XaErrorCode
SunWriteFunc(void *object, void *connectId,
	     XaTime atTime, XaAtom refTime, 
	     char *buffer, CARD32 bitsToWrite, 
	     CARD8 leftPad, CARD32 *bitsWritten,
	     XaTime *deviceTime)
{
    /* XXX mw: For now just assume we're to write byte-aligned audio data. */

    /* Set up.
     * Then do the write.
     */
    XaErrorCode err = XaESuccess;
    int bytesToWrite = bitsToWrite / 8;
    
    if (dev_mode != O_RDONLY)
	*bitsWritten =  write(device_fd, buffer, bytesToWrite) * 8;
    else
	err = XaEFailure;

    return err;
}

int 
SunResetFunc(void *object) {

    /* Set up.
     * Then do the read.
     */
    return lseek (device_fd, 0, SEEK_SET);
}

int 
SunDrainFunc(void *object) {
    /* Set up.
     * Then do the read.
     *return write(devAudio, buffer, size);
     */
    return ioctl (device_fd, AUDIO_DRAIN, NULL);
}

/*
 * Device Attribute Callbacks....
 */
boolean_t
SunCheckChannelsCB (void *object, XaAttributeCBData *data)
{
    boolean_t  status = B_TRUE;

/*
 * Check if newvalue is a valid value
 */
    switch (dev_type) {
    default:     /* if unknown type, assume AMD */
    case SunDeviceAMD:
      if ((int)data->value != 1)
        status = B_FALSE;
      break;
    case SunDeviceDBRI:
    case SunDeviceSBPRO:
    case SunDeviceCODEC:
      if ((int)data->value > 2)
        status = B_FALSE;
      break;
    case SunDeviceSB16:
        /*
         * Place holders for x86 devices, these are treated as if they
         * are equivelant to the SB16 for now.
         */
    case SunDeviceSPECTRUM:
    case SunDeviceMULTISOUND:
      if ((int)data->value > 2)
        status = B_FALSE;
      break;
    }

    return (status);
}

/*********
 * Encoding
 *
 *********/
void
SunGetEncodingCB (void *object, XaAttributeCBData *data)
{
    unsigned value;

    if (dev_mode == O_RDONLY)
      value = curr_state.record.channels;
    else
      value = curr_state.play.channels;

    if (value == AUDIO_ENCODING_ULAW)
      data->value = (void *)XaAencodeUlaw;
    else if (value == AUDIO_ENCODING_ALAW)
      data->value = (void *)XaAencodeAlaw;
    else if (value == AUDIO_ENCODING_LINEAR)
      data->value = (void *)XaAencodeLinear;
}

boolean_t
SunCheckEncodingCB (void *object, XaAttributeCBData *data)
{
    boolean_t  status = B_TRUE;

/*
 * Check if newvalue is a valid value
 */
    switch (dev_type) {
    default:	 /* if unknown type, assume AMD */
    case SunDeviceAMD:
      if ((XaTag)data->value != XaAencodeUlaw &&
	  (XaTag)data->value != XaAencodeAlaw)
        status = B_FALSE;
      break;
    case SunDeviceDBRI:
    case SunDeviceCODEC:
      if ((XaTag)data->value != XaAencodeUlaw &&
          (XaTag)data->value != XaAencodeAlaw &&
	  (XaTag)data->value != XaAencodeLinear) 
        status = B_FALSE;
      break;
    case SunDeviceSBPRO:
      if ((XaTag)data->value != XaAencodeUlaw)
	status = B_FALSE;
      break;
    case SunDeviceSB16:
        /*
         * Place holders for x86 devices, these are treated as if they
         * are equivelant to the SB16 for now.
         */
    case SunDeviceSPECTRUM:
    case SunDeviceMULTISOUND:
      if ((XaTag)data->value != XaAencodeUlaw &&
	  (XaTag)data->value != XaAencodeLinear) 
  	status = B_FALSE;
      break;
    }

    return (status);
}

/********
 * Set Gain
 *
 *********/
void
SunSetGainCB (void *object, XaAttributeCBData *data)
{
    audio_info_t state;
    AUDIO_INITINFO(&state);

    if (dev_mode == O_RDONLY)
      state.record.gain = (unsigned) data->newValue;
    else
      state.play.gain = (unsigned) data->newValue;
 
    if( ioctl(device_fd, AUDIO_SETINFO, &state) == -1 ) {
        perror("ioctl");
	return;
    }
/*
 * Save new value into current state, if successful.
 */
    if (dev_mode == O_RDONLY)
      curr_state.record.gain = (unsigned) data->newValue;
    else
      curr_state.play.gain = (unsigned) data->newValue;
}

void
SunGetGainCB (void *object, XaAttributeCBData *data)
{
    if (dev_mode == O_RDONLY)
      data->value = (void *) curr_state.record.gain;
    else
      data->value = (void *) curr_state.play.gain;
}

boolean_t
SunCheckGainCB (void *object, XaAttributeCBData *data)
{
    boolean_t  status = B_TRUE;
    XaAttributeCBData   cbd;
    XaErrorCode         err;
/*
 * Check if newvalue is a valid value
 */
    if ((int)data->newValue < AUDIO_MIN_GAIN)
	data->newValue = (void *)AUDIO_MIN_GAIN;
    else if ((int)data->newValue > AUDIO_MAX_GAIN)
	data->newValue = (void *)AUDIO_MAX_GAIN;

    return (status);
}

/**************
 * Set/Get Format attributes
 *
 **************/
static unsigned
tag_to_encoding (CARD32 tag)
{
    unsigned value;

    switch (tag) {
    case XaAencodeUlaw:
	value = AUDIO_ENCODING_ULAW;
        break;
    case XaAencodeAlaw:
	value = AUDIO_ENCODING_ALAW;
        break;
    case XaAencodeLinear:
	value = AUDIO_ENCODING_LINEAR;
        break;
    case XaAencodeFloat:
	value = AUDIO_ENCODING_FLOAT;
        break;
    case XaAencodeG721:
	value = AUDIO_ENCODING_G721;
        break;
    case XaAencodeG722:
	value = AUDIO_ENCODING_G722;
        break;
    case XaAencodeG723:
	value = AUDIO_ENCODING_G723;
        break;
    case XaAencodeDVI:
	value = AUDIO_ENCODING_DVI;
        break;
    case XaAencodeLinear8:
	value = AUDIO_ENCODING_LINEAR8;
        break;
      default:
	value = AUDIO_ENCODING_ULAW;
        break;
    }
    return value;
}

/* Return TRUE if the sample rates are within a close tolerance (1%)
 */
static boolean_t 
rate_match(
        unsigned int            rate1,
        unsigned int            rate2)
{
        double                  tol;
 
        tol = ((double) rate2 - (double) rate1) / (double) rate2;
        if (fabs(tol) > .01)
                return (B_FALSE);
        return (B_TRUE);
}

XaBoolean
SunValidateFormatCB (void *object, int noOfAttrs, 
		     XaAttributeCBData **attrs)
{
    void              *format = NULL;
    unsigned           nchannels, sample_rate, encoding, sample_width;
    XaAttributeCBData  cbd;
    int                i;
    XaErrorCode        err;

    for (i = 0; i < noOfAttrs; i++) {
      if (attrs[i]->name == XaAformat)
        format = attrs[i]->newValue;
    }
/*
 * If XaAformat was not set, then
 * return
 */
    if (format == NULL)
      return B_TRUE;
/*
 * Check to see if these format
 * attributes make sense together.
 */
    cbd.name = XaAnumChannels;
    err = XaGetAttributes (format, &cbd, 1);
    if (err != XaESuccess)
      return B_FALSE;
    nchannels = (unsigned)cbd.value;

    cbd.name = XaAsampleRate;
    err = XaGetAttributes (format, &cbd, 1);
    if (err != XaESuccess)
      return B_FALSE;
    sample_rate = (unsigned)cbd.value;

    cbd.name = XaAencoding;
    err = XaGetAttributes (format, &cbd, 1);
    if (err != XaESuccess)
      return B_FALSE;
    encoding = (unsigned)cbd.value;

    switch (dev_type) {
    default:     /* if unknown type, assume AMD  */
    case SunDeviceAMD:
      if ((encoding == XaAencodeUlaw ||
	   encoding == XaAencodeAlaw) &&
           rate_match(sample_rate, 8000) && (nchannels == 1))
        return B_TRUE;
      else
	return B_FALSE;
      break;

    case SunDeviceDBRI:
    case SunDeviceCODEC:
      if ((encoding == XaAencodeUlaw || 
 	   encoding == XaAencodeAlaw) && nchannels == 1)
        return B_TRUE;
      else if (encoding == XaAencodeLinear && nchannels <= 2)
	return B_TRUE;
      else
	return B_FALSE;
      break;

    case SunDeviceSB16:
    case SunDeviceSPECTRUM:
    case SunDeviceMULTISOUND:
      if (encoding == XaAencodeLinear && sample_width != 16)
	return B_FALSE;
      break;	  
    }

    return B_TRUE;

}

void
SunSetFormatCB (void *object, XaAttributeCBData *data)
{
    void *format = data->newValue;
    XaAttributeCBData cbd;
    XaErrorCode       err;
    unsigned          value;

    audio_info_t state;
    AUDIO_INITINFO(&state);
/*
 * Number of channels
 */
    cbd.name = XaAnumChannels;
    err = XaGetAttributes (format, &cbd, 1);
    if (err != XaESuccess)
      return;

    if (dev_mode == O_RDONLY)
      state.record.channels = (unsigned)cbd.value;
    else
      state.play.channels = (unsigned)cbd.value;
/*
 * Sample Rate
 */
    cbd.name = XaAsampleRate;
    err = XaGetAttributes (format, &cbd, 1);
    if (err != XaESuccess)
      return;

    if (dev_mode == O_RDONLY)
      state.record.sample_rate = (unsigned)cbd.value;
    else
      state.play.sample_rate = (unsigned)cbd.value;
/*
 * Bits Per Sample 
 */ 
    cbd.name = XaAbitsPerSample;
    err = XaGetAttributes (format, &cbd, 1);
    if (err != XaESuccess)
      return;

    if (dev_mode == O_RDONLY)
      state.record.precision = (unsigned)cbd.value;
    else
      state.play.precision = (unsigned)cbd.value;
/*
 * Encoding
 */ 
    cbd.name = XaAencoding;
    err = XaGetAttributes (format, &cbd, 1);
    if (err != XaESuccess)
      return;

    if (dev_mode == O_RDONLY)
      state.record.encoding = tag_to_encoding((CARD32)cbd.value);
    else
      state.play.encoding = tag_to_encoding((CARD32)cbd.value);
/*
 * Save new value into current state, if successful.
 */
    if( ioctl(device_fd, AUDIO_SETINFO, &state) == -1 ) {
        perror("ioctl");
	return;
    }
}



boolean_t
SunCheckSampleRateCB (void *object, XaAttributeCBData *data)
{
/*
 * Need to find a way if the sample rate does not
 * match exactly, but is within tolerance, to adjust it.
 */
    switch (dev_type) {
    default:
    case SunDeviceAMD:  /* if unknown type, assume AMD */
      if (rate_match((unsigned int)data->value, 8000))
	data->value = (void *)8000;
      else
        return (B_FALSE);
      break;
    case SunDeviceDBRI:
    case SunDeviceCODEC:
      if (rate_match((unsigned int)data->value, 8000))
        data->value = (void *)8000;
      else if (rate_match((unsigned int)data->value, 9600))
	data->value = (void *)9600;
      else if (rate_match((unsigned int)data->value, 11025))
	data->value = (void *)11025;
      else if (rate_match((unsigned int)data->value, 16000))
	data->value = (void *)16000;
      else if (rate_match((unsigned int)data->value, 18900))
	data->value = (void *)18900;
      else if (rate_match((unsigned int)data->value, 22050))
	data->value = (void *)22050;
      else if (rate_match((unsigned int)data->value, 32000))
	data->value = (void *)32000;
      else if (rate_match((unsigned int)data->value, 37800))
	data->value = (void *)37800;
      else if (rate_match((unsigned int)data->value, 44100))
	data->value = (void *)44100;
      else if (rate_match((unsigned int)data->value, 48000))
	data->value = (void *)48000;
      else
        return (B_FALSE);
      break;
    case SunDeviceSBPRO:
    case SunDeviceSB16:
        /*
         * Place holders for x86 devices, these are treated as if they
         * are equivelant to the SB16 for now.
         */
    case SunDeviceSPECTRUM:
    case SunDeviceMULTISOUND:
	/* Need to check on these */
      if ((unsigned int)data->value < 4000 ||
	  (unsigned int)data->value > 44100)
        return (B_FALSE);
      break;
    }
    return B_TRUE;
}

boolean_t
SunCheckFormatCB (void *object, XaAttributeCBData *data)
{
    void *format = data->newValue;
    XaAttributeCBData cbd;
    XaErrorCode       err;
    boolean_t  status = B_TRUE;
/*
 * Check if format values are valid.
 */

/*
 * Check Number of Channels
 */
    cbd.name = XaAnumChannels;
    err = XaGetAttributes (format, &cbd, 1);
    if (err != XaESuccess)
      return B_FALSE;

    status = SunCheckChannelsCB (object, &cbd);
    if (status == B_FALSE)
      return B_FALSE;
/*
 * Check Sample Rate
 */
    cbd.name = XaAsampleRate;
    err = XaGetAttributes (format, &cbd, 1);
    if (err != XaESuccess)
      return B_FALSE;

    status = SunCheckSampleRateCB (format, &cbd);
    if (status == B_FALSE)
      return B_FALSE;
/*
 * Do an XaSetAttribute in case rate was adjusted
 */
    err = XaSetAttributes (format, &cbd, 1);
    if (err != XaESuccess) 
      return B_FALSE;
    
/*
 * Checking encoding
 */
    cbd.name = XaAencoding;
    err = XaGetAttributes (format, &cbd, 1);
    if (err != XaESuccess) 
      return B_FALSE;

    status = SunCheckEncodingCB (object, &cbd);
    if (status == B_FALSE)
      return B_FALSE;

    return (status);
}




/**********
 * FormatWidth
 *   Set, Get, Check
 **********/
void
SunSetFormatWidthCB (void *object, XaAttributeCBData *data)
{
    audio_info_t state;
    AUDIO_INITINFO(&state);
 
    if (dev_mode == O_RDONLY)
      state.record.precision = (unsigned) data->newValue;
    else
      state.play.precision = (unsigned) data->newValue;
 
    if( ioctl(device_fd, AUDIO_SETINFO, &state) == -1 ) {
        perror("ioctl");
	return;
    }
/*
 * Save new value into current state, if successful.
 */
    if (dev_mode == O_RDONLY)
      curr_state.record.precision = (unsigned) data->newValue;
    else
      curr_state.play.precision = (unsigned) data->newValue;
}


SunValidateCB (void *object, int count, XaAttributeCBData **data)
{
/*
 * Check if these set of attributes are valid as a set.
 */
    return B_TRUE;
}

/*
 * Update the device time (I think -- mw).
 */
static struct timeval
SunUpdateFunc()
{
    struct timeval tv = {0, 0};
    return tv;
}

boolean_t
SunDeviceInit(void *obj, XaBoolean write) 
{
    XaAttributeCBData attr;
    XaAttributeSetCB  setCB;
    void              *connectorObject;

    /* Do device specific initialization
     */

    /*
       XXX mw: Added passthrough connector creation here,
       moved from dia/device.cc since it is only used for
       a subset of vendors -- 29 May 1996
    */
    if (write)
    {
	connectorObject = XaCreatePassThroughConnector(SunUpdateFunc,
						       SunWriteFunc,
						       NULL, NULL);
	if (!connectorObject) return XaEFailure;

	XaSetDeviceConnector(obj, connectorObject);
    }
    
    /*
     * then set some local functions.
     */
    SetDeviceOpenFunc(obj, SunOpenFunc);
    SetDeviceReadFunc(obj, SunReadFunc);
    SetDeviceWriteFunc(obj, SunWriteFunc);
    SetDeviceResetFunc(obj, SunResetFunc);
    SetDeviceDrainFunc(obj, SunDrainFunc);
    SetDeviceCloseFunc(obj, SunCloseFunc);

    attr.name = XaAformat;
    attr.type = XaAformat;
    SetAttributeSetCB (obj, &attr, SunSetFormatCB);
    /*SetAttributeGetCB (obj, &attr, SunGetFormatCB); */
    SetAttributeCheckCB (obj, &attr, SunCheckFormatCB); 

    attr.name = XaAgain;
    attr.type = XaTcard32;
    SetAttributeSetCB (obj, &attr, SunSetGainCB);
    SetAttributeGetCB (obj, &attr, SunGetGainCB);
    SetAttributeCheckCB (obj, &attr, SunCheckGainCB);

    SetAttributeValidateCB(obj, SunValidateFormatCB); 

    return 0;
}

boolean_t
device_init(void *obj, XaBoolean write) 
{
   return (SunDeviceInit (obj, write));
}
