/* $XConsortium: mixbuffer-test.cc /main/5 1996/12/30 16:34:37 swick $ */

/*
Copyright (c) 1996  X Consortium

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from the X Consortium.
*/

#include <Xmd.h>
#include <Xa/atomdefs.h>
#include <Xa/errors.h>
#include <string.h>
#ifdef sun
#include <memory.h>
#endif
#include "transbuffs.h"

#define DATA_SIZE 8000
char dataSent[DATA_SIZE];

void
MixBufInitTestData(char *testData, int testDataSize)
{
    // sawtooth wave, should be sufficient to test basic in/out
    for (int i=0;i<testDataSize;i++)
    {
	testData[i] = i & 0xFF;
    }
}

void
MixBufInitTestChunks(XaInternalBuffer **bufArray, int numBufs,
		     void *dataPtr, int dataSize)
{
    XaInternalBuffer   *newBuf;
    char *dataStart = (char *) dataPtr;
    char *dataEnd = dataStart + dataSize;
    char *dataCurStart = dataStart;
    int chunkSize = dataSize / numBufs;
    int i;

    for (i=0;i<numBufs;i++)
	bufArray[i] = NULL;

    for(i=0;i<numBufs;i++)
    {
	if (i == (numBufs-1))
	    newBuf = new XaInternalBuffer(dataEnd - dataCurStart,
					  (void *) dataCurStart);
        else
	{
	    newBuf = new XaInternalBuffer(chunkSize, (void *) dataCurStart);
	    dataCurStart += chunkSize;
	}
	if (newBuf != NULL)
	{
	    bufArray[i] = newBuf;
	}
	else
	{
	    printf("MixBufInitTestChunks: Couldn't allocate a new "
		"XaInternalBuffer!\n");
	    return; // by this time we're hosed anyway
	}
    }
}

/*
   Test methodology:
   Create an XaMixingBuffer. Write one chunk to it, and attempt
   to read that chunk out again. If the output chunk matches the
   input chunk, we win.
*/
XaBoolean
MixBuf1ChunkTest()
{
    XaMixingBuffer mixingBuffer;
    char *buff= NULL;
    CARD32 bitsWritten, samplesRead;
    XaTime deviceTime;
    char *dataReceived;
    XaBoolean testPassed = TRUE;
    XaErrorCode err;

    MixBufInitTestData(dataSent, DATA_SIZE);
    
    XaInternalBuffer	intBuffer(DATA_SIZE, (void *)dataSent);

    err = mixingBuffer.RingBufConfigure(&buff, DATA_SIZE, // length
					1,	// numChannels
					8,	// sampleWidth
					500,	// transferSize
					1000,	// sampleLimit
					256,	// hardwareTransferLimit
					0,	// CurrentSample
					XaAoutput); // direction
    if (err != XaESuccess)
    {
	printf("Couldn't initialize ring buffer (error %d).\n", err);
		
	testPassed = FALSE;
    }

    if (testPassed)
    {
	err = mixingBuffer.Write(0, 			// atTime
				 XaAlastAccessTime,	// refClock
				 intBuffer,		// buffer
				 DATA_SIZE * 8,	        // bitsToWrite
				 0,			// leftPad
				 bitsWritten,		// bitsWritten
				 deviceTime);	        // deviceTime

	if (err != XaESuccess)
	{
	    printf("Couldn't write to ring buffer (error %d).\n", err);
		
	    testPassed = FALSE;
	}
    }
	

    if (testPassed)
    {
	err = mixingBuffer.WriteTransfer(0,                 // atTime
					 XaAlastAccessTime, // refClock
					 dataReceived,      // buf
					 DATA_SIZE,         // samplesToRead
					 samplesRead,       // <----
					 0);                // deviceTime
	
	if (err != XaESuccess)
	{
	    printf("Couldn't read (WriteTransfer) "
		"from ring buffer (error %d).\n", err);
		
	    testPassed = FALSE;
	}
	if (samplesRead != DATA_SIZE)
	{
	    printf("Samples received didn't match expected result (%d != %d).",
		   samplesRead, DATA_SIZE);
	    testPassed = FALSE;
	}
    }

    if (testPassed)
    {
	// Compare what we got to what we sent.
	if (memcmp(dataReceived, dataSent, samplesRead))
	{
	    printf("Data received didn't match expected result.\n");
	    testPassed = FALSE;
	}
    }

    return testPassed;
}

/*
   Test methodology:
   Create an XaMixingBuffer. Write a chunk to it, and attempt
   to read that chunk out again. Repeat this step NUM_CHUNKS times.
   If the output chunks match the input chunks, we win.
*/
XaBoolean
MixBufSolidNChunksTest()
{
    XaMixingBuffer mixingBuffer;
    char *buff= NULL;
    CARD32 bitsWritten, samplesRead;
    XaTime deviceTime;
    char *dataReceived;
    XaBoolean testPassed = TRUE;
    XaErrorCode err;
    const int NUM_CHUNKS = 30;
    XaInternalBuffer   *bufs[NUM_CHUNKS];
    int i, j, k;

    MixBufInitTestData(dataSent, DATA_SIZE);
    MixBufInitTestChunks(bufs, NUM_CHUNKS, dataSent, DATA_SIZE);
    
    err = mixingBuffer.RingBufConfigure(&buff, DATA_SIZE, // length
					1,	// numChannels
					8,	// sampleWidth
					500,	// transferSize
					1000,	// sampleLimit
					256,	// hardwareTransferLimit
					0,	// CurrentSample
					XaAoutput); // direction
    if (err != XaESuccess)
    {
	printf("Couldn't initialize ring buffer (error %d).\n", err);
		
	testPassed = FALSE;
    }

    for(i=0;(i<NUM_CHUNKS) && testPassed;i++)
    {
	if (bufs[i] == NULL)
	{
	    printf("Buffer %d not properly initialized.\n", i);
	    testPassed = FALSE;
	    continue;
	}
	
	if (testPassed)
	{
	    err = mixingBuffer.Write(0, 		// atTime
				     XaAlastAccessTime,	// refClock
				     *(bufs[i]),	// buffer
				     bufs[i]->GetSize() * 8, // bitsToWrite
				     0,			// leftPad
				     bitsWritten,	// bitsWritten
				     deviceTime);	// deviceTime

	    if (err != XaESuccess)
	    {
		printf("Couldn't write to ring buffer (error %d).\n", err);
		    
		testPassed = FALSE;
	    }

	    if (bitsWritten != (bufs[i]->GetSize() * 8))
	    {
		printf("Buffer %d wasn't completely written (%d != %d).\n",
		       i, bitsWritten, (bufs[i]->GetSize()*8));
		testPassed = FALSE;
	    }
	}


	if (testPassed)
	{
	    err = mixingBuffer.WriteTransfer(0,                 // atTime
					     XaAlastAccessTime, // refClock
					     dataReceived,      // buf
					     bufs[i]->GetSize(),// samplesToRead
					     samplesRead,       // <----
					     0);                // deviceTime

	    if (err != XaESuccess)
	    {
		printf("Couldn't read (WriteTransfer) "
		    "from ring buffer (error %d).\n", err);
		    
		testPassed = FALSE;
	    }

	    if (samplesRead != bufs[i]->GetSize())
	    {
		printf("Buffer %d wasn't completely read out (%d != %d).\n",
		       i, samplesRead, bufs[i]->GetSize());
		testPassed = FALSE;
	    }

	    // Compare what we got to what we sent.
	    if (memcmp(dataReceived, bufs[i]->GetBuffer(), samplesRead))
	    {
		printf("Data received didn't match expected result.\n");
		testPassed = FALSE;
	    }
	}
    }
    return testPassed;
}

/*
   Test methodology:
   Create an XaMixingBuffer. Write two chunk to it, and attempt
   to read three smaller chunks out whose total size is that of
   the original two chunks. Repeat this step (NUM_CHUNKS/2) times.
   If the output chunks match the expected values, we win.
*/
XaBoolean
MixBufSplitNChunksTest()
{
    XaMixingBuffer mixingBuffer;
    char *buff= NULL;
    CARD32 bitsWritten, samplesRead;
    XaTime deviceTime;
    XaBoolean testPassed = TRUE;
    XaErrorCode err;
    const int NUM_CHUNKS = 30;
    const int NUM_IN_CHUNKS = 2;
    const int NUM_OUT_CHUNKS = 3;
    XaInternalBuffer   *bufs[NUM_CHUNKS];
    char *dataReceived;
    int chunkSizes[NUM_OUT_CHUNKS];
    int i, j, k;

    MixBufInitTestData(dataSent, DATA_SIZE);
    MixBufInitTestChunks(bufs, NUM_CHUNKS, dataSent, DATA_SIZE);
    
    err = mixingBuffer.RingBufConfigure(&buff, DATA_SIZE, // length
					1,	// numChannels
					8,	// sampleWidth
					500,	// transferSize
					1000,	// sampleLimit
					256,	// hardwareTransferLimit
					0,	// CurrentSample
					XaAoutput); // direction
    if (err != XaESuccess)
    {
	printf("Couldn't initialize ring buffer (error %d).\n", err);
		
	testPassed = FALSE;
    }

    for(i=0; (i<NUM_CHUNKS) && testPassed; i+=2)
    {
	int chunkSum = 0;

	for(j=0;j<NUM_IN_CHUNKS && testPassed;j++)
	{
	    if (bufs[i+j] == NULL)
	    {
		printf("Buffer not properly initialized.\n", i+j);
		testPassed = FALSE;
		continue;
	    }

	    chunkSum += bufs[i+j]->GetSize();

	    bitsWritten = 0;
	    
	    err = mixingBuffer.Write(0, 		// atTime
				     XaAlastAccessTime,	// refClock
				     *(bufs[i+j]),	// buffer
				     bufs[i+j]->GetSize() * 8, // bitsToWrite
				     0,			// leftPad
				     bitsWritten,	// bitsWritten
				     deviceTime);	// deviceTime
	    
	    if (err != XaESuccess)
	    {
		printf("Couldn't write to ring buffer (error %d).\n", err);
		    
		testPassed = FALSE;
	    }

	    if (bitsWritten != (bufs[i+j]->GetSize() * 8))
	    {
		printf("Buffer %d wasn't completely written (%d != %d).\n",
		       i+j, bitsWritten, (bufs[i+j]->GetSize()*8));
		testPassed = FALSE;
	    }
	}

	chunkSizes[0] = 0;
	int chunkAccum = 0;
	for(j=0;j<NUM_OUT_CHUNKS && testPassed;j++)
	{
	    if (j == (NUM_OUT_CHUNKS-1))
		chunkSizes[j] = chunkSum -
		    (chunkSizes[0] * (NUM_OUT_CHUNKS-1));
	    else
		chunkSizes[j] = chunkSum / NUM_OUT_CHUNKS;

	    err = mixingBuffer.WriteTransfer(0,                 // atTime
					     XaAlastAccessTime, // refClock
					     dataReceived,      // buf
					     chunkSizes[j],// samplesToRead
					     samplesRead,       // <----
					     0);                // deviceTime

	    if (err != XaESuccess)
	    {
		printf("Couldn't read (WriteTransfer) "
		    "from ring buffer (error %d).\n", err);
		    
		testPassed = FALSE;
	    }

	    if (samplesRead != chunkSizes[j])
	    {
		printf("Buffer %d wasn't completely written (%d != %d).\n",
		       (i*NUM_OUT_CHUNKS)+j, samplesRead, chunkSizes[j]);
		testPassed = FALSE;
	    }

	    if (testPassed)
	    {
		// Compare what we got to what we sent.
		if (memcmp(dataReceived,
			   (((char *) bufs[i]->GetBuffer()) + chunkAccum),
			   samplesRead))
		{
		    printf("Data received didn't match expected result.\n");
		    testPassed = FALSE;
		}
	    }
	    chunkAccum += chunkSizes[j];
	}
    }
    
    return testPassed;
}


main (int argc, char **argv)
{
    printf("Single chunk test %s.\n", MixBuf1ChunkTest() ?
	"PASSED" : "FAILED");
    printf("Solid n-chunk test %s.\n",
	MixBufSolidNChunksTest() ?
	"PASSED" : "FAILED");
    printf("Split-read n-chunk test %s.\n",
	MixBufSplitNChunksTest() ?
	"PASSED" : "FAILED");
}




