/* $XConsortium: scheduler.cc /main/4 1996/12/30 16:34:13 swick $ */

/*
Copyright (c) 1996  X Consortium

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from the X Consortium.
*/

// 
// @DEC_COPYRIGHT@
//
// HISTORY
// $Log$
// Revision 1.1.1.4  1995/11/22  15:23:19  Peter_Derr
// 	Fix GrowArray bugs
//
// Revision 1.1.1.3  1995/11/07  19:36:25  Peter_Derr
// 	use ICE authorization
// 
// Revision 1.1.1.2  1995/11/03  20:02:23  Peter_Derr
// 	orig from mw@x.org
// 
// Revision 1.1.1.5  1995/10/25  14:56:45  Peter_Derr
// 	pass a simple protocol message
// 
// Revision 1.1.1.4  1995/10/19  19:57:21  Peter_Derr
// 	handle clients exiting
// 
// Revision 1.1.1.3  1995/10/19  17:39:02  Peter_Derr
// 	accepting ICE connections working
// 
// Revision 1.1.1.2  1995/10/17  17:48:04  Peter_Derr
// 	first pass at setting up ICE connection
// 
// $EndLog$
// 
// @(#)$RCSfile: scheduler.cc $ $Revision: /main/4 $ (DEC) $Date: 1996/12/30 16:34:13 $
// 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#ifdef hpux
#include <sys/time.h>
#endif
#include <unistd.h>
#include <signal.h>

#include "scheduler.h"

#define USE_INVARIANTS

#include <assert.h>
#ifdef USE_INVARIANTS
#define INVARIANT() Invariant()
#else
#define INVARIANT() (void) /* ?? want a no-op here */
#endif

#ifndef FALSE
#define FALSE 0
#endif

#ifndef TRUE
#define TRUE 1
#endif

/*
   --------------------------------------------------------
   Methods for class XaWallTime
*/

void
XaWallTime::Add(const XaWallTime &t1, const XaWallTime& t2,
		XaWallTime& result)
{
    result.tv.tv_sec = t1.tv.tv_sec + t2.tv.tv_sec;
    result.tv.tv_usec = t1.tv.tv_usec + t2.tv.tv_usec;

    if (result.tv.tv_usec > 1000000)
    {
	result.tv.tv_sec++;
	result.tv.tv_usec -= 1000000;
    }
}

/* Subtract t2 from t1, leaving the result in (result).
   Negative results are pinned to 0. This is because of the intended
   use of wall clock subtraction (to determine offsets for poll/select). */
void
XaWallTime::Subtract(const XaWallTime &t1, const XaWallTime& t2,
		     XaWallTime& result)
{
    result.tv.tv_sec = t1.tv.tv_sec - t2.tv.tv_sec;
    result.tv.tv_usec = t1.tv.tv_usec - t2.tv.tv_usec;

    if (result.tv.tv_usec < 0)
    {
	result.tv.tv_sec--;
	result.tv.tv_usec += 1000000;
    }

    if (result.tv.tv_sec < 0)
    {
	result.tv.tv_sec = 0;
	result.tv.tv_usec = 0;
    }
}

void
XaWallTime::Set(const unsigned long seconds, const long usec)
{
    tv.tv_sec = seconds;
    tv.tv_usec = usec;
}

void
XaWallTime::Get(unsigned long& seconds, long& usec)
{
    seconds = tv.tv_sec;
    usec = tv.tv_usec;
}

XaWallTime
XaWallTime::operator + (const XaWallTime& t2) const
{
    XaWallTime result;

    Add(*this, t2, result);
    return result;
}

XaWallTime
XaWallTime::operator - (const XaWallTime& t2) const
{
    XaWallTime result;

    Subtract(*this, t2, result);
    return result;
}

XaWallTime&
XaWallTime::operator += (const XaWallTime& t2)
{
    Add(*this, t2, *this);
    return *this;
}

XaWallTime&
XaWallTime::operator -= (const XaWallTime& t2)
{
    Subtract(*this, t2, *this);
    return *this;
}

int
XaWallTime::operator == (const XaWallTime& t2) const
{
    if (tv.tv_sec != t2.tv.tv_sec)
	return FALSE;
    if (tv.tv_usec != t2.tv.tv_usec)
	return FALSE;

    return TRUE;
}

int
XaWallTime::operator != (const XaWallTime& t2) const
{
    return (! (*this == t2));
}

int
XaWallTime::operator < (const XaWallTime& t2) const
{
    if (tv.tv_sec < t2.tv.tv_sec)
	return TRUE;
    if ((tv.tv_sec == t2.tv.tv_sec) &&
	(tv.tv_usec < t2.tv.tv_usec))
	return TRUE;

    return FALSE;
}

int
XaWallTime::operator >= (const XaWallTime& t2) const
{
    return (! (*this < t2));
}

int
XaWallTime::operator > (const XaWallTime& t2) const
{
    if (tv.tv_sec > t2.tv.tv_sec)
	return TRUE;
    if ((tv.tv_sec == t2.tv.tv_sec) &&
	(tv.tv_usec > t2.tv.tv_usec))
	return TRUE;

    return FALSE;
}

int
XaWallTime::operator <= (const XaWallTime& t2) const
{
    return (! (*this > t2));
}

XaWallTime::operator int() const
{
    int returnVal = (int)((tv.tv_sec * 1000) + (tv.tv_usec / 1000));
    return returnVal;
}

void
XaWallTime::Invariant(void) const
{
    /* Check against outlandish second values. */
    assert(tv.tv_sec < (unsigned long) 0xFFFF0000);

    /* Check against negative microsecond values. */
    assert(tv.tv_usec >= 0);
}

/*
   --------------------------------------------------------
   Methods for class XaInternalOrderedCltn
*/

XaInternalOrderedCltn::XaInternalOrderedCltn(void) : arraySize(0), bufSize(0)
{
    /* Start with NUM_INITIAL_ARRAY_ELEMENTS elements */
    buf = (void **) malloc(NUM_INITIAL_ARRAY_ELEMENTS * sizeof(char *));
    if (buf)
    {
	bufSize = NUM_INITIAL_ARRAY_ELEMENTS;
    }
}

XaInternalOrderedCltn::~XaInternalOrderedCltn(void)
{
    bufSize = arraySize = 0;
    free(buf);
    buf = NULL;
}

void
XaInternalOrderedCltn::Invariant(void) const
{
    // Ensure invariant conditions; bomb if we fail
    assert(buf != NULL);
    assert(arraySize >= 0);
    assert(bufSize >= arraySize);
    for(int i=0;i<arraySize;i++)
    {
	assert(buf[i] != NULL);
    }
}

void
XaInternalOrderedCltn::GrowArray(void)
{
    void *newBuf;

    newBuf = realloc(buf, (bufSize<<1) * sizeof(char *));
    if (newBuf)
    {
	buf = (void **) newBuf;
	bufSize <<= 1;
    }
    INVARIANT();
}

void
XaInternalOrderedCltn::MoveElements(int start, int howMany, int newStart)
{
    memmove(buf+newStart, buf+start, howMany * sizeof(void *));
}

void
XaInternalOrderedCltn::Add(void *elem, int beforeWhich)
{
    int realIndex = beforeWhich;

    /* Constrain the index to a legal value. */
    if (realIndex < 1) realIndex = 1;
    if (realIndex > (arraySize)) realIndex = arraySize;

    /* Make sure we can add one more element to the array. */
    if (bufSize < (arraySize + 1))
	GrowArray();

    /* Move higher-numbered elements aside for the new entry. */
    if (realIndex <= arraySize)
	MoveElements(realIndex, arraySize-realIndex, realIndex+1);

    /* Insert the new element. */
    buf[realIndex] = elem;
    arraySize++;

    INVARIANT();
}

int
XaInternalOrderedCltn::Find(void *elem) const
{
    int i;
    for(i=0;i<arraySize;i++)
    {
	if (buf[i] == elem)
	    return i;
    }
    return -1;
}

void
XaInternalOrderedCltn::Delete(void *elem)
{
    int whichElem = Find(elem);
    (void) AtDelete(whichElem);
    INVARIANT();
}

void *
XaInternalOrderedCltn::AtDelete(int whichElem)
{
    void *victim = NULL;
    
    /* Bounds check */
    if ((whichElem >= 0) && (whichElem < arraySize))
    {
	victim = buf[whichElem];
	MoveElements(whichElem+1, arraySize-(whichElem+1), whichElem);
	arraySize--;
    }
    INVARIANT();
    return victim;
}

/*
   --------------------------------------------------------
   Methods for class XaSortedArray
*/

const int           XaObjectsEqual = 0;
const int           XaObject1Greater = 1;
const int           XaObject1Less = -1;

typedef int         (*XaArraySortFunc)(void *obj1, void *obj2);

class XaSortedArray : public XaInternalOrderedCltn
{
private:
    XaArraySortFunc sortFunc;

    int             GetInsertPoint(void* obj) const;
public:
                    XaSortedArray(XaArraySortFunc func);

    void            Add(void *elem);
    int             Find(void *elem) const;
    int             GetInsertPt(void *elem) const;

    // Invariant checker (debugging aid)
    void            Invariant(void) const;
};

XaSortedArray::XaSortedArray(XaArraySortFunc func) : sortFunc(func)
{
    INVARIANT();
}

/* Perform a binary search in the array for the correct insertion
   point of a new object (obj). */
int
XaSortedArray::GetInsertPt(void* obj) const
{
    int left = 0;               // Left (lesser) side of search window
    int right = GetSize() - 1;  // Right (greater) side of search window
    int mid, cmp;

    if (obj == NULL) return XaEFailure; // don't search for NULL objs

    if (GetSize() == 0) return 0; // no objects yet, so place (obj) first.

    while(left < right)
    {
	// Calculate the midpoint of our search area.
	mid = left + ((right-left) / 2);

	// Apply the sort function to determine on which side of (mid)
	// (obj) should live.
	cmp = (*sortFunc)(obj, At(mid));
	if (cmp == XaObjectsEqual)
	{
	    // Bingo, we found the right position. Return it.
	    return mid;
	}
	else if (cmp == XaObject1Greater)
	{
	    // New object falls within right (greater) side of window;
	    // move the left side of our binary window to the middle.
	    left = mid+1;
	}
	else
	{
	    // New object falls within left (lesser) side of window;
	    // move the right side of our binary window to the middle.
	    right = mid;
	}
    }

    /* At this point, left and right should both be pointing at
       the correct insertion point (compare should be XaObject1Less).
       If we get an XaObject1Less or XaObjectsEqual result, return the
       index; otherwise, report an inconsistency. */
    cmp = (*sortFunc)(obj, At(left));
    if ((cmp == XaObject1Less) || (cmp == XaObjectsEqual))
	return left;
    else
    {
	fprintf(stderr,
		"Error in XaSortedList at %lx: sort function failed!\n",
		this);
    }
}

int
XaSortedArray::Find(void *elem) const
{
    int insertPt = GetInsertPt(elem);
    int returnVal = XaEFailure;

    if ((insertPt >= 0) && (insertPt < GetSize()))
    {
	if (sortFunc(elem, At(insertPt)) == XaObjectsEqual)
	    returnVal = insertPt;
    }

    return returnVal;
}

void
XaSortedArray::Add(void *elem)
{
    int insertPt = GetInsertPt(elem);

    // If (elem) is insertable, insert it.
    if (insertPt >= 0)
	XaInternalOrderedCltn::Add(elem, insertPt);

    INVARIANT();
}

void
XaSortedArray::Invariant(void) const
{
    // Call inherited Invariant to check basic list consistency.
    XaInternalOrderedCltn::Invariant();

    // Check each adjacent pair of objects in the array to
    // make sure they're sorted correctly.
    for(int i=0; i < (GetSize() - 1); i++)
    {
	void *elem1, *elem2;
	int result;
	
	elem1 = At(i);
	elem2 = At(i+1);

	result = (*sortFunc)(elem1, elem2);
	assert((result == XaObjectsEqual) || (result == XaObject1Less));
    }
}

/*
   --------------------------------------------------------
   Methods for class XaTask
*/

XaTask::XaTask(XaTaskProc initialProc, void *initialData)
    : serviceProc(initialProc), userData(initialData)
{
}

Bool
XaTask::HasFD(void) const
{
    return FALSE;
}

/*
   --------------------------------------------------------
   Methods for class XaFileTask
*/

XaFileTask::XaFileTask(XaTaskProc newProc, void *newUserData, int newFD)
                 : XaTask(newProc, newUserData), fd(newFD)
{
    fprintf(stderr,"Created new XaFileTask at %lx, fd:%d.\n",
	    this, newFD);
}

Bool
XaFileTask::HasFD(void) const
{
    return TRUE;
}

/*
   --------------------------------------------------------
   Methods for class XaTimedTask
*/

XaTimedTask::XaTimedTask(XaTaskProc proc, void *userData,
			 const XaWallTime& initialWallTime)
    : XaTask(proc, userData)
{
    when = initialWallTime;
}

Bool
XaTimedTask::HasFD(void) const
{
    return FALSE;
}

void
XaTimedTask::Reschedule(const XaWallTime& newTime)
{
    /* Set the time. */
    when = newTime;

    /* Put us back in the scheduler for execution. */
    Scheduler().AddTask(*this);
}

void
XaTimedTask::Invariant(void) const
{
    // make sure the time is legitimate
    when.Invariant();
}

/*
   --------------------------------------------------------
   Methods for class XaScheduler
*/

int
SortTimedTasks(void *obj1, void *obj2)
{
    XaWallTime t1 = ((XaTimedTask *) obj1)->GetTime();
    XaWallTime t2 = ((XaTimedTask *) obj2)->GetTime();

    if (t1 == t2)
	return XaObjectsEqual;
    else if (t1 < t2)
	return XaObject1Less;
    else
	return XaObject1Greater;
}

XaScheduler::XaScheduler()
{
    fileTaskList = new XaInternalOrderedCltn;
    timedTaskList = new XaSortedArray(SortTimedTasks);
    exception = 0;
    timedTaskReady = XaFalse;
}

void XaScheduler::AddTask(XaTask& newTask)
{
    int index;
    
    if (newTask.HasFD())
    {
	//fprintf(stderr, "Appending task %lx to file task list.\n",&newTask);
	fileTaskList->Append(&newTask);
	index = fileTaskList->GetSize() - 1;
	pollList[index].fd = ((XaFileTask *) &newTask)->GetFD();
	/*
	   XXX For now, assume all fds are input fds.
	   We'll change this later.
	*/
#ifdef sco
	pollList[index].events = POLLIN;
#else
	pollList[index].events = POLLNORM;
#endif
	pollList[index].revents = 0;
    }
    else
    {
	timedTaskList->Append(&newTask);
    }
}


void XaScheduler::RemoveTask(XaTask& task)
{
    int index;
    
    if (task.HasFD())
    {
	index = fileTaskList->Find(&task);
	if (index >= 0)
	{
	    /* Move up poll records for higher-numbered file tasks. */
	    if (index < fileTaskList->GetSize())
		memmove(&pollList[index], &pollList[index+1],
			(fileTaskList->GetSize() - (index+1))
			* sizeof(struct pollfd));

	    /* Then remove the file task from the task list. */
	    fileTaskList->AtDelete(index);
	}
    }
    else
    {
	timedTaskList->Delete(&task);
    }
}

void
XaScheduler::ClearReadyIndexList(void)
{
    readyList[0]=XaEND_FILE_TASKS_READY;
    nextReady=0;
    INVARIANT();
}

int
XaScheduler::NextReadyIndex(void)
{
    int returnVal = readyList[nextReady];
    
    if (readyList[nextReady] != XaEND_FILE_TASKS_READY)
	nextReady++;

    INVARIANT();
    return returnVal;
}

int
XaScheduler::NextTaskTimeOffset(void) const
{
    if (timedTaskList->GetSize() == 0)
	return -1; /* No timed tasks, don't timeout */

    XaTimedTask *theTask = (XaTimedTask *) timedTaskList->At(0);
    XaWallTime now, taskTime = theTask->GetTime();

    now.Now();
    taskTime -= now;

    /* Convert the difference into milliseconds. */
    int returnVal = (int) taskTime;
    return returnVal;
}

XaTask *
XaScheduler::NextReadyTimedTask(void)
{
    if (timedTaskList->GetSize() == 0)
	return NULL;

    /* Get the first task in the timed task list (should be earliest) */
    XaTimedTask *theTask = (XaTimedTask *) timedTaskList->At(0);

    /* if timedTaskReady is TRUE, this means that an alarm signal
       fired, indicating that the first task in the task list is ready.
       Otherwise... */
    if (!timedTaskReady)
    {
	/* task flag not thrown, need to check current wall time */
	XaWallTime    nowTm, taskTm;

	nowTm.Now();
	taskTm = theTask->GetTime();

	/* If the wall clock time is earlier than the first task time,
	   then the task is not ready to run. */
	if (nowTm < taskTm)
	    theTask = NULL;
    }

    /* If the first timed task in the list is ready,
       remove it from the timed task list prior to delivery. */
    if (theTask != NULL)
    {
	//fprintf(stderr,"Removing first task from timed task list.\n");
	(void) timedTaskList->AtDelete(0);
    }

    return theTask;
}


void
XaScheduler::RunTimedTasks()
{
    XaTask	*nextTask;

    // printf ("XaScheduler::RunTimedTasks: ");

    /* Run each ready timed task. */
    nextTask = NextReadyTimedTask();
    while (nextTask)
    {
	// printf ("	running task\n");

	nextTask->Run();
	nextTask = NextReadyTimedTask();
    }
}


XaTask *
XaScheduler::NextReadyFileTask(void)
{
    XaTask *    nextTask = NULL;
    int         nextIndex = NextReadyIndex();
    
    if (nextIndex != XaEND_FILE_TASKS_READY)
    {
	nextTask = (XaTask *) fileTaskList->At(nextIndex);
    }

    INVARIANT();
    return nextTask;
}

XaTask *
XaScheduler::NextReadyTask(void)
{
    XaTask   *theTask;

    theTask = NextReadyTimedTask();
    if (theTask == NULL)
	theTask = NextReadyFileTask();

    INVARIANT();
    return theTask;
}

int
XaScheduler::WaitForReadyTask(void)
{
    register int numFds;
    int nextTaskWait = NextTaskTimeOffset();
    int pollerror;

    if (exception)
    {
	/*
	   If the server was jumpstarted, return immediately since
	   the scheduler state needs to be re-established.
	*/
	fprintf(stderr, "WaitForReadyTask: exception\n");
	fflush(stderr);
	numFds = -1;
    }
    else
    {
	//fprintf(stderr,"Fds available for polling:\n");
	for(int i=0;i<fileTaskList->GetSize(); i++)
	{
	    //fprintf(stderr,"%d:fd %d events %ld revents %ld (task %lx)\n",
		//    i, pollList[i].fd, pollList[i].events,
		//    pollList[i].revents, fileTaskList->At(i));
	}
	//fprintf(stderr,"Timeout interval: %d\n", nextTaskWait);
	
	/* Poll for available input or task timeout. */
	//fprintf(stderr, "WaitForReadyTask: poll...");
	fflush(stderr);
	numFds = poll(pollList, fileTaskList->GetSize(), nextTaskWait);
	//fprintf(stderr, "\n");
	fflush(stderr);
    }
    pollerror = errno; /* scoop up errno immediately for analysis */

    if (numFds <= 0) /* error or timeout */
    {
	/*
	   If (exception) is non-zero, this means that the server
	   was jumpstarted in some way, and the scheduler should
	   reestablish all its connections/tasks. If numFds == 0,
	   then a timeout occurred on the poll.
        */
	if (exception || numFds == 0)
	    return 0;

	if (pollerror == EBADF) /* A client disconnected */
	{
	    /* Check connections */
    
	}
	else if (pollerror != EINTR) /* otherwise not an interrupt? */
	{
	    //fprintf(stderr,"XaScheduler::WaitForReadyTask(): poll error %d\n",
		//    pollerror);
	}
    }
    else
    {
	/* Put indices of all ready fds into the ready list. */
	int i, *pReady = readyList;
	register struct pollfd *pollRecPtr;
	
	for(i=0, pollRecPtr=pollList;
	    (i<fileTaskList->GetSize()) && (numFds > 0);
	    i++, pollRecPtr++)
	{
	    /* If this fd received events, mark it ready. */
	    if (pollRecPtr->revents & pollRecPtr->events)
	    {
//		fprintf(stderr,"Putting task #%d into element %d "
//			"of ready list.\n",
//			i, (pReady-readyList)
		*(pReady++) = i;
	    }
	}

	*pReady = XaEND_FILE_TASKS_READY;
	nextReady = 0;
    }
}

void XaScheduler::Run(void)
{
    register XaTask     *nextTask;

    ClearReadyIndexList();
    while (!exception)
    {
	/* Wait for a task to become ready. */
	(void) WaitForReadyTask();

	/* Run each ready task. */
	nextTask = NextReadyTask();
	while (nextTask)
	{
	    nextTask->Run();
	    nextTask = NextReadyTask();
	}

	/* check invariants before every poll */
	INVARIANT();
    }

    /*
       KillAllClients();
       dispatchException &= ~DE_RESET;
    */
}

void
XaScheduler::Invariant(void) const
{
    fileTaskList->Invariant();
    timedTaskList->Invariant();
}

static XaScheduler *schedulerPtr = NULL;

XaScheduler &
Scheduler(void)
{
    if (!schedulerPtr)
    {
	schedulerPtr = new XaScheduler;
    }
    return *(schedulerPtr);
}
