/* $XConsortium: ringbuffer.cc /main/13 1996/12/30 16:34:03 swick $ */

/*
Copyright (c) 1996  X Consortium

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from the X Consortium.
*/

/*
Copyright (c) 1996 Digital Equipment Corporation

Digital Equipment Corporation makes no representations about
the suitability of this Software for any purpose.  The Software
is provided "as is" without express or implied warranty.


@SUN_COPYRIGHT@

*/

// #include <macros.h>
// #include "ringbuffer.h"
#include "transbuffs.h"
#include <Xa/atomdefs.h>
#include <assert.h>
#include <stdlib.h>
#include <string.h>

#define min(a,b)               ( ((a) < (b)) ? (a) : (b) )

#ifdef DUMP_BITS
#include <fcntl.h>
#include <unistd.h>
#endif

/*
 * Ring Buffer
 *
 *	WARNING:  This code requires that bitsPerSample % 8 == 0
 * 
*/

#ifdef  Not_Yet // __alpha
#include <c_asm.h>
#define MB()    asm("mb")
#else
#define MB()
#endif

#ifdef DEBUG
#include "config.h"

static XaBoolean DoDebug ()
{
    static XaBoolean init = FALSE;
    static XaBoolean doit = FALSE;

    if (!init)
    {
	_XaConfigGetBool(".RingBuffer.debug", "", &doit);
	init = TRUE;
    }
    return doit;
}

#endif /* DEBUG */


XaRingBuffer::XaRingBuffer(XaInternalBuffer *ib,
			   XaTime atTime, XaTime duration,
			   XaFormatCache fmtCache,
			   XaAtom newdirection, CARD8 leftPad)
    : XaAudioBuffer(ib, atTime, duration, leftPad, 
		    fmtCache.bitsPerFrame * duration, &formatCache)
{    
    formatCache = fmtCache;
    direction = newdirection;
    transferSize = latestTransferredSample = earliestTransferredSample = 0;
    earliestSample = latestSample = latestSampleLimit = earliestSampleLimit =0;
    currentSample = 0;
    // bitsPerSample = 8;
    doMixing = XaFalse;
}


XaErrorCode XaRingBuffer::Configure(XaTime length, CARD32 nChannels,
				CARD32 sWidth, XaAtom encoding,
				XaTime newTransferSize, XaTime newSampleLimit, 
				XaTime newTransferLimit,
				XaTime newCurrentSample,
				XaMixingBuffer *parent)
{
    // can't deal with non-byte-aligned samples
    if (sWidth % 8 != 0)
	return XaEValue;

    parentMixBuffer = parent;
    bytesPerChannel = sWidth / 8;
    bytesPerSample =  nChannels * bytesPerChannel;
    CARD32 size = length * bytesPerSample;

    if (!buffer)
    {
	buffer = new XaInternalBuffer(size);
    }
    else if (size > InternalBuffer()->GetSize())
    {
	buffer->Release();  // should be sufficient to delete this
	buffer = new XaInternalBuffer(size);
    }
    if (!buffer)
	return XaEAlloc;

    transferSize = newTransferSize;

    if (direction == XaAoutput)
	latestSampleLimit = newSampleLimit;
    else
	earliestSampleLimit = newSampleLimit;

    transferLimit = newTransferLimit;
    latestSample = latestTransferredSample = currentSample = newCurrentSample;
    zeroTime = currentSample + 1;
    bufSamples = InternalBuffer()->GetSize() / bytesPerSample;
    formatCache.numChannels = nChannels;
    formatCache.encoding = encoding;

    return Zero(zeroTime, bufSamples, NULL);
}

XaTime XaRingBuffer::SampleLimit()
{
    if (direction == XaAoutput)
	return (latestTransferredSample - transferLimit);
    else
	return (earliestTransferredSample - transferLimit);
}

XaErrorCode XaRingBuffer::SetLatestSample(XaTime newLS)
{
    if (newLS == latestSample)
	return XaESuccess;

    if (After(newLS, latestSample))
    {
	// Advance time, fill with silence.

	Zero(latestSample + 1, Difference(newLS, latestSample), NULL);
	latestSample = newLS;
	return XaESuccess;
    }
#ifdef DEBUG
    else if (DoDebug())
    {
	assert (0 == 1);
    }
#endif

    // XXXX  Can't set time backwards!!  What do we do in this case???
    return XaEFailure;
}

XaErrorCode XaRingBuffer::Write(XaTime atTime, XaTag refClock,
				XaBoolean preempt, char *buff,
				XaTime samplesToWrite, XaTime *samplesWritten)
{
    XaTime samplesLeft;
    XaErrorCode status = XaESuccess;

#ifdef DEBUG
    if (DoDebug())
	assert(Difference(latestSample, latestTransferredSample) <= bufSamples);
#endif
    if (samplesToWrite > bufSamples) // Can't write more than the size of
	samplesToWrite = bufSamples; // the buffer

    if (refClock == XaAlastAccessTime) // Write after the last Write
	atTime += latestSample + 1;

    // If this is the first write to this time, zero the appropriate part
    // of the time range.

    if (After(atTime, latestSample))
	Zero(atTime, samplesToWrite, NULL);
    else if (After(atTime + samplesToWrite, latestSample))
    {
	Zero(latestSample + 1,
		Difference(atTime + samplesToWrite, latestSample), NULL);
    }

    XaTime relativeTime = (atTime + bufSamples) - zeroTime;
    if (relativeTime > 0)
	relativeTime = relativeTime % bufSamples;
    else
	relativeTime = bufSamples - ((-relativeTime) % bufSamples);
    char * mixBuff = (char *)InternalBuffer()->GetBuffer();
#ifdef DEBUG
    if (DoDebug())
	assert (relativeTime >= 0);
#endif

    if (samplesWritten)
	*samplesWritten = samplesToWrite;

    if (refClock == XaAlastAccessTime)
    {
	latestSample += samplesToWrite;
    }
    else if (After(atTime + samplesToWrite, latestSample))
    {
	latestSample = atTime + samplesToWrite - 1;
    }

#ifdef DEBUG
    if (DoDebug())
	printf(
	    "XaRingBuffer::Write: latestSample=%d latestTransferredSample=%d\n",
		latestSample, latestTransferredSample);
#endif
#ifdef DUMP_BITS

    static int dumpFd;

    if (!dumpFd)
	dumpFd = creat("RingBufferDump.dat", 0666);

    write(dumpFd, buff, samplesToWrite*bytesPerSample);
#endif

    samplesLeft = samplesToWrite;

    while (samplesLeft > 0)
    {
	int	channelCounter = 0;
	XaTime	thisTime = min(samplesLeft, bufSamples - relativeTime);
	int	bytesToCopy = thisTime * bytesPerSample;

	if (preempt || !doMixing)
	{
	    memcpy((void *)&mixBuff[relativeTime * bytesPerSample], buff,
		    bytesToCopy);
	    buff += bytesToCopy;
	    relativeTime += thisTime;
	}
	else
	{
	    XaTime endTime = relativeTime + thisTime;

	    while (relativeTime < endTime)
	    {
		int buffOffset = relativeTime * bytesPerSample;
		for (channelCounter = 0;
		    channelCounter < formatCache.numChannels; channelCounter++)
		{
		    // Need to do mixing writes per channel.
		    mix_sample(mixBuff + buffOffset, buff);
		    buff += bytesPerChannel;
		    buffOffset += bytesPerChannel;
		}
		relativeTime++;
	    }
	}
	samplesLeft -= thisTime;
	if (relativeTime >= bufSamples)
	    relativeTime = 0;
    }
    if (Before(atTime, latestTransferredSample + 1))
    {
	// Write-through is required.

	if (After(atTime + samplesToWrite, latestTransferredSample))
	    // write-through only the earlier part
	    samplesLeft = Difference(latestTransferredSample, atTime) + 1;
	else
	    samplesLeft = samplesToWrite;

	CARD32 bitsToWrite = samplesLeft * bytesPerSample * 8;


	while (samplesLeft > 0)
	{
	    XaTime samplesWritten2;
	    CARD32 bitsWritten;
	    char *buff;
	    XaTime deviceTime; // XXXX this doesn't belong here!!!

	    status = WriteTransfer(atTime, XaAdeviceTime, &buff,
				    samplesLeft, &samplesWritten2);
#ifdef DEBUG
	    if (DoDebug())
		assert (status == XaESuccess);
#endif
	    if (status != XaESuccess) return status;

	    status = parentMixBuffer->WriteThrough(atTime, XaAdeviceTime, buff,
					   samplesWritten2 * bytesPerSample * 8,
					   0, bitsWritten, deviceTime);
	    // if (status != XaESuccess) return status;
	    // If this chunk is too late, WriteThrough() will return a
	    // failure status, but keep going, the second chunk might be OK.

	    // XXXX bytesWritten, deviceTime ???


	    // XXX make sure this is necesssary.
	    // It would be if ring buffer is mapped to DMA region
	    MB();
	    samplesLeft -= samplesWritten2;
	    atTime += samplesWritten2;
	}
    }


    return status;
}
XaErrorCode XaRingBuffer::WriteTransfer(XaTime atTime, XaTag refTime,
				     char **buff, XaTime samplesToTransfer,
				     XaTime *samplesTransferred)
{
    if (refTime == XaAlastAccessTime)
	atTime = latestTransferredSample + 1;
    // else atTime must be absolute device time

    // Make sure we have the requested samples
    if (After(atTime, latestSample))
    {
	*buff = NULL;
	*samplesTransferred = 0;
	return XaEFailure;
    }
    if (samplesToTransfer > latestSample - atTime + 1)
	samplesToTransfer = latestSample - atTime + 1;

    // Get the buffer address to return
    *buff = (char *)InternalBuffer()->GetBuffer();

    // get the offset from the beginning of the buffer
    XaTime relativeTime = (atTime + bufSamples) - zeroTime;
    if (relativeTime > 0)
	relativeTime = relativeTime % bufSamples;
    else
	relativeTime = bufSamples - ((-relativeTime) % bufSamples);
#ifdef DEBUG
    if (DoDebug())
	assert (relativeTime >= 0);
#endif

    // increment the buffer address
    *buff += relativeTime * bytesPerSample;

    // we can return samples from this point to the end of the buffer.
    if ((relativeTime + samplesToTransfer) <= bufSamples)
	*samplesTransferred = samplesToTransfer;
    else
	*samplesTransferred = bufSamples - relativeTime;

    if (refTime == XaAlastAccessTime)
	latestTransferredSample += *samplesTransferred;
    else if (After(atTime + *samplesTransferred, latestTransferredSample))
	latestTransferredSample = atTime + *samplesTransferred;
#ifdef DEBUG
    if (DoDebug())
    {
	printf(" XaRingBuffer::WriteTransfer: latestSample=%d latestTransferredSample=%d\n",
		latestSample, latestTransferredSample);
	assert(Difference(latestSample, latestTransferredSample) <= bufSamples);
    }
#endif
    
    return XaESuccess;
}

extern unsigned char AF_mix_a[65536];
extern unsigned char AF_mix_u[65536];
extern unsigned char *AF_gain_table_u[];
extern unsigned char *AF_gain_table_a[];
extern int    AF_gain_max_u;
extern int    AF_gain_min_u;
extern int    AF_gain_max_a;
extern int    AF_gain_min_a;

#define MIX_U(cd,sd)      (AF_mix_u[((cd)<<8)|(sd)])
#define MIX_A(cd,sd)      (AF_mix_a[((cd)<<8)|(sd)])


void XaRingBuffer::mix_sample(char *dest, char *src)
{
    //
    //  for one channel, one sample, mix in the source data
    //

    switch (formatCache.encoding)
    {
	int a, b, sum;
	unsigned char *gp;	// gain table pointer

	case XaAencodeLinear:
	    switch (bytesPerChannel)
	    {

		case 1:
		    a = *(unsigned char *)dest;
		    b = *(unsigned char *)src;
		    sum = a + b/2;
		    if (sum < 0) sum = 0;
		    if (sum > 255) sum = 255;
		    *(unsigned char *)dest = sum;

		    break;
		case 2:
		    a = *(short *)dest;
		    b = *(short *)src;
		    sum = a + b/2;
		    if (sum < -32768) sum = -32768;
		    if (sum > 32767) sum = 32767;
		    // if (sum < 0) sum = 0;
		    // if (sum > 0xFFFF) sum = 0xFFFF;
		    *(short *)dest = sum;
		    break;
		default:
		    // XXX what now???
		    assert (0 == 1);
		    break;
	    }
	    break;

	case XaAencodeUlaw:

	    // XXX Could apply an arbitrary gain of -5dB like this:
	    // gp = AF_gain_table_u[-5 - AF_gain_min_u];

	    a = *(unsigned char *)dest;
	    b = *(unsigned char *)src;
	    // b = gp[b];
	    *dest = MIX_U(b, a);

	    break;

	case XaAencodeAlaw:

	    // XXX Could apply an arbitrary gain of -5dB like this:
	    // gp = AF_gain_table_a[-5 - AF_gain_min_a];

	    a = *(unsigned char *)dest;
	    b = *(unsigned char *)src;
	    // b = gp[b];
	    *(unsigned char *)dest = MIX_A(b, a);

	    break;

	default:
	    // XXX We're hosed.  Should never happen.
	    memcpy((void *)src, (void *)dest, bytesPerChannel);
	    break;
    }
}


XaErrorCode XaRingBuffer::Read(XaTime atTime, XaTag refTime,
			     char **buf, XaTime samplesToRead,
			     XaTime *samplesRead)
{

    return XaEUnimplemented;
}

XaErrorCode XaRingBuffer::ReadTransfer(XaTime atTime, XaTag refClock,
					XaBoolean preempt,
					char *buff, XaTime samplesToTransfer, 
					XaTime *samplesTransferred)
{

    return XaEUnimplemented;
}

XaErrorCode XaRingBuffer::Zero(XaTime atTime, XaTime samplesToWrite, 
				XaTime *samplesWritten)
{

    XaTime relativeTime = (atTime + bufSamples) - zeroTime;
    char * buff = (char *)InternalBuffer()->GetBuffer();
    int silenceValue;

    if (formatCache.encoding == XaAencodeLinear)
	silenceValue = 0;
    else
	silenceValue = 0x0ff;

    if (relativeTime > 0)
	relativeTime = relativeTime % bufSamples;
    else
	relativeTime = bufSamples - ((-relativeTime) % bufSamples);

#ifdef DEBUG
    if (DoDebug())
	assert (relativeTime >= 0);
#endif

    if (samplesToWrite > bufSamples)
	samplesToWrite = bufSamples;

    if (samplesWritten)
	*samplesWritten = samplesToWrite;

    while (samplesToWrite > 0)
    {
	XaTime thisTime = min(samplesToWrite, bufSamples - relativeTime);

	memset((void *)&(buff[relativeTime * bytesPerSample]), silenceValue,
		thisTime * bytesPerSample);
	samplesToWrite -= thisTime;
	relativeTime += thisTime;
	if (relativeTime >= bufSamples)
	    relativeTime = 0;
    }
    // XXX make sure this is necesssary.
    // It would be if ring buffer is mapped to DMA region
    MB();

    return XaESuccess;
}


XaErrorCode XaRingBuffer::WriteSilence(XaTime atTime, XaTag refClock,
				XaTime samplesToWrite, XaTime *samplesWritten)
{
    XaTime writeTime;

    if (refClock == XaAdeviceTime)
	writeTime = atTime;
    else if (refClock == XaAlastAccessTime)
	writeTime = latestSample + atTime;

    Zero(writeTime, samplesToWrite, samplesWritten);

    latestSample += *samplesWritten;

    return XaESuccess;
}
