/* $XConsortium: makeatoms.cc /main/3 1996/12/30 16:30:08 swick $ */

/*
Copyright (c) 1996  X Consortium

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from the X Consortium.
*/

//
//
// Builds an initatoms function
// and a atomdefs.h function from
// an input file that has the predefined atoms
// specificaiton.

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <X11/Xmd.h>

// CHANGING THESE IMPLIES A PROTOCOL CHANGE
const CARD32 ParsePartitionBase  = 0x00000001;
const CARD32 ParsePartitionTop   = 0x000003FF;
const CARD32 ServerPartitionBase = 0x00000400;
const CARD32 ServerPartitionTop  = 0x7fffffff;
const CARD32 ClientPartitionBase = 0x80000000;
const CARD32 ClientPartitionTop  = 0xFFFFFFFF;
#define XaTnone (0)

// If these chanage you'll have to hack on imake files.
const char *initFunctionFile = "initatoms.cc";
const char *stringHeaderFile = "atomstrings.h";
const char *atomHeaderFile = "atomdefs.h";
const char *inputFile = "PredefinedAtoms";


const int BuffSize = 256;

enum Partition {Parse, Server, NoPartition};
const char *PartitionToString(Partition p) {
    switch( p) {
	case Parse:
	    return "ParseConstant";
	case Server:
	    return "ServerAllocation";
	case NoPartition:
	default:
	    return "ERROR:NO_PARTITION";
    }
}

//
// This is pretty brittle.
// It requires knowledge of the AtomStoreClass
// and TagRanges.
//
// Tags are assigned in ascending order
// in which they are encountered.
// Each time a line is succesfully parsed
// the atom counter for the appropiate 
// partition is incremented.

class Definition {

public:

    Partition part;

    // The symbol used to refer to the String
    char *symbol;
    // The String.
    char *name;
    // The symbol used to refer to the integer.
    char *atomName;
    // The integer
    CARD32 atom;

    Definition *next;

    Definition() : part(NoPartition), 
                   symbol(NULL), name(NULL), atomName(NULL),
                   atom(0), next(NULL)  {}

    void printInitCache(FILE *f) {
	fprintf(f,
		"    if( a.AddAtom(%d, \"%s\")\n"
		"             != XaESuccess)\n"
		"        return XaEFailure;\n",
		atom, name);
    }

    // Prints out the #define for the atom and the
    // value it should have.
    void printAtom(FILE *f) {
	fprintf(f, "#define %-25s ((XaTag)(%lu))\n", atomName, atom);
    }

    void printStringDef(FILE *f) {
            fprintf(f, "#define %-25s \"%s\"\n", symbol, name);
      }

};

//
// Here is a short grammar for the PredefinedAtoms file
//
// <file>            : <partion_list> <eof>
// <partition_list>  : <partition_list> <partition>
// <parition>        : <partition_name> <definition_list>
//                   : <empty>
// <partition_name>  : "Parse" <nl>
//                   : "Server"   <nl>
// <definition_list> : <definition_list> <defintion>
//                   : <empty>
// <definition>      : <string_name> <atom_name> <atom_string>
// <atom_name>       : <string>
// <atom_string>     : <string>
// <nl>              : "\n"
//
//
// The token stream is produced by
// stripping coments and delimiting tokens by
// white space.
//
// # as the first character is a line comment character
//   the rest of the line is ignored.
//
// 
//  
// Parition <parition_name> 
//    starts a section of definitions. There are only
//    two partitions names: Parse, Server
//
//  
//

class DefinitionList {
private:
    Definition *top, *end;

    Partition part;
    CARD32 parseCount;
    CARD32 serverCount;
public:

    CARD32 &CountFromPartition(Partition p) {
	static CARD32 error = 0;
	switch(p) {
	    case Parse:
		return parseCount;
	    case Server:
		return serverCount;
	    case NoPartition:
		fprintf(stderr, "ERROR-defintion with no partiion\n");
		return error;
	}
    }
    DefinitionList() : top(NULL), end(NULL),
                   part(NoPartition),
	           parseCount(ParsePartitionBase),
                   serverCount(ServerPartitionBase) {}
  
    // Add new defintion and assign an atom.
    void add(Definition *d) {
      d->next = NULL;
      if( top == NULL) {
	end = top = d;
      } else {
	end->next = d;
	end = d;
      }
    }

    enum ParseResult { ParseError, ParseOK };
    ParseResult parseLine(char *buffer) {
     static char *delimiters = " \n\t";
     static char *partition_state_string;
     static Partition partition_state = NoPartition;
    
      // Eat comments.
      if(buffer[0] == '#')
	return ParseOK;

      // It's an error to start off with a number!
      if( isdigit(buffer[0])) {
	fprintf(stderr, "Symbol can't start with number\n");
	return ParseError;
      }
	
      // tokenize
      char *token = strtok(buffer, delimiters);
      if(token == NULL)
	return ParseOK;
    
      // Check for partition start.
      if(!strcmp("Parse", token)) {
	part = Parse;
	return ParseOK;
      }
      if(!strcmp("Server", token)) {
	part = Server;
	return ParseOK;
      }

       // We need to be in a parition now.
      if(part == NoPartition) {
	fprintf(stderr, "No Partition - \"%s\"\n", token);
	return ParseError;
      }

      // Should be ready to read a defintion here:
      // This is quick and dirty so we'll leak memory 
      // here.
      Definition *d = new Definition;

      // Peel off definition.
      if(token == NULL) {
	fprintf(stderr, "Missing symbol name.\n");
	return ParseError;
      } else {
	d->symbol = (char *)malloc(strlen(token) + 1);
	strcpy(d->symbol,token);
      } 

     token = strtok(NULL, delimiters);
      if( token == NULL) {
	fprintf(stderr, "Missing atom name.\n");
	return ParseError;
      } else {
	d->atomName = (char *)malloc(strlen(token) + 1);
	strcpy(d->atomName, token);
      }

      token = strtok(NULL, delimiters);
      if( token  == NULL) {
	fprintf(stderr, "Missing string definition\n");
	return ParseError;
      } else {
	d->name = (char *)malloc(strlen(token) + 1);
	strcpy(d->name, token);
      }

      // Assign the atom.
      d->atom = (part == Parse) ? parseCount++ : serverCount++;
      d->part = part;
      add(d);

      return ParseOK;
    }

    // Build the files.
    void printOn(FILE *initFile, FILE *stringFile, FILE *atomFile) {

      // warnings and disclaimers.
      genericHeader(initFile);
      genericHeader(stringFile);
      genericHeader(atomFile);

      // File Specifics
      stringHeader(stringFile);
      atomHeader(atomFile);
      
      // Spin through the definitions print along the way.
      Partition p = NoPartition;
      Definition *current = top; 
      CARD32 maxParseAtom = 0;
      CARD32 maxServerAtom = 0;

      initCacheHeader(initFile);

      for(; current != NULL; current = current->next) {
	if( p != current->part) {
	  p = current->part;
	  char *partName = ((current->part == Server) ? "Server" : "Parse");
	  fprintf(initFile, "\n    /* %s Partition */\n", partName);
	  fprintf(stringFile, "\n/* %s Partition */\n", partName);
	  fprintf(atomFile, "\n/* %s Partition */\n", partName);
	}
	current->printStringDef(stringFile);
	current->printAtom(atomFile);
	current->printInitCache(initFile);
	if (current->part == Server) {
	    if (current->atom > maxServerAtom)
		maxServerAtom = current->atom;
	} else {
	    if (current->atom > maxParseAtom)
		maxParseAtom = current->atom;
	}
      }

      fprintf(atomFile,
	      "\n#define XA_ATOM_PARSE_LAST_PREDEFINED  (%d)\n",
	      maxParseAtom);
      fprintf(atomFile,
	      "#define XA_ATOM_SERVER_LAST_PREDEFINED  (%d)\n",
	      maxServerAtom);

      // Print out the footers.
      initCacheFooter(initFile);
      stringFooter(stringFile);
      atomFooter(atomFile);

      genericFooter(initFile);
      genericFooter(stringFile);
      genericFooter(atomFile);

    }

      // Generics
      void genericHeader(FILE *f) {
	fprintf(f, "/*\n"
		   " * This file was automatically generated\n"
		   " * and should NOT be edited by hand.\n"
		   "*/\n");
      }

      void genericFooter(FILE *f) {}


      void initCacheHeader(FILE *f) {
	fprintf(f,"//\n// Definition of predefined attoms,\n"
		"// called at server startup.\n");
	fprintf(f,"#include \"errors.h\"\n");
	fprintf(f,"#include \"atom.h\"\n");
	fprintf(f,"#include <Xa/atomdefs.h>\n");
	fprintf(f,"XaErrorCode InitializeCache( XaAtomCache &a) {\n\n");
      }

      void initCacheFooter(FILE *f) {
	fprintf(f, "\n\n    return XaESuccess;\n\n");
	fprintf(f, "}\n");
      }

      // Atom Definitions.
      void atomHeader(FILE *f) {
	fprintf(f, "#ifndef _ATOM_DEFS_H\n");
	fprintf(f, "#define _ATOM_DEFS_H\n");
	fprintf(f, "\n/* Atom Partition Defintions */\n");
	fprintf(f, "#define XA_ATOM_PARSE_BOTTOM  (0x%08X)\n",
		ParsePartitionBase);
	fprintf(f, "#define XA_ATOM_PARSE_TOP     (0x%08X)\n",
		ParsePartitionTop);
	fprintf(f, "#define XA_ATOM_SERVER_BOTTOM (0x%08X)\n",
		ServerPartitionBase);
	fprintf(f, "#define XA_ATOM_SERVER_TOP    (0x%08X)\n",
		ServerPartitionTop);
	fprintf(f, "#define XA_ATOM_CLIENT_BOTTOM (0x%08X)\n",
		ClientPartitionBase);
	fprintf(f, "#define XA_ATOM_CLIENT_TOP    (0x%08X)\n",
		ClientPartitionTop);
	fprintf(f, "\n\n/* Definitions of predefined atoms.*/\n");
	fprintf(f, "#define %-25s ((XaTag)(%lu))\n", "XaTnone", XaTnone);
	
      }

      void atomFooter(FILE *f) {
	fprintf(f, "\n#endif\n\n");
      }

      // String functions
      void stringHeader(FILE *f) {
	fprintf(f, "#ifndef _ATOM_STRINGS_H\n");
	fprintf(f, "#define _ATOM_STRINGS_H\n");
	fprintf(f, "\n/* Definitions of predefined atom names. */\n");
	fprintf(f, "\n#include <Xa/atomdefs.h>\n\n");
	// fprintf(f, "#define %-25s (XaStringFromAtom( %s ))\n", 
        //                                             "XaNnone", "XaTnone");
	fprintf(f, "#define %-25s \"%s\"\n", 
                                                     "XaNnone", "none");
 
      }

      void stringFooter(FILE *f) {
	fprintf(f, "\n#endif\n\n");
      }
  };


main() {

    /* Open up the files ... */
    FILE *strings_file;
    if(	(strings_file = fopen(stringHeaderFile, "w")) == NULL) {
	fprintf(stderr, "Couldn't create file: \"%s\"\n", stringHeaderFile);
	exit(-1);
    }
    
    FILE *atoms_file;
    if((atoms_file = fopen(atomHeaderFile, "w")) == NULL) {
	fprintf(stderr, "Couldn't create file: \"%s\"\n", atomHeaderFile);
	exit(-1);
    }
    
    FILE *init_file;
    if((init_file = fopen(initFunctionFile, "w")) == NULL) {
	fprintf(stderr, "Couldn't create file: \"%s\"\n", initFunctionFile);
	exit(-1);
    }

    FILE *spec_file;
    if((spec_file = fopen(inputFile, "r")) == NULL) {
	fprintf(stderr, "Couldn't read file: \"%s\"\n", inputFile);
	exit(-1);
    }

    // Do the spec file a line at a time.
    DefinitionList::ParseResult result;
    DefinitionList defs;
    int lineNo = 0;
    char buffer[BuffSize];
    while(lineNo++, fgets(buffer, BuffSize, spec_file)) {

        // parse the file a line at a time.
	switch(defs.parseLine(buffer)) {

	case DefinitionList::ParseError:
	  fprintf(stderr, "\"%s\" - Syntax error on line: %d\n",
		  inputFile, lineNo);
	  break;

	case DefinitionList::ParseOK:
	default:
		break;
	}
    }

    // Build the files. */
    defs.printOn(init_file, strings_file, atoms_file);

    // Close files.
    fclose(spec_file);
    fclose(init_file);
    fclose(strings_file);
    fclose(atoms_file);

    return 0;
}
