// This line lets emacs recognize this as -*- C++ -*- Code
/* $XConsortium: dictionary.cc /main/3 1996/12/30 16:29:34 swick $ */

/*
Copyright (c) 1996  X Consortium

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from the X Consortium.
*/


//
//  -------------------------------------------
//  Implementation-specific macros and includes
//  -------------------------------------------
// 

#include "dictionary.h"
#include "pointer.h"
#include <malloc.h>
#include <stdio.h>
//
//  -------------------------------------------
//  XaDictionary method implementations
//  -------------------------------------------
// 

//
// --------- Constructors / destructors
// 

XaDictionary::XaDictionary(unsigned numBuckets) : XaBag(numBuckets)
{
}

XaDictionary::XaDictionary(const XaDictionary& another) 
  : XaBag((const XaBag &)another), values(another.values)
{
}

//
// --------- Object insertion and removal
//

// multiplicative hash from Knuth vol.3 pp 508-512
unsigned XaDictionary::multHash(const XaCollectable& targetKey) const
{
    unsigned mask = allocSize - 1;
    const unsigned long multiplier = 0x9E3779B9;
    return ((multiplier*targetKey.hash())>>((8*sizeof(unsigned))-nBits)) & mask;
    //return ((multiplier*(unsigned long)targetKey)>>((8*sizeof(unsigned))-nBits)) & mask;
}

unsigned XaDictionary::findIndexOf(const XaCollectable& targetKey) const
{
    unsigned mask = allocSize - 1;
    int i;
    for (i=multHash(targetKey); objectArray[i]!=NULL; i=(i-1)&mask)
    {
	if (objectArray[i]->equals(targetKey))
	    return i;
    }
    return i;
}

XaCollectable* XaDictionary::find(const XaCollectable& targetKey) const {
    unsigned index = findIndexOf(targetKey);
    return at(index);
}

XaCollectable* XaDictionary::findKeyAndValue(const XaCollectable& targetKey, 
					     void *&valueResult) {
    unsigned index = findIndexOf(targetKey);
    XaPointer *ptr = (XaPointer *)values.at(index);
    if (ptr)
      valueResult = (void *)*ptr;
    else
      valueResult = NULL;
    return at(index);              
}

void* XaDictionary::findValue(const XaCollectable& targetKey) const {

    unsigned index = findIndexOf(targetKey);
    XaPointer *ptr = (XaPointer *)values.at(index);
    if (ptr)
      return (void *)*ptr;
    else
      return NULL;
}

void* XaDictionary::findValue(const XaCollectable& targetKey, void *newValue) {

    unsigned index = findIndexOf(targetKey);
    XaPointer *pValue = new XaPointer (&newValue);
    XaPointer *ptr = (XaPointer *)values.replaceAtIndex(index, *pValue);
    if (*ptr)
      return (void*)*ptr;
    else
      return NULL;
}

XaCollectable* XaDictionary::insertKeyAndValue(XaCollectable& newKey,
					       void *newValue) {

    // make space if we are half full - this guarantees that the
    // searching will be reasonably efficient
    if (nEntries >= allocSize/2)     {
	if (doubleSize() != XA_SUCCESS)
	    return NULL;
	
	// REMIND: if we run out of memory on this one, the dictionary
	// will be corrupted!
	if (values.doubleSize() != XA_SUCCESS)
	    return NULL;

	// remove and re-insert all the old objects in the new 
	// double sized array, but in the same positions.
	XaBag keyBag(*this);
	XaBag valueBag(values);

        // free space from old arrays.
	XaBag::clear();
	values.clear();

	// redistribute keys/values in the new
	// double sized array.
	XaCollectable* pKey;
	XaPointer* pValue;
	XaBagIterator keyBagIt(keyBag);
	XaBagIterator valueBagIt(valueBag);

	while (pKey = keyBagIt()) {
	    pValue = (XaPointer *)valueBagIt();
	    insertKeyAndValue(*pKey, (void *)*pValue);
	    // We don't need the pointer object anymore
	    // so delete it.
	    // jn: Can't delete it, up to user to clean up.
  	    //     Deleting pValue causes segv.
	    // delete pValue;
	}
	keyBag.clear();
	valueBag.clear();
    }

    int i = findIndexOf(newKey);

    // Don't allow insertion if there is already such a key
    if (objectArray[i] != NULL)
	return NULL;

    XaCollectable* pXa = objectArray[i] = &newKey;
    nEntries++;

    /* Encapsulate the void * in an XaPointer object */
    values.objectArray[i] = new XaPointer (newValue);
    values.nEntries++;

    return pXa;
}

unsigned XaDictionary::occurrencesOf(const XaCollectable& targetKey) const
{
    if (contains(targetKey))
	return 1;
    else
	return 0;
}

void* XaDictionary::remove(const XaCollectable& targetKey,
			   XaCollectable *&rKey) {

    // Get the key and value to remove.
    int i = findIndexOf(targetKey);
    if (objectArray[i] == NULL)
	return NULL;

    XaCollectable* pKey = objectArray[i];
    XaPointer* pValue = (XaPointer*)values.objectArray[i];
    if (pKey == NULL)
	return NULL;

    unsigned mask = allocSize - 1;
    int j, r;
    for (;;)
    {
	// remove the object and shift all the other like-hashing
	// objects down to fill in the hole.
	objectArray[j=i] = NULL;
	values.objectArray[i] = NULL;
	do 
	{
	    i = (i-1) & mask;
	    if (objectArray[i] == NULL)
	    {
		nEntries--;
		values.nEntries--;

		// Exit
		goto cleanUp;
	    }
	    r = multHash(*objectArray[i]);

	    // the complicated test here is to see if r lies cyclically 
	    // between i and j
	} while ((i<=r&&r<j) || (r<j&&j<i) || (j<i&&i<=r));
	objectArray[j] = objectArray[i];
	values.objectArray[j] = values.objectArray[i];
    }

cleanUp:    
//
// Do Not delete the Key.
// But clean up the XaPointer.
// Returning the value of the value.
//
    rKey = pKey;
    if (pValue) {
      void *v = (void *)*pValue;
      delete pValue;
      return v;
    }
    else
      return NULL;
}


void *XaDictionary::remove(const XaCollectable &targetKey) {
    XaCollectable *dummy;
    return XaDictionary::remove(targetKey, dummy);
}

void XaDictionary::removeAndDestroy(const XaCollectable& targetKey,
				     void *&value) {
    XaCollectable *theKey;
    void *v = XaDictionary::remove(targetKey, theKey);
    delete theKey;
    value = v;
}

void XaDictionary::removeAndDestroy(const XaCollectable &targetKey) {
    removeAndDestroy(targetKey);
}

// return the bag of keys.
XaBag *XaDictionary::getKeys() {
    return new XaBag(*this);
}
    
XaDictionary::~XaDictionary(void) {
    XaBag::clear();
    values.clear();
}


// Nondestructive remove.
void XaDictionary::clear() {
    XaBag::clear();
    // This delete the XaPointer objects
    // that are in the Bag.
    // The void* in the XaPointers are left alone.
    values.clearAndDelete();
}


//
// --------- Copy support
//

XaDictionary& XaDictionary::operator=(const XaDictionary& d) {

    values = d.values;
    this->XaBag::operator=(d);
    return *this;
}


#ifdef LATER
XaBoolean XaDictionary::equals(const XaCollectable& anObject) const
{
  if (anObject.isAKindOf(XA_DICTIONARY))
    {
      XaDictionary* pAnotherDict = (XaDictionary*) &anObject;
      if (!this->Inherited::equals(*pAnotherDict))
	return XaFalse
      if (!this->values.equals(pAnotherDict->values))
	return XaFalse;
      return XaTrue;
    }
  else if (anObject.isAKindOf(XA_BAG))
    return anObject.equals(*this);
  else
    return XaFalse;
}
#endif

//
//  --------------------------------------------
//  XaDictionaryIterator method implementations
//  --------------------------------------------
// 

//
// --------- Constructors / destructors
// 

XaDictionaryIterator::XaDictionaryIterator(const XaDictionary& theDict) 
  : XaBagIterator(theDict)
{
      pDict = (XaDictionary *) &theDict;
}

XaDictionaryIterator::~XaDictionaryIterator()
{
//    pDict->release();
}

//
// --------- Iteration and contained object operations
//

XaCollectable* XaDictionaryIterator::findNext(const XaCollectable& target)
{
    currentIndex = pDict->findIndexOf(target);
    XaCollectable* pXa = pDict->at(currentIndex);
    if (pXa == NULL)
	return NULL;
    return pXa;
}

void* XaDictionaryIterator::remove(void) {
    XaCollectable* pXa = pDict->at(currentIndex);
    if (pXa == NULL)
	return NULL;
    return pDict->remove(*pXa);
}

void* XaDictionaryIterator::value(void) const {
    XaPointer* pXa = (XaPointer*) pDict->values.at(currentIndex);
    return (void*)*pXa;
}

void* XaDictionaryIterator::value(void*& newValue) const {
    XaPointer *pValue = new XaPointer (&newValue);
    XaCollectable* pXa = pDict->values.replaceAtIndex(currentIndex, *pValue);
    return pXa;
}

