//This line lets emacs recognize this as -*- C++ -*- Code
/* $XConsortium: bag.cc /main/5 1996/12/30 16:29:12 swick $ */

/*
Copyright (c) 1996  X Consortium

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from the X Consortium.
*/

//

#include "xadefines.h"
#include "bag.h"
#include "collectable.h"
#ifndef USL
#include <malloc.h>
#endif
#include <stdlib.h>
#ifdef hpux
#include <stream.h>
#else
#include <sys/stream.h>
#endif

XaBag::XaBag(unsigned N)
{
    if (N < XA_DEFAULT_COLLECTION_SIZE)
	N = XA_DEFAULT_COLLECTION_SIZE;

    /*
     * round N up to power of 2 (required for XaDictionary)
     */
    nBits = 0;
    unsigned s;
    for (s=N-1; s!=0; s>>=1) 
	nBits++;
    if ((N&(N-1)) != 0)
	nBits++; 	/* round up */
    N = 1<<nBits;

    allocSize   = N;

    finishConstructor();
}

XaBag::XaBag(const XaBag& bag)
{
    allocSize   = bag.allocSize;
    nBits = bag.nBits;
    finishConstructor();

    unsigned i;
    for (i=0; i<allocSize; i++)
	//objectArray[i] = (bag.objectArray[i])->get();
	objectArray[i] = bag.objectArray[i];
    nEntries = bag.nEntries;
}

XaBag& XaBag::operator=(const XaBag& bag)
{
    if( this == &bag ) return *this;

    XaBag::clear();
    if (objectArray != 0)
	free(objectArray);

    allocSize   = bag.allocSize;
    nBits = bag.nBits;
    XaBag::finishConstructor();

    unsigned i;
    for (i=0; i<allocSize; i++)
	objectArray[i] = bag.objectArray[i];
    nEntries = bag.nEntries;
    return *this;
}

void XaBag::finishConstructor()
{
    objectArray = (XaCollectable**) malloc(allocSize*sizeof(XaCollectable*));

#ifdef LATER
// validStatus is from Base
    if (objectArray == 0)
	validStatus = XA_OUT_OF_MEMORY;
#endif
    unsigned i;
    for (i=0; i<allocSize; i++)
	objectArray[i] = NULL;
    nEntries = 0;

}

void XaBag::clear() {
    unsigned i;
    for (i=0; i<allocSize; i++) {
	objectArray[i] = NULL;
    }
    nEntries = 0;
}

void XaBag::clearAndDelete() {
    unsigned i;
    for (i=0; i<allocSize; i++) {
	delete objectArray[i];
	objectArray[i] = NULL;
    }
    nEntries = 0;
}

XaBag::~XaBag()
{
    XaBag::clear();
    if (objectArray != 0)
	free(objectArray);
}

XaCollectable* XaBag::find(const XaCollectable& object) const
{
    unsigned count = 0;
    while (count < allocSize)
    {
	if (objectArray[count] && objectArray[count]->equals(object))
	    return (objectArray[count]);
	count++;
    }
    return 0;
}

// Resizes the arrays, and puts the elements
// from the old arrays into the new ones.
XaErrorCode XaBag::doubleSize() {
    
    unsigned newSize = allocSize*2;
    XaCollectable** oldObjectArray = objectArray;
    objectArray = (XaCollectable**) malloc(newSize*sizeof(XaCollectable*));
    if (objectArray == 0) {
	objectArray = oldObjectArray;
	return XA_OUT_OF_MEMORY;
    }
    
    unsigned i;
    for (i=0;i<allocSize;i++)
        objectArray[i] = oldObjectArray[i];
    for ( ; i<newSize; i++)
	objectArray[i] = NULL;
    allocSize = newSize;
    free(oldObjectArray);
    return XA_SUCCESS;
}

XaCollectable* XaBag::at(unsigned index) const
{
    if (index >= allocSize)
	return NULL;
    XaCollectable* pXa = objectArray[index];
    if (pXa == NULL)
       return NULL;
    return pXa;
}

XaCollectable* XaBag::replaceAtIndex(unsigned index, XaCollectable& theObject)
{
    if (index >= allocSize)
	return NULL;

    XaCollectable* pXa = objectArray[index];

    if (pXa == NULL)
       return NULL;

    // put new object at the index
    objectArray[index] = &theObject;
    return pXa;
}

XaCollectable* XaBag::insert(XaCollectable& object)
{
    // check to see if the bag is full first, and reallocate if it is.
    if (entries() == allocSize)
    {
	if (doubleSize() != XA_SUCCESS)
	    return NULL;
    }

    unsigned count = 0;
    while (count < allocSize)
    {
	// if we find an opening, drop the object in
	if (objectArray[count] == NULL)
	{
	    objectArray[count] = &object;
	    nEntries++;
	    return &object;
	}
	count++;
    }

    // this should never happen:
    return 0;
}

XaBoolean XaBag::isEmpty() const
{
    return (entries() == 0);
}

XaBoolean XaBag::contains(const XaCollectable& object) const
{
    XaCollectable* pXa = find(object);
    if (pXa == NULL)
	return XaFalse;
    //pXa->release();
    return XaTrue;
}

unsigned XaBag::occurrencesOf(const XaCollectable& object) const
{
    unsigned i;
    unsigned count = 0;
    for (i=0; i<allocSize; i++)
    {
	if (objectArray[i] && (objectArray[i])->equals(object))
	    count++;
    }
    return count;
}

void* XaBag::remove(const XaCollectable& object) {

    XaCollectable* pXa = 0;
    unsigned count = 0;
    while (count < allocSize)
    {
	pXa = objectArray[count];
	if (pXa && pXa->equals (object))
	{
	    objectArray[count] = NULL;
	    nEntries--;
	    break;
	}
	count++;
    }
    return pXa;
}

void XaBag::removeAndDestroy(const XaCollectable& object)
{
    XaCollectable* pXa = (XaCollectable*)remove(object);
    delete pXa;
}

//
// --- Implementation of iterator ---
//

XaBagIterator::XaBagIterator(const XaBag& bag)
{
    pBag = (XaBag *)&bag;
    currentIndex = -1;
}

XaBagIterator::~XaBagIterator()
{
    pBag = NULL;
}

// Advances iterator to the next item and returns it.  Returns NULL when
// the end of the collection has been reached.
XaCollectable* XaBagIterator::operator()()
{
    do
    {
	currentIndex++;
	if (currentIndex >= pBag->allocSize)
	    return NULL;
    } while (pBag->objectArray[currentIndex] == NULL);

    // if we get here, we are pointing to the next item
    return (pBag->objectArray[currentIndex]);
}

// Moves iterator to the next item which equals the object pointed
// to by target, and returns it.  If no such item exists, NULL is returned 
// and the state of the iterator is then undefined.
XaCollectable* XaBagIterator::findNext(const XaCollectable& item)
{
#ifndef DO_RELEASES
    while (operator()() != NULL)
	if (pBag->objectArray[currentIndex]->equals(item))
	    return (pBag->objectArray[currentIndex]);
#else
    XaCollectable* pNext = NULL;
    while ((pNext = operator()()) != NULL)
    {
        pNext->release();
	if (pBag->objectArray[currentIndex]->equals(item))
	    return (pBag->objectArray[currentIndex]);
    }
#endif
    return NULL;
}

// Return the key at the current position in the bag.
XaCollectable* XaBagIterator::key() const
{
    if (currentIndex == -1)
	return (pBag->objectArray[0]);
    return (pBag->objectArray[currentIndex]);
}

// Reset the iterator to its initial state.
void XaBagIterator::reset()
{
    currentIndex = -1;
}

XaScalarBag::XaScalarBag(unsigned N) : XaBag(N) {}
XaScalarBag::XaScalarBag(const XaBag&bag) : XaBag(bag) {}

XaScalarBag::~XaScalarBag(void) {}

XaScalarBag &XaScalarBag::operator=(XaScalarBag const &bag) {
    XaBag::operator=(bag);
    return *this;
}

// XXX following routine should use hash for improved speed
XaCollectable* XaScalarBag::find(unsigned scalar) const
{
    unsigned count = 0;
    while (count < allocSize)
    {
	if (objectArray[count] && objectArray[count]->hash() == scalar)
	    return (objectArray[count]);
	count++;
    }
    return 0;
}

XaCollectable* XaScalarBag::find(const XaCollectable& theObject) const
{ return XaBag::find(theObject); }

XaBoolean XaScalarBag::contains(unsigned scalar) const
{
    XaCollectable* pXa = find(scalar);
    if (pXa == NULL)
	return XaFalse;
    //pXa->release();
    return XaTrue;
}

XaBoolean XaScalarBag::contains(const XaCollectable& theObject) const
{ return XaBag::contains(theObject); }

unsigned XaScalarBag::occurrencesOf(unsigned scalar) const
{
    unsigned i;
    unsigned count = 0;
    for (i=0; i<allocSize; i++)
    {
	if (objectArray[i] && (objectArray[i])->hash() == scalar)
	    count++;
    }
    return count;
}

unsigned XaScalarBag::occurrencesOf(const XaCollectable& theObject) const
{ return XaBag::occurrencesOf(theObject); }

void* XaScalarBag::remove(unsigned scalar) {

    XaCollectable* pXa = 0;
    unsigned count = 0;
    while (count < allocSize)
    {
	pXa = objectArray[count];
	if (pXa && pXa->hash() == scalar)
	{
	    objectArray[count] = NULL;
	    nEntries--;
	    break;
	}
	count++;
    }
    return pXa;
}

void *XaScalarBag::remove(const XaCollectable& theObject)
{ return XaBag::remove(theObject); }

void XaScalarBag::removeAndDestroy(unsigned scalar)
{
    XaCollectable* pXa = (XaCollectable*)remove(scalar);
    delete pXa;
}

void XaScalarBag::removeAndDestroy(const XaCollectable& theObject)
{ XaBag::removeAndDestroy(theObject); }

XaScalarBagIterator::XaScalarBagIterator(const XaBag& bag) : XaBagIterator(bag)
{}

XaCollectable* XaScalarBagIterator::findNext(unsigned scalar)
{
#ifndef DO_RELEASES
    while (operator()() != NULL)
	if (pBag->objectArray[currentIndex]->hash() == scalar)
	    return (pBag->objectArray[currentIndex]);
#else
    XaCollectable* pNext = NULL;
    while ((pNext = operator()()) != NULL)
    {
        //pNext->release();
	if (pBag->objectArray[currentIndex]->hash() == scalar)
	    return (pBag->objectArray[currentIndex]);
    }
#endif
    return NULL;
}

XaCollectable* XaScalarBagIterator::findNext(const XaCollectable& theObject)
{ return XaBagIterator::findNext(theObject); }
