/******************************************************************************
 * $RCSfile: xwhoserver.c,v $   $Revision: 1.20 $
 * Copyright (C) 1994  Technische Universitaet Muenchen
 ******************************************************************************/

#include "xwho.h"
#include <misc/math.h>

extern int verboseflag;


/*----------------------------------------------------------------------------*
 * intern variables
 *----------------------------------------------------------------------------*/

static xwhoservers servers = NULL;


/*----------------------------------------------------------------------------*
 * functions
 *----------------------------------------------------------------------------*/

/*--- servers ----------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
void
    Init_servers(conn input)
{
	if (verboseflag)
	    puts("building server-list ...");
	servers = Init_xwhoservers(input);
	if (verboseflag)
	    puts("subscript done");
}


/*----------------------------------------------------------------------------*/
void
    SetLastUpdate_servers(string serverhost)
{
	xwhoserver s = Seek_servers(serverhost);

	if (s != NULL)
	    time(&(s->lastupdate));
}


/*----------------------------------------------------------------------------*/
xwhoserver
    Seek_servers(string serverhost)
{
	return ( Seek_xwhoservers(servers, serverhost) );
}


/*----------------------------------------------------------------------------*/
void
    ReSubscribe_servers(string serverhost,
			int port)
{
	xwhoserver s = Seek_servers(serverhost);

	if (s != NULL)
	    Subscribe_xwhoserver(s, port);
}


/*----------------------------------------------------------------------------*/
void
    Check_servers(int port)
{
	time_t now;
	list l;

	if (verboseflag)
	    printf("checking for lost connections ...\n");
	time(&now);
	now -= XWHO_SERVER_CHECK_TIME;
	for (l = servers; ! Empty_list(l); l = Next_list(l)) {
		xwhoserver s = First_list(l);

		if (s->lastupdate < now)
		    Subscribe_xwhoserver(s, port);
	}
	XtAppAddTimeOut(appctxt, XWHO_SERVER_CHECK_TIME * 1000,
			(XtTimerCallbackProc)Check_servers, (XtPointer)port);
}


/*--- xwhoservers ------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
xwhoservers
    New_xwhoservers()
{
	return ((xwhoservers)NULL);
}


/*----------------------------------------------------------------------------*/
xwhoservers
    Init_xwhoservers(conn input)
{
	xwhoservers sl = New_xwhoservers();
	xwhowidget w;
	list l;

	while ((w = RetrieveHost_widgetlines())  !=  NULL) {
		xwhodaemon d;

		d = GetConfigServer(w->host.name);
		if (d == NULL) {
			if (Seek_xwhoservers(sl, w->host.name) == NULL)
			    d = Init_xwhodaemon(w->host.name);
		}
		else {
			if (Seek_xwhoservers(sl, d->hostname) == NULL)
			    d = Copy_xwhodaemon(d);
			else
			    d = NULL;
		}
		if (d != NULL) {
			xwhoserver s = New_xwhoserver();

			s->daemon = d;
			Add_xwhoservers(&sl, s);
		}
	}

	for (l = (list)sl; ! Empty_list(l); l = Next_list(l)) {
		xwhoserver s = (xwhoserver)First_list(l);

		Subscribe_xwhoserver(s, ((struct conn *)input)->port);
	}

	return (sl);
}


/*----------------------------------------------------------------------------*/
void
    Add_xwhoservers(xwhoservers* sl_ref,
		    xwhoserver new)
{
	AddTrueNewSorted_list((list*)sl_ref, (void*)new, Cmp_xwhoserver);
}


/*----------------------------------------------------------------------------*/
void
    Del_xwhoservers(xwhoservers* sl_ref,
		    xwhoserver s)
{
	xwhoserver del;

	del = (xwhoserver)DelSorted_list((list*)sl_ref, (void*)s, NULL,
					 Cmp_xwhoserver);
	Free_xwhoserver((void**)&del);
}


/*----------------------------------------------------------------------------*/
xwhoserver
    Seek_xwhoservers(xwhoservers sl,
		     string serverhost)
{
	list l;

	for (l = sl; ! Empty_list(l); l = Next_list(l)) {
		xwhoserver s = First_list(l);

		if (strcmp(s->daemon->hostname, serverhost) == 0)
		    return (s);
	}
	return (NULL);
}


/*----------------------------------------------------------------------------*/
void
    Free_xwhoservers(void** sl_ref)
{
	Free_list((list*)sl_ref, Free_xwhoserver);
}


/*--- xwhoserver -------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
xwhoserver
    New_xwhoserver()
{
	xwhoserver s;

	s = TrueReAlloc(NULL, 1, sizeof (struct xwhoserver));
	s->daemon = New_xwhodaemon();
	s->lastupdate = 0;
	return (s);
}


/*----------------------------------------------------------------------------*/
int
    Cmp_xwhoserver(void* server1,
		   void* server2)
{
	xwhoserver s1 = server1;
	xwhoserver s2 = server2;

	return ( Cmp_xwhodaemon((void*)s1->daemon, (void*)s2->daemon) );
}


/*----------------------------------------------------------------------------*/
void
    Subscribe_xwhoserver(xwhoserver s,
			 int port)
{
	conn streamconn;
	int protocol_version;
	xwhodservedhosts hostdata;

	if (verboseflag)
	    printf("subscribing to server on %s ...\n",
		   s->daemon->hostname);

#define WITH_PING
#ifdef WITH_PING
	if (! PingHost(s->daemon->hostname)) {
		Warning("%s not reachable", s->daemon->hostname);
		return;
	}
#endif

	streamconn = Connect_conn(NULL, s->daemon->hostname, INETD_PORT,
				  0, CONN_STREAM);
	Close_conn(&streamconn);

	streamconn = Connect_conn(NULL, s->daemon->hostname, s->daemon->port,
				  2, CONN_STREAM);
	if (streamconn != NULL) {
		protocol_version = SettleProtocol_client(streamconn);
		if (protocol_version < 0) {
			Warning("can't communicate with server on %s",
				s->daemon->hostname);
			Close_conn(&streamconn);
		}
		else {
			xwhodservedhost h;

			SendClientSubscript(streamconn, myhostname, port,
					    protocol_version, &hostdata);
			Close_conn(&streamconn);

			while ((h = TakeOut_xwhodservedhosts(&hostdata))
			       != NULL)
			    Update_widgetlines(h);

			time(&(s->lastupdate));
			s->lastupdate += RandIntInterval(0, 200);
		}
	}
	else
	    Warning("no connection to server on %s", s->daemon->hostname);
}


/*----------------------------------------------------------------------------*/
void
    Free_xwhoserver(void** s_ref)
{
	xwhoserver s = *s_ref;

	if (s != NULL) {
		Free_xwhodaemon((void*)&(s->daemon));
		Free(s_ref);
	}
}
