/******************************************************************************
 * $RCSfile: Protocol.c,v $   $Revision: 1.11 $
 * Copyright (C) 1994  Technische Universitaet Muenchen
 ******************************************************************************/

#include "xwho.h"

#define MY_PROTOCOL 1

#define MESG_COMMUNICATION_END 1
#define MESG_END	       2
#define MESG_NEXT              3
#define MESG_ACKNOWLEDGE       4
#define MESG_PROTOCOL          10
#define MESG_CLIENT_DATA       20
#define MESG_CLIENT_UPDATE     21
#define MESG_CLIENT_TIMEOUT    22


/*----------------------------------------------------------------------------*
 * intern variables
 *----------------------------------------------------------------------------*/

static int protocol_version = 0;


/*----------------------------------------------------------------------------*
 * functions
 *----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
int
    SettleProtocol_server(conn stream)
{
	char* str;

	/* hand-shake */
	SendString_conn(stream, "I'm an XWho-server");
	str = RecvString_conn(stream);
	if (str == NULL  ||  strcmp(str, "I'm an XWho-client") != 0) {
		Warning("protocol error in SettleProtocol()");
		return (-1);
	}

	/* settle protocol */
	if ((protocol_version = RecvINT_conn(stream)) > MY_PROTOCOL)
	    protocol_version = MY_PROTOCOL;
	SendINT_conn(stream, protocol_version);

	return (protocol_version);
}


/*----------------------------------------------------------------------------*/
int
    SettleProtocol_client(conn stream)
{
	char* str;

	/* hand-shake */
	str = RecvString_conn(stream);
	if (str == NULL  ||  strcmp(str, "I'm an XWho-server") != 0) {
		Warning("protocol error in SettleProtocol()");
		return (-1);
	}
	SendString_conn(stream, "I'm an XWho-client");

	/* settle protocol */
	SendINT_conn(stream, MY_PROTOCOL);
	protocol_version = RecvINT_conn(stream);

	return (protocol_version);
}


/*----------------------------------------------------------------------------*/
void
    SendProtocol(conn c,
		 int version)
{
	if (version > MY_PROTOCOL)
	    Error(-1, "can't set protocol version %d", version);
	else
	    protocol_version = version;
	SendString_conn(c, "XWho-message");
	SendINT_conn(c, MESG_PROTOCOL);
	SendINT_conn(c, protocol_version);
}


/*----------------------------------------------------------------------------*/
int
    RecvProtocol(conn c)
{
	char* str;

	str = RecvString_conn(c);
	if (str == NULL || strcmp(str,"XWho-message") != 0)
	    return (-1);
	if (RecvINT_conn(c) != MESG_PROTOCOL) {
		Warning("protocol error in RecvProtocol()");
		protocol_version = -1;
	}
	else
	    if ((protocol_version = RecvINT_conn(c))  >  MY_PROTOCOL) {
		    Warning("can't set protocol version %d", protocol_version);
		    protocol_version = -1;
	    }
	return (protocol_version);
}


/*----------------------------------------------------------------------------*/
void
    SendClientSubscript(conn stream,
			string hostname,
			int port,
			int protocol,
			xwhodservedhosts* hosts_ref)
{
	SendINT_conn(stream, MESG_CLIENT_DATA);
	SendString_conn(stream, hostname);
	SendINT_conn(stream, port);
	SendINT_conn(stream, protocol);
	if (RecvINT_conn(stream) != MESG_ACKNOWLEDGE)
	    Error(-1, "server communication error in SendClientSubscript()");
	*hosts_ref = Recv_xwhodservedhosts(stream);
	CommunicationEnd_client(stream);
}


/*----------------------------------------------------------------------------*/
int
    RecvServerInput(conn stream,
		    string* hostname_ref,
		    int* port_ref,
		    int* protocol_ref,
		    xwhodservedhosts hosts)
{
	if (RecvINT_conn(stream) != MESG_CLIENT_DATA) {
		Warning("client communication error in RecvServerInput()");
		return (-1);
	}
	StrTrueReAlloc(hostname_ref, RecvString_conn(stream));
	*port_ref = RecvINT_conn(stream);
	*protocol_ref = RecvINT_conn(stream);
	SendINT_conn(stream, MESG_ACKNOWLEDGE);
	Send_xwhodservedhosts(stream, hosts);
	CommunicationEnd_server(stream);
	return (0);
}


/*----------------------------------------------------------------------------*/
void
    SendUpdateData(conn clientinput,
		   xwhodservedhost host)
{
	SendINT_conn(clientinput, MESG_CLIENT_UPDATE);
	SendString_conn(clientinput, myhostname);
	Send_xwhodservedhost(clientinput, host);
}


/*----------------------------------------------------------------------------*/
void
    SendTimeOut(conn clientinput)
{
	SendINT_conn(clientinput, MESG_CLIENT_TIMEOUT);
	SendString_conn(clientinput, myhostname);
}


/*----------------------------------------------------------------------------*/
int
    RecvClientInput(conn clientinput,
		    xwhodservedhost* host_ref,
		    string* serverhost_ref)
{
	string serverhost;

	*host_ref = NULL;
	switch (RecvINT_conn(clientinput)) {
	    case MESG_CLIENT_UPDATE:
		if ((serverhost = RecvString_conn(clientinput)) == NULL)
		    return (-1);
		StrTrueReAlloc(serverhost_ref, serverhost);
		*host_ref = Recv_xwhodservedhost(clientinput);
		break;
	    case MESG_CLIENT_TIMEOUT:
		if ((serverhost = RecvString_conn(clientinput)) == NULL)
		    return (-1);
		StrTrueReAlloc(serverhost_ref, serverhost);
		return (1);
	    default:
		return (-1);
	}
	return (0);
}


/*----------------------------------------------------------------------------*/
void
    SendHostData(conn clientinput,
		 string hostname)
{
	SendString_conn(clientinput, hostname);
}


/*----------------------------------------------------------------------------*/
void
    RecvHostData(conn input,
		 string* hostname_ref)
{
	StrTrueReAlloc(hostname_ref, RecvString_conn(input));
}


/*----------------------------------------------------------------------------*/
void
    SendLoginData(conn clientinput,
		  string login,
		  string fullname,
		  string nick,
		  int islocal)
{
	SendString_conn(clientinput, login);
	SendString_conn(clientinput, fullname);
	SendString_conn(clientinput, nick);
	SendINT_conn(clientinput, islocal);
}


/*----------------------------------------------------------------------------*/
void
    RecvLoginData(conn input,
		  string* login_ref,
		  string* fullname_ref,
		  string* nick_ref,
		  int* islocal_ref)
{
	StrTrueReAlloc(login_ref, RecvString_conn(input));
	StrTrueReAlloc(fullname_ref, RecvString_conn(input));
	StrTrueReAlloc(nick_ref, RecvString_conn(input));
	*islocal_ref = RecvINT_conn(input);
}


/*----------------------------------------------------------------------------*/
void
    SendTtyData(conn clientinput,
		string tty,
		string rhost)
{
	SendString_conn(clientinput, tty);
	SendString_conn(clientinput, rhost);
}


/*----------------------------------------------------------------------------*/
void
    RecvTtyData(conn input,
		string* tty_ref,
		string* rhost_ref)
{
	StrTrueReAlloc(tty_ref, RecvString_conn(input));
	StrTrueReAlloc(rhost_ref, RecvString_conn(input));
}


/*----------------------------------------------------------------------------*/
void
    ProtocolNext(conn c)
{
	SendINT_conn(c, MESG_NEXT);
}


/*----------------------------------------------------------------------------*/
void
    ProtocolEnd(conn c)
{
	SendINT_conn(c, MESG_END);
}


/*----------------------------------------------------------------------------*/
int
    ProtocolHasNext(conn c)
{
	switch (RecvINT_conn(c)) {
	    case MESG_NEXT:
		return (1);
	    case MESG_END:
		return (0);
	}
	Warning("protocol error in ProtocolHasNext()");
	return (0);
}


/*----------------------------------------------------------------------------*/
void
    CommunicationEnd_server(conn stream)
{
	SendINT_conn(stream, MESG_COMMUNICATION_END);
	if (RecvINT_conn(stream) != MESG_COMMUNICATION_END)
	    Warning("client communication error in CommunicationEnd()");
}


/*----------------------------------------------------------------------------*/
void
    CommunicationEnd_client(conn stream)
{
	if (RecvINT_conn(stream) != MESG_COMMUNICATION_END)
	    Warning("server communication error in CommunicationEnd()");
	SendINT_conn(stream, MESG_COMMUNICATION_END);
}
