/******************************************************************************
 * $RCSfile: ipc.c,v $   $Revision: 1.20 $
 * Copyright (C) 1993  A.Michael.Leliveld@Informatik.TU-Muenchen.De
 ******************************************************************************/

#include <misc/ipc.h>
#include <misc/str.h>
#include <misc/file.h>
#include <misc/mem.h>
#include <errno.h>


/*----------------------------------------------------------------------------*
 * intern variables
 *----------------------------------------------------------------------------*/

static char* sendbuf = NULL;
static int sendbufpos = 0;
static char* recvbuf = NULL;
static int recvbufpos = 0;
static int recvbuflen = 0;


/*----------------------------------------------------------------------------*
 * intern prototypes
 *----------------------------------------------------------------------------*/

static void
    TestFlags_conn(unsigned  /* flags */
		   );


/*----------------------------------------------------------------------------*
 * functions
 *----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
conn
    Open_conn(char* host,
	      int min_port,
	      int max_port,
	      unsigned flags)
{
	conn c;
	int prottype;
	string localhost = NULL;

	/* test the parameters */
	if (host != NULL  &&  !TestHostname(host)) {
		Warning("illegal hostname \"%s\"", host);
		return (NULL);
	}
	TestFlags_conn(flags);

	/* initialize the variables */
	c = (conn)TrueReAlloc(NULL, 1, sizeof (struct conn));
	c->flags = flags;
	StrTrueReAlloc(&localhost, GetFullHostname());


	bzero((char*)&(c->from), sizeof (c->from));
	if (!(flags & CONN_DATAGRAM)  &&  host != NULL  &&
	    strcmp(localhost, GetHostnameByName(host)) == 0) {
		/* open an Unix-connection */
		c->addrfam = c->from.un.sun_family = AF_UNIX;
		strcpy(c->from.un.sun_path, IPC_UNIXSOCKTEMPL);
		mktemp(c->from.un.sun_path);		/* make a unique
							 * socket name */
		c->addrlen = sizeof (c->from.un.sun_family) +
		    strlen(c->from.un.sun_path);
		prottype = SOCK_STREAM;
	}
	else {
		struct hostent *he;

		/* open an Internet-connection */
		c->addrfam = c->from.in.sin_family = AF_INET;
		he = gethostbyname(localhost);
		bcopy((char*)(he->h_addr_list[0]),
		      (char*)&(c->from.in.sin_addr), he->h_length);
		c->addrlen = sizeof (struct sockaddr_in);
		prottype = (flags & CONN_STREAM) ? SOCK_STREAM : SOCK_DGRAM;
	}

	if (flags & CONN_INETD)
	    c->sock = 0;
	else {
		if ((c->sock = socket(c->addrfam, prottype, 0))  <  0) {
			perror("socket()");
			Free((void**)&c);
			Error(-1, "can't open socket");
		}

		if (c->addrfam == AF_UNIX) {
			c->port = -1;
			if (bind(c->sock, (void*)&(c->from), c->addrlen) < 0) {
				perror("bind()");
				Free((void**)&c);
				unlink(c->from.un.sun_path);
				return (NULL);
			}
		}
		else {
			for (c->port = min_port; c->port <= max_port;
			     c->port++) {
				c->from.in.sin_port = htons(c->port);
				if (bind(c->sock, (void*)&(c->from),
					 c->addrlen) == 0)
				    break;
			}
			if (c->port > max_port) {
				perror("bind()");
				Free((void**)&c);
				Warning("can't bind a port");
				return(NULL);
			}
		}

		if (c->flags & CONN_STREAM) {
			if (listen(c->sock, IPC_LISTENQUEUELEN)  <  0) {
				perror("listen()");
				Free((void**)&c);
				return (NULL);
			}
		}
	}
	c->flags |= CONN_ISLISTENER;

	return (c);
}


/*----------------------------------------------------------------------------*/
conn
    Accept_conn(conn lc,
		long sec)
{
	conn c;

	if (lc == NULL  ||  lc->flags & CONN_DATAGRAM)
	    return (lc);

	c = (conn)TrueReAlloc(NULL, 1, sizeof (struct conn));
	c->flags = lc->flags - CONN_ISLISTENER;

	if (lc->addrfam == AF_UNIX) {
		c->addrfam = AF_UNIX;
		c->addrlen = sizeof (struct sockaddr_un);
	}
	else {
		c->addrfam = AF_INET;
		c->addrlen = sizeof (struct sockaddr_in);
	}
	c->port = lc->port;

	if (Test_conn(lc->sock, CONN_ISREAD, sec)) {
		if ((c->sock = accept(lc->sock, (void*)&(c->to),
				      &(c->addrlen)))  <  0) {
			perror("accept()");
			Free((void**)&c);
		}
	}
	else {
		Free((void**)&c);
	}

	return (c);
}


/*----------------------------------------------------------------------------*/
conn
    Connect_conn(char* sockpath,
		 char* host,
		 int port,
		 long sec,
		 unsigned flags)
{
	conn c;
	int prottype;

#define NONBLOCK_BUG
	/* TODO: nonblocking is yet buggy */
#ifdef NONBLOCK_BUG
	if (sec != -1)
	    sec = -1;  /* sorry for that ;) */
#endif

	TestFlags_conn(flags);

	c = (conn)TrueReAlloc(NULL, 1, sizeof (struct conn));

	c->flags = flags;

	bzero((char*)&(c->to), sizeof (c->to));
	if (port == -1) {
		c->addrfam = c->to.un.sun_family = AF_UNIX;
		strcpy(c->to.un.sun_path, sockpath);
                c->addrlen = sizeof (c->to.un.sun_family) +
		    strlen(c->to.un.sun_path);
		prottype = SOCK_STREAM;
	}
	else {
		struct hostent *he;

		c->addrfam = c->to.in.sin_family = AF_INET;

		if ((he = GetHostByName(host)) == NULL) {
			Warning("illegal hostname \"%s\"", host);
			Free((void**)&c);
			return (NULL);
		}
		bcopy((char*)he->h_addr, (char*)&(c->to.in.sin_addr),
		      he->h_length);

		c->to.in.sin_port = htons(port);

		c->addrlen = sizeof (struct sockaddr_in);
		prottype = (flags & CONN_STREAM) ? SOCK_STREAM : SOCK_DGRAM;
	}

	if ((c->sock = socket(c->addrfam, prottype, 0))  <  0) {
		perror("socket()");
		Free((void**)&c);
		Error(-1, "can't open socket");
	}

	c->port = port;
	if (flags & CONN_STREAM) {
		int connret;
		int connerr;

		if (sec >= 0)
		    fSetFlags(c->sock, O_NONBLOCK);
		connret = connect(c->sock, (void*)&(c->to), c->addrlen);
		connerr = errno;
		if (sec >= 0)
		    fClearFlags(c->sock, O_NONBLOCK);
		if (connret < 0) {
			int noconn = 1;

			while (connerr == EINPROGRESS) {
				connect(c->sock, (void*)&(c->to), c->addrlen);
				connerr = errno;
			}
			if (connerr == EALREADY || connerr == EISCONN) {
				if (Test_conn(c->sock, CONN_ISWRITE, sec))
				    noconn = 0;
			}
			else {
				if (connerr != ECONNREFUSED) {
					errno = connerr;
					perror("connect()");
				}
			}
			if (noconn) {
				close(c->sock);
				Free((void**)&c);
				return (NULL);
			}
		}
	}
	else {
		bzero((char*)&(c->from), sizeof (c->from));
		c->from.in.sin_family = AF_INET;
		c->from.in.sin_addr.s_addr = htonl(INADDR_ANY);
		c->from.in.sin_port = htons(0);
		if (bind(c->sock, (void*)&(c->from), c->addrlen)  <  0) {
			perror("bind()");
			close(c->sock);
			Free((void**)&c);
			return (NULL);
		}
	}

	return (c);
}


/*----------------------------------------------------------------------------*/
int
    SendMesg_conn(conn c,
		  void* mesg,
		  size_t len)
{
	int ret;

	if (c == NULL) {
		if (sendbuf == NULL)
		    sendbuf = TrueReAlloc(NULL, IPC_BUFLEN, sizeof (char));
		if (sendbufpos + sizeof (u_short) + len  <=  IPC_BUFLEN) {
			u_short l = htons((u_short)len);

			memcpy(sendbuf+sendbufpos, &l, sizeof (u_short));
			sendbufpos += sizeof (u_short);
			memcpy(sendbuf+sendbufpos, mesg, len);
			sendbufpos += len;
			return (0);
		}
		else {
			Warning("send-buffer overflow");
			return (-1);
		}
	}
	if (c->flags & CONN_INETD)
	    c->sock = 1;
	if (c->flags & CONN_STREAM) {
		u_short l;

		l = htons((u_short)len);
		if ((ret = write(c->sock, (char *)&l,
				 sizeof (u_short)))  <  0) {
			perror("write()");
			return (-1);
		}
		if (ret != sizeof (u_short)) {
			Warning("SendMesg_conn(): message-length incomplete");
			return (-1);
		}
		if ((ret = write(c->sock, (char *)mesg, (int)len))  <  0) {
			perror("write()");
			return (-1);
		}
	}
	else {
		if (len > IPC_BUFLEN) {
			Warning("SendMesg_conn(): receive-buffer isn't big enough");
			return (-1);
		}
		if ((ret = sendto(c->sock, (char *)mesg, (int)len, 0,
				  (void *)&(c->to), c->addrlen))  <  0) {
			perror("sendto()");
			return (-1);
		}
	}
	if (ret != len) {
		Warning("SendMesg_conn(): message incomplete");
		return (-1);
	}
	return (0);
}


/*----------------------------------------------------------------------------*/
int
    SendBuffer_conn(conn c)
{
	if (sendbuf != NULL && sendbufpos > 0) {
		int ret;

		ret = SendMesg_conn(c, sendbuf, sendbufpos);
		sendbufpos = 0;
		return (ret);
	}
	else
	    return (-1);
}


/*----------------------------------------------------------------------------*/
void*
    RecvMesg_conn(conn c,
		  int* lenptr)
{
	static char* mesg = NULL;
	int len, received;
	int ret;

	StrFree(&mesg);
	if (c == NULL) {
		if (recvbuf != NULL &&
		    recvbufpos + sizeof (u_short)  <=  recvbuflen) {
			u_short l;

			memcpy(&l, recvbuf+recvbufpos, sizeof (u_short));
			recvbufpos += sizeof (u_short);
			len = (int)ntohs(l);
			if (recvbufpos + len  <=  recvbuflen) {
				mesg = (char*)TrueReAlloc(NULL, len,
							  sizeof (char));
				memcpy(mesg, recvbuf+recvbufpos, len);
				recvbufpos += len;
			}
			else {
				Warning("message in receive-buffer incomplete");
				len = -1;
			}
		}
		else {
			Warning("message not in receive-buffer");
			len = -1;
		}
	}
	else {
		if (c->flags & CONN_INETD)
		    c->sock = 0;
		if (c->flags & CONN_STREAM) {
			u_short l;

			for (received = 0; received < sizeof (u_short);
			     received += ret)
			    if ((ret = read(c->sock, (char*)(&l)+received,
					    sizeof (u_short) - received)) < 0) {
				    perror("read()");
				    if (lenptr != NULL)
					*lenptr = -1;
				    return (NULL);
			    }
			len = (int)ntohs(l);
			mesg = (char*)TrueReAlloc(NULL, len, sizeof (char));
			for (received = 0; received < len; received += ret)
			    if ((ret = read(c->sock, (char*)(mesg + received),
					    len - received)) < 0) {
				    perror("read()");
				    if (lenptr != NULL)
					*lenptr = -1;
				    Free((void**)&mesg);
				    return (NULL);
			    }
		}
		else {
			if (recvbuf == NULL)
			    recvbuf = TrueReAlloc(NULL, IPC_BUFLEN,
						  sizeof (char));

			if ((len = recvfrom(c->sock, recvbuf, IPC_BUFLEN, 0,
					    (void*)&(c->to), &(c->addrlen)))
			    < 0) {
				perror("recvfrom()");
				if (lenptr != NULL)
				    *lenptr = -1;
				return (NULL);
			}
			mesg = (char*)TrueReAlloc(NULL, len, sizeof (char));
			bcopy(recvbuf, mesg, len);
		}
	}
	if (lenptr != NULL)
	    *lenptr = len;
	return (mesg);
}


/*----------------------------------------------------------------------------*/
int
    RecvBuffer_conn(conn c)
{
	void* msg;

	if (recvbuf == NULL)
	    recvbuf = TrueReAlloc(NULL, IPC_BUFLEN, sizeof (char));
	msg = RecvMesg_conn(c, &recvbuflen);
	if (recvbuflen == -1)
	    return (-1);
	if (msg != NULL)
	    memcpy(recvbuf, msg, recvbuflen);
	else
	    recvbuflen = 0;
	recvbufpos = 0;
	return (0);
}


/*----------------------------------------------------------------------------*/
int
    SendNumber_conn(conn c,
		    void* number,
		    size_t len)
{
	switch (len) {
	    case sizeof (u_short):
		{
			u_short s;

			memcpy(&s, number, len);
			s = htons(s);
			return (SendMesg_conn(c, (void *)&s, len));
		}
		break;
	    case sizeof (u_long):
		{
			u_long l;

			memcpy(&l, number, len);
			l = htonl(l);
			return (SendMesg_conn(c, (void *)&l, len));
		}
		break;
	    default:
		return (SendMesg_conn(c, number, len));
	}
}


/*----------------------------------------------------------------------------*/
int
    RecvNumber_conn(conn c,
		    void* number,
		    size_t len)
{
	int l;
	void* n;

	if ((n = RecvMesg_conn(c, &l))  ==  NULL) {
		return (-1);
	}
	if (l != len) {
		Warning("RecvNumber_conn(): message-length doesn't match");
		return (-1);
	}
	switch (len) {
	    case sizeof (u_short):
		{
			u_short s;

			memcpy(&s, n, len);
			s = ntohs(s);
			memcpy(number, &s, len);
		}
		break;
	    case sizeof (u_long):
		{
			u_long l;

			memcpy(&l, n, len);
			l = ntohl(l);
			memcpy(number, &l, len);
		}
		break;
	    default:
		memcpy(number, n, len);
	}
	return (0);
}


/*----------------------------------------------------------------------------*/
void
    SendTrueINT_conn(conn c,
		     int i)
{
	if (SendNumber_conn(c, &i, sizeof (int))  <  0)
	    Error(-1, "can't send integer");
}


/*----------------------------------------------------------------------------*/
int
    SendINT_conn(conn c,
		 int i)
{
	return ( SendNumber_conn(c, &i, sizeof (int)) );
}


/*----------------------------------------------------------------------------*/
int
    RecvTrueINT_conn(conn c)
{
	int i;

	if (RecvNumber_conn(c, &i, sizeof (int))  <  0)
	    Error(-1, "can't receive integer");
	return (i);
}


/*----------------------------------------------------------------------------*/
int
    RecvINT_conn(conn c)
{
	int i;

	if (RecvNumber_conn(c, &i, sizeof (int))  <  0) {
		Warning("can't receive integer");
		return (-1);
	}
	return (i);
}


/*----------------------------------------------------------------------------*/
void
    SendTrueString_conn(conn c,
			char* str)
{
	if (SendMesg_conn(c, str, (strlen(str)+1) * sizeof (char))  <  0)
		Error(-1, "can't send string");
}


/*----------------------------------------------------------------------------*/
int
    SendString_conn(conn c,
		    char* str)
{
	return ( SendMesg_conn(c, str, (strlen(str)+1) * sizeof (char)) );
}


/*----------------------------------------------------------------------------*/
char*
    RecvTrueString_conn(conn c)
{
	static string str = NULL;
	void* s;

	if ((s = RecvMesg_conn(c, NULL))  ==  NULL)
	    Error(-1, "can't receive string");
	StrTrueReAlloc(&str, (char *)s);
	return ((char *)str);
}


/*----------------------------------------------------------------------------*/
char*
    RecvString_conn(conn c)
{
	static string str = NULL;
	void* s;

	if ((s = RecvMesg_conn(c, NULL))  !=  NULL)
	    StrTrueReAlloc(&str, (char*)s);
	else
	    StrFree(&str);
	return ((char*)str);
}


/*----------------------------------------------------------------------------*/
void
    Close_conn(conn* cptr)
{
	if (*cptr != NULL) {
		close((*cptr)->sock);
		if ((*cptr)->flags & CONN_ISLISTENER  &&
		    (*cptr)->addrfam == AF_UNIX)
		    unlink((*cptr)->from.un.sun_path);
		Free((void**)cptr);
	}
}


/*----------------------------------------------------------------------------*/
static void
    TestFlags_conn(unsigned flags)
{
	if (flags & CONN_STREAM  &&  flags & CONN_DATAGRAM)
	    Error(-1, "illegal connection flags: CONN_STREAM and CONN_DATAGRAM");
	if (!(flags & CONN_STREAM) && !(flags & CONN_DATAGRAM))
	    Error(-1, "illegal connection flags: CONN_STREAM or CONN_DATAGRAM needed");
	if (flags & CONN_ISLISTENER)
	    Error(-1, "illegal connection flags: CONN_ISLISTENER is intern");
	if (flags & CONN_ISREAD)
	    Error(-1, "illegal connection flags: CONN_ISREAD is intern");
	if (flags & CONN_ISWRITE)
	    Error(-1, "illegal connection flags: CONN_ISWRITE is intern");
}


/*----------------------------------------------------------------------------*/
int
    Test_conn(int sock,
	      int mode,
	      long sec)
{
	fd_set readfs;
	fd_set* rfs = NULL;
	fd_set writefs;
	fd_set* wfs = NULL;
	struct timeval tv, *tvptr = NULL;
	int ret;

	if (mode & CONN_ISREAD) {
		FD_ZERO(&readfs);
		FD_SET(sock, &readfs);
		rfs = &readfs;
	}
	if (mode & CONN_ISWRITE) {
		FD_ZERO(&writefs);
		FD_SET(sock, &writefs);
		wfs = &writefs;
	}
	if (rfs == NULL && wfs == NULL)
	    Error(-1, "Test_conn(): illegal test-mode");

	if (sec != -1) {
		tv.tv_sec = sec;
		tv.tv_usec = 0L;
		tvptr = &tv;
	}
	if (
#ifdef hpux
	    (ret = select(sock + 1, (int*)rfs, (int*)wfs, NULL, tvptr))  <  0
#else
	    (ret = select(sock + 1, rfs, wfs, NULL, tvptr))  <  0
#endif
	    ) {
		perror("select()");
		Error(-1, "select() failed");
	}
	return (ret);
}
