/* xtea - distribute beverages and other resources over the network
 *
 * Copyright (c) 1994 Henning Spruth (spruth@regent.e-technik.tu-muenchen.de)
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or 
 * implied warranty.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>

#include "config.h"
#include "message.h"
#include "sockio.h"


/* The set of message types */
char *msgtype[]=
{ "STARTUP", "ALIVE", "ACK", "OFFER", "USERLIST", "INFORM", "ORDER",
  "CONFIRM","INFORMSERVER","ORDERSERVER","CONFIRMSERVER","END","GOODBYE",
  "QUERYSTATUS", "STATUSDATA"
};



/* ********************************* parse_msgtype() ************************
 *
 * Parse a message type string.
 *
*/
int parse_msgtype(char *s)
{
  int i;
  for(i=0; i<MSGTYPES;i++)
    if(strcmp(s,msgtype[i])==0) return i;
  return -1;
}


/* **************************** init_listen() ****************************
 *
 * Create a socket and prepare to accept connections on it.
 *
*/
int init_listen(int serviceno, int *mastersocket)
{
  struct sockaddr_in srvr;
  int sock;
  
  sock=socket(AF_INET,SOCK_STREAM,0);
  if (sock<0)
    { perror("socket");
      return 1;
    }
  srvr.sin_family=AF_INET;
  srvr.sin_addr.s_addr=INADDR_ANY;

  srvr.sin_port=serviceno;

  if (bind(sock,(struct sockaddr *) &srvr,sizeof(srvr))<0)
  { perror("bind");
    return 1;
  }

  if(listen(sock,4)<0)
  {
    perror("listen");
    return 1;
  }
  *mastersocket=sock;
  return 0;
}
  


/* ********************************** start_message() ************************
 *
 * Start sending a message of given type from the server to a client.
 *
*/
int start_message(char *host, short port, int messagetype, int *newsock)
{
  int sock;
  struct sockaddr_in server;
  struct hostent *hp, *gethostbyname();

  sock=socket(AF_INET, SOCK_STREAM, 0);
  if (sock<0)
    { perror("tready: socket");
      return 1;
    }

  server.sin_family=AF_INET;
  hp=gethostbyname(host);
  if (hp==NULL)
  { perror("gethostbyname");
    close(sock);
    return 1;
  }

  memcpy(&server.sin_addr, hp->h_addr, hp->h_length);

  server.sin_port=port;

  if(connect(sock, (struct sockaddr *) &server, sizeof(server))<0)
    { perror("connect");
      close(sock);
      return 1;
    }

  put_string(sock,msgtype[messagetype]);
  *newsock = sock;
  return 0;
}


/* ******************************** start_servermessage() *******************
 *
 * Start sending a message of given type from a client to the server.
 *
*/
int start_servermessage(int messagetype, int *newsock)
{
  struct servent *se;
  int port;

#ifndef PORTNUMBER
  se=getservbyname(XTEASERVICE,"tcp");
  if(se==NULL)
  { perror("getservbyname");
    return 1;
  }
  port=se->s_port;
#else
  port=PORTNUMBER;
#endif
  return start_message(XTEASERVER, port, messagetype, newsock);
}


/* ********************************** end_of_message() **********************
 *
 * Wait for the last line indicating a message's end.
 *
*/
int end_of_message(int sock)
{
  char *s;
  int status=0;
  if(get_string(sock,&s))
  {
    fprintf(stderr,"Can't receive END line!\n");
    status=1;
  }
  else
  {
    if(strcmp(s,"END")!=0)
    {
      fprintf(stderr,"Bad message end - expected END, got %s\n",s);
      status=1;
    }
    free(s);
  }
  close(sock);
  return status;
}


