/* xtea - distribute beverages and other resources over the network
 *
 * Copyright (c) 1994 Henning Spruth (spruth@regent.e-technik.tu-muenchen.de)
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or 
 * implied warranty.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Command.h>

#include "alloc.h"
#include "inform.h"
#include "sockio.h"
#include "message.h"
#include "client.h"
#include "xresources.h"

extern Widget toplevel;

extern char *xtea_resources[];
extern char my_uname[];
extern appresStruct appres;
extern int verbose;

extern Pixmap c_pixmap[];

extern XtAppContext app_context;


/*
 * This structure holds data for an 'offer' popup for some commodity.
*/
typedef struct AvailableResource
{
  char *uname;
  char *fullname;
  int resource;
  int amount;
  Widget popup,label;
  struct AvailableResource *prev;
  struct AvailableResource *next;
} AvailableResource;


/*
 * This structure contains data on a 'confirm' popup.
*/
typedef struct ConfirmInfo
{
  Widget popup;
  XtIntervalId timer;
} ConfirmInfo;

static AvailableResource *firstavailable, *lastavailable;

/* ********************************* dialog_popdown() ************************
 *
 * Remove the AvailableResource popup for some reason.
 *
*/
static void dialog_popdown(Widget w, AvailableResource *a, void *dummy2)
{
  XtPopdown(a->popup);
  XtDestroyWidget(a->popup);
  free(a->uname);
  free(a->fullname);
  if(firstavailable==a) firstavailable=a->next;
  if(lastavailable==a) lastavailable=a->prev;
  if(a->prev) (a->prev)->next = a->next;
  if(a->next) (a->next)->prev = a->prev;
  free(a);
}


/* ***************************** dialog_accept() **************************
 *
 * We want to take the offered commodity, so send an ORDERSERVER message to
 * xtead.
 *
*/
static void dialog_accept(Widget w, AvailableResource *a, void *dummy2)
{
  int sock;
  char buf[10];

  start_servermessage(ORDERSERVER,&sock);
  sprintf(buf,"%d",a->resource);
  put_string(sock,buf);
  put_string(sock,a->uname);
  put_string(sock,my_uname);
  put_string(sock,"1");
  put_string(sock,"END");
  close(sock);
/*  dialog_popdown(w,a,NULL);*/
}


/* *********************************** dialog_reject() ***********************
 *
 * Tell xtead that we don't want the offered resource.
 *
*/
static void dialog_reject(Widget w, AvailableResource *a, void *dummy2)
{
  int sock;
  char buf[10];

  start_servermessage(ORDERSERVER,&sock);
  sprintf(buf,"%d",a->resource);
  put_string(sock,buf);
  put_string(sock,a->uname);
  put_string(sock,my_uname);
  put_string(sock,"0");
  put_string(sock,"END");
  close(sock);
  dialog_popdown(w,a,NULL);
}


/* ******************************* SearchAvailable() ************************
 *
 * Check if we already have a popup for commodity 'resource' from user 'uname'.
 *
*/
static AvailableResource *SearchAvailable(char *uname, int resource)
{
  AvailableResource *a;
  for(a=firstavailable; a!=NULL; a=a->next)
    if(a->resource==resource && strcmp(a->uname,uname)==0) break;

  return(a);
}



/* **************************** NewAvailable() ****************************
 *
 * A new popup is about to be created, so allocate an AvailableResource
 * buffer.
 *
*/
static AvailableResource *NewAvailable(char *uname, char *fullname, 
					int resource, int amount)
{
  AvailableResource *a;
  a=myalloc(sizeof(AvailableResource));
  a->uname=uname;
  a->fullname=fullname;
  a->resource=resource;
  a->amount=amount;
  a->next=NULL;
  a->prev=lastavailable;
  if(lastavailable) lastavailable->next=a;
  lastavailable=a;
  if(firstavailable==NULL) firstavailable=a;
  return a;
}



/* *************************** process_inform() ***************************
 *
 * We are being sent a new offer or an update to an existing offer for some
 * commodity.
 *
*/
void process_inform(int sock)
{
  char *fullname, *uname,*temp, *info;
  int resource,amount,status;
  Widget form,label,command,icon,line;
  Arg arg[20];
  int n;
  char *buf;
  AvailableResource *a;

  if(get_string(sock,&fullname)) return;
  if(get_string(sock,&uname))
  {
    free(fullname);
    return;
  }
  if(get_string(sock,&temp)) return;
  status=0; if(sscanf(temp,"%d",&resource)!=1) status=1;
  free(temp);
  if(status) 
  {
    free(uname);
    free(fullname);
    return;
  }

  if(get_string(sock,&temp)) return;
  status=0; if(sscanf(temp,"%d",&amount)!=1) status=1;
  free(temp);
  if(status) 
  {
    free(uname);
    free(fullname);
    return;
  }

  if(get_string(sock,&info)) 
  {
    free(uname);
    free(fullname);
    return;
  }

  a=SearchAvailable(uname, resource);

  /* if the remaining amount is 0, remove the popup */
  if(amount==0)
  {
    if(a) 
    {
      /* remove existing query popup */
      dialog_popdown(NULL,a,NULL);
    }
    free(uname);
    free(fullname);
    free(info);	/* free message buffer */
    return;
  }

  /* allocate a buffer that is large enough to hold the info text */	
  n=strlen(appres.infolabel)+strlen(info)+3;
  if(n<200) n=200;
  buf=myalloc(n);

  if(a)
  {
    /* this is an update of the available amount for an existing popup */
    if(verbose) printf("Amount for resource %d set to %d\n", resource,amount);
    a->amount = amount;
    n=0;
    sprintf(buf,"%d",amount);
    XtSetArg(arg[n],XtNlabel, buf); n++;
    XtSetValues(a->label,arg,n);
    free(info);
    free(uname);
    free(fullname);
  }
  else
  {
    /* this is a new offer, create a new popup */

    /* Note: info, uname, and fullname are stored in NewAvailable()
     *       so they must not be freed!
     */
    a=NewAvailable(uname, fullname, resource, amount);

    n=0;
    a->popup= XtCreatePopupShell("informpopup",transientShellWidgetClass,
			    toplevel,arg,n);
    
    n=0;
    form=XtCreateManagedWidget("form",formWidgetClass,a->popup,arg,n);

    n=0;
    XtSetArg(arg[n], XtNbitmap, c_pixmap[resource]); n++;
    XtSetArg(arg[n], XtNleft, XawChainLeft); n++;
    XtSetArg(arg[n], XtNright, XawChainRight); n++;
    XtSetArg(arg[n], XtNtop, XawChainTop); n++;
    icon = XtCreateManagedWidget("informpixmap",labelWidgetClass,form,arg,n);
    
    n=0;
    sprintf(buf,"%d",amount);
    XtSetArg(arg[n],XtNlabel, buf); n++;
    XtSetArg(arg[n], XtNtop, XawChainTop); n++;
    XtSetArg(arg[n], XtNfromHoriz, icon); n++;
    label=XtCreateManagedWidget("amount",labelWidgetClass,
				form,arg,n);
    a->label=label;

    n=0;
    sprintf(buf,appres.gotoffer,appres.resourcename[resource],fullname);
    XtSetArg(arg[n],XtNlabel, buf); n++;
    XtSetArg(arg[n], XtNfromVert, label); n++;
    XtSetArg(arg[n], XtNright, XawChainRight); n++;
    XtSetArg(arg[n], XtNfromHoriz, icon); n++;
    label=XtCreateManagedWidget("text",labelWidgetClass,
				form,arg,n);

    n=0;
    sprintf(buf,"%s %s",appres.infolabel,info);
    XtSetArg(arg[n], XtNlabel, buf); n++;
    XtSetArg(arg[n], XtNleft, XawChainLeft); n++;
    XtSetArg(arg[n], XtNfromVert, label); n++;
    XtSetArg(arg[n], XtNright, XawChainRight); n++;
    label=XtCreateManagedWidget("info",labelWidgetClass,
				form,arg,n);

    line = label;

    n=0;
    XtSetArg(arg[n], XtNleft, XawChainLeft); n++;
    XtSetArg(arg[n], XtNfromVert, line); n++;
    command=XtCreateManagedWidget("order",commandWidgetClass,
				  form,arg,n);
    XtAddCallback(command,XtNcallback, (XtCallbackProc) dialog_accept,a);
    
    n=0;
    XtSetArg(arg[n], XtNright, XawChainRight); n++;
    XtSetArg(arg[n], XtNfromVert, line); n++;
    XtSetArg(arg[n], XtNfromHoriz, command); n++;
    command=XtCreateManagedWidget("cancel",commandWidgetClass,
				  form,arg,n);
    XtAddCallback(command,XtNcallback, (XtCallbackProc) dialog_reject,a);
    
    XtPopup(a->popup,XtGrabNone);

    /* sound the bell if not disabled */
    if(appres.beep)
    {
      XBell(XtDisplay(form),100);
      XBell(XtDisplay(form),100);
    }

    /* create a user-specified event, e.g. playing an audio clip using aplay */
    announce_event(appres.event_arg[resource]?
    	appres.event_arg[resource] : appres.default_event);

    free(info); /* free message buffer */

    if(verbose) printf("Offer of %d units of %s from %s/%s\n",
	   amount,xtea_resources[resource],fullname,uname);
  }

  free(buf);

}




/* ***************************** remove_confirm() ****************************
 *
 * Remove the confirmation popup.
 *
*/
static void remove_confirm(ConfirmInfo *c, void *dummy2)
{
  XtPopdown(c->popup);
  XtDestroyWidget(c->popup);
  free(c);
}

/* *************************** remove_confirm_user() ***********************
 *
 * The user wants to remove the confirm popup, so disarm the timer first.
 *
*/
static void remove_confirm_user(Widget w, ConfirmInfo *c, void *dummy2)
{
  XtRemoveTimeOut(c->timer);
  remove_confirm(c,dummy2);
}


/* ******************************** popup_confirm() ***********************
 *
 * Create a confirmation popup, realize it, and initialize its countdown timer.
 *
*/
void popup_confirm(int sock)
{
  char *fullname,*temp;
  int resource,result;
  Widget dialog;
  int n;
  Arg arg[10];
  char buf[80];
  int status;
  ConfirmInfo *c;

  if(get_string(sock,&temp)) return;
  status=0; if(sscanf(temp,"%d",&resource)!=1) status=1;
  free(temp);
  
  if(get_string(sock,&temp)) return;
  status=0; if(sscanf(temp,"%d",&result)!=1) status=1;
  free(temp);
  
  if(get_string(sock,&fullname)) return;

  if(result<0 || result>2) result=2;
  sprintf(buf,appres.confirmmsg[status], fullname,
	  appres.resourcename[resource]);
  free(fullname);

  c=(ConfirmInfo *) myalloc(sizeof(ConfirmInfo));

  n=0;
  c->popup= XtCreatePopupShell("confirmpopup",transientShellWidgetClass,
			    toplevel,arg,n);

  n=0;
  XtSetArg(arg[n], XtNlabel, buf); n++;
  dialog = XtCreateManagedWidget("dialog", dialogWidgetClass,
				 c->popup,arg,n);
  
  XawDialogAddButton(dialog,"OK", (XtCallbackProc) remove_confirm_user, c);
  XtPopup(c->popup, XtGrabNone);
  
  c->timer = XtAppAddTimeOut(app_context, CONFIRM_LIFETIME , 
			     (XtTimerCallbackProc) remove_confirm, c);
}



/* ****************************** init_inform() **************************
 *
 * Initialize local data.
 *
*/
void init_inform()
{
  lastavailable=firstavailable=NULL;
}
