/*========================================================================*\

Copyright (c) 1990-1999  Paul Vojta

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
PAUL VOJTA BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

NOTE:
	xdvi is based on prior work, as noted in the modification history
	in xdvi.c.

\*========================================================================*/

#include "filf-app.h"	/* application-related defs, etc. */
#include "filefind.h"

#include <errno.h>

#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifndef WEXITSTATUS
# define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
#endif
#ifndef WIFEXITED
# define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
#endif

#if HAVE_VFORK_H
#include <vfork.h>
#endif

#ifdef	X_NOT_STDC_ENV
extern	int	errno;
extern	char	*getenv ARGS((_Xconst char *));
#endif

#ifndef	EXIT_SUCCESS
#ifndef	VMS
#define	EXIT_SUCCESS	0
#else
#define	EXIT_SUCCESS	1
#endif
#endif

/*
 *	If you think you have to change DEFAULT_TAIL, then you haven't read the
 *	documentation closely enough.
 */

#ifndef	VMS
#define	PATH_SEP	':'
#define	DEFAULT_TAIL	"/%f.%d%p"
#define	DEFAULT_VF_TAIL	"/%f.vf"
#else	/* VMS */
#define	PATH_SEP	'/'
#define	DEFAULT_TAIL	":%f.%d%p"
#define	DEFAULT_VF_TAIL	":%f.vf"
#endif	/* VMS */

#ifndef DEFAULT_FONT_SIZES
	/* default sizes for current dpi */
#define	DEFAULT_FONT_SIZES	"m0:m0.5:m1:m2:m3:m4:m5"
#endif

#ifdef MKTEXPK
static	_Xconst	char	*makepkcmd		= NULL;
#endif

/*
 *	Information on how to search for pk and gf files.
 */

static	_Xconst	char	no_f_str_pkgf[]	= DEFAULT_TAIL;

static	struct findrec			search_pkgf	= {
	/* path1	*/	NULL,
#if	CFGFILE
	/* envptr	*/	NULL,
#endif
	/* path2	*/	DEFAULT_FONT_PATH,
	/* type		*/	"font",
	/* fF_etc	*/	"fFdbpm",
	/* x_var_char	*/	'd',
	/* n_var_opts	*/	3,
	/* no_f_str	*/	no_f_str_pkgf,
	/* no_f_str_end	*/	no_f_str_pkgf + sizeof(no_f_str_pkgf) - 1,
	/* abs_str	*/	"%f.%d%p",
#ifdef	USE_GF
	/* no_f_str_flags */	F_FILE_USED | F_PK_USED,
	/* abs_str_flags */	F_FILE_USED | F_PK_USED,
	/* pk_opt_char	*/	'p',
	/* pk_gf_addr	*/	&fF_values[4],
#endif
	/* pct_s_str	*/	"%qfonts/%p/%m//:%qfonts/%p/modeless//",
	{
	  /* v.stephead		*/	NULL,
	  /* v.pct_s_head	*/	NULL,
	  /* v.pct_s_count	*/	0,
	  /* v.pct_s_atom	*/	NULL,
	  /* v.rootp		*/	NULL,
	}
};

#ifdef	DOSNAMES

static	_Xconst	char	no_f_str_dos[]	= "/dpi%d/%f.%p";

static	struct findrec			search_pkgf_dos	= {
	/* path1	*/	NULL,
#if	CFGFILE
	/* envptr	*/	NULL,
#endif
	/* path2	*/	DEFAULT_FONT_PATH,
	/* type		*/	"font",
	/* fF_etc	*/	"fFdbpm",
	/* x_var_char	*/	'd',
	/* n_var_opts	*/	3,
	/* no_f_str	*/	no_f_str_dos,
	/* no_f_str_end	*/	no_f_str_dos + sizeof(no_f_str_dos) - 1,
	/* abs_str	*/	"%f.%d%p",
#ifdef	USE_GF
	/* no_f_str_flags */	F_FILE_USED | F_PK_USED,
	/* abs_str_flags */	F_FILE_USED | F_PK_USED,
	/* pk_opt_char	*/	'p',
	/* pk_gf_addr	*/	&fF_values[4],
#endif
	/* pct_s_str	*/	"%qfonts/%p/%m//:%qfonts/%p/modeless//",
	{
	  /* v.stephead		*/	NULL,
	  /* v.pct_s_head	*/	NULL,
	  /* v.pct_s_count	*/	0,
	  /* v.pct_s_atom	*/	NULL,
	  /* v.rootp		*/	NULL,
	}
};
#endif	/* DOSNAMES */

/*
 *	Information on how to search for vf files.
 */

static	_Xconst	char	no_f_str_vf[]	= DEFAULT_VF_TAIL;

static	struct findrec			search_vf	= {
	/* path1	*/	NULL,
#if	CFGFILE
	/* envptr	*/	NULL,
#endif
	/* path2	*/	DEFAULT_VF_PATH,
	/* type		*/	"vf",
	/* fF_etc	*/	"fF",
	/* x_var_char	*/	'f',		/* i.e., none */
	/* n_var_opts	*/	2,
	/* no_f_str	*/	no_f_str_vf,
	/* no_f_str_end	*/	no_f_str_vf + sizeof(no_f_str_vf) - 1,
	/* abs_str	*/	"%f.vf",
#ifdef	USE_GF
	/* no_f_str_flags */	F_FILE_USED,
	/* abs_str_flags */	F_FILE_USED,
	/* pk_opt_char	*/	'f',		/* none */
	/* pk_gf_addr	*/	NULL,
#endif
	/* pct_s_str	*/	"%qfonts/vf//",
	{
	  /* v.stephead		*/	NULL,
	  /* v.pct_s_head	*/	NULL,
	  /* v.pct_s_count	*/	0,
	  /* v.pct_s_atom	*/	NULL,
	  /* v.rootp		*/	NULL,
	}
};

static	int	*sizes, *sizend;
static	char	default_size_list[]	= DEFAULT_FONT_SIZES;

static	char		bdpi_string[10];
static	char		dpi_string[10];

static	double		magsteps[10]	= {1.0, 1.2, 1.44, 1.728, 2.0736,
					   2.48832, 2.985984, 3.5831808,
					   4.29981696, 5.159780352};

static	int
atosize(p)
	_Xconst	char	*p;
{
	_Xconst	char	*q;
	Boolean		minus;
	double		factor;

	if (*p != 'm')
	    return atoi(p);

	for (q = "agstep";; ++q) {
	    ++p;
	    if (*q == '\0' || *p != *q) break;
	}

	minus = False;
	if (*p == '-') {
	    minus = True;
	    ++p;
	}

	if (*p < '0' || *p > '9')
	    return 0;
	factor = magsteps[*p - '0'];
	++p;

	if (*p == '.' && p[1] == '5')
	    factor *= 1.0954451150103321;	/* sqrt(1.2) */

	return (minus ? pixels_per_inch / factor : pixels_per_inch * factor)
	  + 0.5;
}

static	void
get_sizes(size_list, spp)
	char	*size_list;
	int	**spp;
{
	if (*size_list == PATH_SEP) ++size_list;
	for (;;) {
	    *(*spp)++ = atosize(size_list);
	    size_list = index(size_list, PATH_SEP);
	    if (size_list == NULL) return;
	    ++size_list;
	}
}

void
init_font_open()
{
	char	*size_list;
	int	*sp, *sp1;
	unsigned int n;
	char	*p;

	Sprintf(bdpi_string, "%d", pixels_per_inch);

#if	CFGFILE

#ifndef	XDVIFONTS_ONLY
	if ((search_pkgf.path1 = getenv("XDVIFONTS")) == NULL
	  && (search_pkgf.path1 = getenv("PKFONTS")) == NULL
	  && (search_pkgf.path1 = getenv("TEXPKS")) == NULL)
	    search_pkgf.path1 = getenv("TEXFONTS");
#else
	search_pkgf.path1 = getenv("XDVIFONTS");
#endif
#ifdef	DOSNAMES
	search_pkgf_dos.path1 = search_pkgf.path1;
#endif
	search_pkgf.envptr = ffgetenv("PKFONTS");
	/* clear it if it's a getenv() placeholder */
	if (search_pkgf.envptr != NULL && search_pkgf.envptr->value == NULL)
            search_pkgf.envptr = NULL;
#ifdef	DOSNAMES
	search_pkgf_dos.envptr = search_pkgf.envptr;
#endif

#else	/* not CFGFILE */

	if ((search_pkgf.path1 = getenv("XDVIFONTS")) == NULL
#ifndef	XDVIFONTS_ONLY
		&& (search_pkgf.path1 = getenv("PKFONTS")) == NULL
		&& (search_pkgf.path1 = getenv("TEXPKS")) == NULL
		&& (search_pkgf.path1 = getenv("TEXFONTS")) == NULL
#endif
		) {
	    search_pkgf.path1 = search_pkgf.path2;
	    search_pkgf.path2 = NULL;
	}
#ifdef	DOSNAMES
	search_pkgf_dos.path1 = search_pkgf.path1;
	search_pkgf_dos.path2 = search_pkgf.path2;
#endif

#endif	/* not CFGFILE */

	/*
	 * pk/gf searching is the only kind that uses more than three
	 * characters in fF_etc, so these can be initialized once and then
	 * forgotten.
	 */

	fF_values[2] = dpi_string;
	fF_values[3] = bdpi_string;
#ifndef	USE_GF
	fF_values[4] = "pk";
#endif
	fF_values[5] = resource.mfmode;

#if	CFGFILE

#ifndef	XDVIFONTS_ONLY
	if ((search_vf.path1 = getenv("XDVIVFS")) == NULL)
	    search_vf.path1 = getenv("VFFONTS");
#else
	search_vf.path1 = getenv("XDVIVFS");
#endif
	search_vf.envptr = ffgetenv("VFFONTS");
	/* clear it if it's a getenv() placeholder */
	if (search_vf.envptr != NULL && search_vf.envptr->value == NULL)
            search_vf.envptr = NULL;

#else	/* not CFGFILE */

	if ((search_vf.path1 = getenv("XDVIVFS")) == NULL
#ifndef	XDVIFONTS_ONLY
		&& (search_vf.path1 = getenv("VFFONTS")) == NULL
#endif
		) {
	    search_vf.path1 = search_vf.path2;
	    search_vf.path2 = NULL;
	}

#endif	/* not CFGFILE */

	size_list = getenv("XDVISIZES");
	n = 1;	/* count number of sizes */
	if (size_list == NULL || *size_list == '\0' || *size_list == PATH_SEP)
	    for (p = default_size_list; (p = index(p, PATH_SEP)) != NULL; ++p)
		++n;
	if (size_list != NULL)
	    for (p = size_list; (p = index(p, PATH_SEP)) != NULL; ++p) ++n;
	sizes = xmalloc(n * sizeof(int));
	sizend = sizes;	/* get the actual sizes */
	if (size_list == NULL || *size_list == '\0' || *size_list == PATH_SEP)
	    get_sizes(default_size_list, &sizend);
	if (size_list != NULL && *size_list != '\0')
	    get_sizes(size_list, &sizend);

	/* sort the sizes (insertion sort) */
	for (sp = sizes + 1; sp < sizend; ++sp)
	    if (*sp < sp[-1]) {
		int	i	= *sp;

		sp1 = sp;
		do
		    *sp1 = sp1[-1];
		while (--sp1 > sizes && i < sp1[-1]);
		*sp1 = i;
	    }

	/* eliminate duplicates and erroneous values */
	n = 0;	/* previous value */
	for (sp = sp1 = sizes; sp < sizend; ++sp)
	    if (*sp != n)
		n = *sp1++ = *sp;
	sizend = sp1;
}

/*
 *	Try a given size.
 */

static	FILE *
try_size(font, dpi, ret_path)
	_Xconst char	*font;
	int		dpi;
	_Xconst char	**ret_path;
{
#ifdef	DOSNAMES
	FILE		*retval;
#endif

	Sprintf(dpi_string, "%d", dpi);

#ifdef	DOSNAMES
	retval = filefind(font, &search_pkgf, ret_path);
	if (retval != NULL) return retval;
	return filefind(font, &search_pkgf_dos, ret_path);
#else
	return filefind(font, &search_pkgf, ret_path);
#endif
}


#ifdef MKTEXPK

#ifndef	MKTEXPK_PATH
#ifdef MAKETEXPK
#define	MKTEXPK_PATH	"MakeTeXPK"
#else
#define	MKTEXPK_PATH	"mktexpk"
#endif
#endif

#ifdef MAKETEXPK
#define	MKPK_DEFAULT_MODE	"default"
#else
#define	MKPK_DEFAULT_MODE	"/"
#endif

#define	NOBUILD		29999
#define	MKPKSIZE	256

/*
 *	makefont - call system() to make the font.
 */

static	FILE *
makefont(font, dpi, name, magstepval)
	_Xconst char	*font;
	int		dpi;
	_Xconst char	**name;
	int		magstepval;
{
	int			pipefds[2];
	char			mkpk[MKPKSIZE];
	Boolean			used_fontname = False;
	Boolean			used_mfmode = False;
	int			redirect_to = 1;
	_Xconst	char		*p;
	char			*q;
	static	_Xconst	char	*argv[]	= {"/bin/sh", "-c", NULL, NULL};
	static	Boolean		did_putenv = False;
	pid_t			pid;
	int			status;
	FILE			*f;
	int			pos;

	if (xpipe(pipefds) != 0) {	/* create the pipe */
	    perror("[xdvi] pipe");
	    return NULL;
	}

	/*
	 * Generate the mktexpk command line.
	 */

	if (makepkcmd == NULL) {
	    makepkcmd = getenv("XDVIMAKEPK");
	    if (makepkcmd == NULL) makepkcmd = MKTEXPK_PATH;
	}
	p = makepkcmd;
	q = mkpk;
	for (;;) {
	    if (*p == '%') {
		switch (*++p) {
		case 'n':
		    Strcpy(q, font);
		    used_fontname = True;
		    break;
		case 'd':
		    Sprintf(q, "%d", dpi);
		    break;
		case 'b':
		    Sprintf(q, "%d", pixels_per_inch);
		    break;
		case 'm':
		    if (magstepval == NOMAGSTP)
			Sprintf(q, "%d+%d/%d", dpi / pixels_per_inch,
			    dpi % pixels_per_inch, pixels_per_inch);
		    else if (magstepval >= 0)
			Sprintf(q, "'magstep(%d%s)'", magstepval / 2,
			    magstepval % 2 ? ".5" :"");
		    else
			Sprintf(q, "'magstep(-%d%s)'", -magstepval / 2,
			    magstepval % 2 ? ".5" :"");
		    break;
		case 'o':
		    Strcpy(q, resource.mfmode != NULL ? resource.mfmode
			: MKPK_DEFAULT_MODE);
		    used_mfmode = True;
		    break;
		case 'r':
		    Strcpy(q, "'>&3'");
		    redirect_to = 3;
		    break;
		case '%':
		    *q++ = '%';
		    *q = '\0';
		    break;
		case '\0':
		    --p;
		    *q = '\0';
		    break;
		default:
		    *q++ = '%';
		    *q++ = *p;
		    *q = '\0';
		    break;
		}
		q += strlen(q);
	    }
	    else if (*p == '\0')
		if (used_fontname) break;
		else {
#ifndef MAKETEXPK
		    p = " --mfmode %o --bdpi %b --mag %m --dpi %d %n %r";
#else
#ifdef MKPK_REDIRECT
		    p = " %n %d %b %m %o '' %r";
#else
		    p = " %n %d %b %m";
#endif
#endif
		    continue;
		}
	    else *q++ = *p;
	    ++p;
	}
	if (resource.mfmode != NULL && !used_mfmode) {
	    *q++ = ' ';
	    Strcpy(q, resource.mfmode);
	}
	else *q = '\0';

	/*
	 * Put the metafont mode into the environment, if available.
	 */

	if (!did_putenv) {
	    if (resource.mfmode != NULL)
		xputenv("MAKETEX_MODE", resource.mfmode);
	    did_putenv = True;
	}

	/*
	 * Create a child process.
	 */

	argv[2] = mkpk;
	Printf("- %s\n", mkpk);

	Fflush(stderr);		/* avoid double buffering */
	pid = vfork();
	if (pid == 0) {		/* if child */
	    (void) close(pipefds[0]);
	    if (redirect_to != pipefds[1]) {
		(void) dup2(pipefds[1], redirect_to);
		(void) close(pipefds[1]);
	    }
	    (void) execvp(argv[0], (char **) argv);
	    Fputs("Execvp of /bin/sh failed.\n", stderr);
	    Fflush(stderr);
	    _exit(1);
	}

	(void) close(pipefds[1]);

	if (pid == -1) {
	    (void) close(pipefds[0]);
	    perror("[xdvi] fork");
	    return NULL;
	}

	/*
	 * Now wait until the process terminates, reading whatever it writes
	 * to the pipe.  An eof on the pipe assumes that the child terminated.
	 */

	pos = 0;
	for (;;) {		/* read from pipe until EOF */
	    int bytes;

	    if (pos >= ffline_len)
		expandline(ffline_len);

	    bytes = read(pipefds[0], ffline + pos, ffline_len - pos);
	    if (bytes == -1) continue;
	    if (bytes == 0) break;
	    pos += bytes;
	}

	(void) close(pipefds[0]);

	for (;;) {		/* get termination status */
#if HAVE_WAITPID
	    if (waitpid(pid, &status, 0) != -1) break;
#else
# if HAVE_WAIT4
	    if (wait4(pid, &status, 0, (struct rusage *) NULL) != -1) break;
# else
	    int retval;

	    retval = wait(&status);
	    if (retval == pid) break;
	    if (retval != -1) continue;
# endif
#endif
	    if (errno == EINTR) continue;
	    perror("[xdvi] waitpid");
	    return NULL;
	}

	if (!WIFEXITED(status) || WEXITSTATUS(status) != EXIT_SUCCESS) {
	    Fprintf(stderr, "%s failed.\n", makepkcmd);
	    return NULL;
	}

	if (pos != 0 && ffline[pos - 1] == '\n')	/* trim off last \n */
	    --pos;

		/* if no response, then it probably failed, but look anyway */
	if (pos == 0) {
	    if (debug & DBG_OPEN)
		Printf("No response from %s\n", makepkcmd);
	    return try_size(font, dpi, name);
	}

	if (pos >= ffline_len)
	    expandline(pos);
	ffline[pos++] = '\0';

	if (debug & DBG_OPEN)
	    Printf("%s ---> %s\n", makepkcmd, ffline);
	f = xfopen(ffline, OPEN_MODE);
	if (f == NULL) {
	    perror(ffline);
	    return NULL;
	}

	if (debug & DBG_OPEN)
	    puts("--Success--\n");
	*name = xmemdup(ffline, pos);
	return f;
}

#endif	/* MKTEXPK */

/*
 *	Try a given font name
 */

#ifndef MKTEXPK
#define	PRE_FONT_OPEN(font, fdpi, dpi_ret, name, ignore) \
		pre_font_open(font, fdpi, dpi_ret, name)
#else
#define	PRE_FONT_OPEN	pre_font_open
#endif

static	FILE *
PRE_FONT_OPEN(font, fdpi, dpi_ret, name, magstepval)
	_Xconst char	*font;
	double		fdpi;
	int		*dpi_ret;
	_Xconst char	**name;
#ifdef MKTEXPK
	int	magstepval;
#endif
{
	FILE	*f;
	int	*p1, *p2;
	int	dpi	= fdpi + 0.5;
	int	tempdpi;

	/*
	 * Loop over sizes.  Try actual size first, then closest sizes.
	 * If the pathname is absolutely or explicitly relative, don't
	 * use the usual paths to search for it; just look for it in the
	 * directory specified.
	 */

	f = try_size(font, *dpi_ret = dpi, name);
	if (f != NULL)
	    return f;

	/* Try at one away from the size we just tried, to account
	   for rounding error.  */
	tempdpi = dpi + (dpi < fdpi ? 1 : -1);
	f = try_size(font, tempdpi, name);
	if (f != NULL) {
	    *dpi_ret = tempdpi;
	    return f;
	}

	/* Try a virtual font. */
	f = filefind(font, &search_vf, name);
	if (f != NULL)
	    return f;

#ifdef MKTEXPK
	/* Try to create the font. */
	if (magstepval != NOBUILD && resource.makepk) {
	    f = makefont(font, dpi, name, magstepval);
	    if (f != NULL)
		return f;
	}
#endif

	/* Now try at all the sizes. */
	for (p2 = sizes; p2 < sizend; ++p2) if (*p2 >= dpi) break;
	p1 = p2;
	for (;;) {
		/* find another resolution */
	    if (p1 <= sizes)
		if (p2 >= sizend) return NULL;
		else tempdpi = *p2++;
	    else if (p2 >= sizend || (long) dpi * dpi <= (long) p1[-1] * *p2)
		    tempdpi = *--p1;
		else tempdpi = *p2++;
	    f = try_size(font, *dpi_ret = tempdpi, name);
	    if (f != NULL)
		return f;
	}
}

/* ARGSUSED */
FILE *
font_open(font, font_ret, dpi, dpi_ret, magstepval, name)
	_Xconst char	*font;
	char	**font_ret;
	double	dpi;
	int	*dpi_ret;
	int	magstepval;
	char	**name;
{
	FILE	*f;
	int	actual_pt, low_pt, high_pt, trial_pt;
	char	fn[50], *fnend;

	f = PRE_FONT_OPEN(font, dpi, dpi_ret, name, magstepval);
	if (f != NULL) {
	    *font_ret = NULL;
	    return f;
	}
	Strcpy(fn, font);
	fnend = fn + strlen(fn);
	while (fnend > fn && fnend[-1] >= '0' && fnend[-1] <= '9') --fnend;
	actual_pt = low_pt = high_pt = atoi(fnend);
	if (actual_pt) {
	    low_pt = actual_pt - 1;
	    high_pt = actual_pt + 1;
	    for (;;) {
		if (2 * low_pt >= actual_pt &&
		    (low_pt * high_pt > actual_pt * actual_pt ||
		    high_pt > actual_pt + 5))
			trial_pt = low_pt--;
		else if (high_pt > actual_pt + 5) break;
		else trial_pt = high_pt++;
		Sprintf(fnend, "%d", trial_pt);
		f = PRE_FONT_OPEN(fn, dpi * actual_pt / trial_pt, dpi_ret,
		    name, NOBUILD);
		if (f != NULL) {
		    *font_ret = xstrdup(fn);
		    return f;
		}
	    }
	}
	if (alt_font != NULL) {
	    f = PRE_FONT_OPEN(alt_font, dpi, dpi_ret, name, magstepval);
	    if (f != NULL)
		*font_ret = xstrdup(alt_font);
	}
	return f;
}
