/* fildialog.c -- stuff for the filter dialog				*/
/*
 * Copyright (c) 1993  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */
#include "xdatplot.h"
#include "rfiles.h"

#define	MAXROW	4			/* Max number filters in a row	*/
#define	NO_PIX	((Pixmap) NULL)

#include "arrow.xbm"
static	Pixmap	arrow_pm = NO_PIX;

/*
 * bitmap stuff
 */
typedef	struct	Bitmap {
    int		width;
    int		height;
    char	*bits;
    Pixmap	pixmap;
}		Bitmap;

/*
 * filter info and methods
 */
typedef	struct	FilterParam {
    String	name;			/* unique name for widgets	*/
    String	label;			/* user-visible label		*/
    Bool	visible;		/* managed iff TRUE		*/
    String	type;			/* Xt type			*/
    unsigned	where;			/* offset in FILTER_DATA	*/
    unsigned	size;			/* size in bytes		*/
}		FilterParam;

typedef	struct	FilterInfo {
    int		np;			/* number parameters		*/
    FilterParam	*params;		/* list of params		*/
    String	widget_name;		/* widget names start with...	*/
    Bitmap	icon;			/* icon for filter widget	*/
    FILTER	*(*make_edit)();	/* make filter from edit widget	*/
    void	(*set_edit)();		/* associate edit w/ filter	*/
}		FilterInfo;

#ifdef	__STDC__
static	void	create_edits(void);
static	void	filter_ok_cb(Widget, Widget,
			     XmPushButtonCallbackStruct *);
static	void	filter_insert_cb(Widget, Widget,
				 XmPushButtonCallbackStruct *);
static	void	filter_type_ok(Widget, XtPointer,
			       XmSelectionBoxCallbackStruct *);
static	void	filter_type_help(Widget, XtPointer,
				 XmSelectionBoxCallbackStruct *);
static	void	filter_type_cancel(Widget, XtPointer,
				   XmSelectionBoxCallbackStruct *);
static	FILTER	*current_filter(void);
static	void	filter_type_no_match(Widget, XtPointer,
				     XmSelectionBoxCallbackStruct *);
static	void	filter_delete_cb(Widget, Widget,
				 XmPushButtonCallbackStruct *);
static	void	filter_help_cb(Widget, Widget,
			       	XmPushButtonCallbackStruct *);
static	Widget	make_filter_widget(Widget, FILTER *, int);
static	void	current_cb(Widget, int, XmPushButtonCallbackStruct *);
static	void	set_shadow(int, int);
static	void	set_df_edit(FILTER *);
static	FILTER	*empty_err();
static	FILTER	*make_g_edit(FILTER_TYPE);
static	void	set_g_edit(FILTER *);
static	Widget	find_g_edit(FILTER_TYPE);
static	void	edit_g_field_cb(Widget, FilterInfo *, XmAnyCallbackStruct *);
static	void	edit_df_field_cb(Widget, FilterInfo *, XmAnyCallbackStruct *);
static	int	filternum(FILTER *);
#else	/* __STDC__ */
static	void	create_edits();
static	void	filter_ok_cb();
static	void	filter_insert_cb();
static	void	filter_type_ok();
static	void	filter_type_help();
static	void	filter_type_cancel();
static	FILTER	*current_filter(void);
static	void	filter_type_no_match();
static	void	filter_delete_cb();
static	void	filter_help_cb();
static	void	filter_type_cb();
static	Widget	make_filter_widget();
static	void	current_cb();
static	void	set_shadow();
static	FILTER	*make_g_edit();
static	void	set_g_edit();
static	Widget	find_g_edit();
static	void	edit_g_field_cb();
static	void	edit_df_field_cb();
static	int	filternum();
#endif	/* __STDC__ */

#include "nodatafile.xbm"
#include "datfil.xbm"
static	FilterParam	df_params[] = {
    {"fileName", "File name", TRUE, XtRStringTable,
     XtOffsetOf(DF_DATA, fname), sizeof(String)}
};
#include "hipasstau.xbm"
static	FilterParam	hp_tau_params[] = {
    {"hpTau", "tau", TRUE, XtRTime,
     XtOffsetOf(HP_TAU_DATA, tau), sizeof(double)},
    {"hpTol", "tol", FALSE, XtRDouble,
     XtOffsetOf(HP_TAU_DATA, tol), sizeof(double)}
};
#include "hipassfrq.xbm"
static	FilterParam	hp_f_params[] = {
    {"hpFrequency", "-3 dB frequency", TRUE, XtRFreq,
     XtOffsetOf(HP_F_DATA, f), sizeof(double)},
    {"hpTol", "tol", FALSE,
     XtRDouble, XtOffsetOf(HP_F_DATA, tol), sizeof(double)}
};
#include "lopasstau.xbm"
static	FilterParam	lp_tau_params[] = {
    {"lpTau", "tau", TRUE, XtRTime,
     XtOffsetOf(LP_TAU_DATA, tau), sizeof(double)},
    {"lpTol", "tol", FALSE, XtRDouble,
     XtOffsetOf(LP_TAU_DATA, tol), sizeof(double)}
};
#include "lopassfrq.xbm"
static	FilterParam	lp_f_params[] = {
    {"lpFrequency", "-3 dB frequency", TRUE, XtRFreq,
     XtOffsetOf(LP_F_DATA, f), sizeof(double)},
    {"lpTol", "tol", FALSE, XtRDouble,
     XtOffsetOf(LP_F_DATA, tol), sizeof(double)}
};
#include "minmax.xbm"
static	FilterParam	mm_params[] = {
    {"mmN", "N", TRUE, XtRInt,
     XtOffsetOf(MM_DATA, n), sizeof(int)}
};
#include "scalar.xbm"
static	FilterParam	sc_params[] = {
    {"scFactor", "Scale Factor", TRUE, XtRDouble,
     XtOffsetOf(SC_DATA, factor), sizeof(double)}
};
#include "pick.xbm"
static	FilterParam	pk_params[] = {
    {"pkStart", "Start", TRUE, XtRInt,
     XtOffsetOf(PK_DATA, st), sizeof(int)},
    {"pkN", "N", TRUE, XtRInt,
     XtOffsetOf(PK_DATA, n), sizeof(int)}
};
#include "tpmatch.xbm"
static	FilterParam	tp_params[] = {
    {"tpFile", "Template file", True, XtRString,
     XtOffsetOf(TP_DATA, fname), sizeof(String)},
    {"tpTscale", "Time scale", False, XtRTime,
     XtOffsetOf(TP_DATA, tscale), sizeof(double)},
    {"tpFscale", "F scale", False, XtRVoltage,
     XtOffsetOf(TP_DATA, fscale), sizeof(double)},
    {"tpMaxout", "Max output", False, XtRDouble,
     XtOffsetOf(TP_DATA, maxout), sizeof(double)},
    {"tpMax", "Max", True, XtRDouble,
     XtOffsetOf(TP_DATA, max), sizeof(double)},
    {"tpMin", "Min", True, XtRDouble,
     XtOffsetOf(TP_DATA, min), sizeof(double)}
};
#include "convolve.xbm"
static	FilterParam	cv_params[] = {
    {"cvFile", "Template file", True, XtRString,
     XtOffsetOf(CV_DATA, fname), sizeof(String)},
    {"cvTscale", "Time scale", True, XtRTime,
     XtOffsetOf(CV_DATA, tscale), sizeof(double)}
};

static	FilterInfo	filter_info[F_NUMBER] = {
    {0, NULL, "emptyFilter",
     {nodatafile_width, nodatafile_height, nodatafile_bits, NO_PIX},
     empty_err, set_g_edit},
    {XtNumber(df_params), df_params, "dfFilter",
     {datfil_width, datfil_height, datfil_bits, NO_PIX},
     NULL, set_df_edit},
    {XtNumber(hp_tau_params), hp_tau_params, "hptFilter",
     {hipasstau_width, hipasstau_height, hipasstau_bits, NO_PIX},
     make_g_edit, set_g_edit},
    {XtNumber(hp_f_params), hp_f_params, "hpfFilter",
     {hipassfrq_width, hipassfrq_height, hipassfrq_bits, NO_PIX},
     make_g_edit, set_g_edit},
    {XtNumber(lp_tau_params), lp_tau_params, "lptFilter",
     {lopasstau_width, lopasstau_height, lopasstau_bits, NO_PIX},
     make_g_edit, set_g_edit},
    {XtNumber(lp_f_params), lp_f_params, "lpfFilter",
     {lopassfrq_width, lopassfrq_height, lopassfrq_bits, NO_PIX},
     make_g_edit, set_g_edit},
    {XtNumber(mm_params), mm_params, "mmFilter",
     {minmax_width, minmax_height, minmax_bits, NO_PIX},
     make_g_edit, set_g_edit},
    {XtNumber(sc_params), sc_params, "scFilter",
     {scalar_width, scalar_height, scalar_bits, NO_PIX},
     make_g_edit, set_g_edit},
    {XtNumber(pk_params), pk_params, "pkFilter",
     {pick_width, pick_height, pick_bits, NO_PIX},
     make_g_edit, set_g_edit},
    {XtNumber(tp_params), tp_params, "tpFilter",
     {templatematch_width, templatematch_height, templatematch_bits, NO_PIX},
     make_g_edit, set_g_edit},
    {XtNumber(cv_params), cv_params, "cvFilter",
     {convolve_width, convolve_height, convolve_bits, NO_PIX},
     make_g_edit, set_g_edit}
};

String	scratch = NULL;

static	Widget	filter_sh = NO_WIDGET;
static	Widget	filter_action_frame = NO_WIDGET;
static	Widget	new_filter_form = NO_WIDGET;
static	Widget	new_filter_frame = NO_WIDGET;
static	Widget	current_filter_form = NO_WIDGET;
static	Widget	current_filter_frame = NO_WIDGET;
static	Widget	filter_type_SB = NO_WIDGET;
static	Widget	*current_filter_widgets = NULL;
static	Widget	filter_edit_widgets[F_NUMBER];
static	struct {
    Dimension	w, h;
}		filter_edit_sizes[F_NUMBER];
static	Widget	filter_type_pu = NO_WIDGET;

Widget
make_filter_dialog()
{
    char		lbuf[LLEN];
    String		s;
    Widget		filter_rowcol;
    Widget		filter_action_form;
    Widget		filter_ok_button;
    Widget		filter_insert_button;
    Widget		filter_delete_button;
    Widget		filter_help_button;
    
    /*
     * Create the dialog shell and a RowColumn widget to go in it
     */
    XtVaGetValues(toplevel, XmNtitle, &s, NULL);
    sprintf(lbuf, "%s Analyze Filter", s);
    filter_sh = XtVaAppCreateShell("filterShell", "FilterShell",
	topLevelShellWidgetClass, XtDisplay(toplevel),
	XtNtitle, lbuf,
	XmNiconName, "filters",
	XmNiconPixmap, app_data.icon,
	XmNdeleteResponse, XmUNMAP,
    NULL);
    filter_rowcol = XtVaCreateWidget("filterRowCol",
	xmRowColumnWidgetClass, filter_sh,
	XmNorientation, XmVERTICAL,
	XmNpacking, XmPACK_TIGHT,
	XmNnumColumns, 1,
	XmNresizeHeight, True,
	XmNresizeWidth, True,
	XmNspacing, 0,
	XmNmarginHeight, 0,
	XmNmarginWidth, 0,
    NULL);
    /*
     * create the widgets representing the current filters
     */
    current_filter_frame = XtVaCreateManagedWidget("currentFilterFrame",
	xmFrameWidgetClass, filter_rowcol,
	XmNshadowType, XmSHADOW_ETCHED_IN,
    NULL);
    current_filter_form = XtVaCreateWidget("currentFilterForm",
	xmFormWidgetClass, current_filter_frame,
	XmNuserData, (XtPointer) 0,	/* current filter: 0 = datafile	*/
    NULL);
    make_current_filters();
    XtManageChild(current_filter_form);
    /*
     * Create the widget for editing filters, but leave children unmanaged
     */
    new_filter_frame = XtVaCreateManagedWidget("newFilterFrame",
	xmFrameWidgetClass, filter_rowcol,
	XmNshadowType, XmSHADOW_ETCHED_IN,
    NULL);
    new_filter_form = XtVaCreateWidget("newFilterForm",
	xmFormWidgetClass, new_filter_frame,
    NULL);
    create_edits();
    XtManageChild(new_filter_form);
    /*
     * Create the action area
     */
    filter_action_frame = XtVaCreateManagedWidget("filterActionFrame",
	xmFrameWidgetClass, filter_rowcol,
	XmNshadowType, XmSHADOW_ETCHED_IN,
    NULL);
    filter_action_form = XtVaCreateWidget("filterActionForm",
	xmFormWidgetClass, filter_action_frame,
    NULL);
    filter_ok_button = XtVaCreateManagedWidget("filterOkButton",
	xmPushButtonWidgetClass, filter_action_form,
    NULL);
    AddCallback(filter_ok_button, XmNactivateCallback, filter_ok_cb, filter_sh);
    filter_insert_button = XtVaCreateManagedWidget("filterInsertButton",
	xmPushButtonWidgetClass, filter_action_form,
    NULL);
    AddCallback(filter_insert_button, XmNactivateCallback,
		filter_insert_cb, filter_sh);
    filter_delete_button = XtVaCreateManagedWidget("filterDeleteButton",
	xmPushButtonWidgetClass, filter_action_form,
    NULL);
    AddCallback(filter_delete_button, XmNactivateCallback,
		filter_delete_cb, filter_sh);
    filter_help_button = XtVaCreateManagedWidget("filterHelpButton",
	xmPushButtonWidgetClass, filter_action_form,
    NULL);
    AddCallback(filter_help_button, XmNactivateCallback,
		filter_help_cb, filter_sh);
    XtManageChild(filter_action_form);
    /*
     * manage the dialog and return
     */
    XtManageChild(filter_rowcol);
    current_cb(NO_WIDGET, 0, NULL);	/* make current filter current	*/
    return(filter_sh);
}

static void
create_edits()
{
    char		lbuf[LLEN];
    int			i;
    int			np;
    FilterParam		*p;
    FILTER_TYPE		type;
    Widget		fewt;
    Widget		label;
    Widget		lastlabel;
    Widget		textf = NO_WIDGET;
    Dimension		w1, h1, w2, h2, wf, hf;

    for(type = F_EMPTY; type < F_NUMBER; type++) {
	fewt = XtVaCreateWidget(filter_info[type].widget_name,
	    xmFormWidgetClass, new_filter_form,
	    XmNuserData, NULL,
	    XmNtopAttachment, XmATTACH_FORM,
	    XmNleftAttachment, XmATTACH_FORM,
	    XmNrightAttachment, XmATTACH_FORM,
	    XmNbottomAttachment, XmATTACH_FORM,
	NULL);
	label = XtVaCreateManagedWidget("editLabel",
	    xmLabelWidgetClass, fewt,
	    XtVaTypedArg, XmNlabelString, XmRString,
	    filter_names[type], strlen(filter_names[type]) + 1,
	    XmNtopAttachment, XmATTACH_FORM,
	    XmNleftAttachment, XmATTACH_FORM,
	    XmNrightAttachment, XmATTACH_FORM,
	NULL);
	GetSize(label, wf, hf);
	lastlabel = label;
	np = filter_info[type].np;
	p = filter_info[type].params;
	for(i=0; i<np; i++, p++) {
	    sprintf(lbuf, "%sField", p->name);
	    textf = XtVaCreateWidget(lbuf,
		xmTextFieldWidgetClass, fewt,
	    NULL);
	    AddCallback(textf, XmNactivateCallback,
			edit_g_field_cb, &filter_info[type]);
	    sprintf(lbuf, "%sLabel", p->name);
	    label = XtVaCreateWidget(lbuf,
		xmLabelWidgetClass, fewt,
		XtVaTypedArg, XmNlabelString, XmRString,
		p->label, strlen(p->label)+1,
		XmNalignment, XmALIGNMENT_END,
	    NULL);
	    if (p->visible) {
		XtVaSetValues(textf,
		    XmNtopAttachment, XmATTACH_WIDGET,
		    XmNtopWidget, lastlabel,
		    XmNrightAttachment, XmATTACH_FORM,
		NULL);
		XtManageChild(textf);
		XtVaSetValues(label,
		    XmNleftAttachment, XmATTACH_FORM,
		    XmNrightAttachment, XmATTACH_WIDGET,
		    XmNrightWidget, textf,
		    XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET,
		    XmNtopWidget, textf,
		    XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET,
		    XmNbottomWidget, textf,
		NULL);
		XtManageChild(label);
		lastlabel = label;
		GetSize(label, w1, h1);
		GetSize(textf, w2, h2);
		wf = max(wf, w1 + w2);
		hf += max(h1, h2);
	    }
	}
	if (np > 0)
	    XtVaSetValues(textf, XmNbottomAttachment, XmATTACH_FORM, NULL);
	filter_edit_widgets[type] = fewt;
	filter_edit_sizes[type].w = wf;
	filter_edit_sizes[type].h = hf;
    }
}

void
make_current_filters()
{
    int			n;
    WidgetList		children;
    FILTER		*fp;
    int			nf;		/* Number of filters		*/
    int			cf;		/* current filter		*/
    Dimension		w, h, wr, hr, wf, hf;

    if (NO_WIDGET == current_filter_form) return;
    /*
     * destroy all children and start from scratch
     */
    if (!debug) XtUnmanageChild(current_filter_form);
    XtVaGetValues(current_filter_form,
	XmNchildren, &children,
	XmNnumChildren, &nf,
    NULL);
    for(n=0; n<nf; n++)  {
	XtDestroyWidget(children[n]);
    }
    if (NULL != current_filter_widgets)
	Free(current_filter_widgets);
    /*
     * make a widget for each filter and stick it in the list
     */
    nf = filternum(C_FILTER) + 1;
    current_filter_widgets = (Widget *) XtCalloc(nf, sizeof(Widget));
    XtVaGetValues(current_filter_form, XmNuserData, &cf, NULL);
    if ((0 > cf) || (cf >= nf)) {
	cf = nf - 1;
	XtVaSetValues(current_filter_form, XmNuserData, cf, NULL);
    }
    wf = hf = wr = hr = 0;
    for(
	n = 0, fp = &DF_FIL;
	NULL != fp;
	n++, fp = destination_filter(fp)
    ) {
	current_filter_widgets[n] = make_filter_widget(current_filter_form,
						       fp, n);
	set_shadow(n, (n == cf) ? XmSHADOW_IN : XmSHADOW_OUT);
	GetSize(current_filter_widgets[n], w, h);
	if (0 == n % MAXROW) {		/* start new row		*/
	    XtVaSetValues(current_filter_widgets[n],
		XmNleftAttachment, XmATTACH_FORM,
	    NULL);
	    wr = w;
	    hr = h;
	}
	else {				/* continue same row		*/
	    XtVaSetValues(current_filter_widgets[n],
		XmNleftAttachment, XmATTACH_WIDGET,
		XmNleftWidget, current_filter_widgets[n-1],
	    NULL);
	    wr += w;
	    hr = max(hr, h);
	}
	if (MAXROW > n) {		/* still in top row		*/
	    XtVaSetValues(current_filter_widgets[n],
		XmNtopAttachment, XmATTACH_FORM,
	    NULL);
	}
	else {				/* not in top row		*/
	    XtVaSetValues(current_filter_widgets[n],
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, current_filter_widgets[n-MAXROW],
	    NULL);
	}
	if (				/* end of a row			*/
	    MAXROW-1 == n % MAXROW ||
	    n == nf - 1
	) {
	    wf = max(wf, wr);
	    hf += hr;
	}
    }
    current_cb(NO_WIDGET, cf, NULL);	/* make current filter current	*/
    GetSize(current_filter_form, w, h);
    SetSize(current_filter_form, max(w, wf), hf);
    if (!debug) XtManageChild(current_filter_form);
}

static Widget
make_filter_widget(current_filter_form, fp, n)
Widget		current_filter_form;
FILTER		*fp;
int		n;
{
    Widget		frame;
    Widget		form;
    Widget		arrow;
    Pixel		fg, bg;
    unsigned int	depth;
    Widget		button;
    FilterInfo		*fi;
    Dimension		w1, h1, w2, h2, st;
    int			lo, ro;

    form = XtVaCreateWidget("singleFilterForm",
	xmFormWidgetClass, current_filter_form,
	XmNuserData, fp,
    NULL);
    frame = XtVaCreateManagedWidget("singleFilterFrame",
	xmFrameWidgetClass, form,
    NULL);
    fi = &filter_info[filter_type(fp)];
    if (NO_PIX == fi->icon.pixmap) {
	XtVaGetValues(form,
	    XmNforeground, &fg,
	    XmNbackground, &bg,
	    XmNdepth, &depth,
	NULL);
	fi->icon.pixmap = XCreatePixmapFromBitmapData(
	    XtDisplay(toplevel),
	    XtWindow(toplevel),
	    fi->icon.bits,
	    fi->icon.width,
	    fi->icon.height,
	    fg, bg, depth
	);
    }
    button = XtVaCreateManagedWidget("currentFilterButton",
	xmPushButtonWidgetClass, frame,
	XmNlabelType, XmPIXMAP,
	XmNlabelPixmap, fi->icon.pixmap,
    NULL);
    GetSize(button, w1, h1);
    XtVaGetValues(frame,
	XmNmarginWidth, &w2,
	XmNmarginHeight, &h2,
	XmNshadowThickness, &st,
    NULL);
    w1 += 2*w2 + 2*st;
    h1 += 2*h2 + 2*st;
    SetSize(frame, w1, h1);
    AddCallback(button, XmNactivateCallback, current_cb, n);
    if (NO_PIX == arrow_pm) {
	XtVaGetValues(form,
	    XmNforeground, &fg,
	    XmNbackground, &bg,
	    XmNdepth, &depth,
	NULL);
	arrow_pm = XCreatePixmapFromBitmapData(XtDisplay(toplevel),
					       XtWindow(toplevel),
					       arrow_bits, arrow_width,
					       arrow_height,
					       fg, bg, depth);
    }
    arrow = XtVaCreateManagedWidget("singleFilterArrow",
	xmLabelWidgetClass, form,
	XmNlabelType, XmPIXMAP,
	XmNlabelPixmap, arrow_pm,
	XmNtopAttachment, XmATTACH_FORM,
	XmNbottomAttachment, XmATTACH_FORM,
	XmNleftAttachment, XmATTACH_WIDGET,
	XmNleftWidget, frame,
    NULL);
    GetSize(arrow, w2, h2);
    XtManageChild(form);
    XtVaGetValues(arrow,
	XmNleftOffset, &lo,
	XmNrightOffset, &ro,
    NULL);
    SetSize(form, w1 + w2 + lo + ro, max(h1, h2));
    return(form);
}

/* current_cb -- make the clicked filter current			*/
static void
current_cb(button, n, cbs)
Widget				button;
int				n;
XmPushButtonCallbackStruct	*cbs;
{
    int			old;
    FILTER		*fp;

    XtVaGetValues(current_filter_form, XmNuserData, &old, NULL);
    set_shadow(old, XmSHADOW_OUT);
    XtVaSetValues(current_filter_form, XmNuserData, n, NULL);
    set_shadow(n, XmSHADOW_IN);
    XtVaGetValues(current_filter_widgets[n],
	XmNuserData, &fp,
    NULL);
    (*filter_info[filter_type(fp)].set_edit)(fp);
}

static void
set_shadow(n, shadow)
int		n;
int		shadow;
{
    Widget		frame;

    frame = XtNameToWidget(current_filter_widgets[n], "*singleFilterFrame");
    if (NULL == frame) return;
    XtVaSetValues(frame, XmNshadowType, shadow, NULL);
}

static void
set_df_edit(fp)
FILTER		*fp;
{
    Widget	fewt;
    Widget	field;

    set_g_edit(fp);
    fewt = filter_edit_widgets[filter_type(fp)];
    if (NULL == fewt) return;
    field = XtNameToWidget(fewt, "fileNameField");
    XtRemoveAllCallbacks(field, XmNactivateCallback);
    AddCallback(field, XmNactivateCallback, edit_df_field_cb,
		  filter_info[F_DF].params);
}

static void
edit_df_field_cb(field, p, cbs)
Widget			field;
FilterInfo		*p;
XmAnyCallbackStruct	*cbs;
{
    String		s = XmTextFieldGetString(field);

    new_file(s, True);
}

static FILTER *
empty_err()
{
    String		msg = "You can't do anything with the empty filter";

    PU_error(msg, NULL);
    return(NULL);
}

/* generic edit widget routines						*/
static FILTER *
make_g_edit(type)
FILTER_TYPE	type;
{
    FILTER		*fp;
    char		lbuf[LLEN];
    int			i;
    FilterParam		*p = filter_info[type].params;
    int			np = filter_info[type].np;
    Widget		fewt;
    Widget		field;
    FILTER_DATA		data;
    int			ret;
    String		val;
    String		**params;

    find_g_edit(type);
    fewt = filter_edit_widgets[type];
    add_converters(current_filter());
    bzero(&data, sizeof(data));
    for(i=0; i<np; i++, p++) {
	sprintf(lbuf, "%sField", p->name);
	field = XtNameToWidget(fewt, lbuf);
	if (NO_WIDGET == field) error("make_g_edit: can't happen");
	XtVaGetValues(field, XmNvalue, &val, NULL);
	ret = cvt_from_string(val, field, &data, p->where, p->size,
			      p->type, True);
	if (ret) error("make_g_edit: can't happen");
    }
    add_converters(NULL);
    fp = XtNew(FILTER);
    if (make_filter(type, &data, fp)) {
	PU_error("unable to create new filter", "filters.html");
	Free(fp);
	fp = NULL;
    }
    else {
	XtVaSetValues(fewt, XmNuserData, fp, NULL);
    }
    return(fp);
}

static void
set_g_edit(fp)
FILTER		*fp;
{
    char		lbuf[LLEN];
    int			i;
    FilterParam		*p = filter_info[filter_type(fp)].params;
    int			np = filter_info[filter_type(fp)].np;
    Widget		fewt;
    Widget		field;
    FILTER_DATA		data;
    String		val;

    find_g_edit(filter_type(fp));
    if (!is_active_filter(fp)) return;
    fewt = filter_edit_widgets[filter_type(fp)];
    if (NULL == fewt) return;
    XtVaSetValues(fewt, XmNuserData, fp, NULL);
    if (NULL != filter_data(fp))
	bcopy(filter_data(fp), &data, sizeof(FILTER_DATA));
    add_converters(source_filter(fp));
    for(i=0; i<np; i++, p++) {
	sprintf(lbuf, "%sField", p->name);
	field = XtNameToWidget(fewt, lbuf);
	if (NULL == field) error("set_g_edit: can't happen");
	val = cvt_to_string(field, &data, p->where, p->size, p->type);
	if (NULL == val) error("set_g_edit: can't happen");
	XtVaSetValues(field, XmNvalue, val, NULL);
    }
    add_converters(NULL);
}

/* find_g_edit -- find and manage edit widget, creating if necessary	*/
static Widget
find_g_edit(type)
FILTER_TYPE	type;
{
    WidgetList		children;
    int			nc;
    Widget		fewt = filter_edit_widgets[type];
    Dimension		w1, h1, w2, h2, wf, hf;

    if (
	(NO_WIDGET == fewt) ||
	XtIsManaged(fewt)
    ) return(fewt);
    XtVaGetValues(new_filter_form,
	XtNnumChildren, &nc,
	XtNchildren, &children,
    NULL);
    XtUnmanageChildren(children, nc);
    XtManageChild(fewt);
    return(fewt);
}

static void
edit_g_field_cb(field, fi, cbs)
Widget			field;
FilterInfo		*fi;
XmAnyCallbackStruct	*cbs;
{
    register int	i;
    Widget		fewt = XtParent(field);
    FilterParam		*p = fi->params;
    char		lbuf[LLEN];
    FILTER		*fp;
    FILTER_DATA		data;
    String		s;
    int			cfn;
    Bool		ret;
    String		**params;

					/* get the filter to change	*/
    XtVaGetValues(fewt, XmNuserData, &fp, NULL);
    if (NULL == fp) error("edit_g_field: can't happen");
    bcopy(filter_data(fp), &data, sizeof(FILTER_DATA));
    for(i=0; i<fi->np; i++, p++) {
	sprintf(lbuf, "%sField", p->name);
	field = XtNameToWidget(fewt, lbuf);
	if (NO_WIDGET == field) error("edit_g_field: can't happen");
	s = XmTextFieldGetString(field);
	ret = cvt_from_string(s, field, &data, p->where,
			      p->size, p->type, False);
	if (ret) {
	    sprintf(lbuf, "Invalid value for %s: %s", p->label, s);
	    PU_error(lbuf, "filters.html");
	    return;
	}
    }
    params = save_filter_params(destination_filter(fp));
    if (change_filter(fp, &data)) {
	PU_error("Unable to change filter parameters", "filters.html");
    }
    restore_filter_params(destination_filter(fp), params);
    redo_whole();
}

/* filternum -- find the number of a filter				*/
static int
filternum(fp)
FILTER		*fp;
{
    int			n;
    FILTER		*gp;

    for(
	n = 0, gp = &DF_FIL;
	NULL != gp;
	n++, gp = destination_filter(gp)
    ) {
	if (fp == gp) return(n);
    }
    return(-1);
}

static void
filter_ok_cb(widget, filter_sh, cbs)
Widget				widget;
Widget				filter_sh;
XmPushButtonCallbackStruct	*cbs;
{
    XtPopdown(filter_sh);
}

static void
filter_insert_cb(widget, filter_sh, cbs)
Widget				widget;
Widget				filter_sh;
XmPushButtonCallbackStruct	*cbs;
{
    register int	i;
    char		lbuf[LLEN];
    String		s;
    XmString		names[F_NUMBER - F_DF - 1];
    XmString		title;

    if (NO_WIDGET == filter_type_pu) {
	filter_type_pu = XmCreateSelectionDialog(filter_sh, "filterTypeSB",
						 NULL, 0);
	AddCallback(filter_type_pu, XmNokCallback, filter_type_ok, NULL);
	AddCallback(filter_type_pu, XmNcancelCallback,
		    filter_type_cancel, NULL);
	AddCallback(filter_type_pu, XmNhelpCallback, filter_type_help, NULL);
	AddCallback(filter_type_pu, XmNnoMatchCallback,
		    filter_type_no_match, NULL);
	for(i=0; i<F_NUMBER - F_DF - 1; i++) {
	    names[i] = XmStringCreateSimple((String) filter_names[i+F_DF+1]);
	}
	XtVaGetValues(toplevel, XmNtitle, &s, NULL);
	sprintf(lbuf, "%s Filter Insert", s);
	title = XmStringCreateSimple(lbuf);
	XtVaSetValues(filter_type_pu,
	    XmNdialogTitle, title,
	    XtVaTypedArg, XmNlistLabelString, XmRString,
	    "Filter Types:", strlen("Filter Types:")+1,
	    XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL,
	    XmNlistItems, names,
	    XmNlistItemCount, F_NUMBER - F_DF - 1,
	    XmNmustMatch, TRUE,
	NULL);
	XmStringFree(title);
	for(i=0; i<F_NUMBER - F_DF - 1; i++) {
	    XmStringFree(names[i]);
	}
    }
    XtManageChild(filter_type_pu);
    XtPopup(XtParent(filter_type_pu), XtGrabNone);
}

static void
filter_type_ok(widget, data, cbs)
Widget				widget;
XtPointer			data;
XmSelectionBoxCallbackStruct	*cbs;
{
    register int	i;
    String		s;
    FILTER		*nfp;		/* new filter			*/
    FILTER		*cfp;		/* current filter		*/
    int			cf;		/* current filter number	*/
    String		**params;

    XmStringGetLtoR(cbs->value, XmSTRING_DEFAULT_CHARSET, &s);
    for(i=F_DF; i<F_NUMBER; i++) {
	if (0 == strcmp(filter_names[i], s)) break;
    }
    if (i >= F_NUMBER) error("filter_type_ok: can't happen");
    nfp = (*filter_info[i].make_edit)(i);
    if (NULL == nfp) return;
    cfp = current_filter();
    params = save_filter_params(destination_filter(cfp));
    connect_filters(cfp, nfp, destination_filter(cfp));
    restore_filter_params(destination_filter(nfp), params);
    if (C_FILTER == cfp) C_FILTER = nfp;
    XtVaGetValues(current_filter_form, XmNuserData, &cf, NULL);
    XtVaSetValues(current_filter_form, XmNuserData, cf+1, NULL);
    make_current_filters();
    redo_whole();
    XtUnmanageChild(widget);
    XtPopdown(XtParent(widget));
}

static FILTER *
current_filter()
{
    int			cf;
    FILTER		*cfp;

    XtVaGetValues(current_filter_form, XmNuserData, &cf, NULL);
    XtVaGetValues(current_filter_widgets[cf], XmNuserData, &cfp, NULL);
    return(cfp);
}

static void
filter_type_cancel(widget, data, cbs)
Widget				widget;
XtPointer			data;
XmSelectionBoxCallbackStruct	*cbs;
{
    XtUnmanageChild(widget);
    XtPopdown(XtParent(widget));
}

static void
filter_type_no_match(widget, data, cbs)
Widget				widget;
XtPointer			data;
XmSelectionBoxCallbackStruct	*cbs;
{
    Bell;
}

static void
filter_delete_cb(widget, filter_sh, cbs)
Widget				widget;
Widget				filter_sh;
XmPushButtonCallbackStruct	*cbs;
{
    FILTER		*cfp;
    FILTER		*sfp;
    FILTER		*dfp;
    String		**params;
    int			cf;
    Bool		ret;

    cfp = current_filter();
    if (&DF_FIL == cfp) {
	PU_error("You can't delete the first filter", "filters.html");
	return;
    }
    sfp = source_filter(cfp);
    dfp = destination_filter(cfp);
    params = save_filter_params(dfp);
    ret = kill_filter(cfp);
    restore_filter_params(dfp, params);
    if (ret) {
	PU_error("Couldn't delete filter", "filters.html");
	return;
    }
    if (C_FILTER == cfp) C_FILTER = sfp;
    XtVaGetValues(current_filter_form, XmNuserData, &cf, NULL);
    XtVaSetValues(current_filter_form, XmNuserData, cf-1, NULL);
    make_current_filters();
    redo_whole();
}

static void
filter_help_cb(widget, filter_sh, cbs)
Widget				widget;
Widget				filter_sh;
XmPushButtonCallbackStruct	*cbs;
{
    help_window("filters.html");
}

/* save_filter_params and restore_filter_params -- save and restore
 * filter chains
 *
 * FILTER	*first;
 * String	**params;
 * params = save_filter_params(first);
 *
 * ...
 *
 * restore_filter_params(first, params);
 *
 * save_filter_params scans through the filter chain beginning at
 * first and converts the filter parameters to text.  It returns a
 * pointer to pointers to arrays of pointers to the text values, all
 * allocated from the heap.  restore_filter_params is called some time
 * later to restore these values.  It also frees all the space
 * associated with params.  discard_filter_params(params) frees space
 * associated with the params. 
 */
String **
save_filter_params(first)
FILTER		*first;
{
    char		lbuf[LLEN];
    FILTER		*fp;
    int			nf;
    String		**spp;
    String		**ppp;
    FilterParam		*p;
    int			np;
    int			i;
    FILTER_DATA		*dp;
    String		val;

    for(
	nf = 0, fp = first;
	NULL != fp;
	nf++, fp = destination_filter(fp)
    );
    if (0 == nf) return(NULL);
    spp = (String **) XtCalloc(nf+1, sizeof(String *));
    for(
	fp = first, ppp = spp;
	NULL != fp;
	(fp = destination_filter(fp)), ppp++
    ) {
	p = filter_info[filter_type(fp)].params;
	np = filter_info[filter_type(fp)].np;
	*ppp = (String *) XtCalloc(np+1, sizeof(String));
	if (0 == np) continue;
	add_converters(source_filter(fp));
	dp = filter_data(fp);
	for(i=0; i<np; i++) {
	    val = cvt_to_string(toplevel, dp, p[i].where, p[i].size, p[i].type);
	    if (NULL == val) error("save_filter_params: can't happen");
	    (*ppp)[i] = XtNewString(val);
	}
    }
    add_converters(NULL);
    return(spp);
}

void
restore_filter_params(fp, spp)
FILTER		*fp;
String		**spp;
{
    String		**ppp;
    FilterParam		*p;
    int			np;
    int			i;
    FILTER_DATA		data;
    int			ret;
    int			err = FALSE;

    if (NULL == spp) return;
    bzero(&data, sizeof(data));
    for(
	ppp = spp;
	NULL != fp;
	fp = destination_filter(fp), ppp++
    ) {
	p = filter_info[filter_type(fp)].params;
	np = filter_info[filter_type(fp)].np;
	add_converters(source_filter(fp));
	for(i=0; i<np; i++) {
	    ret = cvt_from_string((*ppp)[i], toplevel, &data, p[i].where,
				  p[i].size, p[i].type, True);
	    if (ret) error("restore_filter_params: can't happen");
	    Free((*ppp)[i]);
	}
	if (NULL != *ppp) Free(*ppp);
	if (change_filter(fp, &data)) err = TRUE;
    }
    Free(spp);
    add_converters(NULL);
    if (err) PU_error("Unable to restore previous filter parameters",
		      "filters.html");
}

void
discard_filter_params(spp)
String		**spp;
{
    String		**ppp;
    String		*pp;

    if (NULL == spp) return;
    for(
	ppp = spp;
	NULL != *ppp;
	ppp++
    ) {
	for(pp = *ppp; NULL != *pp; pp++) {
	    Free(*pp);
	}
	Free(*ppp);
    }
    Free(spp);
}

static void
filter_type_help(widget, data, cbs)
Widget				widget;
XtPointer			data;
XmSelectionBoxCallbackStruct	*cbs;
{
    help_window("filters.html");
}

/* CvtStringToFilterChain -- Xt conversions to and from Filter chains
 * CvtFilterChainToString
 *
 * These are Xt conversion routines for filter chains.  They are
 * really intended for just one special purpose: saving and restoring
 * the filter chain attached to the datafile.  CvtFilterChainToString
 * expects to be passed a pointer to the first filter in a chain,
 * (normally the datafile filter), but the String it creates only
 * describes the subsequent filters.  Similiarly,
 * CvtStringToFilterChain takes a pointer to a filter chain, but
 * leaves the first filter in the chain unchanged, creating a filter
 * chain it attaches to it.  The main reason for this is so that it
 * can disconnect any filters attached to the first filter on call.
 */
Boolean
CvtFilterChainToString(disp, args, nargs, from, to)
Display		*disp;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    int			i;
    FILTER		*fp = (FILTER *) from->addr;
    String		fc = XtNewString("");
    int			nc = 0;
    FilterParam		*p;
    int			np;
    FILTER_DATA		*dp;
    Bool		ret;
    String		s;

    if (0 != *nargs) {
	XtWarningMsg("wrongParameters", "cvtFilterChainToString",
		     "XtToolkitError", "FilterChain to String\
		     conversion needs no extra arguments",
		     NULL, NULL);
    }
    for(
	fp = destination_filter(fp);
	NULL != fp;
	fp = destination_filter(fp)
    ) {
	p = filter_info[filter_type(fp)].params;
	np = filter_info[filter_type(fp)].np;
	fc = append_string(fc, filter_info[filter_type(fp)].widget_name, &nc);
	add_converters(source_filter(fp));
	dp = filter_data(fp);
	for(i=0; i<np; i++) {
	    s = cvt_to_string(toplevel, dp, p[i].where, p[i].size, p[i].type);
	    fc = append_string(fc, " ", &nc);
	    if (0 == strcmp(XtRString, p[i].type)) {
		s = escape_resource(s);
		fc = append_string(fc, "\"", &nc);
		fc = append_string(fc, s, &nc);
		fc = append_string(fc, "\"", &nc);
		Free(s);
	    }
	    else {
		fc = append_string(fc, s, &nc);
	    }
	}
	fc = append_string(fc, "\n", &nc);
    }
    add_converters(NULL);
    if (NULL == to->addr) {
	Nfree(scratch);
	scratch = fc;
	to->addr = (caddr_t) scratch;
	to->size = nc + 1;
	return(True);
    }
    else if (to->size <= nc) {
	to->size = nc + 1;
	Free(fc);
	return(False);
    }
    else {
	strcpy((String) to->addr, fc);
	to->size = nc + 1;
	Free(fc);
	return(True);
    }
}

Boolean
CvtStringToFilterChain(disp, args, nargs, from, to)
Display		*disp;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    FILTER		*fp;
    FILTER		*gp;
    String		fc;
    int			nc;
    char		lbuf[LLEN];
    int			type;
    FilterParam		*p;
    int			np;
    int			i;
    FILTER_DATA		data;
    Bool		ret;
    Bool		err = FALSE;
    String		s;

    if (0 != *nargs) {
	XtWarningMsg("wrongParameters", "cvtStringToFilterChain",
		     "XtToolkitError", "String to FilterChain\
		     conversion needs no extra arguments",
		     NULL, NULL);
    }
    if (NULL == to->addr) {
	XtWarningMsg("wrongValue", "cvtStringToFilterChain",
		     "XtToolkitError", "unknown filter type",
		     NULL, NULL);
	return(False);
    }
    if (sizeof(FILTER) > to->size) {
	to->size = sizeof(FILTER);
	return(False);
    }
    fp = (FILTER *) to->addr;
    if (
	(!is_active_filter(fp)) &&
	(make_filter(F_EMPTY, NULL, fp))
    ) {
	XtWarningMsg("filterError", "cvtStringToFilterChain",
		     "XtToolkitError", "unable to make empty filter",
		     NULL, NULL);
	return(False);
    }
    if (disconnect_filter(fp)) {
	XtWarningMsg("filterError", "cvtStringToFilterChain",
		     "XtToolkitError", "unable to disconnect source filter",
		     NULL, NULL);
	return(False);
    }
    for(
	fc = (String) from->addr;
	(((String) NULL + 1) != fc) && ('\0' != fc[strspn(fc, WHITE)]);
	fc = STRCHR(fc, '\n') + 1
    ) {
	fc += strspn(fc, LWHITE);
	nc = strcspn(fc, WHITE);
	strncpy(lbuf, fc, min(LLEN-1, nc));
	fc += nc;
	lbuf[min(LLEN-1, nc)] = '\0';
	for(i=0; i<F_NUMBER; i++) {
	    if (0 == strcmp(lbuf, filter_info[i].widget_name)) break;
	}
	if (i >= F_NUMBER) {
	    XtWarningMsg("wrongValue", "cvtStringToFilterChain",
			 "XtToolkitError", "unknown filter type",
			 NULL, NULL);
	    continue;
	}
	type = i;
	p = filter_info[type].params;
	np = filter_info[type].np;
	add_converters(fp);
	for(i=0; i<np; i++) {
	    fc += strspn(fc, LWHITE);
	    if ('\"' == *fc) {
		fc++;
		s = quoted_string(&fc, '\"');
		store_string(s);
	    }
	    else {
		nc = strcspn(fc, WHITE);
		strncpy(lbuf, fc, min(LLEN-1, nc));
		fc += nc;
		lbuf[min(LLEN-1, nc)] = '\0';
		if (0 == strcmp(XtRString, p[i].type))
		    s = save_string(lbuf);
		else
		    s = lbuf;
	    }
	    ret = cvt_from_string(s, toplevel, &data, p[i].where,
				  p[i].size, p[i].type, False);
	}
	gp = XtNew(FILTER);
	if (
	    (ret = make_filter(type, &data, gp)) ||
	    (connect_filters(fp, gp, NULL))
	    ) {
	    XtWarningMsg("wrongValue", "cvtStringToFilterChain",
			 "XtToolkitError", "couldn't create filter",
			 NULL, NULL);
	    if (!ret) kill_filter(gp);
	    Free(gp);
	    clear_strings();
	    continue;
	}
	fp = gp;
	clear_strings();
    }
    add_converters(NULL);
    to->size = sizeof(FILTER);
    return(True);
}
