/*-----------------------------------------------------------------------*/
/* text.c --- text processing routines for xcircuit		 	 */
/*-----------------------------------------------------------------------*/

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

/*------------------------------------------------------------------------*/
/* Local includes                                                         */
/*------------------------------------------------------------------------*/

#include "colordefs.h"
#include "xcircuit.h"
#include "alpha.h"

/*------------------------------------------------------------------------*/
/* External Variable definitions                                          */
/*------------------------------------------------------------------------*/

extern Display *dpy;
extern Clientdata areastruct;
extern short textpos;

/*------------------------------------------------------------------------*/
/* Declarations of functions defined externally to functions.c            */
/*------------------------------------------------------------------------*/

short ULength();
extern void UDrawLines();
extern float UTopScale();
extern int *appcolors;

#define HALFSPACE 3

/*---------------------------------------------------------------------*/
/* draw a single character at 0, 0 using current transformation matrix */
/*---------------------------------------------------------------------*/

short UDrawChar(localdata, code, styles)
  objinstptr 	localdata;
  short 	code, styles;
{
   alphaptr drawchar;
   XPoint alphapts[2], *curpoint;
   short  totalpts, i;
   short  ffont = (styles & 6);
   short  localwidth;
   XPoint *pointptr;

   /* set proper font */

   switch (ffont) {
      case 0: drawchar = alphabet[code]; 	break;
      case 2: drawchar = helveticabet[code]; 	break;
      case 4: drawchar = courierbet[code]; 	break;
      case 6: drawchar = symbolbet[code];	break;
   }

   localwidth = (ffont == 4) ? COURIER_WIDTH : drawchar->width;
   totalpts = 1 + drawchar->number[0] + drawchar->number[1] + 
	drawchar->number[2];

   if (styles & 1) USlantCTM(DCTM, 0.25);  /* premultiply by slanting function */
					   /* for italic styles		       */
   if (!(styles & 64)) {
      for (i = 0; i < drawchar->parts; i++)
	 UDrawLines(localdata, drawchar->points[i], drawchar->number[i]);

      /* under- and overlines */
      if (styles & 8)
         alphapts[0].y = alphapts[1].y = -6;
      else if (styles & 16)
         alphapts[0].y = alphapts[1].y = TEXTHEIGHT - 6;
      else if (styles & 32)
         alphapts[0].y = alphapts[1].y = TEXTHEIGHT + 4;
      if (styles & 56) {
         alphapts[0].x = -HALFSPACE; alphapts[1].x = localwidth;
         UDrawSimpleLine(localdata, &alphapts[0], &alphapts[1]);
      }
   }
   return ((styles & 64) ? -localwidth : localwidth);
}

/* setlineattrib and setscript are just conveniences. . . */

void setlineattrib(style)
   short style;
{
   short tmpwidth;
   float tscale = UTopScale();

   tmpwidth = (short)(((style & 2) ? 4.0 : 2.0) * tscale);
   if (tmpwidth < 2) tmpwidth = 0;
   XSetLineAttributes(dpy, areastruct.gc, tmpwidth, LineSolid, CapRound, 
	JoinBevel);
}

void UDrawString(localdata, drawlabel)
  objinstptr 	localdata;
  labelptr	drawlabel;
{
   uchar *strptr = drawlabel->string;
   short  tmplength, fstyle, ffont, fline = 0, backct = 1, tmpwidth;
   short  oldy, oldfont, oldstyle;
   float  tmpscale = 1.0;
   XPoint newpoint, bboxin[2], bboxout[2];
   Boolean xm, ym;

   tmplength = ULength(drawlabel->string, 0.0, 0); /* "natural" (unscaled) length */

   newpoint.x = (drawlabel->justify & NOTLEFT ?
       (drawlabel->justify & RIGHT ? -tmplength : -tmplength >> 1) : 0);
   newpoint.y = (drawlabel->justify & NOTBOTTOM ?
       (drawlabel->justify & TOP ? -TEXTHEIGHT : -HALFHEIGHT) : 0);
   oldy = newpoint.y;

   /* calculate the transformation matrix for this object */
   /* in natural units of the alphabet vectors		  */
   /* (conversion to window units)			  */

   UPushCTM();
   UPreMultCTM(DCTM, drawlabel->position, drawlabel->scale, drawlabel->rotation);

   /* do quick calculation on bounding box; don't draw if off-screen */

   bboxin[0].x = newpoint.x;
   bboxin[0].y = newpoint.y - HALFHEIGHT;
   bboxin[1].x = newpoint.x + tmplength;
   bboxin[1].y = newpoint.y + TEXTHEIGHT + HALFHEIGHT;
   UTransformbyCTM(DCTM, bboxin, bboxout, 2);
   xm = (bboxout[0].x < bboxout[1].x) ? 0 : 1;
   ym = (bboxout[0].y < bboxout[1].y) ? 0 : 1;

   if (bboxout[xm].x < areastruct.width && bboxout[ym].y < areastruct.height &&
       bboxout[!xm].x > 0 && bboxout[!ym].y > 0) {

       for (strptr = drawlabel->string; *strptr != '\0'; strptr++) {

          /* make a character transformation matrix by translating to newpoint */

          UPushCTM();
          UPreMultCTM(DCTM, newpoint, tmpscale, 0);

          /* deal with in-line text format control escape sequences */

          if ((short)(*strptr) == TEXT_ESC) {
	     short control = (short)(*(++strptr));

             if (control >= FONT_START) {
	        if (control < FONT_START + FONTS) {
	           fstyle = (control - FONT_START) % 4;
	           ffont = ((control - FONT_START) >> 1) & 0xfe;
	        }
	        else fstyle = ffont = 0;	/* user-included fonts */
	        if (oldy == newpoint.y) {	/* set top-level font and style */
	           oldfont = ffont;
	           oldstyle = fstyle;
	        }
	        setlineattrib(fstyle);
	        fline = 0;  /* font change disables underline/overline */
             }
             else if (control == SUBSCRIPT) {
	        tmpscale *= SUBSCALE; 
	        newpoint.y -= (short)((TEXTHEIGHT >> 1) * tmpscale);
	        fline = 0;
             }
             else if (control == SUPERSCRIPT) {
	        tmpscale *= SUBSCALE;
	        newpoint.y += (short)(TEXTHEIGHT * tmpscale);
	        fline = 0;
             }
             else if (control == NORMALSCRIPT) {
	        tmpscale = 1.0;
	        ffont = oldfont;	/* revert to top-level font and style */
	        fstyle = oldstyle;
	        newpoint.y = oldy;
	        fline = 0;
             }
             else if (control == UNDERLINE) fline = 8;
             else if (control == OVERLINE) {
	        uchar *quickptr = strptr + 1;
	        static uchar biglittle[7] = {'b', 'd', 'f', 'h', 'k', 'l', 't'};
	        short lowerct = 0, i;

	        /* any control character stops the /overline */ 
	        for (; (*quickptr != TEXT_ESC) && (*quickptr != '\0'); quickptr++)
	           if (islower(*quickptr)) {
	              lowerct++;
	              for (i = 0; i < 7; i++) if (*quickptr == biglittle[i])
		         lowerct--;
	           }
	        fline = ((short)(quickptr - strptr - 1) == lowerct) ? 16 : 32;
             }
             else if (control == NOLINE) fline = 0;
             else if (control == BACKSPACE) {
	        if (strptr + 1 - (backct * 3) >= drawlabel->string) {
	           newpoint.x += UDrawChar(localdata, *(strptr + 1 - (backct * 3)), 
	                  (fstyle & 1) | ffont | 64) * tmpscale;
	        }
	        backct = ((short)(*(strptr + 1)) != TEXT_ESC || (short)(*(strptr + 2))
	    	!= BACKSPACE) ? 1 : backct + 1;
             }
          }
          else
	     newpoint.x += UDrawChar(localdata, *strptr, (fstyle & 1) | ffont | fline)
	    		* tmpscale;
   
          /* pop the character transformation matrix */

          UPopCTM();
      }
   }

   /* pop the string transformation matrix */

   UPopCTM();
}

short ULength(string, newscale, dostop)
  uchar *string;
  float	newscale;
  short dostop;
{
   short backct = 1;
   float oldscale, strscale, xtotal = 0.5;
   uchar *strptr;
   objinstptr *scaleobj;
   alphaptr *somebet;

   if (newscale > 0.0) strscale = UTopScale() * newscale;
   else strscale = 1.0;
     
   oldscale = strscale;

   for (strptr = string; *strptr != '\0'; strptr++) {
      if (dostop & ((short)(strptr - string) == textpos)) break;
      if ((short)(*strptr) == TEXT_ESC) {
	 short control = (short)(*(++strptr));	 

         if (control == SUPERSCRIPT || control == SUBSCRIPT) strscale *= SUBSCALE;
         else if (control == NORMALSCRIPT) strscale = oldscale;
         else if (control == BACKSPACE) {
	    if (strptr + 1 - (backct * 3) >= string) {
	       if (somebet == courierbet)
	          xtotal -= COURIER_WIDTH * strscale;
	       else
	          xtotal -= (float)((*(somebet + *(strptr + 1 - (backct * 3))))->width)
			 * strscale;
	    }
	    backct = (*(strptr + 1) != TEXT_ESC || *(strptr + 2) != BACKSPACE)
		 ? 1 : backct + 1;
         }
         else if (control >= FONT_START) {
	    if (control < FONT_START + FONTS) {
	       switch ((control - FONT_START) >> 2) {
	          case 1:  somebet = helveticabet;	break;
	          case 2:  somebet = courierbet;	break;
	          case 3:  somebet = symbolbet;		break;
	          default: somebet = alphabet;		break;
	       }
	    }
	    else somebet = alphabet;
	 }
      }
      else {
	 if (somebet == courierbet)
	    xtotal += COURIER_WIDTH * strscale;
	 else
	    xtotal += (float)((*(somebet + *strptr))->width) * strscale;
      }
   }
   return ((short)(xtotal - (HALFSPACE * oldscale)));
}

/*------------------------------------------------------------------------*/
/* simple routines for drawing and erasing labels */

undrawtext(settext)
  labelptr settext;
{
   XSetFunction(dpy, areastruct.gc, GXcopy);
   XSetForeground(dpy, areastruct.gc, BACKGROUND);
   UDrawString(areastruct.topobject, settext);
}

redrawtext(settext)
  labelptr settext;
{
   XTopSetForeground(settext->color);
   UDrawString(areastruct.topobject, settext);
}

/*------------------------------------------------------------------------*/
