/*
 * Copyright 1993 by Digital Equipment Corporation, Maynard, Massachusetts.
 * 
 * Permission to use, copy, modify, distribute, and sell this software and its 
 * documentation for any purpose is hereby granted without fee, provided that 
 * the above copyright notice appear in all copies and that both that 
 * copyright notice and this permission notice appear in supporting 
 * documentation, and that the name of Digital not be used in advertising or 
 * publicity pertaining to distribution of the software without specific, 
 * written prior permission.  Digital makes no representations about the 
 * suitability of this software for any purpose.  It is provided "as is" 
 * without express or implied warranty.
 * 
 * DIGITAL DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING 
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL 
 * DIGITAL BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY 
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN 
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF 
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <unistd.h>
#include <sys/file.h>
#include <fcntl.h>
#include <limits.h>
#include <AF/AFlib.h>
#include <AF/AFUtils.h>

#include "xbiff-proto.h"

#define BUFSIZE  2.0			/* buffer size, in seconds */
static int flushflag = 0;
static int bufsize;			/* buffer size, in seconds */

static int int_flag = 0;
static void (*old_handler) XBIFFPROTO((int));

static AFAudioConn *aud = (AFAudioConn *)NULL;
static int device = -2;

static void handle_int XBIFFPROTO((int));
static int FindDefaultDevice XBIFFPROTO((void));

static void handle_int(xxx)
int xxx;
{
  /* notify main loop of the requested exit */
  int_flag = 1;
  /* assure that we don't wait around */
  flushflag = 0;
  /* restore original handler, so that a second ^C will force an exit */
  signal(SIGINT, old_handler);
}

/*
 * Find a suitable default device (the first device not connected to the phone),
 *  returns device number.
 *
 * Returns -1 if no suitable device can be found.
 */
static int FindDefaultDevice()
{
        AFDeviceDescriptor *aDev;
        int     i;

        for(i=0; i<ANumberOfAudioDevices(aud); i++) {
                aDev = AAudioDeviceDescriptor(aud, i);
                if(aDev->inputsFromPhone == 0 && aDev->outputsToPhone == 0)
                        return i;
        }
        return -1;
}

int
AFplay(sound)
const char *sound;
{
  AFSetACAttributes attributes;
  const char *fullname;
  ATime t;
  ATime nact;
  AC ac;
  int fd;
  int nbytes;
  unsigned char *buf;
  int unitSize;		/* sample size, in bytes */
  int srate;			/* sample rate, per second */
  int units;			/* number of units */
  int nsamples;
  float toffset = 0.1;		/* seconds delay to start play */
  unsigned int channels;

  if (sound != (char *) NULL) {

    fullname = findSoundFile(sound);
    if (fullname == NULL) {
      fprintf(stderr, "can't find audio file \"%s\".\n", sound);
      return(1);
    }

    fd = open(fullname, O_RDONLY, 0);
    if (fd < 0) {
      fprintf(stderr, "can't open audio file \"%s\".\n", fullname);
      return(1);
    }
  } else
    fd = 0;

  if (aud == NULL && (aud = AFOpenAudioConn("")) == NULL) {
    fprintf(stderr, "can't open AudioFile connection.\n");
    close(fd);
    return(1);
  }

  /* set up audio context, find sample size and sample rate */
  if((device == -2) || (device >= aud->ndevices))
    device = FindDefaultDevice();

  attributes.preempt = Mix;
  attributes.start_timeout = 0;
  attributes.end_silence = 0;
  attributes.play_gain = 0;
  attributes.rec_gain =  0;
  attributes.type = aud->devices[device].playBufType;
  ac = AFCreateAC(aud, device, (ACPlayGain | ACEncodingType), 
		  &attributes);
  AFSync(aud, 0);	/* Make sure we confirm encoding type support. */
  srate = ac->device->playSampleFreq;
  channels = ac->device->playNchannels;
  unitSize = BytesPerUnit(attributes.type) * channels;
  bufsize = (srate / SampsPerUnit(attributes.type)) * BUFSIZE;

  /* allocate play buffer */
  if((buf = (unsigned char *)malloc(bufsize*unitSize)) == NULL) {
    fprintf(stderr, "Couldn't allocate play buffer\n");
    AFFreeAC(ac);
    close(fd);
    return(1);
  }

  if((nbytes = read(fd, buf, bufsize*unitSize)) <= 0)  {
    free(buf);
    AFFreeAC(ac);
    close(fd);
    return(0);
  }

  old_handler = signal(SIGINT, handle_int);

  t = AFGetTime(ac) + (toffset * srate);
  do {

    units = nbytes / unitSize;
    nsamples = units * SampsPerUnit(attributes.type);
    nact = AFPlaySamples(ac, t, units*unitSize, buf);
/*
    printf("time %d bytes %d units %d unitSize %d samps %d\n", t, nbytes,
	   units, unitSize, nsamples);
*/

    t += nsamples;
    /* AF holds samples from nact to t */
    if (int_flag) break;
  } while ( (nbytes = read(fd, buf, bufsize*unitSize)) > 0);

  if (int_flag) {
    AFSilence(attributes.type, buf, bufsize);
    attributes.preempt = 1;
    AFChangeACAttributes(ac, ACPreemption, &attributes);
    while (nact < t) {
      nsamples = bufsize * SampsPerUnit(attributes.type);
      (void) AFPlaySamples(ac, nact, bufsize*unitSize, buf);
      nact += nsamples;
    }
  } else if (flushflag) {
    while ((((int) AFGetTime(ac)) - ((int) t)) < 0) 
      sleep(1);
  }

  free(buf);
  AFFreeAC(ac);
  close(fd);
  return(0);
}

#ifdef APLAY_MAIN

/* needed by findSound() */
char *sound_path = NULL;

int main XBIFFPROTO((int, char *[]));

int
main(int argc, char **argv)
{
  int i;

  /* Parse the command line */
  for ( i = 1; i < argc; i++ )
    AFplay(argv[i]);

  return 0;
}
#endif /* APLAY_MAIN */
