/* subpanel.c -- xa sub panel creation and event procedures */

/* $Id: subpanel.c,v 1.10 1996/04/19 22:21:42 richards Exp $ */

/*
 * $Log: subpanel.c,v $
 * Revision 1.10  1996/04/19 22:21:42  richards
 * ApplyColorScaling() now takes an argument to fix RANDOM behaviour
 * other minor changes
 *
 * Revision 1.9  1996/03/16 17:09:26  richards
 * renamed xrastool to xa
 * fixed indenting
 *
 * Revision 1.8  1996/03/14 22:11:27  richards
 * added SCALABLE and LOCKABLE macros
 * rationalized prototypes
 * changed behaviour of CUTOFF scaling option
 *
 * Revision 1.7  1996/03/08 21:03:53  richards
 * minor changes
 *
 * Revision 1.6  1996/02/08 01:45:04  richards
 * added color slider lock
 * replaced toggle_slider_scales_exposure() with
 *    toggle_lock_and_scales_exposure()
 * fixed inconsistency with color slider scale for GAMMA scaling
 *
 * Revision 1.5  1996/01/30 20:28:32  richards
 * incorporated COLOR_T (unsigned char) and COLORCELL_T (RGB structure)
 * added last_scaling in ApplyColorScaling() to allow locked random colors
 * replaced InstallCms() with StoreColors() in color.c
 * rationalized behaviour of grayscale color sliders
 * incorporated color macros
 *
 * Revision 1.4  1996/01/27 22:40:37  richards
 * changed some comment formats
 *
 * Revision 1.3  1996/01/25 15:49:57  richards
 * implemented gamma correction
 *
 * Revision 1.2  1996/01/21 21:35:28  richards
 * prototyping, fixed ATTR_LIST problem on Suns, NewCmap functionality
 *
 * Revision 1.1  1996/01/18 16:56:55  richards
 * Initial revision
 *
 */

#include "xa.h"
#include <math.h> /* for GAMMA calculation */

#define MAX_COLOR_INTENSITY (MAX_NUM_COLORS - 1)

#define NUM_CONTOURS 10

#define SCALABLE(scaling)\
  ((scaling) == COLOR || (scaling) == GAMMA || (scaling) == CUTOFF)

#define LOCKABLE(scaling)\
  ((scaling) == COLOR || (scaling) == GAMMA || (scaling) == CUTOFF)

/* Global variables */

Frame SubFrame;
Panel SubPanel;
COLORCELL_T ScaledColors[MAX_NUM_COLORS];

/* Local variables */

static Panel_item options_item, backdrops_item, slider_lock_item, scaling_item,
  red_slider, green_slider, blue_slider, red_slider_scale, green_slider_scale,
  blue_slider_scale;
static Bool slider_lock = FALSE;
static int num_colors;
static COLORCELL_T *colors;

/* Bitmaps */

#include "bitmaps/lock.xbm"

/* Contour colors, in order of increasing brightness */

static COLORCELL_T contour_colors[NUM_CONTOURS] = {
  {  0,   0,   0}, /* black */
  {  0,   0, 255}, /* blue */
  {139,   0, 139}, /* magenta */
  { 93,  71, 139}, /* medium purple */
  {  0, 154, 205}, /* deep sky blue */
  { 50, 205,  50}, /* lime green */
  {238, 154,  73}, /* medium aquamarine */
  {205, 201, 165}, /* lemon */
  {255, 255,   0}, /* yellow */
  {255, 255, 255}  /* white */
};

/* Local functions */

static void set_option(Panel_item item, int value, Event *event);
static void set_backdrop(Panel_item item, int value, Event *event);
static void set_scaling(Panel_item item, int value, Event *event);
static void toggle_slider_lock(Panel_item item, int value, Event *event);
static void change_color_slider(Panel_item item, int value, Event *event);
static void change_color_slider_scale(Panel_item item, int value,
				      Event *event);
static void toggle_lock_and_scales_exposure(Bool flag);
static void set_color_sliders(Panel_item item, Event *event);
static void draw_bw_scale(void);
static void draw_gray_scale(void);
static void draw_color_scale(void);
static void draw_gamma_scale(void);
static void draw_cutoff_scale(void);
static void draw_contour_scale(void);
static void draw_random_scale(void);

/*** END OF PREAMBLE ***/

void MakeSubWindow(void)
{
  /* Creates sub panel */

  Server_image lock_image;
  Rect *rect;
  Attr_avlist attr_list;

  SubFrame = (Frame)
    xv_create(BaseFrame, FRAME_CMD,
	      FRAME_CMD_DEFAULT_PIN_STATE, FRAME_CMD_PIN_IN,
	      XV_X, DFLT_SUB_FRAME_XV_X,
	      XV_Y, DFLT_SUB_FRAME_XV_Y,
	      FRAME_LABEL, "xa: Sub Frame",
	      NULL);

  SubPanel = (Panel) xv_get(SubFrame, FRAME_CMD_PANEL);

  options_item = (Panel_item)
    xv_create(SubPanel, PANEL_TOGGLE,
	      PANEL_CHOICE_NCOLS, 3,
	      PANEL_CHOICE_STRINGS, "No scrolling", "No resizing", "No moving",
	        "No scaling", "No updates", "No backdrops", "Lock colors",
	        "Centering", "Live cursor", NULL,
	      PANEL_VALUE, Options,
	      PANEL_NOTIFY_PROC, set_option,
	      NULL);

  MakeBackdropImages();

  backdrops_item = (Panel_item)
    xv_create(SubPanel, PANEL_CHOICE,
	      PANEL_NEXT_ROW, -1,
	      PANEL_LABEL_STRING, "Backdrops:",
	      PANEL_CHOICE_IMAGES, Backdrop[0], Backdrop[1], Backdrop[2],
	        Backdrop[3], Backdrop[4], Backdrop[5], Backdrop[6],
	        Backdrop[7], Backdrop[8], NULL,
	      PANEL_VALUE, BackdropIndex,
	      PANEL_NOTIFY_PROC, set_backdrop,
	      NULL);

  scaling_item = (Panel_item)
    xv_create(SubPanel, PANEL_CHOICE,
	      PANEL_NEXT_ROW, -1,
	      PANEL_LABEL_STRING, "Scaling:",
	      PANEL_CHOICE_NCOLS, 3,
	      PANEL_CHOICE_STRINGS, "BW", "Gray", "Color", "Gamma", "Cutoff",
	        "Contour", "Random", "", "", NULL,
	      PANEL_VALUE, Scaling,
	      PANEL_NOTIFY_PROC, set_scaling,
	      NULL);

  lock_image = (Server_image)
    xv_create(XV_NULL, SERVER_IMAGE,
	      SERVER_IMAGE_X_BITS, lock_bits,
	      XV_WIDTH, lock_width,
	      XV_HEIGHT, lock_height,
	      NULL);

  rect = (Rect *) xv_get(scaling_item, XV_RECT); /* For nicer positioning */

  slider_lock_item = (Panel_item)
    xv_create(SubPanel, PANEL_TOGGLE,
	      PANEL_CHOICE_IMAGES, lock_image, NULL,
	      PANEL_VALUE, slider_lock,
	      PANEL_NOTIFY_PROC, toggle_slider_lock,
	      XV_X, rect_right(rect) + 24,
	      XV_Y, rect_bottom(rect) - 46,
	      NULL);

  attr_list =
    attr_create_list(PANEL_NEXT_ROW, -1,
		     PANEL_TICKS, 5,
		     PANEL_MIN_VALUE, 0,
		     PANEL_MAX_VALUE, 100,
		     PANEL_SLIDER_WIDTH, 149, /* Line up w/main opts buttons */
		     PANEL_NOTIFY_PROC, change_color_slider,
		     PANEL_NOTIFY_LEVEL, PANEL_ALL,
		     NULL);

  /* NOTE: slider values are set during initialization (Cf. Initialize()) */

  red_slider = (Panel_item)
    xv_create(SubPanel, PANEL_SLIDER,
	      ATTR_LIST, attr_list, /* ATTR_LIST must ALWAYS come first! */
	      PANEL_LABEL_STRING, "R:",
	      PANEL_MAX_VALUE, 100,
	      NULL);

  red_slider_scale = (Panel_item)
    xv_create(SubPanel, PANEL_CHOICE,
	      PANEL_CHOICE_STRINGS, "x", "/", NULL,
	      PANEL_VALUE, RedSliderScale,
	      PANEL_NOTIFY_PROC, change_color_slider_scale,
	      NULL);

  green_slider = (Panel_item)
    xv_create(SubPanel, PANEL_SLIDER,
	      ATTR_LIST, attr_list,
	      PANEL_LABEL_STRING, "G:",
	      NULL);

  green_slider_scale = (Panel_item)
    xv_create(SubPanel, PANEL_CHOICE,
	      PANEL_CHOICE_STRINGS, "x", "/", NULL,
	      PANEL_VALUE, GreenSliderScale,
	      PANEL_NOTIFY_PROC, change_color_slider_scale,
	      NULL);

  blue_slider = (Panel_item)
    xv_create(SubPanel, PANEL_SLIDER,
	      ATTR_LIST, attr_list,
	      PANEL_LABEL_STRING, "B:",
	      NULL);

  blue_slider_scale = (Panel_item)
    xv_create(SubPanel, PANEL_CHOICE,
	      PANEL_CHOICE_STRINGS, "x", "/", NULL,
	      PANEL_VALUE, BlueSliderScale,
	      PANEL_NOTIFY_PROC, change_color_slider_scale,
	      NULL);

  window_fit(SubPanel);
  window_fit(SubFrame);
}

void UpdateOptionsItem(void)
{
  (void) xv_set(options_item, PANEL_VALUE, Options, NULL);
}

void ToggleScalingExposure(Bool flag)
{
  (void) xv_set(scaling_item, PANEL_INACTIVE, flag, NULL);
  (void) xv_set(red_slider, PANEL_INACTIVE, flag, NULL);
  (void) xv_set(green_slider, PANEL_INACTIVE, flag, NULL);
  (void) xv_set(blue_slider, PANEL_INACTIVE, flag, NULL);

  toggle_lock_and_scales_exposure(flag ? TRUE : !SCALABLE(Scaling));
}

void ToggleBackdropsExposure(Bool flag)
{
  (void) xv_set(backdrops_item, PANEL_INACTIVE, flag, NULL);
}

void ApplyColorScaling(Bool sliders_changed)
{
  /*
   * Applies color scaling to current or locked colors. "sliders_changed"
   * should be TRUE if calling routine changed slider values.
   *
   */

  if (Options & LOCK_COLORS) {
    num_colors = NumLockedColors;
    colors = LockedColors;
  }
  else {
    num_colors = Image[Current]->num_colors;
    colors = Image[Current]->colors;
  }

  switch (Scaling) {
  case BW:
    draw_bw_scale();
    break;
  case GRAY:
    draw_gray_scale();
    break;
  case COLOR:
    draw_color_scale();
    break;
  case GAMMA:
    draw_gamma_scale();
    break;
  case CUTOFF:
    draw_cutoff_scale();
    break;
  case CONTOUR:
    draw_contour_scale();
    break;
  case RANDOM:
    if (!sliders_changed) /* Only generate random map if sliders changed */
      return;
    draw_random_scale();
  }

  StoreColors(ScaledColors, num_colors);
}

void SetColorSliders(void)
{
  toggle_lock_and_scales_exposure((Options & NO_SCALING) ||
				  !SCALABLE(Scaling));

  set_color_sliders((Panel_item) NULL, (Event *) NULL);
}

static void set_option(Panel_item item, int value, Event *event)/*ARGSUSED*/
{
  /* Sets option, overriding unlock colors request if unique colormap */

  if (((Options ^ value) & LOCK_COLORS) && (OneCmap || NewCmap))
    UpdateOptionsItem();
  else
    SetOptions(Options ^ value);
}

static void set_backdrop(Panel_item item, int value, Event *event)/*ARGSUSED*/
{
  BackdropIndex = value;

  SetBackdrop();
}

static void set_scaling(Panel_item item, int value, Event *event)/*ARGSUSED*/
{
  /* Sets colors scaling option */

  Scaling = value;

  if (LOCKABLE(Scaling)) { /* Set lock default */
    slider_lock = (Scaling == GAMMA || Scaling == CUTOFF);
    (void) xv_set(slider_lock_item, PANEL_VALUE, slider_lock, NULL);
  }

  SetColorSliderDefaults();

  SetScaling();
}

static void toggle_slider_lock(Panel_item item, int value, Event *event)
/*ARGSUSED*/
{
  /* Toggles color slider locking (only affects certain scaling options) */

  if (!LOCKABLE(Scaling))
    return;

  slider_lock = (Bool) value;
}

static void change_color_slider(Panel_item item, int value, Event *event)
{
  /* Changes color sliders depending on current scaling option */

  switch (Scaling) {
  case GRAY:
    if (item == red_slider) {
      if (GreenSliderValue > BlueSliderValue)
	GreenSliderValue = 100 - value - BlueSliderValue;
      else if (GreenSliderValue < BlueSliderValue)
	BlueSliderValue = 100 - value - GreenSliderValue;
      else
	GreenSliderValue = BlueSliderValue = (100 - value) / 2;
      RedSliderValue = value;
    }
    else if (item == green_slider) {
      if (RedSliderValue > BlueSliderValue)
	RedSliderValue = 100 - value - BlueSliderValue;
      else if (RedSliderValue < BlueSliderValue)
	BlueSliderValue = 100 - value - RedSliderValue;
      else
	RedSliderValue = BlueSliderValue = (100 - value) / 2;
      GreenSliderValue = value;
    }
    else {
      if (RedSliderValue > GreenSliderValue)
	RedSliderValue = 100 - value - GreenSliderValue;
      else if (RedSliderValue < GreenSliderValue)
	GreenSliderValue = 100 - value - RedSliderValue;
      else
	RedSliderValue = GreenSliderValue = (100 - value) / 2;
      BlueSliderValue = value;
    }
    set_color_sliders(item, event);
    break;
  case COLOR:
  case GAMMA:
  case CUTOFF:
    if (item == red_slider) {
      if (slider_lock) {
	int dc = value - RedSliderValue;
	GreenSliderValue += dc;
	BlueSliderValue  += dc;
      }
      RedSliderValue = value;
    }
    else if (item == green_slider) {
      if (slider_lock) {
	int dc = value - GreenSliderValue;
	RedSliderValue  += dc;
	BlueSliderValue += dc;
      }
      GreenSliderValue = value;
    }
    else {
      if (slider_lock) {
	int dc = value - BlueSliderValue;
	RedSliderValue   += dc;
	GreenSliderValue += dc;
      }
      BlueSliderValue = value;
    }
    if (slider_lock)
      set_color_sliders(item, event);
    break;
  case BW:
  case CONTOUR:
  case RANDOM:
    RedSliderValue = GreenSliderValue = BlueSliderValue = value;
    set_color_sliders(item, event);
    break;
  default:
    return;
  }

  if (!UNDEF_IMAGE)
    ApplyColorScaling(TRUE);
}

static void change_color_slider_scale(Panel_item item, int value, Event *event)
/*ARGSUSED*/
{
  /* Changes color slider scale (only affects certain scaling options) */

  if (!SCALABLE(Scaling))
    return;

  if (item == red_slider_scale) {
    RedSliderScale = value;
    if (Scaling == GAMMA) {
      if (value) {
	RedSliderValue *= 10;
	RedSliderValue = MIN(RedSliderValue, 100);
      }
      else
	RedSliderValue /= 10;
    }
  }
  else if (item == green_slider_scale) {
    GreenSliderScale = value;
    if (Scaling == GAMMA) {
      if (value) {
	GreenSliderValue *= 10;
	GreenSliderValue = MIN(GreenSliderValue, 100);
      }
      else
	GreenSliderValue /= 10;
    }
  }
  else {
    BlueSliderScale = value;
    if (Scaling == GAMMA) {
      if (value) {
	BlueSliderValue *= 10;
	BlueSliderValue = MIN(BlueSliderValue, 100);
      }
      else
	BlueSliderValue /= 10;
    }
  }

  if (Scaling == GAMMA)
    set_color_sliders((Panel_item) NULL, (Event *) NULL);

  if (!UNDEF_IMAGE)
    ApplyColorScaling(TRUE);
}

static void toggle_lock_and_scales_exposure(Bool flag)
{
  (void) xv_set(slider_lock_item, PANEL_INACTIVE, flag, NULL);
  (void) xv_set(red_slider_scale, PANEL_INACTIVE, flag, NULL);
  (void) xv_set(green_slider_scale, PANEL_INACTIVE, flag, NULL);
  (void) xv_set(blue_slider_scale, PANEL_INACTIVE, flag, NULL);
}

static void set_color_sliders(Panel_item item, Event *event)
{
  /* Loads current values into color sliders */

  if (LIVE_SLIDERS || !event || event_is_ascii(event) ||
      (event_is_button(event) && !event_is_down(event))) {
    if (item != red_slider)
      (void) xv_set(red_slider, PANEL_VALUE, RedSliderValue, NULL);
    if (item != green_slider)
      (void) xv_set(green_slider, PANEL_VALUE, GreenSliderValue, NULL);
    if (item != blue_slider)
      (void) xv_set(blue_slider, PANEL_VALUE, BlueSliderValue, NULL);
  }
}

static void draw_bw_scale(void)
{
  int i;
  u_long value;

  value = (100 - RedSliderValue) * MAX_COLOR_INTENSITY / 100;
  value = 3 * SQ(value);

  for (i = 0; i < num_colors; i++) {
    if (COLOR_INTEN(colors[i]) > value) /* No semi-colon before "else" */
      SET_GRAY(ScaledColors[i], MAX_COLOR_INTENSITY)
    else
      SET_GRAY(ScaledColors[i], 0);
  }
}

static void draw_gray_scale(void)
{
  int i;
  u_long value;

  for (i = 0; i < num_colors; i++) {
    value = RedSliderValue * colors[i].r + GreenSliderValue * colors[i].g +
      BlueSliderValue * colors[i].b;
    SET_GRAY(ScaledColors[i], value / 100);
  }
}

#define SCALE(index, slider, div)\
  (div ? (slider ? (index) * 100 / (slider) : MAX_COLOR_INTENSITY) :\
         (index) * (slider) / 100)

static void draw_color_scale(void)
{
  int i;
  u_long r, g, b;

  for (i = 0; i < num_colors; i++) {
    r = SCALE(colors[i].r, RedSliderValue,   RedSliderScale);
    g = SCALE(colors[i].g, GreenSliderValue, GreenSliderScale);
    b = SCALE(colors[i].b, BlueSliderValue,  BlueSliderScale);
    SET_COLOR(ScaledColors[i], MIN(r, MAX_COLOR_INTENSITY),
	      MIN(g, MAX_COLOR_INTENSITY), MIN(b, MAX_COLOR_INTENSITY));
  }
}

#undef SCALE

#define MAXI MAX_COLOR_INTENSITY

#define SCALE(index, slider, div)\
  (slider ?\
    pow((double) index / MAXI, (div ? 100.0 : 10.0) / slider) * MAXI : 0)

static void draw_gamma_scale(void)
{
  /* NOTE: gamma value is one tenth slider value */

  int i;

  for (i = 0; i < num_colors; i++)
    SET_COLOR(ScaledColors[i],
	      SCALE(colors[i].r, RedSliderValue,   RedSliderScale),
	      SCALE(colors[i].g, GreenSliderValue, GreenSliderScale),
	      SCALE(colors[i].b, BlueSliderValue,  BlueSliderScale));
}

#undef SCALE
#undef MAXI

#define SCALE(index, slider, lo)\
  (index < (slider * 256) / 100 ? (lo ? 0 : index) : (lo ? index : 0))

static void draw_cutoff_scale(void)
{
  int i;

  for (i = 0; i < num_colors; i++)
    SET_COLOR(ScaledColors[i],
	      SCALE(colors[i].r, RedSliderValue,   RedSliderScale),
	      SCALE(colors[i].g, GreenSliderValue, GreenSliderScale),
	      SCALE(colors[i].b, BlueSliderValue,  BlueSliderScale));
}

#undef SCALE

#define MAX_CONTOUR_VAL 195076 /* 3 * 255**2 + 1 */

static void draw_contour_scale(void)
{
  int i, num_contours, contour;
  u_long bin_size;

  num_contours = MAX(RedSliderValue, 1);

  bin_size = MAX_CONTOUR_VAL / num_contours;

  for (i = 0; i < num_colors; i++) {
    contour = (COLOR_INTEN(colors[i]) / bin_size) % NUM_CONTOURS;
    COPY_COLOR(contour_colors[contour], ScaledColors[i]);
  }
}

#undef MAX_CONTOUR_VAL

static void draw_random_scale(void)
{
  int i;

  for (i = 0; i < num_colors; i++)
    SET_COLOR(ScaledColors[i], lrand48(), lrand48(), lrand48());
}

/*** subpanel.c ***/
