/*
 * Copyright (c) 1994 by Gregory P. Ward.
 * All rights reserved.
 * 
 * This file is part of the MNI front end of the Berkeley MPEG decoder.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE
 * UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER
 * IS ON AN "AS IS" BASIS, AND THE AUTHOR HAS NO OBLIGATION TO PROVIDE
 * MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.  
 */
/*
 * Portions of this software Copyright (c) 1995 Brown University.
 * All rights reserved.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement
 * is hereby granted, provided that the above copyright notice and the
 * following two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT 
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF BROWN
 * UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * BROWN UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"
 * BASIS, AND BROWN UNIVERSITY HAS NO OBLIGATION TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

/* ----------------------------- MNI Header -----------------------------------
@NAME       : wrapper.c
@INPUT      : 
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Functions and variables used in the interface between
              user applications and the Berkely MPEG decoder.  This
              file essentially comprises the MNI front end to the
	      Berkeley decoder; applications should NOT have access to any of 
              these.  The functions in this file are the only ones to
              which users of the MPEG library have access; all other
              functions in the library are private.
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 
@MODIFIED   : 
---------------------------------------------------------------------------- */

#include <stdio.h>
#include <string.h> /* for memcpy() */
#include <errno.h>
#include "video.h"
#include "proto.h"
#include "util.h"
#include "dither.h"
#include "mpeg.h"

#define BUF_LENGTH 100000


/*
 * Global variables shared between this file and the core MPEG decoder
 * (but nowhere else!)  - these are the only globals!
 */
int quietFlag=1, loopFlag=1,qualityFlag=0, requireKeypressFlag=0;
int framerate=0,gammaCorrectFlag=0, chromaCorrectFlag=0, noDisplayFlag=0;
int partialFlag=0,startFrame=-1, endFrame=-1;
double gammaCorrect = 1.0, chromaCorrect=1.0;

static int numMovies=0;

/* Prototypes for functions local to this file: */

void GetMPEGInfo (ImageDesc *Info);

int dprintf (char *format, ...) {}


/* ----------------------------- MNI Header -----------------------------------
@NAME       : OpenMPEG
@INPUT      : MPEGfile - pointer to a stream opened for reading, positioned
                         at the beginning of an MPEG stream
	      ImgInfo  - pointer to an ImageDesc structure which will have
	                 information such as frame height, width, depth
			 and size (total bytes per frame) put in it.
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Creates and initializes the variables needed to start 
              reading/decoding an MPEG stream.  

              This function is part of the MNI front end to the Berkeley
	      MPEG decoder, adapted from the original Berkeley code.
@METHOD     : 
@GLOBALS    : LUM_RANGE, CR_RANGE, CB_RANGE
	      lum_values, cr_values, cb_values
@CALLS      : GetMPEGInfo()
              init_tables()
              InitDither()
@CREATED    : 94/6/16, Greg Ward (adapted from main() in the original
              Berkeley source)
@MODIFIED   : 
---------------------------------------------------------------------------- */
Boolean OpenMPEG (FILE *MPEGfile, ImageDesc *ImgInfo)
{
   VidStream *vid_stream;
   /* 
    * First reinitialize these globals in case we're opening a second
    * (or more) file - thanks to Loren Holding (art054@cs.brown.edu)
    * for the patch
    */

    /* unneeded - done in NewVidStream or ResetVidStream */

   /* 
    * Create the video stream and read the first chunk to get movie
    * stats -- width and height in particular.
    */

   ImgInfo->vid_stream = NULL; /* Hack -- DCR 96-03-14 */

   if (ImgInfo->vid_stream==NULL) {
     ImgInfo->vid_stream = vid_stream =NewVidStream((unsigned int) BUF_LENGTH);
   } else {
     vid_stream=ImgInfo->vid_stream;
     clear_data_stream(vid_stream);
     ResetVidStream(vid_stream);
   }
   vid_stream->input = MPEGfile;
   vid_stream->filename = "mniwrapper";
#ifndef DISABLE_DITHER
   vid_stream->matched_depth = 8;
   vid_stream->ditherType = ORDERED_DITHER; /* default */
#else
   vid_stream->matched_depth = 24;
   vid_stream->ditherType = FULL_COLOR_DITHER; /* default */
#endif

   numMovies = 1; /*numMovies++;*/ /* Hack -- DCR 96-03-26 */
   vid_stream->realTimeStart = 0; /* Hack -- DCR 96-04-12 */

   if (numMovies==1) {
      init_tables();		/* initialize decoding stuff */
      InitCrop();

      InitDither(ImgInfo);	/* initializes dithering structures and */
				/* colormap (i.e. this is where we do */
				/* all dither-specific stuff) */
   }

   if (mpegVidRsrc(0, ImgInfo->vid_stream, 1, NULL )==NULL) {
      return FALSE;
   } else {
     GetMPEGInfo(ImgInfo);
     return TRUE;
   }
}     /* OpenMPEG () */


/* ----------------------------- MNI Header -----------------------------------
@NAME       : CloseMPEG
@INPUT      : (none)
@OUTPUT     : (none)
@RETURNS    : (void)
@DESCRIPTION: Frees up some of the memory allocated by OpenMPEG() (some
              is not freed because the Berkeley code doesn't take into 
	      account the fact that somebody might want to, say, free
	      up the memory it allocates... someday, I'll probably have
	      to hack into it to fix that, but not today thanks.)
@METHOD     : 
@GLOBALS    : lum_values
	      cr_values
	      cb_values
@CALLS      : DestroyVidStream
@CREATED    : 94/6/27, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
void CloseMPEG (ImageDesc *ImgInfo)
{
   DestroyVidStream(ImgInfo->vid_stream, NULL);
   free (lum_values);
   free (cr_values);
   free (cb_values);
}



/* ----------------------------- MNI Header -----------------------------------
@NAME       : RewindMPEG
@INPUT      : MPEGfile - the input stream where the MPEG's coming from
              Image    - image descriptor (just passed to OpenMPEG ())
@OUTPUT     : (none)
@RETURNS    : (void)
@DESCRIPTION: Resets things so that the caller can start reading the MPEG
              stream from the start again.   Note that the caller does NOT
	      need to call OpenMPEG() again -- after a call to RewindMPEG(),
	      the next call to GetMPEGFrame() will return the first frame
	      of the MPEG.
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/7/20, Greg Ward
@MODIFIED   : 
@COMMENTS   : 
---------------------------------------------------------------------------- */
void RewindMPEG (FILE *MPEGfile, ImageDesc *Image)
{
   rewind (MPEGfile);

   OpenMPEG (MPEGfile, Image);
}



/* ----------------------------- MNI Header -----------------------------------
@NAME       : GetMPEGInfo
@INPUT      : vid_stream - a video stream that is open and has had at 
                           least one call to mpegVidRsrc() performed on it
              Info       - a pointer to an ImageDesc struct in the caller's
	                   space (i.e., the argument to OpenMPEG()) where
			   the image information will be copied
@OUTPUT     : 
@RETURNS    : (void)
@DESCRIPTION: From the video stream, determines the width, height, pixel
              size and depth (in bits) and total image size (in bytes)
	      for an MPEG stream.  Sets the fields in the structure
	      pointed to by the Info pointer.
@METHOD     : 
@GLOBALS    :
@CALLS      : 
@CREATED    : 94/6/17, Greg Ward: based on code from ExecuteDisplay() in the
              original Berkeley source
@MODIFIED   : 
---------------------------------------------------------------------------- */
void GetMPEGInfo (ImageDesc *Info)
{
#ifndef DISABLE_DITHER
   switch (Info->vid_stream->ditherType)
   {
      case Twox2_DITHER:
      {
	 Info->Height = Info->vid_stream->mb_height * 32;
	 Info->Width = Info->vid_stream->mb_width * 32; 
	 Info->Depth = 8;
	 Info->PixelSize = 8;
	 Info->BitmapPad = 8;
	 break;
      } 
      case FULL_COLOR_DITHER:
      {
#endif
	 Info->Height = Info->vid_stream->mb_height * 16;
	 Info->Width = Info->vid_stream->mb_width * 16; 
	 Info->Depth = 24;
	 Info->PixelSize = 32;
	 Info->BitmapPad = 32;
#ifndef DISABLE_DITHER
	 break;
      } 
      case MONO_DITHER:
      case MONO_THRESHOLD:
      {
	 Info->Height = Info->vid_stream->mb_height * 16;
	 Info->Width = Info->vid_stream->mb_width * 16; 
	 Info->Depth = 1;
	 Info->PixelSize = 1;
	 Info->BitmapPad = 8;
	 break;
      } 
      default:			/* including GRAY_DITHER and ORDERED_DITHER */
      {
	 Info->Height = Info->vid_stream->mb_height * 16;
	 Info->Width = Info->vid_stream->mb_width * 16; 
	 Info->Depth = 8;
	 Info->PixelSize = 8;
	 Info->BitmapPad = 8;
	 break;
      }
   }     /* switch on ditherType */
#endif

   Info->Size = (Info->Height*Info->Width*Info->PixelSize) / 8;
   Info->PictureRate = Info->vid_stream->picture_rate;
   Info->BitRate = Info->vid_stream->bit_rate;

}     /* GetMPEGInfo () */



/* ----------------------------- MNI Header -----------------------------------
@NAME       : SetMPEGOption
@INPUT      : Option - which option to set
              Value  - what to set it to
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Set an MPEG option.  The options are all assigned intelligent
              defaults when they are created (as global variables), so 
	      calling SetMPEGOption is optional (as you might expect 
	      from the name).  Whether SetMPEGOption() is called before 
	      or after OpenMPEG() is important, but it depends on which
	      option you're setting.  In particular, the dithering type
	      and luminance/chromaticity ranges must be set before
	      OpenMPEG(); but (unless your code is more clever than it
	      needs to be), the colourmap indeces probably won't be set
	      until after OpenMPEG().  RTFM for explanations of what the
	      individual options do.

	      The currently available options are:
                 MPEG_DITHER
		 MPEG_LUM_RANGE
		 MPEG_CR_RANGE
		 MPEG_CB_RANGE
		 MPEG_CMAP_INDEX

@METHOD     : 
@GLOBALS    : Depending on the value of Option, sets one of the MPEG
              decoders global variables:
		 LUM_RANGE
		 CR_RANGE
		 CB_RANGE
@CALLS      : 
@CREATED    : 94/6/17, Greg Ward.
@MODIFIED   : 95/3/18, GW: added MPEG_CMAP_INDEX option.
---------------------------------------------------------------------------- */
void SetMPEGOption (ImageDesc *Info, MPEGOptionEnum Option, int Value)
{
   switch (Option)
   {
      case MPEG_DITHER:    
#ifndef DISABLE_DITHER
	 Info->vid_stream->ditherType = (DitherEnum) Value;
#endif
	 break;
      case MPEG_LUM_RANGE: LUM_RANGE = Value; break;
      case MPEG_CR_RANGE:  CR_RANGE = Value; break;
      case MPEG_CB_RANGE:  CB_RANGE = Value; break;
#ifndef DISABLE_DITHER
      case MPEG_CMAP_INDEX:
      {
	 int            i;
	 unsigned char *cmap_index;

	 cmap_index = (unsigned char *) &Value;
	 for (i = 0; i < Info->ColormapSize; i++)
	 {
	    pixel[i] = cmap_index[i];
	 }
	 break;
      }
#endif
	    
   }
}     /* SetMPEGOption () */



/* ----------------------------- MNI Header -----------------------------------
@NAME       : GetMPEGFrame
@INPUT      : 
@OUTPUT     : Frame - the image data, converted to RGB space, is
              copied to the area pointed to by Frame.  There must be
              enough room for the entire image; the ImageDesc
              structure returned by OpenMPEG() will tell you just how
              much memory this is.  The format of the data depends on
              the dithering type; for full colour dithering, there are
              four bytes per pixel: red, green, blue, and unused.
              (Perfect for passing to lrectwrite() or XPutImage().)
@RETURNS    : TRUE if there are still frames left to decode
              FALSE if we have just decoded the last frame
@DESCRIPTION: Part of the MNI front end to the Berkeley MPEG decoder.  
              Essentially reads, decodes, and converts to RGB space the
	      next frame in the MPEG stream opened with OpenMPEG().
@METHOD     : 
@GLOBALS    : 
@CALLS      : mpegVidRsrc ()
@CREATED    : 94/6/16, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
Boolean GetMPEGFrame (ImageDesc *image, char *Frame)
{
   Boolean   MovieDone = FALSE;
   int frames;
   char *CurrentImage;

   dprintf ("GetMPEGFrame: starting or continuing movie\n");

   frames=image->vid_stream->totNumFrames;
   while (!MovieDone && (frames==image->vid_stream->totNumFrames))
   {
      mpegVidRsrc(0,image->vid_stream,0, NULL);
      MovieDone = image->vid_stream->film_has_ended;
   }

   CurrentImage=(char *) image->vid_stream->current->display;

   dprintf ("\nGetMPEGFrame: just received a finished frame: "
	    "copying from %08X to %08X\n", CurrentImage, Frame);

   memcpy (Frame, CurrentImage, image->Size);
   return (!MovieDone);

}     /* GetMPEGFrame () */

#ifndef DISABLE_DITHER
/* ----------------------------- MNI Header -----------------------------------
@NAME       : DoDitherImage
@INPUT      : l, Cr, Cb - pointers to the luminance, Cr, and Cb planes
              disp - ?
              h, w - height and width of image (?)
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Called when image needs to be dithered. Selects correct
              dither routine based on info in ditherType.
@METHOD     : 
@GLOBALS    : ditherType
@CALLS      : One of the following, depending on the value of ditherType:
                 HybridDitherImage       (hybrid.c)
		 HybridErrorDitherImage  (hybriderr.c)
		 FS2FastDitherImage      (fs2fast.c)
		 FS2DitherImage          (fs2.c)
		 FS4DitherImage          (fs4.c)
		 Twox2DitherImage        (2x2.c)
		 ColorDitherImage        (16bit.c)
		 GrayDitherImage         (gray.c)
		 OrderedDitherImage      (ordered.c)
		 MonoDitherImage         (mono.c)
		 MonoThresholdImage      (mono.c)
		 Ordered2DitherImage     (ordered2.c)
		 MBOrderedDitherImage    (mb_ordered.c)
@CREATED    : (taken from the original Berkeley code)
@MODIFIED   : 
---------------------------------------------------------------------------- */
void
DoDitherImage(vid_stream)
VidStream *vid_stream;
{
 unsigned char *l=vid_stream->current->luminance,
               *Cr=vid_stream->current->Cr,
               *Cb=vid_stream->current->Cb,
               *disp=vid_stream->current->display;
 int h=(int) vid_stream->mb_height * 16;
 int w=(int) vid_stream->mb_width * 16;
 int ditherType=vid_stream->ditherType;
 int matched_depth=vid_stream->matched_depth;

  switch(ditherType) {
  case HYBRID_DITHER:
    HybridDitherImage(l, Cr, Cb, disp, h, w);
    break;

  case HYBRID2_DITHER:
    HybridErrorDitherImage(l, Cr, Cb, disp, h, w);
    break;

  case FS2FAST_DITHER:
    FS2FastDitherImage(l, Cr, Cb, disp, h, w);
    break;

  case FS2_DITHER:
    FS2DitherImage(l, Cr, Cb, disp, h, w);
    break;

  case FS4_DITHER:
    FS4DitherImage(l, Cr, Cb, disp, h, w);
    break;

  case Twox2_DITHER:
    Twox2DitherImage(l, Cr, Cb, disp, h, w);
    break;

  case FULL_COLOR_DITHER:
    Color32DitherImage(l, Cr, Cb, disp, h, w);
    break;

  case GRAY_DITHER:
    GrayDitherImage(l, Cr, Cb, disp, h, w);
    break;

  case NO_DITHER:
    break;

  case PPM_DITHER:
    Color32DitherImage(l, Cr, Cb, disp, h, w);
    break;

  case ORDERED_DITHER:
    OrderedDitherImage(l, Cr, Cb, disp, h, w);
    break;

  case MONO_DITHER:
    MonoDitherImage(l, Cr, Cb, disp, h, w);
    break;

  case MONO_THRESHOLD:
    MonoThresholdImage(l, Cr, Cb, disp, h, w);
    break;

  case ORDERED2_DITHER:
    Ordered2DitherImage(l, Cr, Cb, disp, h, w);
    break;

  case MBORDERED_DITHER:
    MBOrderedDitherImage(l, Cr, Cb, disp, h, w, vid_stream->ditherFlags);
    break;
  }
}   /* DoDitherImage () */
#else
void
DoDitherImage(vid_stream)
VidStream *vid_stream;
{
 unsigned char *l=vid_stream->current->luminance,
               *Cr=vid_stream->current->Cr,
               *Cb=vid_stream->current->Cb,
               *disp=vid_stream->current->display;
 int h=(int) vid_stream->mb_height * 16;
 int w=(int) vid_stream->mb_width * 16;

 Color32DitherImage(l,Cr,Cb,disp,h,w);
}
#endif
