/* mainpanel.c -- xa main panel creation and event procedures */

/* $Id: mainpanel.c,v 1.13 1996/05/04 20:00:22 richards Exp $ */

/*
 * $Log: mainpanel.c,v $
 * Revision 1.13  1996/05/04 20:00:22  richards
 * minor changes
 *
 * Revision 1.12  1996/04/19 22:19:46  richards
 * replaced xv_set(...FRAME_BUSY...) calls with Busy()
 * added -oneatatime support
 *
 * Revision 1.11  1996/04/05 16:36:01  richards
 * minor changes
 *
 * Revision 1.10  1996/03/16 17:07:36  richards
 * renamed xrastool to xa
 * fixed indenting
 * added CanvasCursor() call to restore cursor after BUSY state
 *
 * Revision 1.9  1996/03/14 22:07:24  richards
 * implemented "insert" and "delete" buttons
 * rationalized prototypes
 * put logical operators into pull-down menu
 * all frames now go "busy" during DynamicLoad
 * added free_image() function
 * miscellaneous minor changes
 *
 * Revision 1.8  1996/03/08 21:00:34  richards
 * improved DynamicLoad() functionality and added wait cursor (needs work)
 * added free()'s in quit() procedure for clean exit
 *
 * Revision 1.7  1996/02/08 01:37:57  richards
 * removed INNER_CANVAS_BORDER (absorbed in ScrollMargin)
 * image counter now increments for -nopreload
 * replaced was_waiting with loading_delayed_images (local to file)
 *
 * Revision 1.6  1996/01/30 20:10:17  richards
 * removed stat() includes (now just in loadimages.c)
 * added DynamicLoad() to automagically load images if -nopreload
 * added WaitingForImages flags (used in conjunction with DynamicLoad())
 * replaced stat() calls in rescan() with GetLastChange() in loadimages.c
 *
 * Revision 1.5  1996/01/27 22:38:04  richards
 * changed some comment formats
 * adjusted panel buttons for better placement
 * added "back" keyboard accelerator
 * added DFLT_FAST_TIMER_SCALE and DFLT_FAST_TIMER_VALUE for FAST button
 *
 * Revision 1.4  1996/01/25 15:40:54  richards
 * added Rescan button to reload images that have changed since last access
 * replaced SetFixedSizeItem() with SetSizeItem() to improve "No Resizing"
 * if error occurs when reloading image, old image is retained
 *
 * Revision 1.3  1996/01/21 21:32:58  richards
 * prototyping, NewCmap functionality
 *
 * Revision 1.2  1996/01/18 18:15:57  richards
 * minor updates, including author address change
 *
 * Revision 1.1  1996/01/18 16:56:55  richards
 * Initial revision
 *
 */

#include "xa.h"
#include <X11/Xutil.h> /* for XDestroyImage() */
#include <xview/notice.h>

/* Global variables */

Panel MainPanel;
Bool LoadingDelayedImages = FALSE;

/* Local variables, mostly panel item handles */

static Panel_item image_number, image_name, width_item, height_item,
  sizing_item, set_item, timer_scale_item, timer_slider, dir_switch,
  starting_item, insert_image_name;
static Server_image fwd_switch_image, rev_switch_image;
static Xv_notice notice;
static Bool loading_delayed_images;
static Frame insert_frame;

/* Bitmaps */

#include "bitmaps/fwd.xbm"
#include "bitmaps/rev.xbm"
#include "bitmaps/cycle.xbm"
#include "bitmaps/loopback.xbm"
#include "bitmaps/oneway.xbm"

/* Local functions, mostly panel event procedures */

static Panel_setting set_current(void);
static void new_image(void);
static int  load_image(void);
static void delete_image(void);
static void insert_image(void);
static void set_logical(Panel_item item, int value, Event *event);
static Panel_setting load_image_from_text(void);
static Panel_setting set_width(void);
static Panel_setting set_height(void);
static void set_sizing(Panel_item item, int value, Event *event);
static void set_size(void);
static void change_timer_scale(Panel_item item, int value, Event *event);
static void change_timer_value(Panel_item item, int value, Event *event);
static void set_fast_mode(void);
static void toggle_dir(void);
static void step_image(void);
static void blink_images(Panel_item item, int value, Event *event);
static void cycle_images(void);
static void set_cycle_option(Panel_item item, int value, Event *event);
static void refresh(void);
static void rescan(void);
static void show_sub_frame(void);
static void info(void);
static void quit(void);
static void insert_image_ok(void);
static void insert_image_cleanup(void);
static void free_image(IMAGE_T *image);
static void draw_switch(void);
static void set_timer(int sw);

/*** END OF PREAMBLE ***/

void MakeMainPanel(void)
{
   /* Creates main panel and copyright notice */

   Rect *rect;
   Panel_item dum_item;
   Server_image cycle_image, loop_back_image, one_way_image;
   Panel insert_panel;

   MainPanel = (Panel) xv_create(BaseFrame, PANEL, NULL);

   /* Create LOTS of items... */

   image_number = (Panel_item)
     xv_create(MainPanel, PANEL_NUMERIC_TEXT,
	       PANEL_LABEL_STRING, "Image #",
	       PANEL_VALUE_DISPLAY_LENGTH, 4,
	       PANEL_VALUE_STORED_LENGTH, 4,
	       PANEL_MIN_VALUE, 0,
	       PANEL_MAX_VALUE, MAX_NUM_IMAGES + 1,
	       PANEL_VALUE, Current + 1,
	       PANEL_NOTIFY_PROC, set_current,
	       NULL);

   (void) xv_create(MainPanel, PANEL_BUTTON,
		    PANEL_LABEL_STRING, "New",
		    PANEL_NOTIFY_PROC, new_image,
		    NULL);
   
   (void) xv_create(MainPanel, PANEL_BUTTON,
		    PANEL_LABEL_STRING, "Load",
		    PANEL_NOTIFY_PROC, load_image,
		    NULL);
   
   (void) xv_create(MainPanel, PANEL_BUTTON,
		    PANEL_LABEL_STRING, "Del",
		    PANEL_NOTIFY_PROC, delete_image,
		    NULL);
   
   (void) xv_create(MainPanel, PANEL_BUTTON,
		    PANEL_LABEL_STRING, "Ins",
		    PANEL_NOTIFY_PROC, insert_image,
		    NULL);
   
   (void) xv_create(MainPanel, PANEL_CHOICE_STACK,
		    PANEL_LAYOUT, PANEL_HORIZONTAL,
		    PANEL_LABEL_STRING, "Op",
		    PANEL_DISPLAY_LEVEL, PANEL_NONE,
		    PANEL_CHOICE_STRINGS, "+ (SRC)", "| (OR)", "& (AND)",
		      "x (XOR)", "e (ERA)", "- (INV)", NULL,
		    PANEL_VALUE, Logical,
		    PANEL_NOTIFY_PROC, set_logical,
		    NULL);
   
   image_name = (Panel_item)
     xv_create(MainPanel, PANEL_TEXT,
	       PANEL_NEXT_ROW, -1,
	       PANEL_LABEL_STRING, "Image:",
	       PANEL_VALUE_DISPLAY_LENGTH, 42,
	       PANEL_VALUE_STORED_LENGTH, MAX_STR_LEN,
	       PANEL_NOTIFY_PROC, load_image_from_text,
	       NULL);
   
   width_item = (Panel_item)
     xv_create(MainPanel, PANEL_NUMERIC_TEXT,
	       PANEL_NEXT_ROW, -1,
	       PANEL_LABEL_STRING, "W:",
	       PANEL_VALUE_DISPLAY_LENGTH, 4,
	       PANEL_VALUE_STORED_LENGTH, 4,
	       PANEL_MIN_VALUE, MIN_IMAGE_WIDTH,
	       PANEL_MAX_VALUE, MAX_IMAGE_WIDTH,
	       PANEL_VALUE, FixedWidth,
	       PANEL_NOTIFY_PROC, set_width,
	       NULL);
   
   height_item = (Panel_item)
     xv_create(MainPanel, PANEL_NUMERIC_TEXT,
	       PANEL_LABEL_STRING, "H:",
	       PANEL_VALUE_DISPLAY_LENGTH, 4,
	       PANEL_VALUE_STORED_LENGTH, 4,
	       PANEL_MIN_VALUE, MIN_IMAGE_HEIGHT,
	       PANEL_MAX_VALUE, MAX_IMAGE_HEIGHT,
	       PANEL_VALUE, FixedHeight,
	       PANEL_NOTIFY_PROC, set_height,
	       NULL);
   
   rect = (Rect *) xv_get(height_item, XV_RECT);
   
   sizing_item = (Panel_item)
     xv_create(MainPanel, PANEL_CHOICE,
	       PANEL_CHOICE_STRINGS, "Auto", "Full", "Fixed", NULL,
	       XV_Y, rect->r_top - 3,
	       PANEL_VALUE, Sizing,
	       PANEL_NOTIFY_PROC, set_sizing,
	       NULL);
   
   set_item = (Panel_item)
     xv_create(MainPanel, PANEL_BUTTON,
	       PANEL_LABEL_STRING, " Set ",
	       PANEL_NOTIFY_PROC, set_size,
	       NULL);
   
   timer_scale_item = (Panel_item)
     xv_create(MainPanel, PANEL_CHOICE,
	       PANEL_NEXT_ROW, -1,
	       PANEL_CHOICE_STRINGS, "us", "ms", "sec", NULL,
	       PANEL_VALUE, TimerScale,
	       PANEL_NOTIFY_PROC, change_timer_scale,
	       NULL);
   
   timer_slider = (Panel_item)
     xv_create(MainPanel, PANEL_SLIDER,
	       PANEL_TICKS, 5,
	       PANEL_SLIDER_WIDTH, 129,
	       PANEL_MIN_VALUE, MIN_TIMER_VALUE,
	       PANEL_MAX_VALUE, MAX_TIMER_VALUE,
	       PANEL_VALUE, TimerValue,
	       PANEL_NOTIFY_PROC, change_timer_value,
	       NULL);
   
   (void) xv_create(MainPanel, PANEL_BUTTON,
		    PANEL_LABEL_STRING, "Fast",
		    PANEL_NOTIFY_PROC, set_fast_mode,
		    NULL);
   
   fwd_switch_image = (Server_image)
     xv_create(XV_NULL, SERVER_IMAGE,
	       SERVER_IMAGE_X_BITS, fwd_bits,
	       XV_WIDTH, fwd_width,
	       XV_HEIGHT, fwd_height,
	       NULL);
   
   rev_switch_image = (Server_image)
     xv_create(XV_NULL, SERVER_IMAGE,
	       SERVER_IMAGE_X_BITS, rev_bits,
	       XV_WIDTH, rev_width,
	       XV_HEIGHT, rev_height,
	       NULL);
   
   dir_switch = (Panel_item)
     xv_create(MainPanel, PANEL_MESSAGE,
	       PANEL_NEXT_ROW, -1,
	       PANEL_NOTIFY_PROC, toggle_dir,
	       NULL);
   
   draw_switch();
   
   rect = (Rect *) xv_get(dir_switch, XV_RECT);
   
   (void) xv_create(MainPanel, PANEL_BUTTON,
		    PANEL_LABEL_STRING, "Step",
		    XV_X, rect_right(rect) + 19,
		    PANEL_NOTIFY_PROC, step_image,
		    NULL);
   
   (void) xv_create(MainPanel, PANEL_BUTTON,
		    PANEL_LABEL_STRING, "Blink",
		    PANEL_NOTIFY_PROC, blink_images,
		    NULL);
   
   dum_item = (Panel_item)
     xv_create(MainPanel, PANEL_BUTTON,
	       PANEL_LABEL_STRING, "Cycle",
	       PANEL_NOTIFY_PROC, cycle_images,
	       NULL);
   
   rect = (Rect *) xv_get(dum_item, XV_RECT);
   
   cycle_image = (Server_image)
     xv_create(XV_NULL, SERVER_IMAGE,
	       SERVER_IMAGE_X_BITS, cycle_bits,
	       XV_WIDTH, cycle_width,
	       XV_HEIGHT, cycle_height,
	       NULL);
   
   loop_back_image = (Server_image)
     xv_create(XV_NULL, SERVER_IMAGE,
	       SERVER_IMAGE_X_BITS, loop_back_bits,
	       XV_WIDTH, loop_back_width,
	       XV_HEIGHT, loop_back_height,
	       NULL);
   
   one_way_image = (Server_image)
     xv_create(XV_NULL, SERVER_IMAGE,
	       SERVER_IMAGE_X_BITS, one_way_bits,
	       XV_WIDTH, one_way_width,
	       XV_HEIGHT, one_way_height,
	       NULL);
   
   (void) xv_create(MainPanel, PANEL_CHOICE,
		    PANEL_CHOICE_IMAGES, cycle_image, loop_back_image,
		      one_way_image, NULL,
		    XV_X, rect_right(rect) + 19,
		    XV_Y, rect->r_top - 3,
		    PANEL_VALUE, CycleOption,
		    PANEL_NOTIFY_PROC, set_cycle_option,
		    NULL);
   
   (void) xv_create(MainPanel, PANEL_BUTTON,
		    PANEL_NEXT_ROW, -1,
		    PANEL_LABEL_STRING, "Refresh",
		    PANEL_NOTIFY_PROC, refresh,
		    NULL);
   
   (void) xv_create(MainPanel, PANEL_BUTTON,
		    PANEL_LABEL_STRING, "Rescan",
		    PANEL_NOTIFY_PROC, rescan,
		    NULL);
   
   starting_item = (Panel_item)
     xv_create(MainPanel, PANEL_BUTTON,
	       PANEL_LABEL_STRING, "More...",
	       PANEL_NOTIFY_PROC, show_sub_frame,
	       NULL);
   
   (void) xv_create(MainPanel, PANEL_BUTTON,
		    PANEL_LABEL_STRING, "          Info...         ",
		    PANEL_NOTIFY_PROC, info,
		    NULL);
   
   (void) xv_create(MainPanel, PANEL_BUTTON,
		    PANEL_LABEL_STRING, "Quit",
		    PANEL_NOTIFY_PROC, quit,
		    NULL);
   
   window_fit(MainPanel);
   window_fit(BaseFrame);
   
   /* Special "insert" frame */
   
   insert_frame = (Frame)
     xv_create(BaseFrame, FRAME_CMD,
	       FRAME_CMD_DEFAULT_PIN_STATE, FRAME_CMD_PIN_IN,
	       FRAME_LABEL, "xa: Insert Image",
	       NULL);
   
   insert_panel = (Panel) xv_get(insert_frame, FRAME_CMD_PANEL);
   
   insert_image_name = (Panel_item)
     xv_create(insert_panel, PANEL_TEXT,
	       PANEL_LABEL_STRING, "Image:",
	       PANEL_VALUE_DISPLAY_LENGTH, 30,
	       PANEL_VALUE_STORED_LENGTH, MAX_STR_LEN,
	       PANEL_NOTIFY_PROC, insert_image_ok,
	       NULL);
   
   (void) xv_create(insert_panel, PANEL_BUTTON,
		    PANEL_LABEL_STRING, "OK",
		    PANEL_NOTIFY_PROC, insert_image_ok,
		    NULL);
   
   (void) xv_create(insert_panel, PANEL_BUTTON,
		    PANEL_LABEL_STRING, "Cancel",
		    PANEL_NOTIFY_PROC, insert_image_cleanup,
		    NULL);
   
   window_fit(insert_panel);
   window_fit(insert_frame);
   
   /* Info and copyright notice */
   
   notice = (Xv_notice)
     xv_create(MainPanel, NOTICE,
	       NOTICE_MESSAGE_STRINGS,
	       VERSION,
	       "", "",
	       COPYRIGHT,
	       "", "",
	       "This software may be freely distributed",
	       "under the terms of the GNU General Public License",
	       "", "",
	       "Comments welcome!",
	       "", "",
	       "snail-mail:",
	       "c/o CITA, McLennan Labs",
	       "University of Toronto",
	       "60 St. George Street",
	       "Toronto, Ontario",
	       "CANADA M5S 1A7",
	       "", "",
	       "e-mail: richards@cita.utoronto.ca",
	       NULL,
	       NOTICE_BUTTON, "Done", 0,
	       NOTICE_NO_BEEPING, TRUE,
	       NULL);
}

Notify_value DynamicLoad(Notify_client client, Notify_event event,
			 Notify_arg arg, Notify_event_type type)
{
  /*
   * Loads images one at a time so that the user can watch while they are
   * loading. This function is interposed between the notifier and the
   * main panel event handler. Unfortunately, this function cannot remove
   * itself, so it remains interposed for the duration.
   *
   */

  Notify_value value;

  /* Process the event first and flush the X queue */

  value = notify_next_event_func(client, event, arg, type);

  XFlush(MainDisplay);

  /* If the main panel just appeared, perform the loading sequence */

  if (VisualLoad && event_action((Event *) event) == WIN_REPAINT) {
    LoadingDelayedImages = TRUE;
    Busy(TRUE);
    (void) LoadData(NumDelayedFiles, DelayedFilenames, 0);
    Busy(FALSE);
    LoadingDelayedImages = FALSE;
    if (NumImages > 0)
      Current = NumImages - 1; /* Stay with last image loaded */
    VisualLoad = FALSE;
  }

  /* Force panel info update after displaying last image */

  if (Options & NO_UPDATES)
    UpdateLabels();

  return value;
}

void KeyPressed(char c)
{
  /* Interface for keyboard acceleration called by canvas_event_proc() */

  switch (c) {
  case SET_KEY:
    set_size();
    break;
  case STEP_KEY:
    step_image();
    break;
  case BACK_KEY:
    Direction = (Direction == FWD ? REV : FWD);
    step_image();
    Direction = (Direction == FWD ? REV : FWD);
    break;
  case CYCLE_KEY:
    cycle_images();
    break;
  case QUIT_KEY:
    quit();
    break;
  default:
    /* Ignored */;
  }
}

void UpdateLabels(void)
{
  /* Updates image number, filename, width & height items, and main footer */

  IMAGE_T *image;
  char msg[MAX_STR_LEN];
  int num_colors;

  image = Image[Current];

  (void) xv_set(image_number, PANEL_VALUE, Current + 1, NULL);

  (void) xv_set(image_name, PANEL_VALUE, image->filename, NULL);

  if (!(Options & NO_RESIZING)) {
    (void) xv_set(width_item, PANEL_VALUE,
		  (int) xv_get(CanvasFrame, XV_WIDTH) - ScrollMargin, NULL);
    (void) xv_set(height_item, PANEL_VALUE,
		  (int) xv_get(CanvasFrame, XV_HEIGHT) - ScrollMargin, NULL);
  }

  (void) sprintf(msg, "%i image%s loaded",
		 loading_delayed_images ? Current + 1 : NumImages,
		 PLURAL(NumImages));

  (void) xv_set(BaseFrame, FRAME_LEFT_FOOTER, msg, NULL);

  num_colors = (OneCmap || NewCmap ? NumLockedColors : image->num_colors);

  (void) sprintf(msg, "%i color%s", num_colors, PLURAL(num_colors));

  (void) xv_set(BaseFrame, FRAME_RIGHT_FOOTER, msg, NULL);
}

void SetSizeItem(int sizing)
{
  /* Set sizing button to Fixed */

  Sizing = sizing;

  (void) xv_set(sizing_item, PANEL_VALUE, Sizing, NULL);
}

void TimerOn(void)
{
  /* Activates automatic calls of step_image() for blinking/cycling */

  notify_set_itimer_func(BaseFrame, (Notify_func) step_image, ITIMER_REAL,
			 &Timer, NULL);
}

void TimerOff(void)
{
  /* Deactivates timed calls */

  notify_set_itimer_func(BaseFrame, NOTIFY_FUNC_NULL, ITIMER_REAL, NULL, NULL);
}

void MoveCursorAndCaret(void)
{
  /* Moves cursor onto main panel and positions main panel input caret */

  PointToItem(BaseFrame, (Rect *) xv_get(starting_item, XV_RECT));

  if (NO_IMAGES)
    (void) xv_set(MainPanel, PANEL_CARET_ITEM, image_name, NULL);
  else
    (void) xv_set(MainPanel, PANEL_CARET_ITEM, image_number, NULL);
}

void ToggleSizingExposure(Bool flag)
{
  (void) xv_set(width_item, PANEL_INACTIVE, flag, NULL);
  (void) xv_set(height_item, PANEL_INACTIVE, flag, NULL);
  (void) xv_set(sizing_item, PANEL_INACTIVE, flag, NULL);
  (void) xv_set(set_item, PANEL_INACTIVE, flag, NULL);
}

static Panel_setting set_current(void)
{
  /* Reads image number from panel and displays corresponding image */

  int value;

  value = (int) xv_get(image_number, PANEL_VALUE);

  /* Handle out of range values */

  if (value > NumImages) {
    value = Current + 1;
    (void) xv_set(image_number, PANEL_VALUE, value, NULL);
  }
  else if (value < 1) {
    value = 1;
    (void) xv_set(image_number, PANEL_VALUE, 1, NULL);
  }

  /* Internally images start at 0... */

  Current = value - 1;

  /* Show image */

  if (!UNDEF_IMAGE)
    ShowNewImage();

  /* Do not advance caret */

  return PANEL_NONE;
}

static void new_image(void)
{
  /* Prompts user for filename of new image */

  Current = NumImages;

  (void) xv_set(image_number, PANEL_VALUE, NumImages + 1, NULL);

  (void) xv_set(image_name, PANEL_VALUE, "", NULL);

  (void) xv_set(BaseFrame, FRAME_RIGHT_FOOTER, "Waiting for new image",
		NULL);

  if (Options & LIVE_CURSOR)
    (void) xv_set(CanvasFrame, FRAME_LEFT_FOOTER, "(no image)",
		  NULL);

  (void) xv_set(MainPanel, PANEL_CARET_ITEM, image_name, NULL);

  /* Erase any existing display */

  PaintBackground();
}

static int load_image(void)
{
  /* Reads filename from panel, loads image, and displays it */
  
  int n;
  char *filename[1];
  
  *filename = (char *) xv_get(image_name, PANEL_VALUE);
  
  if (!**filename)
    return 0;

  /* NOTE: Image[Current] may be NULL if inserting image... */

  if ((n = LoadData(1, filename, Current)) <= 0)
    if (Current < NumImages && Image[Current])
      (void) xv_set(image_name, PANEL_VALUE, Image[Current]->filename, NULL);
    else
      (void) xv_set(image_name, PANEL_VALUE, "", NULL);
  else
    ShowNewImage();

  return n;
}
  
static void delete_image(void)
{
  /* Deletes current image from memory */

  int i;

  if (NumImages <= 1 && Current == 0)
    return;

  if (Current == NumImages) {
    --Current;
    ShowNewImage();
    return;
  }

  free_image(Image[Current]);  

  --NumImages;

  for (i = Current; i < NumImages; i++)
    Image[i] = Image[i + 1];

  Image[NumImages] = NULL;

  if (Current == NumImages)
    --Current;

  ShowNewImage();
}

static void insert_image(void)
{
  /* Activates insert frame to immediately prompt for filename */

  Busy(TRUE);

  (void) xv_set(insert_image_name, PANEL_VALUE, "", NULL);

  (void) xv_set(insert_frame, XV_SHOW, TRUE, NULL);
}

static void set_logical(Panel_item item, int value, Event *event)/*ARGSUSED*/
{
  /* Sets logical function */

  Logical = value;

  SetLogical();
}

static Panel_setting load_image_from_text(void)
{
  /* Call load_image() */

  (void) load_image();

  /* Do not advance caret */

  return PANEL_NONE;
}

static Panel_setting set_width(void)
{
  /* Changes FixedWidth and forces fixed sizing */

  FixedWidth = (int) xv_get(width_item, PANEL_VALUE);

  if (Sizing != FIXED)
    SetSizeItem(FIXED);

  ShowNewImage();

  /* Advance caret to height item */

  return PANEL_NEXT;
}

static Panel_setting set_height(void)
{
  /* Changes FixedHeight and forces fixed sizing */

  FixedHeight = (int) xv_get(height_item, PANEL_VALUE);

  if (Sizing != FIXED)
    SetSizeItem(FIXED);

  ShowNewImage();

  /* Move caret to width item */

  return PANEL_PREVIOUS;
}

static void set_sizing(Panel_item item, int value, Event *event)/*ARGSUSED*/
{
  /* Changes sizing value */

  Sizing = value;

  if (Sizing == FULL) {
    if (UNDEF_IMAGE)
      return;
    (void) xv_set(width_item, PANEL_VALUE, Image[Current]->w, NULL);
    (void) xv_set(height_item, PANEL_VALUE, Image[Current]->h, NULL);
  }
  else {
    FixedWidth = (int) xv_get(width_item, PANEL_VALUE);
    FixedHeight = (int) xv_get(height_item, PANEL_VALUE);
  }

  ShowNewImage();
}

static void set_size(void)
{
  /* Fixes size according to current canvas window dimensions */

  FixedWidth = (int) xv_get(CanvasFrame, XV_WIDTH) - ScrollMargin;
  FixedHeight = (int) xv_get(CanvasFrame, XV_HEIGHT) - ScrollMargin;

  FixedWidth = MAX(FixedWidth, MIN_IMAGE_WIDTH);
  FixedHeight = MAX(FixedHeight, MIN_IMAGE_HEIGHT);

  (void) xv_set(width_item, PANEL_VALUE, FixedWidth, NULL);
  (void) xv_set(height_item, PANEL_VALUE, FixedHeight, NULL);

  SetSizeItem(FIXED);

  ShowNewImage();
}

static void change_timer_scale(Panel_item item, int value, Event *event)
/*ARGSUSED*/
{
  /* Changes timer scale */

  TimerScale = value;

  SetTimer();
}

static void change_timer_value(Panel_item item, int value, Event *event)
/*ARGSUSED*/
{
  /* Changes timer value */

  TimerValue = value;

  SetTimer();
}

static void set_fast_mode(void)
{
  /* Activates fast mode */

  SetOptions(Options ^ FAST_MODE);

  UpdateOptionsItem();

  TimerScale = DFLT_FAST_TIMER_SCALE;
  TimerValue = DFLT_FAST_TIMER_VALUE;

  (void) xv_set(timer_scale_item, PANEL_VALUE, TimerScale, NULL);
  (void) xv_set(timer_slider, PANEL_VALUE, TimerValue, NULL);

  SetTimer();
}

static void toggle_dir(void)
{
  /* Changes current direction flag */

  Direction = (Direction == FWD ? REV : FWD);

  if (!(Blinking && (Options & NO_UPDATES)))
    draw_switch();
}

static void step_image(void)
{
  /* Displays next or prev item depending on direction and cycling option */

  if (NumImages == 0)
    return;

  Current += (Direction == FWD ? 1 : -1);

  if (Blinking)
    toggle_dir();

  if (Blinking || !Cycling) {
    if (Current < 0)
      Current = NumImages - 1;
    else if (Current >= NumImages)
      Current = 0;
  }
  else
    switch (CycleOption) {
    case LOOP:
      if (Current < 0)
	Current = NumImages - 1;
      else if (Current >= NumImages)
	Current = 0;
      break;
    case LOOP_BACK:
      if (Current < 0) {
	Current = 1;
	toggle_dir();
      }
      else if (Current >= NumImages) {
	Current = NumImages - 2;
	toggle_dir();
      }
      break;
    case ONE_WAY:
      if (Current < 0) {
	Current = 0;
	cycle_images();
	return;
      }
      else if (Current >= NumImages) {
	Current = NumImages - 1;
	cycle_images();
	return;
      }
      break;
    }

  ShowNewImage();
}

static void blink_images(Panel_item item, int value, Event *event)/*ARGSUSED*/
{
  /* Toggles blinking */

  Blinking = (Bool) value;

  if (NumImages > 1) {
    if (!Blinking) {
      Blinking = TRUE;
      if (Cycling)
	return;
    }
    cycle_images();
  }
}

static void cycle_images(void)
{
  /* Toggles cycling */

  if (NumImages > 1) {
    if (!Cycling && CycleOption == ONE_WAY) {
      if (Current == NumImages - 1 && Direction == FWD) {
	Current = 0;
	ShowNewImage();
      }
      else if (Current == 0 && Direction == REV) {
	Current = NumImages - 1;
	ShowNewImage();
      }
    }
    set_timer(Cycling = !Cycling);
    if (Blinking && !Cycling)
      Blinking = FALSE;
  }
}

static void set_cycle_option(Panel_item item, int value, Event *event)
/*ARGSUSED*/
{
  CycleOption = value;
}

static void refresh(void)
{
  ShowNewImage();
}

static void rescan(void)
{
  /* Reloads images that have been modified since last read */

  int i, first, last;
  IMAGE_T *image;

  /* Loop over all images (or just current image if loading one at a time) */

  if (OneAtATime)
    first = last = Current;
  else {
    first = 0;
    last = NumImages - 1;
  }

  for (i = first; i <= last; i++) {
    image = Image[i];
    if (GetLastChange(image->filename) > image->last_change) {
      int result;
      char *filename;

      filename = (char *) malloc(strlen(image->filename) + 1);
      (void) strcpy(filename, image->filename); /* Local copy */
      result = LoadData(1, &filename, i);
      free((void *) filename);
      if (result != 1) {
	Warning("ShowNewImage(): Error occurred during load.");
	return;
      }
    }
  }

  ShowNewImage();
}

static void show_sub_frame(void)
{
  (void) xv_set(SubFrame, XV_SHOW, TRUE, NULL);
}

static void info(void)
{
  (void) xv_set(notice, XV_SHOW, TRUE, NULL);
}

static void quit(void)
{
  int i;

  /* Turn off cycling before freeing images */

  TimerOff();

  /* For completeness, free all malloc'd resources */

  for (i = 0; i < NumImages; i++)
    free_image(Image[i]);

  if (Verbose && TotalImageMem != 0)
    Warning("Memory accounting did not balance.");
  
  /* Exit cleanly */
  
  (void) xv_destroy_safe(BaseFrame);
}

static void insert_image_ok(void)
{
  /* Performs insertion of requested file */

  int i;

  (void) xv_set(image_name,
		PANEL_VALUE, (char *) xv_get(insert_image_name, PANEL_VALUE),
		NULL);

  insert_image_cleanup();

  ++Current;

  for (i = NumImages; i > Current; i--)
    Image[i] = Image[i - 1];

  ++NumImages;

  Image[Current] = NULL;

  if (load_image() <= 0) {
    --NumImages;
    for (i = Current; i < NumImages; i++)
      Image[i] = Image[i + 1];
    --Current;
    ShowNewImage();
  }
}

static void insert_image_cleanup(void)
{
  (void) xv_set(insert_frame, XV_SHOW, FALSE, NULL);

  Busy(False);
}

static void free_image(IMAGE_T *image)
{
  /* Deallocates space associated with image */

  if (!image)
    return;

  if (image->filename)
    free((void *) image->filename);

  if (image->colors)
    free((void *) image->colors);

  if (UsePixmaps)
    XFreePixmap(MainDisplay, image->mem.pixmap);
  else if (image->mem.ximage) {
    if (OldXImagePtr && *OldXImagePtr == image->mem.ximage) {
      TotalImageMem -= image->l;
      OldXImagePtr = NULL;
    }
    (void) XDestroyImage(image->mem.ximage);
  }

  if (!OneAtATime)
    TotalImageMem -= image->l;

  free((void *) image);
}

static void draw_switch(void)
{
  (void) xv_set(dir_switch, PANEL_LABEL_IMAGE,
		(Direction == FWD ? fwd_switch_image : rev_switch_image),
		NULL);
}

static void set_timer(int sw)
{
  /* Activates or deactivates cycling mode */

  static char frame_header[MAX_STR_LEN];

  if (sw) {
    (void) strcpy(frame_header, (char *) xv_get(CanvasFrame, FRAME_LABEL));
    (void) xv_set(CanvasFrame, FRAME_LABEL, "CYCLING", NULL);
    TimerOn();
  }
  else {
    TimerOff();
    (void) xv_set(CanvasFrame, FRAME_LABEL, frame_header, NULL);
    if (Blinking && (Options & (NO_UPDATES | NO_BACKDROPS)))
      draw_switch();
    if (Options & NO_UPDATES)
      UpdateLabels();
  }
}

/*** mainpanel.c ***/
