/* params.c -- routines for setting xa defaults and handling options */

/* $Id: params.c,v 1.12 1996/05/04 20:00:40 richards Exp $ */

/*
 * $Log: params.c,v $
 * Revision 1.12  1996/05/04 20:00:40  richards
 * fixed -newcmap -fast behaviour
 * other minor fixes
 *
 * Revision 1.11  1996/04/19 22:20:46  richards
 * added -oneatatime (-1) support
 *
 * Revision 1.10  1996/03/16 17:08:22  richards
 * renamed xrastool to xa
 * fixed indenting
 * renamed -logical to -op
 *
 * Revision 1.9  1996/03/14 22:10:25  richards
 * implemented -fastquant option
 *
 * Revision 1.8  1996/03/08 21:02:19  richards
 * replaced -nopreload with -visualload
 * replaced -pixmaps with -serverimages
 * replaced -ximages with -clientimages
 *
 * Revision 1.7  1996/02/21 20:49:49  richards
 * implemented -copy option
 *
 * Revision 1.6  1996/02/08 01:41:21  richards
 * replaced "+/-" toggles with "no" prefix
 * changed comments and formatting, and reordered some options
 * added AllowManualSizing() to get around resizing problem (Cf. canvas.c)
 *
 * Revision 1.5  1996/01/30 20:13:13  richards
 * added RawQuery and WaitingForImages (-nopreload) support
 * removed redundant xv_get(MainCms, CMS_COLORS, TmpColors) call
 *
 * Revision 1.4  1996/01/27 22:40:16  richards
 * changed some comment formats
 * added DFLT_FAST_TIMER_SCALE and DFLT_FAST_TIMER_VALUE for -fast option
 *
 * Revision 1.3  1996/01/25 15:46:38  richards
 * added [-raw [-rw (width)] [-rh (height)]] options for RAW file support
 * added [+|-pixmaps] option to choose between Pixmap and XImage storage
 * added [-gamma] option for image gamma correction
 * fixed bug with [+|-resizing] option
 * removed reference to TmpXImage (obsolete)
 *
 * Revision 1.2  1996/01/21 21:33:31  richards
 * prototyping, improved "Usage" message, added +|-newcmap option
 *
 * Revision 1.1  1996/01/18 16:56:55  richards
 * Initial revision
 *
 */

#include "xa.h"

/* Default values for main parameters */

#define DFLT_NEW_CMAP           FALSE
#define DFLT_ONE_CMAP           FALSE
#define DFLT_USE_FAST_QUANT	FALSE
#define DFLT_BLINKING           FALSE
#define DFLT_SHOW_SUB_PANEL     FALSE
#define DFLT_CYCLING            FALSE
#define DFLT_OPTIONS            0
#define DFLT_RAW		FALSE
#define DFLT_RAW_QUERY          FALSE
#define DFLT_USE_PIXMAPS	TRUE
#define DFLT_VISUAL_LOAD        FALSE
#define DFLT_ONE_AT_A_TIME      FALSE
#define DFLT_VERBOSE            TRUE
#define DFLT_FIRST_IMAGE        0
#define DFLT_COMPRESS_OPTION    PAIR_COMPRESS
#define DFLT_NUM_COLORS_TO_COPY 0
#define DFLT_LOGICAL            SRC
#define DFLT_FIXED_WIDTH        640
#define DFLT_FIXED_HEIGHT       480
#define DFLT_SIZING             AUTO
#define DFLT_TIMER_SCALE        TIMER_MS
#define DFLT_TIMER_VALUE        100
#define DFLT_DIRECTION          FWD
#define DFLT_CYCLE_OPTION       LOOP
#define DFLT_BACKDROP_INDEX     0
#define DFLT_SCALING            COLOR
#define DFLT_RED_SLIDER_VALUE   100
#define DFLT_RED_SLIDER_SCALE   0
#define DFLT_GREEN_SLIDER_VALUE 100
#define DFLT_GREEN_SLIDER_SCALE 0
#define DFLT_BLUE_SLIDER_VALUE  100
#define DFLT_BLUE_SLIDER_SCALE  0
#define DFLT_RAW_WIDTH		0 /* (will prompt user if zero) */
#define DFLT_RAW_HEIGHT		0 /* (ditto) */

#define INITIALIZE -1 /* Useful flag for SetOptions() */

/* Global variables */

int NumImages = 0;
struct itimerval Timer;
int NumLockedColors;
COLORCELL_T *LockedColors;

/* Global parameters */

Bool ForceFirstResize = FALSE;
Bool StartCycling     = FALSE;
Bool NewCmap          = DFLT_NEW_CMAP;
Bool OneCmap          = DFLT_ONE_CMAP;
Bool UseFastQuant     = DFLT_USE_FAST_QUANT;
Bool ShowSubPanel     = DFLT_SHOW_SUB_PANEL;
Bool Blinking         = DFLT_BLINKING;
Bool Cycling          = DFLT_CYCLING;
int  Options          = DFLT_OPTIONS;
Bool Raw	      = DFLT_RAW;
Bool RawQuery         = DFLT_RAW_QUERY;
int  UsePixmaps	      = DFLT_USE_PIXMAPS;
Bool VisualLoad       = DFLT_VISUAL_LOAD;
Bool OneAtATime       = DFLT_ONE_AT_A_TIME;
Bool Verbose          = DFLT_VERBOSE;
int  Current          = DFLT_FIRST_IMAGE;
int  CompressOption   = DFLT_COMPRESS_OPTION;
int  NumColorsToCopy  = DFLT_NUM_COLORS_TO_COPY;
int  Logical          = DFLT_LOGICAL;
int  FixedWidth       = DFLT_FIXED_WIDTH;
int  FixedHeight      = DFLT_FIXED_HEIGHT;
int  Sizing           = DFLT_SIZING;
int  TimerScale       = DFLT_TIMER_SCALE;
int  TimerValue       = DFLT_TIMER_VALUE;
int  Direction        = DFLT_DIRECTION;
int  CycleOption      = DFLT_CYCLE_OPTION;
int  BackdropIndex    = DFLT_BACKDROP_INDEX;
int  Scaling          = DFLT_SCALING;
int  RedSliderValue   = DFLT_RED_SLIDER_VALUE;
int  RedSliderScale   = DFLT_RED_SLIDER_SCALE;
int  GreenSliderValue = DFLT_GREEN_SLIDER_VALUE;
int  GreenSliderScale = DFLT_GREEN_SLIDER_SCALE;
int  BlueSliderValue  = DFLT_BLUE_SLIDER_VALUE;
int  BlueSliderScale  = DFLT_BLUE_SLIDER_SCALE;
int  RawWidth         = DFLT_RAW_WIDTH;
int  RawHeight        = DFLT_RAW_HEIGHT;

/* Local functions */

static void show_usage(char *prog);

/*** END OF PREAMBLE ***/

/* Handy macro for ReadCmdLine() */

#define NEED_ARG {\
  (void) fprintf(stderr, "\007-%s requires argument.\n", argv[i - 1]);\
  show_usage(argv[0]);\
}

void ReadCmdLine(int argc, char *argv[], int *files_start)
{
  /*
   * Reads any command line options and returns files_start (an index into
   * argv[] pointing to the first filename).
   *
   * NOTES:
   *
   * -- Most (but not all!) trailing arguments override leading arguments
   * -- The "no" prefix disables the corresponding option
   * -- New options should not have "no" as the first two letters
   *
   */

  int i;
  Bool enable;

  /* Loop through arguments */

  i = 1;

  while (i < argc && argv[i][0] == '-') { /* Options must start with hyphen */
    ++argv[i]; /* Eat hyphen */
    enable = !(argv[i][0] == 'n' && argv[i][1] == 'o'); /* Check for "no" */
    if (!enable)
      argv[i] += 2; /* Eat "no" */
    if (!strcmp(argv[i], "fast")) { /* Fast mode, always test first */
      if (enable) {
	OneCmap = !NewCmap;
	ForceFirstResize = TRUE;
	FixedWidth = FORCE_SIZE;
	FixedHeight = FORCE_SIZE;
	Sizing = FIXED;
	TimerScale = DFLT_FAST_TIMER_SCALE;
	TimerValue = DFLT_FAST_TIMER_VALUE;
	Options = FAST_MODE;
      }
      else { /* "no" modifier not allowed */
	(void) fprintf(stderr, "\007Cannot use \"no\" modifier with -fast.\n");
	show_usage(argv[0]);
      }
    }
    else if (!strcmp(argv[i], "newcmap")) { /* New ("best") colormap */
      if (enable && (VisualLoad || OneAtATime))
	(void) fprintf(stderr, "\007Cannot use -newcmap with %s.\n",
		       VisualLoad ? "-visualload" : "-oneatatime");
      else
	NewCmap = enable;
      if (NewCmap) {
	Options |= LOCK_COLORS;
	OneCmap = FALSE; /* Mutually exclusive */
      }
      else
	Options &= ~LOCK_COLORS;
    }
    else if (!strcmp(argv[i], "onecmap")) { /* One colormap */
      OneCmap = enable;
      if (OneCmap) {
	Options |= LOCK_COLORS;
	NewCmap = FALSE; /* Mutually exclusive */
      }
      else
	Options &= ~LOCK_COLORS;
    }
    else if (!strcmp(argv[i], "fastquant"))
      UseFastQuant = enable;
    else if (!strcmp(argv[i], "subpanel"))
      ShowSubPanel = enable;
    else if (!strcmp(argv[i], "blink"))
      Blinking = enable;
    else if (!strcmp(argv[i], "cycle"))
      Cycling = enable;
    else if (!strcmp(argv[i], "scrolling"))
      Options = (enable ? Options & ~NO_SCROLLING : Options | NO_SCROLLING);
    else if (!strcmp(argv[i], "resizing")) {
      Options = (enable ? Options & ~NO_RESIZING : Options | NO_RESIZING);
      if (!enable) { /* Ensure first image is correctly framed */
	ForceFirstResize = TRUE;
	FixedWidth = FORCE_SIZE;
	FixedHeight = FORCE_SIZE;
      }
      else
	ForceFirstResize = FALSE;
    }
    else if (!strcmp(argv[i], "moving"))
      Options = (enable ? Options & ~NO_MOVING : Options | NO_MOVING);
    else if (!strcmp(argv[i], "scaling"))
      Options = (enable ? Options & ~NO_SCALING : Options | NO_SCALING);
    else if (!strcmp(argv[i], "updates"))
      Options = (enable ? Options & ~NO_UPDATES : Options | NO_UPDATES);
    else if (!strcmp(argv[i], "backdrops"))
      Options = (enable ? Options & ~NO_BACKDROPS : Options | NO_BACKDROPS);
    else if (!strcmp(argv[i], "lockcolors") && !OneCmap && !NewCmap)
      Options = (enable ? Options | LOCK_COLORS : Options & ~LOCK_COLORS);
    else if (!strcmp(argv[i], "centering"))
      Options = (enable ? Options | CENTERING : Options & ~CENTERING);
    else if (!strcmp(argv[i], "livecursor"))
      Options = (enable ? Options | LIVE_CURSOR : Options & ~LIVE_CURSOR);
    else if (!strcmp(argv[i], "raw"))
      Raw = enable;
    else if (!strcmp(argv[i], "query")) {
      RawQuery = enable;
      if (enable)
	Raw = TRUE;
    }
    else if (!strcmp(argv[i], "serverimages"))
      if (enable && OneAtATime)
	(void) fprintf(stderr, "Server images disabled with -oneatatime.\n");
      else
	UsePixmaps = enable;
    else if (!strcmp(argv[i], "clientimages"))
      UsePixmaps = !enable; /* Exclusive with -pixmaps */
    else if (!strcmp(argv[i], "visualload"))
      if (enable && NewCmap)
	(void) fprintf(stderr, "\007Cannot use -visualload with -newcmap.\n");
      else
	VisualLoad = enable;
    else if (!strcmp(argv[i], "1") || !strcmp(argv[i], "oneatatime")) {
      OneAtATime = enable;
      if (enable) {
	if (NewCmap) {
	  (void) fprintf(stderr, "Warning: -newcmap disabled by -1 option.\n");
	  NewCmap = FALSE;
	  Options &= ~LOCK_COLORS;
	}
	UsePixmaps = FALSE;
      }
    }
    else if (!strcmp(argv[i], "verbose"))
      Verbose = enable;
    else if (!strcmp(argv[i], "quiet"))
      Verbose = !enable; /* Exclusive with -verbose */
    else if (!enable) { /* Remaining options do not accept "no" modifier */
      (void) fprintf(stderr, "\007Cannot use \"no\" modifier with -%s.\n",
		     argv[i]);
      (void) fprintf(stderr, "(%s may not be a recognized option).\n",
		     argv[i]);
      show_usage(argv[0]);
    }
    else if (!strcmp(argv[i], "start")) {
      if (++i < argc) {
	Current = atoi(argv[i]);
	Current = MIN(Current, MAX_NUM_IMAGES);
	Current = MAX(Current, 1);
	--Current; /* Internally, indices start at 0 */
      }
      else
	NEED_ARG /* Note: trailing semi-colon unnecessary */
    }
    else if (!strcmp(argv[i], "private"))
      CompressOption = NO_COMPRESS;
    else if (!strcmp(argv[i], "hist"))
      CompressOption = HIST_COMPRESS;
    else if (!strcmp(argv[i], "pair"))
      CompressOption = PAIR_COMPRESS;
    else if (!strcmp(argv[i], "copy")) {
      if (++i < argc) {
	NumColorsToCopy = atoi(argv[i]);
	NumColorsToCopy = MIN(NumColorsToCopy, MAX_NUM_COLORS);
	NumColorsToCopy = MAX(NumColorsToCopy, 0);
	if (NumColorsToCopy == 0) /* -copy 0 implies -private */
	  CompressOption = NO_COMPRESS;
      }
      else
	NEED_ARG
    }
    else if (!strcmp(argv[i], "op")) { /* Logical functions */
      if (++i < argc) {
	if (!strcmp(argv[i], "SRC"))
	  Logical = SRC;
	else if (!strcmp(argv[i], "OR"))
	  Logical = OR;
	else if (!strcmp(argv[i], "AND"))
	  Logical = AND;
	else if (!strcmp(argv[i], "XOR"))
	  Logical = XOR;
	else if (!strcmp(argv[i], "ERA"))
	  Logical = ERA;
	else if (!strcmp(argv[i], "NEG"))
	  Logical = NEG;
	else {
	  (void) fprintf(stderr, "\007Unrecognized -op argument.\n");
	  show_usage(argv[0]);
	}
      }
      else
	NEED_ARG
    }
    else if (!strcmp(argv[i], "w")) {
      if (++i < argc) {
	FixedWidth = atoi(argv[i]);
	FixedWidth = MIN(FixedWidth, MAX_IMAGE_WIDTH);
	FixedWidth = MAX(FixedWidth, MIN_IMAGE_WIDTH);
      }
      else
	NEED_ARG
    }
    else if (!strcmp(argv[i], "h")) {
      if (++i < argc) {
	FixedHeight = atoi(argv[i]);
	FixedHeight = MIN(FixedHeight, MAX_IMAGE_HEIGHT);
	FixedHeight = MAX(FixedHeight, MIN_IMAGE_HEIGHT);
      }
      else
	NEED_ARG
    }
    else if (!strcmp(argv[i], "auto"))
      Sizing = AUTO;
    else if (!strcmp(argv[i], "full"))
      Sizing = FULL;
    else if (!strcmp(argv[i], "fixed")) {
      Sizing = FIXED;
      if (enable) { /* Ensure first image is correctly framed */
	ForceFirstResize = TRUE;
	FixedWidth = FORCE_SIZE;
	FixedHeight = FORCE_SIZE;
      }
      else
	ForceFirstResize = FALSE;
    }
    else if (!strcmp(argv[i], "timerscale")) { /* Timer scale */
      if (++i < argc) {
	if (!strcmp(argv[i], "us"))
	  TimerScale = TIMER_US;
	else if (!strcmp(argv[i], "ms"))
	  TimerScale = TIMER_MS;
	else if (!strcmp(argv[i], "sec"))
	  TimerScale = TIMER_SEC;
	else {
	  (void) fprintf(stderr, "\007Unrecognized -timerscale argument.\n");
	  show_usage(argv[0]);
	}
      }
      else
	NEED_ARG
    }
    else if (!strcmp(argv[i], "timervalue")) {
      if (++i < argc) {
	TimerValue = atoi(argv[i]);
	TimerValue = MIN(TimerValue, MAX_TIMER_VALUE);
	TimerValue = MAX(TimerValue, MIN_TIMER_VALUE);
      }
      else
	NEED_ARG
    }
    else if (!strcmp(argv[i], "fwd"))
      Direction = FWD;
    else if (!strcmp(argv[i], "rev"))
      Direction = REV;
    else if (!strcmp(argv[i], "loop"))
      CycleOption = LOOP;
    else if (!strcmp(argv[i], "loopback"))
      CycleOption = LOOP_BACK;
    else if (!strcmp(argv[i], "oneway"))
      CycleOption = ONE_WAY;
    else if (!strcmp(argv[i], "backdrop")) {
      if (++i < argc) {
	BackdropIndex = atoi(argv[i]);
	BackdropIndex = MIN(BackdropIndex, NUM_BACKDROPS);
	BackdropIndex = MAX(BackdropIndex, 1);
	--BackdropIndex;
      }
      else
	NEED_ARG
    }
    else if (!strcmp(argv[i], "bw"))
      { Scaling = BW; SetColorSliderDefaults(); }
    else if (!strcmp(argv[i], "gray"))
      { Scaling = GRAY; SetColorSliderDefaults(); }
    else if (!strcmp(argv[i], "color"))
      { Scaling = COLOR; SetColorSliderDefaults(); }
    else if (!strcmp(argv[i], "gamma"))
      { Scaling = GAMMA; SetColorSliderDefaults(); }
    else if (!strcmp(argv[i], "cutoff"))
      { Scaling = CUTOFF; SetColorSliderDefaults(); }
    else if (!strcmp(argv[i], "contour"))
      { Scaling = CONTOUR; SetColorSliderDefaults(); }
    else if (!strcmp(argv[i], "random"))
      { Scaling = RANDOM; SetColorSliderDefaults(); }
    else if (!strcmp(argv[i], "red")) { /* Red slider */
      if (++i < argc) {
	RedSliderValue = atoi(argv[i]);
	if (RedSliderValue < 0) { /* Use neg val to select DIV */
	  RedSliderValue *= -1;
	  RedSliderScale = DIV;
	}
	else
	  RedSliderScale = MULT;
      }
      else
	NEED_ARG
    }
    else if (!strcmp(argv[i], "green")) {
      if (++i < argc) {
	GreenSliderValue = atoi(argv[i]);
	if (GreenSliderValue < 0) {
	  GreenSliderValue *= -1;
	  GreenSliderScale = DIV;
	}
	else
	  GreenSliderScale = MULT;
      }	
      else
	NEED_ARG
    }
    else if (!strcmp(argv[i], "blue")) {
      if (++i < argc) {
	BlueSliderValue = atoi(argv[i]);
	if (BlueSliderValue < 0) {
	  BlueSliderValue *= -1;
	  BlueSliderScale = DIV;
	}
	else
	  BlueSliderScale = MULT;
      }
      else
	NEED_ARG
    }
    else if (!strcmp(argv[i], "rw")) {
      if (++i < argc) {
	RawWidth = atoi(argv[i]);
	RawWidth = MAX(0, RawWidth);
	Raw = TRUE;
      }
      else
	NEED_ARG
    }
    else if (!strcmp(argv[i], "rh")) {
      if (++i < argc) {
	RawHeight = atoi(argv[i]);
	RawHeight = MAX(0, RawHeight);
	Raw = TRUE;
      }
      else
	NEED_ARG
    }
    else {
      (void) fprintf(stderr, "\007Unknown option: %s.\n", argv[i]);
      show_usage(argv[0]);
    }
    ++i; /* Proceed to next option */
  }

  /* Point to first non-option...hopefully it's a filename! */
  
  *files_start = i;
  
  /* Check to make sure no further options trail the first filename */
  
  for (i = *files_start + 1; i < argc; i++)
    if (argv[i][0] == '-') {
      (void) fprintf(stderr, "\007Bad syntax (%s).\n", argv[i]);
      show_usage(argv[0]);
    }
}

#undef NEED_ARG

void Initialize(void)
{
  /* Applies defaults and/or command line options as required */

  SetLogical();

  SetTimer();

  if (Blinking)
    Cycling = TRUE;

  if (Cycling) {
    StartCycling = TRUE;
    Cycling = FALSE; /* For correct toggling behaviour */
  }

  SetOptions(INITIALIZE);

  SetBackdrop();

  SetScaling();
}

void SetLogical(void)
{
  /* Changes logical function of MainGC */

  XGCValues xgcv;

  /* Determine function */

  switch (Logical) {
  case SRC:
    xgcv.function = GXcopy;
    break;
  case OR:
    xgcv.function = GXor;
    break;
  case AND:
    xgcv.function = GXand;
    break;
  case XOR:
    xgcv.function = GXxor;
    break;
  case ERA:
    xgcv.function = GXandInverted;
    break;
  case NEG:
    xgcv.function = GXcopyInverted;
    break;
  default:
    Error("Invalid logical function.");
  }

  /* Apply change */

  XChangeGC(MainDisplay, MainGC, GCFunction, &xgcv);
}

void SetTimer(void)
{
  /* Sets elements of Timer structure */

  /* First stop cycling if applicable */

  if (Cycling && !NO_IMAGES)
    TimerOff();

  /* Scale by 1000 for milliseconds */

  if (TimerScale == TIMER_MS)
    TimerValue *= 1000;

  /* Set microsecond or second elements as appropriate */

  if (TimerScale == TIMER_US || TimerScale == TIMER_MS) {
    Timer.it_value.tv_usec = Timer.it_interval.tv_usec = TimerValue;
    Timer.it_value.tv_sec = Timer.it_interval.tv_sec = 0;
  }
  else {
    Timer.it_value.tv_usec = Timer.it_interval.tv_usec = 0;
    Timer.it_value.tv_sec = Timer.it_interval.tv_sec = TimerValue;
  }

  /* Switch cycling back on if applicable */

  if (Cycling && !NO_IMAGES)
    TimerOn();
}

void SetOptions(int change)
{
  /* Apply change to Options (subpanel options) */

  int new_options;

  /* Ignore change on first call; otherwise determine new (set) options */

  if (change == INITIALIZE)
    change = new_options = Options;
  else
    new_options = Options ^ change;

  /* Toggle canvas scrollbars if requested */

  if (change & NO_SCROLLING)
    SetScrollbars(!(new_options & NO_SCROLLING));

  /*
   * Set fixed sizing if NO_RESIZING selected and hide sizing controls.
   *
   * Note: if resizing is forced on the first image (e.g. with -fast option),
   * the lock-out procedure must be performed by set_canvas_size_and_pos()
   * in canvas.c since the dimensions of the first image are not yet known.
   *
   */

  if (change & NO_RESIZING) {
    static Bool old_sizing = DFLT_SIZING;

    if (new_options & NO_RESIZING) {
      old_sizing = Sizing; /* Remember old sizing so we can switch back */
      SetSizeItem(FIXED);
      if (!ForceFirstResize)
	AllowManualSizing(FALSE);
    }
    else {
      SetSizeItem(old_sizing);
      AllowManualSizing(TRUE);
    }    
    ToggleSizingExposure(new_options & NO_RESIZING);
  }

  /* Hide or expose color scaling options and sliders if NO_SCALING changed */

  if (change & NO_SCALING)
    ToggleScalingExposure(new_options & NO_SCALING);

  /* Hide or expose backdrop choices if NO_BACKDROPS changed */

  if (change & NO_BACKDROPS)
    ToggleBackdropsExposure(new_options & NO_BACKDROPS);

  /* Assign locked colors if requested */

  if ((change & LOCK_COLORS) && (new_options & LOCK_COLORS) && !UNDEF_IMAGE) {
    NumLockedColors = Image[Current]->num_colors;
    LockedColors = Image[Current]->colors;
  }

  /* Toggle cursor if requested */

  if (change & LIVE_CURSOR) 
    if (new_options & LIVE_CURSOR) { /* Show canvas footer */
      if (new_options & NO_RESIZING)
	AllowManualSizing(TRUE); /* Allow frame to grow */
      (void) xv_set(CanvasFrame,
		    FRAME_SHOW_FOOTER, TRUE,
		    FRAME_LEFT_FOOTER, (UNDEF_IMAGE ? "(no image)" :
					"(no cursor)"),
		    NULL);

      /*
       * For some reason, if NO_RESIZING was on, toggling the footer
       * disables manual sizing, so force it if necessary.
       *
       */

      AllowManualSizing(!(new_options & NO_RESIZING));
    }
    else { /* Hide canvas footer */
      if (new_options & NO_RESIZING)
	AllowManualSizing(TRUE); /* Allow frame to shrink */
      (void) xv_set(CanvasFrame,
		    FRAME_SHOW_FOOTER, FALSE,
		    FRAME_LEFT_FOOTER, "",
		    NULL);
      AllowManualSizing(!(new_options & NO_RESIZING));
    }	   

  /* Update options variable */

  Options = new_options;

  /* Redisplay image if applicable */

  ShowNewImage();
}

void SetBackdrop(void)
{
  /* Changes the current backdrop */

  XGCValues values;

  /* Assign pixmap to stipple */

  values.stipple = (Pixmap) xv_get(Backdrop[BackdropIndex], XV_XID);

  /* Apply change */

  XChangeGC(MainDisplay, BackdropGC, GCStipple, &values);

  /* If undefined image, paint entire canvas; otherwise redisplay image */

  if (UNDEF_IMAGE && !NO_IMAGES)
    PaintBackground();
  else
    ShowNewImage();
}


void SetColorSliderDefaults(void)
{
  /* Sets default slider values */

  switch (Scaling) {
  case BW:
    RedSliderValue = GreenSliderValue = BlueSliderValue = 50;
    break;
  case GRAY:
    RedSliderValue = 30; /* Standard values...must add up to 100 */
    GreenSliderValue = 59;
    BlueSliderValue = 11;
    break;
  case COLOR:
    RedSliderValue = GreenSliderValue = BlueSliderValue = 100;
    break;
  case GAMMA:
    RedSliderValue = GreenSliderValue = BlueSliderValue = 10;
    break;
  case CUTOFF:
    RedSliderValue = GreenSliderValue = BlueSliderValue = 100;
    break;
  case CONTOUR:
    RedSliderValue = GreenSliderValue = BlueSliderValue = 25;
    break;
  case RANDOM:
    RedSliderValue = GreenSliderValue = BlueSliderValue = 50;
  }
}

void SetScaling(void)
{
  /* Initializes color sliders and applies scaling */

  SetColorSliders();

  /* Redisplay image with new scaling if applicable */

  if (!UNDEF_IMAGE && !(Options & NO_SCALING))
    ApplyColorScaling(TRUE);
}

#define LINE_WIDTH 79
#define TAB (strlen(prog))

static void show_usage(char *prog)
{
  /* Prints a usage message to stderr and aborts with error status 1 */

  char *options = "[-fast],[-[no]newcmap],[-[no]onecmap],[-[no]fastquant],[-[no]subpanel],[-[no]blink|-[no]cycle],[-[no]scrolling],[-[no]resizing],[-[no]moving],[-[no]scaling],[-[no]updates],[-[no]backdrops],[-[no]lockcolors],[-[no]centering],[-[no]livecursor],[-[no]raw],[-[no]query],[-[no]serverimages|-[no]clientimages],[-[no]visualload],[-[no]1],[-[no]oneatatime],[-[no]verbose|-[no]quiet],[-start (index)],[-private|-hist|-pair],[-copy (num)],[-op SRC|OR|AND|XOR|ERA|NEG],[-w (width)],[-h (height)],[-auto|-full|-fixed],[-timerscale us|ms|sec],[-timervalue (timer)],[-fwd|-rev],[-loop|-loopback|-oneway],[-backdrop (index)],[-bw|-gray|-color|-gamma|-cutoff|-contour|-random],[-red (value)],[-green (value)],[-blue (value)],[-rw (width)],[-rh (height)],[filename...]";

  int t, l;

  (void) fprintf(stderr, "Usage: (standard window options omitted)\n%s", prog);

  t = TAB;

  while (*options) {
    l = 0;
    while (*(options + l) && *(options + l) != ',')
      ++l;
    if (t > LINE_WIDTH || (t + l > LINE_WIDTH && l < LINE_WIDTH - TAB)) {
      (void) printf("\n");
      for (t = 0; t < TAB; t++)
	(void) printf(" ");
    }
    (void) printf(" ");
    ++t;
    for (; l > 0; l--, t++)
      (void) printf("%c", *(options++));
    if (*options)
      ++options;
  }

  (void) fprintf(stderr, "\n\"[ ]\" indicate optional items; "
		 "\"|\" indicates exclusive choices;\n");
  (void) fprintf(stderr, "\"( )\" indicate user-supplied values; "
		 "colors may be negative to denote\n");
  (void) fprintf(stderr, "inverse scale; "
		 "note -visualload overrides -start (index).\n");

  exit(1);
}

#undef TAB
#undef LINE_WIDTH

/*** params.c ***/
