/* worldmap - a world map display program
   Copyright (C) 1998 Andrew Skypeck
  
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * rmap.h - RPC client/server interface for the map program
 */

#ifndef RMAP_H
#define RMAP_H

#include <rpc/rpc.h>

/* C++ support */
#if defined(__cplusplus) || defined(c_plusplus)
extern "C" {
#endif

/* server program number, well known to clients */
#define PROGNUM     ((unsigned long) 0x20000006)

/* program version number, also well known to clients */
#define VERSION     ((unsigned long) 1)

/* procedure number for look-up in server dispatch table */
#define CLEAR        ((unsigned long) 1)
#define FREEZE       ((unsigned long) 2)
#define DRAW_POINTS  ((unsigned long) 3)
#define DRAW_LINES   ((unsigned long) 4)
#define DRAW_MARKERS ((unsigned long) 5)
#define DRAW_STRINGS ((unsigned long) 6)
#define FILL_POLYGON ((unsigned long) 7)
#define CHANGE_GC    ((unsigned long) 8)

/* geographic position */
typedef struct {		/* geographic point */
  float lat;			/* always in degrees */
  float lon;
} RMapGPoint;

typedef struct {		/* RPC transport structure */
  unsigned int npts;
  RMapGPoint *pts;
} RMapGPointArray;

typedef enum {
  RMap_CIRCLE = 0, RMap_SQUARE, RMap_TRIANGLE
} RMapMarkerType;

typedef struct {
  RMapMarkerType marker_type;
  unsigned int marker_width;
  unsigned int npts;
  RMapGPoint *pts;
} RMapMarkerArray;

#define RMAP_STRING_LEN 40
typedef char RMapString[RMAP_STRING_LEN];

typedef struct {
  unsigned int npts;
  RMapGPoint *pts;
  RMapString *str;
} RMapStringArray;

typedef struct {
  unsigned int width;
  unsigned int height;
  RMapString bits;
} RMapStipple;

#define RMapGCFunction              (1L<<0)
#define RMapGCPlaneMask             (1L<<1)
#define RMapGCForeground            (1L<<2)
#define RMapGCBackground            (1L<<3)
#define RMapGCLineWidth             (1L<<4)
#define RMapGCLineStyle             (1L<<5)
#define RMapGCCapStyle              (1L<<6)
#define RMapGCJoinStyle		    (1L<<7)
#define RMapGCFillStyle		    (1L<<8)
#define RMapGCFillRule		    (1L<<9) 
#define RMapGCTile		    (1L<<10)
#define RMapGCStipple		    (1L<<11)
#define RMapGCTileStipXOrigin	    (1L<<12)
#define RMapGCTileStipYOrigin	    (1L<<13)
#define RMapGCFont 		    (1L<<14)
#define RMapGCSubwindowMode	    (1L<<15)
#define RMapGCGraphicsExposures     (1L<<16)
#define RMapGCClipXOrigin	    (1L<<17)
#define RMapGCClipYOrigin	    (1L<<18)
#define RMapGCClipMask		    (1L<<19)
#define RMapGCDashOffset	    (1L<<20)
#define RMapGCDashList		    (1L<<21)
#define RMapGCArcMode		    (1L<<22)
#define RMapGCLastBit		    22

/* Data structure for remotely setting graphics context */
typedef struct {
  int function;			/* logical operation */
  unsigned long plane_mask;	/* plane mask */
  RMapString foreground;	/* foreground pixel */
  RMapString background;	/* background pixel */
  int line_width;		/* line width */
  int line_style;	 	/* LineSolid, LineOnOffDash, LineDoubleDash */
  int cap_style;	  	/* CapNotLast, CapButt, 
				   CapRound, CapProjecting */
  int join_style;	 	/* JoinMiter, JoinRound, JoinBevel */
  int fill_style;	 	/* FillSolid, FillTiled, 
				   FillStippled, FillOpaeueStippled */
  int fill_rule;	  	/* EvenOddRule, WindingRule */
  int arc_mode;			/* ArcChord, arcpieslice */
  RMapString tile;		/* tile pixmap for tiling operations */
  RMapString stipple;		/* stipple 1 plane pixmap for stipping */
  int ts_x_origin;	/* offset for tile or stipple operations */
  int ts_y_origin;
  RMapString font;		/* default text font for text operations */
  int subwindow_mode;		/* ClipByChildren, IncludeInferiors */
  bool_t graphics_exposures;	/* boolean, should exposures be generated */
  int clip_x_origin;		/* origin for clipping */
  int clip_y_origin;
  RMapString clip_mask;		/* bitmap clipping; other calls for rects */
  int dash_offset;		/* patterned/dashed line information */
  char dashes;
} RMapGCValues;

typedef struct {
  unsigned long mask;
  RMapGCValues values;
} RMapGCDesc;

/* public functions returning the RPC client status */
enum clnt_stat RMapClear(char *host);
enum clnt_stat RMapFreeze(char *host);
enum clnt_stat RMapDrawPoints(char *host, RMapGPoint *gp, unsigned int npts);
enum clnt_stat RMapDrawMarkers(char *host, RMapGPoint *gp, unsigned int npts,
			       RMapMarkerType marker_type,
			       unsigned int marker_width);
enum clnt_stat RMapDrawLines(char *host, RMapGPoint *gp, unsigned int npts);
enum clnt_stat RMapDrawStrings(char *host, RMapGPoint *gp, RMapString *s,
			       unsigned int npts);
enum clnt_stat RMapChangeGC(char *host, RMapGCValues values,
			    unsigned int mask);

#if defined(__cplusplus) || defined(c_plusplus)
}
#endif

#endif /* RMAP_H */
