/* worldmap - a world map display program
   Copyright (C) 1998 Andrew Skypeck
  
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * message.c
 */

#include <stdio.h>
#include <assert.h>
#include <string.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#ifdef USE_XPM
#include <X11/xpm.h>

/* XPM */
static char *xlogo32[] = {
/* width height num_colors chars_per_pixel */
"    32    32        2            1",
/* colors */
". c None",
"# c green",
/* pixels */
"########......................##",
".########.....................##",
"..########...................##.",
"...########.................##..",
"...########................##...",
"....########..............##....",
".....########............##.....",
"......########...........##.....",
"......########..........##......",
".......########........##.......",
"........########......##........",
".........########....##.........",
".........########...##..........",
"..........########..##..........",
"...........#######.##...........",
"............#####.##............",
"............####.###............",
".............##.#####...........",
"............##.#######..........",
"...........##..########.........",
"...........##..########.........",
"..........##....########........",
".........##......########.......",
"........##........########......",
".......##.........########......",
"......##...........########.....",
"......##............########....",
".....##..............########...",
"....##...............########...",
"...##.................########..",
"..##...................########.",
".##.....................########"
};
#else /* use X bitmap logo */
#include <X11/bitmaps/xlogo32>
#endif /* USE_XPM */

#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/MainW.h>
#include <Xm/MessageB.h>

#include "resources.h"

/* keep a copy of the message label around so we can write
   to it at any time */
static Widget message_label = NULL;

static String DialogTypeString(int dialog_type);

Widget
CreateMessageArea(Widget main_window)
{
  Widget message_form, logo_label;
  Pixmap label_pixmap;
  XmString xms;

  static Pixmap GetLogoPixmap(Widget label);

  assert(XmIsMainWindow(main_window));

  message_form = \
    XtVaCreateManagedWidget("message_form",
			    xmFormWidgetClass,      main_window,
			    NULL);

  xms = XmStringCreateSimple(" ");
  message_label = \
    XtVaCreateManagedWidget("messageLabel",
			    xmLabelWidgetClass,     message_form,
			    XmNtopAttachment,       XmATTACH_FORM,
			    XmNleftAttachment,      XmATTACH_FORM,
			    XmNrightAttachment,     XmATTACH_NONE,
			    XmNbottomAttachment,    XmATTACH_FORM,
			    XmNalignment,           XmALIGNMENT_BEGINNING,
			    XmNlabelString,         xms,
			    NULL);
  XmStringFree(xms);

  logo_label = \
    XtVaCreateManagedWidget("logoLabel",
			    xmLabelWidgetClass,     message_form,
			    XmNtopAttachment,       XmATTACH_FORM,
			    XmNrightAttachment,     XmATTACH_FORM,
			    XmNleftAttachment,      XmATTACH_NONE,
			    XmNbottomAttachment,    XmATTACH_FORM,
			    XmNalignment,           XmALIGNMENT_END,
			    XmNlabelType,           XmPIXMAP,
			    NULL);

  label_pixmap = GetLogoPixmap(logo_label);
  if (label_pixmap)
    {
      XtVaSetValues(logo_label,
		    XmNlabelPixmap, label_pixmap,
		    NULL);
    }

  return message_form;
}

static Pixmap
GetLogoPixmap(Widget logo_label)
{
  Pixmap label_pixmap;
  Pixel bg, fg;
#ifdef USE_XPM
  XpmAttributes attributes;
  XpmColorSymbol symbol;
  int status;
#else
  unsigned int width, height;
  int x_hot, y_hot;
#endif /* USE_XPM */

  extern InstanceRecord instance_record;

  XtVaGetValues(logo_label,
		XmNforeground, &fg,
		XmNbackground, &bg,
		NULL);

#ifdef USE_XPM
  symbol.name = NULL;
  symbol.value = "None";
  symbol.pixel = bg;

  attributes.colorsymbols = &symbol;
  attributes.numsymbols = 1;
  attributes.valuemask = XpmColorSymbols;
#endif /* USE_XPM */

  if (instance_record.logo == NULL)
    {
#ifdef USE_XPM
      status = \
	XpmCreatePixmapFromData(XtDisplay(logo_label),
				XRootWindowOfScreen(XtScreen(logo_label)),
				xlogo32, &label_pixmap, NULL, &attributes);
#else /* use X bitmap logo */
      label_pixmap = \
	XCreatePixmapFromBitmapData(XtDisplay(logo_label),
				    XRootWindowOfScreen(XtScreen(logo_label)),
				    xlogo32_bits, xlogo32_width,
				    xlogo32_height, fg, bg,
                                  DefaultDepthOfScreen(XtScreen(logo_label)));
#endif /* USE_XPM */
    }
  else
    {
#ifdef USE_XPM
      status = \
	XpmReadFileToPixmap(XtDisplay(logo_label),
			    XRootWindowOfScreen(XtScreen(logo_label)),
			    instance_record.logo,
			    &label_pixmap, NULL, &attributes);
#else /* use X bitmap logo */
	XReadBitmapFile(XtDisplay(logo_label),
			XRootWindowOfScreen(XtScreen(logo_label)),
			instance_record.logo,
			&width, &height, &label_pixmap, &x_hot, &y_hot);
#endif /* USE_XPM */
    }

#ifdef USE_XPM
  return (status == XpmSuccess) ? label_pixmap : (Pixmap) 0;
#else
  return label_pixmap;
#endif
}

/* put a message into the message area */
void
SetMessageArea(String new_message)
{
  XmString xms;

  assert(XmIsLabel(message_label));

  xms = XmStringCreateSimple(new_message);
  XtVaSetValues(message_label,
		XmNlabelString, xms,
		NULL);
  XmStringFree(xms);
}

void
PopupDialog(Widget parent, int dialog_type, char *msg)
{
  Widget dialog;
  XmString message = XmStringCreateLocalized(msg);
  XmString title;

  dialog = XmCreateMessageDialog(parent, "dialog", NULL, 0);
  title = XmStringCreateSimple(DialogTypeString(dialog_type));
  XtVaSetValues(dialog,
		XmNdialogType, dialog_type,
		XmNdialogTitle, title,
		XmNmessageString, message,
		NULL);
  XmStringFree(message);
  XmStringFree(title);

  XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_CANCEL_BUTTON));
  XtSetSensitive(XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON), False);
  XtAddCallback(dialog, XmNokCallback, (XtCallbackProc) XtDestroyWidget, NULL);
  XtManageChild(dialog);
}

static String
DialogTypeString(int dialog_type)
{
  String s;

  switch (dialog_type)
    {
    case XmDIALOG_ERROR:
      s = (String) strdup("Error Dialog");
      break;
    case XmDIALOG_INFORMATION:
      s = (String) strdup("Information Dialog");
      break;
    case XmDIALOG_MESSAGE:
      s = (String) strdup("Message Dialog");
      break;
    case XmDIALOG_QUESTION:
      s = (String) strdup("Question Dialog");
      break;
    case XmDIALOG_TEMPLATE:
      s = (String) strdup("Template Dialog");
      break;
    case XmDIALOG_WARNING:
      s = (String) strdup("Warning Dialog");
      break;
    case XmDIALOG_WORKING:
      s = (String) strdup("Working Dialog");
      break;
    default:
      s = NULL;
      break;
    }

  return s;
}
