/*
 * MapDraw.c
 */
#include "MapDraw.h"
#include "MapP.h"
#include "MapProject.h"

#include <sys/ipc.h>
#include <sys/shm.h>
#include <assert.h>
#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <string.h>

void
MapCreatePixmap(MapWidget mw)
{
  Display *display = XtDisplay(mw);
  Screen *scr = XtScreen(mw);

  if (mw->map.map_pixmap != XmUNSPECIFIED_PIXMAP)
    XFreePixmap(display, mw->map.map_pixmap);

  mw->map.map_pixmap = XCreatePixmap(display, RootWindowOfScreen(scr),
				     mw->core.width, mw->core.height,
				     DefaultDepthOfScreen(scr));
}

void
MapCreateGC(MapWidget mw)
{
  Display *display = XtDisplay(mw);
  XtGCMask mask;
  XGCValues values;

  if (mw->map.copy_gc != NULL)
    XFreeGC(display, mw->map.copy_gc);

  if (mw->map.xor_gc != NULL)
    XFreeGC(display, mw->map.xor_gc);

  mask = GCFunction | GCForeground | GCBackground | GCLineStyle | GCLineWidth
         | GCFillRule | GCFont;

  values.function = GXcopy;
  values.foreground = mw->primitive.foreground;
  values.background = mw->core.background_pixel;
  values.line_style = LineSolid;
  values.line_width = 1;
  values.fill_rule = WindingRule;
  values.font = mw->map.font->fid;
  mw->map.copy_gc = XCreateGC(display, mw->map.map_pixmap, mask, &values);

  values.function = GXxor;
  mw->map.xor_gc = XCreateGC(display, mw->map.map_pixmap, mask, &values);
}

void
MapChangeGC(Display *display, GC gc, XtGCMask in_mask, XGCValues *in_values)
{
  XtGCMask mask = 0;
  XGCValues values;

  /* Get the current graphic context values */
  assert(XGetGCValues(display, gc, in_mask, &values));

  if (in_mask & GCForeground && values.foreground != in_values->foreground)
  {
    values.foreground = in_values->foreground;
    mask |= GCForeground;
  }

  if (in_mask & GCLineWidth && values.line_width != in_values->line_width)
  {
    values.line_width = in_values->line_width;
    mask |= GCLineWidth;
  }

  if (in_mask & GCLineStyle && values.line_style != in_values->line_style)
  {
    values.line_style = in_values->line_style;
    mask |= GCLineStyle;
  }

  if (mask != 0)
    {
      XChangeGC(display, gc, mask, &values);
    }
}

void
MapSetFont(MapWidget mw)
{
  XmFontContext context;
  XmStringCharSet charset;
  XFontStruct *font;

  /* Make a private copy of the FontList */
  mw->map.font_list = XmFontListCopy(mw->map.font_list);

  /* Get XFontStruct from FontList */
  if (!XmFontListInitFontContext(&context, mw->map.font_list))
  {
    XtAppErrorMsg(XtWidgetToApplicationContext((Widget) mw),
		  "newFont", "badFont", "XmMap",
		  "XmMap: XmFontListInitFontContext failed, bad fontList",
		  NULL, 0);
  }

  if (!XmFontListGetNextFont(context, &charset, &font))
  {
    XtAppErrorMsg(XtWidgetToApplicationContext((Widget) mw),
		  "newFont", "badFont", "XmMap",
		  "XmMap: XmFontListGetNextFont failed, "
		  "cannot get font from fontList",
		  NULL, 0);
  }

  XtFree(charset);
  XmFontListFreeFontContext(context);

  mw->map.font = font;
}

void
MapCreateShmImage(MapWidget mw)
{
#if USE_SHM
  Display *display = XtDisplay(mw);
  unsigned int depth = DefaultDepthOfScreen(XtScreen(mw));
  XImage *image;

  /* Destroy the current shared memory and image if they exist */
  if (mw->map.basemap_image != NULL)
    XmMapDestroyShm((Widget) mw);

  /* Create a new shared memory image */
  mw->map.basemap_image = \
  image = XShmCreateImage(display, NULL, depth, ZPixmap, NULL,
			  &mw->map.shm_info, mw->core.width, mw->core.height);
  if (image == NULL)
    XtError("XShmCreateImage failed");

  /* Get a new shared memory segment */
  mw->map.shm_info.shmid = shmget(IPC_PRIVATE,
				  image->bytes_per_line * image->height,
				  IPC_CREAT | 0777);
  if (mw->map.shm_info.shmid == -1)
    XtError("Map: shared memory error");
    /* XtError(strerror(errno)); */

  /* Attach the process to the new shared memory */
  mw->map.basemap_image->data = \
  mw->map.shm_info.shmaddr = (char*) shmat(mw->map.shm_info.shmid, NULL, 0);
  if (mw->map.shm_info.shmaddr == (char*) -1)
    XtError("Map: shared memory error");
    /* XtError(strerror(errno)); */

  /* Make it so that we can read and write to the shared memory pool */
  mw->map.shm_info.readOnly = False;

  /* Tell the X server to attach to the shared memory */
  if (!XShmAttach(display, &mw->map.shm_info))
    XtError("XShmAttach failed");
#else
  XtWarning("Map: shared memory extension not configured");
#endif /* USE_SHM */
}

void
MapSetBasemapImage(MapWidget mw)
{
  Display *display = XtDisplay(mw);

  if (mw->map.use_shm)
  {
#ifdef USE_SHM
    if (!XShmGetImage(display, mw->map.map_pixmap, mw->map.basemap_image,
		      0, 0, AllPlanes))
    {
      XtError("XShmGetImage failed");
    }
#else
    XtWarning("Map: shared memory extension not configured");
#endif /* USE_SHM */
  }
  else
  {
    if (mw->map.basemap_image != NULL)
      XDestroyImage(mw->map.basemap_image);

    mw->map.basemap_image = XGetImage(display, mw->map.map_pixmap, 0, 0,
				      mw->core.width, mw->core.height,
				      AllPlanes, XYPixmap);
  }
}

void
MapGetBasemapImage(MapWidget mw)
{
  Display *display = XtDisplay(mw);

  if (mw->map.use_shm)
  {
    if (!XShmPutImage(display, mw->map.map_pixmap, mw->map.copy_gc,
		      mw->map.basemap_image, 0, 0, 0, 0,
		      mw->core.width, mw->core.height, False))
    {
      XtError("XShmPutImage failed");
    }
  }
  else
  {
    XPutImage(display, mw->map.map_pixmap, mw->map.copy_gc,
	      mw->map.basemap_image,
	      0, 0, 0, 0, mw->core.width, mw->core.height);
  }

  if (mw->map.refresh)
    XmMapRefresh((Widget) mw);
}

void
MapDrawPoints(MapWidget mw, GC gc, const XmMapGPoint *gpts, Cardinal npts)
{
  Cardinal i;
  Cardinal xpts = 0;
  Display *display = XtDisplay(mw);
  XPoint *xp = (XPoint *) XtMalloc(npts * sizeof(XPoint));

  for (i = 0; i < npts; i++)
    if (!MapGPointToXPoint(mw, &gpts[i], &xp[xpts++]))
      --xpts;

  if (xpts > 0)
  {
    XDrawPoints(display, mw->map.map_pixmap, gc, xp, xpts, CoordModeOrigin);

    if (mw->map.refresh && XtIsRealized(mw))
    {
      XDrawPoints(display, XtWindow(mw), gc, xp, xpts, CoordModeOrigin);
    }
  }

  XtFree((char*) xp);
}

void
MapDrawStrings(MapWidget mw, GC gc, const XmMapGPoint *gpts, String* s,
	       Cardinal npts)
{
  Cardinal i;
  Display *display = XtDisplay(mw);

  for (i = 0; i < npts; i++)
  {
    XPoint xp;

    if (MapGPointToXPoint(mw, &gpts[i], &xp))
    {
      XDrawString(display, mw->map.map_pixmap, gc, xp.x, xp.y,
		  s[i], strlen(s[i]));

      if (mw->map.refresh && XtIsRealized(mw))
      {
	XDrawString(display, XtWindow(mw), gc, xp.x, xp.y, s[i], strlen(s[i]));
      }
    }
  }
}

void
MapDrawMarkers(MapWidget mw, GC gc, const XmMapGPoint *gpts, Cardinal npts,
	       XmMarkerType marker_type, Cardinal marker_size)
{
  Cardinal i;
  Display *display = XtDisplay(mw);

  for (i = 0; i < npts; i++)
  {
    XPoint xp;

    if (MapGPointToXPoint(mw, &gpts[i], &xp))
    {
      if (marker_type == XmTRIANGLE)
      {
	XPoint tri[3];
	double ydiff = marker_size * sqrt(3.0) / 4.0;

	tri[0].x = xp.x;
	tri[0].y = (int)(xp.y - ydiff);

	tri[1].x = tri[0].x + marker_size / 2;
	tri[1].y = (int)(xp.y + ydiff);

	tri[2].x = tri[1].x - marker_size;
	tri[2].y = (int)(xp.y + ydiff);

	XFillPolygon(display, mw->map.map_pixmap, gc, tri,
		     3, Convex, CoordModeOrigin);

	if (mw->map.refresh && XtIsRealized(mw))
	{
	  XFillPolygon(display, XtWindow(mw), gc, tri, 3, Convex,
		       CoordModeOrigin);
	}
      }
      else if (marker_type == XmSQUARE)
      {
	xp.x -= marker_size / 2;
	xp.y -= marker_size / 2;

	XFillRectangle(display, mw->map.map_pixmap, gc,
		       xp.x, xp.y, marker_size, marker_size);

	if (mw->map.refresh && XtIsRealized(mw))
        {
	  XFillRectangle(display, XtWindow(mw), gc,
			 xp.x, xp.y, marker_size, marker_size);
	}
      }
      else /* Draw a circle */
      {
	xp.x -= marker_size / 2;
	xp.y -= marker_size / 2;

	XFillArc(display, mw->map.map_pixmap, gc,
		 xp.x, xp.y, marker_size, marker_size, 0, 64 * 360);

	if (mw->map.refresh && XtIsRealized(mw))
	{
	  XFillArc(display, XtWindow(mw), gc,
		   xp.x, xp.y, marker_size, marker_size, 0, 64 * 360);
	}
      }
    }
  }
}

/* Assigns ix,iy to the intesection point of lines {a,b} and {c,d} and
   returns the distance between point a and point i. Returns -1 if the
   two lines don't intersect */
static float
CrossPoint(float ax, float ay, float bx, float by,
	   float cx, float cy, float dx, float dy,
	   float* ix, float* iy)
{
  double rd = (bx - ax) * (dy - cy) - (by - ay) * (dx - cx);

  if (rd != 0.0)
  {
    double r = ((ay - cy) * (dx - cx) - (ax - cx) * (dy - cy)) / rd;
    double s = ((ay - cy) * (bx - ax) - (ax - cx) * (by - ay)) / rd;

    if (0.0 <= r && r <= 1.0 && 0.0 <= s && s <= 1.0)
    {
      double d_x;
      double d_y;

      *ix = ax + r * (bx - ax);
      *iy = ay + r * (by - ay);

      d_x = ax - *ix;
      d_y = ay - *iy;

      return sqrt(d_x*d_x + d_y*d_y);
    }
  }

  return -1.0;
}

/* Returns True if the the line segment [from,to] crosses the 180 degree
   longitude zipper and assigns crossPoint to the cross location on the
   from side. Returns False if the line segment does not cross the zipper */
static Boolean
CrossesZipper(const XmMapGPoint *from, const XmMapGPoint *to,
	      XmMapGPoint *crossPoint)
{
  if (fabs(from->lon - to->lon) > 180.0)
  {
    float delta = (from->lon > 0.0) ? 360.0 : -360.0;

    return (CrossPoint(from->lon, from->lat, to->lon + delta, to->lat,
		       delta / 2.0, -90.0, delta / 2.0, 90.0,
		       &crossPoint->lon, &crossPoint->lat) > 0.0);
  }
  return False;
}

/* Assigns xp to the intesection point of the line segment [xp1,xp2] and
   the rectangle. If there are more than one intersection point, xp will be
   assigned to the one closest to xp1. Return False if the line segment
   does not intersect any sides of the rectangle */
static Boolean
GetLineSegment(MapWidget mw, const XPoint *xp1, const XPoint *xp2, XPoint *xp)
{
  float min_dis = HUGE;
  float dis;
  float px;
  float py;

  dis = CrossPoint(xp1->x, xp1->y, xp2->x, xp2->y, 0.0, 0.0, 0.0,
		   mw->core.height, &px, &py);
  if (dis != -1.0 && dis < min_dis)
  {
    min_dis = dis;
    xp->x = px;
    xp->y = py;
  }

  dis = CrossPoint(xp1->x, xp1->y, xp2->x, xp2->y, 0.0, 0.0, mw->core.width,
		   0.0, &px, &py);
  if (dis != -1.0 && dis < min_dis)
  {
    min_dis = dis;
    xp->x = px;
    xp->y = py;
  }

  dis = CrossPoint(xp1->x, xp1->y, xp2->x, xp2->y, 0.0, mw->core.height,
		   mw->core.width, mw->core.height, &px, &py);
  if (dis != -1.0 && dis < min_dis)
  {
    min_dis = dis;
    xp->x = px;
    xp->y = py;
  }

  dis = CrossPoint(xp1->x, xp1->y, xp2->x, xp2->y, mw->core.width, 0.0,
		   mw->core.width, mw->core.height, &px, &py);
  if (dis != -1.0 && dis < min_dis)
  {
    min_dis = dis;
    xp->x = px;
    xp->y = py;
  }

  return (min_dis != HUGE);
}

static void
CopyPoint(const XPoint *x, XPoint *xp, Cardinal* xpts)
{
  if (*xpts == 0 || x->x != xp[*xpts - 1].x || x->y != xp[*xpts - 1].y)
  {
    xp[*xpts].x = x->x;
    xp[(*xpts)++].y = x->y;
  }
}

static void
DrawLines(MapWidget mw, GC gc, XPoint *xp, Cardinal xpts)
{
  if (xpts > 1)
  {
    Display *display = XtDisplay(mw);

    XDrawLines(display, mw->map.map_pixmap, gc, xp, xpts,
	       CoordModeOrigin);

    if (mw->map.refresh && XtIsRealized(mw))
    {
      XDrawLines(display, XtWindow(mw), gc, xp, xpts,
		 CoordModeOrigin);
    }
  }
}

static void
AddLinePoint(MapWidget mw, GC gc, const XmMapGPoint *gpt,
	     XPoint *xp, Cardinal* xpts,
	     Boolean* inside, XPoint *lastPoint)
{
  XPoint x;
  Boolean visible = MapGPointToXPoint(mw, gpt, &x);

  if (visible)
  {
    /* Get the line segment point if we are coming in from the outside */
    if (*inside == False)
      assert(GetLineSegment(mw, lastPoint, &x, &xp[(*xpts)++]));

    CopyPoint(&x, xp, xpts);
  }
  else
  {
    /* Draw the line segments if we are going from inside to outside */
    if (*xpts != 0 && *inside)
    {
      assert(GetLineSegment(mw, &xp[*xpts - 1], &x, &xp[*xpts]));

      DrawLines(mw, gc, xp, *xpts + 1);
      *xpts = 0;
    }

    lastPoint->x = x.x;
    lastPoint->y = x.y;
  }

  *inside = visible;
}

void
MapDrawLines(MapWidget mw, GC gc, const XmMapGPoint *gpts, Cardinal npts)
{
  Cardinal xpts = 0;
  XPoint *xp = (XPoint*) XtMalloc(npts * sizeof(XPoint));
  Boolean inside = True;
  XPoint lastPoint;

  Cardinal i;
  for (i = 0; i < npts; i++)
  {
    XmMapGPoint crossPoint;

    if (i > 0 && CrossesZipper(&gpts[i - 1], &gpts[i], &crossPoint))
    {
      AddLinePoint(mw, gc, &crossPoint, xp, &xpts, &inside, &lastPoint);

      DrawLines(mw, gc, xp, xpts);
      xpts = 0;
      inside = True;

      crossPoint.lon = (gpts[i].lon < 0) ? -180.0 : 180.0;
      AddLinePoint(mw, gc, &crossPoint, xp, &xpts, &inside, &lastPoint);
    }
    else
      AddLinePoint(mw, gc, &gpts[i], xp, &xpts, &inside, &lastPoint);
  }

  DrawLines(mw, gc, xp, xpts);

  XtFree((char*) xp);
}

static void
RearrangePoints(XPoint** xp, Boolean** b, Cardinal offset, Cardinal npts)
{
  Boolean* b1 = *b;
  Boolean* b2 = (Boolean*) XtMalloc(npts * sizeof(Boolean));
  Cardinal nmo = npts - offset;
  XPoint *xp1 = *xp;
  XPoint *xp2 = (XPoint*) XtMalloc(npts * sizeof(XPoint));

  memcpy(b2, &b1[offset], nmo * sizeof(Boolean));
  memcpy(&b2[nmo], b1, offset * sizeof(Boolean));

  memcpy((char*) xp2, (char*) &xp1[offset], nmo * sizeof(XPoint));
  memcpy((char*) &xp2[nmo], (char*) xp1, offset * sizeof(XPoint));

  XtFree(b1);
  XtFree((char*) xp1);

  *xp = xp2;
  *b = b2;
}

static float
GetAngleDiff(MapWidget mw, const XPoint *xp1, const XPoint *xp2)
{
  double cx = (double) mw->core.width / 2.0;
  double cy = (double) mw->core.height / 2.0;
  double ax = (double) xp1->x - cx;
  double ay = (double) xp1->y - cy;
  double bx = (double) xp2->x - cx;
  double by = (double) xp2->y - cy;
  const double one = 0.99999999999;
  double num1;

  /* normalize the vectors */
  double len_a = sqrt(ax*ax + ay*ay);
  double len_b = sqrt(bx*bx + by*by);
  double dotp = ax*bx + ay*by;

  /* catch roundoff error when num1 = 1 */
  num1 = dotp / (len_a * len_b);
  if (fabs(num1) > one) num1 = one;

  return (float) copysign(acos(num1), ax*by - ay*bx);
}

static Dimension
GetQuadrient(MapWidget mw, const XPoint *xp)
{
  if (xp->y == 0)
    return 0;
  else if (xp->x == mw->core.width)
    return 1;
  else if (xp->y == mw->core.height)
    return 2;
  else
    return 3;
}

static void
GetCornerPoints(MapWidget mw, const XPoint *xp1, const XPoint *xp2,
		float alpha, XPoint *p, Cardinal* Xpts)
{
  Dimension x[4] = {0, 0, 0, 0};
  Dimension y[4] = {0, 0, 0, 0};
  Cardinal s_quad = GetQuadrient(mw, xp1);
  Cardinal e_quad = GetQuadrient(mw, xp2);
  Cardinal xpts = *Xpts;

  x[1] = x[2] = mw->core.width;
  y[2] = y[3] = mw->core.height;

  while (s_quad != e_quad)
  {
    if (alpha > 0) /* We are going clockwise */
    {
      s_quad = (s_quad + 1) % 4;

      p[xpts].x = x[s_quad];
      p[xpts++].y = y[s_quad];
    }
    else /* We are going counter-clockwise */
    {
      p[xpts].x = x[s_quad];
      p[xpts++].y = y[s_quad];

      s_quad = (s_quad - 1) % 4;
    }
  }

  *Xpts = xpts;
}

static Boolean
PointInsidePolygon(short x, short y, XPoint *xp, Cardinal npts)
{
  Cardinal i;
  Cardinal j;
  Boolean c = False;

  for (i = 0, j = npts - 1; i < npts; j = i++)
  {
    if ((((xp[i].y <= y) && (y < xp[j].y))
	 || ((xp[j].y <= y) && (y < xp[i].y)))
	&& (x < (xp[j].x - xp[i].x) * (y - xp[i].y)
	    / (xp[j].y - xp[i].y) + xp[i].x))
    {
      c = !c;
    }
  }

  return c;
}

void
MapFillPolygon(MapWidget mw, GC gc, const XmMapGPoint *gpts, Cardinal npts)
{
  Boolean* visible = (Boolean*) XtMalloc(npts * sizeof(Boolean));
  short i;
  Cardinal offset = -1;
  Display *display = XtDisplay(mw);
  XPoint *xp = (XPoint*) XtMalloc(npts * sizeof(XPoint));
  short j = 0;

  /* Get all the XPoints, which ones are visible, and the first visible
     point */
  for (i = npts - 1; i >= 0; --i)
  {
    ++j;
    if (visible[i] = MapGPointToXPoint(mw, &gpts[i], &xp[i]))
      offset = i;
  }

  /* Offset is greater than -1 if at least one point is visible */
  if (offset != -1)
  {
    float alpha;
    Boolean inside = True;
    Cardinal total_pts = 100;
    Cardinal xpts = 0;
    XPoint *last_point;
    XPoint *ixp = (XPoint*) XtMalloc(total_pts * sizeof(XPoint));

    /* Rearrange the points so that the first point is always visible */
    if (offset > 0)
      RearrangePoints(&xp, &visible, offset, npts);

    /* Loop through all the points and form a polygon which is inside the
       window */
    for (i = 0; i < npts; i++)
    {
      /* Handle if we are running out of points */
      if (total_pts - xpts < 5)
      {
	total_pts += 100;
	ixp = (XPoint*) XtRealloc((char*) ixp, total_pts * sizeof(XPoint));
      }

      /* Handle if we are going from inside to inside */
      /* This should always happen to the first point */
      if (inside && visible[i])
	CopyPoint(&xp[i], ixp, &xpts);

      /* Handle if we are going from inside to outside */
      else if (inside && !visible[i])
      {
	XPoint p;

	assert(GetLineSegment(mw, &xp[i - 1], &xp[i], &p));
	alpha = GetAngleDiff(mw, &p, &xp[i]);

	CopyPoint(&p, ixp, &xpts);
	last_point = &xp[i];
      }

      /* Handle if we are going from outside to inside */
      else if (!inside && visible[i])
      {
	XPoint p;

	/* Get the line segment coming in and place it in a temp. var */
	assert(GetLineSegment(mw, last_point, &xp[i], &p));

	/* Get the angle difference */
	alpha += GetAngleDiff(mw, last_point, &p);

	/* Get the corner points */
	GetCornerPoints(mw, &ixp[xpts - 1], &p, alpha, ixp, &xpts);

	/* Append the line segment points to the end of the list */
	CopyPoint(&p, ixp, &xpts);
      }

      /* Handle if we are going outside to outside */
      else
      {
	alpha += GetAngleDiff(mw, last_point, &xp[i]);
	last_point = &xp[i];
      }

      inside = visible[i];
    }

    /* Get the corner points if we are still outside the rectangle */
    if (!inside)
    {
      XPoint p;

      /* Get the location we come in at */
      assert(GetLineSegment(mw, &xp[npts - 1], &xp[0], &p));

      /* Get the angle difference */
      alpha += GetAngleDiff(mw, &xp[npts - 1], &p);

      /* Get the corner points */
      GetCornerPoints(mw, &ixp[xpts - 1], &p, alpha, ixp, &xpts);

      /* Append the line segment points to the end of the list */
      CopyPoint(&p, ixp, &xpts);
    }

    /* Sanity check */
    assert(xpts > 0);

    /* Draw the polygon on the pixmap */
    XFillPolygon(display, mw->map.map_pixmap, gc, ixp, xpts,
		 Complex, CoordModeOrigin);

    if (mw->map.refresh && XtIsRealized(mw))
    {
      XFillPolygon(display, XtWindow(mw), gc, ixp, xpts, Complex,
		   CoordModeOrigin);
    }

    XtFree((char*) ixp);
  }
  else if (PointInsidePolygon(0.0, 0.0, xp, npts))
  {
    XFillRectangle(display, mw->map.map_pixmap, gc, 0, 0,
		   mw->core.width, mw->core.height);

    if (mw->map.refresh && XtIsRealized(mw))
    {
      XFillRectangle(display, XtWindow(mw), gc, 0, 0,
		     mw->core.width, mw->core.height);
    }
  }

  XtFree(visible);
  XtFree((char*) xp);
}
