#include "MapBasemap.h"
#include "MapDraw.h"
#include "MapP.h"

#include <sys/param.h>
#include <memory.h>
#include <stdio.h>
#include <string.h>

static void
MapDrawPolygonFile(MapWidget mw, const String file_dir,
		   const String file_name, Boolean draw_filled)
{
  FILE *f;
  char file[MAXPATHLEN];

  sprintf(file, "%s/%s", file_dir, file_name);

  f = fopen(file, "r");
  if (f != NULL)
    {
      while (!feof(f))
	{
	  unsigned int npts;
	  if (fread((char *) &npts, sizeof(npts), 1, f) == 1)
	    {
	      XmMapGPoint *gpts = \
		(XmMapGPoint*)XtMalloc(npts * sizeof(XmMapGPoint));

	      if (fread((char *) gpts, sizeof(XmMapGPoint), npts, f) == npts)
		{
		  if (draw_filled)
		    MapFillPolygon(mw, mw->map.copy_gc, gpts, npts);
		  else
		    {
		      /* Connect the last point to the first point
			 before drawing as line segments */
		      gpts = \
			(XmMapGPoint*) XtRealloc((char *) gpts,
						 ++npts * sizeof(XmMapGPoint));

		      memcpy((char *) &gpts[npts - 1], (char *) &gpts[0],
			     sizeof(XmMapGPoint));

		      MapDrawLines(mw, mw->map.copy_gc, gpts, npts);
		    }
		}
	      XtFree((char *) gpts);
	    }
	}

      fclose(f);
    }
  else
    {
      char s[80];

      sprintf(s, "Map: unable to open map file \"%s\" for reading\n", file);
      XtWarning(s);
    }
}

static void
MapDrawLineFile(MapWidget mw, const char *file_dir, const char *file_name)
{
  FILE* f;
  char file[MAXPATHLEN];

  sprintf(file, "%s/%s", file_dir, file_name);

  f = fopen(file, "r");
  if (f != NULL)
  {
    while(!feof(f))
    {
      unsigned int npts;

      if (fread((char *) &npts, sizeof(npts), 1, f) == 1)
      {
	XmMapGPoint *gpts = \
	  (XmMapGPoint *) XtMalloc(npts * sizeof(XmMapGPoint));

	if (fread((char *) gpts, sizeof(XmMapGPoint), npts, f) == npts)
	  MapDrawLines(mw, mw->map.copy_gc, gpts, npts);

	XtFree((char *) gpts);
      }
    }

    fclose(f);
  }
  else
    {
      char s[80];

      sprintf(s, "Map: unable to open map file \"%s\" for reading\n", file);
      XtWarning(s);
    }
}

static void
MapFillBackground(MapWidget mw)
{
  Display *display = XtDisplay(mw);
  XmMapGPoint *gpts = (XmMapGPoint *) XtMalloc(1080 * sizeof(XmMapGPoint));
  Cardinal npts = 0;
  short i;

  /* Set the GC for the map background */
  XtGCMask mask = GCForeground;
  XGCValues values;
  values.foreground = mw->core.background_pixel;
  MapChangeGC(display, mw->map.copy_gc, mask, &values);

  /* Fill the entire map with the background color */
  XFillRectangle(display, mw->map.map_pixmap, mw->map.copy_gc, 0, 0,
		 mw->core.width, mw->core.height);

  if (mw->map.refresh && XtIsRealized(mw))
  {
    XFillRectangle(display, XtWindow(mw), mw->map.copy_gc, 0, 0,
		   mw->core.width, mw->core.height);
  }

  /* Set the GC to the ocean color */
  mask = GCForeground;
  values.foreground = mw->map.ocean_color;
  MapChangeGC(display, mw->map.copy_gc, mask, &values);

  /* Calculate the ocean boundry points */
  for (i = -90; i < 90; i++)
  {
    gpts[npts].lat = i;
    gpts[npts++].lon = 180.0;
  }

  for (i = 180.0; i > -180; i--)
  {
    gpts[npts].lat = 90.0;
    gpts[npts++].lon = i;
  }

  for (i = 90; i > -90; i--)
  {
    gpts[npts].lat = i;
    gpts[npts++].lon = -180.0;
  }

  for (i = -180.0; i < 180; i++)
  {
    gpts[npts].lat = -90.0;
    gpts[npts++].lon = i;
  }

  /* Draw the ocean */
  MapFillPolygon(mw, mw->map.copy_gc, gpts, npts);

  XtFree((char *) gpts);
}

static void
MapDrawLand(MapWidget mw)
{
  Display *display = XtDisplay(mw);
  XtGCMask mask;
  XGCValues values;

  mask = GCForeground;
  values.foreground = mw->map.land_color;
  MapChangeGC(display, mw->map.copy_gc, mask, &values);

  MapDrawPolygonFile(mw, mw->map.map_dir, mw->map.coastlines_file, True);

  if (mw->map.draw_coastline)
  {
    mask = GCForeground | GCLineWidth | GCLineStyle;
    values.foreground = mw->map.coastline_color;
    values.line_width = 1;
    values.line_style = LineSolid;
    MapChangeGC(display, mw->map.copy_gc, mask, &values);

    MapDrawPolygonFile(mw, mw->map.map_dir, mw->map.coastlines_file, False);
  }
}

static void
MapDrawLakes(MapWidget mw)
{
  XtGCMask mask;
  XGCValues values;

  mask = GCForeground;
  values.foreground = mw->map.lake_color;
  MapChangeGC(XtDisplay(mw), mw->map.copy_gc, mask, &values);

  MapDrawPolygonFile(mw, mw->map.map_dir, mw->map.lakes_file, True);
}

static void
MapDrawRivers(MapWidget mw)
{
  XtGCMask mask;
  XGCValues values;

  mask = GCForeground | GCLineWidth | GCLineStyle;
  values.foreground = mw->map.river_color;
  values.line_width = 1;
  values.line_style = LineSolid;
  MapChangeGC(XtDisplay(mw), mw->map.copy_gc, mask, &values);

  MapDrawLineFile(mw, mw->map.map_dir, mw->map.rivers_file);
}

static void
MapDrawBoundaries(MapWidget mw)
{
  XtGCMask mask;
  XGCValues values;

  mask = GCForeground | GCLineWidth | GCLineStyle;
  values.foreground = mw->map.boundary_color;
  values.line_width = 1;
  values.line_style = LineSolid;
  MapChangeGC(XtDisplay(mw), mw->map.copy_gc, mask, &values);

  MapDrawLineFile(mw, mw->map.map_dir, mw->map.boundaries_file);
}

static void
MapDrawGrid(MapWidget mw)
{
  XtGCMask mask;
  XGCValues values;
  XmMapGPoint *gpts;
  Cardinal npts;
  float i;
  float j;

  mask = GCForeground;
  values.foreground = mw->map.grid_color;
  MapChangeGC(XtDisplay(mw), mw->map.copy_gc, mask, &values);

  npts = (360.0 / mw->map.minor_grid_size + 1.0)
         * (180.0 / mw->map.major_grid_size + 1.0)
         * sizeof(XmMapGPoint);

  gpts = (XmMapGPoint *) XtMalloc(npts);

  /* Draw latitude lines */
  npts = 0;
  for (i = 0.0; i <= 180.0; i += mw->map.major_grid_size)
    for (j = 0.0; j <= 90.0; j+= mw->map.minor_grid_size)
    {
      gpts[npts].lat = j;
      gpts[npts++].lon = i;

      if (i != 0.0)
      {
	gpts[npts].lat = j;
	gpts[npts++].lon = -i;
      }

      if (j != 0.0)
      {
	gpts[npts].lat = -j;
	gpts[npts++].lon = i;
      }

      if (i != 0.0 && j != 0.0)
      {
	gpts[npts].lat = -j;
	gpts[npts++].lon = -i;
      }
    }

  MapDrawPoints(mw, mw->map.copy_gc, gpts, npts);

  /* Draw longitude lines */
  npts = 0;
  for (j = 0.0; j <= 90.0; j += mw->map.major_grid_size)
    for (i = 0.0; i <= 180.0; i += mw->map.minor_grid_size)
    {
      gpts[npts].lat = j;
      gpts[npts++].lon = i;

      if (i != 0.0)
      {
	gpts[npts].lat = j;
	gpts[npts++].lon = -i;
      }

      if (j != 0.0)
      {
	gpts[npts].lat = -j;
	gpts[npts++].lon = i;
      }

      if (i != 0.0 && j != 0.0)
      {
	gpts[npts].lat = -j;
	gpts[npts++].lon = -i;
      }
    }

  MapDrawPoints(mw, mw->map.copy_gc, gpts, npts);

  XtFree((char *) gpts);
}

void
MapDrawBasemap(MapWidget mw)
{
  MapFillBackground(mw);
  MapDrawLand(mw);

  if (mw->map.draw_lakes)
    MapDrawLakes(mw);

  if (mw->map.draw_rivers)
    MapDrawRivers(mw);

  if (mw->map.draw_boundaries)
    MapDrawBoundaries(mw);

  if (mw->map.draw_grid)
    MapDrawGrid(mw);
}
